/****h* gnocl/widget/sourceview
 * NAME
 * 	sourceview.c
 * AUTHOR
 * 	William J Giddings
 * SYNOPSIS
 * 	 This file implements the sourceview widget.
 * FUNCTION
 * NOTES
 *
 * Getting a hangle on the sourceView buffer
 *
 * buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (view));
 * gtk_text_buffer_set_text (GTK_TEXT_BUFFER( buffer ), "Your 1st GtkSourceView widget!", -1);
 *
 *
 * BUGS
 * SEE ALSO
 *****/
/*
 * $Id: sourceview.c,v 0.1 2008/10/10 19:29:04 giddings Exp $
 *
 * This file implements the sourceview widget
 *
 * Copyright (c) 2008 William J Giddings
 * Copyright (c) 2001 - 2004 Peter G. Baum  http://www.dr-baum.net
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

/*
   History:
   2008-10: Begin of developement

 */

#include "sourceview.h"
#include <string.h>
#include <assert.h>

/****v* text/constants
 * SOURCE
 */
static const int scrollBarIdx = 0;
/*****/

/****f* callbackFunction/doOnUndoRedo
 * NAME
 * AUTHOR
 * 	PGB
 * SYNOPSIS
 * FUNCTION
 *	Callback function for
 * NOTES
 * BUGS
 * SEE ALSO
 * SOURCE
 */
static void doOnUndoRedo  ( GtkSourceView *sourceview, gpointer user_data )
{
	GnoclCommandData *cs = ( GnoclCommandData * ) user_data;

	GnoclPercSubst ps[] =
	{
		{ 'w', GNOCL_STRING },  /* widget */
		{ 0 }
	};

	ps[0].val.str = gnoclGetNameFromWidget ( sourceview );
	gnoclPercentSubstAndEval ( cs->interp, ps, cs->command, 1 );
}

/*****/

/****f* sourceview/parseOptions/gnoclOptOnUndoRedo
 * NAME
 * AUTHOR
 * 	WJG
 * SYNOPSIS
 * FUNCTION
 * NOTES
 *	The events trapped here are GtkWidget events, not signals.
 * BUGS
 * SEE ALSO
 * SOURCE
 */

int gnoclOptOnUndoRedo ( Tcl_Interp *interp, GnoclOption *opt, GObject *obj, Tcl_Obj **ret )
{
	assert ( *opt->propName == 'U' || *opt->propName == 'R' );
	return gnoclConnectOptCmd ( interp, obj, *opt->propName == 'U' ?  "undo" : "redo", G_CALLBACK ( doOnUndoRedo ), opt, NULL, ret );

}

/*****/

/****f* text/scrollToMark
 * NAME
 *	scrollToMark
 * PURPOSE
 * AUTHOR
 *	PGB
 *	Modifications by WJG to include suuport for marks.
 * CREATION DATE
 * USAGE
 * ARGUMENTS
 *	GtkTextView *view,			pointer to the GtkTextView object
 * 	GtkTextBuffer *buffer,		pointer to the GtkTextBuffer itself
 * 	Tcl_Interp *interp,			pointer to Tcl interpreter for error messaging etc.,
 * 	int objc,					?
 * 	Tcl_Obj * const objv[]		?
 * RETURN VALUE
 *	TCL_ERROR
 *	TCL_OK
 * NOTES
 * ERRORS
 * BUGS
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 *	WJG 02/08/08
 * SOURCE
 */
static int scrollToMark ( GtkTextView *view, GtkTextBuffer *buffer, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[] )
{

	printf ( "scrollToMark 1>  %s \n", Tcl_GetString ( objv[2] ) );


	GtkTextMark *mark;

	if ( objc < 3 )
	{
		Tcl_WrongNumArgs ( interp, 2, objv, "index ?-option val ...?" );
		return TCL_ERROR;
	}

	/* check to see that the mark exists */
	mark = gtk_text_buffer_get_mark ( buffer, Tcl_GetString ( objv[2] ) );

	/* reposition */
	if ( mark == NULL )
	{
		Tcl_SetResult ( interp, "This mark does not exist.", TCL_STATIC );
		return TCL_ERROR;
	}

	gtk_text_view_scroll_mark_onscreen  ( view, mark );

	return TCL_OK;

}

/*****/


/****f* text/posToIter
 * NAME
 *	posToIter
 * PURPOSE
 *	Convert at text index in the form of {row col} into a GtkTextBuffer iter(ator).
 * AUTHOR
 *	PGB
 * CREATION DATE
 * 	2001-06:
 * USAGE
 *	   if( posToIter( interp, objv[2], buffer, &iter ) != TCL_OK ) return TCL_ERROR;
 * ARGUMENTS
 *
 *	Tcl_Interp		*interp		pointer to Tcl interpreter from which this funcition was called
 *	Tcl_Obj			*obj		parameter passed
 *	GtkTextBuffer	*buffer		the buffer of current text widget
 *	GtkTextIter		*iter		iterator which will recieve value
 * RETURN VALUE
 *	TCL_ERROR
 *	TCL_OK
 * NOTES
 * ERRORS
 *	(text_cursor_keyWords_test.tcl:13461): Gtk-WARNING **: Invalid text buffer iterator: either the iterator is uninitialized,
 *	or the characters/pixbufs/widgets in the buffer have been modified since the iterator was created.
 *	You must use marks, character numbers, or line numbers to preserve a position across buffer modifications.
 *	You can apply tags and insert marks without invalidating your iterators,
 *	but any mutation that affects 'indexable' buffer contents (contents that can be referred to by character offset)
 *	will invalidate all outstanding iterators
 *
 * Causes: moving the inter does not act on the buffer, once the iter has been moved, then it n
 *
 * TODO
 *	Include new keywords
 *		sentenceStart
 *		sentenceEnd
 *		-paragraphStart
 *		-paragraphEnd
 *		wordStart
 *		wordEnd
 * 		lineEnd
 * USES
 * USED BY
 *	applyTag
 *	scrollToPos
 *	gnoclTextCommand
 * MODIFICATION HISTORY
 *	2008-06-27	Began implementation of new keywords for text position. See TODO.
 * SOURCE
 */
static int posToIter ( Tcl_Interp *interp, Tcl_Obj *obj, GtkTextBuffer *buffer, GtkTextIter *iter )
{
	char errMsg[] = "Position must be eiter a list of row and column "
					"or a keyword plus offset";
	char errEndOffset[] = "offset to \"end\" must be negative";

	int len;

	/* error check the arguments passed to the function */

	if ( Tcl_ListObjLength ( interp, obj, &len ) != TCL_OK || len < 1 || len > 2 )
	{
		Tcl_SetResult ( interp, errMsg, TCL_STATIC );
		return TCL_ERROR;
	}

	/* this is right */

	if ( len == 2 )
	{
		int idx[2];
		int isEnd[2] = { 0, 0 };
		int k;

		for ( k = 0; k < 2; ++k )
		{
			Tcl_Obj *tp;

			if ( Tcl_ListObjIndex ( interp, obj, k, &tp ) != TCL_OK )
			{
				Tcl_SetResult ( interp, errMsg, TCL_STATIC );
				return TCL_ERROR;
			}

			if ( Tcl_GetIntFromObj ( NULL, tp, idx + k ) != TCL_OK )
			{
				char *txt = Tcl_GetString ( tp );

				if ( strncmp ( txt, "end", 3 ) == 0 )
				{
					if ( gnoclPosOffset ( interp, txt + 3, idx + k ) != TCL_OK )
						return TCL_ERROR;

					if ( idx[k] > 0 )
					{
						Tcl_SetResult ( interp, errEndOffset, TCL_STATIC );
						return TCL_ERROR;
					}

					isEnd[k] = 1;
				}

				else
				{
					Tcl_AppendResult ( interp, "unknown row or column index \"", txt, "\" must be integer or end plus offset" );
					return TCL_ERROR;
				}

			}
		}

		gtk_text_buffer_get_start_iter ( buffer, iter );

		if ( isEnd[0] )
		{
			gtk_text_iter_set_line ( iter, -1 );
			gtk_text_iter_backward_lines ( iter, -idx[0] );
		}

		else
			gtk_text_iter_set_line ( iter, idx[0] );

		if ( isEnd[0] )
		{
			gtk_text_iter_forward_to_line_end ( iter );
			gtk_text_iter_backward_chars ( iter, -idx[1] );
		}

		else
			gtk_text_iter_forward_chars ( iter, idx[1] );
	}

	else if ( Tcl_GetIntFromObj ( NULL, obj, &len ) == TCL_OK )
	{
		if ( len < 0 )
		{
			Tcl_SetResult ( interp, "character offset must be greater zero.", TCL_STATIC );
			return TCL_ERROR;
		}

		gtk_text_buffer_get_iter_at_offset ( buffer, iter, len );
	}

	else
	{
		const char *txt = Tcl_GetString ( obj );
		const char *last;
		int offset;

		/* get a fresh iterator, it may have already been altered due to a previous call */
		gtk_text_buffer_get_iter_at_mark ( buffer, iter, gtk_text_buffer_get_insert ( buffer ) );


		if ( strncmp ( txt, "end", 3 ) == 0 )
		{
			gtk_text_buffer_get_end_iter ( buffer, iter );
			last = txt + 3;
		}

		else if ( strncmp ( txt, "cursor", 6 ) == 0 )
		{
			last = txt + 6;
			gtk_text_buffer_get_iter_at_mark ( buffer, iter, gtk_text_buffer_get_insert ( buffer ) );
		}

		else if ( strncmp ( txt, "selectionStart", 14 ) == 0 )
		{
			GtkTextIter end;
			gtk_text_buffer_get_selection_bounds ( buffer, iter, &end );
			last = txt + 14;
		}

		else if ( strncmp ( txt, "selectionEnd", 12 ) == 0 )
		{
			GtkTextIter start;
			gtk_text_buffer_get_selection_bounds ( buffer, &start, iter );
			last = txt + 12;
		}

		else if ( strncmp ( txt, "wordStart", 9 ) == 0 )
		{
			/* get a fresh itersator, it may have already been altered */
			//gtk_text_buffer_get_iter_at_mark( buffer, iter, gtk_text_buffer_get_insert( buffer ) );
			gtk_text_iter_backward_word_start ( iter );
			last = txt + 9;
		}

		else if ( strncmp ( txt, "wordEnd", 7 ) == 0 )
		{
			/* get a fresh itersator, it may have already been altered */
			//gtk_text_buffer_get_iter_at_mark( buffer, iter, gtk_text_buffer_get_insert( buffer ) );
			gtk_text_iter_forward_word_end ( iter );
			last = txt + 7;
		}

		/* WJG CURRENTLY WORKING HERE. Nothing happening, also text insert now kaput! */

		else if ( strncmp ( txt, "sentenceStart", 13 ) == 0 )
		{
			/* get a fresh itersator, it may have already been altered */
			//gtk_text_buffer_get_iter_at_mark( buffer, iter, gtk_text_buffer_get_insert( buffer ) );
			gtk_text_iter_backward_sentence_start ( iter );
			last = txt + 13;
		}

		else if ( strncmp ( txt, "sentenceEnd", 11 ) == 0 )
		{
			/* get a fresh itersator, it may have already been altered */
			//gtk_text_buffer_get_iter_at_mark( buffer, iter, gtk_text_buffer_get_insert( buffer ) );
			gtk_text_iter_forward_sentence_end ( iter );
			last = txt + 11;
		}

		else if ( strncmp ( txt, "lineStart", 9 ) == 0 )
		{
			/* move iterator to an offset of 0 */
			/* get a fresh itersator, it may have already been altered */
			//gtk_text_buffer_get_iter_at_mark( buffer, iter, gtk_text_buffer_get_insert( buffer ) );
			gtk_text_iter_forward_sentence_end ( iter );
			last = txt + 9;
		}

		else if ( strncmp ( txt, "lineEnd", 7 ) == 0 )
		{
			/* move iterator to the start of the next line, then move back one offset */
			/* get a fresh itersator, it may have already been altered */
			//gtk_text_buffer_get_iter_at_mark( buffer, iter, gtk_text_buffer_get_insert( buffer ) );
			gtk_text_iter_forward_sentence_end ( iter );
			last = txt + 7;
		}

		else
		{
			Tcl_AppendResult ( interp, "unknown index \"", txt,
							   "\", must be a list of row and column, "
							   "an integer as character offset, "
							   "or one of end, cursor, wordStart, wordEnd, sentenceStart, sentenceEnd, lineStart, lineEnd, selectionStart, or selectionEnd",
							   NULL );
			return TCL_ERROR;
		}

		if ( gnoclPosOffset ( interp, last, &offset ) != TCL_OK )
			return TCL_ERROR;

		if ( offset > 0 )
			gtk_text_iter_forward_chars ( iter, offset );
		else if ( offset < 0 )
			gtk_text_iter_backward_chars ( iter, -offset );
	}

	return TCL_OK;
}

/*****/


/****f* text/configure
 * NAME
 *	configure
 * PURPOSE
 * AUTHOR
 * CREATION DATE
 * USAGE
 * ARGUMENTS
 * RETURN VALUE
 *	TCL_ERROR
 *	TCL_OK
 * NOTES
 * 	taken from text.c
 * 	All these operations effect the sourceView widget, commands effect the buffer!
 * ERRORS
 * BUGS
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
static int configure ( Tcl_Interp *interp, GtkScrolledWindow *scrolled, GtkTextView *text, GnoclOption options[] )
{
	if ( options[scrollBarIdx].status == GNOCL_STATUS_CHANGED )
	{
		GtkPolicyType hor, vert;

		if ( gnoclGetScrollbarPolicy ( interp, options[scrollBarIdx].val.obj, &hor, &vert ) != TCL_OK )
		{
			return TCL_ERROR;
		}

		gtk_scrolled_window_set_policy ( scrolled, hor, vert );
	}

	return TCL_OK;
}

/*****/


/****f*  sourceview/gnoclSourceViewCmd
 * NAME
 * AUTHOR
 * 	WJG
 * SYNOPSIS
 * 	Sets the right margin line on the sourceView view.
 * FUNCTION
 * NOTES
 * BUGS
 * SEE ALSO
 * SOURCE
 */
int gnoclOptRightMargin ( Tcl_Interp *interp, GnoclOption *opt, GObject *obj, Tcl_Obj **ret )
{

	if ( 1 )
	{
		gtk_source_view_set_show_right_margin ( obj , 1 );

		return TCL_OK;
	}

	return TCL_ERROR;
}

/*****/

/****v* sourceview/sourceViewOptions
 * NAME
 * SYNOPSIS
 * FUNCTION
 * NOTES
 * 	As long as the options for the GtkScrolledWindow are not set
   	automatically, we don't need any special handling in gnoclSetOptions.
 * BUGS
 * SEE ALSO
 * SOURCE
*/
static GnoclOption sourceViewOptions[] =
{
	/* sourceView properties */
	{ "-autoIdent", GNOCL_BOOL, "auto-indent"},
	{ "-drawSpaces", GNOCL_STRING, "draw-spaces" },
	{ "-highlightCurrentLine", GNOCL_BOOL, "highlight-current-line" },
	{ "-indentOnTab", GNOCL_BOOL, "indent-on-tab" },
	{ "-indentWidth", GNOCL_INT, "indent-width" },
	{ "-insertSpace", GNOCL_BOOL, "insert-spaces-instead-of-tabs" },
	{ "-rightMargin", GNOCL_INT, "right-margin-position" },
	{ "-showLineMarks", GNOCL_BOOL, "show-line-marks" },
	{ "-showLineNumbers", GNOCL_BOOL, "show-line-numbers" },
	{ "-showRightMargin", GNOCL_BOOL, "show-right-margin" },
	{ "-smartHomeEnd", GNOCL_BOOL, "smart-home-end" },
	{ "-tabWidth", GNOCL_INT, "tab-width"  },

	/* set these options through commands */
	{ "-baseFont", GNOCL_OBJ, "Sans 14", gnoclOptGdkBaseFont },
	{ "-baseColor", GNOCL_OBJ, "normal", gnoclOptGdkColorBase },

	/* textView events */
	{ "-onButtonPress", GNOCL_OBJ, "P", gnoclOptOnButton },
	{ "-onButtonRelease", GNOCL_OBJ, "R", gnoclOptOnButton },
	{ "-onKeyPress", GNOCL_OBJ, "", gnoclOptOnKeyPress },
	{ "-onKeyRelease", GNOCL_OBJ, "", gnoclOptOnKeyRelease },
	{ "-onMotion", GNOCL_OBJ, "", gnoclOptOnMotion },

	/* sourceView signals */
	{ "-onUndo", GNOCL_OBJ, "U", gnoclOptOnUndoRedo },
	{ "-onRedo", GNOCL_OBJ, "R", gnoclOptOnUndoRedo },

	{ NULL }
};
/*****/

/****f* text/scrollToPos
 * NAME
 *	scrollToPos
 * PURPOSE
 * AUTHOR
 *	PGB
 *	Modifications by WJG to include suuport for marks.
 * CREATION DATE
 * USAGE
 * ARGUMENTS
 * RETURN VALUE
 *	TCL_ERROR
 *	TCL_OK
 * NOTES
 * ERRORS
 * BUGS
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 *	WJG 02/08/08
 * SOURCE
 */
static int scrollToPos ( GtkTextView *view, GtkTextBuffer *buffer, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[] )
{

	printf ( "scrollToPos 1>  %s \n",  Tcl_GetString ( objv[2] ) );

	GnoclOption options[] =
	{
		{ "-margin", GNOCL_DOUBLE, NULL },    /* 0 */
		{ "-align", GNOCL_OBJ, NULL },        /* 1 */
		{ NULL }
	};
	const int marginIdx = 0;
	const int alignIdx  = 1;

	int   ret = TCL_ERROR;

	double      margin = .0;
	int         useAlign = 0;
	gfloat      xAlign = 0.5, yAlign = 0.5;
	GtkTextIter iter;
	GtkTextMark *mark;


	if ( objc < 3 )
	{
		Tcl_WrongNumArgs ( interp, 2, objv, "index ?-option val ...?" );
		return TCL_ERROR;
	}

	if ( posToIter ( interp, objv[2], buffer, &iter ) != TCL_OK )
	{
		return TCL_ERROR;
	}

	if ( gnoclParseOptions ( interp, objc - 2, objv + 2, options ) != TCL_OK )
	{
		goto clearExit;
	}

	if ( options[alignIdx].status == GNOCL_STATUS_CHANGED )
	{
		if ( gnoclGetBothAlign ( interp, options[alignIdx].val.obj, &xAlign, &yAlign ) != TCL_OK )
		{
			goto clearExit;
		}

		useAlign = 1;
	}

	if ( options[marginIdx].status == GNOCL_STATUS_CHANGED )
	{
		margin = options[marginIdx].val.d;

		if ( margin < 0.0 || margin >= 0.5 )
		{
			Tcl_SetResult ( interp, "-margin must be between 0 and 0.5", TCL_STATIC );
			goto clearExit;
		}
	}


	mark = gtk_text_buffer_create_mark ( buffer, "__gnoclScrollMark__", &iter, 0 );


	gtk_text_view_scroll_to_mark ( view, mark, margin, useAlign, xAlign, yAlign );

	gtk_text_buffer_delete_mark ( buffer, mark );

	ret = TCL_OK;

clearExit:
	gnoclClearOptions ( options );
	return ret;
}

/*****/

/****f* text/textInsert
 * NAME
 *	textInsert
 * PURPOSE
 *	Insert formatted string into specified text widget at a given location.
 * AUTHOR
 *	PGB
 * CREATION DATE
 *	2001-06:
 * USAGE
 *	if( posToIter( interp, objv[2], buffer, &iter ) != TCL_OK ) return TCL_ERROR;
 * ARGUMENTS
 *	GtkTextView		*view
 *	GtkTextBuffer	*buffer
 *	Tcl_Interp		*interp
 *	int				objc
 *	Tcl_Obj			* const objv[]
 * RETURN VALUE
 *	TCL_ERROR
 *	TCL_OK
 * NOTES
 * ERRORS
 * BUGS
 *	Recent changes to posToIter causing this function to crash.
 *  But, applyTag, which is hack of this function does not cause a crash!
 * TODO
 * USES
 * 	posToIter
 * USED BY
 *	gnoclTextCommand
 * MODIFICATION HISTORY
 *	2008-06-27	Began implementation of new keywords for text position. See TODO.
 * SOURCE
 */
static int textInsert ( GtkTextBuffer *buffer, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[], int cmdNo )
{
	GnoclOption insertOptions[] =
	{
		{ "-tags", GNOCL_LIST, NULL },
		{ NULL }
	};
	const int tagsIdx = 0;
	gint      startOffset;
	int       ret = TCL_ERROR;

	GtkTextIter   iter;

	if ( objc < cmdNo + 2 )
	{
		Tcl_WrongNumArgs ( interp, cmdNo, objv, "position text ?-option val ...?" );
		return TCL_ERROR;
	}


	if ( posToIter ( interp, objv[cmdNo], buffer, &iter ) != TCL_OK )
	{
		return TCL_ERROR;
	}

	if ( gnoclParseOptions ( interp, objc - cmdNo - 1, objv + cmdNo + 1, insertOptions ) != TCL_OK )
	{
		goto clearExit;
	}

	startOffset = gtk_text_iter_get_offset ( &iter );

	gtk_text_buffer_insert ( buffer, &iter, gnoclGetString ( objv[cmdNo+1] ), -1 );

	if ( insertOptions[tagsIdx].status == GNOCL_STATUS_CHANGED )
	{
		GtkTextIter start;
		int k, no;
		Tcl_Obj *obj = insertOptions[tagsIdx].val.obj;

		gtk_text_buffer_get_iter_at_offset ( buffer, &start, startOffset );

		if ( Tcl_ListObjLength ( interp, obj, &no ) != TCL_OK )
		{
			goto clearExit;
		}

		for ( k = 0; k < no; ++k )
		{
			Tcl_Obj *tp;

			if ( Tcl_ListObjIndex ( interp, obj, k, &tp ) != TCL_OK )
			{
				Tcl_SetResult ( interp, "Could not read tag list", TCL_STATIC );
				goto clearExit;
			}

			gtk_text_buffer_apply_tag_by_name ( buffer, Tcl_GetString ( tp ), &start, &iter );
		}
	}

	ret = TCL_OK;

clearExit:
	gnoclClearOptions ( insertOptions );

	return ret;
}

/*****/

/****f* sourceview/sourceViewFunc
 * AUTHOR
 *	WJG, based upon original code created by Peter G. Baum
 * FUNCTION
 *  Widget creation function.
 * NOTES
 * 	This function effects the sourceBuffer and not the view!
 * SOURCE
 */

// int sourceViewFunc ( GtkTextBuffer *buffer, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[], int cmdNo, int isTextWidget )

int sourceViewFunc ( ClientData data, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[] )
{
	GtkScrolledWindow	*scrolled = GTK_SCROLLED_WINDOW ( data );
	GtkTextView			*text = GTK_TEXT_VIEW ( gtk_bin_get_child ( GTK_BIN ( scrolled ) ) );
	GtkTextBuffer		*buffer = gtk_text_view_get_buffer ( text );

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "command" );
		return TCL_ERROR;
	}

	switch ( gnoclSourceViewCommand ( buffer, interp, objc, objv, 1, 1 ) )
	{
		case 0:
			break;	/* return TCL_OK */
		case 1:   	/* delete */
			return gnoclDelete ( interp, GTK_WIDGET ( scrolled ), objc, objv );
		case 2:   	/* configure */
			{
				int ret = TCL_ERROR;

				if ( gnoclParseAndSetOptions ( interp, objc - 1, objv + 1, sourceViewOptions, G_OBJECT ( text ) ) == TCL_OK )
				{
					ret = configure ( interp, scrolled, text, sourceViewOptions );
				}

				gnoclClearOptions ( sourceViewOptions );

				return ret;
			}

			break;
		case 3: /* scrollToPosition */
			return scrollToPos ( text, buffer, interp, objc, objv );
		case 4: /* scrollToMark */
			return scrollToMark ( text, buffer, interp, objc, objv );
		default:
			return TCL_ERROR;
	}

	return TCL_OK;
}

/*****/

/****f* text/gnoclTextCommand
 * NAME
 * 	gnoclTextCommand
 * PURPOSE
 * AUTHOR
 * CREATION DATE
 * USAGE
 * ARGUMENTS
 * RETURN VALUE
 *	TCL_ERROR
 *	TCL_OK
 * NOTES
 * ERRORS
 * BUGS
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 *	21/07/08	WJG		Added emitSignal command
 * SOURCE
 */
int gnoclSourceViewCommand ( GtkTextBuffer *buffer, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[], int cmdNo, int isTextWidget )
{
	printf ( "sourceViewFunc\n" );

	const char *cmds[] = { "class", "delete", "configure", "scrollToPosition",  "scrollToMark",
						   "erase", "select", "get", "cut", "copy", "paste", "cget", "getLineCount", "getWordLength",
						   "getLength", "getLineLength", "setCursor", "getCursor", "insert", "tag", "dump", "XYgetCursor",
						   "signalEmit", "mark", "gotoWordStart", "gotoWordEnd", "search",
						   NULL
						 };
	enum cmdIdx { ClassIdx, DeleteIdx, ConfigureIdx, ScrollToPosIdx, ScrollToMarkIdx,
				  EraseIdx, SelectIdx, GetIdx, CutIdx, CopyIdx, PasteIdx, CgetIdx, GetLineCountIdx, GetWordLengthIdx,
				  GetLengthIdx, GetLineLengthIdx, SetCursorIdx, GetCursorIdx, InsertIdx, TagIdx, DumpIdx, XYgetCursorIdx,
				  SignalEmitIdx, MarkIdx, GotoWordStartIdx, GotoWordEndIdx, SearchIdx
				};

	int idx;

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "command" );
		{
			return TCL_ERROR;
		}
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmds, "command", TCL_EXACT, &idx ) != TCL_OK )
	{
		return TCL_ERROR;
	}


	switch ( idx )
	{
		case SearchIdx:

			printf ( "SearchIDx 0 = %s 1 = %s 2 = %s \n",
					 Tcl_GetString ( objv[cmdNo+0] ) ,
					 Tcl_GetString ( objv[cmdNo+1] ) ,
					 Tcl_GetString ( objv[cmdNo+2] ) );


			break;
		case InsertIdx:
			printf ( "SearchIDx 0 = %s 1 = %s 2 = %s \n",
					 Tcl_GetString ( objv[cmdNo+0] ) ,
					 Tcl_GetString ( objv[cmdNo+1] ) ,
					 Tcl_GetString ( objv[cmdNo+2] ) );


			/*
			 * GET A DIRECT INPUT OF TEXT...
			 */

			// gtk_text_buffer_set_text (GTK_TEXT_BUFFER( buffer ), "Your 1st GtkSourceView widget!", -1);

			if ( textInsert ( buffer, interp, objc, objv, cmdNo + 1 ) != TCL_OK )
			{
				return -1;
			}

			/* this action needs to emit an insert text signal */

			break;

		default:
			assert ( 0 );

			return -1;
	}

	// return 0;

	return TCL_OK;
}

/*****/

/****f* sourceview/gnoclSourceViewCmd
 * AUTHOR
 *	PGB
 * FUNCTION
 *  Widget cretion function.
 * SOURCE
 */
int gnoclSourceViewCmd ( ClientData data, Tcl_Interp *interp, int objc, Tcl_Obj * const objv[]  )
{
	int               ret;
	GtkWidget*  sourceView;
	GtkScrolledWindow *scrolled;

	printf ( "gnoclSourceViewCmd -1\n" );

	if ( gnoclParseOptions ( interp, objc, objv, sourceViewOptions ) != TCL_OK )
	{
		gnoclClearOptions ( sourceViewOptions );
		return TCL_ERROR;
	}

	sourceView = gtk_source_view_new ();

	if ( 0 )
	{
		gtk_source_view_set_show_line_numbers ( sourceView, TRUE );
		gtk_source_view_set_highlight_current_line ( sourceView, TRUE );
	}

	scrolled =  GTK_SCROLLED_WINDOW ( gtk_scrolled_window_new ( NULL, NULL ) );

	gtk_scrolled_window_set_policy ( scrolled, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC );
	gtk_container_add ( GTK_CONTAINER ( scrolled ), GTK_WIDGET ( sourceView ) );
	gtk_widget_show_all ( GTK_WIDGET ( scrolled ) );

	ret = gnoclSetOptions ( interp, sourceViewOptions, G_OBJECT ( sourceView ), -1 );

	if ( ret == TCL_OK )
	{
		ret = configure ( interp, scrolled, sourceView, sourceViewOptions );
	}

	gnoclClearOptions ( sourceViewOptions );


	if ( ret != TCL_OK )
	{
		gtk_widget_destroy ( GTK_WIDGET ( scrolled ) );
		return TCL_ERROR;
	}

	return gnoclRegisterWidget ( interp, GTK_WIDGET ( scrolled ), sourceViewFunc );
}

/*****/

