#!/usr/bin/env python
#
# Copyright 2007 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#




"""Base class for implementing API proxy stubs."""







from __future__ import with_statement



import os
import threading
import urllib

from google.appengine.api import apiproxy_rpc
from google.appengine.runtime import apiproxy_errors


MAX_REQUEST_SIZE = 1 << 20


class RequestData(object):
  """Allows stubs to lookup state linked to the request making the API call."""

  def get_request_url(self, request_id):
    """Returns the URL the request e.g. 'http://localhost:8080/foo?bar=baz'.

    Args:
      request_id: The string id of the request making the API call.

    Returns:
      The URL of the request as a string.
    """
    raise NotImplementedError()


class _LocalRequestData(RequestData):
  """Lookup information about a request using environment variables."""

  def get_request_url(self, request_id):
    """Returns the URL the request e.g. 'http://localhost:8080/foo?bar=baz'.

    Args:
      request_id: The string id of the request making the API call.

    Returns:
      The URL of the request as a string.
    """
    try:
      host = os.environ['HTTP_HOST']
    except KeyError:
      host = os.environ['SERVER_NAME']
      port = os.environ['SERVER_PORT']
      if port != '80':
        host += ':' + port
    url = 'http://' + host
    url += urllib.quote(os.environ.get('PATH_INFO', '/'))
    if os.environ.get('QUERY_STRING'):
      url += '?' + os.environ['QUERY_STRING']
    return url


_local_request_data = _LocalRequestData()


class APIProxyStub(object):
  """Base class for implementing API proxy stub classes.

  To implement an API proxy stub:
    - Extend this class.
    - Override __init__ to pass in appropriate default service name.
    - Implement service methods as _Dynamic_<method>(request, response).
  """



  _ACCEPTS_REQUEST_ID = False

  def __init__(self, service_name, max_request_size=MAX_REQUEST_SIZE,
               request_data=None):
    """Constructor.

    Args:
      service_name: Service name expected for all calls.
      max_request_size: int, maximum allowable size of the incoming request.  A
        apiproxy_errors.RequestTooLargeError will be raised if the inbound
        request exceeds this size.  Default is 1 MB.
      request_data: A RequestData instance used to look up state associated with
          the request that generated an API call.
    """
    self.__service_name = service_name
    self.__max_request_size = max_request_size
    self.request_data = request_data or _local_request_data



    self._mutex = threading.RLock()
    self.__error = None

  def CreateRPC(self):
    """Creates RPC object instance.

    Returns:
      a instance of RPC.
    """
    return apiproxy_rpc.RPC(stub=self)

  def MakeSyncCall(self, service, call, request, response, request_id=None):
    """The main RPC entry point.

    Args:
      service: Must be name as provided to service_name of constructor.
      call: A string representing the rpc to make.  Must be part of
        the underlying services methods and impemented by _Dynamic_<call>.
      request: A protocol buffer of the type corresponding to 'call'.
      response: A protocol buffer of the type corresponding to 'call'.
      request_id: A unique string identifying the request associated with the
          API call.
    """
    assert service == self.__service_name, ('Expected "%s" service name, '
                                            'was "%s"' % (self.__service_name,
                                                          service))
    if request.ByteSize() > self.__max_request_size:
      raise apiproxy_errors.RequestTooLargeError(
          'The request to API call %s.%s() was too large.' % (service, call))
    messages = []
    assert request.IsInitialized(messages), messages

    if self.__error:


      raise self.__error
    else:
      method = getattr(self, '_Dynamic_' + call)
      if self._ACCEPTS_REQUEST_ID:
        method(request, response, request_id)
      else:
        method(request, response)

  def SetError(self, error):
    """Set an error condition that is always raised when calls made to stub.

    Args:
      error: An instance of apiproxy_errors.Error or None for no error.
    """
    assert error is None or isinstance(error, apiproxy_errors.Error)
    self.__error = error


def Synchronized(method):
  """Decorator to acquire a mutex around an APIProxyStub method.

  Args:
    method: An unbound method of APIProxyStub or a subclass.

  Returns:
    The method, altered such it acquires self._mutex throughout its execution.
  """

  def WrappedMethod(self, *args, **kwargs):
    with self._mutex:
      return method(self, *args, **kwargs)

  return WrappedMethod
