
/***************************************************/
/*                                                 */
/* output.c-                                       */
/*                                                 */
/* HLA Code Generation functions are in this file. */
/* (iterim note: actually, the functions are split */
/* between this file and codegen.c;  as clean-up   */
/* progresses I will move files from codegen.c to  */
/* this file).                                     */
/*                                                 */
/***************************************************/

/*
** Must include debug.h first!
*/

#include "debug.h"

/*
** Note that this file must include "common.h" before
** "hla.h".  hla.h is generated automatically by bison,
** and it uses symbols defined in "common.h", so the
** following sequence must be maintained manually.
*/


#include "common.h"

#ifdef DEBUGUTILS
#define PathLogging
/*#define TestStmtCoverage*/
#define LogFuncCalls
#define DEBUG
#endif

#include "ratc.h"


#include "common.h"
#include "symbol.h"
#include "hlaparse.h"
#include "output.h"
#include "asm.h"
#include "version.h"




#define isFasm(t,f) _ifx(assembler==fasm,t,f)
#define isFGM(f,g,m) _ifx( assembler==fasm, f, _ifx(assembler==gas,g,m))

extern char* SizeMismatchStr;


struct bpList_t
{
	struct	bpList_t	*Next;
			char		*bpatch;
};

struct bpList_t *bpList = NULL;

static	struct	extRecs	*extHashTable[2048];


void PrintString( unsigned char *s, int zeroTerminate );


/*
** Register maps - maps register numbers to output strings.
*/


char *gpregmap[ numGPregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas	   ?? 
		{	"al", 	"al",	"al",	"al",  "" },
		{	"cl", 	"cl",	"cl",	"cl",  "" },
		{	"dl", 	"dl",	"dl",	"dl",  "" },
		{	"bl", 	"bl",	"bl",	"bl",  "" },

		{	"ah", 	"ah",	"ah",	"ah",  "" },
		{	"ch", 	"ch",	"ch",	"ch",  "" },
		{	"dh", 	"dh",	"dh",	"dh",  "" },
		{	"bh", 	"bh",	"bh",	"bh",  "" },

		{	"ax", 	"ax",	"ax",	"ax",  "" },
		{	"cx", 	"cx",	"cx",	"cx",  "" },
		{	"dx", 	"dx",	"dx",	"dx",  "" },
		{	"bx", 	"bx",	"bx",	"bx",  "" },

		{	"sp", 	"sp",	"sp",	"sp",  "" },
		{	"bp", 	"bp",	"bp",	"bp",  "" },
		{	"si", 	"si",	"si",	"si",  "" },
		{	"di", 	"di",	"di",	"di",  "" },

		{	"eax", 	"eax",	"eax",	"eax", "" },
		{	"ecx", 	"ecx",	"ecx",	"ecx", "" },
		{	"edx", 	"edx",	"edx",	"edx", "" },
		{	"ebx", 	"ebx",	"ebx",	"ebx", "" },
		{	"esp", 	"esp",	"esp",	"esp", "" },
		{	"ebp", 	"ebp",	"ebp",	"ebp", "" },
		{	"esi", 	"esi",	"esi",	"esi", "" },
		{	"edi", 	"edi",	"edi",	"edi", "" }
		
	}; 


char *fpregmap[ numFPregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas		?? 
		{	"st(0)","ST0",	"st(0)","st(0)","" },
		{	"st(1)","ST1",	"st(1)","st(1)","" },
		{	"st(2)","ST2",	"st(2)","st(2)","" },
		{	"st(3)","ST3",	"st(3)","st(3)","" },
		{	"st(4)","ST4",	"st(4)","st(4)","" },
		{	"st(5)","ST5",	"st(5)","st(5)","" },
		{	"st(6)","ST6",	"st(6)","st(6)","" },
		{	"st(7)","ST7",	"st(7)","st(7)","" }
		
	}; 


char *mmxregmap[ numMMXregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas		?? 
		{	"mm0",	"mm0",	"mm0",	"mm0",	"" },
		{	"mm1",	"mm1",	"mm1",	"mm1",	"" },
		{	"mm2",	"mm2",	"mm2",	"mm2",	"" },
		{	"mm3",	"mm3",	"mm3",	"mm3",	"" },
		{	"mm4",	"mm4",	"mm4",	"mm4",	"" },
		{	"mm5",	"mm5",	"mm5",	"mm5",	"" },
		{	"mm6",	"mm6",	"mm6",	"mm6",	"" },
		{	"mm7",	"mm7",	"mm7",	"mm7",	"" }
		
	}; 


char *xmmregmap[ numXMMregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas		?? 
		{	"xmm0",	"xmm0",	"xmm0",	"xmm0",	"" },
		{	"xmm1",	"xmm1",	"xmm1",	"xmm1",	"" },
		{	"xmm2",	"xmm2",	"xmm2",	"xmm2",	"" },
		{	"xmm3",	"xmm3",	"xmm3",	"xmm3",	"" },
		{	"xmm4",	"xmm4",	"xmm4",	"xmm4",	"" },
		{	"xmm5",	"xmm5",	"xmm5",	"xmm5",	"" },
		{	"xmm6",	"xmm6",	"xmm6",	"xmm6",	"" },
		{	"xmm7",	"xmm7",	"xmm7",	"xmm7",	"" }
		
	}; 


char *ctrlregmap[ numCTRLregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas		?? 
		{	"cr0",	"cr0",	"cr0",	"cr0",	"" },
		{	"cr1",	"cr1",	"cr1",	"cr1",	"" },
		{	"cr2",	"cr2",	"cr2",	"cr2",	"" },
		{	"cr3",	"cr3",	"cr3",	"cr3",	"" },
		{	"cr4",	"cr4",	"cr4",	"cr4",	"" },
		{	"cr5",	"cr5",	"cr5",	"cr5",	"" },
		{	"cr6",	"cr6",	"cr6",	"cr6",	"" },
		{	"cr7",	"cr7",	"cr7",	"cr7",	"" }
		
	}; 


char *dbgregmap[ numDBGregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas		?? 
		{	"dr0",	"dr0",	"dr0",	"dr0",	"" },
		{	"dr1",	"dr1",	"dr1",	"dr1",	"" },
		{	"dr2",	"dr2",	"dr2",	"dr2",	"" },
		{	"dr3",	"dr3",	"dr3",	"dr3",	"" },
		{	"dr4",	"dr4",	"dr4",	"dr4",	"" },
		{	"dr5",	"dr5",	"dr5",	"dr5",	"" },
		{	"dr6",	"dr6",	"dr6",	"dr6",	"" },
		{	"dr7",	"dr7",	"dr7",	"dr7",	"" }
		
	}; 


char *segregmap[ numSEGregs ][ numAssemblers ] =
	{
		//	masm	fasm	tasm	gas		?? 
		{	"es",	"es",	"es",	"es",	"" },
		{	"cs",	"cs",	"cs",	"cs",	"" },
		{	"ss",	"ss",	"ss",	"ss",	"" },
		{	"ds",	"ds",	"ds",	"ds",	"" },
		{	"fs",	"fs",	"fs",	"fs",	"" },
		{	"gs",	"gs",	"gs",	"gs",	"" }
	}; 



// Data Declaration Directives:

char *byteDrctv[ numAssemblers ] =
		//	masm	fasm	tasm	gas		 ?? 
		{	"db",	"db",	"db",	".byte", "" };

char *wordDrctv[ numAssemblers ] =
		//	masm	fasm	tasm	gas		 ?? 
		{	"dw",	"dw",	"dw",	".word", "" };

char *dwordDrctv[ numAssemblers ] =
		//	masm	fasm	tasm	gas		 ?? 
		{	"dd",	"dd",	"dd",	".long", "" };





/*
** ptype2type-
**	A pType is an index into this table, returning the base type.
*/

struct SymNode *ptype2type[] =
{
	&boolean_ste,   //  tBoolean,	//0 
	&byte_ste,	    //  tEnum,		//1
	&uns8_ste,	    //  tUns8,		//2
	&uns16_ste,	    //  tUns16,		//3
	&uns32_ste,	    //  tUns32,		//4
	&uns64_ste,	    //  tUns64,		//5
	&uns128_ste,    //  tUns128,	//6
	&byte_ste,	    //  tByte,		//7
	&word_ste,	    //  tWord,		//8
	&dword_ste,	    //  tDWord,		//9
	&qword_ste,	    //  tQWord,		//10
	&tbyte_ste,	    //  tTByte,		//11
	&lword_ste,	    //  tLWord,		//12
	&int8_ste,	    //  tInt8,		//13
	&int16_ste,	    //  tInt16,		//14
	&int32_ste,	    //  tInt32,		//15
	&int64_ste,	    //  tInt64,		//16
	&int128_ste,    //  tInt128,	//17
	&char_ste, 	    //  tChar, 		//18
	&wchar_ste,	    //  tWChar,		//19
	&real32_ste,    //  tReal32,	//20
	&real64_ste,    //  tReal64, 	//21
	&real80_ste,    //  tReal80,	//22
	&real128_ste,   //  tReal128,	//23
	&string_ste,    //  tString,	//24
	&zstring_ste,   //  tZString,	//25
	&wstring_ste,   //  tWString,	//26
	&cset_ste,	    //  tCset,		//27
	&byte_ste,	    //  tArray,		//28
	&byte_ste,	    //  tRecord,	//29
	&byte_ste,	    //  tUnion,		//30
	&regex_ste,	    //  tRegex,		//31
	&byte_ste,	    //  tClass,		//32
	&byte_ste,	    //  tProcptr,	//33
	&thunk_ste,	    //  tThunk,		//34
	&pointer_ste    //  tPointer	//35
};

char *ptype2str[] =
	{
		"Boolean",
		"Enum",
		
		"Uns8",
		"Uns16",
		"Uns32",
		"Uns64",
		"Uns128",
		
		"Byte",
		"Word",
		"DWord",
		"QWord",
		"TByte",
		"LWord",

		"Int8",
		"Int16",
		"Int32",
		"Int64",
		"Int128",

		"Char",
		"WChar",

		"Real32",
		"Real64",
		"Real80",
		"Real128",
		 
		"String",
		"ZString",
		"WString",
		"Cset",

		"Array",
		"Record",
		"Union",
		"Regex",
		"Class",
		"Procptr",
		"Thunk",
		"Pointer",

		"Label",
		"Proc",
		"Method",
		"ClassProc",
		"ClassIter",
		"Iterator",
		"Program",
		"Macro",
		"Text",
		"RegexMac",
					
		"Namespace",
		"Segment",
		"AnonRec",
		"AnonUnion",
		"Variant",
		"Error"
	};	

/*
** The following function exists simply to force
** the compiler to link in a floating point version
** of printf.
*/

static void ForceFPLink( void )
_begin( ForceFPLink )

	long double f;
	
	printf( "%LE", f );

_end( ForceFPLink )


/*********************************************************/
/*                                                       */
/* asmPrintf-                                            */
/* Formats and writes a "printf-style" string to asmBuf. */
/*                                                       */
/*********************************************************/

int
asmPrintf( char *fmtstr, ...)
_begin( asmPrintf )

	int	result;
	va_list	ap;
	
	va_start( ap, fmtstr );
	_if( asmBuf->offset + 1024 >= asmBuf->size )
	
		asmBuf->size *= 2;
		asmBuf->base = realloc( asmBuf->base, asmBuf->size );
		assert( asmBuf->base != NULL );
		
	_endif
	result = vsprintf( asmBuf->base + asmBuf->offset, fmtstr, ap );
	asmBuf->offset += result;
	va_end( ap );
	_return result;

_end( asmPrintf )


void
asmPuts( char *s )
_begin( asmPuts )

	int	result;
	int len = strlen(s);
	
	_if( asmBuf->offset + len + 2 >= asmBuf->size )
	
		asmBuf->size *= 2;
		asmBuf->base = realloc( asmBuf->base, asmBuf->size );
		assert( asmBuf->base != NULL );
		
	_endif
	memcpy( asmBuf->base + asmBuf->offset, s, len );
	asmBuf->offset += len;

_end( asmPuts )


void
asmCpy( char *s, int len )
_begin( asmCpy )

	int	result;
	
	_if( asmBuf->offset + len + 2 >= asmBuf->size )
	
		asmBuf->size += asmBuf->size + len + 2;
		asmBuf->base = realloc( asmBuf->base, asmBuf->size );
		assert( asmBuf->base != NULL );
		
	_endif
	memcpy( asmBuf->base + asmBuf->offset, s, len );
	asmBuf->offset += len;

_end( asmCpy )


void
asmPutc( char c )
_begin( asmPutc )

	int	result;
	
	_if( asmBuf->offset + 2 >= asmBuf->size )
	
		asmBuf->size *= 2;
		asmBuf->base = realloc( asmBuf->base, asmBuf->size );
		assert( asmBuf->base != NULL );
		
	_endif
	asmBuf->base[ asmBuf->offset ] = c;
	++asmBuf->offset;

_end( asmPutc )



/*************************************************************/
/*                                                           */
/* CheckStatic-                                              */
/*                                                           */
/* This function scans through the "StaticList" linked list  */
/* to ensure that all statics used in the current program    */
/* unit have been properly declared before the end of the    */
/* current declaration section.                              */
/*                                                           */
/*************************************************************/

void
CheckStatic
( 
	struct StaticListType *StaticList,
	int	   MainPgm 
)
_begin( CheckStatic )

	char					msg[ 256 ];
	struct	SymNode			*StaticSymEntry;
	struct	SymNode			*SaveSymTbl = SymbolTable;

	_while( StaticList != NULL )

		assert( StaticList->Name != NULL );
		assert( StaticList->StaticName != NULL );

		_if( !StaticList->Fixed )

			assert( StaticList->Context != NULL );

			_if
			( 
					StaticList->DefinedSym != NULL 
				&&	StaticList->DefinedSym->StaticName != NULL
			)

				EmitBackPatchss
				(
					StaticList->StaticName,
					StaticList->DefinedSym->StaticName
				);
				StaticList->Fixed = 1;

			_else

				StaticSymEntry = 
					lookupin
					( 
						StaticList->Name, 
						StaticList->Context->u.proc.Locals 
					);

				_if( StaticSymEntry == NULL &&	MainPgm )

					sprintf
					( 
						msg, 
						"Object \"%s\" was used as an address expression in\n"
						"the statement at line %d, it must be defined within\n"
						"the scope of the object",
						StaticList->Name,
						StaticList->LineNumber
					);
					yyerror( msg );

				_elseif
				( 
						StaticSymEntry != NULL 
					&&	StaticSymEntry->StaticName == NULL 
				)

					sprintf
					(
						msg,
						"%s is not a static object",
						StaticList->Name
					);
					yyerror( msg );

				_elseif
				( 
						StaticSymEntry != NULL 
					&&	StaticSymEntry->StaticName != NULL 
				)

					EmitBackPatchss
					(
						StaticList->StaticName,
						StaticSymEntry->StaticName
					);
					StaticList->Fixed = 1;
								
				_endif

			_endif

		_endif
		StaticList = StaticList->Next;

	_endwhile
	SymbolTable = SaveSymTbl;

_end( CheckStatic )



/***********************************************************/
/*                                                         */
/* Add2PtrList-                                            */
/*                                                         */
/* We've got some sort of pointer declaration and the base */
/* type is currently undefined. So add an entry to the     */
/* "pointer list" that tracks such things.                 */
/*                                                         */
/***********************************************************/

void
Add2PtrList
( 
	struct SymNode *reference, 
	char *undefdID 
)
_begin( Add2PtrList )



	struct	PointerListType	*ptrList;
	struct	PointerListType **lastPtr;
	struct	RefListType 	*ref;
	struct	RefListType		*r;
	struct	RefListType		*NewRef;

	ptrList = PointerList;
	_if( ptrList != NULL )

		/*
		** Find the symbol or the last node in
		** the list.
		*/

		lastPtr = &PointerList;
		_while
		( 
				ptrList != NULL 
			&&	_strne( ptrList->Name, undefdID ) 
		)

			lastPtr = &ptrList->Next;
			ptrList = ptrList->Next;

		_endwhile

		/*
		** if the list is not empty, there are two cases;
		** either ptrList is NULL and we have a new entry,
		** or ptrList is not NULL and we have an additional
		** occurrence of a pointer to this symbol.
		*/

		_if( ptrList == NULL )

			/*
			**	New entry to add to the Pointer List 
			*/

			ptrList = 
				malloc2( sizeof( struct PointerListType ));

			ref = 
				malloc2( sizeof( struct RefListType ));
			
			ptrList->Next = NULL;
			ptrList->ReferenceList = ref;
			ptrList->Name = hlastrdup( undefdID );

			*lastPtr = ptrList;

			ref->Next = NULL;
			ref->Symbol = reference;	// Was SymbolTable
			ref->LineNumber = LineCnt;
			ref->FileName = FileName;


		_else


			/*
			** This has already been referenced, so
			** add the current symbol table entry to
			** the end of the referenced list.
			*/

			r = ptrList->ReferenceList;
			_while( r->Next != NULL )

				r = r->Next;

			_endwhile
			NewRef =
				malloc2( sizeof( struct RefListType ));
			
			NewRef->Next = NULL;
			NewRef->Symbol = reference; // Was SymbolTable;
			NewRef->LineNumber = LineCnt;
			NewRef->FileName = FileName;
			r->Next = NewRef;

		_endif

	_else

		/*
		** if the list is empty, just add this
		** symbol to the list.
		*/

		ref = malloc2( sizeof( struct RefListType ));
		
		ref->Next = NULL;
		ref->Symbol = reference; // WasvSymbolTable;
		ref->LineNumber = LineCnt;
		ref->FileName = FileName;

		PointerList = 
			malloc2( sizeof( struct PointerListType ));

		PointerList->Next = NULL;
		PointerList->ReferenceList = ref;
		PointerList->Name = hlastrdup( undefdID );


	_endif

_end( Add2PtrList )




/*****************************************************
/*                                                  */
/* CheckPtrs-                                       */
/*                                                  */
/* Scans through the Pointer List to make sure that */
/* all forward-referenced static addresses assigned */
/* to pointers have been defined.                   */
/*                                                  */
/****************************************************/

void
CheckPtrs( void )
_begin( CheckPtrs )

	char					msg[ 256 ];
	struct	RefListType		*ref;
	struct	PointerListType	*FreeMe;
	struct	RefListType		*FreeRef;
	struct	SymNode			*s;
	struct	PatchListType	*lastPBL;

	_while( PointerList != NULL )

		ref = PointerList->ReferenceList;
		s = lookup( PointerList->Name, 1 );
		_if( s != NULL )

			/*
			** The symbol was defined, so go in and patch
			** all the references to this symbol.
			*/

			_while( ref != NULL )

				ref->Symbol->Base = s;
				FreeRef = ref;
				ref = ref->Next;
				free2( vss FreeRef );

			_endwhile

		_else

			/*
			** The symbol was never defined, so we have an error.
			*/

			_while( ref != NULL )

				/*
				** Patch this symbol with bogus information to prevent
				** cascading errors.
				*/

				ref->Symbol->Base = NULL;
				ref->Symbol->Type = &pointer_ste;
				ref->Symbol->pType = tPointer;

				/*
				** Print an error message.
				*/

				sprintf
				(
					msg,
					"Type \"%s\" was used as a pointer base type for\n"
					"object \"%s\" but was not declared within the\n"
					"current scope. Definition at line %d in '%s'",
					PointerList->Name,
					ref->Symbol->TrueName,
					ref->LineNumber,
					ref->FileName
				);
				yyerror( msg );

				/*
				** On to the next reference.
				*/


				FreeRef = ref;
				ref = ref->Next;
				free2( vss FreeRef );

			_endwhile

		_endif
		FreeMe = PointerList;
		PointerList = PointerList->Next;
		free2( vss FreeMe->Name );
		free2( vss FreeMe );

	_endwhile

	/*
	** Okay, go and patch all the variable entries in the PatchBaseList.
	*/

	_while( PatchBaseList != NULL )

		lastPBL = PatchBaseList;
		PatchBaseList->Symbol->Base = PatchBaseList->Symbol->Type->Base;
		PatchBaseList = PatchBaseList->Next;
		free2( vss lastPBL );

	_endwhile

_end( CheckPtrs )



// PatchPtrs- Just like CheckPtrs, except it only cleans up
// the pointer references that were found -- it doesn't report
// any errors if there are any undefined references.

void
PatchPtrs( void )
_begin( PatchPtrs )

	struct	RefListType		*ref;
	struct	PointerListType	*FreeMe;
	struct	PointerListType **prev;
	struct	PointerListType *cur;
	struct	RefListType		*FreeRef;
	struct	SymNode			*s;

	cur = PointerList;
	prev = &PointerList;
	_while( cur != NULL )

		ref = cur->ReferenceList;
		s = lookup( cur->Name, 1 );
		_if( s != NULL )

			/*
			** The symbol was defined, so go in and patch
			** all the references to this symbol.
			*/

			_while( ref != NULL )

				ref->Symbol->Base = s;
				FreeRef = ref;
				ref = ref->Next;
				free2( vss FreeRef );

			_endwhile
			
			*prev = cur->Next;
			FreeMe = cur;
			cur = cur->Next;
			free2( vss FreeMe->Name );
			free2( vss FreeMe );

		_else
		
			prev = &cur->Next;
			cur = cur->Next;
			
		_endif

	_endwhile


_end( PatchPtrs )



/***************************************************************/
/*                                                             */
/* CheckFwdRef-                                                */
/*                                                             */
/* This function scans through the FwdLabelsList linked list   */
/* searching for any forward label references.  If any exist,  */
/* this is an error because there is some jump to a label that */
/* has not been defined in the current procedure.              */
/*                                                             */
/***************************************************************/

void
CheckFwdRef( void )
_begin( CheckFwdRef )

	struct FwdRefLabelType	*flist;
	struct FwdRefLabelType	**prev;
	struct FwdRefLabelType	*f;
	struct SymNode			*s;
	

	flist = FwdLabelsList;
	prev = &FwdLabelsList;
	_while( flist != NULL )

		_if( flist->lexLevel > CurLexLevel )
		
			_if( !flist->isExternal )
			
				s = lookup( flist->label, 1 );
				_if( s == NULL )

					ErrorNear
					( 
						"Undefined statement label referenced "
						"in procedure body",
						flist->label,
						__LINE__,
						__FILE__
					);
					flist->isExternal = 1; // To prevent cascading errors.

				_else

					EmitBackPatchss
					(
						flist->StaticName,
						s->StaticName
					);

				_endif
				
			_endif
			free2( vss flist->label );
			free2( vss flist->StaticName );
			f = flist;
			*prev = flist->Next;	/* Skip over current struct in list */
			flist = flist->Next;
			free2( vss f );
		
		_else
		
			prev = &flist->Next;
			flist = flist->Next;
		_endif

	_endwhile
	
_end( CheckFwdRef )



/***********************************************************/
/*                                                         */
/* BuildAdrs-                                              */
/*                                                         */
/* Fills in an adrsYYS structure with the parameter values */
/*                                                         */
/***********************************************************/



void 
BuildAdrs
(
	struct	adrsYYS 	*adrs,
	unsigned			Size,
	unsigned			ObjectSize,
	char				*StaticName,
	char				*BaseReg,
	char				*IndexReg,
	unsigned			Scale,
	int					Disp,
	struct	SymNode		*Sym,
	struct	SymNode		*Type,
	enum	PrimType	pType,
	enum	ClassType	SymClass,
	enum	ParmClass	pClass,
	struct	SymNode		*BaseType
)
_begin( BuildAdrs )

	adrs->Size 			=  Size;
	adrs->ObjectSize 	=  ObjectSize;
	adrs->StaticName 	=  StaticName;
	adrs->BaseReg 		=  BaseReg;
	adrs->IndexReg 		=  IndexReg;
	adrs->Scale 		=  Scale;
	adrs->Disp 			=  Disp;
	adrs->Sym 			=  Sym;
	adrs->Type 			=  Type;
	adrs->pType 		=  pType;
	adrs->SymClass 		=  SymClass;
	adrs->pClass 		=  pClass;
	adrs->BaseType		=  BaseType;

_end( BuildAdrs )



/******************************************************************/
/*                                                                */
/* LabelToOfs-                                                    */
/*                                                                */
/* Given a string that is a label representing a memory address,  */
/* this function applies the appropriate operator (e.g. OFFSET32) */
/* to take the address of that object at assembly time.           */
/*                                                                */
/******************************************************************/

void
LabelToOfs( char *dest, char *src )
_begin( LabelToOfs )

	_if
	(
			assembler == masm
		||	assembler == tasm
	)

		strcpy( dest, "offset32 " );

	_elseif( assembler == gas )
	
		strcpy( dest, "offset " );
		
	_elseif( assembler == fasm )

		dest[0] = '\0';

	_else

		yyerror( "Internal HLA error (Bad assembler value)" );
		dest[0] = '\0';

	_endif;
	strcat( dest, src );

_end( LabelToOfs )





/*****************************************************/
/*                                                   */
/* LabelToMem-                                       */
/*                                                   */
/* This function is given a string representing the  */
/* name of a memory location and a string specifying */
/* the size of the memory object (byte, word, etc.). */
/* This function creates a string of the form:       */
/* "size PTR [label]" for use as a memory operand on */
/* output.                                           */
/*                                                   */
/* Note: If the caller passes in a 32-bit register,  */
/* this code creates a string of the form            */
/* "size PTR [reg32]" which is perfectly legal.      */
/*                                                   */
/*****************************************************/


void
LabelToMem( char *dest, char *src, char *size )
_begin( LabelToMem )

	_if( *src != '[' )
	
		strcpy( dest, size );
		strcat( dest, " ptr [" );
		strcat( dest, src );
		strcat( dest, "]" );
		
	_else
	
		strcpy( dest, size );
		strcat( dest, " ptr " );
		strcat( dest, src );
		
	_endif

_end( LabelToMem )


/**********************************************************/
/*                                                        */
/* LabelTomemPlusOfs-                                     */
/*                                                        */
/* Like the function above but also factors in an integer */
/* offset into the address expression.                    */
/*                                                        */
/**********************************************************/

void
LabelToMemPlusOfs( char *dest, char *src, char *size, int ofs )
_begin( LabelToMemPlusOfs )


	_if( ofs == 0 )
	
		// If the offset is zero, don't emit it to the
		// code stream because some assemblers (e.g., GAS) will
		// generate a byte displacement if you explicitly specify
		// an offset of zero.
		
		LabelToMem( dest, src, size );
	
	_else
	
		strcpy( dest, size );
		_if( *src != '[' )
		
			strcat( dest, " ptr [" );
			
		_else
		
			strcat( dest, " ptr " );
			
		_endif
		strcat( dest, src );
		sprintf
		( 
			dest+strlen( dest ) - (*src == '['), 
			"%s%d]",
			_ifx( ofs<0, "", "+"),
			ofs
		);
		
	_endif

_end( LabelToMemPlusOfs )








/******************************************************************/
/*                                                                */
/* CheckForwardDecls-                                             */
/*                                                                */
/* Scans through the symbol table up to the current procedure     */
/* we're processing to see if there are any symbol table entries  */
/* for a procedure that are forward declarations.  If we find     */
/* any, then the specified forward procedure has not been defined */
/* and this is an error.                                          */
/*                                                                */
/******************************************************************/


void 
CheckForwardDecls( struct SymNode *CurProc )
_begin( CheckForwardDecls )

	struct	SymNode	*CurSym;
	char	msg[ 256 ];

	CurSym = SymbolTable;

	assert( CurProc != NULL );
	_while( CurSym != CurProc && CurSym != NULL )

		_if
		( 
				CurSym->pType == tProc 
			&&	CurSym->u.proc.Forward != NULL 
			&&	!CurSym->IsExternal
		)

			sprintf
			( 
				msg,
				"%s is declared forward without a corresponding "
				"procedure definition",
				CurSym->TrueName
			);
			yyerror( msg );
			CurSym->u.proc.Forward = NULL;

		_endif
		CurSym = CurSym->Next;

	_endwhile

_end( CheckForwardDecls )






/*******************************************************/
/*                                                     */
/* CopyParms-                                          */
/*                                                     */
/* Used by CopySymbols to copy the parameter list of a */
/* procedure or method encountered while copying the   */
/* fields of a class.                                  */
/*                                                     */
/*******************************************************/

struct SymNode *recCopyParms( struct SymNode *toCopy );

void
CopyParms( struct SymNode *proc )
_begin( CopyParms )

	int				lexlevel;
	struct	SymNode	*plist;
	struct	SymNode	*bt;
	struct	SymNode	*_parms_;
	struct	SymNode	*symsave;
	struct	SymNode	value;

	assert( proc != NULL );
	bt = _ifx( proc->pType == tProcptr, GetBaseType( proc ), proc );
	assert( bt->u.proc.parms != NULL );

	plist= recCopyParms( bt->u.proc.parms );
	proc->u.proc.parms = plist;

	_if( proc->u.proc.Locals == NULL )
	
		value.u.intval = 0;
		lexlevel = 0;
	
	_else
	
		symsave = SymbolTable;
		SymbolTable = proc->u.proc.Locals;
		_parms_ = lookup( "_parms_", 1 );
		SymbolTable = symsave;
		_if( _parms_ != NULL )

			value.u.intval = _parms_->u.intval;
			lexlevel = _parms_->LexLevel;

		_else

			value.u.intval = 0;
			lexlevel = 0;

		_endif
		
	_endif
	InsertSym
	( 
		"_parms_", 
		&uns32_ste,
		tUns32, 
		cConstant, 
		0,
		NULL,
		0, 
		YYS &value,
		4,
		0,
		NULL,
		NULL,
		NULL,
		0
	);
	SymbolTable->LexLevel = lexlevel;


	proc->u.proc.Locals = SymbolTable;
	SymbolTable = SymbolTable->Next;
	proc->u.proc.Locals->Next = proc->u.proc.parms;
	
_end( CopyParms )



struct SymNode*
recCopyParms( struct SymNode *toCopy )
_begin( recCopyParms )

	struct SymNode *temp;
	struct SymNode *prevItem;
	
	assert( toCopy != NULL );

	prevItem = NULL;

	_if( toCopy->Next != NULL )

		prevItem = recCopyParms( toCopy->Next );
		temp = malloc2( sizeofSymNode );
		memcpy( temp, toCopy, sizeofSymNode );

	_else

		temp = malloc2( sizeofSymNode );
		memcpy( temp, toCopy, sizeofSymNode );

	_endif
	temp->Next = prevItem;
	_return temp;
		
_end( recCopyParms )


/*************************************************************/
/*                                                           */
/* CopySymbols-                                              */
/*                                                           */
/* This function makes a copy of a symbol table list.  It is */
/* primarily used to copy fields from a base class into the  */
/* class that inherits those fields.                         */
/*                                                           */
/*************************************************************/


struct SymNode*
CopySymbols( struct SymNode *SymbolsToCopy, char *VMTName )
_begin( CopySymbols )

	struct	SymNode	*temp;
	struct	SymNode	*NextItem;
	char			msg[ 256 ];
	struct	SymNode value;


	assert( VMTName != NULL );

	_if( SymbolsToCopy == NULL )

		struct SymNode *Save = SymbolTable;


		sprintf
		( 
			msg,
			"Internal error: SymbolsToCopy is NULL in %s at line %d",
			__FILE__,
			__LINE__,
			__FILE__
		);
		yyerror( msg );

		/*
		** Create a dummy entry to return to prevent
		** system crashes.
		*/

		value.u.unsval = 0;
		InsertSym
		( 
			"_dummy_", 
			&uns32_ste,
			tUns32, 
			cConstant, 
			0,
			NULL,
			0, 
			YYS &value,
			4,
			0,
			NULL,
			NULL,
			NULL,
			0
		);
		temp = SymbolTable;
		SymbolTable = Save;
		_return temp;

	_endif
	assert( SymbolsToCopy != NULL );

	temp = malloc2( sizeofSymNode );

	/*
	** Recursively call this function until we reach the
	** end of the list.  On return from the function, insert
	** the symbols into the symbol table.  It is important
	** to do it this way so that we insert the symbols into
	** the symbol table in the same order they were entered
	** for the original class.
	*/


	_if( _strne( SymbolsToCopy->TrueName, "_VMT_" ))

		NextItem = CopySymbols( SymbolsToCopy->Next, VMTName );


		/*
		** If this is a value object then do a deep copy.
		** This prevents modifications to this field in other
		** objects from affecting the value of this object.
		**
		** If it's not a value object, then we can do a shallow
		** copy because the underlying values cannot change.
		**
		** If this is a class object, we will also need to do
		** a deeper copy since the class object may contain
		** VALUE fields.
		*/

		_if( SymbolsToCopy->SymClass == cValue )

			DeepCopy( temp, SymbolsToCopy );
			temp->Next = NextItem;

		_elseif( SymbolsToCopy->pType == tClass )

			memcpy( temp, SymbolsToCopy, sizeofSymNode );
			temp->Next = NextItem;
			temp->Fields = CopySymbols( SymbolsToCopy->Fields, VMTName );
			NullTerminate( temp->Fields, SymbolTable );

		_elseif
		( 
				IsProc( SymbolsToCopy->pType ) 
			||	SymbolsToCopy->pType == tProcptr
		)

			memcpy( temp, SymbolsToCopy, sizeofSymNode );
			CopyParms( temp );
			temp->Next = NextItem;


		_else	  

			memcpy( temp, SymbolsToCopy, sizeofSymNode );
			temp->Next = NextItem;

		_endif

		// Must copy the StaticName field because we sometimes
		// change its value and we don't want to change the
		// original value.
		
		_if( temp->StaticName != NULL )
		
			temp->StaticName = hlastrdup2( temp->StaticName );
			
		_endif

	_else

		/*
		** If we get to this point, then the Next field contains NULL.
		** That means that we're looking at the VMT entry for this class.
		** We need to treat the VMT entry specially because we have to
		** attach a new static name to it.
		*/

		memcpy( temp, SymbolsToCopy, sizeofSymNode );
		temp->StaticName = hlastrdup( VMTName );
		temp->Next = SymbolTable;

	_endif
	_return temp;

_end( CopySymbols )




/*************************************************************/
/*                                                           */
/* CopyRecSymbols-                                           */
/*                                                           */
/* This function makes a copy of a symbol table list.  It is */
/* primarily used to copy fields from a base record into the */
/* record that inherits those fields.                        */
/*                                                           */
/*************************************************************/


struct SymNode*
CopyRecSymbols( struct SymNode *SymbolsToCopy )
_begin( CopyRecSymbols )

	struct	SymNode	*temp;
	struct	SymNode	*NextItem;

	/*
	** Recursively call this function until we reach the
	** end of the list.  On return from the function, insert
	** the symbols into the symbol table.  It is important
	** to do it this way so that we insert the symbols into
	** the symbol table in the same order they were entered
	** for the original record.
	*/


	_if( SymbolsToCopy != NULL )


		NextItem = CopyRecSymbols( SymbolsToCopy->Next );


		temp = malloc2( sizeofSymNode );

		memcpy( temp, SymbolsToCopy, sizeofSymNode );
		temp->Next = NextItem;


		// Must copy the StaticName field because we sometimes
		// change its value and we don't want to change the
		// original value.
		
		_if( temp->StaticName != NULL )
		
			temp->StaticName = hlastrdup2( temp->StaticName );
			
		_endif
		_return temp;


	_endif
	
	/*
	** Drop down here when we reach the end of the list.
	** Return NULL to mark the end of this process.
	*/
	
	_return NULL;
	
_end( CopyRecSymbols )




/***************************************************************/
/*                                                             */
/* NullTerminate-                                              */
/*                                                             */
/* Searches through the symbol table pointed at by "list"      */
/* until it hits the entry pointed at by "Last".  When this    */
/* occurs, this code sets the Next field of the entry pointing */
/* at "Last" to NULL.                                          */
/*                                                             */
/***************************************************************/

void 
NullTerminate( struct SymNode *list, struct SymNode *Last )
_begin( NullTerminate )

	assert( list != NULL );
	assert( list->Next != NULL );
	assert( Last != NULL );

	_while( list->Next != Last && list->Next != NULL )

		list = list->Next;

	_endwhile
	list->Next = NULL;

_end( NullTerminate )



/***************************************************************/
/*                                                             */
/* BuildVMT-                                                   */
/*                                                             */
/* The parameter contains a pointer to a newly created class.  */
/* This function traverses the fields of the class and locates */
/* all methods of that class.  For each method, this function  */
/* emits a pointer to the virtual method table and updates the */
/* offset field of the method entry to contain the offset into */
/* the VMT where a pointer to that method's code can be found. */
/*                                                             */
/***************************************************************/

/*
** OutputVMT-
**
**	A recursive routine that outputs the VMT table entries
**  in the reverse order that they appear in the symbol table
**  (so they are output in order of declaration).
*/


static void
OutputVMT( struct SymNode *ClassPtr, int *Offset )
_begin( OutputVMT )

	_if( ClassPtr != NULL )

		OutputVMT( ClassPtr->Next, Offset );
		_if
		( 
				ClassPtr->SymClass == cMethod 
			||	ClassPtr->SymClass == cClassIter 
		)

			*Offset += 4;
			EmitAdrs
			(
				_ifx
				( 
					strlen( ClassPtr->StaticName ) > 0,
					ClassPtr->StaticName,
					"abstract" sympost
				)
			); 
			ClassPtr->Offset = *Offset;
			
			// Add an "external" declaration for this symbol, just
			// in case we're inheriting fields from some class that
			// isn't defined in the current module:
			
			extLookup( ClassPtr, ClassPtr->StaticName, "near32", 0, 1, 0 );

		_endif

	_endif
	
_end( OutputVMT )


void
BuildVMT( struct SymNode *ClassPtr, char *VMTname )
_begin( BuildVMT )

	int	VMToffset;

	EmitPublic( VMTname );
	EmitTypedExtern( NULL, "abstract" sympost, "near32" );
	EmitTypedLabel( VMTname, "dword" );
	VMToffset = -4;
	OutputVMT( ClassPtr->Fields, &VMToffset );

_end( BuildVMT );



/*************************************************/
/*                                               */
/* UpdateVMTOffsets-                             */
/*                                               */
/* This routine recursively traverses the fields */
/* of a class and updates the OFFSET fields of   */
/* all the class methods and iterators.          */
/*                                               */
/*************************************************/

static void
SetVMTOffsets( struct SymNode *ClassPtr, int *Offset )
_begin( SetVMTOffsets )

	_if( ClassPtr != NULL )

		SetVMTOffsets( ClassPtr->Next, Offset );
		_if
		( 
				ClassPtr->SymClass == cMethod 
			||	ClassPtr->SymClass == cClassIter 
		)

			*Offset += 4; 
			ClassPtr->Offset = *Offset;

		_endif

	_endif
	
_end( SetVMTOffsets )


void
UpdateVMTOffsets( struct SymNode *ClassPtr )
_begin( UpdateVMTOffsets )

	int VMToffset = -4;
	SetVMTOffsets( ClassPtr->Fields, &VMToffset );

_end( UpdateVMTOffsets )





/******************************************************************/
/*                                                                */
/* CheckOrdinal-                                                  */
/*                                                                */
/* Checks a value to see if it is ordinal (or a pointer constant) */
/* and within the range specified by the second operand.          */
/*                                                                */
/******************************************************************/


char *
CheckOrdinal( union YYSTYPE *value, unsigned size )
_begin( CheckOrdinal )

	char Result[ 256 ];

	Result[0] = '0';
	Result[1] = '\0';
	
	_if( numBits( value ) > 32 )
	
		yyerror( "Constant value is out of range" );
		
	_elseif( IsOrdinal( value->v.pType ))


		_if( IsInt( value->v.pType ))

			_if
			( 
					(
							size == 8 
						&&	( value->v.u.intval < -128 || value->v.u.intval > 127 )
					)

				||	(
							size == 16 
						&&	( 
									value->v.u.intval < -32768 
								||	value->v.u.intval > 32676 
							)
					)
			)


				yyerror( "Constant value is out of range" );
				value->v.u.intval = 0;

			_endif

			sprintf
			(
				Result,
				"%d",
				value->v.u.intval
			);

		_else

			_if
			( 
					(
							size == 8 
						&&	( value->v.u.unsval > 255 )
					)
				||	(
							size == 16 
						&&	( value->v.u.unsval > 65535 )
					)
			)



				yyerror( "Constant value is out of range" );
				value->v.u.unsval = 0;

			_endif

			sprintf
			(
				Result,
				"%u",
				value->v.u.unsval
			);

		_endif

	_elseif( value->v.pType == tPointer )

		_if( size != 32 )

			yyerror( "Unexpected pointer constant" );

		_else

			LabelToOfs( Result, value->v.u.strval );

		_endif

	_else

		yyerror
		( 
			"Unexpected constant type, expected ordinal or pointer constant" 
		);

	_endif
	_return hlastrdup2( Result );

_end( CheckOrdinal )






char *
CheckUnsigned( union YYSTYPE *value, unsigned size )
_begin( CheckUnsigned )

	char Result[ 256 ];

	Result[0] = '0';
	Result[1] = '\0';
	_if( numBits( value ) > 32 )
	
		yyerror( "Constant value is out of range" );
		
	_elseif( IsOrdinal( value->v.pType ))

		_if( IsInt( value->v.pType ))

			yyerror( "Expected an unsigned constant" );

		_else

			_if
			( 
					(
							size == 3 
						&&	( value->v.u.unsval > 7 )
					)
				
				||	(
							size == 4 
						&&	( value->v.u.unsval > 15 )
					)

				||	(
							size == 5 
						&&	( value->v.u.unsval > 31 )
					)

				||	(
							size == 8 
						&&	( value->v.u.unsval > 255 )
					)

				||	(
							size == 16 
						&&	( value->v.u.unsval > 65535 )
					)
				||	numBits( value ) > 32
			)


				yyerror( "Constant value is out of range" );
				value->v.u.unsval = 0;

			_endif

			sprintf
			(
				Result,
				"%u",
				value->v.u.unsval
			);

		_endif

	_else

		yyerror
		( 
			"Unexpected unsigned constant type" 
		);

	_endif
	FreeValue( value );
	_return hlastrdup( Result );

_end( CheckUnsigned )


// putScale-
//	Utility function to output a scaled addressing mode item.

char *

rtnScale( union YYSTYPE *adrs )
_begin( rtnScale )

	_return
		_ifx
		( 
			adrs->adrs.Scale == 8, "*8",
			_ifx
			( 
				adrs->adrs.Scale == 4,  "*4",
				_ifx
				( 
					adrs->adrs.Scale == 2, "*2", 
					_ifx
					( 
						adrs->adrs.Scale ==1, "*1", 
						""
					)
				)
			)
		);

_end( rtnScale )


/************************************************************/
/*                                                          */
/* StoreAdrs-                                               */
/*                                                          */
/* "adrs" parameter takes the following form:               */
/*                                                          */
/* 	struct                                                  */
/* 	{                                                       */
/* 		unsigned	Size;                                   */
/* 		char		*StaticName;                            */
/* 		char		*BaseReg;                               */
/* 		char		*IndexReg;                              */
/* 		unsigned	Scale;                                  */
/* 		int			Disp;                                   */
/* 		char		*Comment;                               */
/* 	}adrs;                                                  */
/*                                                          */
/* This routine converts a MASM-compatible address expr     */
/* based on the components supplied in the adrs structure.  */
/* The output string is one of the following forms:         */
/*                                                          */
/* 	StaticName[0]                                           */
/*                                                          */
/* 	StaticName[BaseReg+0]                                   */
/* 	StaticName[BaseReg+Disp]                                */
/* 	StaticName[BaseReg-Disp]                                */
/*                                                          */
/* 	StaticName[IndexReg*Scale+0]                            */
/* 	StaticName[IndexReg*Scale+Disp]                         */
/* 	StaticName[IndexReg*Scale-Disp]                         */
/* 	                                                        */
/* 	StaticName[BaseReg + IndexReg*Scale + 0]                */
/* 	StaticName[BaseReg + IndexReg*Scale + Disp]             */
/* 	StaticName[BaseReg + IndexReg*Scale - Disp]             */
/*                                                          */
/*                                                          */
/* 	[BaseReg+0]                                             */
/* 	[BaseReg+Disp]                                          */
/* 	[BaseReg-Disp]                                          */
/*                                                          */
/* 	[IndexReg*Scale+0]                                      */
/* 	[IndexReg*Scale+Disp]                                   */
/* 	[IndexReg*Scale-Disp]                                   */
/*                                                          */
/* 	[BaseReg + IndexReg*Scale + 0]                          */
/* 	[BaseReg + IndexReg*Scale + Disp]                       */
/* 	[BaseReg + IndexReg*Scale - Disp]                       */
/*                                                          */
/*                                                          */
/* where BaseReg and IndexReg are any of the 32-bit general */
/* purpose registers, Scale is 1, 2, 4, or 8, and Disp is   */
/* an integer constant.                                     */
/*                                                          */
/************************************************************/

/*
** Variations:
**
** StoreAdrsSize -	Lets you control whether it outputs a size prefix.
** StoreAdrs - 		Full address output.
** StoreAdrsStr -	Only outputs the address string, no length prefix
**					and no brackets around the string.
*/



int
MakeAdrsStr( char *str, union YYSTYPE *adrs, int EmitSize, int PutBrackets )
_begin( MakeAdrsStr )

	int pType;
	int retval;		/* Return specified size here	*/
	int snLen;
	char *sn;

	assert( str != NULL );
	assert( adrs != NULL );


	/*
	** Grand Kludge!
	**
	**	Parameters passed in registers return the
	** register name in the StaticName field.  If
	** this particular address is a register, don't
	** adorn the address with anything but the
	** register name.
	*/

	sn = adrs->adrs.StaticName;
	_if( sn != NULL )

		snLen = strlen( sn );
		_if( snLen == 3 && sn[0] == 'e' )

			_if
			( 
					_streq( sn, "eax" )
				||	_streq( sn, "ebx" )
				||	_streq( sn, "ecx" )

				||	_streq( sn, "edx" )
				||	_streq( sn, "ebp" )
				||	_streq( sn, "esp" )
				||	_streq( sn, "esi" )
				||	_streq( sn, "edi" )
			)

				strcpy( str, sn );
				_return(4);

			_endif

		_elseif( snLen == 2 )

			_if
			( 
					_streq( sn, "al" )
				||	_streq( sn, "bl" )
				||	_streq( sn, "cl" )
				||	_streq( sn, "dl" )

				||	_streq( sn, "ah" )
				||	_streq( sn, "bh" )
				||	_streq( sn, "ch" )
				||	_streq( sn, "dh" )
			)

				strcpy( str, sn );
				_return(1);


			_elseif
			(
					_streq( sn, "ax" )
				||	_streq( sn, "bx" )
				||	_streq( sn, "cx" )
				||	_streq( sn, "dx" )
				||	_streq( sn, "bp" )
				||	_streq( sn, "sp" )
				||	_streq( sn, "si" )
				||	_streq( sn, "di" )
			)

				strcpy( str, sn );
				_return(2);

			_endif

		_endif


	_endif


	*str = '\0';

	assert( adrs->adrs.pType != tArray || adrs->adrs.Type != NULL );
	pType = _ifx
			( 
				adrs->adrs.pType != tArray, 
				adrs->adrs.pType,
				adrs->adrs.Type->pType
			);


	/*
	** Output the size (if applicable)
	*/

	retval = adrs->adrs.Size;
	_if( PutBrackets && EmitSize )

		retval = adrs->adrs.Size;
		_if( retval == 1 )

			strcpy( str, "byte ptr " );

		_elseif( retval == 2 )

			strcpy( str, "word ptr " );

		_elseif( retval == 4 )

			_if( pType == tReal32 )

				strcpy( str, isFGM( "dword ptr ", "dword ptr ", "real4 ptr " ));

			_else

				strcpy( str, "dword ptr " );

			_endif

		_else

			_if( pType == tReal64 )

				strcpy( str, isFGM( "qword ptr ", "qword ptr ", "real8 ptr " ));

			_elseif( pType == tReal80 )

				strcpy( str, isFGM( "tword ptr ", "tbyte ptr", "real10 ptr " ));

			_elseif
			( 
					pType == tQWord 
				||	pType == tUns64
				||	pType == tInt64
			)

				strcpy( str, "qword ptr " );

			_elseif( pType == tTByte )

				strcpy
				( 
					str, 
					_ifx( assembler == fasm, "tword ptr ", "tbyte ptr ") 
				);

			_endif
			retval = 0;

		_endif

	_endif
	

	/*
	** If there is a static name component to this
	** address, print it first.
	*/

	_if( PutBrackets )

		_if
		( 
				assembler == masm 
			||	assembler == fasm  
			||	assembler == gas
		)

			strcat( str, "[" );

		_elseif( assembler == tasm )

			strcat( str, "ds:[" );

		_else

			yyerror( "Internal HLA error (Bad assembler value)" );

		_endif

	_endif
	_if( adrs->adrs.StaticName != NULL && *adrs->adrs.StaticName != '\0' )

		strcat( str, adrs->adrs.StaticName );

	_endif

	/*
	** Print the Base Register (if present)
	*/

	_if( adrs->adrs.BaseReg != NULL )

		_if( adrs->adrs.StaticName != NULL && *adrs->adrs.StaticName != '\0' )
		
			strcat( str, "+" );
			
		_endif

		strcat( str, adrs->adrs.BaseReg );

	_endif

	/*
	** Print the index register, if present.  If there was
	** also a base register value, separate the two register
	** with a "+".  Also print the appropriate scaled index
	** value (default is 1).
	*/

	_if( adrs->adrs.IndexReg != NULL )

		sprintf
		(

			str + strlen(str),
			"%s%s%s",
			_ifx
			( 
					adrs->adrs.BaseReg != NULL 
				|| 	(
							adrs->adrs.StaticName != NULL 
						&&	*adrs->adrs.StaticName != '\0'
					), 
				"+", 
				"" 
			),
			adrs->adrs.IndexReg,
			rtnScale( adrs )
		);
		
	_endif

	/*
	** Print the displacement value.  If the displacement
	** value is positive, also emit a "+" if there was a
	** base or index register output.
	**
	** Note: don't output a displacement of zero because GAS
	** will generate a byte displacement value (of zero) if you
	** explicitly specify a displacement.
	*/

	_if( adrs->adrs.Disp != 0 )
	
		sprintf
		(
			str + strlen( str ),
			"%s%d",
			_ifx( adrs->adrs.Disp > 0, "+", "" ),
			adrs->adrs.Disp
		);
		
	_endif

	_if( PutBrackets )
	
		strcat( str, "]" );

	_endif
	return retval;

_end( MakeAdrsStr )


int
StoreAdrsSize( char *str, union YYSTYPE *adrs, int EmitSize )
_begin( StoreAdrsSize )

	_return MakeAdrsStr( str, adrs, EmitSize, 1 );

_end( StoreAdrsSize )


int
StoreAdrs( char *str, union YYSTYPE *adrs )
_begin( StoreAdrs )

	_return StoreAdrsSize( str, adrs, 1 );

_end( StoreAdrs )



static void
setOffset32( char *str )
_begin( setOffset32 )

	_if( assembler == gas )

		strcpy( str, "offset " );

	_elseif( assembler == fasm )
	
		*str = '\0';
	
	_else

		strcpy( str, "offset32 " );

	_endif

_end( setOffset32 )




/*********************************************************
/*                                                      */
/* StoreAdrsOfs-                                        */
/*                                                      */
/* str is destination where the address string goes.    */
/* adrs the the address value.                          */
/* offset, if true, says "offset32" precedes the value. */
/*                                                      */
/********************************************************/

int
StoreAdrsOfs
( 
	char *str, 
	union YYSTYPE *adrs, 
	int offset 
)
_begin( StoreAdrsOfs )

	int Result;

	_if
	(
			assembler == masm
		||	assembler == tasm
		||	assembler == gas
	)

		_if( offset )

			setOffset32( str );
			Result = StoreAdrsSize( str+strlen( str ), adrs, 0 );

		_else

			Result = StoreAdrs( str, adrs );

		_endif

	_elseif( assembler == fasm )

		_if( offset )

			Result = MakeAdrsStr( str, adrs, 0, 0 );
			
		_else

			Result = StoreAdrs( str, adrs );

		_endif

	_else

		Result = StoreAdrs( str, adrs );
		yyerror( "Internal HLA error (Bad assembler value)" );

	_endif
	return Result;

_end( StoreAdrsOfs )


/*************************************************************
/*                                                          */
/* StoreAdrsForceSize-                                      */
/*                                                          */
/* Converts an HLA address to a string with the prefix size */
/* specified by the "Size" parameter.                       */
/*                                                          */
/************************************************************/

int
StoreAdrsForceSize( char *str, union YYSTYPE *adrs, unsigned Size )
_begin(	StoreAdrsForceSize )

	_if( Size <= 1 )

		strcpy( str, "byte ptr " );

	_elseif( Size == 2 )

		strcpy( str, "word ptr " );

	_else

		strcpy( str, "dword ptr " );

	_endif
	_return StoreAdrsSize( str + strlen( str ), adrs, 0 );

_end( StoreAdrsForceSize )


/****************************************************/
/*                                                  */
/* FreeAdrs-                                        */
/*                                                  */
/* Frees the string associated with the StaticName, */
/* BaseReg, IndexReg, and Comment fields of an      */
/* adrs structure.                                  */
/*                                                  */
/****************************************************/


void

FreeAdrs( union YYSTYPE *adrs )
_begin( FreeAdrs )

	free2( vss adrs->adrs.StaticName );
	adrs->adrs.StaticName = NULL;

	free2( vss adrs->adrs.BaseReg );
	adrs->adrs.BaseReg = NULL;
	
	free2( vss adrs->adrs.IndexReg );
	adrs->adrs.IndexReg = NULL;
	
_end( FreeAdrs )



/*
** As above, but returns the variable's name as an allocated string
*/

void

adrsToStr( char *dest, union YYSTYPE *adrs )
_begin( adrsToStr )

	char scale[3];

	assert( adrs != NULL );
	scale[0] = '*';
	scale[1] = adrs->adrs.Scale | '0';
	scale[2] = 0;
	
	_if( adrs->adrs.Disp != 0 )
	
		sprintf
		(
			dest,
			"%s[%s%s%s%s%s%d]",
			_ifx
			( 
						adrs->adrs.StaticName != NULL 
					&&	adrs->adrs.Sym != NULL 
					&&	adrs->adrs.Sym->TrueName != NULL, 
					adrs->adrs.Sym->TrueName,
					""
			),
			_ifx( adrs->adrs.BaseReg != NULL, adrs->adrs.BaseReg, "" ),
			_ifx
			( 
					adrs->adrs.BaseReg != NULL 
				&& 	adrs->adrs.IndexReg != NULL, 
				"+", 
				""
			),
			_ifx( adrs->adrs.IndexReg != NULL, adrs->adrs.IndexReg, "" ),
			_ifx( adrs->adrs.IndexReg != NULL, scale, "" ),
			_ifx( (adrs->adrs.BaseReg != NULL || adrs->adrs.IndexReg != NULL) &&
					adrs->adrs.Disp >= 0, "+", "" ),
			adrs->adrs.Disp
		);
		
	_else
	
		// Don't emit a displacement of zero as GAS will emit
		// an actual displacement byte if you do this.
		
		sprintf
		(
			dest,
			"%s[%s%s%s%s]",
			_ifx
			( 
						adrs->adrs.StaticName != NULL 
					&&	adrs->adrs.Sym != NULL 
					&&	adrs->adrs.Sym->TrueName != NULL, 
					adrs->adrs.Sym->TrueName,
					""
			),
			_ifx( adrs->adrs.BaseReg != NULL, adrs->adrs.BaseReg, "" ),
			_ifx
			( 
					adrs->adrs.BaseReg != NULL 
				&& 	adrs->adrs.IndexReg != NULL, 
				"+", 
				""
			),
			_ifx( adrs->adrs.IndexReg != NULL, adrs->adrs.IndexReg, "" ),
			_ifx( adrs->adrs.IndexReg != NULL, scale, "" )
		);
		
	_endif

_end( adrsToStr )


char *
FreeAdrs2( union YYSTYPE *adrs )
_begin( FreeAdrs2 )

	char toReturn[256];
	
	adrsToStr( toReturn, adrs );
	free2( vss adrs->adrs.StaticName );
	free2( vss adrs->adrs.BaseReg );
	free2( vss adrs->adrs.IndexReg );
	_return hlastrdup2( toReturn );

_end( FreeAdrs2 )



/****************************************************************/
/*                                                              */
/* CombineAddresses-                                            */
/*                                                              */
/* This routine combines the components of two $<adrs> objects. */
/*                                                              */
/****************************************************************/

void
CombineAddresses( union YYSTYPE *dest, union YYSTYPE *src )
_begin( CombineAddresses )

	int		regCnt;
	char	*temp;
	char	comment[ 256 ];

	/*
	** If the base and/or index registers are the same,
	** then don't count them twice!
	*/

	_if
	( 
			dest->adrs.BaseReg != NULL
		&&	src->adrs.BaseReg != NULL
		&&	_streq( dest->adrs.BaseReg, src->adrs.BaseReg )
	)

		src->adrs.BaseReg = NULL;

	_endif

	_if
	(
			dest->adrs.IndexReg != NULL
		&&	src->adrs.IndexReg != NULL
		&&	dest->adrs.Scale == src->adrs.Scale
		&&	_streq( dest->adrs.IndexReg, src->adrs.IndexReg )
	)

		src->adrs.IndexReg = NULL;

	_endif


	/*
	** First, make sure that there aren't too many base
	** or index registers.
	*/

	regCnt =
			( dest->adrs.BaseReg != NULL )
		+	( dest->adrs.IndexReg != NULL )
		+	( src->adrs.BaseReg != NULL )
		+	( src->adrs.IndexReg != NULL );

	_if
	( 
			regCnt > 2 
		||	(
					dest->adrs.IndexReg != NULL
				&&	src->adrs.IndexReg != NULL
			)
	)
		yyerror
		( 
			"Too many index registers (may be due to using a VAR\n" 
			"variable in expression, which implies use of EBP)"
		);


	_endif


	/*
	** Merge the registers together here.
	*/

	_if( dest->adrs.BaseReg == NULL )

		dest->adrs.BaseReg = src->adrs.BaseReg;
		src->adrs.BaseReg = NULL;
		_if( dest->adrs.IndexReg == NULL )

			dest->adrs.IndexReg = src->adrs.IndexReg;
			dest->adrs.Scale = src->adrs.Scale;
			src->adrs.IndexReg = NULL;

		_endif

	_elseif( dest->adrs.IndexReg == NULL )

		_if( src->adrs.BaseReg != NULL )

			dest->adrs.IndexReg = src->adrs.BaseReg;
			dest->adrs.Scale = 1;
			src->adrs.BaseReg = NULL;

		_else

			dest->adrs.IndexReg = src->adrs.IndexReg;
			dest->adrs.Scale = src->adrs.Scale;
			src->adrs.IndexReg = NULL;

		_endif

	_endif

	/*
	** Merge the displacements.
	*/

	dest->adrs.Disp += src->adrs.Disp;
	
	// If the index register is ESP and the scaling factor is one,
	// swap the index register with the base register.
	
	_if
	(
			dest->adrs.IndexReg != NULL
		&&	dest->adrs.IndexReg[2] == 'p'	// Much faster than strcmp!
		&&	dest->adrs.IndexReg[1] == 's'
		&&	dest->adrs.IndexReg[0] == 'e'
	)
	
		_if( dest->adrs.Scale <= 1 )
		
			temp = dest->adrs.IndexReg;
			dest->adrs.IndexReg = dest->adrs.BaseReg;
			dest->adrs.BaseReg = temp;
			
		_else
		
			yyerror
			( 
				"Cannot use ESP as an index register (possible two\n"
				"indexes on an automatic variable?)"
			);
		
		_endif
	
	_endif

_end( CombineAddresses )


/************************************************************/
/*                                                          */
/* Output routines for the general instruction productions. */
/* OutputGenCM- source is constant, dest is memory.         */
/*                                                          */
/************************************************************/


static void
OutGenCMbwd
( 	
	char *instr, 
	union YYSTYPE *constant, 
	union YYSTYPE *dest, 
	unsigned size 
)
_begin( OutGenCMbwd )

	char *Operand;
	char adrs[128];

	assert( instr != NULL );
	assert( constant != NULL );
	assert( dest != NULL );

	Operand = CheckOrdinal( constant, size );
	StoreAdrs( adrs, dest );
	Emit2mc( instr, adrs, Operand );
	free2( vss Operand );
	FreeValue( constant );

_end( OutGenCMbwd )





void 
OutputGenCM( char *instr, union YYSTYPE *constant, union YYSTYPE *dest )
_begin( OutputGenCM )

	assert( instr != NULL );
	assert( constant != NULL );
	assert( dest != NULL );

	_if( IsOrdinal( constant->v.pType ))

		_if( dest->adrs.Size == 1 )

			OutGenCMbwd( instr, constant, dest, 8 );

		_elseif( dest->adrs.Size == 2 )

			OutGenCMbwd( instr, constant, dest, 16 );

		_elseif( dest->adrs.Size == 4 )

			OutGenCMbwd( instr, constant, dest, 32 );

		_else

			yyerror( "Memory operand must be 1, 2, or 4 bytes" );
			FreeValue( constant );

		_endif

	_elseif( constant->v.pType == tPointer )

		_if( dest->adrs.Size != 4 )

			yyerror( "Object must be a dword" );

		_else

			char adrs[128];
			char ptr[128];

			StoreAdrs( adrs, dest );
			ptr[0] = '\0';
			_if
			(
					assembler == masm
				||	assembler == tasm
				||	assembler == gas
			)
			
				setOffset32( ptr );
			
			_endif
			strcat( ptr, constant->v.u.strval );
			Emit2mc( instr, adrs, ptr );

		_endif
		FreeValue( constant );

	_else

		yyerror( "Expected an ordinal or address value" );
		FreeValue( constant );

	_endif

_end( OutputGenCM )




/************************************************************/
/*                                                          */
/* Output routine for mem-mem MOV instruction.				*/
/* OutputMemMM- source is memory, dest is memory.           */
/* OutputMemAM- source is anonymous memory, dest is memory.	*/
/* OutputMemMA- source is memory, dest is anonymous memory.	*/
/*                                                          */
/************************************************************/


void 
OutputMovMM( char *instr, union YYSTYPE *src, union YYSTYPE *dest )
_begin( OutputMovMM )

	char address[ 256 ];

	assert( instr != NULL );
	assert( src != NULL );
	assert( dest != NULL );

	_if( src->adrs.Size != dest->adrs.Size )

		yyerror( "Operand sizes must be the same" );

	_endif
	_if( src->adrs.pType == tReal32 )
	

		StoreAdrsForceSize( address, src, 4 );
		
	_else
	
		StoreAdrs( address, src );
		
	_endif
	Emit1M( "push", address );
	_if( dest->adrs.pType == tReal32 )
	
		StoreAdrsForceSize( address, dest, 4 );
		
	_else
	
		StoreAdrs( address, dest );
		
	_endif
	Emit1M( "pop", address );
	
_end( OutputMovMM )






void 
OutputMovAM( char *instr, union YYSTYPE *src, union YYSTYPE *dest )
_begin( OutputMovAM )

	char	address[ 256 ];
	int		OldSize;

	assert( instr != NULL );
	assert( src != NULL );
	assert( dest != NULL );

	OldSize = src->adrs.Size;
	src->adrs.Size = dest->adrs.Size;
	_if( src->adrs.pType == tReal32 )
	
		StoreAdrsForceSize( address, src, 4 );
		
	_else
	
		StoreAdrs( address, src );
		
	_endif
	Emit1M( "push", address );
	src->adrs.Size = OldSize;
	_if( dest->adrs.pType == tReal32 )
	
		StoreAdrsForceSize( address, dest, 4 );
		
	_else
	
		StoreAdrs( address, dest );
		
	_endif
	Emit1M( "pop", address );
	
_end( OutputMovAM )


void 
OutputMovMA( char *instr, union YYSTYPE *src, union YYSTYPE *dest )
_begin( OutputMovMA )

	char	address[ 256 ];
	int		OldSize;

	assert( instr != NULL );
	assert( src != NULL );
	assert( dest != NULL );

	_if( src->adrs.pType == tReal32 )
	
		StoreAdrsForceSize( address, src, 4 );
		
	_else
	
		StoreAdrs( address, src );
		

	_endif
	Emit1M( "push", address );
	OldSize = dest->adrs.Size;
	dest->adrs.Size = src->adrs.Size;
	_if( dest->adrs.pType == tReal32 )
	
		StoreAdrsForceSize( address, dest, 4 );
		
	_else
	
		StoreAdrs( address, dest );
		
	_endif
	Emit1M( "pop", address );
	dest->adrs.Size = OldSize;
	
_end( OutputMovMA )









/***************************************************/
/*                                                 */
/* OutputGenRM-                                    */
/*                                                 */
/* Outputs a generic reg, memory instruction.      */
/* Returns the string corresponding to the         */
/* memory address (HLA version) as the result.     */
/* It is the responsibility of the caller to       */
/* eventually free the storage associated with the */
/* returned string.                                */
/*                                                 */
/***************************************************/

char * 
OutputGenRM
( 
	char *instr, 
	char *reg, 
	union YYSTYPE *dest,
	unsigned size 
)
_begin( OutputGenRM )

	char address[ 256 ];

	assert( instr != NULL );
	assert( reg != NULL );
	assert( dest != NULL );

	_if( dest->adrs.Size == size )

		_if( dest->adrs.pType == tReal32 )
		
			StoreAdrsForceSize( address, dest, 4 );
			
		_else
		
			StoreAdrsSize( address, dest, 1 );
			
		_endif
		Emit2mr
		( 
			instr, 
			address,
			reg 
		);

	_else

		yyerror( SizeMismatchStr );


	_endif

	/*
	** Free up the strings this routine needs to clean up.
	*/

	_return FreeAdrs2( dest );

_end( OutputGenRM )


/***************************************************/
/*                                                 */
/* OutputGenRA-                                    */
/*                                                 */
/* Outputs a generic reg, anonmem instruction.     */
/* Returns the string corresponding to the         */
/* memory address (HLA version) as the result.     */
/* It is the responsibility of the caller to       */
/* eventually free the storage associated with the */
/* returned string.                                */
/*                                                 */
/***************************************************/

char * 
OutputGenRA
( 
	char *instr, 
	char *reg, 
	union YYSTYPE *dest,
	unsigned size 
)
_begin( OutputGenRA )

	char address[ 256 ];
	char name[256];

	assert( instr != NULL );
	assert( reg != NULL );
	assert( dest != NULL );

	StoreAdrs( address, dest );
	Emit2mr
	( 
		instr, 
		address,
		reg 
	);


	/*
	** Free up the strings this routine needs to clean up.
	*/

	adrsToStr( name, dest );
	sprintf
	( 
		address, 
		"(type %s %s)",
		_ifx
		( 
			size == 1, 
			"byte",
			_ifx( size == 2, "word", "dword" )
		),
		name
	);

	FreeAdrs( dest );
	_return hlastrdup( address );

_end( OutputGenRA )


/*******************************************************/
/*                                                     */
/* OutputGenMR-                                        */
/*                                                     */
/* Like OutputGenRM above, except the source is memory */
/* and the dest is a register.                         */
/*                                                     */
/*******************************************************/

char * 
OutputGenMR
( 
	char *instr, 
	union YYSTYPE *src,
	char *reg, 
	unsigned size 
)
_begin( OutputGenMR )

	char address[ 256 ];

	assert( instr != NULL );
	assert( src != NULL );
	assert( reg != NULL );

	_if( src->adrs.Size == size )

		_if( src->adrs.pType == tReal32 )
		
			StoreAdrsForceSize( address, src, 4 );
			
		_else
		
			StoreAdrsSize( address, src, 1 );
			
		_endif
		Emit2rm
		( 
			instr, 
			reg, 
			address 
		);

	_else

		yyerror( SizeMismatchStr );

	_endif

	/*
	** Free up the strings this routine needs to clean up.
	*/

	FreeAdrs( src );
	_return hlastrdup2( reg );

_end( OutputGenMR )






/*******************************************************/
/*                                                     */
/* OutputGenAR-                                        */
/*                                                     */
/* Like OutputGenMR above, except the source is anon   */
/* memory and the dest is a register.                  */
/*                                                     */
/*******************************************************/

char * 
OutputGenAR
(
	char *instr, 
	union YYSTYPE *src,
	char *reg 
)
_begin( OutputGenAR )

	char address[ 256 ];

	assert( src != NULL );
	assert( instr != NULL );
	assert( reg != NULL );

	StoreAdrs( address, src );
	Emit2rm
	( 
		instr, 
		reg, 
		address 
	);


	/*
	** Free up the strings this routine needs to clean up.
	*/

	FreeAdrs( src );
	_return hlastrdup2( reg );

_end( OutputGenAR )




/*********************************************************/
/*                                                       */
/* OutputLxS-                                            */
/*                                                       */
/* This function handles the LDS, LES, LFS, LGS, and LSS */
/* instructions (that require special handling for the   */
/* 48-bit memory operand).                               */
/*                                                       */
/*********************************************************/




char * 
OutputLxS
( 
	char *instr, 
	union YYSTYPE *src,
	char *reg 
)
_begin( OutputLxS )

	char address[ 256 ];

	assert( instr != NULL );
	assert( src != NULL );
	assert( reg != NULL );

	_if( src->adrs.Size == 4 || src->adrs.Size == 6 )
	
		strcpy( address, "fword ptr " );
		StoreAdrsSize( address+10, src, 0 );
		Emit2rm
		( 
			instr, 
			reg, 
			address 
		);

	_else

		yyerror( SizeMismatchStr );

	_endif

	/*
	** Free up the strings this routine needs to clean up.
	*/

	FreeAdrs( src );
	_return hlastrdup2( reg );

_end( OutputLxS )




/*********************************************************/
/*                                                       */
/* OutputLxS-                                            */
/*                                                       */
/* This function handles the LDS, LES, LFS, LGS, and LSS */
/* instructions (that require special handling for the   */
/* 48-bit memory operand).                               */
/*                                                       */
/*********************************************************/




void 
OutputLigdt
( 
	char *instr, 
	union YYSTYPE *src 
)
_begin( OutputLigdt )

	char address[ 256 ];

	assert( instr != NULL );
	assert( src != NULL );


	_if( assembler == gas )

		strcpy( address, "dword ptr " );

	_else

		strcpy( address, "fword ptr " );

	_endif

	StoreAdrsSize( address+10, src, 0 );
	Emit1M
	( 
		instr, 
		address
	);


	/*
	** Free up the strings this routine needs to clean up.
	*/

	FreeAdrs( src );
	_return;

_end( OutputLigdt )








/***************************************************/
/*                                                 */
/* OutputCmpRA-                                    */
/*                                                 */
/* Outputs a generic reg, anonmem instruction.     */
/* Returns the string corresponding to the         */
/* memory address (HLA version) as the result.     */
/* It is the responsibility of the caller to       */
/* eventually free the storage associated with the */
/* returned string.                                */
/*                                                 */
/***************************************************/

char * 
OutputCmpRA
( 
	char *reg, 
	union YYSTYPE *dest,
	unsigned size 
)
_begin( OutputCmpRA )

	char address[ 256 ];
	char name[256];

	assert( reg != NULL );
	assert( dest != NULL );

	StoreAdrs( address, dest );
	Emit2rm
	( 
		"cmp", 
		reg, 
		address 
	);

	/*
	** Update the Comment to reflect the known size.
	*/

	adrsToStr( name, dest );
	sprintf
	( 
		address, 
		"(type %s %s)",
		_ifx
		( 
			size == 1, 
			"byte",
			_ifx( size == 2, "word", "dword" )
		),
		name
	);

	/*
	** Free up the strings this routine needs to clean up.
	*/

	FreeAdrs( dest );
	_return hlastrdup( address );

_end( OutputCmpRA )







/*******************************************************/
/*                                                     */
/* OutputCmpAR-                                        */
/*                                                     */
/* Like OutputGenMR above, except the source is anon   */
/* memory and the dest is a register.                  */
/*                                                     */
/*******************************************************/

char * 
OutputCmpAR
( 
	union YYSTYPE *src,
	char *reg 
)
_begin( OutputCmpAR )

	char address[ 256 ];

	assert( src != NULL );
	assert( reg != NULL );

	StoreAdrs( address, src );
	Emit2mr
	( 
		"cmp", 
		address,
		reg 
	);


	/*
	** Free up the strings this routine needs to clean up.
	*/

	FreeAdrs( src );
	_return hlastrdup2( reg );

_end( OutputCmpAR )






/************************************
/*                                 */
/* OutputCXAR                      */
/*                                 */
/* Outputs CMPXCHG( AnonMem, Reg ) */
/*                                 */
/***********************************/


void 
OutputCXAR
( 
	union YYSTYPE *src,
	char *reg,
	int hasLock 
)
_begin( OutputCXAR )

	char address[ 256 ];

	assert( src != NULL );
	assert( reg != NULL );

	StoreAdrs( address, src );
	_if( hasLock )
	
		Emit2mr( "lock cmpxchg", address, reg  );
		
	_else
	
		Emit2mr( "cmpxchg", address, reg  );
		
	_endif
	FreeAdrs( src );

_end( OutputCXAR )


/*********************
/*                  */
/* OutputCXmr-      */
/* 	                */
/* 	cmpxchg mem,reg */
/*                  */
/********************/

void
OutputCXmr
( 
	union YYSTYPE *mem,
	char *reg,
	unsigned size,
	int hasLock 
)
_begin( OutputCXmr )

	char address[ 256 ];

	assert( mem != NULL );
	assert( reg != NULL );

	StoreAdrs( address, mem );
	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_else
	
		_if( hasLock )
		
			Emit2mr( "lock cmpxchg", address, reg  );
			
		_else
		
			Emit2mr( "cmpxchg", address, reg  );
			
		_endif

	_endif
	FreeAdrs( mem );

_end( OutputCXmr )







/***********************************
/*                                */
/* OutputDivAR                    */
/*                                */
/* Outputs DivMul( AnonMem, Reg ) */
/*                                */
/**********************************/


void 
OutputDivMulAR
(
	char			*instr, 
	union YYSTYPE	*src,
	unsigned		size 
)
_begin( OutputDivMulAR )

	char address[ 256 ];

	assert( instr != NULL );
	assert( src != NULL );

	StoreAdrsForceSize( address, src, size );
	Emit1M( instr, address );
	FreeAdrs( src );

_end( OutputDivMulAR )




/***************************************************************
/*                                                            */
/* OutputGenSRcm-                                             */
/*                                                            */
/* Shift and rotate instructions with a constant shift value. */
/*                                                            */
/**************************************************************/

extern char *OutputGenSRcm
( 
	char *instr, 
	union YYSTYPE *constant,
	union YYSTYPE *mem 
)
_begin( OutputGenSRcm )

	char *Shift;
	char address[ 256 ];

	assert( instr != NULL );
	assert( constant != NULL );
	assert( mem != NULL );

	StoreAdrs( address, mem );
	_if( mem->adrs.Size == 1 )

		Shift = CheckUnsigned( constant, 3 );
		Emit2mc( instr, address, Shift  );
		free2( vss Shift );

	_elseif( mem->adrs.Size == 2 )

		Shift = CheckUnsigned( constant, 4 );
		Emit2mc( instr, address, Shift );
		free2( vss Shift );

	_elseif( mem->adrs.Size == 4 )
	
		Shift = CheckUnsigned( constant, 5 );
		Emit2mc( instr, address, Shift );
		free2( vss Shift );

	_else

		yyerror( "Illegal memory operand size" );
		FreeValue( constant );

	_endif
	free2( vss instr );
	_return FreeAdrs2( mem );

_end( OutputGenSRcm )





/**************************************
/*                                   */
/* OutputGenSRclm-                   */
/*                                   */
/* Shift/Rotate using mem,cl syntax. */
/*                                   */
/*************************************/


char *
OutputGenSRclm
( 
	char *instr, 
	union YYSTYPE *mem 
)
_begin( OutputGenSRcm )

	char adrs[ 256 ];

	assert( instr != NULL );
	assert( mem != NULL );

	StoreAdrs( adrs, mem );
	_if
	(
			mem->adrs.Size == 1 
		||	mem->adrs.Size == 2
		||	mem->adrs.Size == 4
	)


		Emit2mr( instr, adrs, "cl" );

	_else

		yyerror( "Illegal memory operand size" );

	_endif
	free2( vss instr );
	_return FreeAdrs2( mem );

_end( OutputGenSRclm )





/***************************
/*                        */
/* OutputBndRM-           */
/* 	                      */
/* 	Bound reg,mem output. */
/*                        */
/**************************/

char *
OutputBndRM
( 
	char *reg, 
	union YYSTYPE *mem,
	unsigned size 
)
_begin( OutputBndRM )

	enum PrimType pSave;
	unsigned sizeSave;
	char address[ 256 ];

	assert( reg != NULL );
	assert( mem != NULL );

	sizeSave = mem->adrs.Size;
	pSave = mem->adrs.pType;
	_if( assembler == tasm )
	
		/*
		** TASM requires "dword ptr" and "qword ptr"
		** instead of "word ptr" and "dword ptr" on
		** the bound operands.
		*/

		_if( sizeSave == 2 )
		
			mem->adrs.Size = 4;
			mem->adrs.pType = tDWord;
			
		_else
		
			mem->adrs.Size = 8;
			mem->adrs.pType = tQWord;
			
		_endif
		
	_endif

	StoreAdrs( address, mem );

	_if( assembler == tasm )
	
		mem->adrs.Size = sizeSave;
		mem->adrs.pType = pSave;
		
	_endif
	
	_if( mem->adrs.Size < size )

		yyerror( SizeMismatchStr );

	_elseif( CompileBound )

		Emit2rm( "bound", reg, address );

	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputBndRM )






/***************************************
/*                                    */
/* OutputBndRCC -                     */
/*                                    */
/* 	Bound( reg, const, const) output. */
/*                                    */
/**************************************/

void
OutputBndRCC
( 
	char *reg, 
	unsigned lower,
	unsigned upper,
	unsigned size,
	char *LBound,
	char *UBound
	 
)
_begin( OutputBndRCC )

	char 	lbound[64];
	char 	p_lbound[64];
	char 	ubound[64];

	sprintf( lbound, "L%d_LBound" sympost, LblCntr );
	sprintf( ubound, "L%d_UBound" sympost, LblCntr );
	_if( lower < upper )
	
		_if( assembler == tasm )

			/*
			** TASM requires "dword ptr" and "qword ptr"
			** instead of "word ptr" and "dword ptr" on
			** the bound operands.
			*/
			
			_if( size == 2 )

				startStrSeg();
				EmitData( lbound, "dword", LBound );
				EmitData( ubound, "dword", UBound );
				endStrSeg();
				LabelToMem( p_lbound, lbound, "dword" );
				Emit2rm( "bound", reg, p_lbound );
				++LblCntr;

			_else

				startStrSeg();
				EmitData( lbound, "qword", LBound );
				EmitData( ubound, "qword", UBound );
				endStrSeg();
				LabelToMem( p_lbound, lbound, "qword" );
				Emit2rm( "bound", reg, p_lbound );
				++LblCntr;

			_endif
		
		_else
		
			_if( size == 2 )

				startStrSeg();
				EmitData( lbound, "word", LBound );
				EmitData( ubound, "word", UBound );
				endStrSeg();
				LabelToMem( p_lbound, lbound, "word" );
				Emit2rm( "bound", reg, p_lbound );
				++LblCntr;

			_else

				startStrSeg();
				EmitData( lbound, "dword", LBound );
				EmitData( ubound, "dword", UBound );
				endStrSeg();
				LabelToMem( p_lbound, lbound, "dword" );
				Emit2rm( "bound", reg, p_lbound );
				++LblCntr;

			_endif
			
		_endif
		
	_else
	
		yyerror( "Lower bound must be less than upper bound" );
		
	_endif

_end( OutputBndRCC )






/*****************************
/*                          */
/* OutputBBMR -             */
/*                          */
/* 	BSR/BSF mem, reg output */
/*                          */










/****************************/

char *
OutputBBMR
(
	char *instr, 
	char *reg, 
	union YYSTYPE *mem,
	unsigned size 
)
_begin( OutputBBMR )

	char address[ 256 ];

	assert( instr != NULL );
	assert( reg != NULL );
	assert( mem != NULL );

	StoreAdrs( address, mem );
	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_endif
	Emit2rm( instr, reg, address );
	free2( vss instr );
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputBBMR )




/****************************************
/*                                     */
/* OutputBTcm -                        */
/*                                     */
/* Handles BT/BTR/BTC/BTS constant,mem */
/*                                     */
/***************************************/

char *
OutputBTcm
( 
	char *instr, 
	union YYSTYPE *constant,
	union YYSTYPE *mem 
)
_begin( OutputBTcm )

	char	address[ 256 ];
	char	cons[32];

	assert( instr != NULL );
	assert( constant != NULL );
	assert( mem != NULL );

	StoreAdrs( address, mem );
	sprintf( cons, "%u", constant->v.u.unsval ); 
	_if( IsOrdinal( constant->v.pType ) && numBits( constant ) <= 32 )

		
		_if( constant->v.u.unsval >= 256 )

			yyerror( "Bit number must be between 0 and 255" );

		_elseif
		( 
				mem->adrs.Size == 2
			|| 	mem->adrs.Size == 4
		)

			Emit2mc( instr, address, cons );

		_elseif( mem->adrs.pType == tCset )


			StoreAdrsForceSize( address, mem, 4 );
			Emit2mc( instr, address, cons );


		_else

			yyerror( "Illegal operand size" );

		_endif

	_else

		yyerror( "Expected ordinal value for bit number" );

	_endif
	free2( vss instr );
	FreeValue( constant );
	_return FreeAdrs2( mem );

_end( OutputBTcm )




/***************************
/*                        */
/* OutputBTrm             */
/*                        */
/* BT/BTS/BTC/BTR reg,mem */
/*                        */
/**************************/

char *
OutputBTrm
( 
	char *instr, 
	char *reg,
	union YYSTYPE *mem,
	unsigned size 
)
_begin( OutputBTrm )

	char address[ 256 ];

	assert( instr != NULL );
	assert( reg != NULL );
	assert( mem != NULL );

	StoreAdrsForceSize( address, mem, size );
	_if
	( 
			mem->adrs.Size == size 
		||	mem->adrs.pType == tCset
	)
		
		Emit2mr( instr, address, reg );

	_else

		yyerror( "Operands must be the same size" );

	_endif
	free2( vss instr );
	_return FreeAdrs2( mem );

_end( OutputBTrm )



/****************
/*             */
/* OutputCOrm  */
/*             */
/* cmp reg,mem */
/*             */
/***************/

char *
OutputCOrm
( 
	char *reg,
	union YYSTYPE *mem,
	unsigned size 
)
_begin( OutputCOrm )

	char address[ 256 ];

	assert( reg != NULL );
	assert( mem != NULL );

	StoreAdrs( address, mem );
	_if( mem->adrs.Size == size )

		Emit2rm
		( 
			"cmp", 
			reg, 
			address 
		);

	_else

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputCOrm )




/****************
/*             */
/* OutputCOmr  */
/*             */
/* cmp mem,reg */
/*             */
/***************/

char *
OutputCOmr
( 
	union YYSTYPE *mem,
	char *reg,
	unsigned size 
)
_begin( OutputCOmr )

	char address[ 256 ];

	assert( mem != NULL );
	assert( reg != NULL );

	StoreAdrs( address, mem );
	_if( mem->adrs.Size == size )

		Emit2mr
		( 
			"cmp", 
			address,
			reg 
		);

	_else

		yyerror( SizeMismatchStr );

	_endif
	_return FreeAdrs2( mem );

_end( OutputCOmr )




/*********************
/*                  */
/* OutputCOmc       */
/*                  */
/* cmp mem,constant */
/*                  */
/********************/

char *
OutputCOmc
( 
	union YYSTYPE *mem,
	union YYSTYPE *constant 
)
_begin( OutputCOmc )

	char *Operand;
	char address[ 256 ];

	assert( mem != NULL );
	assert( constant != NULL );

	StoreAdrs( address, mem );
	Operand = CheckOrdinal( constant, mem->adrs.Size * 8 );

	_if( constant->v.pType != tPointer )
			   
		Emit2mc( "cmp", address, Operand );

	_else

		_if( mem->adrs.Size == 4 )

			Emit2mc( "cmp", address, Operand );
			
		_else

			yyerror( "memory object must be a dword" );

		_endif

	_endif
	free2( vss Operand );
	FreeValue( constant );
	_return FreeAdrs2( mem );

_end( OutputCOmc )






/****************
/*             */
/* OutputDm -  */
/* 	           */
/* 	div mem    */
/*             */
/***************/

char *
OutputDm
( 
	union YYSTYPE *mem 
)
_begin( OutputDm )

	char *returns;
	char address[ 256 ];

	assert( mem != NULL );

	StoreAdrs( address, mem );
	Emit1M( "div", address );
	returns = "eax";
	_if( mem->adrs.Size == 1 )

		returns = "al";

	_elseif( mem->adrs.Size == 2 )

		returns = "ax";

	_elseif( mem->adrs.Size != 4 )

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup( returns );

_end( OutputDm )




/****************
/*             */
/* OutputDm -  */
/*             */
/* 	mod mem    */
/*             */
/***************/

char *
OutputMm
( 
	union YYSTYPE *mem 
)
_begin( OutputDm )

	char *returns;
	char address[ 256 ];

	assert( mem != NULL );


	StoreAdrs( address, mem );
	Emit1M( "div", address );
	returns = "edx";
	_if( mem->adrs.Size == 1 )

		returns = "ah";

	_elseif( mem->adrs.Size == 2 )

		returns = "dx";

	_elseif( mem->adrs.Size != 4 )

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup( returns );

_end( OutputDm )


/*****************
/*              */
/* OutputIDm -  */
/*              */
/* 	idiv mem    */
/*              */
/****************/

char *
OutputIDm
( 
	union YYSTYPE *mem 
)
_begin( OutputIDm )

	char *returns;
	char address[ 256 ];

	assert( mem != NULL );

	StoreAdrs( address, mem );
	Emit1M( "idiv", address );
	returns = "eax";
	_if( mem->adrs.Size == 1 )

		returns = "al";

	_elseif( mem->adrs.Size == 2 )

		returns = "ax";

	_elseif( mem->adrs.Size != 4 )

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup( returns );


_end( OutputIDm )


/*****************
/*              */
/* OutputIMm -  */
/*              */
/* 	imod mem    */
/*              */
/****************/

char *
OutputIMm
( 
	union YYSTYPE *mem 
)
_begin( OutputIDm )

	char *returns;
	char address[ 256 ];

	assert( mem != NULL );

	StoreAdrs( address, mem );
	Emit1M( "idiv", address );
	returns = "edx";
	_if( mem->adrs.Size == 1 )

		returns = "ah";

	_elseif( mem->adrs.Size == 2 )

		returns = "dx";

	_elseif( mem->adrs.Size != 4 )

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup( returns );

_end( OutputIDm )


/******************
/*               */
/* OutputMULm -  */
/*               */
/* 	mul mem      */
/*               */
/*****************/

char *
OutputMULm
( 
	union YYSTYPE *mem 
)
_begin( OutputMULm )

	char *returns;
	char address[ 256 ];

	assert( mem != NULL );

	StoreAdrs( address, mem );
	Emit1M( "mul", address );
	returns = "edx:eax";
	_if( mem->adrs.Size == 1 )

		returns = "ax";

	_elseif( mem->adrs.Size == 2 )

		returns = "dx:ax";

	_elseif( mem->adrs.Size != 4 )

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup( returns );

_end( OutputMULm )




/*******************
/*                */
/* OutputIMULm -  */
/*                */
/* 	imul mem      */
/*                */
/******************/

char *
OutputIMULm
( 
	union YYSTYPE *mem 
)
_begin( OutputIMULm )

	char *returns;
	char address[ 256 ];


	assert( mem != NULL );

	StoreAdrs( address, mem );
	Emit1M( "imul", address );
	returns = "edx:eax";
	_if( mem->adrs.Size == 1 )

		returns = "ax";

	_elseif( mem->adrs.Size == 2 )

		returns = "dx:ax";

	_elseif( mem->adrs.Size != 4 )

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup( returns );


_end( OutputIMULm )





/*********************
/*                  */
/* OutputIntMulmr-  */
/*                  */
/* 	imul mem, reg   */
/*                  */
/********************/

char *
OutputIntMulmr
( 
	union YYSTYPE *mem,
	char *reg,
	unsigned size 
)
_begin( OutputIntMulmr )

	char adrs[128];

	assert( mem != NULL );
	assert( reg != NULL );

	StoreAdrs( adrs, mem );
	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_else

		Emit2rm( "imul", reg, adrs );

	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputIntMulmr )


/**********************************************
/*                                           */
/* OutputIntMulAR                            */
/*                                           */
/* Outputs the intmul( const, anonmem, reg ) */
/* instruction.                              */
/*                                           */
/*********************************************/


char *
OutputIntMulAR
(
	union YYSTYPE *src,
	union YYSTYPE *adrs,
	char *reg,
	unsigned size 
)
_begin( OutputIntMulAR )

	char	*multiplier;
	char	address[128];

	assert( src != NULL );
	assert( adrs != NULL );
	assert( reg != NULL );

	multiplier = CheckOrdinal( src, size*8 );
	StoreAdrs( address, adrs );
	Emit3rmc( "imul", reg, address, multiplier );
	free2( vss multiplier );
	FreeAdrs( adrs );
	FreeValue( src );
	_return hlastrdup2( reg );

_end( OutputIntMulAR )







/**************************
/*                       */
/* OutputIntMulcmr -     */
/* 	imul const, mem, reg */
/*                       */
/*************************/

char *
OutputIntMulcmr
( 
	union YYSTYPE *constant,
	union YYSTYPE *mem,
	char *reg,
	unsigned size
)
_begin( OutputIntMulcmr )

	char adrs[128];
	char *theConst;
	
	assert( reg != NULL );
	assert( mem != NULL );
	assert( constant != NULL );

	StoreAdrs( adrs, mem );
	theConst = CheckOrdinal( constant, size*8 );

	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_else

		Emit3rmc( "imul", reg, adrs, theConst );

	_endif

	free2( vss theConst );

	FreeValue( constant );
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputIntMulcmr )




/*******************
/*                */
/*  -  */
/*                */
/* 	lea mem, reg  */
/*                */
/******************/

char *
OutputLEAmr
( 
	union YYSTYPE *mem,
	char *reg
)
_begin( OutputLEAmr )

	char adrs[ 128 ];

	assert( reg != NULL );
	assert( mem != NULL );

	StoreAdrsSize( adrs, mem, assembler != gas );
	Emit2rm( "lea", reg, adrs );
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputLEAmr )




/***************************************************************
/*                                                            */
/* OutputSXops-                                               */
/* 	                                                          */
/* 	Handles the sign extension and zero extension operations. */
/*                                                            */
/**************************************************************/

extern char *OutputSXops
( 
	char *instr, 
	union YYSTYPE *mem,
	char *reg, 
	unsigned size 
)
_begin( OutputSXops )

	char adrs[128];

	assert( reg != NULL );
	assert( instr != NULL );
	assert( mem != NULL );

	_if( mem->adrs.Size <= size )

		StoreAdrs( adrs, mem );
		Emit2mr( instr, reg, adrs );

	_else

		yyerror( SizeMismatchStr );

	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputSXops )



/*******************************************************
/*                                                    */
/* OutputSHxDcrm-                                     */
/*                                                    */
/* 	Outputs the SHRD and SHLD const, reg, mem instrs. */
/*                                                    */
/******************************************************/

char *
OutputSHxDcrm
(
	char *instr, 
	union YYSTYPE *constant,
	char *reg,
	union YYSTYPE *mem, 
	unsigned size 
)
_begin( OutputSHxDcrm )

	char *count;
	unsigned SizeInBits;
	char	adrs[128];

	assert( reg != NULL );
	assert( constant != NULL );
	assert( instr != NULL );
	assert( mem != NULL );

	SizeInBits = _ifx( size == 2, 4, 5 );
	count = CheckUnsigned( constant, SizeInBits );
	StoreAdrs( adrs, mem );
	Emit3mrc( instr, adrs, reg, count );
	free2( vss count );
	_return FreeAdrs2( mem );

_end( OutputSHxDcrm )



/***********************************************************
/*                                                        */
/* OutputSHxDcra                                          */
/*                                                        */
/* 	Outputs the SHRD and SHLD const, reg, anonmem instrs. */
/*                                                        */
/**********************************************************/

char *
OutputSHxDcra
( 
	char *instr,
	union YYSTYPE *constant,
	char *reg,
	union YYSTYPE *mem, 
	unsigned sizeInBits 
)
_begin( OutputSHxDcra )

	char 	*count;
	char	adrs[128];

	assert( reg != NULL );
	assert( constant != NULL );
	assert( instr != NULL );
	assert( mem != NULL );

	count = CheckUnsigned( constant, sizeInBits );
	StoreAdrs( adrs, mem );
	Emit3mrc( instr, adrs, reg, count );
	free2( vss count );
	_return FreeAdrs2( mem );

_end( OutputSHxDcra )



/********************************************************
/*                                                     */
/* OutputSHxDclra                                      */
/*                                                     */
/* 	Outputs the SHRD and SHLD cl, reg, anonmem instrs. */
/*                                                     */
/*******************************************************/

char *
OutputSHxDclra
( 
	char *instr,
	char *reg,
	union YYSTYPE *mem 
)
_begin( OutputSHxDclra )
	  
	char	adrs[128];

	assert( reg != NULL );
	assert( instr != NULL );
	assert( mem != NULL );

	StoreAdrs( adrs, mem );
	Emit3mrr( instr, adrs, reg, "cl" );
	_return FreeAdrs2( mem );


_end( OutputSHxDclra )



/***************************************************
/*                                                */
/* OutputSHxDclrm-                                */
/*                                                */
/* Outputs the SHRD and SHLD cl, reg, mem instrs. */
/*                                                */
/**************************************************/

char *
OutputSHxDclrm
(
	char *instr, 
	char *reg,
	union YYSTYPE *mem,
	unsigned size 
)
_begin( OutputSHxDcrm )

	char	adrs[128];

	assert( reg != NULL );
	assert( instr != NULL );
	assert( mem != NULL );

	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_else

		StoreAdrs( adrs, mem );
		Emit3mrr( instr, adrs, reg, "cl" );

	_endif
	_return FreeAdrs2( mem );

_end( OutputSHxDcrm )




/********************
/*                 */
/* OutputXADDmr -  */
/*                 */
/* 	xadd mem, reg  */
/*                 */
/*******************/

char *
OutputXADDmr
( 
	union YYSTYPE *mem,
	char *reg,
	unsigned size,
	int hasLock 
)
_begin( OutputXADDmr )

	char	adrs[128];

	assert( reg != NULL );
	assert( mem != NULL );

	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_endif
	StoreAdrs( adrs, mem );
	_if( hasLock )
	
		Emit2mr( "lock xadd", adrs, reg );
		
	_else
	
		Emit2mr( "xadd", adrs, reg );
		
	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputXADDmr )




/********************
/*                 */
/* OutputXADDar -  */
/*                 */
/* 	xadd anon, reg */
/*                 */
/*******************/

char *
OutputXADDar
( 
	union YYSTYPE *mem,
	char *reg,
	unsigned size,
	int hasLock 
)
_begin( OutputXADDar )

	char	adrs[128];

	assert( reg != NULL );
	assert( mem != NULL );

	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_endif
	StoreAdrs( adrs, mem );
	_if( hasLock )
	
		Emit2mr( "xadd", adrs, reg );
		
	_else
	
		Emit2mr( "xadd", adrs, reg );
		
	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputXADDar )




/********************
/*                 */
/* OutputXCHGmr -  */
/*                 */
/* 	xchg mem, reg  */
/*                 */
/*******************/

char *
OutputXCHGmr
( 
	union YYSTYPE *mem,
	char *reg,
	unsigned size,
	int hasLock 
)
_begin( OutputXCHGmr )

	char	adrs[128];

	assert( reg != NULL );
	assert( mem != NULL );

	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_endif
	StoreAdrs( adrs, mem );
	_if( hasLock )
	
		Emit2mr( "lock xchg", adrs, reg );
		
	_else
	
		Emit2mr( "xchg", adrs, reg );
		
	_endif
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputXCHGmr )



/********************
/*                 */
/* OutputXCHGrm -  */
/* 	               */
/* 	xchg reg, mem  */
/*                 */
/*******************/

char *
OutputXCHGrm
( 
	char *reg,
	union YYSTYPE *mem,
	unsigned size,
	int hasLock 
)
_begin( OutputXCHGrm )

	char	adrs[128];

	assert( reg != NULL );
	assert( mem != NULL );

	_if( mem->adrs.Size != size )

		yyerror( SizeMismatchStr );

	_endif
	StoreAdrs( adrs, mem );
	_if( hasLock )
	
		Emit2mr( "lock xchg", adrs, reg );
		
	_else
	
		Emit2mr( "xchg", adrs, reg );
		
	_endif
	_return FreeAdrs2( mem );

_end( OutputXCHGrm )



/********************************
/*                             */
/* OutputIRMS                  */
/*                             */
/* Output:	instr reg, anonmem */
/*                             */
/*******************************/

char* 
OutputIRMS
(
	char *instr,
	char *reg,
	union YYSTYPE *mem,
	unsigned size,
	int hasLock
)
_begin( OutputIRMS )

	char 	returns[ 256 ];
	char	adrs[ 128 ];
	char	instr2[256];

	assert( reg != NULL );
	assert( instr != NULL );
	assert( mem != NULL );

	StoreAdrs( adrs, mem );
	adrsToStr( instr2, mem );
	sprintf
	( 
		returns, 
		"(type %s %s)", 
		_ifx
		( 
			size == 1,
			"byte",
			_ifx( size == 2, "word", "dword" )
		),
		instr2
	);
	sprintf( instr2, "%s%s", _ifx( hasLock, "lock ", ""), instr );
	Emit2mr( instr2, adrs, reg );
	FreeAdrs( mem );
	_return hlastrdup2( returns );

_end( OutputIRMS )





/*******************************
/*                            */
/* OutputIMRS-                */
/*                            */
/* 	Output instr anonmem, reg */
/*                            */
/******************************/

char* 
OutputIMRS
(
	char *instr,
	union YYSTYPE *mem,
	char *reg,
	int hasLock
)
_begin( OutputIMRS )

	char	adrs[ 128 ];
	char	instr2[32];

	assert( mem != NULL );
	assert( reg != NULL );
	assert( instr != NULL );

	StoreAdrs( adrs, mem );
	sprintf( instr2, "%s%s", _ifx( hasLock, "lock ", ""), instr );
	Emit2rm( instr2, reg, adrs );
	FreeAdrs( mem );
	_return hlastrdup2( reg );

_end( OutputIMRS )









/************************************************
/*                                             */
/* IsReg-                                      */
/*                                             */
/* 	Returns true if the parameter is a string  */
/* 	that matches one of the integer registers. */
/*                                             */
/***********************************************/

#define cmp2(str,one,two) (str[0] == one && str[1] == two)
#define cmp3(str,one,two,three) (str[0]==one && str[1]==two && str[2]==three)

static int
IsReg( char *reg )
_begin( IsReg )

	int len;
	char str[4];

	_if( reg == NULL )

		_return 0;

	_endif

	len = strlen( reg );
	_if( len == 2 )

		
		strcpy( str, reg );
		strlwr( str );
		_return
		(
				cmp2( str, 'a', 'l' )
			||	cmp2( str, 'b', 'l' )
			||	cmp2( str, 'c', 'l' )
			||	cmp2( str, 'd', 'l' )
			||	cmp2( str, 'a', 'h' )
			||	cmp2( str, 'b', 'h' )

			||	cmp2( str, 'c', 'h' )
			||	cmp2( str, 'd', 'h' )

			||	cmp2( str, 'a', 'x' )
			||	cmp2( str, 'b', 'x' )
			||	cmp2( str, 'c', 'x' )
			||	cmp2( str, 'd', 'x' )
			||	cmp2( str, 'd', 'i' )
			||	cmp2( str, 's', 'i' )
			||	cmp2( str, 'b', 'p' )
			||	cmp2( str, 's', 'p' )
		);


	_elseif( len == 3 )
	
		strcpy( str, reg );
		strlwr( str );
		_return
		(
				cmp3( str, 'e', 'a', 'x' )
			||	cmp3( str, 'e', 'b', 'x' )
			||	cmp3( str, 'e', 'c', 'x' )
			||	cmp3( str, 'e', 'd', 'x' )
			||	cmp3( str, 'e', 'd', 'i' )
			||	cmp3( str, 'e', 's', 'i' )
			||	cmp3( str, 'e', 'b', 'p' )
			||	cmp3( str, 'e', 's', 'p' )
		);

	_endif
	_return 0;

_end( IsReg )



/************************************************************
/*                                                         */
/* PushReg32-                                              */
/*                                                         */
/* 	Emits a push instruction that pushes a 32-bit register */
/* 	onto the stack.                                        */
/*                                                         */
/***********************************************************/

static void
PushReg32( char *reg )
_begin( PushReg32 )

	char regStr[8];
	char regStr8[8];

	tolower( *reg );
	_if( strlen( reg ) == 3 )

		/*
		** Just push 32-bit registers onto the stack:
		*/

		Emit1R( "push", reg );

	_elseif( reg[1] != 'h' )

		/*
		** Push xL 8-bit and 16-bit registers onto the stack as dwords here.
		*/

		sprintf
		(
			regStr,
			"e%c%c",
			reg[1],
			_ifx( reg[1] == 'l', 'x', reg[1] )
		);

		Emit1R( "push", regStr  );

	_else
	
	
		/*
		**	8-bit "xH" registers are problematic, handle them here.
		*/

		Emit2rc( "sub", "esp", "4" );
	 	Emit2mr( "mov", "[esp]", reg );

	_endif

_end( PushReg32 )





/******************************************************************/
/*                                                                */
/* OutputMemParm-                                                 */
/*                                                                */
/* "formal" is the type of the formal parameter.                  */
/* "actual" is the info associated with the actual parameter.     */
/* If reasonable, emit the code to push the actual parameter onto */
/* the stack.                                                     */
/*                                                                */
/******************************************************************/


void 
OutputMemParm
( 
	struct SymNode	*theProc, 
	struct SymNode	*formal, 
	union YYSTYPE	*actual,
	int				valPrefix 
)
_begin( OutputMemParm )

	struct	SymNode		*abt;
	struct	SymNode 	*fbt;
	enum	PrimType 	apType;
	struct	SymNode		*aType;
	unsigned			aSize;
	unsigned			fSize;
	char				adrs[128];

	assert( formal != NULL );
	assert( actual != NULL );

	abt = 	_ifx
			( 
				actual->adrs.Type != NULL, 
				GetBaseType( actual->adrs.Type ),
				actual->adrs.Sym 
			);
	fbt = 	_ifx
			(
				formal->Type != NULL,
				GetBaseType( formal->Type ),
				formal
			);

	aSize = actual->adrs.ObjectSize;
	fSize = 
		_ifx
		( 
			formal->Type != NULL,
			formal->Type->ObjectSize,
			4
		);
		

	_switch( formal->pClass )

		/*
		** Handle parameters passed by value here.
		*/

		_case( valp_pc )

			_if( valPrefix )
			
				yyerror
				( 
					"VAL prefix is illegal for pass by value parameters" 
				);
				
				
			/*
			** If the actual parameter is a static variable,
			** a local variable, or a parameter passed into the
			** current procedure by value, handle it here.
			*/

			_elseif
			( 
					actual->adrs.SymClass == cStatic
				||	actual->adrs.SymClass == cVar 
				||	(
							actual->adrs.SymClass == cParm
						&&	actual->adrs.pClass == valp_pc
					)
			)

				/*
				** Be sure the formal and actual parameter
				** types agree.
				*/


				_if
				( 
						(
								formal->pType == actual->adrs.pType
							&&	formal->pType != tArray
						)
					||	(
								formal->pType == tArray
							&&	fbt == abt
							&&	(unsigned) formal->ObjectSize == 
										(unsigned) actual->adrs.ObjectSize
						)
					||	(
								actual->adrs.ObjectSize == 16 
							&&	formal->pType == tLWord
						)
					||	(
								actual->adrs.ObjectSize == 8 
							&&	formal->pType == tQWord
						)
					||	(
								actual->adrs.ObjectSize == 4 
							&&	formal->pType == tDWord
						)
					||	(
								actual->adrs.ObjectSize == 2 
							&&	formal->pType == tWord
						)
					||	(
								actual->adrs.ObjectSize == 1 
							&&	formal->pType == tByte
						)
						
					||	(
								formal->Type != NULL
							&&	(

									(		formal->Type->ObjectSize == 16 
										&&	actual->adrs.pType == tLWord
									)
								||	(
											formal->Type != NULL
										&&	formal->Type->ObjectSize == 8 
										&&	actual->adrs.pType == tQWord
									)
								||	(
											formal->Type != NULL
										&&	formal->Type->ObjectSize == 4 
										&&	actual->adrs.pType == tDWord
									)
								||	(
											formal->Type != NULL
										&&	formal->Type->ObjectSize == 2 
										&&	actual->adrs.pType == tWord
									)
								||	(
											formal->Type != NULL
										&&	formal->Type->ObjectSize == 1 
										&&	actual->adrs.pType == tByte
									)
								)
						)
				)

					int FormalIsReg;
					int ActualIsReg;

					FormalIsReg = IsReg( formal->StaticName );
					ActualIsReg = IsReg( actual->adrs.StaticName );
					_if( !FormalIsReg && !ActualIsReg )

						/*
						** Formal and actual parameters are not
						** registers, so just push the memory object
						** onto the stack.
						*/

						PushActualValue( actual, theProc->u.proc.use );

					_elseif( FormalIsReg && !ActualIsReg )

						/*
						** Formal parameter is a register, actual
						** parameter is a memory location, move the
						** memory location into the register.
						*/

						StoreAdrs( adrs, actual );
						Emit2rm
						( 
							"mov", 
							formal->StaticName, 
							adrs  
						);

					_elseif( !FormalIsReg && ActualIsReg )

						/*
						** Formal parameter is memory, the actual
						** parameter is a register or static object.  
						** Push the actual onto the stack.
						*/
						PushReg32( actual->adrs.Sym->StaticName );

					_else

						/*
						** Formal and actual parameters are both
						** registers.  If they are not the same
						** register, move the actual register to
						** the formal register.
						*/

						_if
						( 
								actual->adrs.Sym->StaticName == NULL
							||	_strne
								( 
									formal->StaticName, 
									actual->adrs.Sym->StaticName 
								)
						)

							Emit2rr
							( 
								"mov", 
								formal->StaticName, 
								actual->adrs.Sym->StaticName  
							);

						_endif


					_endif


				_else

					char msg[128];
					
					sprintf
					(
						msg,
						"Parameter type mismatch (%s)",
						formal->TrueName
					);
					yyerror( msg );

				_endif




			/*
			** If the actual parameter is itself a parameter
			** that was passed into the current routine and the
			** code is passing it on as a parameter to another
			** routine, the code generation depends upon the
			** class of the incoming and outgoing parameters.
			** The outgoing class is "Pass by Value".  The following
			** switch statment handles the incoming classes.
			*/

			_elseif( actual->adrs.SymClass == cParm )

				int					SizeToReserve;
				char				address[ 256 ];
				char				anonadrs[ 256 ];
				char				Value[ 32 ];


				_switch( actual->adrs.pClass )

					/*
					** Actual parameter was passed in by reference
					** and we're passing it to the new procedure
					** by value.  This code must dereference the
					** pointer passed to us and pass the value on
					** through.
					*/


					_case( refp_pc )


						/*
						** Set aside an appropriate amount of space on
						** the stack for this parameter.  Also, we're
						** going to need at least one register to play
						** with, so reserve ECX for this purpose (ECX
						** was chosen because it contains byte/word
						** components *and* is used by string instrs).
						*/

						SizeToReserve = ( formal->ObjectSize + 3 ) & ~3;
						sprintf( Value, "%u", SizeToReserve );

						/*
						** Note: for reference parameter, a pointer to
						** the type appears in the adrs.BaseType field.
						*/
							
						_if( actual->adrs.BaseType != NULL )
						
							aType = GetBaseType( actual->adrs.BaseType );
							apType = actual->adrs.BaseType->pType;
							aSize = actual->adrs.BaseType->ObjectSize;
							
						_else
						
							yyerror
							( 
								"Cannot pass untyped reference parameter "
								"as a value parameter"
							);
							aType = &dword_ste;
							apType = tDWord;
							aSize = 4;
							
						_endif;

						_if
						(
								(
										formal->pType == apType
									&&	formal->pType != tArray
								)
							||	(
										formal->pType == tArray
									&&	fbt == aType
									&&	formal->ObjectSize == aSize
								)
							||	(
										actual->adrs.ObjectSize == 16 
									&&	formal->pType == tLWord
								)
							||	(
										actual->adrs.ObjectSize == 8 
									&&	formal->pType == tQWord
								)
							||	(
										actual->adrs.ObjectSize == 4 
									&&	formal->pType == tDWord
								)
							||	(
										aSize == 2 
									&&	formal->pType == tWord
								)
							||	(
										aSize == 1 
									&&	formal->pType == tByte
								)
								
							||	(
										formal->ObjectSize == 16 
									&&	apType == tLWord
								)
							||	(
										formal->ObjectSize == 8 
									&&	apType == tQWord
								)
							||	(
										formal->ObjectSize == 4 
									&&	apType == tDWord
								)
							||	(
										formal->ObjectSize == 2 
									&&	apType == tWord
								)
							||	(
										formal->ObjectSize == 1 
									&&	apType == tByte
								)
						)


							/*
							** Okay, the type check is complete,
							** get the address.
							*/

							StoreAdrs( address, actual );


							/*
							** Special case some of the more
							** common sizes so we can generate
							** better code for them.
							*/

							_if( formal->StaticName != NULL )

								/*
								** Okay, the parameter is going into
								** a register, so this is easy.
								** Note: if the register length is three, 
								** then we've got a 32-bit register.
								*/

								_if( strlen( formal->StaticName ) == 3 )

									/*
									** Pass value in the specified 32-bit
									** register (assume it's 32-bits if
									** the length is three).
									*/

									_if( _strne( formal->StaticName, address ))

										Emit2rm
										( 
											"mov",
											formal->StaticName,
											address 
										);
										sprintf
										( 
											anonadrs, 
											"[%s]", 
											formal->StaticName 
										);
										Emit2rm
										(
											"mov",
											formal->StaticName,
											anonadrs 
										);

									_else

										/*
										** If the formal and actual
										** parameters are the same, this
										** is easy.
										*/

										sprintf
										( 
											anonadrs, 
											"[%s]", 
											formal->StaticName 
										);
										Emit2rm
										(
											"mov",
											formal->StaticName,
											anonadrs 
										);

									_endif


								// If the register length is not three, then
								// it's a 16 or 8 bit register.

								_elseif
								( 
										stricmp( formal->StaticName, "ax" ) 
									==	0 
								)

									/*
									** Pass value in AX.

									*/
									
									char *mem = theProc->u.proc.use;
									char dest[32];

									_if( mem == NULL )
									
										Emit1R( "push", "ebx" );
										mem = "ebx";
										
									_endif
										
									Emit2rm
									( 
										"mov", 
										mem, 
										address 
									);
									sprintf( dest, "word ptr [%s]", mem );
									Emit2rm
									( 
										"mov", 
										"ax", 
										dest  
									);
									
									_if( theProc->u.proc.use == NULL )
									 
										Emit1R( "pop", "ebx" );
										
									_endif

								_elseif
								( 
										stricmp( formal->StaticName, "al" ) 
									==	0 
								)

									/*
									** Pass value in AL.
									*/

									char *mem = theProc->u.proc.use;
									char dest[32];

									_if( mem == NULL )
									
										Emit1R( "push", "ebx" );
										mem = "ebx";
										
									_endif
									Emit2rm
									( 
										"mov", 
										"ebx", 
										address 
									);
									sprintf( dest, "byte ptr [%s]", mem );
									Emit2rm
									( 
										"mov", 
										"al", 
										dest  
									); 
									_if( theProc->u.proc.use == NULL )
									 
										Emit1R( "pop", "ebx" );
										
									_endif

								_elseif
								( 
										stricmp( formal->StaticName, "ah" ) 
									==	0 
								)
								
									char *mem = theProc->u.proc.use;
									char dest[64];

									/*
									** Pass value in AH.
									*/

									_if( mem == NULL )
									
										Emit1R( "push", "ebx" );
										mem = "ebx";
										
									_endif
									Emit2rm
									( 
										"mov", 
										"ebx", 
										address 
									);
									sprintf( dest, "byte ptr [%s]", mem );
									Emit2rm
									( 
										"mov", 
										"ah", 
										dest  
									); 
									_if( theProc->u.proc.use == NULL )
									 
										Emit1R( "pop", "ebx" );
										
									_endif


								_else

									/*
									** Pass the data in some register
									** other than the accumulator.
									*/

									_if
									( 
											theProc->u.proc.use == NULL
										||	_strne( theProc->u.proc.use, "eax" )
									)
									
										Emit1R( "push", "eax" );
										
									_endif
									Emit2rm
									( 
										"mov", 
										"eax", 
										address 
									);
									Emit2rm
									( 
										"mov", 
										formal->StaticName, 
										"byte ptr [eax]"  
									); 
									_if
									( 
											theProc->u.proc.use == NULL
										||	_strne( theProc->u.proc.use, "eax" )
									)
									
										Emit1R( "pop", "eax" );
										
									_endif

								_endif
			
									
							_elseif( formal->ObjectSize == 1 )

								Pushd( 0 );
								Emit1R( "push", "eax" );
								Emit2rm
								( 
									"mov", 
									"eax", 
									address  
								);
								Emit2rm
								( 
									"mov", 
									"al", 
									"byte ptr [eax]"  
								);
								Emit2mr
								( 
									"mov", 
									"byte ptr [esp+4]", 
									"al"  
								);
								Emit1R( "pop", "eax" );
									

							_elseif( formal->ObjectSize == 2 )

								Pushd( 0 );
								Emit1R( "push", "ecx" );
								Emit2rm
								( 
									"mov", 
									"ecx", 
									address  
								);
								Emit2rm( "mov", "cx", "word ptr [ecx]"  );
								Emit2mr( "mov", "word ptr [esp+4]", "cx" );
								Emit1R( "pop", "ecx" );

							_elseif( formal->ObjectSize == 3 )

								Emit1R( "push", "eax" );
								Pushw( 0 );
								Emit2rm
								( 
									"mov", 
									"eax", 
									address  
								);
								Emit1M( "push", "word ptr [eax]" );
								Emit2rm( "mov", "al", "byte ptr [eax+2]");
								Emit2mr( "mov", "byte ptr [esp+2]", "al");
								Emit2rm
								( 
									"mov", 
									"eax", 
									"dword ptr [esp+4]"  
								);
								Emit1R( "pop", "dword ptr [esp]" );


							_elseif( formal->ObjectSize == 4 )

								_if( theProc->u.proc.use == NULL )
								
									Pushd( 0 );
									Emit1R( "push", "eax" );
									Emit2rm
									( 
										"mov", 
										"eax", 
										address  
									);
									Emit2rm
									( 
										"mov", 
										"eax", 
										"dword ptr [eax]"  
									);
									Emit2mr
									( 
										"mov", 
										"dword ptr [esp+4]", 
										"eax"  
									);
									Emit1R( "pop", "eax" );
									
								_else

									char dest[64];
									
									Emit2rm
									( 
										"mov", 
										theProc->u.proc.use, 
										address  
									);
									sprintf
									( 
										dest, 
										"dword ptr [%s]", 
										theProc->u.proc.use 
									);
									Emit1M( "push", dest );
									
								
								_endif

							_elseif( formal->ObjectSize == 8 )

								_if( theProc->u.proc.use == NULL );
								
									Emit2rm( "lea", "esp", "[esp-16]" );
									Emit2mr
									( 
										"mov", 
										"dword ptr [esp+4]", 
										"edx"  
									);
									Emit2mr
									(
										"mov",
										"dword ptr [esp]",
										"eax" 
									);
									Emit2rm
									(
										"mov",
										"eax",
										address 
									);
									Emit2rm
									(
										"mov",
										"edx",
										"dword ptr [eax]" 
									);
									Emit2rm
									(
										"mov",
										"eax",
										"dword ptr [eax+4]" 
									);
									Emit2mr
									(
										"mov",
										"dword ptr [esp+8]",
										"edx" 
									);
									Emit2mr
									(
										"mov",
										"dword ptr [esp+12]",
										"eax" 
									);
									Emit1R( "pop", "eax" );
									Emit1R( "pop", "edx" );
									
								_else
								
									char dest[64];
									
									Emit2rm
									( 
										"mov",
										theProc->u.proc.use,
										address 
									);
									sprintf
									( 
										dest, 
										"dword ptr [%s+4]", 
										theProc->u.proc.use 
									);
									Emit1M( "push", dest );
									sprintf
									( 
										dest, 
										"dword ptr [%s]", 
										theProc->u.proc.use 
									);
									Emit1M( "push", dest );
									
								_endif

							_elseif( formal->ObjectSize == 10 )


								Emit1R( "push", "eax" );
								Emit2rm
								( 
									"mov", 
									"eax", 
									address  
								);
								Emit1M( "push", "dword ptr [eax+4]" ); 
								Emit1M( "push", "dword ptr [eax+0]" );
								Emit1M( "push", "word ptr [eax+8]" );
								Emit2rm
								( 
									"mov", 
									"eax", 
									"dword ptr [esp+10]"  
								);
								Emit1M( "pop", "word ptr [esp+8]" ); 


							/*
							** Special case for parameters that
							** are an even multiple of four bytes
							** in length and are less than or equal
							** to 64 bytes long.
							*/

							_elseif
							( 
									formal->ObjectSize <= 64
								&&	(formal->ObjectSize & 3) == 0
							)

								Emit1R( "push", "eax" );
								Emit2rm
								( 
									"mov", 
									"eax", 
									address 
								);
								_for
								( 
									int i=formal->ObjectSize-8, 
									i >= 0, 
									i-=4 
								)

									sprintf
									( 	
										anonadrs, 
										"dword ptr [eax+%d]", 
										i 
									);
									Emit1M( "push", anonadrs  );

								_endfor
								sprintf
								( 
									anonadrs, 
									"dword ptr [eax+%d]", 
									formal->ObjectSize-4
								);
								Emit1M( "push", anonadrs );
								sprintf
								( 
									anonadrs, 
									"dword ptr [esp+%d]", 
									formal->ObjectSize
								);
								Emit2rm( "mov", "eax", anonadrs  );
								sprintf
								( 
									anonadrs, 

									"dword ptr [esp+%d]", 
									formal->ObjectSize-4
								);
								Emit1M( "pop", anonadrs );


							/*
							** Okay, we're done with the special cases,
							** Now the code starts to get ugly.
							**
							**	First, handle a parameter less than 64
							** bytes in length (that must not be an
							** even multiple of four bytes at this point).
							*/

							_elseif( formal->ObjectSize <= 64 )

								unsigned long index;


								/*
								** Figure out how many dwords to push
								*/

								index = formal->ObjectSize >> 2;

								Emit1R( "push", "eax" );
								Emit2rm
								( 
									"mov", 
									"eax", 
									address 
								);

								/*
								** Push all the whole dwords in the value:
								*/

								_for
								( 
									int i = (formal->ObjectSize & ~3) - 4, 
									i >= 0, 
									i-=4 
								)

									sprintf
									( 
										anonadrs, 
										"dword ptr [eax+%d]", 
										i 
									);
									Emit1M( "push", anonadrs );

								_endfor

								/*
								** If there are at least two leftover bytes, 
								** deal with them here.
								*/

								
								_if( (formal->ObjectSize & 3) == 3 )

									Emit2rm( "lea", "esp", "[esp-2]" );
									sprintf
									( 
										anonadrs, 
										"word ptr [eax+%d]", 
										index << 2 
									);
									Emit1M( "push", anonadrs );
									sprintf
									( 
										anonadrs, 
										"word ptr [eax+%d]", 
										(index << 2) + 2 
									);
									Emit2rm( "mov", "al", anonadrs );
									Emit2mr
									( 
										"mov", 
										"byte ptr [esp+2]", 
										"al"  
									);
									 
								_elseif( (formal->ObjectSize & 3) == 2 )

									Pushw( 0 );
									sprintf
									(
										anonadrs,
										"word ptr [eax+%d]",
										index << 2
									);
									Emit1M( "push", anonadrs );


								/*
								** If this value had an odd number of bytes...
								*/

								_elseif( (formal->ObjectSize & 3) == 1 )

									sprintf
									(
										anonadrs,
										"byte ptr [eax+%d]",
										formal->ObjectSize - 1
									);
									Emit2rm( "mov", "al", anonadrs );
									Emit1R( "push", "eax" );

								_endif

								sprintf
								(
									anonadrs,
									"dword ptr [esp+%d]",
									(index << 2) + 4
								);
								Emit2rm( "mov", "eax", anonadrs );
								sprintf
								(
									anonadrs,
									"dword ptr [esp+%d]",
									(index << 2)
								);
								Emit1M( "pop", anonadrs );



							/*
							** Okay, the parameter is more than 64 bytes.
							** Let's use the movs instructions to copy
							** the parameter.
							*/

							_else

								unsigned long Size;

								Size = (formal->ObjectSize + 3) & ~3;
								sprintf( anonadrs, "[esp-%d]", Size );
								Emit2rm( "lea", "esp", anonadrs );
								Emit1R( "push", "ecx" );
								Emit1R( "push", "esi" );
								Emit1R( "push", "edi" );
								Emit0( "pushfd" );
								Emit0( "cld" );
								Emit2rm( "lea", "edi", "[esp+16]" );
								sprintf
								( 
									anonadrs, 
									"%d", 
									formal->ObjectSize >> 2 
								);
								Emit2rm
								( 
									"mov", 
									"esi", 
									address  
								);
								Emit0( "rep\tmovsd" );

								_if( (formal->ObjectSize & 2) == 2 )
								
									Emit0( "movsw" );
									
								_endif
								 
								_if( (formal->ObjectSize & 1) == 1 )
								
									Emit0( "movsb" );
									
								_endif
								Emit0( "popfd" );
								Emit1R( "pop", "edi" );
								Emit1R( "pop", "esi" );
								Emit1R( "pop", "ecx" );

							_endif


						_else

							yyerror( "Reference parameter type mismatch" );

						_endif

					_endcase



					
					_case( vrp_pc )
					_case( result_pc )
					_case( name_pc )
					_case( lazy_pc )

						yyerror
						(
							"You must explicitly copy the data for this\n"
							"parameter class (value/result, result, name,\n"
							"or lazy) to the called procedure"
						);

					_endcase



					_default

						yyerror( "Illegal parameter class" );

				_endswitch

			_else
			
				char msg[128];

				sprintf
				(
					msg,
					"Parameter class/type mismatch (%s)",
					formal->TrueName
				);
				yyerror( msg );

			_endif

		_endcase



		/*
		** Handle parameters passed by reference, value/result, or
		** result here (we're passing in the address of the actual
		** parameter).
		*/

		_case( refp_pc )
		_case( vrp_pc )
		_case( result_pc )

			/*
			** If the actual parameter is a static variable,
			** a local variable, or a parameter passed into the
			** current procedure by value, take the address of that
			** object and pass the address as the parameter.
			*/
			
			_if( actual->adrs.SymClass == cStatic )

				char adrs[128];

				_if( valPrefix )
				
					_if
					(
							(
									actual->adrs.Size == 4
								&&	(
											actual->adrs.pType == tDWord
										||	IsStr( actual->adrs.pType )
									)
							)
						||	actual->adrs.pType == tPointer
					)
					
						PushActualValue( actual, theProc->u.proc.use );
						
					_else
					
						yyerror
						( 
							"With VAL prefix, operand must be a pointer "
							" or DWORD"
						);
						
					_endif
					
				
				
				_elseif
				( 
						formal->Type == actual->adrs.Type
					||	(
								formal->pType == tArray
							&&	fbt == abt
						)
					||	(
								actual->adrs.pType == tArray
							&&	fbt == abt
						) 
					||	formal->Type == &variant_ste
					
					||	(
								actual->adrs.pType == tByte
							&&	formal->Type->ObjectSize == 1
						)
					||	(
								actual->adrs.pType == tWord
							&&	formal->Type->ObjectSize == 2
						)
					||	(
								actual->adrs.pType == tDWord
							&&	formal->Type->ObjectSize == 4
							&&	formal->pType != tPointer
						)
					||	(
								actual->adrs.pType == tQWord
							&&	formal->Type->ObjectSize == 8
						)
					||	(
								actual->adrs.pType == tLWord
							&&	formal->Type->ObjectSize == 16
						)
						
						
					||	(
								actual->adrs.Size == 1
							&&	formal->pType == tByte
						)
					||	(
								actual->adrs.Size == 2
							&&	formal->pType == tWord
						)
					||	(
								actual->adrs.Size == 4
							&&	formal->pType == tDWord
						)
					||	(
								actual->adrs.Size == 8
							&&	formal->Type->pType == tQWord
						)
					||	(
								actual->adrs.Size == 16
							&&	formal->Type->pType == tLWord
						)
						
						
					||	(
								actual->adrs.pType == tPointer
							&&	actual->adrs.Sym != NULL
							&&	GetBaseType( actual->adrs.Sym->Type ) == fbt
						)
				)
				


					_if( formal->StaticName == NULL )

						/*
						** Okay, the formal parameter is not
						** in a register, so pass it on the stack.
						*/

						/* Fix ME!  Should just be a PUSH */

						_if
						( 
								actual->adrs.IndexReg == NULL 
							&&	actual->adrs.BaseReg == NULL 
						)
						
						
							//StoreAdrsSize( adrs, actual, 0 );
							MakeAdrsStr( adrs, actual, 0, 0 );
							PushStaticAdrs( adrs );
						
						_else
						
							StoreAdrsSize( adrs, actual, 0 );
							Emit1R( "push", "eax" );
							Emit1R( "push", "eax" );
							Emit2rm( "lea", "eax", adrs );
							Emit2mr( "mov", "dword ptr [esp+4]", "eax" );
							Emit1R( "pop", "eax" );
						
						_endif


					_else

						/*
						** Okay, the formal parameter is in a register, 
						** so pass it in that register.
						*/

						_if
						( 
								actual->adrs.BaseReg == NULL 
							&&	actual->adrs.IndexReg == NULL 
						)

							// If it's a static address, use a
							// simple MOV instruction.


							StoreAdrsOfs( adrs, actual, 1 );
							Emit2rc
							( 
								"mov", 

								formal->StaticName, 
								adrs 
							);  

						_else

							// If the address involves index or base
							// registers, use the LEA instruction.

							StoreAdrsSize( adrs, actual, 0 );
							Emit2rm
							( 
								"lea",
								formal->StaticName,
								adrs 
							); 

						_endif

					_endif
					
				_elseif
				( 
						actual->adrs.pType == tPointer
				)

					/*
					** Okay, we've got a pointer to the object.
					** Dereference the pointer and pass that address.
					*/				   

					PushActualValue( actual, theProc->u.proc.use );

				_else
				
					char msg[128];
					
					sprintf
					(
						msg,
						"Static reference parameter type mismatch (%s)",
						formal->TrueName
					);

					yyerror( msg );

				_endif

			_elseif
			(
			 		actual->adrs.SymClass == cVar 
				||	(
							actual->adrs.SymClass == cParm
						&&	actual->adrs.pClass == valp_pc
					)
			)


				_if( valPrefix )
				
					_if
					(
							(
									actual->adrs.Size == 4
								&&	(
											formal->pType == tDWord
										||	formal->pClass != valp_pc
									)
							)
						||	actual->adrs.pType == tPointer
					)
					
						PushActualValue( actual, theProc->u.proc.use );
						
					_else
					
						yyerror
						( 
							"With VAL prefix, operand must be a pointer "
							" or DWORD"
						);
						
					_endif
					
				
				
				/*
				** Be sure the formal and actual parameter
				** types agree.
				*/


				_elseif
				(
						formal->Type == &variant_ste
					||	(
								formal->pType == actual->adrs.pType
							&&	formal->pType != tArray
						)
					||	(
								(
										formal->pType == tArray
									||	actual->adrs.pType == tArray
								)
							&&	fbt == abt
							// &&	fSize == aSize // Allow any size array!
						)
					||	(
								aSize == 16 
							&&	formal->pType == tLWord
						)
					||	(
								aSize == 8 
							&&	formal->pType == tQWord
						)
					||	(
								aSize == 4 
							&&	formal->pType == tDWord
						)
					||	(
								aSize == 2 
							&&	formal->pType == tWord
						)
					||	(
								aSize == 1 
							&&	formal->pType == tByte
						)
						
					||	(
								fSize == 16 
							&&	actual->adrs.pType == tLWord
						)
					||	(
								fSize == 8 
							&&	actual->adrs.pType == tQWord
						)
					||	(
								fSize == 4 
							&&	actual->adrs.pType == tDWord
						)
					||	(
								fSize == 2 
							&&	actual->adrs.pType == tWord
						)
					||	(
								fSize == 1 
							&&	actual->adrs.pType == tByte
						)
				)

					/*
					** If we've got an anonymous memory reference
					** of the form "(type xxxx [reg32])" then just
					** push the register.  Otherwise take the address
					** of the operand.
					*/

					_if
					( 
							actual->adrs.Sym == NULL
						&&	actual->adrs.BaseReg != NULL
						&&	actual->adrs.IndexReg == NULL
						&&	actual->adrs.Disp == 0
						&&  (
									actual->adrs.StaticName == NULL
								||	*actual->adrs.StaticName == '\0'
							)
					)

						_if( formal->StaticName != NULL )

							_if
							( 
								_strne
								( 
									formal->StaticName, 
									actual->adrs.BaseReg 
								)
							)

								/*
								** If the base register is not the
								** same as the parameter IN <REG>
								** value, the move the base register
								** into the parameter register.  If the
								** two registers are the same, don't
								** do anything.
								*/

								Emit2rm
								( 
									"mov", 
									formal->StaticName,
									actual->adrs.BaseReg 
								);

							_endif

						_else

							Emit1R( "push", actual->adrs.BaseReg );

						_endif

					_else
					
						/*
						** We've got a more complex addressing mode
						*/

						char adrs[256];

						StoreAdrsSize( adrs, actual, 0 );
						_if( formal->StaticName != NULL )

							/*
							** Parameter is passed in a register:
							*/


							Emit2rm
							(
								"lea",
								formal->StaticName,
								adrs 
							);

						_else

							/*
							** Parameter is not passed in a register,
							** so pass it on the stack:
							*/

							_if( theProc->u.proc.use == NULL )
							
								_if
								( 
										actual->adrs.IndexReg == NULL 
									&&	actual->adrs.BaseReg != NULL
								)
								
									// If it's just a simple local variable
									// or parameter, and we're passing it
									// by reference to the caller, then
									// push EBP and add the variable's offset
									// to the value pushed onto the stack:
								
									char operand[ 64];
								
									sprintf( operand, "%d", actual->adrs.Disp );
									Emit1R( "push", actual->adrs.BaseReg );
									Emit2mc
									( 
										"add", 
										"dword ptr [esp]", 
										operand  
									);
									
								
								
								_else
								
									// If it's a fancier address, then use
									// the LEA instruction to pass the address
									// on the stack.
								
									Emit1R( "push", "eax" );
									Emit1R( "push", "eax" );
									Emit2rm
									(
										"lea",
										"eax",
										adrs 
									);
									Emit2mr
									( 
										"mov", 
										"dword ptr [esp+4]", 
										"eax"  
									);
									Emit1R( "pop", "eax" );
									
								_endif
								
							_else
							
								Emit2rm
								(
									"lea",
									theProc->u.proc.use,
									adrs 
								);
								Emit1R( "push", theProc->u.proc.use );
								
							_endif

						_endif

					_endif

				_elseif( actual->adrs.pType == tPointer )

					/*
					** Okay, we've got a pointer to the object.
					** Dereference the pointer and pass that address.
					*/

					PushActualValue( actual, theProc->u.proc.use );


				_else

					char msg[ 256 ];

					sprintf
					(
						msg,
						"Parameter type mismatch: %s(formal:%s)\n"
						"is passed as type %s and should be type %s",
						actual->adrs.Sym->TrueName,
						formal->TrueName,
						actual->adrs.Type->TrueName,
						formal->Type->TrueName
					);
					yyerror( msg );

				_endif


			/*
			** If the actual parameter is itself a parameter
			** that was passed into the current routine and the
			** code is passing it on as a parameter to another
			** routine, the code generation depends upon the
			** class of the incoming and outgoing parameters.
			** The outgoing class is "Pass by Reference".  The following
			** switch statment handles the incoming classes.
			*/

			_elseif( actual->adrs.SymClass == cParm )

				struct SymNode *abt = actual->adrs.BaseType;

				
				/*
				** Actual parameter type could be an isomophism of
				** an array type.  Formal parameter has already
				** dereferenced the link.  So we need to do the same
				** to the actual parameter here.
				*/

				_if( abt != NULL && abt->pType == tArray )

					abt = GetBaseType( abt );

				_endif
 
				_switch( actual->adrs.pClass )

					/*
					** Actual parameter was passed in by reference
					** and we're passing it to the new procedure
					** by reference.  This code must simply copy the
					** pointer through.
					*/

					_case( refp_pc )

						_if
						(
								formal->Type == &variant_ste
							||	(
										fbt == abt
									&&	fSize == 
											actual->adrs.BaseType->ObjectSize
								)

							||	(
										actual->adrs.BaseType->ObjectSize == 4 
									&&	formal->pType == tDWord
								)

							||	(
										actual->adrs.BaseType->pType == tDWord
									&&	formal->ObjectSize == 4
								)
						)


							_if( formal->StaticName != NULL )

								char adrs[ 256 ];

								/*
								** Formal parameter is a register, so pass
								** the address in that register.
								*/

								StoreAdrs( adrs, actual );
								Emit2rm
								( 
									"mov",
									formal->StaticName,
									adrs 
								);

							_else

								char adrs[ 128 ];

								/*
								** Formal parameter is not a register
								** parameter, so push the address onto
								** the stack.
								*/

								StoreAdrsForceSize( adrs, actual, 4 );
								Emit1M( "push", adrs );

							_endif

						_else
							
							sprintf
							( 
								adrs, 
								"Reference parameter type mismatch for %s, "
								"expected type %s",
								formal->TrueName,
								formal->Type->TrueName
							);
								

							yyerror( adrs );

						_endif

					_endcase



					
					_case( vrp_pc )
					_case( result_pc )
					_case( name_pc )
					_case( lazy_pc )

						yyerror
						(
							"You must explicitly copy the address for this\n"
							"parameter class (value/result, result, name,\n"
							"or lazy) to the called procedure"
						);

					_endcase



					_default

						yyerror( "Illegal parameter class" );

				_endswitch

			_else

				yyerror( "Parameter type mismatch" );

			_endif

		_endcase




		/*
		** Handle parameters passed by name and by lazy eval here.
		*/

		_case( name_pc )
		_case( lazy_pc )

			_if
			(
					formal->pClass == actual->adrs.pClass 
				&&	(
							actual->adrs.pClass == name_pc 
						||	actual->adrs.pClass == lazy_pc
					)
			)

				char adrsHO[128];
				char adrsLO[128];


				actual->adrs.Disp += 4;
				StoreAdrsForceSize( adrsHO, actual, 4 );
				actual->adrs.Disp -= 4;
				StoreAdrsForceSize( adrsLO, actual, 4 );
				Emit1M( "push", adrsHO );
				Emit1M( "push", adrsLO );

			_else

				yyerror
				( 
					"Must specify an immediate thunk() for pass by name\n"
					"and pass by lazy evaluation" 
				);

			_endif

		_endcase



		_default
			yyerror( "Illegal parameter class" );

	_endswitch

_end( OutputMemParm )







/***********************************************/
/*                                             */
/* EmitRegopReg-                               */
/*                                             */
/* Emits code for the IF and WHILE statements. */
/* Operands are two registers (left and right) */
/* as well as the instruction to return if the */
/* registers are signed or unsigned.           */
/*                                             */
/***********************************************/

char *
EmitRegopReg
( 
	union YYSTYPE *left, 
	union YYSTYPE *right, 
	char *SignedInstr,
	char *UnsignedInstr
)
_begin( EmitRegopReg )


	_if( left->reg.Size == right->reg.Size )


		Emit2rr
		( 
			"cmp",  
			regStrs[ left->reg.encoding ], 
			regStrs[ right->reg.encoding ]  
		);

	_else

		yyerror( "Register sizes must be the same" );

	_endif
	_returnif
	( 
			left->reg.IsSigned
		||	right->reg.IsSigned
		 
	) hlastrdup2( SignedInstr );
	_return hlastrdup2( UnsignedInstr );

_end( EmitRegopReg )







char *
EmitRegopMem
( 
	union YYSTYPE *reg, 
	union YYSTYPE *mem, 
	char *SignedInstr,
	char *UnsignedInstr
)
_begin( EmitRegopMem )

	char adrs[ 256 ];

	_if( (unsigned) reg->reg.Size == (unsigned) mem->adrs.Size )

		StoreAdrs( adrs, mem );
		Emit2rm
		( 
			"cmp", 
			regStrs[ reg->reg.encoding ], 
			adrs  
		);


	_else

		yyerror( "Operand sizes must be the same" );

	_endif
	FreeAdrs( mem );
	_returnif
	( 
			IsInt( mem->adrs.pType )
		||	reg->reg.IsSigned

	) hlastrdup2( SignedInstr );
	_return hlastrdup2( UnsignedInstr );

_end( EmitRegopMem )







char *
EmitMemopReg
( 
	union YYSTYPE *mem, 
	union YYSTYPE *reg, 
	char *SignedInstr,
	char *UnsignedInstr
)
_begin( EmitMemopReg )

	char adrs[ 256 ];

	_if( (unsigned) reg->reg.Size == (unsigned) mem->adrs.Size )

		StoreAdrs( adrs, mem );
		Emit2mr
		( 
			"cmp", 
			adrs, 
			regStrs[ reg->reg.encoding ]  

		);

	_else

		yyerror( "Operand sizes must be the same" );

	_endif
	FreeAdrs( mem );
	_returnif
	( 
			IsInt( mem->adrs.pType )
		||	reg->reg.IsSigned

	) hlastrdup2( SignedInstr );
	_return hlastrdup2( UnsignedInstr );

_end( EmitMemopReg )





static int
IsEightBits( union YYSTYPE *theConst )
_begin( IsEightBits )

	_returnif( numBits( theConst ) > 32 ) 0;
	_returnif( theConst->v.Type->ObjectSize == 1 ) 1;
	_returnif
	(
			IsInt( theConst->v.pType )
		&&	(
					theConst->v.u.intval >= -128
				&&	theConst->v.u.intval <= 127
			)
	) 1;
	_returnif( theConst->v.u.unsval <= 255 ) 1;
	_return 0;

_end( IsEightBits )




static int
Is16Bits( union YYSTYPE *theConst )
_begin( Is16Bits )

	_returnif( numBits( theConst ) > 32 ) 0;
	_returnif( theConst->v.Type->ObjectSize == 2 ) 1;
	_returnif
	(
			IsInt( theConst->v.pType )
		&&	(
					theConst->v.u.intval >= -32768
				&&	theConst->v.u.intval <= 32767
			)
	) 1;
	_returnif( theConst->v.u.unsval <= 65535 ) 1;
	_return 0;

_end( Is16Bits )




void
EmitRegInRange4
( 
	union YYSTYPE *reg, 
	union YYSTYPE *startConst, 
	union YYSTYPE *endConst,
	int target,
	int condition
)
_begin( EmitRegInRange4 )

	int		IsSigned;
	char	label[32];

	assert( startConst->v.Type != NULL );
	assert( endConst->v.Type != NULL );

	IsSigned = 
			IsInt( startConst->v.pType ) 
		||	IsInt( endConst->v.pType )
		||	reg->reg.IsSigned;

	_if( !IsOrdinal( startConst->v.pType ) || !IsOrdinal( endConst->v.pType ))

		yyerror( "Expected an ordinal constant" );

	_elseif
	(
			reg->reg.Size == startConst->v.Type->ObjectSize
		&&	reg->reg.Size == endConst->v.Type->ObjectSize 
		||	(
					reg->reg.Size == 1
				&&	IsEightBits( startConst )
				&&	IsEightBits( endConst )
			)
		||	(
					reg->reg.Size == 2
				&&	Is16Bits( startConst )
				&&	Is16Bits( endConst )
			)
	)

		_if( condition )

			/*
			** If condition is true, then we will jump
			** to the %d_false label if the value is
			** within the specified range.
			** Otherwise we fall through to the %d_true
			** label (yes, the choice of target labels
			** is non-intuitive, this is forced on us
			** by the fact that IF/WHILE/UNTIL/etc emit
			** the target label and the true/false labels
			** make more sense in the context of those stmts.)
			*/
			
			EmitIRC( "cmp", regStrs[ reg->reg.encoding], YYS startConst );
			sprintf( label, "L%d_true" sympost, LblCntr );
			Emit1L( _ifx( IsSigned, "jl", "jb" ), label );

			EmitIRC( "cmp", regStrs[ reg->reg.encoding], YYS endConst );
			sprintf( label, "L%d_false" sympost, target );
			Emit1L( _ifx( IsSigned, "jng", "jna" ), label );

			EmitStmtLblNum( "L%d_true" sympost, LblCntr ); 
			
		_else
		
			/*
			** If condition is false, transfer control to
			** the %d_false label if the value is within
			** the specified range, otherwise fall through
			** to the %d_true label.
			*/

			EmitIRC( "cmp", regStrs[ reg->reg.encoding], YYS startConst );
			sprintf( label, "L%d_false" sympost, target );
			Emit1L( _ifx( IsSigned, "jl", "jb" ), label );

			EmitIRC( "cmp", regStrs[ reg->reg.encoding], YYS endConst );
			sprintf( label, "L%d_false" sympost, target );
			Emit1L( _ifx( IsSigned, "jg", "ja" ), label );
			

		_endif

	_else

		yyerror( "Operand sizes must be the same" );

	_endif
	FreeValue( startConst );
	FreeValue( endConst );
	++LblCntr;

_end( EmitRegInRange4 )





void
EmitMemInRange4
( 
	union YYSTYPE *adrs, 
	union YYSTYPE *startConst, 
	union YYSTYPE *endConst,
	int target,
	int condition
)
_begin( EmitMemInRange4 )

	int 	IsSigned;
	char	label[ 32 ];
	char	address[ 256 ];

	assert( startConst->v.Type != NULL );
	assert( endConst->v.Type != NULL );

	IsSigned = 
			IsInt( startConst->v.pType ) 
		||	IsInt( endConst->v.pType )
		||	IsInt( adrs->adrs.pType );

	_if( !IsOrdinal( startConst->v.pType ) || !IsOrdinal( endConst->v.pType ))

		yyerror( "Expected an ordinal constant" );

	_elseif
	(
			adrs->adrs.ObjectSize == startConst->v.Type->ObjectSize
		&&	adrs->adrs.ObjectSize == endConst->v.Type->ObjectSize 
		||	(
					adrs->adrs.ObjectSize == 1
				&&	IsEightBits( startConst )
				&&	IsEightBits( endConst )
			)
		||	(
					adrs->adrs.ObjectSize == 2
				&&	Is16Bits( startConst )
				&&	Is16Bits( endConst )
			)
	)

		StoreAdrs( address, adrs );
		_if( condition )

			/*
			** If condition is true, then we will jump
			** to the %d_false label if the value is
			** within the specified range.
			** Otherwise we fall through to the %d_true
			** label (yes, the choice of target labels
			** is non-intuitive, this is forced on us
			** by the fact that IF/WHILE/UNTIL/etc emit
			** the target label and the true/false labels
			** make more sense in the context of those stmts.)
			*/
			
			EmitIMC( "cmp", address, YYS startConst );
			sprintf( label, "L%d_true" sympost, LblCntr );
			Emit1L( _ifx( IsSigned, "jl", "jb" ), label );
			
			EmitIMC( "cmp", address, YYS endConst );
			sprintf( label, "L%d_false" sympost, target );
			Emit1L( _ifx( IsSigned, "jng", "jna" ), label );
			
			EmitStmtLblNum( "L%d_true" sympost, LblCntr ); 
			
		_else
		
			/*
			** If condition is false, transfer control to
			** the %d_false label if the value is within
			** the specified range, otherwise fall through
			** to the %d_true label.
			*/

			EmitIMC( "cmp", address, YYS startConst );
			sprintf( label, "L%d_false" sympost, target );
			Emit1L( _ifx( IsSigned, "jl", "jb" ), label );
			
			EmitIMC( "cmp", address, YYS endConst );
			sprintf( label, "L%d_false" sympost, target );
			Emit1L( _ifx( IsSigned, "jg", "ja" ), label );
			

		_endif

	_else

		yyerror( "Operand sizes must be the same" );

	_endif
	FreeAdrs( adrs );
	FreeValue( startConst );
	FreeValue( endConst );
	++LblCntr;

_end( EmitMemInRange4 )





void
EmitRegInRange5
( 
	union YYSTYPE *reg, 
	union YYSTYPE *v, 
	int condition, 
	int label 
)
_begin( EmitRegInRange5 )

	enum regnums regToUse;
	char sn[16];
	char slabel[32];

	_if( v->v.pType != tCset )

		yyerror( "Expected a character set constant" );

	_elseif( reg->reg.Size != 1 )

		yyerror( "Expression requires an eight-bit register" );

	_else


		_if( reg->reg.encoding == reg_al || reg->reg.encoding == reg_ah )

			regToUse = reg_eax;

		_elseif( reg->reg.encoding == reg_bl || reg->reg.encoding == reg_bh )

			regToUse = reg_ebx;


		_elseif( reg->reg.encoding == reg_cl || reg->reg.encoding == reg_ch )

			regToUse = reg_ecx;

		_else

			regToUse = reg_edx;

		_endif
		startStrSeg();
		sprintf( sn, "L%d_cset" sympost, label );
		OutValue( sn, v->v.Type, v );
		endStrSeg();
		Emit1R( "push", regStrs[ regToUse ]  );
		Emit2rr
		( 
			"movzx", 
			regStrs[ regToUse ], 
			regStrs[ reg->reg.encoding ]  
		);
		LabelToMem( slabel, sn, "dword" );
		Emit2mr( "bt", slabel, regStrs[ regToUse ]  );
		Emit1R( "pop", regStrs[ regToUse ]  );
		sprintf( sn, "L%d_false" sympost, label );
		Emit1L( _ifx( condition, "jc", "jnc" ), sn  );

	_endif

_end( EmitRegInRange5 )









void
EmitRegInRange6
( 
	union YYSTYPE *reg, 
	union YYSTYPE *adrs, 
	int condition, 
	int label 
)
_begin( EmitRegInRange6 )

	enum regnums regToUse;
	char sn[256];

	_if( adrs->adrs.pType != tCset )

		yyerror( "Expected a character set variable" );

	_elseif( reg->reg.Size != 1 )

		yyerror( "Expression requires an eight-bit register" );

	_else


		_if( reg->reg.encoding == reg_al || reg->reg.encoding == reg_ah )

			regToUse = reg_eax;

		_elseif( reg->reg.encoding == reg_bl || reg->reg.encoding == reg_bh )

			regToUse = reg_ebx;


		_elseif( reg->reg.encoding == reg_cl || reg->reg.encoding == reg_ch )

			regToUse = reg_ecx;

		_else

			regToUse = reg_edx;

		_endif
		StoreAdrsForceSize( sn, adrs, 4 );
		Emit1R( "push", regStrs[ regToUse ] );
		Emit2rr
		( 
			"movzx", 
			regStrs[ regToUse ], 
			regStrs[ reg->reg.encoding]  
		);
		Emit2mr( "bt", sn, regStrs[ regToUse ]  );
		Emit1R( "pop", regStrs[ regToUse ]  );

		sprintf( sn, "L%d_false" sympost, label );
		Emit1L( _ifx( condition, "jc", "jnc" ), sn  );


	_endif

_end( EmitRegInRange6 )

/*
** OutPaddedValue-
**
**	Outputs an arbitrary data constant to the readonly segment
** with at least four bytes of padding at the end of the data.
*/


static void
OutPaddedValue
( 
	char			*Name, 
	struct	SymNode *Type, 
	union	YYSTYPE *Value 
)
_begin( OutPaddedValue )

	assert( Value != NULL );
	assert( Type != NULL );

	startStrSeg();
	OutValue( Name, Type, Value );
	EmitDwordConst( 0, "\t\t;Padding" );
	EmitAlign(4);
	endStrSeg();

_end( OutPaddedValue )



/*
** EmitCMOVrr - Emits a CMOV instruction with two register operands.
** EmitCMOVmr - Emits a CMOV instruction with memory and register operands.
*/

char*
EmitCMOVrr
( 
	char	*instr,
	char	*srcReg, 
	char	*destReg
)
_begin( EmitCMOVrr )

	Emit2rr( instr, destReg, srcReg );
	free2( vss instr );
	_return destReg;

_end( EmitCMOVrr )





char*
EmitCMOVmr
( 
	char			*instr,
	union YYSTYPE	*adrs, 
	char			*destReg,
	unsigned		Size
)
_begin( EmitCMOVmr )

	char address[128];

	_if( adrs->adrs.Size != Size )

		yyerror( "CMOV operands must be the same size" );

	_else

		StoreAdrs( address, adrs );
		Emit2rm( instr, destReg, address );

	_endif
	free2( vss instr );
	FreeAdrs( adrs );
	_return destReg;

_end( EmitCMOVmr )





/*****************************************************/
/*                                                   */
/* CopyValResParms-                                  */
/*                                                   */
/* This function emits the code to copy Value/Result */
/* parameters from the actual parameter to the local */
/* copy of the value upon entry into the procedure.  */
/*                                                   */
/*****************************************************/


void 
CopyValResParms( struct SymNode *ParmList )
_begin( CopyValResParms )

	struct	SymNode	*s;
	struct	SymNode	*d;
	int 	NeedsMOVS = 0;
	char	adrs[128];


	assert( ParmList != NULL );

	/*
	** First, determine if there are any value/result
	** parameters that exceed 16 bytes in length.
	*/

	s = ParmList->u.proc.parms;
	_while( s != NULL )

		NeedsMOVS = 
				NeedsMOVS 
			||	(
						(s->pClass == vrp_pc )
					&&	s->Type != NULL
					&&	s->Type->ObjectSize > 16
				);

		s = s->Next;

	_endwhile;


	/*
	** Preserve the registers we will need to copy the
	** actual parameter data to the local variables.
	*/

	Emit1R( "push", "esi" );
	Emit1R( "push", "ecx" );
	_if( NeedsMOVS )

		Emit1R( "push", "edi"  );
		Emit0( "pushfd" );
		Emit0( "cld" );

	_endif
	s = ParmList->u.proc.parms;
	_while( s != NULL )

		/*
		** For each value/result parameter, copy its
		** data to the corresponding local variable.
		*/

		_if( s->pClass == vrp_pc  )

			/*
			** Get the source pointer into ESI.
			*/

			sprintf( adrs, "dword ptr [ebp+%d]", s->Offset );
			Emit2rm( "mov", "esi", adrs );
			
			/*
			** Get the address of the destination block.
			*/

			d = lookup( s->TrueName, 1 );
			assert( d != NULL );
			
			/*
			** Emit the code to copy the data.
			** Special case 1-16 byte objects.
			*/

			_switch( _ifx( s->Type != NULL, s->Type->ObjectSize, 4 ) )

				_case( 1 )

					Emit2rm( "mov", "cl", "byte ptr [esi]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "cl" );

				_endcase

				_case( 2 )

					Emit2rm( "mov", "cx", "word ptr [esi]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "cx" );

				_endcase

				_case( 3 )

					Emit2rm( "mov", "cx", "word ptr [esi]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "cx" );
					Emit2rm( "mov", "cl", "byte ptr [esi+2]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+2 );
					Emit2mr( "mov", adrs, "cl" );

				_endcase

				_case( 4 )

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );
				
				_endcase

				_case( 5 )

					Emit2rm( "mov", "cl", "byte ptr [esi+4]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "cl" );

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 6 )
				
					Emit2rm( "mov", "cx", "word ptr [esi+4]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "cx" );
					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 7 )

					Emit2rm( "mov", "cl", "byte ptr [esi+6]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+6 );
					Emit2mr( "mov", adrs, "cl" );

					Emit2rm( "mov", "cx", "word ptr [esi+4]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "cx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 8 )

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 9 )

					Emit2rm( "mov", "cl", "byte ptr [esi+8]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "cl" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 10 )

					Emit2rm( "mov", "cx", "word ptr [esi+8]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "cx" );


					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 11 )

					Emit2rm( "mov", "cl", "byte ptr [esi+10]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "cl" );

					Emit2rm( "mov", "cx", "word ptr [esi+8]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "cx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 12 )


					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+8]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_case( 13 )

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+8]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "cl", "byte ptr [esi+12]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+12 );
					Emit2mr( "mov", adrs, "cl" );

				_endcase

				_case( 14 )

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx"  );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+8]"  );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "cx", "word ptr [esi+12]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset+12 );
					Emit2mr( "mov", adrs, "cx" );

				_endcase

				_case( 15 )

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+8]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "cx", "word ptr [esi+12]" );
					sprintf( adrs, "word ptr [ebp%d]", d->Offset+12 );
					Emit2mr( "mov", adrs, "cx"  );

					Emit2rm( "mov", "cl", "byte ptr [esi+14]" );
					sprintf( adrs, "byte ptr [ebp%d]", d->Offset+14 );
					Emit2mr( "mov", adrs, "cl" );

				_endcase


				_case( 16 )

					Emit2rm( "mov", "ecx", "dword ptr [esi]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+4]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+4 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+8]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+8 );
					Emit2mr( "mov", adrs, "ecx" );

					Emit2rm( "mov", "ecx", "dword ptr [esi+12]" );
					sprintf( adrs, "dword ptr [ebp%d]", d->Offset+12 );
					Emit2mr( "mov", adrs, "ecx" );

				_endcase

				_default

					/*
					** For parameters that are not
					** 1..16 bytes long, use
					** a movs instruction to copy the data.
					*/

					sprintf( adrs, "%d", d->ObjectSize / 4 );
					Emit2rc( "mov", "ecx", adrs );
					sprintf
					( 
						adrs, 
						"[ebp%s%d]",
						_ifx( d->Offset < 0, "", "+" ),
						d->Offset
					);
					Emit2rm( "lea", "edi", adrs );
					_if( (d->ObjectSize & 3) >= 2 )

						Emit0( "movsw" );

					_endif
					_if( d->ObjectSize & 1 )

						Emit0( "movsb" );

					_endif

			_endswitch

		_endif
		s = s->Next;

	_endwhile

	_if( NeedsMOVS )

		Emit0( "popfd" );
		Emit1R( "pop", "edi" );

	_endif
	Emit1R( "pop", "ecx" );
	Emit1R( "pop", "esi" );


_end( CopyValResParms )





/*********************************************
/* 	                                        */
/* ebpPlus-                                 */
/*                                          */
/* Utitility function that creates a string */
/* of the form "[ebp +/- disp]"             */
/*                                          */
/********************************************/
 	 
static void
ebpPlus( int disp, char* dest )
_begin( ebpPlus )

	sprintf
	(
		dest,
		"[ebp%s%d]",
		_ifx( disp < 0, "", "+" ),
		disp
	);

_end( ebpPlus )





/****************************************************/
/*                                                  */
/* StoreValResParms-                                */
/*                                                  */
/* This procedure generates the code that stores    */
/* away local copies of valres or result parameters */
/* into the actual parameters (code is emitted at   */
/* the end of the procedure).                       */
/*                                                  */
/****************************************************/



void 
StoreValResParms( struct SymNode *ParmList )
_begin( StoreValResParms )

	struct	SymNode	*s;
	struct	SymNode	*d;
	char			adrs[6][16];
	int 			NeedsMOVS 		= 0;
	int				HadPassByValRes = 0;



	assert( ParmList != NULL );

	/*
	** First, determine if there are any value/result or result
	** parameters and also see if there are any that exceed 16 bytes 
	** in length.
	*/

	s = ParmList->u.proc.parms;
	_while( s != NULL && s != ParmList )

		NeedsMOVS = 
				NeedsMOVS 
			||	(
						(
								s->pClass == vrp_pc 
							||	s->pClass == result_pc
						)
					&&	s->Type != NULL && s->Type->ObjectSize > 16
				);

		HadPassByValRes =
				HadPassByValRes 
			|| 	s->pClass == vrp_pc 
			||	s->pClass == result_pc;

		s = s->Next;

	_endwhile;


	/*
	** Preserve the registers we will need to copy the
	** actual parameter data to the local variables.
	*/

	_if( HadPassByValRes )

		Emit1R( "push", "edi" );
		Emit1R( "push", "ecx" );
		_if( NeedsMOVS )
			
			Emit1R( "push", "esi" );
			Emit0( "pushfd" );
			Emit0( "cld" );

		_endif

		d = ParmList->u.proc.parms;
		_while( d != NULL )

			_if( d->pClass == vrp_pc || d->pClass == result_pc  )


				/*
				** Get the destination pointer into EDI.
				*/

				ebpPlus( d->Offset, adrs[0] );
				Emit2rm( "mov", "edi", adrs[0] );
				
				/*
				** Get the address of the destination block.
				*/

				s = lookup( d->TrueName, 1 );
				assert( s != NULL );
				
				/*
				** Emit the code to copy the data.
				** Special case 1, 2, 3, 4, and 8 byte objects.
				*/

				_switch( _ifx( d->Type != NULL, d->Type->ObjectSize, 4 ))
				
					_case( 3 )

						ebpPlus( s->Offset+2, adrs[1] );

					_case( 1 )
					_case( 2 )
					_case( 4 )
					_default

						ebpPlus( s->Offset, adrs[0] );

					_endcase

					_case( 7 )

						ebpPlus( s->Offset+6, adrs[2] );

					_case( 5 )
					_case( 6 )
					_case( 8 )

						ebpPlus( s->Offset, adrs[0] );
						ebpPlus( s->Offset+4, adrs[1] );

					_endcase

					_case( 11 )

						ebpPlus( s->Offset+10, adrs[3] );

					_case( 9 )
					_case( 10 )
					_case( 12 )

						ebpPlus( s->Offset, adrs[0] );
						ebpPlus( s->Offset+4, adrs[1] );
						ebpPlus( s->Offset+8, adrs[2] );

					_endcase

					_case( 15 )

						ebpPlus( s->Offset+14, adrs[4] );

					_case( 13 )
					_case( 14 )
					_case( 16 )

						ebpPlus( s->Offset, adrs[0] );
						ebpPlus( s->Offset+4, adrs[1] );
						ebpPlus( s->Offset+8, adrs[2] );
						ebpPlus( s->Offset+12, adrs[3] );

					_endcase

				_endswitch


				_switch( _ifx( d->Type != NULL, d->Type->ObjectSize, 4 ))
				
					_case( 1 )

						Emit2rm( "mov", "cl", adrs[0] );
						Emit2mr( "mov", "[edi]", "cl" );

					_endcase

					_case( 2 )

						Emit2rm( "mov", "cx", adrs[0] );
						Emit2mr( "mov", "[edi]", "cx" );

					_endcase

					_case( 3 )

						Emit2rm( "mov", "cx", adrs[0] );
						Emit2mr( "mov", "[edi]", "cx" );
						Emit2rm( "mov", "cl", adrs[1] );
						Emit2mr( "mov", "[edi+2]", "cl" );

					_endcase

					_case( 4 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );

					_endcase


					_case( 5 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "cl", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "cl" );

					_endcase

					_case( 6 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "cx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "cx" );

					_endcase

					_case( 7 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "cx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "cx" );
						Emit2rm( "mov", "cl", adrs[2] );
						Emit2mr( "mov", "[edi+6]", "cl" );

					_endcase

					_case( 8 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );

					_endcase

					_case( 9 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "cl", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "cl" );

					_endcase

					_case( 10 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "cx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "cx" );

					_endcase


					_case( 11 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "cx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "cx" );
						Emit2rm( "mov", "cl", adrs[3] );
						Emit2mr( "mov", "[edi+10]", "cl" );

					_endcase

					_case( 12 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "ecx" );

					_endcase


					_case( 13 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "ecx" );
						Emit2rm( "mov", "cl", adrs[3] );
						Emit2mr( "mov", "[edi+12]", "cl" );

					_endcase

					_case( 14 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx"  );
						Emit2rm( "mov", "ecx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "ecx" );
						Emit2rm( "mov", "cx", adrs[3] );
						Emit2mr( "mov", "[edi+12]", "cx" );

					_endcase

					_case( 15 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "ecx" );
						Emit2rm( "mov", "cx", adrs[3] );
						Emit2mr( "mov", "[edi+12]", "cx" );
						Emit2rm( "mov", "cl", adrs[5] );
						Emit2mr( "mov", "[edi+14]", "cl" );

					_endcase

					_case( 16 )

						Emit2rm( "mov", "ecx", adrs[0] );
						Emit2mr( "mov", "[edi]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[1] );
						Emit2mr( "mov", "[edi+4]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[2] );
						Emit2mr( "mov", "[edi+8]", "ecx" );
						Emit2rm( "mov", "ecx", adrs[3]  );
						Emit2mr( "mov", "[edi+12]", "ecx" );

					_endcase

					_default

						/*
						** For parameters that are not
						** 1, 2, 3, 4, or 8 bytes long, use
						** a movs instruction to copy the data.
						*/

						sprintf( adrs[1], "%d", s->ObjectSize / 4 );
						Emit2rc( "mov", "ecx", adrs[1]  );
						Emit2rm( "mov", "esi", adrs[0]  );
						_if( (s->ObjectSize & 3) >= 2 )

							Emit0( "movsw" );

						_endif
						_if( s->ObjectSize & 1 )

							Emit0( "movsb" );

						_endif

				_endswitch

			_endif
			d = d->Next;

		_endwhile

		_if( NeedsMOVS )

			Emit0( "popfd" );
			Emit1R( "pop", "esi" );

		_endif
		Emit1R( "pop", "ecx" );
		Emit1R( "pop", "edi" );		

	_endif


_end( StoreValResParms )



/******************************************/
/*                                        */
/* Division and Mod instruction emission: */
/*                                        */
/******************************************/

void
EmitdivmodConst( char* Instr, unsigned size, union YYSTYPE *Value )
_begin( divmodEmit )

	char	*divisor;
	char	*instrSize;
	char	label[64];
	int		lblcnt;

	_if( *Instr == 'i' )

		divisor = CheckOrdinal( Value, size );

	_else

		divisor = CheckUnsigned( Value, size );

	_endif
	sprintf( label, "L%d_divisor" sympost, LblCntr );
	lblcnt = LblCntr++;

	startStrSeg();
	
	instrSize =
		_ifx
		(
			size == 8,
			"byte",
			_ifx( size == 16, "word", "dword" )
		);
	
	EmitData
	( 
		label,
		instrSize, 
		divisor 
	);
	endStrSeg();

	sprintf
	( 
		label, 
		"%s ptr %s[L%d_divisor" sympost "]", 
		instrSize, 
		_ifx( assembler == tasm, "ds:", ""),
		lblcnt 
	);
	Emit1L( Instr, label );
	free2( vss divisor );

_end( divmodEmit ) 



/*
** PassValpConst-
**
**	Emits the code to pass a constant by value as a parameter.
*/

void 
PassValpConst( struct SymNode *sym, union YYSTYPE *value )
_begin( EmitValpConst )

	int		Lbl;
	char	label[256];
	char	labelSize[256];

	/*
	** If a scalar object, it's really easy to push the
	** constant onto the stack.  So handle scalar objects
	** as a special case here.
	*/


	_if( IsScalar( value->v.pType ) )

		/*
		** Make sure the actual and formal parameters are compatible.
		*/

		_if( IsCompatible( sym->Type, value ))


			/*
			** If the static name field of the parameter is not
			** NULL, then it contains a register name and we
			** are passing the specified parameter in that
			** register.  Emit the code for that here.
			*/

			_if( sym->StaticName != NULL )

				int r;
				
				r = RegStrToReg( sym->StaticName );
				mov_cr( value, r );
				//strcpy( label, sym->StaticName );
				//EmitIRC( "mov", label, SSN value );

			/*
			** Have to create pseudo-vars for REAL constants.
			** Do that here.
			*/

			_elseif( sym->pType == tReal32 )

				sprintf( label, "L%d_flt" sympost, LblCntr++ );
				startStrSeg();
				EmitTypedLabel( label, isFasm( "dword", "real4" ));
				EmitReal4Const( value->v.u.fltval.f.f );
				endStrSeg();
				LabelToMem( labelSize, label, "dword" );
				Emit1M( "push", labelSize );

			_elseif( sym->pType == tReal64 )

				sprintf( label, "L%d_flt" sympost, LblCntr++ );
				startStrSeg();
				EmitTypedLabel( label, isFasm( "qword", "real8" ));
				EmitReal8Const( value->v.u.fltval.f.d );
				endStrSeg();
				LabelToMemPlusOfs( labelSize, label, "dword", 4 );
				Emit1M( "push", labelSize );
				LabelToMem( labelSize, label, "dword" );
				Emit1M( "push", labelSize );

			_elseif( sym->pType == tReal80 )

				/*
				** Note: we must always push a multiple of
				** four bytes onto the stack, hence the
				** extra PUSHW 0 here.
				*/

				char label0[128];
				char label4[128];
				char label8[128];

				sprintf( label, "L%d_flt" sympost, LblCntr++ );
				startStrSeg();
				EmitTypedLabel( label, isFasm( "tword", "real10" ));
				EmitReal10Const( value->v.u.fltval );
				endStrSeg();
				LabelToMem( label0, label, "dword" );
				LabelToMemPlusOfs( label4, label, "dword", 4 );
				LabelToMemPlusOfs( label8, label, "word", 8 );
				Pushw( 0 );
				Emit1M( "push", label8 );
				Emit1M( "push", label4 );
				Emit1M( "push", label0 );

			_elseif( sym->ObjectSize == 16 )
			
				/*
				** 16-bit constant (LWord, Int128, or Uns128).
				*/
				
				Pushd( value->v.u.lwordval[3] );
				Pushd( value->v.u.lwordval[2] );
				Pushd( value->v.u.lwordval[1] );
				Pushd( value->v.u.lwordval[0] );
			
			_elseif( sym->ObjectSize == 10 )
			
				/*
				** 10-byte constant (TByte).
				*/
				
				Pushw( 0 );
				Pushw( value->v.u.lwordval[2] );
				Pushd( value->v.u.lwordval[1] );
				Pushd( value->v.u.lwordval[0] );
			
			_elseif( sym->ObjectSize == 8 )
			
				/*
				** 16-byte constant (QWord, Int64, or Uns64).
				*/
				
				Pushd( value->v.u.lwordval[1] );
				Pushd( value->v.u.lwordval[0] );
			
			_else /* Must be a four-byte scalar */

				Pushd( value->v.u.unsval ); 

			_endif


		/*
		** Okay, it's not a scalar value.  Now the work gets a little
		** harder.  First, check to see if it's a pointer constant
		** with a NULL actual parameter value (which is still easy).
		*/

		_elseif
		( 
				IsNumber( value->v.pType )
			&&	numBits( value ) <= 32  
			&&	value->v.u.unsval == 0
			&&	( sym->pType == tPointer || sym->pType == tProcptr )
		)

			/*
			** Special case to allow NULL constant
			** for a pointer.
			*/

			Pushd( 0 );


		/*
		** If it's a string parameter, allow a dword object.
		** This will mainly let us pass NULL as a string pointer
		** value.  Also, some Win32 APIs allow a numeric constant
		** in place of a pointer to a character string.  This is
		** bad style, but Windows requires it so we must allow it.
		*/

		_elseif
		( 
				IsNumber( value->v.pType )
			&&	numBits( value ) <= 32 
			&&	IsStr ( sym->pType ) 
		)

			Pushd( value->v.u.unsval );
			_if( value->v.u.unsval != 0 )

				WarnNear
				( 
					"Non-NULL numeric constant passed as string parameter", 
					yytext 
				);

			_endif

		_else

			yyerror
			(
				"Type of constant actual parameter does not\n"
				"agree with type of formal parameter"
			);

		_endif

	/*
	** Deal with string actual parameters here.
	*/

	_elseif( value->v.pType == tString )

		// We can pass a string to a zstring parameter!
		
		_if( IsStr( sym->pType ))

			char adrs[128];

			Lbl = LblCntr++;
			EmitString( value->v.u.strval, Lbl );
			sprintf( adrs, "L%d_str" sympost, Lbl );
			PushStaticAdrs( adrs );

		_else

			yyerror( "Type mismatch (requires string parameter)" );

		_endif



	_elseif( value->v.pType == tZString )

		// ZStrings can only be passed as Zstrings!
		
		_if( sym->pType == tZString )

			char adrs[128];

			Lbl = LblCntr++;
			EmitString( value->v.u.strval, Lbl );
			sprintf( adrs, "L%d_str" sympost, Lbl );
			PushStaticAdrs( adrs );

		_else

			yyerror( "Type mismatch (requires zstring parameter)" );

		_endif




	/*
	** Handle pointer parameters here.  If the actual parameter is
	** a pointer, this means the parameter has taken the form:
	** "&varName".
	*/

	_elseif
	( 
			value->v.pType == tPointer 
		&&	(
					sym->pType == tDWord 
				||	sym->pType == tProcptr
				||	sym->pType == tPointer
				||	IsStr( sym->pType )
			)
	)

		/*
		** If the static name field of the parameter is not
		** NULL, then it contains a register name and we
		** are passing the specified parameter in that
		** register.  Emit the code for that here.
		*/

		_if( sym->StaticName != NULL )

			int r;
			
			r = RegStrToReg( sym->StaticName );
			mov_cr( value, r );
			
		_else

			PushStaticAdrs( value->v.u.strval );
			
		_endif
		
	/*
	** Handle character set parameters here.
	*/

	_elseif( value->v.pType == tCset )
	

		_if( sym->pType == tCset )

			Pushd( value->v.u.lwordval[3] );
			Pushd( value->v.u.lwordval[2] );
			Pushd( value->v.u.lwordval[1] );
			Pushd( value->v.u.lwordval[0] );

		_else

			yyerror( "Illegal character set parameter" );

		_endif


	/*
	** Okay, now it starts to get a little harder.
	** The actual parameter is an array constant.  If it's
	** compatible, push it onto the stack four bytes at a
	** time if it's 64 bytes or less.  If it's more than
	** 64 bytes, create a readonly variable initialized with
	** the constant and then block copy the data.
	*/

	_elseif( sym->pType == tArray )

		_if( ArraysAreCompatible( (union YYSTYPE *) sym, value ))

			char name[ 32 ];
			char bytes[ 64 ];
			int	 Size;
			int	 i;
			struct SymNode *s;

			// If we're passing 64 bytes or fewer as a constant,
			// then simply emit the pushes for these constants.

			_if( sym->ObjectSize <= 64 )

				// Who knows what the constant elements really are.
				// We've got to grab the data and pack it into byte
				// pieces.
				
				i = 0;
				s = GetBaseType( sym->Type );
				_for( int j=0, j < sym->NumElements, ++j )

					// Break up each element of the array into
					// bytes and place the bytes in the "bytes" array.

					_switch( s->ObjectSize % 4 )

					  _case( 0 )

						bytes[i++] = 
							(char)
							  (value->v.u.ArrayOfValues[j].u.unsval & 0xff);
							  
						bytes[i++] = 
							(char)

							  ((value->v.u.ArrayOfValues[j].u.unsval >> 8) & 0xff);

						bytes[i++] =
							(char) 
							  ((value->v.u.ArrayOfValues[j].u.unsval >> 16) & 0xff);

						bytes[i++] =
							(char) 
							  ((value->v.u.ArrayOfValues[j].u.unsval >> 24) & 0xff);

						break;

					  _case( 1 )

						bytes[i++] = 
							(char)
							  (value->v.u.ArrayOfValues[ j ].u.unsval & 0xff);
						break;

					  _case( 2 )

						bytes[i++] = 
							(char)
							  (value->v.u.ArrayOfValues[ j ].u.unsval & 0xff);
						bytes[i++] = 
							(char)
							  ((value->v.u.ArrayOfValues[ j ].u.unsval >> 8 ) & 0xff);

						break;

					  _case( 3 )

						bytes[i++] = 
							(char)
						 	  (value->v.u.ArrayOfValues[ j ].u.unsval & 0xff);
							  
						bytes[i++] = 
							(char)
							  ((value->v.u.ArrayOfValues[j].u.unsval >> 8) & 0xff);

						bytes[i++] =
							(char) 
							  ((value->v.u.ArrayOfValues[j].u.unsval >> 16) & 0xff);
							  
						break;

					_endswitch

				_endfor

				// Round the number of bytes up to the next multiple of four.

				Size = (( s->ObjectSize + 3 ) >> 2) << 2;


				// Output the Pushes to push these values on the stack.

				_for( int j=Size-4, j >= 0, j-=4 )
						
					Pushd
					( 
							(long) bytes[j] 
						+	(((long) bytes[j+1]) << 8)
						+	(((long) bytes[j+2]) << 16)
						+	(((long) bytes[j+3]) << 24)
					);

				_endfor

			_else

				char	val[ 32 ];
				char	label[32];

				Size = ( sym->ObjectSize + 3 ) >> 2;

				// If we're passing more than 64 bytes, create
				// a memory array with the data and then use a
				// MOVSD instruction to actually push the data.

				sprintf( val, "%d", Size*4 );
				sprintf( name, "L%d_array" sympost, LblCntr++ );
				OutPaddedValue( name, sym->Type, value );


				Emit2rc( "sub", "esp", val );
				Emit1R( "push", "ecx" );
				Emit1R( "push", "esi" );
				Emit1R( "push", "edi" );
				Emit0( "pushfd" );
				Emit0( "cld" );

				sprintf( val, "%d", Size );
				Emit2rc( "mov", "ecx", val );
				_if
				( 
						assembler == masm
					||	assembler == tasm
					||	assembler == gas
				)

					setOffset32( label );
					strcat( label, name );

				_elseif( assembler == fasm )

					strcpy( label, name );

				_else
					
					yyerror( "Internal error (Bad assembler value)" );

				_endif
				Emit2rc( "mov", "esi", label  );
				Emit2rm( "lea", "edi", "[esp+16]"  );
				Emit0( "rep\tmovsd" );
				Emit0( "popfd" );
				Emit1R( "pop", "edi" );
				Emit1R( "pop", "esi" );
				Emit1R( "pop", "ecx" );

			_endif

		_else

			yyerror
			( 
				"Actual array parameter type does not match formal type"
			);

		_endif


	_elseif( sym->pType == tRecord && value->v.pType == tRecord )

		_if( FieldsAreCompatible( sym->Type, value ))

			char	name[ 64  ];
			char	adrs[ 128 ];
			int		Size;
			int		i;


			sprintf( name, "L%d_record" sympost, LblCntr++ );
			OutPaddedValue( name, sym, value );

			Size = ( sym->ObjectSize + 3 ) >> 2;


			_if( Size <= 16 )

				_for( i=Size-1, i >=0, --i )


					LabelToMemPlusOfs( adrs, name, "dword", i*4 );
					Emit1M( "push", adrs );
			
				_endfor

			_else

				sprintf( adrs, "%d", Size*4 );
				Emit2rc( "sub", "esp", adrs );
				Emit1R( "push", "ecx" );
				Emit1R( "push", "esi" );
				Emit1R( "push", "edi" );
				Emit0( "pushfd" );
				Emit0( "cld" );
				sprintf( adrs, "%d", Size );
				Emit2rc( "mov", "ecx", adrs );
				LabelToOfs( adrs, name );
				Emit2rc( "mov", "esi", adrs );
				Emit2rm( "lea", "edi", "[esp+16]" );
				Emit0( "rep\tmovds" );
				Emit1R( "pop", "edi" );
				Emit1R( "pop", "esi" );
				Emit1R( "pop", "ecx" );

			_endif


		_else

			yyerror
			( 
				"Actual record parameter type does not match formal type"
			);

		_endif

			
	_else

		yyerror
		( 
			"Cannot pass specified type as a value parameter" 
		);

	_endif

_end( EmitValpConst )



// CheckDisp-
//	Checks a displacement associated with an addressing mode
//	to be sure it's valid.

int
CheckDisp( union YYSTYPE *disp )
_begin( CheckDisp )

	int d;
	
	assert( disp != NULL );
	_returnif
	( 
			IsOrdinal( disp->v.pType ) 
		&&	numBits( disp ) <= 32 
	)
		disp->v.u.intval;
		

	yyerror( "Expected a 32-bit ordinal displacement value" );
	FreeValue( disp );
	disp->v.pType = tInt32;
	disp->v.u.intval = 0;
	_return( 0 );
	
_end( CheckDisp )


void 
initAdrs
( 
	struct adrsYYS *adrs,
	int				baseReg, 
	int				indexReg,
	int				scale,
	int				disp
)
_begin( initAdrs )

	assert( adrs != NULL );
	adrs->Size = 0;
	adrs->ObjectSize = 0;
	adrs->StaticName = NULL;
	adrs->BaseReg = _ifx( baseReg < 0, NULL, hlastrdup( regStrs[ baseReg ] ));
	adrs->IndexReg = _ifx( indexReg < 0, NULL, hlastrdup( regStrs[ indexReg ] ));
	adrs->Scale = scale;
	adrs->Disp = disp;
	adrs->Sym = NULL;
	adrs->Type = NULL;
	adrs->pType = tVariant;	/* No real type 	*/
	adrs->pClass = valp_pc;	/* No real pClass	*/
	adrs->SymClass = cNone;	/* No real class	*/

_end( initAdrs )



// mergeMem-
//	Given two memYYS type objects (from the same addressing
// mode expression, generally), merge the two into a single
// memory object. Used to combine two addressing mode sequences
// into a single object.
 
void 
mergeMem( struct memYYS *dest, struct memYYS *src )
_begin( mergeMem )

	dest->disp += src->disp;
	_if( src->base != -1 )
	
		_if( dest->base == -1 )
		
			dest->base = src->base;
		
		_else
		
			_if( dest->index != -1 || src->index != -1 )
			
				yyerror
				( 
					"Too many index registers in address expression" 
				);
			
			_endif
			dest->index = src->base;
			dest->scale = 0;
			
		_endif
	
	_endif	
	_if( src->index != -1 )
					
		_if( dest->index != -1 )
		
			yyerror( "Too many index registers in address expression" );
		
		_endif
		dest->index = src->index;
		dest->scale = src->scale;
	
	_endif	

_end( mergeMem )


// processCondJump-
//
//	Handle target operands that follow a conditional jump instruction.

void
processCondJump( char *instr, char *target, int TrueLabel, int FalseLabel )
_begin( processCondJump )

	struct SymNode	*sym;
	char			label[ 256 ];

	sym = lookup( target, 1 );
	_if
	(
			sym == &true_ste
	)

		_if( TrueLabel != -1 )

			sprintf( label, "L%d_true" sympost, TrueLabel );
			Emit1L( instr, label ); 

		_else

			yyerror( "\"true\" cannot be used as a label here" );

		_endif


	_elseif
	(
			sym == &false_ste
	)

		_if( FalseLabel != -1 )

			sprintf( label, "L%d_false" sympost, FalseLabel );
			Emit1L( instr, label ); 

		_else

			yyerror( "\"false\" cannot be used as a label here" );

		_endif


	_elseif
	(
			sym != NULL 
		&&	sym->LexLevel == CurLexLevel
		&&	sym->SymClass != cLabel
	)

		/*
		** Attempting to jump to something that is not a label.
		*/

		yyerror( "Illegal branch target" );

	_elseif
	( 
			sym != NULL 
		&&	sym->LexLevel == CurLexLevel
		&&	sym->SymClass == cLabel
	)

		/*
		** We've got a jump to a label that is already defined.
		*/

		Emit1L( instr, sym->StaticName );

	_else

		/*
		** We've got a jump to a label that has yet to be defined
		** in this procedure (note that the label could be defined
		** outside the current procedure as anything, but we'll
		** defer judgement on this until we finish processing the
		** current procedure).
		*/

		struct FwdRefLabelType *flist;


		/*
		** First, let's see if there is already a reference
		** to this particular identifier so we can use the
		** existing StaticName.
		*/

		flist = FwdLabelsList;
		_while( flist != NULL )

			_breakif( _streq( target, flist->label ));
			flist = flist->Next;

		_endwhile

		_if( flist != NULL )

			/*
			** We found the symbol in the forward reference
			** list.  So just use the StaticName field of the
			** symbol we located as the target label.
			*/

			Emit1L( instr, flist->StaticName );

		_else

			/*
			** We didn't find the symbol in the forward reference
			** list, so add it to that list.
			*/

			struct FwdRefLabelType *temp;
			char sn[ 256 ];

			temp = malloc2( sizeof( struct FwdRefLabelType ));
			temp->Next = FwdLabelsList;
			temp->label = hlastrdup( target );
			temp->lexLevel = CurLexLevel;
			temp->isExternal = 0;
			sprintf( sn, "L%d_%s" sympost, LblCntr++, target );
			temp->StaticName = hlastrdup( sn );
			assert( temp->StaticName != NULL );
			Emit1L( instr, temp->StaticName );
			FwdLabelsList = temp;

		_endif

	_endif

_end( processCondJump )


// condJumpHere, condJumpHere2-
//
//	Processes the @here and @here[expr] operands for a conditional jump.

void 
condJumpHere( char *instr )
_begin( condJumpHere )

	_if( assembler == masm || assembler == tasm || assembler == fasm )
	
		Emit1L( instr, "$" );
		
	_elseif( assembler == gas )
	
		Emit1L( instr, "." );
		
	_else
	
		yyerror( "Unknown assembler (internal HLA error)" );
		Emit1L( instr, "$" );
		
	_endif

_end( condJumpHere )	



void 
condJumpHere2( char *instr, union YYSTYPE *ofs )
_begin( condJumpHere2 )

	char StaticName[64];
	
	_if( !IsOrdinal( ofs->v.pType ) || numBits( ofs ) > 32 )

		yyerror( "Expected 32-bit ordinal type in index expression" );
	
	_endif
	sprintf
	( 
		StaticName, 
		"(%s+%lu)", 
		_ifx
		(
			( 
					assembler == masm 
				||	assembler == tasm 
				||	assembler == fasm 
			), 
			"$",
			_ifx
			( 
				assembler == gas, 
				".", 
				"???" 
			)
		), 
		ofs->v.u.intval 
	);
	Emit1L( instr, StaticName );

_end( condJumpHere2 )	


// callUndefSym-
//	Handles the case when the program calls a symbol that has yet to be defined.

void
callUndefSym( char *undefSym )
_begin( callUndefSym )

	struct FwdRefLabelType	*flist;
	struct FwdRefLabelType	*temp;
	struct SymNode			*sym;
	char					label[ 256 ];
	char					sn[ 256 ];

	sym = lookup( undefSym, 1 );
	_if
	( 
			sym != NULL 
		&&	(
					(
							sym->LexLevel == CurLexLevel
						&&	sym->SymClass == cLabel
					)
				||	sym->SymClass == cProc
				||	sym->SymClass == cIterator
			)
	)

		/*
		** We've got a call to a label that is already defined.
		*/

		Emit1L( "call", sym->StaticName );
		SetReferenced( sym );

	_else

		/*
		** We've got a jump to a label that has yet to be defined
		** in this procedure (note that the label could be defined
		** outside the current procedure as anything, but we'll
		** defer judgement on this until we finish processing the
		** current procedure).
		*/



		/*
		** First, let's see if there is already a reference
		** to this particular identifier so we can use the
		** existing StaticName.
		*/

		flist = FwdLabelsList;
		_while( flist != NULL )

			_breakif( _streq( undefSym, flist->label ));
			flist = flist->Next;

		_endwhile

		_if( flist != NULL )

			/*
			** We found the symbol in the forward reference
			** list.  So just use the StaticName field of the
			** symbol we located as the target label.
			*/

			Emit1L( "call", flist->StaticName );

		_else

			/*
			** We didn't find the symbol in the forward reference
			** list, so add it to that list.
			*/

			temp = malloc2( sizeof( struct FwdRefLabelType ));
			temp->Next = FwdLabelsList;
			temp->label = hlastrdup2( undefSym );
			temp->lexLevel = CurLexLevel;
			temp->isExternal = 0;
			assert( temp->label != NULL );
			sprintf( sn, "L%d_%s" sympost, LblCntr++, undefSym );
			temp->StaticName = hlastrdup2( sn );
			assert( temp->StaticName != NULL );
			Emit1L( "call", temp->StaticName );
			FwdLabelsList = temp;

		_endif

	_endif

_end( callUndefSym )



// callHere-
//	Handles "call @here[disp]"

void
callHere( int disp )
_begin( callHere )

	char StaticName[64];
	
	_if( disp == 0 )
	
		_if
		( 
				assembler == masm 
			||	assembler == tasm 
			||	assembler == fasm 
		)
		
			Emit1L( "call", "$" );
			
		_elseif( assembler == gas )
		
			Emit1L( "call", "." );
			
		_else

		
			yyerror( "Internal HLA error -- bad assembler" );
			
		_endif 
		
	
	_else
	
		sprintf
		( 
			StaticName, 
			"(%s%s%lu)", 
			_ifx
			(
				( 
						assembler == masm 
					||	assembler == tasm 
					||	assembler == fasm 
				), 
				"$",
				_ifx
				( 
					assembler == gas, 
					".", 
					"???" 
				)
			), 
			_ifx( disp<0, "-", "+" ), 
			disp 
		);
		Emit1L( "call", StaticName );
		
	_endif

_end( callHere )



// jmpTargetID-
//	Handle identifiers appearing after a JMP instruction:

void 
jmpTargetID( char *jmpSym  )
_begin( jmpTargetID )

	struct SymNode *sym;
	char	label[ 256 ];

	sym = lookup( jmpSym, 1 );
	_if
	(
			sym == &true_ste
	)

		_if( TrueLabel != -1 )

			sprintf
			(
				label,
				"L%d_true" sympost,
				TrueLabel
			);
			Emit1L( "jmp", label );

		_else

			yyerror( "\"true\" cannot be used as a label here" );

		_endif


	_elseif
	(
			sym == &false_ste
	)

		_if( FalseLabel != -1 )

			sprintf
			(
				label,
				"L%d_false" sympost,
				FalseLabel
			);
			Emit1L( "jmp", label );

		_else

			yyerror( "\"false\" cannot be used as a label here" );

		_endif


	_elseif
	( 
			sym != NULL 
		&&	sym->LexLevel == CurLexLevel
		&&	sym->SymClass == cLabel
	)

		/*
		** We've got a jump to a label that is already defined.
		*/

		Emit1L( "jmp", sym->StaticName );
		SetReferenced( sym );

	_else

		/*
		** We've got a jump to a label that has yet to be defined
		** in this procedure (note that the label could be defined
		** outside the current procedure as anything, but we'll
		** defer judgement on this until we finish processing the
		** current procedure).
		*/

		struct FwdRefLabelType *flist;


		/*
		** First, let's see if there is already a reference
		** to this particular identifier so we can use the
		** existing StaticName.
		*/

		flist = FwdLabelsList;
		_while( flist != NULL )

			_breakif( _streq( jmpSym, flist->label ));
			flist = flist->Next;

		_endwhile

		_if( flist != NULL )

			/*
			** We found the symbol in the forward reference
			** list.  So just use the StaticName field of the
			** symbol we located as the target label.
			*/

			Emit1L( "jmp", flist->StaticName );

		_else

			/*
			** We didn't find the symbol in the forward reference
			** list, so add it to that list.
			*/

			struct FwdRefLabelType *temp;
			char sn[ 256 ];

			temp = malloc2( sizeof( struct FwdRefLabelType ));
			temp->Next = FwdLabelsList;
			temp->label = hlastrdup2( jmpSym );
			temp->lexLevel = CurLexLevel;
			temp->isExternal = 0;
			assert( temp->label != NULL );
			sprintf( sn, "L%d_%s" sympost, LblCntr++, jmpSym );
			temp->StaticName = hlastrdup2( sn );
			assert( temp->StaticName != NULL );
			Emit1L( "jmp", temp->StaticName );
			FwdLabelsList = temp;

		_endif

	_endif

_end( jmpTargetID )



void 
jmpHere( int disp )
_begin( jmpHere )

	char StaticName[64];

	_if( disp == 0 )
	
		_if( assembler == masm || assembler == tasm || assembler == fasm )
		
			Emit1L( "jmp", "$" );
			
		_elseif( assembler == gas )
		
			Emit1L( "jmp", "." );
			
		_else
		
			yyerror( "Unknown assembler (internal HLA error)" );
			Emit1L( "jmp", "$" );
			
		_endif
		
	_else
			
		sprintf
		( 
			StaticName, 
			"(%s%s%lu)", 
			_ifx
			(
				( 
						assembler == masm 
					||	assembler == tasm 
					||	assembler == fasm 
				), 
				"$",
				_ifx
				( 
					assembler == gas, 
					".", 
					"???" 
				)
			),
			_ifx( disp<0, "-", "+" ), 
			disp 
		);
		Emit1L( "jmp", StaticName );

	_endif

_end( jmpHere )



/*
** needsOffsetStr - Returns an appropriate string prefix for the
**					current assembler that converts an address to
**					an offset if the parameter is true.
*/

char*
needsOffsetStr( int needsOffset )
_begin( needsOffsetStr )

	_if( needsOffset)
	
		_switch( assembler )
		
				
			_case( fasm )
				
				_return( "" );
			
			_case( gas )
				_return "offset ";
					
			_case( masm )
			_case( tasm )
			
				_return "offset32 ";
				
		_endswitch
	
	_endif
	_return "";
		
_end( needsOffsetStr )



				

/****************************************************************/
/*                                                              */
/* MakeMemStr-                                                  */
/*                                                              */
/* Takes an address object and creates a string that            */
/* is acceptable to the current assembler that represents       */
/* that memory address.  If EmitSize is true, and the assembler */
/* supports type checking, then this function also emits size   */
/* information for the memory access (e.g., "byte ptr").        */
/*                                                              */
/****************************************************************/
		
void
MakeMemStr( char *str, union YYSTYPE *adrs, int EmitSize )
_begin( MakeMemStr )

	int pType;
	int snLen;
	char *sn;
	char *lenStr;
	int	 size;
	
	#define notEmpty(s) ((s)!=NULL && *(s)!='\0')
	#define putStr(s)   (notEmpty(s) ? (s) : "" )
	#define putPlus(x)  ((x) ? "+" : "")

	assert( str != NULL );
	assert( adrs != NULL );


	/*
	** Grand Kludge!
	**
	**	Parameters passed in registers return the
	** register name in the StaticName field.  If
	** this particular address is a register, don't
	** adorn the address with anything but the
	** register name.
	*/

	sn = adrs->adrs.StaticName;
	_if( sn != NULL )

		snLen = strlen( sn );
		_if( snLen == 3 && sn[0] == 'e' )

			_if
			( 
					_streq( sn, "eax" )
				||	_streq( sn, "ebx" )
				||	_streq( sn, "ecx" )
				||	_streq( sn, "edx" )
				||	_streq( sn, "ebp" )
				||	_streq( sn, "esp" )
				||	_streq( sn, "esi" )
				||	_streq( sn, "edi" )
			)

				strcpy( str, sn );
				_return;

			_endif

		_elseif( snLen == 2 )

			_if
			( 
					_streq( sn, "al" )
				||	_streq( sn, "bl" )
				||	_streq( sn, "cl" )
				||	_streq( sn, "dl" )

				||	_streq( sn, "ah" )
				||	_streq( sn, "bh" )
				||	_streq( sn, "ch" )
				||	_streq( sn, "dh" )
			)

				strcpy( str, sn );
				_return;


			_elseif
			(
					_streq( sn, "ax" )
				||	_streq( sn, "bx" )
				||	_streq( sn, "cx" )
				||	_streq( sn, "dx" )
				||	_streq( sn, "bp" )
				||	_streq( sn, "sp" )
				||	_streq( sn, "si" )
				||	_streq( sn, "di" )
			)

				strcpy( str, sn );
				_return;

			_endif

		_endif


	_endif


	*str = '\0';

	assert( adrs->adrs.pType != tArray || adrs->adrs.Type != NULL );
	pType = _ifx
			( 
				adrs->adrs.pType != tArray, 
				adrs->adrs.pType,
				adrs->adrs.Type->pType
			);


	/*
	** Output the size (if applicable)
	*/

	

	lenStr = "";
	_if( EmitSize )
	
		size = adrs->adrs.Size;
		_if( size == 1 )

			_switch( assembler )
			
				_case( masm )
				_case( tasm )
				_case( gas  )
				_case( fasm )
				
					lenStr = "byte ptr";
					
				_endcase
				
				
			_endswitch

		_elseif( size == 2 )

			_switch( assembler )
			
				_case( masm )
				_case( tasm )

				_case( gas  )
				_case( fasm )
				
					lenStr = "word ptr";
					
				_endcase
				
				
			_endswitch


		_elseif( size == 4 )

			_if( pType == tReal32 )

				_switch( assembler )
				
					_case( masm )
					_case( tasm )
					
						lenStr = "real4 ptr";
						
					_endcase
					
					_case( gas  )

					_case( fasm )
					
						lenStr = "dword ptr";
						
					_endcase
					
					
				_endswitch

			_else

				_switch( assembler )
				
					_case( masm )
					_case( tasm )
					_case( gas  )
					_case( fasm )
					
						lenStr = "dword ptr";
						
					_endcase
					
					
				_endswitch

			_endif


		_else

			_if( pType == tReal64 )

				_switch( assembler )
				
					_case( masm )
					_case( tasm )
					
						lenStr = "real8 ptr";
						
					_endcase
					
					_case( gas  )
					_case( fasm )
					
						lenStr = "qword ptr";
						
					_endcase
					
					
				_endswitch

			_elseif( pType == tReal80 )

				_switch( assembler )
				
					_case( masm )
					_case( tasm )
					
						lenStr = "real10 ptr";
						
					_endcase
					
					_case( gas  )
					
						lenStr = "tbyte ptr";
						
					_endcase
					
					_case( fasm )
					
						lenStr = "tword ptr";
						
					_endcase
					
					
				_endswitch

			_elseif
			( 
					pType == tQWord 
				||	pType == tUns64
				||	pType == tInt64
			)

				strcpy( str, "qword ptr " );
				_switch( assembler )
				
					_case( masm )
					_case( tasm )
					_case( gas  )
					_case( fasm )
					
						lenStr = "qword ptr";
						
					_endcase
					
					
				_endswitch

			_elseif( pType == tTByte )

				_switch( assembler )
				
					_case( masm )
					_case( tasm )
					_case( gas  )
					
						lenStr = "tbyte ptr";
						
					_endcase
					
					_case( fasm )
					
						lenStr = "tword ptr";
						
					_endcase
					
					
				_endswitch

			_endif

		_endif
		
	_endif

	

	/*
	** If there is a static name component to this
	** address, print it first.
	*/


	_if( adrs->adrs.Disp != 0 )
			
		sprintf
		( 
			str, 
			"%s %s[%s%s%s%s%s%s%s%d]", 
			lenStr,
			_ifx( assembler == tasm, "ds:", "" ),
			putStr(  adrs->adrs.StaticName ),
			putPlus
			( 
					notEmpty( adrs->adrs.StaticName ) 
				&&	notEmpty( adrs->adrs.BaseReg )
			), 
			putStr(	adrs->adrs.BaseReg ),
			putPlus
			( 
					(
							notEmpty( adrs->adrs.StaticName )
						||	notEmpty(	adrs->adrs.BaseReg )
					)
				&&	notEmpty( adrs->adrs.IndexReg )
			),
			putStr( adrs->adrs.IndexReg ),
			_ifx
			(
				notEmpty( adrs->adrs.IndexReg ),
				rtnScale( adrs ), 
				""
			),
			_ifx
			(
					adrs->adrs.BaseReg != NULL 
				||	adrs->adrs.IndexReg != NULL
				||	adrs->adrs.StaticName != NULL,
				_ifx( adrs->adrs.Disp < 0, "", "+" ),
				""
			),
			adrs->adrs.Disp
		);
		
	_else
	
		// Don't emit a displacement value of zero as GAS will
		// actually generate a zero displacement byte if you do this.
		
		sprintf
		( 
			str, 
			"%s %s[%s%s%s%s%s%s]", 
			lenStr,
			_ifx( assembler == tasm, "ds:", "" ),
			putStr(  adrs->adrs.StaticName ),
			putPlus
			( 
					notEmpty( adrs->adrs.StaticName ) 
				&&	notEmpty( adrs->adrs.BaseReg )
			), 
			putStr(	adrs->adrs.BaseReg ),
			putPlus
			( 
					(
							notEmpty( adrs->adrs.StaticName )
						||	notEmpty(	adrs->adrs.BaseReg )
					)
				&&	notEmpty( adrs->adrs.IndexReg )

			),
			putStr( adrs->adrs.IndexReg ),
			_ifx
			(
				notEmpty( adrs->adrs.IndexReg ),
				rtnScale( adrs ), 
				""
			)
		);

	_endif		
		


_end( MakeMemStr )



void
MakeMemStrSize( char *address, union YYSTYPE *adrs, int size )
_begin( MakeMemStrSize )

	int sizeSave;
	int objsizeSave;
	
	sizeSave = adrs->adrs.Size;
	objsizeSave = adrs->adrs.ObjectSize;
	adrs->adrs.Size = size;
	adrs->adrs.ObjectSize = size;
	MakeMemStr( address, adrs, 1 );
	adrs->adrs.Size = sizeSave;
	adrs->adrs.ObjectSize = objsizeSave;
	
_end( MakeMemStrSize )



void
MakeOfsStr( char *str, union YYSTYPE *adrs )
_begin( MakeMemStr )

	char *offset;
	char *endoffset;
	
	#define notEmpty(s) ((s)!=NULL && *(s)!='\0')
	#define putStr(s)   (notEmpty(s) ? (s) : "" )
	#define putPlus(x)  ((x) ? "+" : "")

	assert( str != NULL );
	assert( adrs != NULL );


	*str = '\0';

	assert( adrs->adrs.pType != tArray || adrs->adrs.Type != NULL );


	/*
	** If there is a static name component to this
	** address, print it first.
	*/
	
	offset = "";
	endoffset = "";
	_switch( assembler )
	
		_case( masm )
		_case( tasm )
			offset = "offset32 [";
			endoffset = "]";
		_endcase
		
		_case( gas  )
			offset = "offset [";
			endoffset = "]";
		_endcase
	
	_endswitch;
	


	_if( adrs->adrs.Disp != 0 )
			
		sprintf
		( 
			str, 
			"%s %s%s%s%s%s%s%s%d %s",
			offset ,
			putStr(  adrs->adrs.StaticName ),
			putPlus
			( 
					notEmpty( adrs->adrs.StaticName ) 
				&&	notEmpty( adrs->adrs.BaseReg )
			), 
			putStr(	adrs->adrs.BaseReg ),
			putPlus
			( 
					(
							notEmpty( adrs->adrs.StaticName )
						||	notEmpty(	adrs->adrs.BaseReg )
					)
				&&	notEmpty( adrs->adrs.IndexReg )
			),
			putStr( adrs->adrs.IndexReg ),
			_ifx
			(
				notEmpty( adrs->adrs.IndexReg ), 
				rtnScale( adrs ), 
				""
			),
			_ifx
			(
					adrs->adrs.BaseReg != NULL 
				||	adrs->adrs.IndexReg != NULL
				||	adrs->adrs.StaticName != NULL,
				_ifx( adrs->adrs.Disp < 0, "", "+" ),
				""
			),
			adrs->adrs.Disp,
			endoffset
		);
		
	_else
	
		// Don't emit an explicit displacement of zero
		// because GAS will generate a zero byte to the
		// code stream if you do this.
		
		sprintf
		( 
			str, 
			"%s %s%s%s%s%s%s %s",
			offset ,
			putStr(  adrs->adrs.StaticName ),
			putPlus
			( 
					notEmpty( adrs->adrs.StaticName ) 
				&&	notEmpty( adrs->adrs.BaseReg )
			), 
			putStr(	adrs->adrs.BaseReg ),
			putPlus
			( 
					(
							notEmpty( adrs->adrs.StaticName )
						||	notEmpty(	adrs->adrs.BaseReg )
					)
				&&	notEmpty( adrs->adrs.IndexReg )
			),
			putStr( adrs->adrs.IndexReg ),
			_ifx
			(
				notEmpty( adrs->adrs.IndexReg ), 
				rtnScale( adrs ), 
				""
			),
			endoffset
		);
		
	_endif

_end( MakeOfsStr )









static char *
rootName( char *pathName )
_begin( rootName )

	char *root;

	root = strrchr( pathName, '\\' );
	_if( root == NULL )

		root = pathName;

	_else

		++root;

	_endif
	_return root;

_end( rootName )


void
SkeletalOutput( void )
_begin( SkeletalOutput )

	/*
	** Output the skeletal assembly language file.
	**
	** Here's the record format used by an exception record.
	** FS:[0] points at this.
	**
	**		record
	**			next:		dword;	//offset 0
	**			hwHandler:	dword;	//offset 4
	**			Coroutine:	dword;	//offset 8
	**			SavedEBP:	dword;	//offset 12
	**			HLAhndlr:	dword;	//offset 16
	**		endrecord;
	*/

	_if( assembler == masm ) // It's MASM

		asmPrintf
		( 
			"; Assembly code emitted by HLA compiler\n"
			"; %s\n"
			"; HLA compiler written by Randall Hyde\n"
			"; MASM compatible output\n"
			"\n"
			"\t\tif\t@Version lt 612\n"
			"\t\t.586p\n"
			"\t\telse\n" 
			"\t\t.686p\n"
			"\t\t.mmx\n"
			"\t\t.xmm\n" 
			"\t\tendif\n"
			"\t\t.model\tflat, syscall\n"
			"\t\toption\tnoscoped\n"
			"\n"
			"\n"
			"offset32\tequ\t<offset flat:>\n"
			"\n"
			"\t\tassume\tfs:nothing\n"
			"ExceptionPtr" sympost "\tequ\t<(dword ptr fs:[0])>\n"
			"\n",
			VersionInformation

		);

	_elseif( assembler == fasm ) // It's FASM
			
		_if( targetOS == windows_os )
						  
			asmPrintf
			( 
				"; Assembly code emitted by HLA compiler\n"
				"; %s\n"
				"; HLA compiler written by Randall Hyde\n"
				"; FASM compatible output\n"
				"\n"
				"\t\tformat\tMS COFF\n"
				"\n"
				"ExceptionPtr" sympost "\tequ\tfs:0\n",
				VersionInformation
			);
			
		_else
		
			asmPrintf
			( 
				"; Assembly code emitted by HLA compiler\n"
				"; %s\n"
				"; HLA compiler written by Randall Hyde\n"
				"; FASM compatible output\n"
				"\n"
				"\t\tformat\tELF\n",
				VersionInformation
			);
		
		_endif
		asmPrintf
		(
			"\n"
			"\n"
			"offset32\tequ\t \n"
			"ptr\tequ\t \n"	
			
			"\n"
			"macro global [symbol]\n"
			"{\n"
			" local isextrn\n"
			" if defined symbol & ~ defined isextrn\n"
			"   public symbol\n"
			" else if used symbol\n"
			"   extrn symbol\n"
			"   isextrn = 1\n"
			" end if\n"
			"}\n"
			"\n"			
			"macro global2 [symbol,type]\n"
			"{\n"
			" local isextrn\n"
			" if defined symbol & ~ defined isextrn\n"
			"   public symbol\n"
			" else if used symbol\n"
			"   extrn symbol:type\n"
			"   isextrn = 1\n"
			" end if\n"
			"}\n"
			"\n"			
			"\n"
		);

	_elseif( assembler == tasm )	// It's TASM

		asmPrintf
		( 
			"; Assembly code emitted by HLA compiler\n"
			"; %s\n"		  
			"; HLA compiler written by Randall Hyde\n"
			"; TASM 5.3 compatible output\n"
			"\n"
			"\t\t.686p\n"
			"\t\t.mmx\n"
			"\t\t.model\tflat, syscall\n"
			"\t\tmasm\n"
			"\t\tquirks\n"
			"\t\toption\tnoscoped\n"
			"\n"
			"offset32\tequ\t<offset FLAT:>\n"
			"\n"
			"\t\tassume\tfs:nothing\n"
			"ExceptionPtr" sympost "\tequ\t<(dword ptr fs:[0])>\n"
			"\n",
			VersionInformation
		);

	_elseif( assembler == gas ) // It's GAS

		asmPrintf
		( 
			"// Assembly code emitted by HLA compiler\n"
			"// %s\n"		  
			"// HLA compiler written by Randall Hyde\n"
			"// GAS compatible output\n"
			"\n"
			"\t\t.intel_syntax\tnoprefix\n"
			"\n",
			VersionInformation

		);
		


	_else

		fprintf( MsgOut, "Internal HLA error (bad assembler value)\n" );
		_return;			  

	_endif


	EmitImmExtern( "abstract" sympost, "near32" ); 
	EmitImmExtern( "Raise" sympost, "near32" );
	_if( targetOS == windows_os ) 
	
		EmitImmExtern( "HWexcept" sympost, "near32" ); 
		EmitImmExtern( "shortDfltExcept" sympost, "near32" ); 
		EmitImmExtern( "__imp__ExitProcess@4", "dword" ); 
		EmitImmExtern( "__imp__MessageBoxA@16", "dword" );
		
	_else
	
		// For Linux/BSD/MacOS, we need an external declaration for
		// the ExceptionPtr__hla_ variable:
		
		EmitImmExtern( ExceptionPtr, "dword" ); 
		EmitImmExtern( "shortDfltExcept" sympost, "near32" ); 
	
	_endif	
	asmPrintf( "\n\n\n" );
	
	// After emiting the skeletal output to the preamble buffer,
	// set asmBuf to point at the codeBuf:
	
	asmBuf = &codeBuf;


_end( SkeletalOutput )



   
/*****************************/
/*                           */
/* Segment naming functions: */
/*                           */
/*****************************/

char* DsegName		= "";
char* DsegAlign 	= "";
char* DsegClass		= "DATA";
int	  DsegUsed		= 0;

char* CsegName		= "";
char* CsegAlign		= "";
char* CsegClass		= "CODE";
int	  CsegUsed		= 0;

char* ROsegName		= "";
char* ROsegAlign	= "";
char* ROsegClass	= "CODE";
int	  ROsegUsed		= 0;

char* StrSegName	= "";
char* StrSegAlign	= "";
char* StrSegClass	= "CODE";
int	  StrSegUsed	= 0;

char* BssSegName	= "";
char* BssSegAlign	= "";
char* BssSegClass	= "BSS";
int	  BssSegUsed	= 0;


static outputBuf *outStack[256];
static int	outStk2[ 256 ][ 5 ];
static int	outSP = 0;

static void
PushOut( void )
_begin( PushOut )

	outStack[ outSP ] = asmBuf;
	outStk2[ outSP ][ 0 ] = DsegUsed;
	outStk2[ outSP ][ 1 ] = CsegUsed;
	outStk2[ outSP ][ 2 ] = BssSegUsed;
	outStk2[ outSP ][ 3 ] = ROsegUsed;
	outStk2[ outSP ][ 4 ] = StrSegUsed;
	++outSP;
	assert( outSP < 256 );

_end( PushOut );

static void
PopOut( void )
_begin( PopOut )

	assert( outSP > 0 );
	--outSP;
	asmBuf = outStack[ outSP ];
	DsegUsed = outStk2[ outSP ][ 0 ];
	CsegUsed = outStk2[ outSP ][ 1 ];
	BssSegUsed = outStk2[ outSP ][ 2 ];
	ROsegUsed = outStk2[ outSP ][ 3 ];
	StrSegUsed = outStk2[ outSP ][ 4 ];

_end( PopOut )



void 
startDseg( void )
_begin( startDseg )

	PushOut();
	asmBuf = &dataBuf;
	DsegUsed = 1;
	CsegUsed = 0;
	ROsegUsed = 0;
	BssSegUsed = 0;
	StrSegUsed = 0;

_end( startDseg )

void 
endDseg( void )
_begin( endDseg )

	PopOut();

_end( endDseg )

void 
startCseg( void )
_begin( startCseg )

	PushOut();
	asmBuf = &codeBuf;
	_if( assembler == gas )

		asmPrintf( "\t\t.text\n" );
		
	_endif
	DsegUsed = 0;
	CsegUsed = 1;
	ROsegUsed = 0;
	BssSegUsed = 0;
	StrSegUsed = 0;

_end( startCseg )

void 
endCseg( void )
_begin( endCseg )

	PopOut();

_end( endCseg )

void 
startROseg( void )
_begin( startROseg )

	PushOut();
	asmBuf = &roBuf;
	DsegUsed = 0;
	CsegUsed = 0;
	ROsegUsed = 1;
	BssSegUsed = 0;
	StrSegUsed = 0;

_end( startROseg )

void 
endROseg( void )
_begin( endROseg )

	PopOut();

_end( endROseg )

void 
startStrSeg( void )
_begin( startStrSeg )

	PushOut();
	asmBuf = &constBuf;
	DsegUsed = 0;
	CsegUsed = 0;
	ROsegUsed = 0;
	BssSegUsed = 0;
	StrSegUsed = 1;

_end( startStrSeg )

void 
endStrSeg( void )
_begin( endStrSeg )

	PopOut();

_end( endStrSeg )

void 
startBssSeg( void )
_begin( startBssSeg )

	PushOut();
	asmBuf = &bssBuf;
	DsegUsed = 0;
	CsegUsed = 0;
	ROsegUsed = 0;
	BssSegUsed = 1;
	StrSegUsed = 0;

_end( startBssSeg )

void 
endBssSeg( void )
_begin( endBssSeg )

	PopOut();

_end( endBssSeg )




static void
EmitExternDataSymbols( void )
_begin( EmitExternDataSymbols )
	
	int						index;
	struct		extRecs		*CurSym;

	_for( index=0, index < 2048, ++index )
	
		CurSym = extHashTable[ index ];
		_while( CurSym != NULL )

			_if
			(	
					!CurSym->IsPublic 
				&&	(
							CurSym->ForceRef				
						||	CurSym->theSym == NULL
					)
				&&	_strne( CurSym->Type, "near32" ) 
			)
															   
				_if( assembler == gas )

					asmPrintf
					(
						"\t\t.extern\t%s\n",
						CurSym->Name
					);
					
				_elseif( assembler == fasm )

					asmPrintf
					(
						"\t\tglobal2\t%s,%s\n",
						CurSym->Name,
						CurSym->Type
					);
					
				_else

					asmPrintf
					(
						"\t\texterndef %s:%s\n",

						CurSym->Name,
						CurSym->Type
					);

				_endif

			_endif
			CurSym = CurSym->Next;

		_endwhile

	_endfor
	
_end( EmitExternDataSymbols )


static void
EmitExternCodeSymbols( void )
_begin( EmitExternCodeSymbols )
	
	int						index;
	struct		extRecs		*CurSym;

	_for( index=0, index < 2048, ++index )
	
		CurSym = extHashTable[ index ];
		_while( CurSym != NULL )

			_if
			(
					!CurSym->IsPublic 
				&&	(
							CurSym->ForceRef				
						||	CurSym->theSym == NULL 
					)
				&&	_streq( CurSym->Type, "near32" ) 
			)

				_if( assembler == gas )

					asmPrintf
					(
						"\t\t.extern\t%s\n",
						CurSym->Name
					);

				_elseif( assembler == fasm )

					asmPrintf
					(
						"\t\tglobal\t%s\n",
						CurSym->Name
					);
						
				_else

					asmPrintf
					(
						"\t\texterndef %s:%s\n",
						CurSym->Name,
						CurSym->Type
					);

				_endif
				
			_endif
			CurSym = CurSym->Next;

		_endwhile

	_endfor
	
_end( EmitExternCodeSymbols )





void
EmitDataSegments( outputBuf *dest )
_begin( EmitDataSegments )

	outputBuf *save = asmBuf;
	asmBuf = dest;

	
	/*
	** Emit data segments to control their order
	** and so the linker won't complain about missing
	** segments in the ".link" file.
	*/

	_if( assembler == fasm )
	
		_if( targetOS == windows_os )
		
			asmPrintf
			(
				"\n\n" 
				"\t\tsection\t'.data' data readable writeable align 16\n\n"
			);
			
		_else
		
			asmPrintf
			( 
				"\n\n" 
				"\t\tsection\t'.data' writeable align 16\n\n"
			);
			
		_endif
		
		// Emit the external data symbols here:
		
		EmitExternDataSymbols();
		
		// Now emit all the data associated with the data section:
		
		asmCpy( dataBuf.base, dataBuf.offset );
		
		// Note: FASM thinks that this section contains uninitialized
		// data if there are any uninitialized objects at the end of
		// the section. Emit an honest-to-God data declaration here
		// to prevent that from ever happening. 
		
		asmPrintf( "\t\tdd\t0ffffffffh ;Dummy, to mark end of data section\n\n" );
				
		
	_elseif( assembler == gas )
	
		asmPrintf
		( 
			"\n\n"
			"\t\t.data\n\n"
		);
		
		// Emit the external data symbols here:
		
		EmitExternDataSymbols();
		
		// Emit the data for this section:

		asmCpy( dataBuf.base, dataBuf.offset );

	_elseif
	( 
			strlen( DsegName ) == 0 
		||	_streq( DsegName, ".data" ) 
		||	assembler == gas
	)
		
		asmPrintf
		( 
			"\n\n"
			"\t\t.data\n\n"
		);
		
		// Emit the external data symbols here:
		
		EmitExternDataSymbols();
		
		// Emit the data for this section:

		asmCpy( dataBuf.base, dataBuf.offset );
		
		// Emit extra data at the end to guarantee that
		// this segment contains something (in case they're
		// using polink, which complains if there is no data
		// in the section):
		
		asmPrintf
		( 
			"\t\tdd\t0ffffffffh ;Dummy, to mark end of data section\n\n" 
		);
		
	_else
	

		EmitSegment( DsegName, DsegAlign, DsegClass );
		
		// Emit the external data symbols here:
		
		EmitExternDataSymbols();
		
		// Emit the section's data:
		
		asmCpy( dataBuf.base, dataBuf.offset );
		
		// Emit extra data at the end to guarantee that
		// this segment contains something (in case they're
		// using polink, which complains if there is no data
		// in the section):
		
		asmPrintf
		( 
			"\t\tdd\t0ffffffffh ;Dummy, to mark end of data section\n\n" 
		);
		
		EndSeg( DsegName );
				
	_endif
		

	// Emit the BSS segment here

	
	_if( assembler == gas )

			 
		asmPrintf
		(
			"\n\n" 
			"\t\t.bss\n\n"
		);
		asmCpy( bssBuf.base, bssBuf.offset );
		
	_elseif( assembler == fasm )
	
		_if( targetOS == windows_os )

			_if( useCFASM )
			
				// If using internal FASM, create a "BSS" typed section.
				
				asmPrintf
				( 
					"\n\n"
					"\t\tsection\t'.bss' bss readable writeable align 16\n\n"
				);
				asmCpy( bssBuf.base, bssBuf.offset );
				
			_else
			
				asmPrintf
				( 
					"\n\n"
					"\t\tsection\t'.bss'  readable writeable align 16\n\n"
				);
				asmCpy( bssBuf.base, bssBuf.offset );
				
			
			_endif
			asmPrintf
			( 
				"\t\tdd\t?\t;Forces FASM to set the uninitialized data flag\n"
				"\n" 
			);
			
		_else

			asmPrintf
			( 
				"\n\n"
				"\t\tsection\t'.bss' writeable align 16\n\n"
				
			);
			asmCpy( bssBuf.base, bssBuf.offset );
			
		_endif
		
		
		
	_elseif( strlen( BssSegName ) == 0 || _streq( BssSegName, ".data?" ))
		
		asmPrintf
		( 
			"\n\n"
			"\t\t.data?\n\n"
		);
		asmCpy( bssBuf.base, bssBuf.offset );		
		
	_else
	
		EmitSegment( BssSegName, BssSegAlign, BssSegClass );
		asmCpy( bssBuf.base, bssBuf.offset );
		EndSeg( BssSegName );
				
	_endif
		
	asmPrintf( "\n\n" );	
	asmBuf = save;	

_end( EmitDataSegments )







void
EmitReadonlySegments( outputBuf *dest, int codeFirst )
_begin( EmitReadonlySegments )

	outputBuf *save = asmBuf;
	asmBuf = dest;

	_if( !codeFirst )
	
		// All the readonly sections (const, readonly, and code) go in the
		// same segment -- the .code/.text segment.	
			

		// Emit the CONST segment here

		
		_if( assembler == gas )


			asmPrintf
			( 
				"\n\n"
				"\t\t.text\n\n"
			);
			asmCpy( constBuf.base, constBuf.offset );


		_elseif( assembler == fasm )
		
			_if( targetOS == windows_os )
				
				asmPrintf
				( 
					"\n\n"
					"\t\tsection\t'.text' code readable executable align 16\n\n"
				);
				
			_else
			
				asmPrintf
				( 
					"\n\n"
					"\t\tsection\t'.text' executable align 16\n\n"
				);
				
			_endif
			asmCpy( constBuf.base, constBuf.offset );
			
			
		_elseif( strlen( StrSegName ) == 0 || _streq( StrSegName, ".code" ))
			
			asmPrintf
			( 
				"\n\n"
				"\t\t.code\n\n" 
			);
			asmCpy( constBuf.base, constBuf.offset );
			
		_else
		
			EmitSegment( StrSegName, StrSegAlign, StrSegClass );
			asmCpy( constBuf.base, constBuf.offset );
			EndSeg( StrSegName );
			
		_endif
			
			

		// Emit the read-only segment here


			
		_if( assembler == gas )
		
			_if( constBuf.offset == 0 )
			
				asmPrintf
				( 
					"\n\n"
					"\t\t.text\n\n"
				);
				
			_endif
			asmCpy( roBuf.base, roBuf.offset );

			
		_elseif( assembler == fasm )
		
			_if( constBuf.offset == 0 )
			
				_if( targetOS == windows_os )
					
					asmPrintf
					( 
						"\n\n"
						"\t\tsection\t'.text' code "
						              "readable executable align 16\n\n"
					);
					
				_else
				
					asmPrintf
					( 
						"\n\n"
						"\t\tsection\t'.text' executable align 16\n\n"
					);
					
				_endif
				
			_endif
			asmCpy( roBuf.base, roBuf.offset );
			
			
		_elseif( strlen( ROsegName ) == 0 || _streq( ROsegName, ".code" ))

			
			_if( constBuf.offset == 0 )
			
				asmPrintf
				( 
					"\n\n"
					"\t\t.code\n\n" 
				);
				
			_endif
			asmCpy( roBuf.base, roBuf.offset );

		_else
		
			EmitSegment( ROsegName, ROsegAlign, ROsegClass );
			asmCpy( roBuf.base, roBuf.offset );
			EndSeg( ROsegName );

		_endif
		
	_endif // !codeFirst
		
	// Start us off in the code segment:
	
	 
	 _if( assembler == gas )

		asmPrintf
		( 
			"\n"
			"\n" 
			"\t\t.text\n"
			"\n" 
		);

	_elseif( assembler == fasm )
	
	
		_if( targetOS == windows_os )
			
			asmPrintf
			( 
				"\n\n"
				"\t\tsection\t'.text' code readable executable align 16\n\n"
			);
			
		_else
		
			asmPrintf
			( 
				"\n\n"
				"\t\tsection\t'.text' executable align 16\n\n"
			);
			
		_endif
	
		
	_elseif( strlen( CsegName ) == 0 || _streq( CsegName, ".code" ))
		
		asmPrintf
		( 
			"\n"
			"\n" 
			"\t\t.code\n"
			"\n" 
		);
		
	_else
	
		EmitSegment( CsegName, CsegAlign, CsegClass );
		
	_endif
	
	asmPrintf( "\n\n" );	
	asmBuf = save;	

_end( EmitReadonlySegments )


void
extPubIterator( outputBuf *output )
_begin( extPubIterator )

	unsigned				index;
	struct		extRecs		*CurSym;
	struct		bpList_t 	*curBP;
	struct		bpList_t 	*freeBP;
	outputBuf				*save = asmBuf;
	struct		SymNode		*sym;
	
	// This function used to emit the external symbols.
	// However, that code was moved to the segment output function.

	asmBuf = output;

	// Emit the public symbols (must go in the text segment):

	_if( assembler == fasm )
	
		_if( targetOS == windows_os )
			
			asmPrintf
			( 
				"\n\n"
				"\t\tsection\t'.text' code readable executable align 16\n\n"
			);
			
		_else
		
			asmPrintf
			( 
				"\n\n"
				"\t\tsection\t'.text' executable align 16\n\n"
			);
			
		_endif
				
	_else

		asmPrintf
		( 
			"\n\t\t%s\n", ifgas( ".text", ".code" )
		);
		
	_endif

	asmPrintf( "\n" );
	_for( index=0, index < 2048, ++index )
	
		CurSym = extHashTable[ index ];
		_while( CurSym != NULL )

			_if( CurSym->IsPublic )

				asmPrintf
				(
					"\t\t%s\t%s\n",
					ifgas( ".global", "public" ),
					CurSym->Name
				);

			_endif
			CurSym = CurSym->Next;

		_endwhile

	_endfor
	
	// Emit all the external code symbols here:
	
	EmitExternCodeSymbols();	
	asmPrintf( "\n\n" );
	
	
	// Output all the forward EQU definitions:

	curBP = bpList;
	_while( curBP != NULL )

		asmPrintf
		(
			"%s\n",
			curBP->bpatch
		);
		free2( vss curBP->bpatch );
		freeBP = curBP;
		curBP = curBP->Next;
		free2( vss freeBP );
									
	_endwhile
	asmPrintf( "\n\n" );
	asmBuf = save;
	
_here;

_end( extPubIterator );







/*****************************************************
/*                                                  */
/* Procedure emission functions-                    */
/*                                                  */
/* Output the procedure beginning and ending        */
/* statements.                                      */
/*                                                  */
/* StartProc-	Emits  xxxx proc near32  (or        */
/* 			whatever syntax the assembler           */
/* 			uses for the start of a procedure).     */
/*                                                  */
/* EndProc-	Emits xxxx endp (or whatever syntax     */
/* 			the assembler uses).                    */
/*                                                  */
/* Input:                                           */
/*                                                  */
/* 	procname-	Name of the procedure to start/end. */
/*                                                  */
/****************************************************/

void
StartProc( char *ProcName )
_begin( StartProc )

	_if( assembler == gas )
	
		_if( targetOS == windows_os )

			asmPrintf
			(
				"%-15s:\n",
				ProcName
			);
		
		_else // Linux, MacOS, or FreeBSD
		
			asmPrintf
			(
				"\t\t.type\t%s,@function\n"
				"%-15s:\n",
				ProcName,
				ProcName
			);
			
		_endif
	
	_elseif( assembler == fasm )
	
		asmPrintf
		(
			"%-15s:\n",
			ProcName
		);

	_else
	
		asmPrintf
		(
			"%-15s proc\tnear32\n",
			ProcName
		);
		
	_endif

_end( StartProc )


void
EndProc( char *ProcName )
_begin( EndProc )

	_if( assembler == gas )

		
		asmPrintf
		(
			"End_%s_%d:\n"
			"\t\t.size\t%s,End_%s_%d-%s\n\n",
			ProcName,
			LblCntr,
			ProcName,
			ProcName,
			LblCntr,
			ProcName
		);
			
		++LblCntr;
		
	_elseif( assembler == fasm )
		
		asmPrintf
		(
			";%-15s endp\n"
			"\n",
			ProcName
		);

	_else
	
		asmPrintf
		(
			"%-15s endp\n"
			"\n",
			ProcName
		);

	_endif

_end( EndProc )


/*************************************************
/*                                              */
/* AlignStack-                                  */
/*                                              */
/* Emits code to align the stack on a four-byte */
/* boundary (or whatever the OS wants)          */
/*                                              */
/************************************************/

void
AlignStack( void )
_begin( AlignStack )


	asmPrintf
	(
		"\t\tand\tesp, 0%sfffffffc%s\n",
		ifgas( "x", "" ),
		ifgas( "", "h" )
	);


_end( AlignStack )


/*********************************************************
/*                                                      */
/* AllocateLocals-                                      */
/*                                                      */
/* Allocates the specified amount of local variable     */
/* storage by dropping the stack down the specified     */
/* number of bytes.                                     */
/*                                                      */
/* Inputs                                               */
/*                                                      */
/* 	localSize-	Number of bytes of local storage.       */
/* 				Note: value must be a multiple of four. */
/*                                                      */
/********************************************************/

void
AllocateLocals( unsigned localSize )
_begin( AllocateLocals )

	_if( localSize > 0 )

		asmPrintf
		(
			"\t\tsub\tesp,%5d\n",
			localSize
		);

	_endif

_end( AllocateLocals )



/**************************************************
/*                                               */
/* StdEntry-                                     */
/*                                               */
/* Sets up the activation record upon entry into */
/* a procedure.                                  */
/*                                               */
/*************************************************/

void
StdEntry( int RoundedOffset, int NoEnter )
_begin( StdEntry )

	_if( NoEnter )
	
		asmPrintf
		( 
			"\t\tpush\tebp\n"
			"\t\tmov\tebp, esp\n" 
		);
		AllocateLocals( RoundedOffset );
		
	_else
	
		asmPrintf( "\t\tenter\t%d,0\n", RoundedOffset );
		
	_endif
	


_end( StdEntry )




/*******************************************************
/*                                                    */
/* StdExit-                                           */
/*                                                    */
/* Converse of the above.  Cleans up the activation   */
/* record and returns from a procedure.               */
/*                                                    */
/* Inputs-                                            */
/* 	StkSize-	Number of bytes to remove from stack  */
/* 				upon returning from this procedure.   */
/*                                                    */
/* 	cdeclProc-	True if this is a CDECL procedure and */
/* 				it's the callers responsibility to    */
/* 				clean up the parameters on the stack. */
/*                                                    */
/******************************************************/


void
StdExit( int StkSize, int cdeclProc, int NoLeave )
_begin( StdExit )

	_if( !NoLeave )
	
		asmPrintf( "\t\tleave\n" );
		
	_else
	
		asmPrintf( "\t\tmov\tesp, ebp\n" );
		asmPrintf( "\t\tpop\tebp\n" );
		
	_endif
	_if( StkSize == 0 || cdeclProc )

		asmPrintf( "\t\tret\n" );

	_else

		asmPrintf( "\t\tret\t%d\n", StkSize );

	_endif

_end( StdExit )


/*
** IteratorExit - StdExit for iterators.
** Note that iterators always using the Pascal calling convention,
** so there is no need for the cdeclProc parameter.
*/

void
IteratorExit( int StkSize )
_begin( IteratorExit )

	asmPrintf
	( 
		"\t\tmov\tesp, ebp\n" 
		"\t\tpop\tebp\n"
		"\t\tadd\tesp, 4\t;/*Remove succeed address*/\n" 
	);
	_if( StkSize == 0 )

		asmPrintf( "\t\tret\n" );


	_else

		asmPrintf( "\t\tret\t%d\n", StkSize );

	_endif

_end( IteratorExit )


/***********************************************************
/*                                                        */
/* EmitSegment-                                           */
/*                                                        */
/* Emits a segment declaration of the form (MASM syntax): */
/*                                                        */
/* 	segname  SEGMENT  align  PUBLIC 'className'           */
/*                                                        */
/**********************************************************/

void
EmitSegment
(
	char *segName,
	char *align,
	char *className
)
_begin( EmitSegment )

	_if( assembler == gas || assembler == fasm )

		yyerror( "SEGMENT directive illegal when emitting GAS/FASM output" );

	_endif
	asmPrintf
	(
		"%-15s segment\t%s public '%s'\n",
		segName,
		align,
		className
	);

_end( EmitSegment )

void
EndSeg( char *SegName )
_begin( EndSeg )

	_if( assembler != gas )

		asmPrintf
		(
			"%-15s ends\n\n",
			SegName
		);

	_endif

_end( EndSeg )


/****************************************************
/*                                                 */
/* ConstructDisplay-                               */
/*                                                 */
/* Builds the "display" in the activation record.  */
/*                                                 */
/* Input:                                          */
/* 	CurLexLevel-	Current lex level for the      */
/* 					procedure whose display we are */
/* 					constructing.                  */
/*                                                 */
/***************************************************/

void
ConstructDisplay( int CurLexLevel, int  RoundedOffset, int NoEnter )
_begin( ConstructDisplay )

	_if( !NoEnter )
	
		asmPrintf( "\t\tenter\t%d, %d\n", RoundedOffset, CurLexLevel+1 );
	
	_else
	
		// Construct the display manually:
		
		asmPrintf
		( 
			"\t\tpush\tebp\t\t;/*Dynamic link*/\n" 
		);
		_for( int i=1, i <= CurLexLevel, ++i )

			asmPrintf
			(
				"\t\tpush\tdword ptr [ebp-%d]"
				"\t;/*Display for lex level %d*/\n",
				i*4,
				i - 1
			);

		_endfor

		asmPrintf
		(
			"\t\tlea\tebp,[esp+%d]\t;/*Get frame ptr*/\n"
			"\t\tpush\tebp\t\t;/*Ptr to this proc's A.R.*/\n",
			CurLexLevel * 4
		);

		AllocateLocals( RoundedOffset );
		
	_endif

_end( ConstructDisplay )





/**************************************************************
**
** Handle all the calls:
**
** call_proc- Calls a procedure referenced by a SymNode object.
** call_thunk- calls a thunk object specified by a memory address.
** call_mem- calls a procedure via a dword pointer.
*/

void 
call_proc
( 
	struct SymNode *proc 
)
_begin( call_proc )

	assert( proc != NULL );

	asmPrintf
	(
		"\t\tcall\t%s\n",
		proc->StaticName
	);
			

_end( call_proc )


void
call_thunk( union YYSTYPE *adrs )
_begin( call_thunk )

	push_mem( adrs, 4 );
	adrs->adrs.Disp += 4;
	call_mem( adrs );
	adrs->adrs.Disp -= 4;
	SetReferenced( adrs->adrs.Sym );

_end( call_thunk )



void
call_mem( union YYSTYPE *adrs )
_begin( call_mem )

	char address[ 256 ];

	MakeMemStrSize( address, adrs, 4 );
	asmPrintf
	(
		"\t\tcall\t%s\n",
		address
	);
			
	SetReferenced( adrs->adrs.Sym );

_end( call_mem )


/**************************************************************
**
** cld instruction output
**
*/


void
cld( void )
_begin( cld )

	_switch( assembler )
	
		_case( masm )
		_case( fasm )
		_case( tasm )
		_case( gas )

			asmPrintf
			(
				"\t\tcld\n"
			);
			
		_endcase
					
		

		_default

			yyerror( "Internal HLA error (Bad assembler value)" );

	_endswitch

_end( cld )




/**************************************************************
**
** lea instruction output
**
**	lea_rm - loads a 32-bit register with the address of a mem location.
**
*/

void
lea_rm( int theReg, union YYSTYPE *adrs )
_begin( lea_rm )

	char address[256];
	
	MakeMemStr( address, YYS adrs, 0 );

	// Special case to avoid reloading a register with itself
	// for method invocations:


	_returnif
	( 
			theReg == reg_esi 
		&&	(
					_streq( address, " [esi+0]" )
				||	
					_streq( address, " [esi]" )
			)
	);
		
	asmPrintf
	(
		"\t\tlea\t%s, %s\n",
		gpregmap[theReg][assembler],
		address
	);
			


_end( lea_rm )


/**************************************************************
**
** mov instruction output
**
**	mov_or - loads the 32-bit offset of a mem location into a 32-bit register.
**	mov_cr - loads a constant into a register.
**	mov_ir - like mov_cr, expect the constant argument is an immediate constant
**	mov_mr - moves the mem location into a register.
**	mov_rm - moves the register into a mem location.
**	mov_rr - moves data between two registers.
**
*/

void
mov_or( union YYSTYPE *adrs, int theReg )
_begin( mov_or )

	char address[256];
	
	MakeOfsStr( address, YYS adrs );
	asmPrintf
	(
		"\t\tmov\t%s, %s\n",
		gpregmap[theReg][assembler],
		address
	);
			


_end( mov_or )


void
mov_cr( union YYSTYPE *theConst, int theReg )
_begin( mov_cr )

	int	 needsOffset;
	char constant[256];
	
	needsOffset = StaticConstToStr( theConst->v.Type, YYS theConst, constant );
	_switch( assembler )
	
		_case( masm )
		_case( tasm )
		_case( fasm )
		_case( gas )

			asmPrintf
			(
				"\t\tmov\t%s, %s%s\n",
				gpregmap[theReg][assembler],
				needsOffsetStr( needsOffset ),
				constant
			);
			
		_endcase
		
		
		_default
			yyerror( "Internal HLA error" );

	_endswitch		
	
_end( mov_cr )



void
mov_ir( unsigned theConst, int theReg )
_begin( mov_ir )

	_switch( assembler )
	
		_case( masm )
		_case( tasm )
		_case( fasm )
		_case( gas )

			asmPrintf
			(
				"\t\tmov\t%s, %d\n",
				gpregmap[theReg][assembler],
				theConst
			);
			
		_endcase
		
		

		_default
			yyerror( "Internal HLA error" );

	_endswitch		
	
_end( mov_ir )


void
mov_mr( union YYSTYPE *adrs, int theReg, int size )
_begin( mov_mr )

	char address[256];
	
	_if( size == 0 )
	
		MakeMemStr( address, YYS adrs, 1 );
		
	_else
	
		MakeMemStrSize( address, YYS adrs, size );
		
	_endif
	asmPrintf
	(
		"\t\tmov\t%s, %s\n",
		gpregmap[theReg][assembler],
		address
	);

_end( mov_mr )


void
mov_rm( int theReg, union YYSTYPE *adrs, int size )
_begin( mov_rm )

	char address[256];
	
	_if( size == 0 )
	
		MakeMemStr( address, YYS adrs, 1 );
		
	_else
	
		MakeMemStrSize( address, YYS adrs, size );
		
	_endif
	asmPrintf
	(
		"\t\tmov\t%s, %s\n",
		address,
		gpregmap[theReg][assembler]
	);

_end( mov_rm )



void 
mov_rr( int srcReg, int destReg )
_begin( mov_rr )

	asmPrintf
	(
		"\t\tmov\t%s, %s\n",
		gpregmap[destReg][assembler],
		gpregmap[srcReg][assembler]
	);


_end( mov_rr )


/**************************************************************
**
** movd instruction output
**
**	movd_r32_mmx
**	movd_mmx_r32
**	movd_mmx_m
**
*/




void 
movd_r32_mmx( int r32, int mmx )
_begin( movd_r32_mmx )

	_switch( assembler )
	
		_case( masm )
		_case( tasm )
		_case( fasm )
		_case( gas  )

			asmPrintf
			(
				"\t\tmovd\t%s, %s\n",
				mmxregmap[mmx][assembler],
				gpregmap[r32][assembler]
			);
			
		_endcase
		
		_default
			yyerror( "Internal HLA error" );
		
	_endswitch


_end( movd_r32_mmx )





void 
movd_mmx_r32( int mmx, int r32 )
_begin( movd_mmx_r32 )

	_switch( assembler )
	
		_case( masm )
		_case( tasm )
		_case( fasm )
		_case( gas  )

			asmPrintf
			(
				"\t\tmovd\t%s, %s\n",
				gpregmap[r32][assembler],
				mmxregmap[mmx][assembler]
			);
			
		_endcase
		
		_default
			yyerror( "Internal HLA error" );
		
	_endswitch


_end( movd_mmx_r32 )




void 
movd_mmx_m( int mmx, union YYSTYPE *adrs )
_begin( movd_mmx_m )

	char address[256];
	
	MakeMemStrSize( address, YYS adrs, 4 );
	asmPrintf
	(
		"\t\tmovd\t%s, %s\n",
		address,
		mmxregmap[mmx][assembler]
	);

_end( movd_mmx_m )







void 
movd_m_mmx( union YYSTYPE *adrs, int mmx )
_begin( movd_m_mmx )

	char address[256];
	
	MakeMemStrSize( address, YYS adrs, 4 );
	asmPrintf
	(
		"\t\tmovd\t%s, %s\n",
		mmxregmap[mmx][assembler],
		address
	);
			

_end( movd_m_mmx )



/**************************************************************
**
** movs instruction output
**
**	movsb
**	movsd
**	movsw
**
*/


void
movsb( void )
_begin( movsb )

	asmPrintf
	(
		"\t\tmovsb\n"
	);
			
_end( movsb )





void
movsd( void )
_begin( movsd )

	asmPrintf
	(
		"\t\tmovsd\n"
	);

_end( movsd )





void
movsw( void )
_begin( movsw )

	asmPrintf
	(
		"\t\tmovsw\n"
	);
			

_end( movsw )







/**************************************************************
**
** movzx instruction output
**
**	movzx_mr - moves the mem location into a register.
**
*/




void
movzx_mr( union YYSTYPE *adrs, int theReg )
_begin( movzx_mr )

	char address[256];
	
	MakeMemStr( address, YYS adrs, 1 );
	asmPrintf
	(
		"\t\tmovzx\t%s, %s\n",
		gpregmap[theReg][assembler],
		address
	);
			

_end( movzx_mr )




/**************************************************************
**
** pop instruction output
**
**	pop_m	- pops a memory value from the stack.
**	pop_r	- pops a register value from the stack.
**
*/

void
pop_m( union YYSTYPE *adrs, int forcedSize )
_begin( pop_m )

	char address[ 256 ];

	_if( forcedSize == 0 )

		StoreAdrsSize( address, adrs, 0 );	
		
	_else
	
		StoreAdrsForceSize( address, adrs, forcedSize );
		
	_endif
	asmPrintf
	(
		"\t\tpop\t%s\n",
		address
	);

_end( pop_m )



void
pop_r( int srcReg )
_begin( pop_r )

	asmPrintf
	(
		"\t\tpop\t%s\n",
		gpregmap[srcReg][assembler]
	);

_end( pop_r )






void
popfd( void )
_begin( popfd )

	asmPrintf
	(
		"\t\tpopfd\n"
	);

_end( popfd )




/**************************************************************
**
** push instruction output
**
**	push_mem	- pushes a memory value onto stack.
**	push_r		- pushes a register onto the stack.
**	push_const	- pushes a constant value onto the stack.
**	push_offset - pushes an offset of a label onto the stack.
**	pushd		- dword immediate version of push_const.
**	pushw		- word immediate version of push_const.
**	pushfd		- pushes the flags register onto the stack.
**	pushf		- pushes the 16-bit flags register onto the stack.
**
*/

void
push_mem( union YYSTYPE *adrs, int forcedSize )
_begin( push_mem )

	char address[ 256 ];

	_if( forcedSize == 0 )

		StoreAdrsSize( address, adrs, 0 );	

		
	_else
	
		StoreAdrsForceSize( address, adrs, forcedSize );
		
	_endif
	asmPrintf
	(
		"\t\tpush\t%s\n",
		address
	);
			

_end( push_mem )



void
push_r( int srcReg )
_begin( push_r )

	asmPrintf
	(
		"\t\tpush\t%s\n",
		gpregmap[srcReg][assembler]
	);
			

_end( push_r )



void
push_const( union YYSTYPE *theConst, int size )
_begin( push_const )

	int  needsOffset;
	char constant[ 256 ];
	
	
	needsOffset = StaticConstToStr( theConst->v.Type, YYS theConst, constant );
	asmPrintf
	(
		"\t\tpush%s\t%s%s\n",
		_ifx( size != 4, "w", "d" ),
		needsOffsetStr( needsOffset ),
		constant
	);

_end( push_const )

void
Pushd( unsigned d )
_begin( Pushd )

	_switch( assembler )
	
		_case( fasm )
		_case( masm )

			asmPrintf
			(
				"\t\tpushd\t0%xh\n",
				d
			);
		
		_endcase

		_case( tasm )


			asmPrintf
			(
				"\t\tpush\t0%xh\n",
				d
			);
			
		_endcase

		_case( gas )


			asmPrintf
			(
				"\t\tpushd\t0x%x\n",
				d
			);
		
		_endcase


		_default

			yyerror( "Internal HLA error (Bad assembler value)" );

	_endswitch

_end( Pushd )


void
Pushw( unsigned w )
_begin( Pushw )

	w &= 0xffff;
	_switch( assembler )
	
		_case( fasm )
		_case( masm )
		
			asmPrintf
			(
				"\t\tpushw\t0%xh\n",
				w
			);
			
		_endcase

		_case( tasm )


			asmPrintf
			(
				"\t\tpush\t0%xh\n",
				w
			);
			
		_endcase

		_case( gas )


			asmPrintf
			(
				"\t\tpushw\t0x%x\n",
				w
			);
			
		_endcase
		

		_default

			yyerror( "Internal HLA error (Bad assembler value)" );

	_endswitch

_end( Pushw )



void
pushfd( void )
_begin( pushfd )

	asmPrintf
	(
		"\t\tpushfd\n"
	);
			
_end( pushfd )





void
pushf( void )
_begin( pushf )

	asmPrintf
	(
		"\t\tpushf\n"
	);

_end( pushf )






void
push_offset( union YYSTYPE *theConst, int size )
_begin( push_offset )

	int  needsOffset;
	char constant[ 256 ];
	
	
	StaticConstToStr( theConst->v.Type, YYS theConst, constant );

	asmPrintf
	(
		"\t\tpush%s\t%s%s\n",
		_ifx( size != 4, "w", "d" ),
		needsOffsetStr( 1 ),
		constant
	);

_end( push_offset )


/**************************************************************
**
** rep_xxx instruction output
**
**	rep_movsb
**	rep_movsd
**	rep_movsw
**
*/


void
rep_movsb( void )
_begin( rep_movsb )

	asmPrintf
	(
		"\trep\tmovsb\n"
	);

_end( rep_movsb )





void
rep_movsd( void )
_begin( rep_movsd )

	asmPrintf
	(
		"\trep\tmovsd\n"
	);

_end( rep_movsd )





void
rep_movsw( void )
_begin( rep_movsw )

	asmPrintf
	(
		"\trep\tmovsw\n"
	);

_end( rep_movsw )





/**************************************************************
**
** sub instruction output
**
**	sub_ir - subtract an immediate constant from a register
**
*/

void
sub_ir( int theConst, int dest )
_begin( sub_ir )

	asmPrintf
	(
		"\t\tsub\t%s, %d\n",
		gpregmap[dest][assembler],
		theConst
	);

_end( sub_ir )







/**************************************************************
**
** xor instruction output
**
**	xor_rr - register to register XOR operation
**
*/

void
xor_rr( int src, int dest )
_begin( xor_rr )

		
	asmPrintf
	(
		"\t\txor\t%s, %s\n",
		gpregmap[dest][assembler],
		gpregmap[src][assembler]
	);
			

_end( xor_rr )









/*************************************************************
/*                                                          */
/* Label emission functions:                                */
/*                                                          */
/* EmitLabel- Emits byte, word, dword, or qword labels,     */
/* depending on the value of the second parameter.          */
/*                                                          */
/* EmitTypedLabel- Emits an arbitrary typed label where     */
/* the second parameter is a string specifying the label's  */
/* type.                                                    */
/*                                                          */
/* EmitTypedLabelNum - The label string contains a "%d"     */
/* format operand into which this function must insert      */
/* the decimal representation of the "num" parameter.       */                                             
/*                                                          */
/************************************************************/





void
EmitLabel
( 
	char *theLabel, 
	unsigned size 
)
_begin( EmitLabel )

	_if( assembler == gas )

		asmPrintf( "%s:\n", theLabel );

	_elseif( assembler == fasm )
	
		asmPrintf
		( 
			"\t\tlabel\t%s %s\n", 
			theLabel,
			_ifx(  size == 2, "word",
			 _ifx( size == 4, "dword",
			 _ifx( size == 8, "qword",
			 "byte"
			)))
		);

	_else 

		asmPrintf
		( 
			"%-15s label\t%s\n", 
			theLabel,
			_ifx(  size == 2, "word",
			 _ifx( size == 4, "dword",
			 _ifx( size == 8, "qword",
			 "byte"
			)))
		);

	_endif

	
_end( EmitLabel )

void
EmitTypedLabel
( 
	char *theLabel, 
	char *labelType 
)
_begin( EmitTypedLabel )

	_if( assembler == gas )

		asmPrintf( "%s:\n", theLabel );
		
	_elseif( assembler == fasm )

		_if( labelType[0] == 'd' && labelType[1] == 'b' )
		
			asmPrintf
			( 
				"\t\tlabel\t%s byte\n", 
				theLabel
			);
		
		_elseif(  labelType[0] == 'd' && labelType[1] == 'w' )
		
			asmPrintf
			( 
				"\t\tlabel\t%s word\n", 

				theLabel
			);
		
		_elseif(  labelType[0] == 'd' && labelType[1] == 'd' )
		
			asmPrintf
			( 
				"\t\tlabel\t%s dword\n", 
				theLabel
			);
		
		_elseif(  labelType[0] == 'd' && labelType[1] == 'q' )
		
			asmPrintf
			( 
				"\t\tlabel\t%s qword\n", 
				theLabel
			);
		
		_elseif(  labelType[0] == 'd' && labelType[1] == 't' )
		
			asmPrintf
			( 
				"\t\tlabel\t%s tword\n", 
				theLabel
			);
		
		_elseif( _streq( labelType, "near32" ))
			
			asmPrintf
			( 
				"%s:\n", 
				theLabel
			);
			
		_else
		
			asmPrintf
			( 
				"\t\tlabel\t%s byte\n", 
				theLabel
			);
			
		_endif

	_else

		asmPrintf
		( 
			"%-15s label\t%s\n", 
			theLabel,
			labelType
		);

	_endif

	
_end( EmitTypedLabel )



void
EmitTypedLabelNum
( 
	char *theLabel, 
	char *labelType,
	int	 num 
)
_begin( EmitTypedLabelNum )

	char lbl[256];

	sprintf( lbl, theLabel, num );

	_if( assembler == gas )

		asmPrintf( "%s:\n", lbl );

	_else

		EmitTypedLabel( lbl, labelType );

	_endif
	
_end( EmitTypedLabelNum )




void
extLookup
( 
	struct	SymNode	*sym, 
	char	*theLabel, 
	char	*theType, 
	char	IsPublic,
	char	ForceRef,
	char	isVMT 
)
_begin( extLoopkup )

	unsigned				index;
	char					*CurChar;
	struct			extRecs	*CurSym; 
	
	/*
	** Cheesy hash table function.
	*/

	index = 0;
	CurChar = theLabel;
	_while( *CurChar != '\0' )

		/*
		** index = (index byteROL 1) ^ CurrentChar
		*/

		index = (index << 1);
		_if( index >= 2048 )
		
			index = index ^ 2049;

		_endif
		index ^= *CurChar;
		++CurChar;

	_endwhile

	// Just for good measure, XOR in the length of the string
	// too.

	index ^= (CurChar - theLabel ) & 2047;
		
	// Okay, this gives us an index into the hash table.
	// See if there's a corresponding entry in the hash table.

	_if( extHashTable[ index ] == NULL )

		// We're in business.  There is no hash entry so there
		// cannot be an instance of this symbol yet.

		extHashTable[ index ] = malloc2( sizeof( struct extRecs));
		extHashTable[ index ]->Next = NULL;
		extHashTable[ index ]->Name = hlastrdup2( theLabel );
		extHashTable[ index ]->Type = theType;
		extHashTable[ index ]->IsPublic = 0;	// Assume it's external.
		extHashTable[ index ]->theSym = sym;
		extHashTable[ index ]->ForceRef = ForceRef;
		CurSym = extHashTable[ index ];

	_else

		// If there's an entry in the hash table slot, then
		// we've got to search through the list to see if we've
		// got an instance of this label.

		CurSym = extHashTable[ index ];
		_while
		(
				CurSym != NULL
			&& _strne( CurSym->Name, theLabel )
		)

			CurSym = CurSym->Next;

		_endwhile
		_if( CurSym == NULL )

			// Didn't find the symbol.  So enter it into
			// the Hash Table and return true.

			CurSym = malloc2( sizeof( struct extRecs));
			CurSym->Next = extHashTable[ index ];
			extHashTable[ index ] = CurSym;
			CurSym->Name = hlastrdup2( theLabel );
			CurSym->Type = theType;
			CurSym->IsPublic = 0;		// Assume it's external.
			CurSym->theSym = sym;
			CurSym->ForceRef = ForceRef;
			
		_else

		
			CurSym->ForceRef |= ForceRef;

		_endif

	_endif

	// IsPublic is a "sticky" field.  Once it's public, it's
	// forever public.

	CurSym->IsPublic |= IsPublic;

	// Ditto for isVMT:
	
	CurSym->isVMT |= isVMT;

	
	// Set the IsReferenced field for whomever calls this function
	// (check for NULL because symbols HLA manually enters pass NULL
	// in for Sym).
	
	_if( sym != NULL )
	
		sym->IsReferenced = CurSym;
		
	_endif
	_return;

_end( extLookup );

void
EmitImmExtern
( 
	char *theSymbol, 
	char *labelType 
)
_begin( EmitImmExtern )

	struct SymNode *sym;

	extLookup( NULL, theSymbol, labelType, 0, 1, 0 );

_end( EmitImmExtern )



void
EmitExtern
( 
	struct SymNode *sym, 
	unsigned size 
)
_begin( EmitExtern )
	
	extLookup
	(
		sym, 
		sym->StaticName,
		_ifx
		(  
			size == 2, "word",
			_ifx
			( 
				size == 4, "dword",
				_ifx( size == 8, "qword", "byte" )
			)
		),
		0,  // External, not public.
		0,	// Don't force emission of extern directive.
		0	// This is not a VMT
	);

_end( EmitExtern )

void
EmitTypedExtern
( 
	struct SymNode *sym,
	char *theSymbol, 
	char *labelType 
)
_begin( EmitTypedExtern )

	extLookup( sym, theSymbol, labelType, 0, 0, 0 );

_end( EmitTypedExtern )



void
EmitVMTExtern
( 
	struct SymNode *sym,
	char *theSymbol 
)
_begin( EmitTypedExtern )

	extLookup( sym, theSymbol, "dword", 0, 0, 1 );

_end( EmitTypedExtern )



/*
** Note that a symbol is PUBLIC so we can emit the PUBLIC record
** to the appropriate include file later.
*/

void
EmitPublic( char *theLabel )
_begin( EmitPublic )

	extLookup( NULL, theLabel, "", 1, 0, 0 );

_end( EmitPublic )



/*
** EmitAdrs-
**
**	Emits a pointer to the specified label to the output stream.
*/

void
EmitAdrs( char *theLabel )
_begin( EmitPublic )

	char *o32 = offset32;

	_if( theLabel == NullPointer )
	
		o32 = "";
		
	_endif

	_if( assembler == gas )

		asmPrintf( "\t\t.long\t%s\n", theLabel );
		
	_elseif( assembler == fasm )

		asmPrintf( "\t\tdd\t%s\n", theLabel );

	_else

		asmPrintf( "\t\tdword\t%s%s\n", o32, theLabel );

	_endif

_end( EmitPublic )





void
PushStaticAdrs( char *theLabel )
_begin( PushStaticAdrs )

	char *o32 = offset32;

	_if( theLabel == NullPointer )
	
		o32 = "";
		
	_endif
	_if( assembler == masm )

		asmPrintf
		(
			"\t\tpush\t%s%s\n",
			o32,
			theLabel
		);

	_elseif( assembler == tasm )

		asmPrintf
		(
			"\t\tpush\t%s%s\n",
			o32,
			theLabel
		);


	_elseif( assembler == gas )

		asmPrintf
		(
			"\t\tpush\t%s %s\n",
			_ifx( o32 == offset32, "offset", ""),
			theLabel
		);

	_elseif( assembler == fasm )

		asmPrintf
		(
			"\t\tpush\t%s\n",
			theLabel
		);

	_else

		yyerror( "Internal compiler error (Bad assembler variable)" );

	_endif

_end( PushStaticAdrs )



/************************************************
/*                                             */
/* EmitStmtLbl-                                */
/* EmitStmtLblNum-                             */
/*                                             */
/* Outputs the specified statement label using */
/* the assembler's syntax.                     */

/*                                             */
/* Input                                       */
/* 	label-	String containing label to emit.   */
/* 	num-	(EmitStmtLblNum only) Numeric      */
/* 			value to attach to label.          */
/*                                             */
/***********************************************/

void
EmitStmtLbl( char *label )
_begin( EmitStmtLbl )

	asmPrintf
	(
		"%s:\n",
		label
	);


_end( EmitStmtLbl )


void
EmitGlobalStmtLbl( char *label )
_begin( EmitGlobalStmtLbl )

	asmPrintf
	(
		"%s:%s\n",
		label,
		_ifx( (assembler==masm || assembler==tasm), ";", "" )
	);

_end( EmitGlobalStmtLbl )


void
EmitStmtLblNum( char *label, int num )
_begin( EmitStmtLblNum )

	asmPrintf
	(
		label,
		num
	);

	asmPrintf
	(
		":\n"
	);

_end( EmitStmtLblNum )


/*
** NewLn- Writes a new line to the output file
*/

void
NewLn( void )
_begin( NewLn )

	asmPrintf( "\n" );

_end( NewLn )

/*******************************************
/*                                        */
/* EmitDwordConst-                        */
/*                                        */
/* Emits a numeric dword value along with */
/* an optional comment.                   */
/*                                        */
/******************************************/


void 
EmitDwordConst( unsigned theConst, char *comment )
_begin( EmitDwordConst )

	asmPrintf
	( 
		"\t\t%s\t0%s%x%s%s\n", 
		ifgas( ".long", _ifx( assembler == fasm, "dd", "dword" )),
		ifgas( "x", "" ), 
		theConst,
		ifgas( "", "h" ), 
		comment 
	);

_end( EmitDwordConst )


void 
EmitLwordConst( void *theConst )
_begin( EmitLwordConst )

	asmPrintf
	( 
		"\t\t%s\t", 
		ifgas( ".byte", _ifx( assembler == fasm, "db", "byte" ))
	);
	
	_for( int i=0, i<16, ++i )
	
		asmPrintf
		( 
			"0%s%x%s%s",
			ifgas( "x", "" ), 
			*((unsigned char*) theConst+i),
			ifgas( "", "h" ),
			_ifx( i < 15, ",", "" )
		);
		
	_endfor
	asmPrintf( "\n" );

_end( EmitLwordConst )


void 
EmitTbyteConst( void *theConst )
_begin( EmitTbyteConst )

	asmPrintf
	( 
		"\t\t%s\t", 
		ifgas( ".byte", _ifx( assembler == fasm, "db", "byte" ))
	);
	
	_for( int i=0, i<10, ++i )
	
		asmPrintf
		( 
			"0%s%x%s%s",
			ifgas( "x", "" ), 
			*((unsigned char*) theConst+i),
			ifgas( "", "h" ),
			_ifx( i < 9, ",", "" )
		);
		
	_endfor
	asmPrintf( "\n" );

_end( EmitTbyteConst )


void 
EmitQwordConst( void *theConst )
_begin( EmitQwordConst )

	asmPrintf
	( 
		"\t\t%s\t0%s%x%s,0%s%x%s\n", 
		ifgas( ".long", _ifx( assembler == fasm, "dd", "dword" )),
		ifgas( "x", "" ), 
		*((unsigned*) theConst),
		ifgas( "", "h" ),
		ifgas( "x", "" ), 
		*((unsigned*) theConst+1),
		ifgas( "", "h" )
	);

_end( EmitQwordConst )



void 
EmitWordConst( unsigned theConst )
_begin( EmitWordConst )

	asmPrintf
	( 
		"\t\t%s\t0%s%x%s\n", 
		ifgas( ".word", _ifx( assembler == fasm, "dw", "word" )),
		ifgas( "x", "" ), 
		theConst & 0xffff,
		ifgas( "", "h" )
	);

_end( EmitWordConst )


void 
EmitByteConst( unsigned theConst )
_begin( EmitByteConst )

	asmPrintf
	( 
		"\t\t%s\t0%s%x%s\n", 
		ifgas( ".byte", _ifx( assembler == fasm, "db", "byte" )),
		ifgas( "x", "" ), 
		theConst & 0xff,
		ifgas( "", "h" )
	);

_end( EmitByteConst )


void
EmitReal4Const( float theConst )
_begin( EmitReal4Const )

	asmPrintf
	(
		"\t\t%s\t%15.8e\n",
		ifgas( ".float", _ifx( assembler == fasm, "dd", "real4" )),
		theConst
	);

_end( EmitReal4Const )




void
EmitReal8Const( double theConst )
_begin( EmitReal8Const )


	asmPrintf
	(
		"\t\t%s\t%24.18e\n",
		ifgas( ".double", _ifx( assembler == fasm, "dq", "real8" ) ),
		theConst
	);

_end( EmitReal8Const )




void
EmitReal10Const( struct flt80 theConst )
_begin( EmitReal10Const )

	char realStr[32];

	e80Str( realStr, theConst );
	asmPrintf
	(
		"\t\t%s\t%s\n",
		ifgas( ".tfloat", _ifx( assembler == fasm, "dt", "real10" )),
		realStr
	);

_end( EmitReal10Const )




void
EmitLword( void )
_begin( EmitLword )

	_if( assembler == gas )

		asmPrintf( "\t\t.space\t16\n" );

	_elseif( assembler == fasm )

		asmPrintf( "\t\trq\t1\n" );
		asmPrintf( "\t\trq\t1\n" );

	_else

		asmPrintf( "\t\tdword\t?\n" );
		asmPrintf( "\t\tdword\t?\n" );
		asmPrintf( "\t\tdword\t?\n" );
		asmPrintf( "\t\tdword\t?\n" );

	_endif

_end( EmitLword )



void
EmitQword( void )
_begin( EmitQword )

	_if( assembler == gas )

		asmPrintf( "\t\t.space\t8\n" );

	_elseif( assembler == fasm )

		asmPrintf( "\t\trq\t1\n" );

	_else

		asmPrintf( "\t\tdword\t?\n" );
		asmPrintf( "\t\tdword\t?\n" );

	_endif

_end( EmitQword )



void
EmitTbyte( void )
_begin( EmitTbyte )

	_if( assembler == gas )

		asmPrintf( "\t\t.space\t10\n" );

	_elseif( assembler == fasm )

		asmPrintf( "\t\trt\t1\n" );

	_else

		asmPrintf( "\t\tdword\t?\n" );
		asmPrintf( "\t\tdword\t?\n" );
		asmPrintf( "\t\tword\t?\n" );

	_endif

_end( EmitTbyte )


void
EmitDword( void )
_begin( EmitDword )

	_if( assembler == gas )

		asmPrintf( "\t\t.space\t4\n" );
		
	_elseif( assembler == fasm )

		asmPrintf( "\t\trd\t1\n" );

	_else

		asmPrintf( "\t\tdword\t?\n" );

	_endif

_end( EmitDword )


void
EmitWord( void )
_begin( EmitWord )

	_if( assembler == gas )

		asmPrintf( "\t\t.space\t2\n" );

	_elseif( assembler == fasm )

		asmPrintf( "\t\trw\t1\n" );

	_else

		asmPrintf( "\t\tdword\t?\n" );

	_endif

_end( EmitWord )


void
EmitByte( void )
_begin( EmitByte )

	_if( assembler == gas )

		asmPrintf( "\t\t.space\t1\n" );
		
	_elseif( assembler == fasm )


		asmPrintf( "\t\trb\t1\n" );

	_else

		asmPrintf( "\t\tbyte\t?\n" );

	_endif

_end( EmitByte )



void
EmitArray
(
	char	*aType,
	int		elements
)
_begin( EmitArray )

	_if( assembler == gas )

		unsigned size;

		_if( _streq( aType, "word" ) )

			size = 2;

		_elseif( _streq( aType, "dword" ) || _streq( aType, "real4" ))
										 
			size = 4;

		_elseif( _streq( aType, "qword" ) || _streq( aType, "real8" ))

			size = 8;

		_elseif( _streq( aType, "real10" ))

			size = 10;

		_else

			size = 1;

		_endif

		asmPrintf
		( 
			"\t\t.space\t%d\n",
			size * elements
		);
	
	_elseif( assembler == fasm )

		_if( _streq( aType, "byte" ))
			
			asmPrintf
			( 
				"\t\trb\t%d\n",
				elements
			);

		_elseif( _streq( aType, "word" ))
			
			asmPrintf
			( 
				"\t\trw\t%d\n",
				elements
			);

		_elseif( _streq( aType, "dword" ) || _streq( aType, "real4" ))
			

			asmPrintf
			( 
				"\t\trd\t%d\n",
				elements
			);

		_elseif( _streq( aType, "qword" ) || _streq( aType, "real8" ))
			
			asmPrintf
			( 
				"\t\trq\t%d\n",
				elements
			);
			

		_elseif( _streq( aType, "real10" ))
			
			asmPrintf
			( 
				"\t\trt\t%d\n",
				elements
			);
			
		_else

			asmPrintf
			( 
				"\t\trb\t%d\n",
				elements
			);
		
		_endif

	_else

		asmPrintf
		( 
			"\t\t%s\t%d dup (?)\n",
			aType,
			elements
		);

	_endif

_end( EmitArray )



void
EmitArrayConst
(
	char		*aType,
	int			elements,
	unsigned	value
)
_begin( EmitArrayConst )

	_if( assembler == gas )

		char *gType;
		int  gSize = 1;

		_if( _streq( aType, "word" ) )

			gType = ".word";
			gSize = 2;

		_elseif( _streq( aType, "dword" ))
										 
			gType = ".long";
			gSize = 4;

		_else

			gType = ".byte";

		_endif

		_if( gSize == 1 || value < 256 )

			asmPrintf( "\t\t.space\t%d,%d\n", gSize*elements, value );

		_else

			asmPrintf( "\t\t.rept\t%d\n", elements );
			asmPrintf( "\n\t\t%s\t%d", gType, value );
			asmPrintf( "\t\t.endr\n" );

		_endif
		
	_elseif( assembler == fasm )
	
		char *fType;


		_if( _streq( aType, "word" ) )

			fType = "dw";

		_elseif( _streq( aType, "dword" ))
										 
			fType = "dd";

		_elseif( _streq( aType, "qword" ))
										 
			fType = "dq";

		_else

			fType = "db";

		_endif
		asmPrintf( "\t\trepeat\t%d\n", elements );
		asmPrintf( "\n\t\t%s\t%d", fType, value );
		asmPrintf( "\t\tend\trepeat\n" );		

	_else

		asmPrintf
		( 
			"\t\t%s\t%d dup (%d)\n",
			aType,
			elements,
			value
		);

	_endif

_end( EmitArrayConst )


void
EmitByteString
(
	char	*theString,
	int		zeroTerminate
)
_begin( EmitByteString )

	_if( strlen( theString ) != 0 || zeroTerminate )
	
		PrintString( theString, zeroTerminate );

	_endif

_end( EmitByteString )


void
EmitWordString
(
	char	*theString
)
_begin( EmitWordString )

	int	 i;
	char *s = theString;

	_for( i=0, i<strlen( theString ), ++i )
	
		EmitWordConst( *s++ );
		
	_endfor

_end( EmitWordString )




void
ReserveStorage( unsigned size )
_begin( ReserveStorage )

	_if( assembler == gas )
	
		asmPrintf( "\t\t.space\t%d\n", size );
		
	_elseif( assembler == fasm )
	
		asmPrintf( "\t\trb\t%d\n", size );

	_else
	
		asmPrintf( "\t\tbyte\t%d dup (?)\n", size );

	_endif
			
_end( ReserveStorage )


void
ReserveTypedStorage( struct SymNode *theType )
_begin( ReserveTypedStorage )

	_if( assembler == gas )
	
		asmPrintf( "\t\t.space\t%d\n", theType->ObjectSize );
		
	_elseif( assembler == fasm )
	
		asmPrintf( "\t\trb\t%d\n", theType->ObjectSize );

	_else
	
		asmPrintf( "\t\tbyte\t%d dup (?)\n", theType->ObjectSize );

	_endif
			
_end( ReserveTypedStorage )

/*********************************************
/*                                          */
/* EmitAlign-                               */
/*                                          */
/* Emits an appropriate alignment directive */
/*                                          */
/********************************************/

void
EmitAlign( unsigned alignment )
_begin( EmitAlign )

	int bitCnt = 0;
	int value = alignment;

	_while( value != 0 )

		bitCnt = bitCnt + (value & 1);
		value >>= 1;

	_endwhile
	_if( bitCnt != 1 )
	
		yyerror( "Alignment value must be a power of two" );
		
	_elseif( alignment > 16 && assembler == fasm )
	
		yyerror( "FASM output: align value must be 16 or less" );
		
	_endif

	_if( alignment > 1 )
	
		_if( assembler == gas )


			asmPrintf
			(
				"\t\t.balign\t%d\n",
				alignment
			);



		_else

			asmPrintf
			(
				"\t\talign\t%d\n",
				alignment 
			);

		_endif
		
	_endif
	
_end( EmitAlign )




void
EmitCodeAlign( unsigned alignment )
_begin( EmitCodeAlign )

	EmitAlign( alignment );
		
_end( EmitCodeAlign )



/************************************************************
/*                                                         */
/* strLookup-                                              */
/*                                                         */
/* This is a utility routine for the EmitString function.  */
/* If the user hasn't turned off string optimization, then */
/* HLA (by default) creates a single instance of each      */
/* unique string constant in the program.  The following   */
/* hash table and function keep track of all the string    */
/* constants in the program.  If a duplicate occurs, then  */
/* EmitString can emit an equate rather than a duplicate   */
/* string value.                                           */
/*                                                         */
/* Inputs-                                                 */
/* 	theString-	String to look for.                        */
/* 	theLabel-	Label value (e.g., "?%d_str", the value    */
/* 				for the %d part) to associate with this    */
/* 				string if it's not already present.        */
/* 				                                           */
/* Returns-                                                */
/* 	NULL		If this is the first occurrence of this    */
/* 				string constant in the hash table.  Also   */
/* 				enters the string into the table.          */
/* 				                                           */
/* 	pointer to                                             */
/* 	strRecs		If this is not the first occurrence of     */
/* 				this string in the hash table.  It is      */
/* 				the caller's responsibility to emit an     */
/* 				equate to the original string.             */
/* 				                                           */
/***********************************************************/
				                                         



 /* Hash table for the string lookup function: */

struct strRecs
{
	struct	strRecs	*Next;
			char	*str;
			int		Label;
};
static	struct	strRecs	*strHashTable[8192];



static struct strRecs*
strLookup( char *theString, int theLabel )
_begin( strLookup )

	unsigned				index;
	char					*CurChar;
	struct			strRecs	*CurStr; 

	/*
	** Cheesy hash table function.
	*/

	index = 0;
	CurChar = theString;
	_while( *CurChar != '\0' )

		/*
		** index = (index byteROL 1) ^ CurrentChar
		*/

		index = (index << 1);
		_if( index >= 8192 )
		
			index = index ^ 8193;

		_endif
		index ^= *CurChar;
		++CurChar;

	_endwhile

	// Just for good measure, XOR in the length of the string
	// too.

	index ^= (CurChar - theString ) & 8191;

	// Okay, this gives us an index into the hash table.
	// See if there's a corresponding entry in the hash table.

	_if( strHashTable[ index ] == NULL )

		// We're in business.  There is no hash entry so there
		// cannot be an instance of this symbol yet.

		strHashTable[ index ] = malloc2( sizeof( struct strRecs));
		strHashTable[ index ]->Next = NULL;
		strHashTable[ index ]->str = hlastrdup2( theString );
		strHashTable[ index ]->Label = theLabel;
		CurStr = NULL;

	_else

		// If there's an entry in the hash table slot, then
		// we've got to search through the list to see if we've
		// got an instance of this label.

		CurStr = strHashTable[ index ];
		_while
		(
				CurStr != NULL
			&& _strne( CurStr->str, theString )
		)

			CurStr = CurStr->Next;

		_endwhile
		_if( CurStr == NULL )

			// Didn't find the symbol.  So enter it into
			// the Hash Table and return true.

			CurStr = malloc2( sizeof( struct strRecs));
			CurStr->Next = strHashTable[ index ];
			strHashTable[ index ] = CurStr;
			CurStr->str = hlastrdup2( theString );
			CurStr->Label = theLabel;
			CurStr = NULL;  // Indicate that this is a new string.

		_endif

	_endif
	_return CurStr;

_end( strLookup );


/*********************************************************
/*                                                      */
/* EmitString-                                          */
/*                                                      */
/* Emits a string constant to the CONST segment.        */
/* "theStr" is the string to emit.                      */
/* "theLabel" is the LblCntr value to attach to this	*/ 
/*   string.											*/
/* Caller can refer to the string using the label       */
/* "?%d_str" (substituting lbl's value for %d).         */
/*                                                      */
/********************************************************/

void 
EmitString
( 
	char *theStr, 
	int theLabel 
)
_begin( EmitString )

	struct	strRecs *sr;
	int				length;
	char			lbl[64];

	length = strlen( theStr );
	startStrSeg();

	// Determine if this string has already appeared and
	// if we should just emit an equate for it.

	sr = NULL;	 
	_if( OptimizeStrings )

		sr = strLookup( theStr, theLabel );

	_endif

	_if( sr != NULL )

		// Okay, the string has already appeared, just emit an
		// equate to the original string value.

		EmitBackPatchddc
		(
			"L%d_str" sympost,
			theLabel,
			"L%d_str" sympost,
			sr->Label,
			""
		); 


	_else

		// Okay, this is the first time this string appears, or the
		// user has chosen not to optimize string constants.  Emit
		// a constant record for this string.

		asmPrintf( "\n" );
		EmitAlign( 4 );
		sprintf( lbl, "L%d_len" sympost, theLabel );
		EmitTypedLabel( lbl, "dword" );
		EmitDwordConst( length, "" );
		EmitDwordConst( length, "" );
		sprintf( lbl, "L%d_str" sympost, theLabel );
		EmitTypedLabel( lbl, "byte" );
		PrintString( theStr, 1 );

		/*
		** Be sure emitted string length is
		** a multiple of four characters long.
		*/

		++length;
		asmPrintf( "\n" );
		_while( (length & 3) != 0 )

			asmPrintf
			( 
				"\t\t%s\t0\n", 
				ifgas( ".byte", _ifx( assembler == fasm, "db", "byte" ))
			);
			++length;

		_endwhile

	_endif
	endStrSeg();

_end( EmitString )




void 
EmitWString
( 
	char *theStr, 
	int theLabel 
)
_begin( EmitWString )

	struct	strRecs *sr;
	unsigned short	*wStr;
	int				length;
	int				i;
	char			lbl[64];

	wStr = (unsigned short *) theStr;

	length = 0;
	while( wStr[ length ] != 0 )
	{
		++length;
	};
	startStrSeg();


	// No optimization for Unicode strings yet.
	// Need to add this someday!

	asmPrintf( "\n" );
	EmitAlign( 4 );
	sprintf( lbl, "L%d_len" sympost, theLabel );
	EmitTypedLabel( lbl, "dword" );
	EmitDwordConst( length, "" );
	EmitDwordConst( length, "" );
	sprintf( lbl, "L%d_str" sympost, theLabel );
	EmitTypedLabel( lbl, "word" );
	_for( i=0, i<length, ++i )
	
		EmitWordConst( wStr[i] );
	
	_endfor
	EmitWordConst( 0 ); // Emit zero terminating word

	/*
	** Be sure emitted string length is
	** a multiple of four characters long.
	*/

	_if( (i & 1) == 0 )
	
		EmitWordConst( 0 );
		
	_endif

	endStrSeg();

_end( EmitWString )



/************************************************************
/*                                                         */
/* EmitLabelledString-                                     */
/*                                                         */
/* This is used for special purposes in HLA (e.g., to      */
/* emit TRACE strings.  Don't bother trying to optimize    */
/* these strings (if called via the trace operation, there */
/* is no need to optimize for space since this is a debug  */
/* compile).                                               */
/*                                                         */
/* Inputs-                                                 */
/* 	theStr-		String to emit to the consts section.      */
/* 	theLabel-	name of this string object.                */
/*                                                         */
/***********************************************************/

void 
EmitLabelledString
( 
	char *theStr, 
	char *theLabel 
)
_begin( EmitLabelledString )

	struct	strRecs *sr;
	int				length;
	char			lbl[64];

	length = strlen( theStr );
	startStrSeg();


	// These strings are only used for special purposes
	// in HLA.  We aren't going to merge these strings
	// with others in the system.  Just emit the string
	// with the specified label.

	NewLn();
	EmitAlign( 4 );
	EmitDwordConst( length, "" );
	EmitDwordConst( length, "" );
	EmitTypedLabel( theLabel, "byte" );
	PrintString( theStr, 1 );

	/*
	** Be sure emitted string length is
	** a multiple of four characters long.
	*/

	++length;
	asmPrintf( "\n" );
	_while( (length & 3) != 0 )

		asmPrintf
		( 
			"\t\t%s\t0\n", 
			ifgas( ".byte", _ifx( assembler == fasm, "db", "byte" ))
		);
		++length;

	_endwhile
	NewLn();
	endStrSeg();

_end( EmitLabelledString )







/*************************************************************
/*                                                          */
/* EmitBackPatchXXXX functions-                             */
/*                                                          */
/* This functions emit EQU directives to handling           */
/* "backpatching" for HLA.                                  */
/*                                                          */
/* EmitBackPatchss-    Sets first symbol equal to second.   */
/*                                                          */
/* EmitBackPatchds -   1st symbol has a %d field,           */
/*                     the 2nd symbol is a simple string.   */
/*                                                          */
/* EmitBackPatchdsc-   Like the routine above, except       */
/*                     this one has a comment parameter to  */
/*                     append to the string.                */
/*                                                          */
/* EmitBackPatchddc-	Like the EmitBackPatchdsc routine,  */
/*                     but both labels have %d fields.      */
/*                                                          */
/************************************************************/

void
EmitBackPatchss
(
	char	*sym,
	char	*equals
)
_begin( EmitBackPatchds )

	char	bp[256];
	struct	bpList_t *thisBP;


	_if( assembler == gas )

		sprintf
		(
			bp,
			"\t\t.set\t%s, %s\n",
			sym,
			equals
		);

	_else

		sprintf
		(
			bp,
			"%-15s equ\t%s",
			sym,
			equals
		);

	_endif

	thisBP = malloc2( sizeof( struct bpList_t ));
	thisBP->Next = bpList;
	thisBP->bpatch = hlastrdup2( bp );
	bpList = thisBP;

_end( EmitBackPatchds )


void
EmitBackPatchds
(
	char	*sym,
	int		symNum,
	char	*equals
)
_begin( EmitBackPatchds )


	char	sn[256];
	char	bp[256];
	struct	bpList_t *thisBP;

	sprintf( sn, sym, symNum );
	_if( assembler == gas )

		sprintf
		(
			bp,
			"\t\t.set\t%s, %s\n",
			sn,
			equals
		);

	_else

		sprintf
		(
			bp,
			"%-15s equ\t%s",
			sn,
			equals
		);

	_endif
	thisBP = malloc2( sizeof( struct bpList_t ));
	thisBP->Next = bpList;
	thisBP->bpatch = hlastrdup2( bp );
	bpList = thisBP;

_end( EmitBackPatchds )




void
EmitBackPatchdsc
( 
	char	*sym, 
	int		symNum,
	char	*equals,
	char	*comment  // Ignoring this for now.
)
_begin( EmitBackPatchdsc )

	char	sn[256];
	char	bp[256];
	struct	bpList_t *thisBP;

	sprintf( sn, sym, symNum );
	_if( assembler == gas )

		sprintf
		(
			bp,
			"\t\t.set\t%s, %s\n",
			sn,
			equals
		);

	_else

		sprintf
		(
			bp,
			"%-15s equ\t%s\t;%s",
			sn,
			equals,
			comment
		);

	_endif
	thisBP = malloc2( sizeof( struct bpList_t ));
	thisBP->Next = bpList;
	thisBP->bpatch = hlastrdup2( bp );
	bpList = thisBP;

_end( EmitBackPatchdsc )






void
EmitBackPatchddc
( 
	char	*sym, 
	int		symNum,
	char	*equals,
	int		eqNum,
	char	*comment  // Ignoring this for now. 
)
_begin( EmitBackPatchddc )

	char	sn[256];
	char	eq[256];
	char	bp[256];
	struct	bpList_t *thisBP;

	sprintf( sn, sym, symNum );
	sprintf( eq, equals, eqNum );
	_if( assembler == gas )

		sprintf
		(
			bp,
			"\t\t.set\t%s, %s\n",
			sn,
			eq
		);

	_else

		sprintf
		(
			bp,
			"%-15s equ\t%s\t;%s",
			sn,
			eq,
			comment
		);

	_endif
	thisBP = malloc2( sizeof( struct bpList_t ));
	thisBP->Next = bpList;
	thisBP->bpatch = hlastrdup2( bp );
	bpList = thisBP;

_end( EmitBackPatchddc )


void
IterateBP( void )
_begin( IterateBP )

	/*
	** This code was moved to ExtPubIterator
	*/
	
_end( IterateBP )
	



/*************************************************************
/*                                                          */
/* ComparableNodes-                                         */
/*                                                          */
/* Checks nodes in an expression Abstract Syntax Tree       */
/* to see if the two operands on either side are comparable */
/* via a CMP instruction.                                   */
/*                                                          */
/************************************************************/


void
ComparableNodes
( 
	struct operandYYS *left, 
	struct operandYYS *right, 
	char *operator 
)
_begin( ComparableNodes )

	struct	operandYYS	*l = (struct operandYYS *) left;
	struct	operandYYS	*r = (struct operandYYS *) right;
	char				msg[ 256 ];


	_if( l->operandType == reg_optype )

		_if( r->operandType == reg_optype )

			_if( l->o.reg->Size != r->o.reg->Size )

				sprintf
				( 
					msg,
					"Registers must be the same size around %s operator",
					operator 
				);
				yyerror( msg );

			_endif

		_elseif( r->operandType == mem_optype )

			_if
			( 
					(
							!IsOrdinal( r->o.adrs->pType ) 
						&&	!IsStr( r->o.adrs->pType )
						&&	r->o.adrs->pType != tPointer 
					)
				||	l->o.reg->Size != r->o.adrs->Size
			)

				sprintf
				(
					msg,
					"Type mismatch error around %s operator",
					operator
				);
				yyerror( msg );

			_endif

		_elseif( r->operandType == const_optype )

			_if
			( 
					!IsOrdinal( r->o.v.pType )
				&&	r->o.v.pType != tPointer
			)
			
				sprintf
				(
					msg,
					"Expected an ordinal constant near %s operator",
					operator
				);
				yyerror( msg );

			_elseif( r->o.v.pType == tPointer )

				/* It's okay */

			_elseif( l->o.reg->IsSigned )

				_if
				(
						(
								IsInt( r->o.v.pType )
							&&	(  
										(
												l->o.reg->Size == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													-128, 
													127 
												)
										)
									||	( 
												l->o.reg->Size == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													-32768, 
													32767 
												)
										)
								)
						)

					||	(
								IsUns( r->o.v.pType )
							&&	(  
										(
												l->o.reg->Size == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													127 
												)
										)
									||	( 
												l->o.reg->Size == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													32767 
												)
										)
									||	( r->o.v.u.unsval & 0x80000000 )
								)
						)

				)

					yyerror( "Constant is out of range for register size" );

				_endif

			_else // Register can be signed or unsigned.

				_if
				(
						(
								IsInt( r->o.v.pType )
							&&	(  
										(
												l->o.reg->Size == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													-127, 
													255 
												)
										)
									||	( 
												l->o.reg->Size == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													-32768, 
													65535 
												)
										)
								)
						)

					||	(
								IsUns( r->o.v.pType )
							&&	(  
										(
												l->o.reg->Size == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													255 
												)
										)
									||	( 
												l->o.reg->Size == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													65535 
												)
										)
								)
						)

				)

					yyerror( "Constant is out of range for register size" );

				_endif

			_endif

				
		_else

			yyerror( "Unknown right operand type (Internal HLA Error)" );

		_endif

	
	_elseif( l->operandType == mem_optype )
	
		_if( r->operandType == reg_optype )

			_if
			( 
					(
							!IsOrdinal( l->o.adrs->pType ) 
						&&	!IsStr( l->o.adrs->pType ) 
					)
				||	r->o.reg->Size != l->o.adrs->Size
			)

				sprintf
				(
					msg,
					"Type mismatch error around %s operator",
					operator
				);
				yyerror( msg );

			_endif

		_elseif( r->operandType == mem_optype )

			// Should never happen, but just in case:

			yyerror
			( 
				"Memory to memory operations are illegal (internal HLA error)"
			);

		_elseif( r->operandType == const_optype )

			_if( l->o.adrs->Size > 4 ) 
			
				yyerror( "Memory operand must be 32 bits or less" );
				
			_elseif
			( 
					!( IsOrdinal( r->o.v.pType ) && r->o.v.ObjectSize <= 4 )
				&&	r->o.v.pType != tPointer
			)
			
				sprintf
				(
					msg,
					"Expected a 32-bit ordinal constant near %s operator",
					operator
				);
				yyerror( msg );

			_elseif( r->o.v.pType == tPointer && l->o.adrs->Size != 4 )

				yyerror( "Size mismatch in operands" );

			_elseif(  r->o.v.pType == tPointer )

				/* then it's okay */

			_elseif( IsInt( l->o.adrs->pType ) )

				_if
				(
						(
								IsInt( r->o.v.pType )
							&&	(  
										(
												l->o.adrs->ObjectSize == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													-128, 
													127 
												)
										)
									||	( 
												l->o.adrs->ObjectSize == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													-32768, 
													32767 
												)
										)
								)
						)

					||	(
								IsUns( r->o.v.pType )
							&&	(  
										(
												l->o.adrs->ObjectSize == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													127 
												)
										)
									||	( 
												l->o.adrs->ObjectSize == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													32767 
												)
										)
									||	( r->o.v.u.unsval & 0x80000000 )
								)
						)

				)

					yyerror( "Constant is out of range for data size" );

				_endif

			_else // Variable must be unsigned.

				_if
				(
						(
								IsInt( r->o.v.pType )
							&&	(  
										(
												l->o.adrs->ObjectSize == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													127 
												)
										)
									||	( 
												l->o.adrs->ObjectSize == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													32767 
												)
										)
								)
						)

					||	(
								IsUns( r->o.v.pType )
							&&	(  
										(
												l->o.adrs->ObjectSize == 1 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													255 
												)
										)
									||	( 
												l->o.adrs->ObjectSize == 2 
											&&	!IntRange
												( 
													r->o.v.u.intval, 
													0, 
													65535 
												)
										)
								)
						)

				)

					yyerror( "Constant is out of range for register size" );

				_endif

			_endif

		_else

			yyerror( "Unknown right operand type (Internal HLA Error)" );

		_endif
	
	_else
	
		yyerror( "Left operand must be a register or memory location" );

	_endif 

_end( ComparableNodes )


static char*
RtnOperand( struct operandYYS *oprnd, char **dest, unsigned Size )
_begin( RtnOperand )

	char op[256];			// Disgusting and non-reentrant
							// but okay for this program.

	static unsigned SizeMask[] =
	{
		0,
		0xff,
		0xffff,
		0xffffff, // not used, but what the heck.
		0xffffffff
	};

	assert( oprnd != NULL );
	assert( dest != NULL );
	_if( oprnd->operandType == reg_optype )

		assert( oprnd->regname != NULL );
		strcpy( op, oprnd->regname );

	_elseif( oprnd->operandType == mem_optype )

		StoreAdrs( op, YYS oprnd->o.adrs );

	_elseif( oprnd->operandType == const_optype );

		_if( IsOrdinal( oprnd->o.v.pType ) && numBits( YYS &oprnd->o.v ) <= 32 )

			sprintf
			( 
				op, 
				"0%s%x%s", 
				ifgas( "x", "" ),
				( oprnd->o.v.u.unsval & SizeMask[ Size ] ),
				ifgas( "", "h" )
			);

		_elseif( oprnd->o.v.pType == tPointer )

			sprintf
			( 
				op, 
				"%s%s", 
				_ifx( oprnd->o.v.u.strval == NullPointer, "", offset32), 
				oprnd->o.v.u.strval 
			);

		_else

			yyerror( "Unexpected operand type (Internal HLA error)" );
			op[0] = '0';
			op[1] = 0;

		_endif

	_endif

	_return *dest = hlastrdup2( op );

_end( RtnOperand )



/******************************************************/
/*                                                    */
/* EmitConstValue-  Converts an ordinal value to it's */
/* native string format.                              */
/*                                                    */
/******************************************************/

char *
EmitConstValue
( 
	char *comment, 
	struct SymNode *v
)
_begin( EmitConstValue )

	_switch( v->pType )

		_case( tBoolean )
		
			_if( v->u.unsval )
			
				strcpy( comment, "true" );
				
			_else
			
				strcpy( comment, "false" );
				
			_endif
			
		_endcase
		
		_case( tEnum )
		_case( tUns8 )
		_case( tUns16 )
		_case( tUns32 )


			sprintf( comment, "%u", v->u.unsval );

		_endcase
		
		_case( tUns64 )
		_case( tUns128 )
			
			UnsToStr( comment, &v->u.lwordval[0] );
			
		_endcase
			
		

		_case( tByte )
		_case( tWord )
		_case( tDWord )

			sprintf( comment, "$%x", v->u.unsval );

		_endcase
		

		_case( tQWord )
		_case( tLWord )
		
			_if( v->u.lwordval[3] == 0 )

			
		
				_if( v->u.lwordval[2] == 0 )
				
					_if( v->u.lwordval[1] == 0 )
					
						sprintf( comment, "$%x", v->u.unsval );
						
					_else
					
						sprintf
						( 
							comment, 
							"$%x_%08x", 
							v->u.lwordval[1],
							v->u.lwordval[0] 
						);
						
					_endif
					
				_else
					
					sprintf
					( 
						comment, 
						"$%x_%08x_%08x", 
						v->u.lwordval[2],
						v->u.lwordval[1],
						v->u.lwordval[0] 
					);
						
				_endif
									
			_else
			
				sprintf
				( 
					comment, 
					"$%x_%08x_%08x_%08x", 
					v->u.lwordval[3],
					v->u.lwordval[2],
					v->u.lwordval[1],
					v->u.lwordval[0] 
				);
					
			_endif

		_case( tInt8 )
		_case( tInt16 )
		_case( tInt32 )

			sprintf( comment, "%d", v->u.intval );

		_endcase
		
		_case( tInt64 )
		_case( tInt128 )
		
			IntToStr( comment, &v->u.lwordval[0] );
			
		_endcase

		_case( tChar )
		_case( tWChar )

			_if( v->u.charval < ' ' || v->u.charval >= 0x7f )

				sprintf( comment, "#$%x", v->u.charval );

			_else

				sprintf( comment, "'%c'", v->u.charval );

			_endif

		_endcase


		_case( tCset )

			strcpy( comment, "cset" );

		_endcase

		_case( tPointer )

			strcpy( comment, "pointer constant" );

		_endcase

		_case( tProcptr )

			strcpy( comment, "procptr constant" );

		_endcase

	 		

		_case( tReal32 )
		_case( tReal64 )
		_case( tReal80 )

			yyerror( "Illegal ordinal type: real (Internal HLA Error)" );

		_endcase
	
 
		_case( tString )

			yyerror( "Illegal ordinal type: string (Internal HLA Error)" );

		_endcase
		
			
		_case( tZString )

			yyerror( "Illegal ordinal type: zstring (Internal HLA Error)" );

		_endcase
		
			
		_case( tTByte )

			yyerror
			( 
				"Illegal ordinal type: tbyte (Internal HLA Error)" 
			);

		_endcase
				

		_case( tArray )
		_case( tRecord )
		_case( tUnion )
		_case( tClass )
		_case( tThunk )
		_case( tLabel )
		_case( tProc )
		_case( tMethod )
		_case( tClassProc )
		_case( tClassIter )
		_case( tIterator )
		_case( tProgram )
		_case( tMacro )
		_case( tText )
		_case( tNamespace )
		_case( tSegment )
		_case( tVariant )
		_case( tError )

			yyerror
			( 
				"Illegal structured type (Internal HLA Error)" 
			);

		_endcase

		_default

			yyerror
			(
				"Unknown primitive data type in HLA (internal error)"
			);

	_endswitch
	return comment;

_end( EmitConstValue )


///**************************************************************/
///*                                                            */
///* RtnComment-  Returns a pointer to a memory field's comment */
///* (i.e., the HLA variable name).  For a constant, returns    */
///* a string containing the constant's native form.            */
///* Returns the empty string if this is a register object.     */
///*                                                            */
///**************************************************************/
//
//
//char *
//RtnComment( struct opnodeYYS *o )
//_begin( RtnComment )
//
//	// Okay, the use of a static variable for the return value
//	// is disgusting a fraught with error.  However, there's only
//	// one comment per output line and this code doesn't need to
//	// be reentrant, so we'll get away with this.
//
//	static char comment[ 256 ];
//
//	assert( o != NULL );
//	assert( o->l.leftOperand != NULL );
//
//	comment[ 0 ] = ';';
//	comment[ 1 ] = '/';
//	comment[ 2 ] = '*';
//	comment[ 3 ] = ' ';
//	comment[ 4 ] = '\0';
//
//	_if( o->l.leftOperand->operandType == mem_optype )
//
//		int len;

//
//		strcpy( &comment[ 4 ], o->l.leftOperand->o.adrs->Comment );
//		_if
//		( 
//				o->r.rightOperand != NULL 
//			&&	o->r.rightOperand->operandType == const_optype 
//		)
//
//			// We've got a memory operand and a constant, create
//			// a comment containing both.
//
//			len = strlen( comment );
//			comment[ len ] = ',';
//
//			EmitConstValue
//

//			( 
//				&comment[ len + 1 ], 
//				&o->r.rightOperand->o.v 
//			);
//
//
//		_endif
//
//
//	_elseif
//	( 
//			o->r.rightOperand != NULL 
//		&&	o->l.leftOperand->operandType == mem_optype 
//	) 
//	
//		strcpy( &comment[ 4 ], o->r.rightOperand->o.adrs->Comment );
//
//	_elseif( o->r.rightOperand->operandType == const_optype )
//
//		EmitConstValue
//		( 
//			&comment[ 4 ], 
//			&o->r.rightOperand->o.v 
//		);
//
//	_else // must be a register.  Don't return any comment.
//
//		comment[ 0 ] = '\0';
//
//	_endif
//	
//	_if( *comment != '\0' )
//	
//		strcat( comment, " */" );
//		
//	_endif
//
//	_return comment;
//
//_end( RtnComment )


/************************************************************/
/*                                                          */
/* FreeOperand-                                             */
/*                                                          */
/* Frees all the storage associated with an operand object. */
/* Recursively processes all operands/operators/etc/ in     */
/* the abstract syntax tree for this expression.            */
/*                                                          */
/************************************************************/

void
FreeOperand( struct opnodeYYS *o )
_begin( FreeOperand )

	assert( o != NULL );
	assert( o->l.leftOperand != NULL );

	_if
	( 
			o->operator != and_astop
		&&	o->operator != or_astop
		&&	o->operator != not_astop 
	)

	
		_if( o->l.leftOperand->operandType == reg_optype )

			assert( o->l.leftOperand->o.reg != NULL );
			free2( vss o->l.leftOperand->regname );
			free2( vss o->l.leftOperand->o.reg );

		_elseif( o->l.leftOperand->operandType == mem_optype )

			assert( o->l.leftOperand->o.adrs != NULL );
			free2( vss o->l.leftOperand->o.adrs->StaticName );
			o->l.leftOperand->o.adrs->StaticName = NULL;
			free2( vss o->l.leftOperand->o.adrs->BaseReg );
			o->l.leftOperand->o.adrs->BaseReg = NULL;
			free2( vss o->l.leftOperand->o.adrs->IndexReg );
			o->l.leftOperand->o.adrs->IndexReg = NULL;

		_elseif( o->l.leftOperand->operandType == const_optype )

			FreeValue( YYS &o->l.leftOperand->o.v );

		_elseif( o->l.leftOperand->operandType == flag_optype )

			/* Don't have to do anything */

		_else

			yyerror( "Unexpected operand type (internal HLA error)" );

		_endif
		free2( vss o->l.leftOperand->text );
		free2( vss o->l.leftOperand );

		_if( o->r.rightOperand != NULL )

			_if( o->r.rightOperand->operandType == reg_optype )

				assert( o->r.rightOperand->o.reg != NULL );
				free2( vss o->r.rightOperand->regname );
				free2( vss o->r.rightOperand->o.reg );

			_elseif( o->r.rightOperand->operandType == mem_optype )

				assert( o->r.rightOperand->o.adrs != NULL );
				free2( vss o->r.rightOperand->o.adrs->StaticName );
				o->r.rightOperand->o.adrs->StaticName = NULL;
				free2( vss o->r.rightOperand->o.adrs->BaseReg );
				o->r.rightOperand->o.adrs->BaseReg = NULL;
				free2( vss o->r.rightOperand->o.adrs->IndexReg );
				o->r.rightOperand->o.adrs->IndexReg = NULL;

			_elseif( o->r.rightOperand->operandType == const_optype )

				FreeValue( YYS &o->r.rightOperand->o.v );

			_else

				yyerror( "Unexpected operand type (internal HLA error)" );

			_endif
			free2( vss o->r.rightOperand->text );
			free2( vss o->r.rightOperand );

		_endif

	_endif
	free2( vss o );

_end( FreeOperand )







/**********************************************************/
/*                                                        */
/* EmitConditionalJump-                                   */
/*                                                        */
/* This code emits the Jcc instructions corresponding to  */
/* the use of a CPU flag (e.g., @c, @nc) as a boolean     */
/* expression.                                            */
/*                                                        */
/* The "t" and "f" strings are the "cc" portion to emit   */
/* if tfjmp is true or false.                             */
/*                                                        */
/**********************************************************/


void
EmitConditionalJmp
( 
	int		tfjmp, 
	char	*t, 
	char	*f, 
	char	*target,
	char	*text 
)
_begin( EmitConditionalJmp )

	asmPrintf
	(
		"%s"
		"\t\tj%s\t%s\n",
		_ifx( text == NULL, "", text ),
		_ifx( tfjmp, t, f),
		target
	);

_end( EmitConditionalJmp )




void
InvertTree( struct opnodeYYS *o )
_begin( InvertTree )

	_switch( o->operator )

		_case( and_astop )

			o->operator = or_astop;
			InvertTree( o->l.leftSubexpression );
			InvertTree( o->r.rightSubexpression );

		_endcase


		_case( or_astop )


			o->operator = and_astop;
			InvertTree( o->l.leftSubexpression );
			InvertTree( o->r.rightSubexpression );


		_endcase

		

		_case( not_astop )

			// If we encounter a second not operation, go ahead
			// and invert everything underneath, we will uninvert it
			// later.

			InvertTree( o->l.leftSubexpression );

		_endcase

		
		_case(  eq_astop ) 		o->operator = ne_astop;	_endcase
		_case(  ne_astop ) 		o->operator = eq_astop;		_endcase
		_case(  b_astop ) 		o->operator = ae_astop;		_endcase
		_case(  be_astop ) 		o->operator = a_astop;		_endcase
		_case(  a_astop ) 		o->operator = be_astop; 	_endcase
		_case(  ae_astop ) 		o->operator = b_astop;		_endcase
		_case(  l_astop ) 		o->operator = ge_astop;		_endcase
		_case(  le_astop ) 		o->operator = g_astop;		_endcase
		_case(  g_astop ) 		o->operator = le_astop;		_endcase
		_case(  ge_astop ) 		o->operator = l_astop;		_endcase
		_case(  eq0_astop )		o->operator = ne0_astop;	_endcase
		_case( ne0_astop ) 		o->operator = eq0_astop;	_endcase
		_case(  in_astop ) 		o->operator = notin_astop;	_endcase
		_case(  notin_astop )	o->operator = in_astop;		_endcase
		_case( c_flag ) 		o->operator = nc_flag;		_endcase
		_case( nc_flag ) 		o->operator = c_flag;		_endcase
		_case( s_flag ) 		o->operator = ns_flag;		_endcase
		_case( ns_flag ) 		o->operator = s_flag;		_endcase
		_case( o_flag ) 		o->operator = no_flag;		_endcase
		_case( no_flag ) 		o->operator = o_flag;		_endcase
		_case( z_flag ) 		o->operator = nz_flag;		_endcase
		_case( nz_flag ) 		o->operator = z_flag;		_endcase
		_case( a_flag ) 		o->operator = na_flag;		_endcase
		_case( na_flag ) 		o->operator = a_flag;		_endcase
		_case( ae_flag ) 		o->operator = nae_flag;		_endcase
		_case( nae_flag ) 		o->operator = ae_flag;		_endcase
		_case( b_flag ) 		o->operator = nb_flag;		_endcase
		_case( nb_flag ) 		o->operator = b_flag;		_endcase
		_case( be_flag ) 		o->operator = nbe_flag;		_endcase
		_case( nbe_flag ) 		o->operator = be_flag;		_endcase
		_case( l_flag ) 		o->operator = nl_flag;		_endcase
		_case( nl_flag ) 		o->operator = l_flag;		_endcase
		_case( le_flag ) 		o->operator = nle_flag;		_endcase
		_case( nle_flag ) 		o->operator = le_flag;		_endcase
		_case( g_flag ) 		o->operator = ng_flag;		_endcase
		_case( ng_flag ) 		o->operator = g_flag;		_endcase
		_case( ge_flag ) 		o->operator = nge_flag;		_endcase
		_case( nge_flag ) 		o->operator = ge_flag;		_endcase
		_case( e_flag ) 		o->operator = ne_flag;		_endcase
		_case( ne_flag ) 		o->operator = e_flag;		_endcase
		_case( pe_flag ) 		o->operator = po_flag;		_endcase
		_case( po_flag ) 		o->operator = pe_flag;		_endcase
		_case( p_flag ) 		o->operator = np_flag;		_endcase
		_case( np_flag ) 		o->operator = p_flag;		_endcase

		_default

			yyerror( "Illegal operand type (internal HLA error)" );


	_endswitch

_end( InvertTree )



/*************************************************************************/
/*                                                                       */
/* EmitBooleanExpr-                                                      */
/*                                                                       */
/* 	"o" points at an abstract syntax tree.                               */
/* 	label is the label suffix for the ?xxxx_true and ?xxxx_false labels. */
/* 	tfjmp determines the sense of the jmp (i.e., should we swap the      */
/* 	true/false labels).                                                  */
/*                                                                       */
/* 	This routine emits the code to evaluate the boolean expression       */
/* 	parsed and placed in the abstract syntax tree pointed at by "o".     */
/*                                                                       */
/* 	It is this routine's responsibility to free the memory in use by     */
/* 	the abstract syntax tree.                                            */
/*                                                                       */
/*************************************************************************/

#define leftSize									\
(													\
		o->l.leftOperand->operandType == reg_optype	\
	? 	o->l.leftOperand->o.reg->Size 				\
	:	o->l.leftOperand->o.adrs->Size 				\
)


#define leftOpText							\
	_ifx									\
	(										\
		o->l.leftOperand->text == NULL, 	\
		"", 								\
		o->l.leftOperand->text				\
	)


#define rightOpText							\
	_ifx									\
	(										\
		o->r.rightOperand->text == NULL, 	\
		"", 								\
		o->r.rightOperand->text				\
	)


void 
EmitBooleanExpr
( 
	struct opnodeYYS *o, 
	char *target,
	int tfjmp 
)
_begin( EmitBooleanExpr )
   
   char *left = NULL;
   char *right = NULL;
   int	newLbl;
   char	newLblc[32];

	assert( o != NULL );
	assert( target != NULL );

	// Special case (i.e., a kludge).  This tree parser
	// really wants to handle binary operators.  So handle
	// single operands as a special case.

	_switch( o->operator )

		_case( and_astop )

			// We've got two subexpressions with an "AND" (conjunction)
			// operation, recursively call this procedure to emit
			// the appropriate code.

			assert( o->l.leftSubexpression != NULL );
			assert( o->r.rightSubexpression != NULL );


			newLbl = LblCntr++;
			_if( tfjmp ) // Branch on true condition.

				sprintf( newLblc, "L%d" sympost, newLbl );
				EmitBooleanExpr
				( 
					o->l.leftSubexpression, 
					newLblc, 
					0				
				);

				EmitBooleanExpr
				( 
					o->r.rightSubexpression, 
					target, 
					1
				);
				asmPrintf
				(
					"%s:\n",
					newLblc
				);

			_else  // Fall through on true condition.

				EmitBooleanExpr
				( 
					o->l.leftSubexpression, 
					target, 
					0			// Branch on false to target				
				);

				EmitBooleanExpr
				( 
					o->r.rightSubexpression, 
					target, 
					0			// Branch on false to target.
				);


			_endif


		_endcase


		_case( or_astop )

			// We've got two subexpressions with an "OR" (disjunction)
			// operation, recursively call this procedure to emit
			// the appropriate code.

			assert( o->l.leftSubexpression != NULL );
			assert( o->r.rightSubexpression != NULL );

			newLbl = LblCntr++;
			_if( tfjmp ) // Branch on true condition.

				EmitBooleanExpr
				( 
					o->l.leftSubexpression, 
					target, 
					1				
				);

				EmitBooleanExpr
				( 
					o->r.rightSubexpression, 
					target, 
					1
				);

			_else  // Fall through on true condition.

				sprintf( newLblc, "L%d" sympost, newLbl );
				EmitBooleanExpr
				( 
					o->l.leftSubexpression, 
					newLblc, 
					1			// Branch on true to newLbl				
				);

				EmitBooleanExpr
				( 
					o->r.rightSubexpression, 
					target, 
					0			// Branch on false to target.
				);

				asmPrintf
				(
					"%s:\n",
					newLblc
				);

			_endif

		_endcase

		

		_case( not_astop )

			// We've got one subexpression with a "!" (NOT)
			// operation.  This is ugly, we've got to recursively
			// descend into the tree and flip all the comparisons,
			// ands/ors, etc.

			assert( o->l.leftSubexpression != NULL );
			assert( o->r.rightSubexpression == NULL );

			InvertTree( o->l.leftSubexpression );
			EmitBooleanExpr
			( 
				o->l.leftSubexpression, 
				target, 
				tfjmp 
			);

		_endcase

		
		_case(  eq_astop )

			// Emit code for "op = op" here

			_if
			( 
					o->l.leftOperand->operandType == reg_optype
				&&	o->r.rightOperand->operandType == const_optype
				&&	o->r.rightOperand->o.v.u.unsval == 0
			)

				// Optimize comparison against zero to a TEST instr.

				asmPrintf
				(
					"%s"
					"\t\ttest\t%s,%s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->l.leftOperand, &right, 0 ),
					_ifx( tfjmp, "e", "ne"),
					target
				);

			_else




				asmPrintf
				(
					"%s"
					"%s"
					"\t\tcmp\t%s, %s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					rightOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->r.rightOperand, &right, leftSize ),
					_ifx( tfjmp, "e", "ne"),
					target
				);

			_endif

		_endcase



		_case(  ne_astop )

			// Emit code for "op <> op" here

			_if
			( 
					o->l.leftOperand->operandType == reg_optype
				&&	o->r.rightOperand->operandType == const_optype
				&&	o->r.rightOperand->o.v.u.unsval == 0
			)

				// Optimize comparison against zero to a TEST instr.

				asmPrintf
				(
					"%s"
					"\t\ttest\t%s,%s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->l.leftOperand, &right, 0 ),
					_ifx( tfjmp, "ne", "e"),
					target
				);

			_else

				asmPrintf
				(
					"%s"
					"%s"
					"\t\tcmp\t%s, %s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					rightOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->r.rightOperand, &right, leftSize ),
					_ifx( tfjmp, "ne", "e"),
					target
				);

			_endif

		_endcase



		_case(  b_astop )

			// Emit code for "op < op" (unsigned) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "b", "nb"),
				target
			);

		_endcase



		_case(  be_astop )

			// Emit code for "op <= op" (unsigned) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "be", "nbe"),
				target
			);

		_endcase



		_case(  a_astop )

			// Emit code for "op > op" (unsigned) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "a", "na"),
				target
			);

		_endcase



		_case(  ae_astop )

			// Emit code for "op >= op" (unsigned) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "ae", "nae"),
				target
			);

		_endcase



		_case(  l_astop )

			// Emit code for "op < op" (signed) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "l", "nl"),
				target
			);

		_endcase



		_case(  le_astop )

			// Emit code for "op <= op" (signed) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "le", "nle"),
				target
			);

		_endcase



		_case(  g_astop )

			// Emit code for "op > op" (signed) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "g", "ng"),
				target
			);

		_endcase



		_case(  ge_astop )

			// Emit code for "op >= op" (signed) here

			asmPrintf
			(
				"%s"
				"%s"
				"\t\tcmp\t%s, %s\n"
				"\t\tj%s\t%s\n",
				leftOpText,
				rightOpText,
				RtnOperand( o->l.leftOperand, &left, 0 ),
				RtnOperand( o->r.rightOperand, &right, leftSize ),
				_ifx( tfjmp, "ge", "nge"),
				target
			);

		_endcase



		_case(  eq0_astop )

			_if( o->l.leftOperand->operandType == reg_optype )

				asmPrintf
				(
					"%s"
					"\t\ttest\t%s,%s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->l.leftOperand, &right , leftSize),
					_ifx( tfjmp, "e", "ne"),
					target
				);

			_elseif( o->l.leftOperand->operandType == mem_optype )

				asmPrintf
				(
					"%s"
					"\t\tcmp\t%s, 0\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					_ifx( tfjmp, "e", "ne"),
					target
				);

			_else

				yyerror( "Unexpected operand type (internal HLA error)" );

			_endif

		_endcase



		_case( ne0_astop )

			_if( o->l.leftOperand->operandType == reg_optype )

				asmPrintf
				(
					"%s"
					"\t\ttest\t%s,%s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->l.leftOperand, &right, leftSize ),
					_ifx( tfjmp, "ne", "e"),
					target
				);

			_elseif( o->l.leftOperand->operandType == mem_optype )

				asmPrintf
				(
					"%s"
					"\t\tcmp\t%s, 0\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					_ifx( tfjmp, "ne", "e"),
					target
				);

			_else

				yyerror( "Unexpected operand type (internal HLA error)" );

			_endif

		_endcase



		_case(  in_astop )

			_if( o->r.rightOperand->operandType == const_optype )

				_if
				( 
						o->l.leftOperand->operandType == reg_optype
					&&	o->l.leftOperand->o.reg->Size == 1
					&&	o->r.rightOperand->o.v.pType == tCset 
				)

					char sn[256];

					startStrSeg();
					sprintf( sn, "L%d_cset" sympost, LblCntr++ );
					OutValue
					( 
						sn, 
						o->r.rightOperand->o.v.Type, 
						(union YYSTYPE *) &o->r.rightOperand->o.v 
					);
					endStrSeg();

					asmPrintf
					(
						"%s"
						"\t\tpush\teax\n"
						"\t\tmovzx\teax, %s\n"
						"\t\tbt\tdword ptr %s, eax\n"
						"\t\tpop\teax\n"
						"\t\tj%s\t%s\n",
						leftOpText,
						RtnOperand( o->l.leftOperand, &left, 0 ),
						sn,
						_ifx( tfjmp, "c", "nc"),
						target
					);


				_else

					yyerror
					( 
						"IN operator requires a register and a CSET or a range" 
					);

				_endif

			_elseif( o->r.rightOperand->operandType == mem_optype )

				char *reg32;

				_switch( tolower( *o->l.leftOperand->regname ) )

					_case( 'a' ) reg32 = "eax"; _endcase
					_case( 'b' ) reg32 = "ebx"; _endcase
					_case( 'c' ) reg32 = "ecx"; _endcase
					_case( 'd' ) reg32 = "edx"; _endcase
					_default
					
						yyerror( "Illegal register value" );
						reg32 = "eax";

				_endswitch


				asmPrintf
				(
					"%s"
					"\t\tpush\t%s\n"
					"\t\tmovzx\t%s, %s\n"
					"\t\tbt\tdword ptr %s, %s\n"
					"\t\tpop\t%s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					reg32,
					reg32,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->r.rightOperand, &right, 0 ),
					reg32,

					reg32,
					_ifx( tfjmp, "c", "nc"),
					target
				);
			
			_else
			
				yyerror( "Unexpected cset type (Internal HLA error)" );

			_endif 

		_endcase



		_case(  notin_astop )

			_if( o->r.rightOperand->operandType == const_optype )

				_if
				( 
						o->l.leftOperand->operandType == reg_optype
					&&	o->l.leftOperand->o.reg->Size == 1
					&&	o->r.rightOperand->o.v.pType == tCset 
				)

					char sn[256];

					startStrSeg();
					sprintf( sn, "L%d_cset" sympost, LblCntr++ );
					OutValue
					( 
						sn, 
						o->r.rightOperand->o.v.Type, 
						(union YYSTYPE *) &o->r.rightOperand->o.v 
					);
					endStrSeg();

					asmPrintf
					(
						"%s"
						"\t\tpush\teax\n"
						"\t\tmovzx\teax, %s\n"
						"\t\tbt\tdword ptr %s, eax\n"
						"\t\tpop\teax\n"
						"\t\tj%s\t%s\n",
						leftOpText,
						RtnOperand( o->l.leftOperand, &left, 0 ),
						sn,
						_ifx( tfjmp, "c", "nc"),
						target
					);


				_else

					yyerror
					( 
						"IN operator requires a register and a CSET or a range" 
					);

				_endif

			_elseif( o->r.rightOperand->operandType == mem_optype )

				char *reg32;

				_switch( tolower( *o->l.leftOperand->regname ) )

					_case( 'a' ) reg32 = "eax"; _endcase
					_case( 'b' ) reg32 = "ebx"; _endcase
					_case( 'c' ) reg32 = "ecx"; _endcase
					_case( 'd' ) reg32 = "edx"; _endcase
					_default
					
						yyerror( "Illegal register value" );
						reg32 = "eax";

				_endswitch


				asmPrintf
				(
					"%s"
					"\t\tpush\t%s\n"
					"\t\tmovzx\t%s, %s\n"
					"\t\tbt\tdword ptr %s, %s\n"
					"\t\tpop\t%s\n"
					"\t\tj%s\t%s\n",
					leftOpText,
					reg32,
					reg32,
					RtnOperand( o->l.leftOperand, &left, 0 ),
					RtnOperand( o->r.rightOperand, &right, 0 ),
					reg32,
					reg32,
					_ifx( tfjmp, "nc", "c"),
					target
				);
			
			_else
			
				yyerror( "Unexpected cset type (Internal HLA error)" );

			_endif 

		_endcase


		_case( c_flag )
		
			EmitConditionalJmp
			(
				tfjmp,
				"c",
				"nc",
				target,
				o->l.leftOperand->text
			);
			
		_endcase

											   
		_case( nc_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nc",
				"c",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( s_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"s",
				"ns",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( ns_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"ns",
				"s",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( o_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"o",
				"no",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( no_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"no",
				"o",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( z_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"z",
				"nz",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( nz_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nz",
				"z",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( a_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"a",
				"na",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( na_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"na",
				"a",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( ae_flag ) 			

		
			EmitConditionalJmp
			(
				tfjmp,
				"ae",
				"nae",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( nae_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nae",
				"ae",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( b_flag ) 			
		
			EmitConditionalJmp

			(
				tfjmp,
				"b",
				"nb",
				target,
				o->l.leftOperand->text
			); 

			
		_endcase


		_case( nb_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nb",
				"b",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase



		_case( be_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"be",
				"nbe",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( nbe_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nbe",
				"be",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( l_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"l",
				"nl",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( nl_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nl",
				"l",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( le_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"le",
				"nle",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( nle_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nle",
				"le",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( g_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"g",
				"ng",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( ng_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"ng",
				"g",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( ge_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"ge",
				"nge",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( nge_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"nge",
				"ge",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase



		_case( e_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"e",
				"ne",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( ne_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"ne",
				"e",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase

		_case( pe_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"pe",
				"po",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( po_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"po",
				"pe",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( p_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"p",
				"np",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase


		_case( np_flag ) 			
		
			EmitConditionalJmp
			(
				tfjmp,
				"np",
				"p",
				target,
				o->l.leftOperand->text
			); 
			
		_endcase



		_default

			yyerror( "Illegal operand type (internal HLA error)" );

	_endswitch
	_here;
	FreeOperand( o );
	_here;
	free2( vss left );
	_here;
	free2( vss right );
	_here;

_end( EmitBooleanExpr )


/***************************************************************
/*                                                            */
/* BeginMain-                                                 */
/* EmitExceptionCode-                                         */
/*                                                            */
/* This functions emit code at the start of the main program. */
/*                                                            */
/**************************************************************/


void
BeginMain( int CurOffset )
_begin( BeginMain )

	_if
	(
			targetOS == linux_os 
		||	targetOS == freeBSD_os 
		||	targetOS == macOS_os
	)
	
		_if( assembler == gas )
		 
			asmPrintf
			(
				"\n"
				"\n"
				"\t\t.data\n"
				"\t\t.global\t_envp__hla_\n"
				"\t\t.global\t_argc__hla_\n"
				"\t\t.global\t_argv__hla_\n"
				"_argc__hla_:\t.long\t0\n"
				"_argv__hla_:\t.long\t0\n"
				"_envp__hla_:\t.long\t0\n"
				"\t\t.text\n"
			);
			
		_elseif( assembler == fasm && targetOS == linux_os )
		
			asmPrintf
			(
				"\n" 
				"\t\tsection\t'.data' writeable align 16\n"
				"\t\tglobal\t_envp__hla_\n"
				"\t\tglobal\t_argc__hla_\n"
				"\t\tglobal\t_argv__hla_\n"
				"_argc__hla_\tdd\t0\n"
				"_argv__hla_\tdd\t0\n"
				"_envp__hla_\tdd\t0\n"
				"\n"
				"\t\tsection\t'.text' executable align 16\n"
			);
			
		_else
		
			yyerror( "Unexpected assembler used for this target OS" );
			
		_endif
	
	_endif
	EmitPublic( "_" "HLAMain" );
	NewLn();
	NewLn();
	StartProc( "_" "HLAMain" );
	
	_switch( targetOS )
	
		_case( linux_os )
	
			_if( assembler == gas )
			
				asmPrintf( "\t\t.global\t_start\n" );
				asmPrintf( "_start:\n" );
				asmPrintf( "\t\tmov\teax,esp\n");
				asmPrintf( "_findEnvp_: add\teax,4\n" );
				asmPrintf( "\t\tcmp dword ptr [eax], 0\n" );
				asmPrintf( "\t\tjne\t_findEnvp_\n" );
				asmPrintf( "\t\tadd\teax,4\n" );
				asmPrintf( "\t\tmov\t_envp__hla_,eax\n" );
				asmPrintf( "\t\tlea\teax, [esp+4]\n" );
				asmPrintf( "\t\tmov\t_argv__hla_, eax\n" );
				asmPrintf( "\t\tmov\teax, [esp+0]\n" );
				asmPrintf( "\t\tmov\t_argc__hla_, eax\n" );
				
			_elseif( assembler == fasm )
			
				asmPrintf
				(
					"\t\tpublic\t_start\n" 
					"_start:\n" 
					"\t\tmov\teax,esp\n"
					"_findEnvp_: add\teax,4\n" 
					"\t\tcmp dword [eax], 0\n" 
					"\t\tjne\t_findEnvp_\n" 
					"\t\tadd\teax,4\n" 
					"\t\tmov\t[_envp__hla_],eax\n" 
					"\t\tlea\teax, [esp+4]\n" 
					"\t\tmov\t[_argv__hla_], eax\n" 
					"\t\tmov\teax, [esp+0]\n" 
					"\t\tmov\t[_argc__hla_], eax\n" 
				);
				
			_else
			
				yyerror( "Unknown assembler under Linux (HLA internal error)" );
				
			_endif
			
		_endcase
		
		_case( freeBSD_os )
	 		
			asmPrintf( "\t\t.global\t_start\n" );
			asmPrintf( "_start:\n" );
			asmPrintf( "\t\tmov\teax,esp\n");
			asmPrintf( "_findEnvp_: add\teax,4\n" );
			asmPrintf( "\t\tcmp dword ptr [eax], 0\n" );
			asmPrintf( "\t\tjne\t_findEnvp_\n" );
			asmPrintf( "\t\tadd\teax,4\n" );
			asmPrintf( "\t\tmov\t_envp__hla_,eax\n" );
			asmPrintf( "\t\tlea\teax, [esp+4]\n" );
			asmPrintf( "\t\tmov\t_argv__hla_, eax\n" );
			asmPrintf( "\t\tmov\teax, [esp+0]\n" );
			asmPrintf( "\t\tmov\t_argc__hla_, eax\n" );
			
		_endcase
		
		_case( macOS_os )
		
			yyerror( "Need to fix argc/argv setup for macOS" );
			
		_endcase
		
		_case( windows_os )
		
			// Nothing to do...
			
		_endcase
		
		_default
		
			yyerror( "Undefined targetOS value (internal HLA error)" );
			
	_endswitch
	asmPrintf
	(
		"\n"

		/*
		** As per exception handling requirements,
		** push the following:
		**
		**	Ptr to HLA Exception Handling Code.
		**	Current context (EBP)
		**	Ptr to Windows Exception Handling Routine.
		**	Old TIP (FS:[0]/?ExceptionPtr) value.
		**
		** then point FS:[0] at this record we've
		** just created.
		*/

		"\n"
		";/* Set up the Structured Exception Handler record */\n"
		";/* for this program. */\n\n"

		"\t\tcall\tBuildExcepts" sympost "\n"		
		"\t\tpushd\t0\t\t;/* No Dynamic Link. */\n"
		"\t\tmov\tebp, esp\t;/* Pointer to Main's locals */\n"
		"\t\tpush\tebp\t\t;/* Main's display. */\n"
	);

	_if( -CurOffset > 4 )

		int CurOfs4;

		CurOfs4 = -CurOffset - 4;
		_if( ( CurOfs4 & 3 ) != 0 )

			CurOfs4 = ( CurOfs4 + 4 ) & ~3;

		_endif
		AllocateLocals( CurOfs4 );
		AlignStack();

	_endif
	NewLn();
	NewLn();

_end( BeginMain )



void
EmitExceptionCode( int FullExceptions )
_begin( EmitExceptionCode )

	_if( FullExceptions )

		startCseg();
		EmitImmExtern( "HardwareException" sympost, "near32" );
		EmitImmExtern( "Raise" sympost, "near32" );
		EmitImmExtern( "DefaultExceptionHandler" sympost, "near32" );
		EmitImmExtern( "BuildExcepts" sympost, "near32" );
		EmitPublic( "DfltExHndlr" sympost );
		_if( targetOS == windows_os )
		
			EmitPublic( "HWexcept" sympost ); 
			

			asmPrintf
			(
				"\n"
				"\n"
				"\n"
				";/* " 
				"HWexcept" sympost 
				" gets called when Windows raises the exception. */\n"
				"\n"
			);
			StartProc( "HWexcept" sympost );
			asmPrintf
			(
				"\t\tjmp\tHardwareException" sympost "\n"
			);
			EndProc( "HWexcept" sympost );
			
		_elseif( targetOS == linux_os || targetOS == freeBSD_os )

			EmitPublic( "InstallSignals" sympost );
			EmitImmExtern( "excepts_install_signals", "near32" );
			StartProc( "InstallSignals" sympost );
			asmPrintf
			(
				"\t\tjmp\t\texcepts_install_signals\n"
			);
			EndProc( "InstallSignals" sympost );	
			EmitImmExtern( "InstallSignals" sympost, "near32" );
			
		_else
		
			yyerror( "Need to add code to set up argc/arv for this OS" );
	
		_endif
		
		StartProc( "DfltExHndlr" sympost );
		asmPrintf
		(
			"\t\tjmp\tDefaultExceptionHandler" sympost "\n"
		);
		EndProc ( "DfltExHndlr" sympost );

	_else
		   
		startCseg();
		EmitImmExtern( "shorthwExcept" sympost, "near32" );
		EmitImmExtern( "shortDfltExcept" sympost, "near32" );
		EmitImmExtern( "BuildExcepts" sympost, "near32" );
		EmitImmExtern( "Raise" sympost, "near32" );
		EmitPublic( "DfltExHndlr" sympost );
		_if( targetOS == windows_os )
		
			EmitPublic( "HWexcept" sympost );
			asmPrintf
			(
				"\n"
				"\n"
				"\n"
			);
			StartProc( "HWexcept" sympost );
			asmPrintf
			(
				"\t\tjmp\tshorthwExcept" sympost "\n"
			);
			EndProc( "HWexcept" sympost );
			
		_elseif( targetOS == linux_os  || targetOS == freeBSD_os )
		
			/*
			** In the short exceptions mode, we aren't going
			** to link in the code to install Linux/BSD signals.
			*/
			
			EmitPublic( "InstallSignals" sympost );
			StartProc( "InstallSignals" sympost );
			asmPrintf
			(
				"\t\tret\n"
			);
			EndProc( "InstallSignals" sympost );
			
		_else
		
			yyerror( "Need to add code to set up argc/arv for this OS" );
	
		_endif
		StartProc( "DfltExHndlr" sympost );
		asmPrintf
		(

			"\t\tjmp\tshortDfltExcept" sympost "\n"
		);
		EndProc( "DfltExHndlr" sympost );
		

	_endif

_end( EmitExceptionCode )


void
EmitExit( void )
_begin( EmitExit )

	char mem[ 256 ];

	_switch( targetOS )
	
		_case( windows_os )

			Pushd( 0 );
			LabelToMem( mem, "__imp__ExitProcess@4", "dword" );
			Emit1L( "call", mem );

		_endcase
		
		_case( linux_os )

			Emit2cr( "mov", "eax", "1" ); 
			Emit2rr( "xor", "ebx", "ebx" );
			Emit1C( "int", "0x80" );
	
		_endcase
		
		_case( freeBSD_os )

			Emit2cr( "mov", "eax", "1" ); 
			Emit1C( "push", "0" );
			Emit1C( "push", "0" );
			Emit1C( "int", "0x80" );
			
		_endcase
		
		_default
		
			yyerror( "Need to fix EXIT call for this target OS" );
			
	_endswitch	

_end( EmitExit )


void
EndMain( void )
_begin( EndMain )

	_if( assembler == gas || assembler == fasm )

		asmPrintf( "QuitMain" sympost ":\n");
	
	_else
	
		asmPrintf( "QuitMain" sympost "::\n");
		
	_endif
	EmitExit();
	EndProc( "_HLAMain" );

_end( EndMain )


/******************************************************
/*                                                   */
/* Instruction code emission-                        */
/*                                                   */
/* Pushd-	Pushes a 32-bit constant onto the stack. */
/*                                                   */
/*****************************************************/



/**************************************************************************/
/***																	***/
/***					W A R N I N G !									***/
/***																	***/
/***	Yes, many of the following functions are identical (except, of	***/
/***	course, for their names).  Resist the temptation to merge these	***/
/***	functions back together.  The whole point of separating them	***/
/***	in this fashion is to make it easier to port this code to 		***/
/***	different assemblers.											***/
/***																	***/
/**************************************************************************/



/*************************************************
/*                                              */
/* Emit0-                                       */
/*                                              */
/* Emits an implied (zero) operand instruction. */
/*                                              */
/************************************************/

void 
Emit0
( 
	char *Instr 
)
_begin( Emit0 )

	assert( Instr != NULL );

	asmPrintf
	(
		"\t\t%s\n",
		Instr
	);

_end( Emit0 )


/*****************************************************
/*                                                  */
/* EmitRep-                                         */
/*                                                  */
/* Emits a string instruction with a repeat prefix. */
/*                                                  */
/****************************************************/

void 
EmitRep
(
	char *Prefix, 
	char *Instr
)
_begin( EmitRep )

	char *newInstr;
	char *suffix;
	
	assert( Instr != NULL );
	assert( Prefix != NULL );

	newInstr = Instr;
	suffix = NULL;
	_if( assembler == gas )
	
		_if( _streq( newInstr, "movsd" ) )
		
			newInstr = "movsl";
			Emit0( ".att_syntax" );
			suffix = ".intel_syntax\tnoprefix";
			
		_elseif( _streq( newInstr, "cmpsd" ) )
		
			newInstr = "cmpsl";
			Emit0( ".att_syntax" );
			suffix = ".intel_syntax\tnoprefix";
			
		_endif
	
	_endif
	
	asmPrintf
	(
		"\t%s\t%s\n",
		Prefix,
		newInstr
	);
	_if( suffix != NULL )
	
		asmPrintf( "\t\t.intel_syntax\tnoprefix\n" );
		
	_endif

_end( EmitRep )




/**************************************************************
/* Emit1L-                                                   */
/*                                                           */
/* This function emits an instruction that has a single      */
/* operand.  That operand must be a statement label (i.e.,   */
/* this is usually a jump or call instruction of some sort). */

/* 	                                                         */
/*************************************************************/

void 
Emit1L
( 
	char *Instr, 
	char *Operand 
)
_begin( Emit1L )

	assert( Instr != NULL );
	assert( Operand != NULL );

	asmPrintf
	(
		"\t\t%s\t%s\n",
		Instr,
		Operand
	);

_end( Emit1L )



/***************************************************************
/*                                                            */
/*  Emit1M-                                                   */
/*                                                            */
/*  This function emits an instruction that has a single      */
/*  operand.  That operand is a memory address.               */
/*                                                            */
/**************************************************************/

void 
Emit1M
( 
	char *Instr, 
	char *Operand 
)
_begin( Emit1M )

	assert( Instr != NULL );
	assert( Operand != NULL );

	asmPrintf
	(
		"\t\t%s\t%s\n",
		Instr,
		Operand
	);

_end( Emit1M )


/*************************************************
/*                                              */
/* Emit1R-                                      */
/*                                              */
/* A single-operand instruction with a register */
/* operand.                                     */
/*                                              */
/************************************************/

void 
Emit1R
( 
	char *Instr, 
	char *Operand 
)
_begin( Emit1R )

	assert( Instr != NULL );
	assert( Operand != NULL );

	asmPrintf
	(
		"\t\t%s\t%s\n",
		Instr,
		Operand

	);

_end( Emit1R )


/**************************************************
/*                                               */
/* Emit1C-                                       */
/*                                               */
/* A single operand instruction with a constant. */
/*                                               */
/*************************************************/

void 
Emit1C
( 
	char *Instr, 
	char *Operand 
)
_begin( Emit1C )

	assert( Instr != NULL );
	assert( Operand != NULL );

	asmPrintf
	(
		"\t\t%s\t%s\n",
		Instr,
		Operand
	);

_end( Emit1C )


/****************************************************
/*                                                 */
/* Emit2rr-                                        */
/*                                                 */
/* This function emits the code for an instruction */
/* with two operands and they are both registers.  */
/*                                                 */
/***************************************************/

void 
Emit2rr
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2rr )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2rr )



/************************************************
/*                                             */
/* Emit2rm-                                    */
/*                                             */
/* Emits a two-operand instruction whose       */
/* dest operand is a register and whose source */
/* operand is a memory location.               */
/*                                             */
/***********************************************/

void 
Emit2rm
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2rm )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2rm )


/******************************************************
/*                                                   */
/* Emit2mr-                                          */
/*                                                   */
/* Emits a two-operand instruction whose destination */
/* is a memory location and whose source operand is  */
/* a register.                                       */
/*                                                   */
/*****************************************************/

void 
Emit2mr
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2mr )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2mr )


/******************************************************
/*                                                   */
/* Emit2rc-                                          */
/*                                                   */
/* Emits a two-operand instruction whose destination */
/* operand is a register and whose source operand is */
/* a constant.                                       */
/*                                                   */
/*****************************************************/

void 
Emit2rc
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2rc )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2rc )


/************************************************************
/*                                                         */
/* EmitIRC-                                                */
/*                                                         */
/* Similar to Emit2rc but with different typed parameters. */
/*                                                         */
/***********************************************************/

void
EmitIRC
(
	char *Instr,
	char *reg,
	union YYSTYPE *theConst
)
_begin( EmitIRC )

	asmPrintf
	(
		"\t\t%s\t%s, ",
		Instr,
		reg
	);
	OutStaticConst( theConst->v.Type, theConst );

_end( EmitIRC )


/**********************************************************
/*                                                       */
/* Emit2mc-                                              */
/*                                                       */
/* Emits a two-operand instruction whose destination     */
/* operand is a memory location and whose source operand */
/* is a constant.                                        */
/*                                                       */
/*********************************************************/

void 
Emit2mc
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2mc )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2mc )


/*******************************************************
/*                                                    */
/* EmitIMC-                                           */
/*                                                    */
/* Similar to Emit2mc except for the parameter types. */
/*                                                    */
/******************************************************/

void
EmitIMC
(
	char *Instr,
	char *mem,
	union YYSTYPE *theConst
)
_begin( EmitIMC )

	asmPrintf
	(
		"\t\t%s\t%s, ",
		Instr,
		mem
	);
	OutStaticConst( theConst->v.Type, theConst );

_end( EmitIMC )



/*************************************************************
/*                                                          */
/* Emit2cc-                                                 */
/*                                                          */
/* Emits an instruction with two constant operands (ENTER). */
/*                                                          */
/************************************************************/

void 
Emit2cc
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2cc )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2cc )



/*******************************************************
/*                                                    */
/* Emit2cr-                                           */
/*                                                    */
/* Emits a two-operand instruction, the first operand */
/* is a constant, the second is a register (OUT).     */
/*                                                    */
/******************************************************/

void 
Emit2cr
( 
	char *Instr, 
	char *Op1,
	char *Op2 
)
_begin( Emit2cr )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s\n",
		Instr,
		Op1,
		Op2
	);

_end( Emit2cr )




/*****************************************************
/*                                                  */
/* Emit3rrc-                                        */
/*                                                  */
/* Emits an instruction with three operands (IMUL). */
/* Reg, Reg, const format.                          */
/*                                                  */
/****************************************************/

void 
Emit3rrc
( 
	char *Instr, 
	char *Op1,
	char *Op2,
	char *Op3 
)
_begin( Emit3rrc )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );
	assert( Op3 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s, %s\n",
		Instr,
		Op1,
		Op2,
		Op3
	);

_end( Emit3rrc )


/*****************************************************
/*                                                  */
/* Emit3mrc-                                        */
/*                                                  */
/* Emits an instruction with three operands (IMUL). */
/* Mem, Reg, const format.                          */
/*                                                  */
/****************************************************/

void 
Emit3mrc
( 
	char *Instr, 
	char *Op1,
	char *Op2,
	char *Op3 
)
_begin( Emit3mrc )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );
	assert( Op3 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s, %s\n",
		Instr,
		Op1,
		Op2,
		Op3
	);

_end( Emit3mrc )


/***********************************************
/*                                            */
/* Emit3rmc-                                  */
/*                                            */
/* Emits a three operand instruction that has */
/* a register, memory, and constant operand.  */
/*                                            */
/**********************************************/


void 
Emit3rmc
( 
	char *Instr, 
	char *Op1,
	char *Op2,
	char *Op3 
)
_begin( Emit3rmc )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );
	assert( Op3 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s, %s\n",
		Instr,
		Op1,
		Op2,
		Op3
	);

_end( Emit3rmc )



/******************************************************
/*                                                   */
/* Emit3rrr-                                         */
/*                                                   */
/* Emits an instruction with three register operands */
/* (e.g., SHLD)                                      */
/*                                                   */
/*****************************************************/

void 
Emit3rrr
( 
	char *Instr, 
	char *Op1,
	char *Op2,
	char *Op3 
)
_begin( Emit3rrr )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );
	assert( Op3 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s, %s\n",
		Instr,
		Op1,
		Op2,
		Op3
	);

_end( Emit3rrr )



/****************************************************************************
/*                                                                         */
/* Emit3mrr-                                                               */
/*                                                                         */
/* Emits an instruction with a memory operand and                          */
/* two register operands (e.g., SHLD)                                      */
/*                                                                         */
/***************************************************************************/




void 
Emit3mrr
( 
	char *Instr, 
	char *Op1,
	char *Op2,
	char *Op3 
)
_begin( Emit3mrr )

	assert( Instr != NULL );
	assert( Op1 != NULL );
	assert( Op2 != NULL );
	assert( Op3 != NULL );

	asmPrintf
	(
		"\t\t%s\t%s, %s, %s\n",
		Instr,
		Op1,
		Op2,
		Op3
	);

_end( Emit3mrr )



/*************************************************************************
/*                                                                      */
/* EmitComment-	Emits a comment on a line by itself to the output file. */
/* EmitData-		Emits a label, pseudo-opcode, and operand value.    */
/*                                                                      */
/************************************************************************/

void
EmitComment( char *comment )
_begin( EmitComment )

	asmPrintf( ";/* %s */\n", comment );

_end( EmitComment )


void
EmitData( char *label, char *pseudoOp, char* Operand )
_begin( EmitData )

	char *directive;
	
	_if( assembler == gas )
	
		_if( _streq( pseudoOp, "byte" ) )
			
			directive = ".byte";
			
		_elseif( _streq( pseudoOp, "word" ) )
			
			directive = ".word";
			
		_elseif( _streq( pseudoOp, "dword" ) )
			
			directive = ".long";
			
		_elseif( _streq( pseudoOp, "qword" ))
		
			directive = ".quad";
			
		_endif
		asmPrintf
		( 
			"%-15s: %s\t%s\n", 
			label,
			directive, 
			Operand 
		);
		
	_elseif( assembler == fasm )
	
		_if( _streq( pseudoOp, "byte" ) )
			
			directive = "db";
			
		_elseif( _streq( pseudoOp, "word" ) )
			
			directive = "dw";
			
		_elseif( _streq( pseudoOp, "dword" ) )
			
			directive = "dd";
			
		_elseif( _streq( pseudoOp, "qword" ))
		
			directive = "dq";
			
		_endif
		asmPrintf
		( 
			"%-15s: %s\t%s\n", 
			label,
			directive, 
			Operand 
		);
		
	_else

		asmPrintf
		( 
			"%-15s %s\t%s\n", 
			label,
			pseudoOp, 
			Operand 
		);
		
	_endif

_end( EmitData )



/****************************************************
/*                                                 */
/* RtnBaseType-                                    */
/*                                                 */
/* Returns a pointer to a static string containing */
/* the low-level representation of the base type   */
/* of the parameter.                               */
/*                                                 */
/***************************************************/



static char*
RtnBaseType( struct SymNode *Type )
_begin( RtnBaseType )

	assert( Type != NULL );
	_switch( Type->pType )

		_case( tBoolean )
		_case( tUns8 )
		_case( tByte )
		_case( tInt8 )
		_case( tChar )
		_case( tCset )

				_return gfm_asm( ".byte", "db", "byte" );



		_case( tUns16 )
		_case( tWord )
		_case( tInt16 )
		_case( tWChar )

				_return gfm_asm( ".word", "dw ", "word" );


		_case( tUns32 )
		_case( tDWord )
		_case( tInt32 )
		_case( tProcptr )
		_case( tPointer )
		_case( tString )
		_case( tZString )
		_case( tWString )

				_return gfm_asm( ".long", "dd", "dword" );

		_case( tUns64 )
		_case( tInt64 )
		_case( tQWord )
		
				_return gfm_asm( ".byte", "dq", "qword" );
				
		_case( tTByte )
		
				_return gfm_asm( ".byte", "db", "byte" );
				
		_case( tUns128 )
		_case( tInt128 )
		_case( tLWord );
		
				_return gfm_asm( ".byte", "db", "byte" );
				

		_case( tEnum )

			_returnif( EnumSize == 1 ) gfm_asm( ".byte", "db", "byte" );
			_returnif( EnumSize == 2 ) gfm_asm( ".word", "dw", "word" );
			_return gfm_asm( ".long", "dd", "dword" );



		_case( tReal32 )

			_return gfm_asm( ".float", "dd", "real4" );



		_case( tReal64 )

			_return gfm_asm( ".double", "dq", "real8" );



		_case( tReal80 )

			_return gfm_asm( ".tfloat", "dt", "real10" );



		_case( tArray )

			assert( Type->Type != NULL );
			_return RtnBaseType( Type->Type );
			



	_endswitch
	_return "";

_end( RtnBaseType )






/*****************************************************************/
/*                                                               */
/* OutNameAndType-                                               */
/*                                                               */
/* This function outputs the StaticName of a static variable and */
/* a MASM directive associated with its type.  If this is not a  */
/* simple type, this procedure outputs an equate for the symbol. */
/*                                                               */
/*****************************************************************/

char*
OutName( char *StaticName, struct SymNode *Type )
_begin( OutName )

	char 			*BaseType;
	struct	SymNode	*bt;

	BaseType = RtnBaseType( Type );
	_if( IsPrimitive( Type->pType ))

		EmitTypedLabel( StaticName, BaseType );

	_elseif( Type->pType == tArray )

		_if( strlen( BaseType ) == 0 )

			BaseType = gfm_asm( ".byte", "db", "byte" );

		_endif
		EmitTypedLabel( StaticName, BaseType );

	_else

		bt = GetBaseType( Type );
		BaseType =
			_ifx
			(
				bt->ObjectSize == 1,
				gfm_asm( ".byte", "db", "byte" ),
				_ifx
				(
					bt->ObjectSize == 2,
					gfm_asm( ".word", "dw", "word" ),
					_ifx
					(
						bt->ObjectSize == 4,
						gfm_asm( ".dword", "dd", "dword" ),
						gfm_asm( ".byte", "db", "byte" )
					)
				)
			);

		EmitTypedLabel( StaticName, BaseType );

	_endif
	_return BaseType;

_end( OutName )


void
OutNameAndType( char *StaticName, struct SymNode *Type )
_begin( OutNameAndType )

	char *BaseType;

	_if( Type->pType == tPointer || Type->pType == tProcptr )
	
		/* Handled Elsewhere */
		
	_else
	 
		BaseType = OutName( StaticName, Type );
		asmPrintf( "\t\t%s\t",BaseType );

	_endif

_end( OutNameAndType )



/**********************************************************/
/*                                                        */
/* EmitFields- A recursive routine that emits the fields  */
/* 			of a record constant in reverse order since   */
/* 			they are stored that way in the symbol table. */
/*                                                        */
/**********************************************************/

static int
EmitFields( struct SymNode *Fields, union YYSTYPE *Values )
_begin( EmitFields )

	int				offset;
	int				fOffset;
	int				size;
	int				FieldCnt;
	struct	SymNode	*f;
	union	YYSTYPE	*v;

	_returnif( Fields == NULL ) 0;
	
	FieldCnt = 1;
	fOffset = Fields->Offset;
	size = Fields->ObjectSize;
	f = Fields;		// Save ptr to current field in case we have anon union.
	_if( Fields->pType == tAnonUnion )
	
		// All of the offsets of the objects in the anon union are
		// the same.  Grab the value of the first such field as use
		// that as our ultimate offset:
		
		assert( Fields->Next != NULL );
		fOffset = Fields->Next->Offset;
		
		// Skip over the anonymous union fields and compute the
		// union's size while we're doing this:
		
		size = 0;
		_do
		
			++FieldCnt;
			size = max( size, Fields->ObjectSize );
			Fields = Fields->Next;
			assert( Fields != NULL );
			
		_until( Fields->pType == tAnonUnion )
		
	_endif;
	offset = EmitFields( Fields->Next, Values + FieldCnt);
	_if( offset < fOffset )

		EmitArray( "byte", fOffset - offset );

	_endif				   
	
	// Okay, emit the value for the current field.
	// But have to handle anon unions specially:
	
	offset = 0;
	_if( f->pType == tAnonUnion )
	
		offset = Values->v.Offset;
		size = Values[offset].v.ObjectSize;
		FieldCnt = Values->v.u.unsval;
		_for( int i=0, i < FieldCnt, ++i )
		

			f = f->Next;
			assert( f != NULL );
			
		_endfor
			
	_endif
	OutValue( NULL, f, &Values[ offset ] );
	return( fOffset + size );

_end( EmitFields )




/*******************************************************************/
/*                                                                 */
/* OutValue- Passed a pointer to a value object.                   */
/* 		  Outputs that value with appropriate pseudo-opcode to the */
/* 		  MASM file.                                               */
/*                                                                 */
/*******************************************************************/

static void
EmitID( int cntr, char *id )
_begin( EmitID )

	char idStr[256];

	sprintf( idStr, "L%d%s" sympost, cntr, id );
	asmPrintf( "%-15s ", idStr );

_end( EmitID )



void
OutValue
( 
	char *Name, 
	struct SymNode *Type, 
	union YYSTYPE *Value 
)
_begin( OutValue )

	int				lastAlign;
	char			*BaseType;
	struct SymNode	*realType;
	
	char			label[ 32 ];

	assert( Value != NULL );

	_if( Type == NULL )
	
		yyerror
		( 
			"HLA internal consistency error "
			"(Probably due to cascading errors)"
		);
		_return;
		
	_endif
		
	assert( Type != NULL );

	BaseType = RtnBaseType( Type );

	_if( Name != NULL )

		_if( *BaseType == '\0' ) 
			
			BaseType = "byte ";

		_endif
		EmitTypedLabel( Name, BaseType );

	_endif


	_if( IsPrimitive( Type->pType ))

		_switch( Type->pType )

			_case( tBoolean )
			_case( tUns8 )
			_case( tByte )
			_case( tChar )
			_case( tInt8 )

				EmitByteConst( Value->v.u.unsval );

			_endcase



			_case( tUns16 )
			_case( tWord )
			_case( tWChar )
			_case( tInt16 )

				EmitWordConst( Value->v.u.unsval );

			_endcase



			_case( tUns32 )
			_case( tDWord )
			_case( tInt32 )

				_if( Type->pType == tDWord && Value->v.pType == tPointer )

					EmitAdrs( Value->v.u.strval );

				_else

					EmitDwordConst( Value->v.u.unsval, "" );

				_endif

			_endcase



			_case( tUns64 )
			_case( tQWord )
			_case( tInt64 )

				EmitQwordConst( &Value->v.u.unsval );

			_endcase



			_case( tUns128 )
			_case( tLWord )
			_case( tInt128 )

				EmitLwordConst( &Value->v.u.unsval );

			_endcase





			_case( tEnum )

				_if( EnumSize == 1 ) 

					EmitByteConst( Value->v.u.unsval );

				_elseif( EnumSize == 2 ) 

					EmitWordConst( Value->v.u.unsval );

				_else

					EmitDwordConst( Value->v.u.unsval, "" );

				_endif

			_endcase


			_case( tReal32 )

				EmitReal4Const( Value->v.u.fltval.f.f );

			_endcase



			_case( tReal64 )

				EmitReal8Const( Value->v.u.fltval.f.d );

			_endcase


			_case( tReal80 )

				EmitReal10Const( Value->v.u.fltval );

			_endcase


			_case( tTByte )

				EmitTbyteConst( &Value->v.u.unsval );

			_endcase

			_case( tString )
			_case( tZString )

				_if( Value->v.u.strval == NullPointer )
				
					EmitDwordConst( 0, "" );
					
				_elseif( Value->v.pType == tPointer )
				
					// If it's a pointer type, just emit
					// the offset of the static expression
					// as the string address:
					
					EmitAdrs( Value->v.u.strval );
				
			
				_else
					
					sprintf( label, "L%d_str" sympost, LblCntr );
					EmitAdrs( label );
					EmitString( Value->v.u.strval, LblCntr++ );
					
				_endif

			_endcase

			_case( tWString )

				_if( Value->v.u.strval == NullPointer )
				
					EmitDwordConst( 0, "" );
					
				_else
					
					sprintf( label, "L%d_str" sympost, LblCntr );
					EmitAdrs( label );
					EmitWString( Value->v.u.strval, LblCntr++ );
					
				_endif

			_endcase

			_case( tCset )

				
				_for( int i=0, i < CSetSizeInBytes, ++i )

					EmitByteConst( Value->v.u.csetval[ i ] );

				_endfor

			_endcase
			
			_default
			
				yyerror( "Unhandled base type" );

		_endswitch

	_elseif( Type->pType == tArray )

		union YYSTYPE *CurValue;

		assert( Type->Type != NULL );
		_if( IsStr( Type->Type->pType ))

			// If it is a string array, we've got to use special
			// code because we've got to emit pointers and the
			// string data, and the two cannot be interspersed.

			//
			// Begin by emiting the pointers to the strings.

			_for
			( 
				int CurElement = 0, 
				CurElement < Value->v.NumElements, 
				++CurElement 
			)

				char s[128];

				_if
				( 
					Value->v.u.ArrayOfValues[ CurElement ].u.strval == 
						NullPointer
				)
					
					EmitDwordConst( 0, "" );
					
				_elseif
				( 
					Value->v.u.ArrayOfValues[ CurElement ].pType == tPointer
				)
					
					// If it's a pointer type, just emit
					// the offset of the static expression
					// as the string address:
					
					EmitAdrs( Value->v.u.ArrayOfValues[ CurElement ].u.strval );

				_else				
						
					sprintf
					( 
						s, 
						"L%d_str" sympost, 
						LblCntr + CurElement 
					);
					EmitAdrs( s );
					
				_endif

			_endfor

			// Okay, now emit the actual string data:

			EmitAlign( 4 );
			_for
			( 
				int CurElement = 0, 
				CurElement < Value->v.NumElements, 
				++CurElement 
			)

				_if
				( 
						Value->v.u.ArrayOfValues[ CurElement ].u.strval != 
							NullPointer
					&&	Value->v.u.ArrayOfValues[ CurElement ].pType !=
							tPointer 
				)
					
					EmitString
					( 
						Value->v.u.ArrayOfValues[ CurElement ].u.strval, 
						LblCntr + CurElement
					);
					
				_endif

			_endfor
			LblCntr += Value->v.NumElements;

		
		_elseif( Type->Type->pType == tWString )

			// If it is a unicode string array, we've got to use special
			// code because we've got to emit pointers and the
			// string data, and the two cannot be interspersed.
			//
			// Begin by emiting the pointers to the strings.

			_for
			( 
				int CurElement = 0, 
				CurElement < Value->v.NumElements, 
				++CurElement 
			)

				char s[128];

				_if
				( 
					Value->v.u.ArrayOfValues[ CurElement ].u.strval == 
						NullPointer
				)
					
					EmitDwordConst( 0, "" );
					
				_else				
						
					sprintf
					( 
						s, 
						"L%d_str" sympost, 
						LblCntr + CurElement 
					);
					EmitAdrs( s );
					
				_endif

			_endfor

			// Okay, now emit the actual string data:

			EmitAlign( 4 );
			_for
			( 
				int CurElement = 0, 
				CurElement < Value->v.NumElements, 
				++CurElement 
			)

				_if
				( 
					Value->v.u.ArrayOfValues[ CurElement ].u.strval != 
						NullPointer 
				)
					
					EmitWString
					( 
						Value->v.u.ArrayOfValues[ CurElement ].u.strval, 
						LblCntr + CurElement
					);
					
				_endif

			_endfor
			LblCntr += Value->v.NumElements;

		
		_else
		

			// If this isn't a string array, just emit
			// each element to the output file by recursively
			// calling OutValue.

			CurValue = (union YYSTYPE *) Value->v.u.ArrayOfValues;
			_for
			( 
				int CurElement=0, 
				CurElement < Value->v.NumElements,
				++CurElement
			)

				realType = CurValue->v.Type;
				_if( realType == NULL )
				
					realType = Value->v.Type;
					
				_endif
				_if( CanBeType( CurValue->v.pType ))
				
					realType = 	ptype2type[ CurValue->v.pType ];
					
				_endif
				OutValue( NULL, realType, CurValue );
				++CurValue;

			_endfor

		_endif

	_elseif( Type->pType == tRecord )

		_if( Value->v.pType != tError && Value->v.Type != NULL )

			lastAlign =
				EmitFields
				( 
					Type->Fields, 
					(union YYSTYPE *) Value->v.u.FieldValues
				);

			_if( lastAlign < Type->ObjectSize )

				EmitArray( "byte", Type->ObjectSize - lastAlign );

			_endif

		_endif

	_elseif( Type->pType == tUnion )

		_if( Value->v.pType != tError && Value->v.Type != NULL )
			
			struct	SymNode	*save;

			assert(	Value->v.u.FieldValues != NULL );
			OutValue
			( 
				NULL, 
				Value->v.CurField, 
				YYS &Value->v.u.FieldValues[ Value->v.CurIndex ] 
			);

			_if( Value->v.u.FieldValues->ObjectSize < Type->ObjectSize )

				EmitArray
				( 
					"byte", 
					Type->ObjectSize - Value->v.u.FieldValues->ObjectSize 
				);

			_endif

		_endif

	_elseif
	( 
			( Type->pType == tPointer || Type->pType==tProcptr )
		&&	( Value->v.pType == tPointer || Value->v.pType == tProcptr)
	)

		EmitAdrs( Value->v.u.strval );

	_elseif
	( 
			( Type->pType == tPointer || Type->pType==tProcptr )
		&&	IsNumber( Value->v.pType )
		&&	Value->v.u.unsval == 0
	)

		OutValue( NULL, &dword_ste, Value );

	_endif


_end( OutValue )







/*************************************************************************/
/*                                                                       */
/* StaticConstToStr-                                                     */
/*                                                                       */
/* "type" is the type of the object and "value" is the value associated  */
/* with this object.  Output the associated constant that appears as the */
/* operand field of the text output by "OutNameAndType" above.           */
/*                                                                       */
/*************************************************************************/


static int
HexToStr( unsigned long hex, char *dest )
_begin( HexToStr )

		_switch( assembler )
		
			_case( gas )
			
				_return sprintf( dest, "0x%x", hex );
				
			_endcase
			
			_case( masm )
			_case( tasm )
			_case( fasm )
			
				_return sprintf( dest, "0%xh", hex );
				
			_endcase
			
		_endswitch
		yyerror( "Internal HLA error (unexpected assembler value)" );
		_return 0;

_end( HexToStr )

static int
HexToStr64( union YYSTYPE *value, char *dest )
_begin( HexToStr64 )

		_switch( assembler )
		
			_case( gas )
			
				_return 
					sprintf
					( 
						dest, 
						"0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x", 
						value->v.u.bytes[0], 
						value->v.u.bytes[1], 
						value->v.u.bytes[2], 
						value->v.u.bytes[3], 
						value->v.u.bytes[4], 
						value->v.u.bytes[5], 
						value->v.u.bytes[6], 
						value->v.u.bytes[7] 
					);
				
			_endcase
			
			_case( masm )
			_case( tasm )
			_case( fasm )
			
				_return 
					sprintf
					( 
						dest, 
						"0%x%08xh", 
						value->v.u.lwordval[1], 
						value->v.u.lwordval[0] 
					);
				
			_endcase
			
		_endswitch
		yyerror( "Internal HLA error (unexpected assembler value)" );
		_return 0;

_end( HexToStr64 )



static int
HexToStr80( union YYSTYPE *value, char *dest )
_begin( HexToStr80 )

		_switch( assembler )
		
			_case( gas )
			
				_return 
					sprintf
					( 
						dest, 
						"0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x," 
						"0x%x,0x%x", 
						value->v.u.bytes[0], 
						value->v.u.bytes[1], 
						value->v.u.bytes[2], 
						value->v.u.bytes[3], 
						value->v.u.bytes[4], 
						value->v.u.bytes[5], 
						value->v.u.bytes[6], 
						value->v.u.bytes[7], 
						value->v.u.bytes[8], 
						value->v.u.bytes[9] 
					);
				
			_endcase
			
			_case( masm )
			_case( tasm )
			_case( fasm )
			
				_return 
					sprintf
					( 
						dest, 
						"0%xh,0%xh,0%xh,0%xh,0%xh,0%xh,0%xh,0%xh," 
						"0%xh,0%xh", 
						value->v.u.bytes[0], 
						value->v.u.bytes[1], 
						value->v.u.bytes[2], 
						value->v.u.bytes[3], 
						value->v.u.bytes[4], 
						value->v.u.bytes[5], 
						value->v.u.bytes[6], 
						value->v.u.bytes[7], 
						value->v.u.bytes[8], 
						value->v.u.bytes[9] 
					);
				
			_endcase
			
		_endswitch
		yyerror( "Internal HLA error (unexpected assembler value)" );
		_return 0;

_end( HexToStr80 )


static int
HexToStr128( union YYSTYPE *value, char *dest )
_begin( HexToStr128 )

		_switch( assembler )
		
			_case( gas )
			
				_return 
					sprintf
					( 
						dest, 
						"0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x," 
						"0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x,0x%x", 
						value->v.u.bytes[0], 
						value->v.u.bytes[1], 
						value->v.u.bytes[2], 
						value->v.u.bytes[3], 
						value->v.u.bytes[4], 
						value->v.u.bytes[5], 
						value->v.u.bytes[6], 
						value->v.u.bytes[7], 
						value->v.u.bytes[8], 
						value->v.u.bytes[9], 
						value->v.u.bytes[10], 
						value->v.u.bytes[11], 
						value->v.u.bytes[12], 
						value->v.u.bytes[13], 
						value->v.u.bytes[14], 
						value->v.u.bytes[15] 
					);
				
			_endcase
			
			_case( masm )
			_case( tasm )
			_case( fasm )
			
				_return 
					sprintf
					( 
						dest, 
						"0%xh,0%xh,0%xh,0%xh,0%xh,0%xh,0%xh,0%xh," 
						"0%xh,0%xh,0%xh,0%xh,0%xh,0%xh,0%xh,0%xh", 
						value->v.u.bytes[0], 
						value->v.u.bytes[1], 
						value->v.u.bytes[2], 
						value->v.u.bytes[3], 
						value->v.u.bytes[4], 
						value->v.u.bytes[5], 
						value->v.u.bytes[6], 
						value->v.u.bytes[7], 
						value->v.u.bytes[8], 
						value->v.u.bytes[9], 
						value->v.u.bytes[10], 
						value->v.u.bytes[11], 
						value->v.u.bytes[12], 
						value->v.u.bytes[13], 
						value->v.u.bytes[14], 
						value->v.u.bytes[15] 
					);
				
			_endcase
			
		_endswitch
		yyerror( "Internal HLA error (unexpected assembler value)" );
		_return 0;

_end( HexToStr128 )


/*
** StaticConstToStr-
**
**	Converts a constant to a string that can be output.
**	Returns one if the constant is a static label and the caller
**	must take the offset of this label.
*/

int 
StaticConstToStr( struct SymNode *type, union YYSTYPE *value, char *dest )
_begin( StaticConstToStr )

	char msg[ 128 ];
	unsigned long theValue;
	int index;
	int needsOffset = 0;

	theValue = value->v.u.lwordval[0];

	_switch( type->pType )

		_case( tBoolean )

			theValue = theValue & 1;

		_case( tByte )
		_case( tUns8 )

			theValue = theValue & 0xff;

		_case( tWord )
		_case( tUns16 )

			theValue = theValue & 0xffff;

		_case( tDWord )
		_case( tUns32 )
		_case( tEnum )

			HexToStr( theValue, dest );

		_endcase

		_case( tQWord )
		_case( tInt64 )
		_case( tUns64 )

			
			HexToStr64( value, dest );

		_endcase


		_case( tTByte )
			
			HexToStr80( value, dest );

		_endcase


		_case( tLWord )
		_case( tUns128 )
		_case( tInt128 )

			HexToStr128( value, dest );

		_endcase
		
		
		
		_case( tInt8 )

			theValue = (unsigned long )(theValue & 0xff);

		_case( tInt16 )

			theValue = (unsigned long )(theValue & 0xffff);

		_case( tInt32 )

			HexToStr( theValue, dest );

		_endcase

		

		_case( tChar )
		_case( tWChar )

			theValue = theValue & 0xff;
			HexToStr( theValue, dest );

		_endcase


		_case( tString )
		_case( tZString )
		{

			_if( value->v.u.strval == NullPointer )
			
				dest[0] = '0';
				dest[1] = '\0';
				
			_elseif( value->v.pType == tPointer )
			
				// If it's a pointer type, just emit
				// the offset of the static expression
				// as the string address:
				
				needsOffset = 1;
				strcpy( dest, value->v.u.strval );
				
			
			_else
				
				needsOffset = 1;
				sprintf
				( 
					dest, 
					"L%d_str" sympost,
					LblCntr 
				);
				EmitString( value->v.u.strval, LblCntr++ );
				
			_endif
		}
		_endcase



		_case( tWString )
		{

			_if( value->v.u.strval == NullPointer )
			
				dest[0] = '0';
				dest[1] = '\0';
				
			_else
				
				needsOffset = 1;
				sprintf
				( 
					dest, 
					"L%d_str" sympost,
					LblCntr 
				);
				EmitWString( value->v.u.strval, LblCntr++ );
				
			_endif
		}
		_endcase



		_case( tReal32 )
		_case( tReal64 )
		_case( tReal80 )
		
			_switch( value->v.pType )
			
				_case( tReal32 ) 
		
					sprintf( dest, "%15.8e", *((float*) &value->v.u.fltval) );
					
				_endcase
				
				_case( tReal64 ) 

					sprintf( dest, "%24.18e", *((double*) &value->v.u.fltval) );
					
				_endcase
				
				_case( tReal80 ) 
		
					e80Str( dest, value->v.u.fltval );
					
				_endcase
				
			_endswitch				
			
		_endcase


		_case( tCset )

			index = 0;
			_for( int i=0, i < (CSetSizeInBytes-1), ++i )

				index += HexToStr( value->v.u.csetval[i], dest+index );
				dest[ index++ ] = ',';

			_endfor
			HexToStr( value->v.u.csetval[CSetSizeInBytes-1], dest+index );

		_endcase

		_case( tPointer )

			needsOffset = 1;
			strcpy( dest, value->v.u.strval );

		_endcase

		_default

			sprintf( msg, "Internal HLA error (pType=%d)", type->pType );
			yyerror( msg );
			strcpy( dest, "" );

	_endswitch
	_return needsOffset;

_end( StaticConstToStr )







/*************************************************************************/
/*                                                                       */
/* OutStaticConst-                                                       */
/*                                                                       */
/* "type" is the type of the object and "value" is the value associated  */
/* with this object.  Output the associated constant that appears as the */
/* operand field of the text output by "OutNameAndType" above.           */
/*                                                                       */
/*************************************************************************/



void 
OutStaticConst( struct SymNode *type, union YYSTYPE *value )
_begin( OutStaticConst )

	int needsOffset;
	char staticConst[ 256 ];
	
	needsOffset = StaticConstToStr( type, value, staticConst );
	asmPrintf
	( 
		"%s%s\n", 
		_ifx
		( 
			!needsOffset, 
			"",
			_ifx
			( 
				assembler == masm || assembler == tasm,
				"offset32 ",
				""
			)
		),
		staticConst
	);
	
_end( OutStaticConst )



/*********************************************************************/
/*                                                                   */
/* PrintString- Emits a string in a form suitable for the underlying */
/*              assembler.                                           */
/* 			                                                         */
/* s -             The string to emit.                               */
/* zeroTerminate - true if this function should emit a zero          */
/*                 terminating byte at the end of the string.        */
/*                                                                   */
/*********************************************************************/



void
PrintString( unsigned char *s, int ZeroTerminate )
_begin( PrintString )

	int	charCounter;
	
	_while( *s != '\0' )
			
	
		/*
		** if the character is printable, go ahead and
		** print it, if not, print it in hexadecimal
		** notation.
		**
		** Note: this code prints the quotes as a hexadecimal value.
		*/
		
		_if( ( isgraph( *s ) && *s != '"') || *s == ' ' )

			_switch( assembler )
			
				_case( gas )
				
					asmPrintf( "\n\t\t.ascii\t\"" );
					
				_endcase
				
				_case( masm )
				_case( tasm )
				_case( fasm )
				
					asmPrintf( "\t\tdb\t\"" );
					
				_endcase
				
			_endswitch
			
			charCounter = 64;
			_while( (isgraph( *s ) && *s != '"') || *s == ' ' )
				
				_if( *s == '\\' && assembler == gas )
				
					asmPrintf( "\\" );
					
				_endif
				asmPrintf( "%c", *s );
				++s;
				--charCounter;
				_if( charCounter == 0 && isgraph(*s) )
				
					_switch( assembler )
					
						_case( gas )
						
							asmPrintf( "\"\n\t\t.ascii\t\"" );
							
						_endcase
						
						_case( masm )
						_case( tasm )
						_case( fasm )
						
							asmPrintf( "\"\n\t\tdb\t\"" );
							
						_endcase
						
					_endswitch
					charCounter = 64;
				
				_endif

			_endwhile;
			asmPrintf( "\"\n" );

		_endif
		_if( *s != '\0' )

			_while( ( !isgraph( *s ) || *s == '"') && *s != ' ' && *s != '\0' )
				
				_switch( assembler )
				
					_case( gas )
					
						asmPrintf( "\t\t.byte\t0x%x\n", *s );
						
					_endcase
					
					_case( masm )
					_case( tasm )
					_case( fasm )
					
						asmPrintf( "\t\tdb\t0%xh\n", *s );
						
					_endcase
					
				_endswitch
				++s;

			_endwhile;

		_endif
	
	_endwhile
	
	/*
	** Zero terminate the string
	*/

	_if( ZeroTerminate )
	
		_switch( assembler )
		
			_case( gas )
			
				asmPrintf( "\t\t.byte\t0\n" );
				
			_endcase
			
			_case( masm )
			_case( tasm )
			_case( fasm )
			
				asmPrintf( "\t\tdb\t0\n" );
				
			_endcase
			
		_endswitch

	_endif

_end( PrintString )


/*****************************************************************************
**
** Code below this point *should* be independent of the assembler that
** we're generating code for.
*/



/**********************************************************
**
** RegStrToStr - Converts a general purpose register string to a regnums
**				 value.
*/

enum regnums
RegStrToReg( char *reg )
_begin( RegStrToReg )

	char lc[3];
	
	lc[0] = tolower( reg[0] );
	lc[1] = tolower( reg[1] );
	lc[2] = tolower( reg[2] );
	

	_switch( lc[0] )
	
		_case( 'a' )
		
			_returnif( lc[1] == 'l' ) reg_al;
			_returnif( lc[1] == 'h' ) reg_ah;
			_return reg_ax;
			
			
		_case( 'b' )
		

			_returnif( lc[1] == 'l' ) reg_bl;
			_returnif( lc[1] == 'h' ) reg_bh;
			_returnif( lc[1] == 'x' ) reg_bx;
			_return reg_bp;
			
			
		_case( 'c' )
		
			_returnif( lc[1] == 'l' ) reg_cl;
			_returnif( lc[1] == 'h' ) reg_ch;
			_return reg_cx;
			
			
		_case( 'd' )
		
			_returnif( lc[1] == 'l' ) reg_dl;
			_returnif( lc[1] == 'h' ) reg_dh;
			_returnif( lc[1] == 'x' ) reg_dx;
			_return reg_di;
			
			
		_case( 'e' )
		
			_returnif( lc[1] == 'a' ) reg_eax;
			_if( lc[1] == 'b' ) 
				
				_returnif( lc[2] == 'x' ) reg_ebx;
				_return reg_ebp;
				
			_endif
			_returnif( lc[1] == 'c' ) reg_ecx;
			_if( lc[1] == 'd' ) 
				
				_returnif( lc[2] == 'x' ) reg_edx;
				_return reg_edi;
				
			_endif
			_returnif( lc[2] == 'p' ) reg_esp;
			_return reg_esi;
			
		_default
		
			_returnif( lc[1] == 'i' ) reg_si;
			_return reg_sp;
			
	_endswitch
	_return reg_eax; //never happens
		
		
_end( RegStrToReg )







/**********************************************************/
/*                                                        */
/* PushActualValue-                                       */
/*                                                        */
/* Pushes the value of the actual parameter on the stack. */
/*                                                        */
/**********************************************************/

void
PushActualValue( union YYSTYPE *actual, char *useReg )
_begin( PushActualValue )

	struct adrsYYS adrs;
	enum regnums rnum;

	_if( actual->adrs.ObjectSize == 1 )

		_if( actual->adrs.SymClass == cParm )
		
			// If it's a byte parameter that we're passing along
			// as a parameter to the current call, then just push
			// the whole dword:
			
			push_mem( actual, 4 );
			
		
		_elseif( useReg == NULL )
		
			Pushd( 0 );
			push_r( reg_eax );

			mov_mr( actual, reg_al, 1 );
			BuildAdrs
			( 
				&adrs, 
				1, 				// Size
				1, 				// ObjectSize
				NULL,			 // StaticName
				"ESP", 			// BaseReg
				NULL, 			// IndexReg
				0, 				// Scale
				4,		 		// Disp
				NULL,			// Sym
				&byte_ste,		// Type
				tByte,			// pType
				cNone,			// SymClass
				none_pc,		// pClass
				NULL			// BaseType
			);
			mov_rm( reg_al, YYS &adrs, 1 );
			pop_r( reg_eax );
			
			
		_else // "USE" register is available
											
			rnum = RegStrToReg( useReg );
				
			_switch( rnum )
			
				_case( reg_eax )
				
					mov_mr( actual, reg_al, 1 );
					push_r( reg_eax );
					
				_endcase
				
				_case( reg_ebx )
				
					mov_mr( actual, reg_bl, 1 );
					push_r( reg_ebx );
					
				_endcase
				
				_case( reg_ecx )
				
					mov_mr( actual, reg_cl, 1 );
					push_r( reg_ecx );
					
				_endcase
				
				_case( reg_edx )
				
					mov_mr( actual, reg_dl, 1 );
					push_r( reg_edx );
					
				_endcase
				
				_default
				
					movzx_mr( actual, rnum );
					push_r( rnum );
					
			_endswitch			
			
		_endif			
			

	_elseif( actual->adrs.ObjectSize == 2 )
	
		_if( actual->adrs.SymClass == cParm )
		
			// If it's a word parameter that we're passing along
			// as a parameter to the current call, then just push
			// the whole dword:
			
			push_mem( actual, 4 );
			
		_else
			
			Pushw(0);
			push_mem( actual, 2 );
			
		_endif

	_elseif( actual->adrs.ObjectSize == 3 )


		Pushw( 0 );
		push_mem( actual, 2 );
		
		_if( useReg == NULL || tolower( useReg[2] ) != 'x' )
		
			push_r( reg_eax );
			
			actual->adrs.Disp += 2;
			mov_mr( actual, reg_al, 1 );
			actual->adrs.Disp -=2;
			BuildAdrs
			( 
				&adrs, 
				1, 				// Size
				1, 				// ObjectSize
				NULL,			 // StaticName
				"ESP", 			// BaseReg
				NULL, 			// IndexReg
				0, 				// Scale
				6,		 		// Disp
				NULL,			// Sym
				&byte_ste,		// Type
				tByte,			// pType
				cNone,			// SymClass
				none_pc,		// pClass
				NULL			// BaseType
			);
			mov_rm( reg_al, YYS &adrs, 1 );
			pop_r( reg_eax );
			
		_else
		
			enum regnums ureg;
			
			rnum = RegStrToReg( useReg );
			ureg = reg_eax;
			_switch( rnum )
			
				_case( reg_ebx )
				
					ureg = reg_bl;
					
				_endcase
			
				_case( reg_ecx )
				
					ureg = reg_cl;
					
				_endcase
			
				_case( reg_edx )
				
					ureg = reg_dl;
					
				_endcase
				
			_endswitch
			actual->adrs.Disp += 2;
			mov_mr( actual, ureg, 0 );
			actual->adrs.Disp -=2;
			push_r( ureg );
		
		_endif


	_elseif( actual->adrs.ObjectSize == 4 )

		push_mem( actual, 4 );


	_elseif( actual->adrs.ObjectSize == 8 )

		/*
		** Output the H.O. DWord of this eight-byte object.
		*/

		actual->adrs.Disp += 4;
		push_mem( actual, 4 );

		/*
		** Output the L.O. DWord of this eight-byte object.
		*/

		actual->adrs.Disp -= 4;
		push_mem( actual, 4 );

	
	_elseif( actual->adrs.ObjectSize == 10 )

		/*
		** Output the H.O. word of this ten-byte object.
		*/


		Pushw( 0 );
		actual->adrs.Disp += 8;
		push_mem( actual, 2 );

		/*
		** Output bytes 4-7 of this ten-byte object.
		*/

		actual->adrs.Disp -= 4;
		push_mem( actual, 4 );


		/*
		** Output the L.O. DWord of this ten-byte object.
		*/

		actual->adrs.Disp -= 4;
		push_mem( actual, 4 );
	
	_elseif
	( 
			( actual->adrs.ObjectSize & 3 ) == 0 
		&&	actual->adrs.ObjectSize <= 64
	)

		/*
		** Output the dwords of this object from H.O. down to L.O.
		**
		**	This code only handles objects that are an even multiple
		** of four bytes long.
		*/


		actual->adrs.Disp += actual->adrs.ObjectSize;
		_for( int i=actual->adrs.ObjectSize, i > 0, i=i-4 )

			actual->adrs.Disp -= 4;
			push_mem( actual, 4 );

		_endfor


	/*
	** Okay, handle one of the odd sizes under 64 bytes here.
	*/

	_elseif( actual->adrs.ObjectSize < 64 )

		unsigned long SaveDisp;
		unsigned long index;

		/*
		** Figure out how many dwords to push
		*/

		index = actual->adrs.ObjectSize >> 2;
		SaveDisp = actual->adrs.Disp;
		actual->adrs.Disp += (index << 2);

		/*
		** Allow for an extra dword since the size is not an
		** even multiple of four bytes.
		**
		** First, check to see if there are an odd number of
		** bytes and "push" the odd byte if there are.
		*/

		_if( (actual->adrs.ObjectSize & 3) == 1 )

			_if( useReg == NULL )
			
				Pushd( 0 );
				push_r( reg_eax );
				mov_mr( actual, reg_al, 1 );
				BuildAdrs
				( 
					&adrs, 
					1, 				// Size
					1, 				// ObjectSize
					NULL,			 // StaticName
					"ESP", 			// BaseReg
					NULL, 			// IndexReg
					0, 				// Scale
					4,		 		// Disp
					NULL,			// Sym
					&byte_ste,		// Type
					tByte,			// pType
					cNone,			// SymClass
					none_pc,		// pClass
					NULL			// BaseType
				);
				mov_rm( reg_al, YYS &adrs, 1 );
				pop_r( reg_eax );
				
			_else
			
				rnum = RegStrToReg( useReg );
				movzx_mr( actual, rnum );
				push_r( rnum );
				
			_endif;			

		_elseif( (actual->adrs.ObjectSize & 3) == 2 )

			// Okay, there are an even number of bytes.
			// Push the extra word onto the stack before
			// we start pushing dwords.

			Pushw( 0 );
			push_mem( actual, 2 );

		_elseif( (actual->adrs.ObjectSize & 3) == 3 )

			Pushw( 0 );
			push_mem( actual, 2 );

			push_r( reg_eax );
			actual->adrs.Disp += 2;
			mov_mr( actual, reg_al, 1 );
			actual->adrs.Disp -= 2;
			BuildAdrs
			( 
				&adrs, 
				1, 				// Size
				1, 				// ObjectSize
				NULL,			 // StaticName
				"ESP", 			// BaseReg
				NULL, 			// IndexReg
				0, 				// Scale
				6,		 		// Disp
				NULL,			// Sym
				&byte_ste,		// Type
				tByte,			// pType
				cNone,			// SymClass
				none_pc,		// pClass
				NULL			// BaseType
			);
			mov_rm( reg_al, YYS &adrs, 0 );
			pop_r( reg_eax );
			
		_else

			// Should never happen...

			assert( 0 );

		_endif

		/*
		** Push all the whole dwords in the value:
		*/

		_for( unsigned i = 0, i < index, ++i )

			actual->adrs.Disp -= 4;
			push_mem( actual, 4 );

		_endfor
		actual->adrs.Disp = SaveDisp;


	/*
	** Handle larger objects here.
	*/

	_else

		/*
		** Allocate storage on the stack for the value.
		*/

		sub_ir( ( actual->adrs.ObjectSize + 3 ) & ~3, reg_esp );
		push_r( reg_esi );
		push_r( reg_edi );
		push_r( reg_ecx );
		pushfd();
		cld();
		lea_rm( reg_esi, actual );
		mov_ir( ( actual->adrs.ObjectSize + 3 ) >> 2, reg_ecx );
		BuildAdrs
		( 
			&adrs, 
			1, 				// Size
			1, 				// ObjectSize
			NULL,			 // StaticName
			"ESP", 			// BaseReg
			NULL, 			// IndexReg
			0, 				// Scale
			16,		 		// Disp
			NULL,			// Sym
			&byte_ste,		// Type
			tByte,			// pType
			cNone,			// SymClass
			none_pc,		// pClass
			NULL			// BaseType
		);
		lea_rm( reg_edi, YYS &adrs );
		rep_movsd();

		_if( actual->adrs.ObjectSize & 2 )

			movsw();

		_endif
		_if( actual->adrs.ObjectSize & 1 )

			movsd();

		_endif
		popfd();
		pop_r( reg_ecx );
		pop_r( reg_edi );
		pop_r( reg_esi );
		
	_endif

_end( PushActualValue )


#define encode_EAX 0
#define encode_ECX 1
#define encode_EDX 2
#define encode_EBX 3
#define encode_ESP 4
#define encode_EBP 5
#define encode_ESI 6
#define encode_EDI 7
#define no_reg 8
#define no_index_reg (4<<3)
#define encode_sib 4

#define encode_disp_0 0
#define encode_disp_1 1
#define encode_disp_4 2
#define encode_disp_sym 3

unsigned
encodeReg( char *reg )
_begin( encodeReg )

	_returnif( reg == NULL ) no_reg;
	_returnif( stricmp( reg, "EAX" ) == 0 ) encode_EAX;
	_returnif( stricmp( reg, "ECX" ) == 0 ) encode_ECX;
	_returnif( stricmp( reg, "EDX" ) == 0 ) encode_EDX;
	_returnif( stricmp( reg, "EBX" ) == 0 ) encode_EBX;
	_returnif( stricmp( reg, "ESP" ) == 0 ) encode_ESP;
	_returnif( stricmp( reg, "EBP" ) == 0 ) encode_EBP;
	_returnif( stricmp( reg, "ESI" ) == 0 ) encode_ESI;
	_returnif( stricmp( reg, "EDI" ) == 0 ) encode_EDI;
	_return no_reg;

_end( encodeReg )


// EncodeAdrs-
//
//	Given a YYSTYPE object, computes the mod-reg-r/m byte,
// the SIB byte, and the displacement for that address.
//
//	adrs-
//		Input YYSTYPE address object
//
//	modRm-
//		mod-reg-r/m value is returned here. Note that the
//		reg field is always returned containing zero.
//		Because this function encodes memory addresses,
//		the MOD field always contains 00, 01, or 10 (11 is
//		not a legal encoding for memory addresses).
//
//	hasSib-
//		True if addressing mode has an SIB byte.
//
//	sib-
//		SIB byte; only valid if hasSib is true.
//
//	dispType-
//		One of the following four values:
//			0: no displacement value
//			1: 1-byte displacement constant, returned in disp
//			2: 4-byte displacement constant, returned in disp
//			3: Label + constant value. Label is stored in
//				the "dispLabel" string, constant is stored
//				in the disp parameter. Always assume that this
//				is a four-byte displacement.

void 
EncodeAdrs
(
	union YYSTYPE *adrs,
	unsigned *modRm,
	unsigned *hasSib,
	unsigned *sib,
	unsigned *dispType,
	int *disp,
	char *dispLabel
)
_begin( EncodeAdrs )

	char *indexReg;
	char *baseReg;
	char *staticName;
	unsigned iReg;
	unsigned bReg;
	unsigned scale;

	_here;
	assert( modRm != NULL );
	assert( hasSib != NULL );
	assert( sib != NULL );
	assert( dispType != NULL );
	assert( disp != NULL );
	assert( dispLabel != NULL );

	scale = adrs->adrs.Scale;
	indexReg = 
		_ifx
		( 
			adrs->adrs.IndexReg == NULL || adrs->adrs.IndexReg[0] == '\0',
			NULL,
			adrs->adrs.IndexReg
		);

	baseReg = 
		_ifx
		( 
			adrs->adrs.BaseReg == NULL || adrs->adrs.BaseReg[0] == '\0',
			NULL,
			adrs->adrs.BaseReg
		);

	// indexReg should never be ESP. The HLA grammar should prevent this.

	iReg = encodeReg( indexReg );
	assert( iReg != encode_ESP );

	bReg = encodeReg( baseReg );

	staticName = 
		_ifx
		( 
			adrs->adrs.StaticName == NULL || adrs->adrs.StaticName[0] == '\0',
			NULL,
			adrs->adrs.StaticName
		);

	*modRm = 0;
	*hasSib = 0;
	*sib = 0;
	*dispType = encode_disp_0;
	*disp = 0;
	*dispLabel = '\0';

	///// Set up the MOD field of the mod-reg-r/m byte:

	_if( staticName != NULL )

		// We have a label.

		*dispType = encode_disp_sym;
		*disp = adrs->adrs.Disp;
		_if( *disp < 0 )

			sprintf( dispLabel, "(%s%d)", staticName, *disp );

		_elseif( *disp != 0 )

			sprintf( dispLabel, "(%s+%d)", staticName, *disp );

		_else

			strcpy( dispLabel, staticName );
		
		_endif 
		_if( bReg == no_reg )

			*modRm = 0;
			bReg = encode_EBP;

		_else

			*modRm = 0x80;

		_endif

	_else
		
		*disp = adrs->adrs.Disp;
		_if( bReg == no_reg )

			// Kludge for displacement-only/no-base addressing mode

			*dispType = encode_disp_4;
			*modRm = 0;
			bReg = encode_EBP;

		_elseif( *disp == 0 )

			_if( bReg == encode_EBP )

				// Kludge! If [EBP] addressing mode,
				// we have to force [EBP+disp] because
				// [EBP] is used for the displacement-only
				// addressing mode.

				*dispType = encode_disp_1;
				*modRm = 0x40;	// MOD = %01;

			_else

				*dispType = encode_disp_0;
				// MOD = %00

			_endif


		_elseif( *disp >= -128 && *disp <= 127  )

			*dispType =  encode_disp_1;
			*modRm = 0x40; // MOD = %01.

		_else

			*dispType = encode_disp_4;
			*modRm = 0x80; // MOD = %10.

		_endif

	_endif


	//// Set up the SIB byte here:

	// Set up the scaling for the SIB byte (may not be necessary, but
	// it's convenient to do it here).

	* sib =
		_ifx( scale == 8, 0xc0,
			_ifx( scale == 4, 0x80,
				_ifx( scale == 2, 0x40,	
					/* assume scale == 1 */ 0 )));

	_if( bReg == encode_ESP )

		// We have to handle ESP has a special case. The encoding
		// for ESP as a base register requires an SIB byte.

		*hasSib = 1;
		_if( iReg == no_reg )
		
			*sib |= no_index_reg | encode_ESP;
			
		_else
		
			*sib |= (iReg<<3) | encode_ESP;
			
		_endif  

	_elseif( iReg != no_reg )

		// We have an index register and, therefore,
		// an SIB addressing mode (or the base register is
		// ESP, which forces an SIB byte).
		//
		// Note that the HLA grammar for addressing modes
		// will automatically swap the index and base registers
		// if ESP is specified as the second register in the
		// mode without an explicit scale. Therefore, iReg will
		// never be ESP at this point (unless there was a syntax
		// error, which has already been caught).

		*hasSib = 1;
		*sib |= (iReg<<3) | bReg;

	_endif


	///// Now set up the r/m field of the mod-reg-r/m byte:

	_if( *hasSib )

		*modRm |= encode_sib;	// SIB indication

	_elseif( bReg != no_reg )

		*modRm |= bReg;

	_endif
							
_end( EncodeAdrs )


void
EmitModRegRm( unsigned reg, union YYSTYPE *adrs )
_begin( EmitModRegRm )

	unsigned hasSib;
	unsigned sib;
	unsigned modRm;
	unsigned dispType;
	int disp;
	char dispLabel[256];

	dispLabel[0] = 0;
	EncodeAdrs
	( 
		adrs, 
		&modRm, 
		&hasSib,
		&sib, 
		&dispType, 
		&disp,
		dispLabel 
	);

	EmitByteConst( modRm | (reg<<3) );
	_if( hasSib )
	
		EmitByteConst( sib );
		 
	_endif
	_switch( dispType )

		_case(0)
			// Do nothing, no displacement
		_endcase

		_case(1)
			EmitByteConst( disp );
		_endcase

		_case(2)
			EmitDwordConst( disp, "" );
		_endcase

		_case(3)
			EmitAdrs( dispLabel );
		_endcase
		
		_default
			yyerror( "Internal HLA error" );
		
	_endswitch 

_end( EmitModRegRm )




// EmitShuffleOpcode-
//	Emits the opcode for the pshufd (1), shufpd (2), or shufps (3)
// instructions.

void
EmitShuffleOpcode( unsigned which )
_begin( EmitShuffleOpcode )

	_switch( which )

		_case(0) //pshufd

			asmPrintf( "\n\t\t;/* pshufd */\n" );
			EmitByteConst( 0x66 );
			EmitByteConst( 0x0f );
			EmitByteConst( 0x70 );

		_endcase

		_case(1) //shufpd

			asmPrintf( "\n\t\t;/* shufpd */\n" );
			EmitByteConst( 0x66 );
			EmitByteConst( 0x0f );
			EmitByteConst( 0xc6 );

		_endcase


		_case(2) //shufps

			asmPrintf( "\n\t\t;/* shufps */\n" );
			EmitByteConst( 0x0f );
			EmitByteConst( 0xc6 );

		_endcase

		_default
			yyerror( "Internal HLA error" );

	_endswitch

_end( EmitShuffleOpcode )




// EmitXshiftReg-
//	Emits the opcode bytes for the mmx/xmm shift instructions.

void 
EmitXshiftOpcode( unsigned which )
_begin( EmitXshiftOpcode )

	_switch( which )

		_case(0) //psllw
			EmitByteConst( 0x0f );
			EmitByteConst( 0xf1 );
		_endcase

		_case(1) //pslld
			EmitByteConst( 0x0f );
			EmitByteConst( 0xf2 );
		_endcase

		_case(2) //psllq
			EmitByteConst( 0x0f );
			EmitByteConst( 0xf3 );
		_endcase

		_case(3) //psraw
			EmitByteConst( 0x0f );
			EmitByteConst( 0xe1 );
		_endcase

		_case(4) //psrad
			EmitByteConst( 0x0f );
			EmitByteConst( 0xe2 );
		_endcase

		_case(5) //psrlw
			EmitByteConst( 0x0f );
			EmitByteConst( 0xd1 );
		_endcase

		_case(6) //psrld
			EmitByteConst( 0x0f );
			EmitByteConst( 0xd2 );
		_endcase

		_case(7) //psrlq
			EmitByteConst( 0x0f );
			EmitByteConst( 0xd3 );
		_endcase

	_endswitch

_end( EmitXshiftOpcode )




static void 
EmitXshiftOpImm( unsigned which )
_begin( EmitXshiftOpImm )

	_switch( which )

		_case(0) //psllw
			EmitByteConst( 0x0f );
			EmitByteConst( 0x71 );
		_endcase

		_case(1) //pslld
			EmitByteConst( 0x0f );
			EmitByteConst( 0x72 );
		_endcase

		_case(2) //psllq
			EmitByteConst( 0x0f );
			EmitByteConst( 0x73 );
		_endcase

		_case(3) //psraw
			EmitByteConst( 0x0f );
			EmitByteConst( 0x71 );
		_endcase

		_case(4) //psrad
			EmitByteConst( 0x0f );
			EmitByteConst( 0x72 );
		_endcase

		_case(5) //psrlw
			EmitByteConst( 0x0f );
			EmitByteConst( 0x71 );
		_endcase

		_case(6) //psrld
			EmitByteConst( 0x0f );
			EmitByteConst( 0x72 );
		_endcase

		_case(7) //psrlq
			EmitByteConst( 0x0f );
			EmitByteConst( 0x73 );
		_endcase

	_endswitch

_end( EmitXshiftOpImm )




void
EmitXshiftImm( unsigned which, unsigned reg )
_begin( EmitXshiftImm )

	EmitXshiftOpImm( which );
	_switch( which )

		_case(0) //psllw
		_case(1) //pslld
		_case(2) //psllq
			EmitByteConst( 0xc0 | (0x6<<3) | reg );
		_endcase


		_case(3) //psraw
		_case(4) //psrad
			EmitByteConst( 0xc0 | (0x4<<3) | reg );
		_endcase


		_case(5) //psrlw
		_case(6) //psrld
		_case(7) //psrlq
			EmitByteConst( 0xc0 | (0x2<<3) | reg );
		_endcase

	_endswitch

_end( EmitXshiftImm )
