/********************************************************************

   Module Name:     SimpleTable.java
   Creation Date:   8/28/97
   Description:     Simple Infobus RowsetAccess source

*********************************************************************/

/*
 * Copyright (c) 1997 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

import java.awt.datatransfer.Transferable;
import java.io.*;
import java.net.*;
import java.sql.Types;
import java.sql.SQLException;
import java.util.*;

import javax.infobus.*;

class SimpleTable implements RowsetAccess, ScrollableRowsetAccess, DataItemChangeListener
{
    private static final String     whoami="SimpleTable:"; // for error messages

    private String                  m_tableName=null;
    private InfoBusDataProducer     m_rowsetProducer=null;
    private DbAccess                m_dbAccess=null;
    private String[]                m_columnName=null;
    private int[]                   m_columnDatatypeNumber=null;
    private Vector                  m_data=null; // Vector of Vector of data values; each element is a row
    private int                     m_nCols=0;
    private int                     m_currRowx=(-1); // 0-based current row, for .next()
    private SimpleDataItem[]        m_columnItem=null; // to return column values in row
    private boolean                 m_isNew=false; // current row created by .newRow()
    private boolean                 m_isModified=false; // current row modified via RowsetAccess.setColumnValue
                                                        // ImmediateAccess.setValue

   // --- Constructors for SimpleTable ---------------------------------------

    /** 
     * SimpleTable Constructor 
     * @param String[] columnName array of column names
     * @param int[] columnDatatypeNumber array of datatype numbers (see java.sql.Types)
     * @param Object[][] array of data values
     *
     */
    SimpleTable(String tableName, InfoBusDataProducer rowsetProducer, DbAccess dbAccess,
                 String[] columnName, int[] columnDatatypeNumber, Object[][] data)
        throws Exception
    {
        int nrows;
        Vector rowVector=null;

        if (tableName==null)
            throw new Exception(whoami+"no table name");
        m_tableName = tableName;
        m_rowsetProducer = rowsetProducer;  // for DataItem.getSource()
        m_dbAccess = dbAccess; // for RowsetAccess.getDb()
        m_columnName = columnName;
        m_columnDatatypeNumber = columnDatatypeNumber;
        // Copy data array to Vector of Vectors:
        nrows = data.length; 
        m_nCols = m_columnName.length;
        m_data = new Vector(nrows);
        for (int r=0; r<nrows; r++)
        {
            rowVector = new Vector(m_nCols);
            for (int c=0; c<m_nCols; c++)
            {
                rowVector.addElement(data[r][c]);
            }
            m_data.addElement(rowVector);
        }
    }

    /**
     * SimpleTable Constructor for newCursor
     * @param SimpleTable existing SimpleTable object; we will share data
     */
    SimpleTable(SimpleTable existing)
    {
        m_tableName = existing.m_tableName;
        m_rowsetProducer = existing.m_rowsetProducer;  // for DataItem.getSource()
        m_columnName = existing.m_columnName;
        m_columnDatatypeNumber = existing.m_columnDatatypeNumber;

        m_data = existing.m_data;
        m_nCols = m_columnName.length;
    }


    // --- interface DataItem method implementations -------------

   protected Vector m_changedListeners=null;

   /**
    * Add the indicated change listener to our list.  all listeners on the
    * list will be notified when the DataItem changes value.
    */
   public  void addDataItemChangedListener(DataItemChangeListener l)
   {
      // vector to manage listeners is created here if it does not already exist
      if (null==m_changedListeners)
         m_changedListeners = new Vector( 5,5 );

      // add the indicated element
      m_changedListeners.addElement(l);
   }

   /**
    * Remove the indicated change listener from our list.
    */
   public  void removeDataItemChangedListener(DataItemChangeListener l)
   {
      // m_changedListeners is null until addDataItemChangedListener is called
      if (null!=m_changedListeners)
      {
         // remove the indicated element
         m_changedListeners.removeElement(l);

         // removed last element? delete the vector.
         if (m_changedListeners.isEmpty())
            m_changedListeners = null;
      }
   }

   /**
    * SimpleTable does not support clipboard requests, so we return
    * null to indicate this to the caller.
    */
   public  Transferable getTransferable()
   {
      return null;
   }

   /**
    * SimpleTable does not support data flavors, so return null to indicate this 
    * to the caller.
    * @return an array of MIME_TYPE flavors for the data item, or null if
    * not supported.
    */
   public java.awt.datatransfer.DataFlavor[] getDataFlavors()
   {
       return null;
   }


   /**
    * When asked for the source of the data item, return the InfoBusDataProducer class
    * that called our constructor:
    */
	public InfoBusEventListener getSource()
	{
		return m_rowsetProducer;
	}

    /**
    * SimpleTable does not support any properties, so returns null
    * regardless of propertyName
    * @param propertyName the name of the property desisred
    * @return always null because SimpleTable supports no properties
    */
    public Object getProperty( String propertyName )
    {
        return null;
    }

    /**
    * SimpleTable has no external resources to release; real producers may need
    * to release database cursors etc.
    */
    public void release()
    {
    }

    // --- interface RowsetAccess method implementations -------------

    // Metadata methods


    public int getColumnCount()
    {
   	   return m_nCols;
    }

   public String getColumnName(int columnNumber)
   {
       int colx = _checkColx(columnNumber);
       return m_columnName[colx];
   }

   public String getColumnDatatypeName(int columnNumber)
   {
       int colx = _checkColx(columnNumber);
	   return RowsetAccessHelper.getDatatypeName(m_columnDatatypeNumber[colx]);
   }

   public int getColumnDatatypeNumber(int columnNumber)
   {
       int colx = _checkColx(columnNumber);
       return m_columnDatatypeNumber[colx];
   }

   // Data methods

    /**
    * Set the fetch buffer size.  This is a hint for performance, and does nothing if not
    * supported (does not throw an exception.)
    * @param fetchSize number of rows to fetch at once
    */
    public void setFetchSize(int fetchSize)
    {
        return;
    }

    /**
    * Get the fetch buffer size.  Returns 1 if setting fetch buffer size is not supported.
    * @return current fetch buffer size
    */
    public int getFetchSize()
    {
        return 1;
    }

    /**
    * Move the row cursor to the next row.
    * @return true if successful, false otherwise
    */

    public boolean next() throws SQLException, RowsetValidationException
    {
        return _changeRowx(m_currRowx+1, false);
    }

    public boolean hasMoreRows()
    {
        if (m_currRowx==m_data.size()) // @EOF, trying to go fwd
            return false;
        else
            return false;
    }

    /**
    * Given a one-based column number, return a DataItem which represents and tracks the column value.
    * @return DataItem which implements ImmediateAccess and represents the current value of the column.
    */
   public Object getColumnItem(int columnNumber) throws IndexOutOfBoundsException
   {
       int colx = _checkColx(columnNumber);
       return _getColumnItem(colx);
   }

    /**
    * Given the name of a column in the RowsetAccess item, return a DataItem which represents and tracks the column value.
    * @return DataItem which implements ImmediateAccess and represents the current value of the column.
    */
    public Object getColumnItem(String colName) throws ColumnNotFoundException, DuplicateColumnException
    {
       for (int colx=0; colx<m_nCols; colx++)
       {
           if (colName.equals(m_columnName[colx]))
               return _getColumnItem(colx);
       }
	   throw new ColumnNotFoundException(whoami+m_tableName+":"+colName);
    }


   /**
    * Create a new row and set the row cursor to this row.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public void newRow()
        throws SQLException, RowsetValidationException
    {
        _changeRowx(0, true);
    }

    /**
    * Given the one-based index of a column, set the value of the column in the current row.
    * Note that this may also be done via ImmediateAccess.setValue.
    * @param columnIndex one-based column number
    * @param object new value for the column
    * @exception IndexOutOfBoundsException if the column index is not valid
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException if the new value is invalid
    */
    public void setColumnValue(int columnNumber, Object object)
        throws IndexOutOfBoundsException, SQLException, RowsetValidationException
    {
        int colx = _checkColx(columnNumber);

        if (!(-1<m_currRowx && m_currRowx<m_data.size()))
            throw new SQLException("no current row");
        
        Object o;
        if (object instanceof ImmediateAccess)
        {
            ImmediateAccess ia = (ImmediateAccess) object;
            o = ia.getValueAsObject();
        }
        else
            o = object;
        // For simplicity, this example only handles a few datatypes:
        switch(m_columnDatatypeNumber[colx])
        {
        case java.sql.Types.VARCHAR:
            if (!(o instanceof String))
                throw new RowsetValidationException("new value not String", this, null);
            break;
        case java.sql.Types.INTEGER:
            if (!(o instanceof Integer))
                throw new RowsetValidationException("new value not Integer", this, null);
            break;
        case java.sql.Types.DOUBLE:
            if (!(o instanceof Double))
                throw new RowsetValidationException("new value not Double", this, null);
            break;
        // NEEDSWORK -- add a Date example?
        default:
            throw new RowsetValidationException("new value's datatype not suppported", this, null);
        }

        // Reset the value in the simple data store:
        Vector rowVector = (Vector) m_data.elementAt(m_currRowx);
        rowVector.setElementAt(o, colx);

        // Reset value in DataItem representing column in current row if necessary:
        if (m_columnItem!=null) // some data consumer called getColumnItem on some column
        {
            if (m_columnItem[colx]!=null)
                m_columnItem[colx].setValue(o);
        }

    }

    /**
    * Given the name of a column, set the value of the column in the current row.
    * @param columnName the name of the column
    * @param object new value for column
    * @exception ColumnNotFoundException if a column matching colName is not
    * present in the rowset
    * @exception DuplicateColumnException if more than one column in the 
    * rowset matches colName
    * @exception java.sql.SQLException if the backend rejects the change
    * @see java.sqlSQLException
    * @exception RowsetValidationException if the new value is invalid
    */
    public void setColumnValue(String columnName, Object object)
        throws ColumnNotFoundException, DuplicateColumnException, 
        SQLException, RowsetValidationException
    {
        for (int colx=0; colx<m_nCols; colx++)
        {
            if (m_columnName[colx].equals(columnName))
            {
                setColumnValue(colx, object);
                return;
            }
        }
        throw new ColumnNotFoundException(columnName);
    }

    /**
    * Delete the current row.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException if the new value is invalid
    */
    public void deleteRow()
        throws SQLException, RowsetValidationException
    {
        m_data.removeElementAt(m_currRowx);
        _resetColumnItems();
    }

    /**
    * Validate changes.
    * @exception java.sql.SQLException if the backend rejects some change
    * @see java.sqlSQLException
    * @exception RowsetValidationException if the change is invalid
    */
    public void validate()
        throws SQLException, RowsetValidationException
    {
        return; // this example does not do any validation
    }

    /**
    * Flush all changes to the underlying database.
    * @exception java.sql.SQLException if the backend rejects some change
    * @see java.sqlSQLException
    * @exception RowsetValidationException if the change is invalid
    */
    public void flush()
        throws SQLException, RowsetValidationException
    {
        return; // in this example, add data is kept in the Java variables/Objects.
    }

    /**
    * Lock the current row.  Does nothing if this is not supported by the producer.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public void lockRow()
        throws SQLException, RowsetValidationException
    {
        return;  // not supported in this example, just return
    }


    /** Get the RowsetAccess item's database
    * @return an instance of DbAccess representing the database 
    * associated with the RowsetAccess item
    */
    public DbAccess getDb()
    {
        return m_dbAccess;
    }

    /**
    * Determine if inserting rows is allowed
    * @return true if inserts are allowed
    */
    public boolean canInsert()
    {  
        return true;
    }

    /**
    * Determine if updating rows is allowed
    * @return true if updates are allowed
    */
    public boolean canUpdate()
    {
        return true;
    }

    /**
    * Returns true if modifying the items in the specified column is 
    * allowed, false otherwise.
    * @return true if updating the specified column is allowed, false otherwise
    */
    public boolean canUpdate(String columnName)
        throws ColumnNotFoundException, DuplicateColumnException
    {
        return true;
    }

    /**
    * Returns true if modifying the items in the specified column is 
    * allowed, false otherwise.
    * @return true if updating the specified column is allowed, false otherwise
    */
    public boolean canUpdate(int columnNumber)
        throws IndexOutOfBoundsException
    {
        return true;
    }

    /**
    * Determine if deleting rows is allowed
    * @return true if deletes are allowed
    */
    public boolean canDelete()
    {
        return true;
    }

    //

    /**
    * Get a new, independent cursor for the rowset.
    * @return a new ScrollableRowsetAccess object having the same underlying set of rows
    */
    public ScrollableRowsetAccess newCursor()
    {
        return new SimpleTable(this);
    }

    /**
    * Ask the data provider to keep a specified number of rows immediately available.
    * Does nothing if not supported.
    * @param size number of rows to keep immediately available.
    */
    public void setBufferSize(int size)
    {
    }

    /**
    * Get the current buffer size.  If the provider does not support setting the
    * buffer this, this will be 1.
    * @return current buffere size.
    */
    public int getBufferSize()
    {
        return 1; // this example does not implement a variable size buffer
    }

    /**
    * Determine the number of rows seen so far by the producer
    * @return number of rows
    */
    public int getHighWaterMark()
    {
        return getRowCount();
    }

    /**
    * Determine the total number of rows in the rowset
    * @return number of rows
    */
    public int getRowCount()
    {
        if (m_data!=null)
            return m_data.size();
        else
            return 0;
    }

    /**
    * Move the row cursor to the previous row.
    * @return true if there is a previous row, false otherwise.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public boolean previous()
        throws SQLException, RowsetValidationException
    {
        return _changeRowx(m_currRowx-1, false);
    }

    /**
    * Move the row cursor to the first row.
    * @return true if there is a first row, false if there are no rows.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public boolean first()
        throws SQLException, RowsetValidationException
    {
        return _changeRowx(0, false);
    }

    /**
    * Move the row cursor to the last row.
    * @return true if there is a last row, false if there are no rows.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public boolean last()
        throws SQLException, RowsetValidationException
    {
        return _changeRowx(m_data.size()-1, false);
    }

    /**
    * Move the row cursor forwards or backwards by the specified number of rows.
    * @param relativeNumRows the number of rows to move forward (backwards if negative).
    * @return true if the specified row exists, false otherwise.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public boolean relative(int relativeNumRows)
        throws SQLException, RowsetValidationException
    {
        return _changeRowx(m_currRowx+relativeNumRows, false); // relativeNumRows may be zero or negative
    }

    /**
    * Get the 1-based row number of the current row.
    * @return a number greater than 0 if there is a current row, else 0.
    */
    public int getRow()
    {
        // remember, rowx is 0 based, but RowsetAccess interface uses 1 based row number
        return m_currRowx+1; // 0 if before beginning, -1 if after end., else valid row number
    }

    /**
    * Set the row cursor to the specified row
    * @param rowNumber the 1-based number of the row as returned by getRow.
    * @return true if the specified row exisits, false otherwise.
    * @exception java.sql.SQLException
    * @see java.sqlSQLException
    * @exception RowsetValidationException
    */
    public boolean absolute(int rowNumber)
        throws SQLException, RowsetValidationException
    {
        // remember, rowx is 0 based, but RowsetAccess interface uses 1 based row number
        return _changeRowx(rowNumber-1, false); // -1 to convert from row number to rowx
    }


    // --- interface DataItemChangeListener method implementations -------------------------

	/**
    * handle data item change notifications to note column value has changed
	* @param event contains change information
	*/
	public void dataItemValueChanged(DataItemValueChangedEvent event)
	{
        m_isModified = true;
	}

	/**
    * handle data item change notifications to note column value has changed
	* @param event contains details of the addition
	*/
	public void dataItemAdded(DataItemAddedEvent event)
	{
        m_isModified = true;
	}

	/**
	* Indicates that the data is being deleted by source.
	* @param event contains details of the deletion
	*/
	public void dataItemDeleted(DataItemDeletedEvent event)
	{
        // we own the per column data items
	}


	/** 
	* Indicates that the data is being deleted by source, so we drop references
	* to it.
	* @param event contains details of the revoked data
	*/
	public void dataItemRevoked(DataItemRevokedEvent event)
	{
        // we own the per column data items
	}


	/**
	* We don't handle incoming RowsetAccess data, so ignores this event
	*/
	public void rowsetCursorMoved(RowsetCursorMovedEvent event)
	{
	}


	/*
		*** Old DICE scheme preserved for correctness check on new above

    public void itemValueChanged(DataItemChangeEvent e)
    {
        // SimpleDataItem for some column item for current row changed
        m_isModified = true;
    }

    public void itemSizeChanged(DataItemChangeEvent e)
    {
        // SimpleDataItem for some column item for current row changed
        m_isModified = true;
    }

    public void itemDeleted(DataItemChangeEvent e)
    {
        // We ignore this case; we *ARE* the issuer
    }

    public void itemRevoked(DataItemChangeEvent e)
    {
        // We ignore this case; we *ARE* the issuer
    }

		*** end of old dice scheme
	*/

    // --- internal utility methods -------------

   /** 
    * Centralize checking if supplied column index is valid:
    */
   private int _checkColx(int columnNumber)  throws IndexOutOfBoundsException
   {
       if (columnNumber<1 || columnNumber>m_nCols)
          throw new IndexOutOfBoundsException(whoami+m_tableName+":"+columnNumber);
       return columnNumber-1;   // InfoBus spec defines column indexes to be 1 based;
                        // in this sample we actually use 0 based Vectors and arrays
   }

   /**
    * Get a SimpleDataItem representing the current value in the specified column.
    * This creates a SimpleDataItem if necessary.
    */
   private DataItem _getColumnItem(int colx) // this colx is 0 based
   {
       Object o;

       // caller has already validated/mapped columnNumber using colx = _checkColx(columnNumber)

       if (m_currRowx<0 || m_currRowx>=m_data.size())   // no current row
          o = null;
       else                                     // use value in specified column of current row
       {
           Vector rowVector = (Vector) m_data.elementAt(m_currRowx);
           o = rowVector.elementAt(colx);
       }
       if (m_columnItem==null)
           m_columnItem = new SimpleDataItem[m_nCols];
       if (m_columnItem[colx]==null)    // create the simple data item
       {
           m_columnItem[colx] = new SimpleDataItem(o, m_rowsetProducer);
           ((DataItemChangeManager) m_columnItem[colx]).addDataItemChangeListener(this);
       }
       else                             // reset the simple item (generates a DataItemChangedEvent)
       {
           m_columnItem[colx].setValue(o);
       }

       return m_columnItem[colx];
   }

    /**
     * Add one empty row at end of m_data:
     */
    private void _addEmptyRow()
    {
        Vector rowVector = new Vector(m_nCols);
        for (int c=0; c<m_nCols; c++)
        {
            rowVector.addElement(null);
        }
        m_data.addElement(rowVector);
    }

    private void _validateRow()
    {
    }

    private void _saveRow()
    {
    }


    private void _resetColumnItems()
    {
        // Reset values in DataItems for row if necessary:
        if (m_columnItem!=null) // some data consumer called getColumnItem on some column
                                // so we created at least one column item
        {
            for (int colx=0; colx<m_nCols; colx++)
            {
                if (m_columnItem[colx]!=null)
                {
                    Object o = null;
                    if (-1<m_currRowx && m_currRowx<m_data.size())
                    {
                        Vector rowVector = (Vector) m_data.elementAt(m_currRowx);
                        o = rowVector.elementAt(colx);
                    }
                    m_columnItem[colx].setValue(o); // sets value to null for new row
                }
            }
        }
    }

    /**
     * Change the row cursor; this may validate current row and send changes to "backend"
     */
    private boolean _changeRowx(int rowx, boolean newRow)
        throws RowsetValidationException, SQLException, IndexOutOfBoundsException

    {
        int newRowx=rowx;

        if (!newRow)
        {
            if (m_currRowx==(-1) && rowx<0) // @BOF, trying to back up
                return false;
            else
            if (m_currRowx==m_data.size() && rowx>=m_data.size()) // @EOF, trying to go fwd
                return false;
            if (rowx<0)
                newRowx = -1; // before beginning, BOF
            else
            if (rowx<m_data.size())
                newRowx = rowx;
            else
                newRowx = m_data.size(); // after end, EOF

            if  (newRowx == m_currRowx)
                return true; // not actually changing row
        }

        if (m_isModified) // old row modified
        {
            _validateRow(); // 
            _saveRow();
            m_isModified = false;
        }

        if (newRow)
        {
            _addEmptyRow();
            m_currRowx = m_data.size()-1;
            m_isNew = true;
        }
        else
            m_currRowx = newRowx;

        _resetColumnItems();

        if (-1<m_currRowx && m_currRowx<m_data.size())
            return true;  // there is a current row.
        else
            return false; // no current row, before beginning or after end
    }


} // class SimpleTable

