/******************************************************************************
* Symb_lib.h - header file for the SYMBolic library.			      *
* This header is also the interface header to the world.		      *
* This library is closely related to symb_lib and should be linked with it.   *
*******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                 *
*******************************************************************************
* Written by Gershon Elber, Mar. 90.					      *
******************************************************************************/

#ifndef SYMB_LIB_H
#define SYMB_LIB_H

#include <stdio.h>
#include "irit_sm.h"
#include "cagd_lib.h"

typedef enum {
    SYMB_ERR_WRONG_SRF,
    SYMB_ERR_BZR_CRV_EXPECT,
    SYMB_ERR_BZR_SRF_EXPECT,
    SYMB_ERR_BSP_CRV_EXPECT,
    SYMB_ERR_BSP_SRF_EXPECT,
    SYMB_ERR_RATIONAL_EXPECTED,
    SYMB_ERR_NO_CROSS_PROD,
    SYMB_ERR_POWER_NO_SUPPORT,
    SYMB_ERR_CRV_FAIL_CMPT,
    SYMB_ERR_SRF_FAIL_CMPT,
    SYMB_ERR_UNDEF_CRV,
    SYMB_ERR_UNDEF_SRF,
    SYMB_ERR_UNDEF_GEOM,
    SYMB_ERR_OUT_OF_RANGE,
    SYMB_ERR_DIR_NOT_CONST_UV,
    SYMB_ERR_REPARAM_NOT_MONOTONE,
    SYMB_ERR_BSPLINE_NO_SUPPORT,
    SYMB_ERR_WRONG_PT_TYPE,
    SYMB_ERR_ONLY_2D_OR_3D,
    SYMB_ERR_ONLY_2D,
    SYMB_ERR_ONLY_3D,
    SYMB_ERR_RATIONAL_NO_SUPPORT,
    SYMB_ERR_SRFS_INCOMPATIBLE,
    SYMB_ERR_CRVS_INCOMPATIBLE,
    SYMB_ERR_CANNOT_COMP_NORMAL,
    SYMB_ERR_TOO_COMPLEX,
    SYMB_ERR_UNSUPPORT_PT,
    SYMB_ERR_W_NOT_SAME,
    SYMB_ERR_SCALAR_EXPECTED,
    SYMB_ERR_POLY_CONST_SRF,
    SYMB_ERR_COPLANAR_GEOMETRY,
    SYMB_ERR_ILLEGAL_PARAMETERS,
    SYMB_ERR_INCONSIST_EDGE_BHOLE,
    SYMB_ERR_BIARC_FIT_FAIL,
    SYMB_ERR_SPL_PROD_FAILED,
    SYMB_ERR_MATCH_FAILED,
    SYMB_ERR_MINIMUM_LINEAR,
    SYMB_ERR_DIV_ZERO,
    SYMB_ERR_INVALID_AXIS,

    SYMB_ERR_UNDEFINE_ERR
} SymbFatalErrorType;

typedef enum {
    SYMB_SUBDIV_STRAT_ALTERNATE = 1,
    SYMB_SUBDIV_STRAT_MIN_MAX,
    SYMB_SUBDIV_STRAT_MIN_MIN
} SymbPlSubdivStrategyType;

typedef enum {
    SYMB_CRV_APPROX_UNIFORM = 0,
    SYMB_CRV_APPROX_TOLERANCE,
    SYMB_CRV_APPROX_CURVATURE
} SymbCrvApproxMethodType;

typedef enum {                       /* Type of relation between two curves. */
    SYMB_CRVREL_NO_RELATION = 0,
    SYMB_CRVREL_SAME_CRVS,
    SYMB_CRVREL_OVERLAPPING_CRVS,		
    SYMB_CRVREL_DISTINCT_CRVS
} SymbCrvRelType;

typedef struct SymbMultiResCrvStruct {
    struct SymbMultiResCrvStruct *Pnext;
    CagdCrvStruct **HieCrv;
    int RefineLevel, Levels, Periodic;
} SymbMultiResCrvStruct;

typedef struct SymbNormalConeStruct {
    CagdVType ConeAxis;
    CagdRType ConeAngle;
} SymbNormalConeStruct;

typedef struct SymbArcStruct {
    struct SymbArcStruct *Pnext;
    struct IPAttributeStruct *Attr;
    int Arc;
    CagdPType Pt1, Cntr, Pt2;
} SymbArcStruct;

typedef CagdCrvStruct *(*SymbAdapIsoDistSqrFuncType)(int,
						     CagdCrvStruct *,
						     CagdCrvStruct *,
						     CagdCrvStruct *,
						     CagdCrvStruct *);
typedef CagdRType (*SymbPlErrorFuncType)(CagdSrfStruct *Srf,
					 CagdSrfDirType Dir,
					 int SubdivDepth);
typedef CagdCrvStruct *(*SymbOffCrvFuncType)(CagdCrvStruct *Crv,
					     CagdRType Off,
					     CagdBType B);
typedef CagdCrvStruct *(*SymbVarOffCrvFuncType)(CagdCrvStruct *Crv,
						CagdCrvStruct *VarOff,
						CagdBType B);
typedef int (*SymbUniformAprxSrfPtImportanceFuncType)(CagdSrfStruct *Srf,
						      CagdRType u,
						      CagdRType v);

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

/******************************************************************************
* Routines to handle objects generically.				      *
******************************************************************************/
SymbArcStruct *SymbArcArrayNew(int Size);
SymbArcStruct *SymbArcNew(int Arc);
SymbArcStruct *SymbArcCopy(SymbArcStruct *Arc);
SymbArcStruct *SymbArcCopyList(SymbArcStruct *ArcList);
void SymbArcFree(SymbArcStruct *Arc);
void SymbArcFreeList(SymbArcStruct *ArcList);
void SymbArcArrayFree(SymbArcStruct *ArcArray, int Size);

/******************************************************************************
* Routines to handle curves generically.				      *
******************************************************************************/
CagdPolylineStruct *SymbCrv2Polyline(CagdCrvStruct *Crv,
				     CagdRType TolSamplesPerCurve,
				     SymbCrvApproxMethodType Method,
				     CagdBType OptiLin);
CagdCrvStruct *SymbSubdivCrvsAtPoles(CagdCrvStruct *Crvs, CagdRType Eps);
CagdCrvStruct *SymbCrvAdd(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvSub(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvMult(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvInvert(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrvScalarScale(CagdCrvStruct *Crv, CagdRType Scale);
CagdCrvStruct *SymbCrvDotProd(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvVecDotProd(CagdCrvStruct *Crv, CagdVType Vec);
CagdCrvStruct *SymbCrvMultScalar(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvCrossProd(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvVecCrossProd(CagdCrvStruct *Crv, CagdVType Vec);
CagdCrvStruct *SymbCrvRtnlMult(CagdCrvStruct *Crv1X,
			       CagdCrvStruct *Crv1W,
			       CagdCrvStruct *Crv2X,
			       CagdCrvStruct *Crv2W,
			       CagdBType OperationAdd);
CagdCrvStruct *SymbCrvEnclosedArea(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrv2DCurvatureSqr(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrv3DCurvatureSqr(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrv3DRadiusNormal(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrv2DUnnormNormal(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrv3DCurvatureNormal(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrv2DCurvatureSign(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrvGenSignedCrvtr(CagdCrvStruct *Crv,
				     int Samples,
				     int Order,
				     int ArcLen);
CagdCrvStruct *SymbSignedCrvtrGenCrv(CagdCrvStruct *Crvtr,
				     CagdRType Tol,
				     int Order,
				     int Periodic);
CagdCrvStruct *SymbMakePosCrvCtlPolyPos(CagdCrvStruct *OrigCrv);
CagdPtStruct *SymbCrv2DInflectionPts(CagdCrvStruct *Crv, CagdRType Epsilon);
CagdPtStruct *SymbCrvExtremCrvtrPts(CagdCrvStruct *Crv, CagdRType Epsilon);
CagdCrvStruct **SymbCrvSplitScalarN(CagdCrvStruct *Crv);
void SymbCrvSplitScalar(CagdCrvStruct *Crv,
			CagdCrvStruct **CrvW,
			CagdCrvStruct **CrvX,
			CagdCrvStruct **CrvY,
			CagdCrvStruct **CrvZ);
CagdCrvStruct *SymbCrvMergeScalarN(CagdCrvStruct *CrvW, 
				   CagdCrvStruct **CrvVec, 
				   int NumCrvs);
CagdCrvStruct *SymbCrvMergeScalar(CagdCrvStruct *CrvW,
				  CagdCrvStruct *CrvX,
				  CagdCrvStruct *CrvY,
				  CagdCrvStruct *CrvZ);
CagdCrvStruct *SymbCrvUnitLenScalar(CagdCrvStruct *OrigCrv,
				    CagdBType Mult,
				    CagdRType Epsilon);
CagdCrvStruct *SymbCrvUnitLenCtlPts(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrvSqrtScalar(CagdCrvStruct *OrigCrv, CagdRType Epsilon);
CagdCrvStruct *SymbCrvArcLenSclrCrv(CagdCrvStruct *Crv, CagdRType Epsilon);
CagdCrvStruct *SymbCrvArcLenCrv(CagdCrvStruct *Crv,
				CagdRType Fineness,
				int Order);
CagdRType SymbCrvArcLen(CagdCrvStruct *Crv, CagdRType Epsilon);
CagdPtStruct *SymbCrvArcLenSteps(CagdCrvStruct *Crv,
				 CagdRType Length,
				 CagdRType Epsilon);
int SymbCrvMonotoneCtlPt(CagdCrvStruct *Crv, int Axis);
CagdCrvStruct *SymbComposeCrvCrv(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *SymbComposePeriodicCrvCrv(CagdCrvStruct *Crv1,
					 CagdCrvStruct *Crv2,
					 CagdRType Epsilon);
int SymbComposeSrfSetCache(int Cache);
void SymbComposeSrfClrCache(CagdSrfStruct *Srf);
CagdCrvStruct *SymbComposeSrfCrv(CagdSrfStruct *Srf, CagdCrvStruct *Crv);
CagdCrvStruct *SymbComposePeriodicSrfCrv(CagdSrfStruct *Srf,
					 CagdCrvStruct *Crv,
					 CagdRType Epsilon);
CagdCrvStruct *SymbDecomposeCrvCrv(CagdCrvStruct *Crv);
CagdRType SymbDistCrvPoint(CagdCrvStruct *Crv,
			   CagdPType Pt,
			   CagdBType MinDist,
			   CagdRType Epsilon);
CagdPtStruct *SymbLclDistCrvPoint(CagdCrvStruct *Crv,
				  CagdPType Pt,
				  CagdRType Epsilon);
CagdRType SymbDistCrvLine(CagdCrvStruct *Crv,
			  CagdLType Line,
			  CagdBType MinDist,
			  CagdRType Epsilon);
CagdPtStruct *SymbLclDistCrvLine(CagdCrvStruct *Crv,
				 CagdLType Line,
				 CagdRType Epsilon,
				 CagdBType InterPos,
				 CagdBType ExtremPos);
CagdPtStruct *SymbCrvRayInter(CagdCrvStruct *Crv,
			      CagdPType RayPt,
			      CagdVType RayDir,
			      CagdRType Epsilon);
CagdRType SymbDistBuildMapToCrv(CagdCrvStruct *Crv,
				CagdRType Tolerance,
				CagdRType *XDomain,
				CagdRType *YDomain,
				CagdRType **DiscMap,
				CagdRType DiscMapXSize,
				CagdRType DiscMapYSize);
CagdPtStruct *SymbCrvZeroSet(CagdCrvStruct *Crv, int Axis, CagdRType Epsilon);
CagdPtStruct *SymbCrvExtremSet(CagdCrvStruct *Crv,
			       int Axis,
			       CagdRType Epsilon);
CagdPtStruct *SymbCrvConstSet(CagdCrvStruct *Crv,
			      int Axis,
			      CagdRType Epsilon,
			      CagdRType ConstVal);
CagdBType SymbCrvPosNegWeights(CagdCrvStruct *Crv);
CagdCrvStruct *SymbCrvOffset(CagdCrvStruct *Crv,
			     CagdRType OffsetDist,
			     CagdBType BezInterp);
CagdCrvStruct *SymbCrvVarOffset(CagdCrvStruct *Crv,
				CagdCrvStruct *VarOffsetDist,
				CagdBType BezInterp);
CagdCrvStruct *SymbCrvSubdivOffset(CagdCrvStruct *Crv,
				   CagdRType OffsetDist,
				   CagdRType Tolerance,
				   CagdBType BezInterp);
CagdCrvStruct *SymbCrvAdapOffset(CagdCrvStruct *OrigCrv,
				 CagdRType OffsetDist,
				 CagdRType OffsetError,
				 SymbOffCrvFuncType OffsetAprxFunc,
				 CagdBType BezInterp);
CagdCrvStruct *SymbCrvAdapVarOffset(CagdCrvStruct *OrigCrv,
				    CagdCrvStruct *VarOffsetDist,
				    CagdRType OffsetError,
				    SymbVarOffCrvFuncType VarOffsetAprxFunc,
				    CagdBType BezInterp);
CagdCrvStruct *SymbCrvAdapOffsetTrim(CagdCrvStruct *OrigCrv,
				     CagdRType OffsetDist,
				     CagdRType OffsetError,
				     SymbOffCrvFuncType OffsetAprxFunc,
				     CagdBType BezInterp);
CagdCrvStruct *SymbCrvLeastSquarOffset(CagdCrvStruct *Crv,
				       CagdRType OffsetDist,
				       int NumOfSamples,
				       int NumOfDOF,
				       int Order,
				       CagdRType *Tolerance);
CagdCrvStruct *SymbCrvMatchingOffset(CagdCrvStruct *Crv,
				     CagdRType OffsetDist,
				     CagdRType Tolerance);
CagdCrvStruct *SymbCrvTrimGlblOffsetSelfInter(CagdCrvStruct *Crv,
					      CagdCrvStruct *OffCrv,
					      CagdRType Tolerance,
					      CagdRType TrimAmount,
					      CagdRType NumerTol);
CagdCrvStruct *SymbCrvCrvConvolution(CagdCrvStruct *Crv1,
				     CagdCrvStruct *Crv2,
				     CagdRType OffsetDist,
				     CagdRType Tolerance);
CagdSrfStruct *SymbEnvOffsetFromCrv(CagdCrvStruct *Crv,
				    CagdRType Height,
				    CagdRType Tolerance);
CagdRType *SymbUniformAprxPtOnCrvDistrib(CagdCrvStruct *Crv,
					 CagdBType ParamUniform,
					 int n);
CagdPtStruct *SymbGetParamListAndReset(CagdRType TMin, CagdRType TMax);
void SymbInsertNewParam(CagdRType t);

/******************************************************************************
* Routines to handle surfaces generically.				      *
******************************************************************************/
CagdPolygonStruct *SymbSrf2Polygons(CagdSrfStruct *Srf,
				    int FineNess,
				    CagdBType ComputeNormals,
				    CagdBType FourPerFlat,
				    CagdBType ComputeUV);
CagdPolylineStruct *SymbSrf2Polylines(CagdSrfStruct *Srf,
				      int NumOfIsocurves[2],
				      CagdRType TolSamples,
				      SymbCrvApproxMethodType Method);
CagdCrvStruct *SymbSrf2Curves(CagdSrfStruct *Srf, int NumOfIsocurves[2]);
CagdSrfStruct *SymbSrfAdd(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *SymbSrfSub(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *SymbSrfMult(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *SymbSrfInvert(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfScalarScale(CagdSrfStruct *Srf, CagdRType Scale);
CagdSrfStruct *SymbSrfDotProd(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *SymbSrfVecDotProd(CagdSrfStruct *Srf, CagdVType Vec);
CagdSrfStruct *SymbSrfMultScalar(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *SymbSrfCrossProd(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *SymbSrfVecCrossProd(CagdSrfStruct *Srf, CagdVType Vec);
CagdSrfStruct *SymbSrfRtnlMult(CagdSrfStruct *Srf1X,
			       CagdSrfStruct *Srf1W,
			       CagdSrfStruct *Srf2X,
			       CagdSrfStruct *Srf2W,
			       CagdBType OperationAdd);
CagdSrfStruct *SymbSrfNormalSrf(CagdSrfStruct *Srf);
void SymbMeshAddSub(CagdRType **DestPoints,
		    CagdRType **Points1,
		    CagdRType **Points2,
		    CagdPointType PType,
		    int Size,
		    CagdBType OperationAdd);
CagdSrfStruct **SymbSrfSplitScalarN(CagdSrfStruct *Srf);
void SymbSrfSplitScalar(CagdSrfStruct *Srf,
			CagdSrfStruct **SrfW,
			CagdSrfStruct **SrfX,
			CagdSrfStruct **SrfY,
			CagdSrfStruct **SrfZ);
CagdSrfStruct *SymbSrfMergeScalarN(CagdSrfStruct *SrfW, 
				   CagdSrfStruct **SrfVec, 
				   int NumSrfs);
CagdSrfStruct *SymbSrfMergeScalar(CagdSrfStruct *SrfW,
				  CagdSrfStruct *SrfX,
				  CagdSrfStruct *SrfY,
				  CagdSrfStruct *SrfZ);
CagdCrvStruct *SymbPrmtSclrCrvTo2D(CagdCrvStruct *Crv,
				   CagdRType Min,
				   CagdRType Max);
CagdSrfStruct *SymbPrmtSclrSrfTo3D(CagdSrfStruct *Srf,
				   CagdRType UMin, CagdRType UMax,
				   CagdRType VMin, CagdRType VMax);
CagdRType *SymbExtremumCntPtVals(CagdRType **Points,
				 int Length,
				 CagdBType FindMinimum);
void SymbSetAdapIsoExtractMinLevel(int MinLevel);
CagdCrvStruct *SymbAdapIsoExtract(CagdSrfStruct *Srf,
				  CagdSrfStruct *NSrf,
				  SymbAdapIsoDistSqrFuncType AdapIsoDistFunc,
				  CagdSrfDirType Dir,
				  CagdRType Eps,
				  CagdBType FullIso,
				  CagdBType SinglePath);

CagdSrfStruct *SymbSrfVolume1Srf(CagdSrfStruct *Srf, CagdBType Integrate);
CagdRType SymbSrfVolume1(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfVolume2Srf(CagdSrfStruct *Srf, CagdBType Integrate);
CagdRType SymbSrfVolume2(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfFirstMomentSrf(CagdSrfStruct *Srf,
				     int Axis,
				     CagdBType Integrate);
CagdRType SymbSrfFirstMoment(CagdSrfStruct *Srf, int Axis);
CagdSrfStruct *SymbSrfSecondMomentSrf(CagdSrfStruct *Srf,
				      int Axis1,
				      int Axis2,
				      CagdBType Integrate);
CagdRType SymbSrfSecondMoment(CagdSrfStruct *Srf, int Axis1, int Axis2);

void SymbSrfFff(CagdSrfStruct *Srf,
		CagdSrfStruct **DuSrf,
		CagdSrfStruct **DvSrf,
		CagdSrfStruct **FffG11,
		CagdSrfStruct **FffG12,
		CagdSrfStruct **FffG22);
void SymbSrfSff(CagdSrfStruct *DuSrf,
		CagdSrfStruct *DvSrf,
		CagdSrfStruct **SffL11,
		CagdSrfStruct **SffL12,
		CagdSrfStruct **SffL22,
		CagdSrfStruct **SNormal);
void SymbSrfTff(CagdSrfStruct *Srf,
		CagdSrfStruct **TffL11,
		CagdSrfStruct **TffL12,
		CagdSrfStruct **TffL22);
CagdSrfStruct *SymbSrfDeterminant2(CagdSrfStruct *Srf11,
				   CagdSrfStruct *Srf12,
				   CagdSrfStruct *Srf21,
				   CagdSrfStruct *Srf22);
CagdSrfStruct *SymbSrfDeterminant3(CagdSrfStruct *Srf11,
				   CagdSrfStruct *Srf12,
				   CagdSrfStruct *Srf13,
				   CagdSrfStruct *Srf21,
				   CagdSrfStruct *Srf22,
				   CagdSrfStruct *Srf23,
				   CagdSrfStruct *Srf31,
				   CagdSrfStruct *Srf32,
				   CagdSrfStruct *Srf33);
CagdCrvStruct *SymbCrvDeterminant2(CagdCrvStruct *Crv11,
				   CagdCrvStruct *Crv12,
				   CagdCrvStruct *Crv21,
				   CagdCrvStruct *Crv22);
CagdCrvStruct *SymbCrvDeterminant3(CagdCrvStruct *Crv11,
				   CagdCrvStruct *Crv12,
				   CagdCrvStruct *Crv13,
				   CagdCrvStruct *Crv21,
				   CagdCrvStruct *Crv22,
				   CagdCrvStruct *Crv23,
				   CagdCrvStruct *Crv31,
				   CagdCrvStruct *Crv32,
				   CagdCrvStruct *Crv33);
CagdSrfStruct *SymbSrfGaussCurvature(CagdSrfStruct *Srf, CagdBType NumerOnly);
CagdSrfStruct *SymbSrfMeanNumer(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfMeanCurvatureSqr(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfMeanEvolute(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfIsoFocalSrf(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdSrfStruct *SymbSrfCurvatureUpperBound(CagdSrfStruct *Srf);
CagdSrfStruct *SymbSrfIsoDirNormalCurvatureBound(CagdSrfStruct *Srf,
						 CagdSrfDirType Dir);
CagdSrfStruct *SymbSrfDistCrvCrv(CagdCrvStruct *Crv1,
				 CagdCrvStruct *Crv2,
				 int DistType);
CagdPtStruct *SymbSrfDistFindPoints(CagdSrfStruct *Srf,
				    CagdRType Epsilon,
				    CagdBType SelfInter);
CagdPtStruct *SymbCrvCrvInter(CagdCrvStruct *Crv1,
			      CagdCrvStruct *Crv2,
			      CagdRType CCIEpsilon,
			      CagdBType SelfInter);
CagdSrfStruct *SymbConicDistCrvCrv(CagdCrvStruct *Crv1,
				   CagdCrvStruct *Crv2,
				   CagdRType Dist);
CagdCrvStruct *SymbTwoCrvsMorphing(CagdCrvStruct *Crv1,
				   CagdCrvStruct *Crv2,
				   CagdRType Blend);
CagdCrvStruct *SymbTwoCrvsMorphingCornerCut(CagdCrvStruct *Crv1,
					    CagdCrvStruct *Crv2,
					    CagdRType MinDist,
					    CagdBType SameLength,
					    CagdBType FilterTangencies);
CagdCrvStruct *SymbTwoCrvsMorphingMultiRes(CagdCrvStruct *Crv1,
					   CagdCrvStruct *Crv2,
					   CagdRType BlendStep);
CagdSrfStruct *SymbTwoSrfsMorphing(CagdSrfStruct *Srf1,
				   CagdSrfStruct *Srf2,
				   CagdRType Blend);
void SymbSrf2OptPolysCurvatureErrorPrep(CagdSrfStruct *Srf);
void SymbSrf2OptPolysIsoDirCurvatureErrorPrep(CagdSrfStruct *Srf);
CagdRType SymbSrf2OptPolysCurvatureError(CagdSrfStruct *Srf,
					 CagdSrfDirType Dir,
					 int SubdivLevel);
CagdRType SymbSrf2OptPolysBilinPolyError(CagdSrfStruct *Srf,
					 CagdSrfDirType Dir,
					 int SubdivLevel);
CagdPolygonStruct *SymbSrf2OptimalPolygons(CagdSrfStruct *Srf,
				CagdRType Tolerance,
				SymbPlSubdivStrategyType SubdivDirStrategy,
				SymbPlErrorFuncType SrfPolyApproxErr,
				CagdBType ComputeNormals,
				CagdBType FourPerFlat,
				CagdBType ComputeUV);
CagdSrfStruct *SymbSrfOffset(CagdSrfStruct *Srf, CagdRType OffsetDist);
CagdSrfStruct *SymbSrfSubdivOffset(CagdSrfStruct *Srf,
				   CagdRType OffsetDist,
				   CagdRType Tolerance);
CagdUVType *SymbUniformAprxPtOnSrfDistrib(CagdSrfStruct *Srf,
			CagdBType ParamUniform,
			int n,
			SymbUniformAprxSrfPtImportanceFuncType EvalImportance);
void SymbUniformAprxPtOnSrfPrepDistrib(CagdSrfStruct *Srf, int n);
CagdUVType *SymbUniformAprxPtOnSrfGetDistrib(CagdSrfStruct *Srf, int *n);
void SymbEvalSrfCurvPrep(CagdSrfStruct *Srf, CagdBType Init);
int SymbEvalSrfCurvature(CagdSrfStruct *Srf,
			 CagdRType U,
			 CagdRType V,
			 CagdBType DirInUV,
			 CagdRType *K1,
			 CagdRType *K2,
			 CagdVType D1,
			 CagdVType D2);
int SymbEvalSrfAsympDir(CagdSrfStruct *Srf,
			CagdRType U,
			CagdRType V,
			CagdBType DirInUV,
			CagdVType AsympDir1,
			CagdVType AsympDir2);

/******************************************************************************
* Routines to handle Bezier curves.					      *
******************************************************************************/
CagdCrvStruct *BzrCrvMult(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
void BzrCrvMultPtsVecs(CagdRType *Pts1,
		       int Order1,
		       CagdRType *Pts2,
		       int Order2,
		       CagdRType *ProdPts);
CagdCrvStruct *BzrCrvMultList(CagdCrvStruct *Crv1Lst, CagdCrvStruct *Crv2Lst);
CagdCrvStruct *BzrCrvDeriveRational(CagdCrvStruct *Crv);
CagdCrvStruct *BzrApproxBzrCrvAsCubics(CagdCrvStruct *Crv,
				       CagdRType Tol,
				       CagdRType MaxLen);
CagdCrvStruct *BzrApproxBzrCrvAsCubicPoly(CagdCrvStruct *Crv, CagdRType Tol2);
CagdCrvStruct *BzrApproxBzrCrvAsQuadratics(CagdCrvStruct *Crv,
					   CagdRType Tol,
					   CagdRType MaxLen);
CagdCrvStruct *BzrApproxBzrCrvAsQuadraticPoly(CagdCrvStruct *Crv,
					      CagdRType Tol2);
CagdCrvStruct *BzrComposeCrvCrv(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *BzrComposeSrfCrv(CagdSrfStruct *Srf, CagdCrvStruct *Crv);

/******************************************************************************
* Routines to handle Bezier surfaces.					      *
******************************************************************************/
CagdSrfStruct *BzrSrfMult(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *BzrSrfDeriveRational(CagdSrfStruct *Srf, CagdSrfDirType Dir);

/******************************************************************************
* Routines to handle Bspline curves.					      *
******************************************************************************/
int BspMultInterpFlag(int BspMultUsingInter);
CagdCrvStruct *BspCrvMult(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *BspCrvDeriveRational(CagdCrvStruct *Crv);
void SymbBspBasisInnerProdPrep(CagdRType *KV, int Len, int Order1, int Order2);
void SymbBspBasisInnerProdPrep2(CagdRType *KV1,
				CagdRType *KV2,
				int Len1,
				int Len2,
				int Order1,
				int Order2);
CagdRType SymbBspBasisInnerProd(int Index1, int Index2);
CagdRType **SymbBspBasisInnerProdMat(CagdRType *KV,
				     int Len,
				     int Order1,
				     int Order2);
CagdRType SymbBspBasisInnerProd2(CagdRType *KV,
				 int Len,
				 int Order1,
				 int Order2,
				 int Index1,
				 int Index2);

/******************************************************************************
* Routines to handle layout (prisa) of surfaces.			      *
******************************************************************************/
CagdSrfStruct *SymbAllPrisaSrfs(CagdSrfStruct *Srfs,
				int SamplesPerCurve,
				CagdRType Epsilon,
				CagdSrfDirType Dir,
				CagdVType Space);
CagdSrfStruct *SymbPiecewiseRuledSrfApprox(CagdSrfStruct *Srf,
					   CagdBType ConsistentDir,
					   CagdRType Epsilon,
					   CagdSrfDirType Dir);
CagdSrfStruct *SymbPrisaRuledSrf(CagdSrfStruct *Srf,
				 int SamplesPerCurve,
				 CagdRType Space,
				 CagdVType Offset);
CagdCrvStruct *SymbPrisaGetCrossSections(CagdSrfStruct *RSrfs,
					 CagdSrfDirType Dir,
					 CagdVType Space);
CagdCrvStruct *SymbPrisaGetOneCrossSection(CagdSrfStruct *RSrf,
					   CagdSrfDirType Dir,
					   CagdBType Starting,
					   CagdBType Ending);

/******************************************************************************
* Routines to handle Multi Resolution Bspline curves.                         *
******************************************************************************/
int SymbCrvMultiResKVBuild(CagdCrvStruct *Crv,
			   int Discont,
			   CagdRType ***KVList,
			   int **KVListSizes,
			   int *KVListSize);
SymbMultiResCrvStruct *SymbCrvMultiResDecomp(CagdCrvStruct *Crv, int Discont);
SymbMultiResCrvStruct *SymbCrvMultiResDecomp2(CagdCrvStruct *Crv,
					      int Discont,
					      int SameSpace);
CagdCrvStruct *SymbCrvMultiResCompos(SymbMultiResCrvStruct *MRCrv);
CagdCrvStruct *SymbCrvMultiResComposAtT(SymbMultiResCrvStruct *MRCrv,
					CagdRType T);
void SymbCrvMultiResEdit(SymbMultiResCrvStruct *MRCrv,
			 CagdRType t,
			 CagdVType TransDir,
			 CagdRType Level,
			 CagdRType FracLevel);
CagdRType *SymbCrvMultiResRefineLevel(SymbMultiResCrvStruct *MRCrv,
				      CagdRType T,
				      int SpanDiscont);
CagdCrvStruct *SymbCrvMultiResBWavelet(CagdRType *KV,
				       int Order,
				       int Len,
				       int KnotIndex);
void SymbCrvMultiResFree(SymbMultiResCrvStruct *MRCrv);
SymbMultiResCrvStruct *SymbCrvMultiResNew(int Levels, CagdBType Periodic);
SymbMultiResCrvStruct *SymbCrvMultiResCopy(SymbMultiResCrvStruct *MRCrvOrig);

/******************************************************************************
* Routines to compute the convex hull of piecewise freeforms		      *
******************************************************************************/
CagdCrvStruct *SymbCrvCnvxHull(CagdCrvStruct *Crv, CagdRType FineNess);
struct IPPolygonStruct *SymbCrvDiameter(CagdCrvStruct *Crv, CagdRType FineNess);
CagdRType *SymbCrvDiameterMinMax(CagdCrvStruct *Crv,
				 struct IPPolygonStruct *Cntrs,
				 int Min);
CagdPtStruct *SymbCrvPtTangents(CagdCrvStruct *Crv,
				CagdPType Pt,
				CagdRType Tolerance);
CagdPtStruct *SymbTangentToCrvAtTwoPts(CagdCrvStruct *Crv,
				       CagdRType FineNess);
CagdPtStruct *SymbCircTanTo2Crvs(CagdCrvStruct *Crv1,
				 CagdCrvStruct *Crv2,
				 CagdRType Radius,
				 CagdRType Tol);

struct IPObjectStruct *SymbSrfCnvxHull(CagdSrfStruct *Srf,
				       CagdRType Fineness);
struct IPObjectStruct *SymbTwoSrfTangencies(CagdSrfStruct *Srf1,
					    CagdSrfStruct *Srf2,
					    CagdRType Fineness);
struct IPObjectStruct *SymbThreeSrfTangencies(CagdSrfStruct *Srf1,
					      CagdSrfStruct *Srf2,
					      CagdSrfStruct *Srf3,
					      CagdRType Fineness);

/******************************************************************************
* Routines to compute the bisector of piecewise freeforms		      *
******************************************************************************/
CagdCrvStruct *SymbCrvBisectors(CagdCrvStruct *Crv,
				int BisectFunc,
				CagdRType Tolerance,
				CagdBType NumerImprove,
				CagdBType SameNormal,
				CagdBType SupportPrms);
CagdSrfStruct *SymbCrvBisectorsSrf(CagdCrvStruct *Crv, int BisectFunc);
CagdSrfStruct *SymbCrvBisectorsSrf2(CagdCrvStruct *Crv);
CagdSrfStruct *SymbCrvBisectorsSrf3(CagdCrvStruct *Crv);
CagdSrfStruct *SymbCrvCrvBisectorSrf3D(CagdCrvStruct *Crv1,
				       CagdCrvStruct *Crv2,
				       CagdRType Alpha);
CagdSrfStruct *SymbCrvBisectorsSrf3D(CagdSrfStruct *Srf1,
				     CagdSrfStruct *Srf2,
				     CagdSrfStruct *DSrf1,
				     CagdSrfStruct *DSrf2,
				     CagdRType Alpha);
CagdCrvStruct *SymbCrvPtBisectorCrv2D(CagdCrvStruct *Crv,
				      CagdPType Pt,
				      CagdRType Alpha);
CagdSrfStruct *SymbCrvPtBisectorSrf3D(CagdCrvStruct *Crv,
				      CagdPType Pt,
				      CagdRType RulingScale);
CagdSrfStruct *SymbSrfPtBisectorSrf3D(CagdSrfStruct *Srf, CagdPType Pt);

CagdCrvStruct *SymbPtCrvBisectOnSphere(CagdPType Pt, CagdCrvStruct *Crv);
CagdCrvStruct *SymbPtCrvBisectOnSphere2(CagdPType Pt,
					CagdCrvStruct *Crv,
					CagdRType Tolerance);
CagdSrfStruct *SymbCrvCrvBisectOnSphere(CagdCrvStruct *Crv1,
					CagdCrvStruct *Crv2);
CagdCrvStruct *SymbCrvCrvBisectOnSphere2(CagdCrvStruct *Crv1,
					 CagdCrvStruct *Crv2,
					 CagdRType Tolerance);
CagdSrfStruct *SymbCrvCrvBisectOnSphere3(CagdCrvStruct *Crv1,
					 CagdCrvStruct *Crv2);
CagdSrfStruct *SymbPlanePointBisect(CagdPType Pt, CagdRType Size);
CagdSrfStruct *SymbCylinPointBisect(CagdPType CylPt,
				    CagdVType CylDir,
				    CagdRType CylRad,
				    CagdPType Pt,
				    CagdRType Size);
CagdSrfStruct *SymbConePointBisect(CagdPType ConeApex,
				   CagdVType ConeDir,
				   CagdRType ConeAngle,
				   CagdPType Pt,
				   CagdRType Size);
CagdSrfStruct *SymbSpherePointBisect(CagdPType SprCntr,
				     CagdRType SprRad,
				     CagdPType Pt);
CagdSrfStruct *SymbTorusPointBisect(CagdPType TrsCntr,
				    CagdVType TrsDir,
				    CagdRType TrsMajorRad,
				    CagdRType TrsMinorRad,
				    CagdPType Pt);
CagdSrfStruct *SymbPlaneLineBisect(CagdVType LineDir, CagdRType Size);
CagdSrfStruct *SymbConeLineBisect(CagdVType ConeDir,
				  CagdRType ConeAngle,
				  CagdVType LineDir,
				  CagdRType Size);
CagdSrfStruct *SymbSphereLineBisect(CagdPType SprCntr,
				    CagdRType SprRad,
				    CagdRType Size);
CagdSrfStruct *SymbSpherePlaneBisect(CagdPType SprCntr,
				     CagdRType SprRad,
				     CagdRType Size);
CagdSrfStruct *SymbCylinPlaneBisect(CagdPType CylPt,
				    CagdVType CylDir,
				    CagdRType CylRad,
				    CagdRType Size);
CagdSrfStruct *SymbConePlaneBisect(CagdPType ConeApex,
				   CagdVType ConeDir,
				   CagdRType ConeAngle,
				   CagdRType Size);
CagdSrfStruct *SymbCylinSphereBisect(CagdPType CylPt,
				     CagdVType CylDir,
				     CagdRType CylRad,
				     CagdPType SprCntr,
				     CagdRType SprRad,
				     CagdRType Size);
CagdSrfStruct *SymbSphereSphereBisect(CagdPType SprCntr1,
				      CagdRType SprRad1,
				      CagdPType SprCntr2,
				      CagdRType SprRad2);
CagdSrfStruct *SymbConeSphereBisect(CagdPType ConeApex,
				    CagdVType ConeDir,
				    CagdRType ConeAngle,
				    CagdPType SprCntr,
				    CagdRType SprRad,
				    CagdRType Size);
CagdSrfStruct *SymbConeConeBisect(CagdVType Cone1Dir,
				  CagdRType Cone1Angle,
				  CagdVType Cone2Dir,
				  CagdRType Cone2Angle,
				  CagdRType Size);
CagdSrfStruct *SymbTorusSphereBisect(CagdPType TrsCntr,
				     CagdVType TrsDir,
				     CagdRType TrsMajorRad,
				     CagdRType TrsMinorRad,
				     CagdPType SprCntr,
				     CagdRType SprRad);
CagdSrfStruct *SymbCylinCylinBisect(CagdVType Cyl1Pos,
				    CagdVType Cyl1Dir,
				    CagdRType Cyl1Rad,
				    CagdVType Cyl2Pos,
				    CagdVType Cyl2Dir,
				    CagdRType Cyl2Rad);
CagdSrfStruct *SymbConeConeBisect2(CagdVType Cone1Pos,
				   CagdVType Cone1Dir,
				   CagdRType Cone1Angle,
				   CagdVType Cone2Pos,
				   CagdVType Cone2Dir,
				   CagdRType Cone2Angle);
CagdSrfStruct *SymbConeCylinBisect(CagdVType Cone1Pos,
				   CagdVType Cone1Dir,
				   CagdRType Cone1Angle,
				   CagdVType Cyl2Pos,
				   CagdVType Cyl2Dir,
				   CagdRType Cyl2Rad);

/******************************************************************************
* Routines to handle Bspline surfaces.					      *
******************************************************************************/
CagdSrfStruct *BspSrfMult(CagdSrfStruct *Srf1, CagdSrfStruct *Srf2);
CagdSrfStruct *BspSrfDeriveRational(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdSrfStruct *BzrSrfFactorBilinear(CagdSrfStruct *Srf, CagdRType *A);
CagdSrfStruct *BspSrfFactorUMinusV(CagdSrfStruct *Srf);

/******************************************************************************
* Routines to handle Bezier surfaces.					      *
******************************************************************************/
CagdSrfStruct *BzrSrfFactorUMinusV(CagdSrfStruct *Srf);

/******************************************************************************
* Routines to handle Normal Cones.					      *
******************************************************************************/
SymbNormalConeStruct *SymbTangentConeForCrv(CagdCrvStruct *Crv, int Planar);
SymbNormalConeStruct *SymbNormalConeForSrf(CagdSrfStruct *Srf);
CagdBType SymbNormalConeOverlap(SymbNormalConeStruct *NormalCone1,
				SymbNormalConeStruct *NormalCone2);

/******************************************************************************
* Routines to handle Reflection Lines.					      *
******************************************************************************/
void SymbRflctLnPrepSrf(CagdSrfStruct *Srf,
			CagdVType ViewDir,
			CagdVType LnDir,
			char *AttribName);
CagdSrfStruct *SymbRflctLnGen(CagdSrfStruct *Srf,
			      CagdVType ViewDir,
			      CagdPType LnPt,
			      CagdVType LnDir,
			      char *AttribName);
void SymbRflctLnFree(CagdSrfStruct *Srf, char *AttribName);
void SymbRflctCircPrepSrf(CagdSrfStruct *Srf,
			  CagdVType ViewDir,
			  CagdPType SprCntr,
			  char *AttribName);
CagdSrfStruct *SymbRflctCircGen(CagdSrfStruct *Srf,
				CagdVType ViewDir,
				CagdPType SprCntr,
				CagdRType ConeAngle,
				char *AttribName);
void SymbRflctCircFree(CagdSrfStruct *Srf, char *AttribName);
void SymbHighlightLnPrepSrf(CagdSrfStruct *Srf,
			    CagdVType LnDir,
			    char *AttribName);
CagdSrfStruct *SymbHighlightLnGen(CagdSrfStruct *Srf,
				  CagdPType LnPt,
				  CagdVType LnDir,
				  char *AttribName);
void SymbHighlightLnFree(CagdSrfStruct *Srf, char *AttribName);

/******************************************************************************
* Routines to handle K-orthotomics, silhouettes, and isoclines.		      *
******************************************************************************/
CagdCrvStruct *SymbCrvOrthotomic(CagdCrvStruct *Crv,
				 CagdPType P,
				 CagdRType K);
CagdSrfStruct *SymbSrfOrthotomic(CagdSrfStruct *Srf,
				 CagdPType P,
				 CagdRType K);
struct IPPolygonStruct *SymbSrfSilhouette(CagdSrfStruct *Srf,
					  CagdVType VDir,
					  CagdRType Tolerance,
					  CagdBType Euclidean);
struct IPPolygonStruct *SymbSrfPolarSilhouette(CagdSrfStruct *Srf,
					       CagdVType VDir,
					       CagdRType Tolerance,
					       CagdBType Euclidean);
struct IPPolygonStruct *SymbSrfIsocline(CagdSrfStruct *Srf,
					CagdVType VDir,
					CagdRType Theta,
					CagdRType Tolerance,
					CagdBType Euclidean);

/******************************************************************************
* Routines to construct surfaces algebraically.				      *
******************************************************************************/
CagdSrfStruct *SymbAlgebraicSumSrf(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdSrfStruct *SymbAlgebraicProdSrf(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdSrfStruct *SymbSwungAlgSumSrf(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);

/******************************************************************************
* Routines to construct dual curves and surfaces.			      *
******************************************************************************/
CagdCrvStruct *SymbCrvDual(CagdCrvStruct *Crv);
CagdSrfStruct *SymbSrfDual(CagdSrfStruct *Srf);

/******************************************************************************
* Routines to handle surface surface intersections.			      *
******************************************************************************/
CagdSrfStruct *SymbRuledRuledZeroSetFunc(CagdCrvStruct *C1,
					 CagdCrvStruct *C2,
					 CagdCrvStruct *D1,
					 CagdCrvStruct *D2);
CagdCrvStruct *SymbRuledRuledIntersection(CagdCrvStruct *C1,
					  CagdCrvStruct *C2,
					  CagdCrvStruct *D1,
					  CagdCrvStruct *D2,
					  CagdRType Tolerance,
					  CagdCrvStruct **PCrvs1,
					  CagdCrvStruct **PCrvs2);
CagdCrvStruct *SymbRingRingIntersection(CagdCrvStruct *C1,
					CagdCrvStruct *r1,
					CagdCrvStruct *C2,
					CagdCrvStruct *r2,
					CagdRType Tolerance,
					CagdCrvStruct **PCrvs1,
					CagdCrvStruct **PCrvs2);
CagdSrfStruct *SymbRingRingZeroSetFunc(CagdCrvStruct *C1,
				       CagdCrvStruct *r1,
				       CagdCrvStruct *C2,
				       CagdCrvStruct *r2);

/******************************************************************************
* Routines to handle knot removal of Bspline curves.             	      *
******************************************************************************/
CagdCrvStruct *SymbRmKntBspCrvRemoveKnots(CagdCrvStruct *OriginalCrv, 
					  int RemoveKnotsNumber, 
					  int MaxInIteration, 
					  int SamplesNum);
CagdCrvStruct *SymbRmKntBspCrvRemoveKnotsError(CagdCrvStruct *OriginalCrv, 
					       CagdRType MaxError, 
					       CagdRType MaxErrorInIteration, 
					       int SamplesNum);
CagdCrvStruct *SymbRmKntBspCrvCleanKnots(CagdCrvStruct *OriginalCrv, 
					 int SamplesNum);

/******************************************************************************
* Routines to handle reverse engineering and surface recognition.	      *
******************************************************************************/
CagdBType SymbIsConstCrv(CagdCrvStruct *Crv,
			 CagdCtlPtStruct **ConstVal,
			 CagdRType Eps);
CagdBType SymbIsCircularCrv(CagdCrvStruct *Crv,
			    CagdPType Center,
			    CagdRType *Radius,
			    CagdRType Eps);
CagdBType SymbIsLineCrv(CagdCrvStruct *Crv,
			CagdPType LnPos,
			CagdVType LnDir,
			CagdRType Eps);
CagdBType SymbIsConstSrf(CagdSrfStruct *Srf,
			 CagdCtlPtStruct **ConstVal,
			 CagdRType Eps);
CagdBType SymbIsSphericalSrf(CagdSrfStruct *Srf,
			     CagdPType Center,
			     CagdRType *Radius,
			     CagdRType Eps);
CagdBType SymbIsExtrusionSrf(CagdSrfStruct *Srf,
			     CagdCrvStruct **Crv,
			     CagdVType ExtDir,
			     CagdRType Eps);
CagdBType SymbIsDevelopSrf(CagdSrfStruct *Srf, CagdRType Eps);
CagdBType SymbIsRuledSrf(CagdSrfStruct *Srf,
			 CagdCrvStruct **Crv1,
			 CagdCrvStruct **Crv2,
			 CagdRType Eps);
CagdBType SymbIsSrfOfRevSrf(CagdSrfStruct *Srf,
			    CagdCrvStruct **CrossSec,
			    CagdPType AxisPos,
			    CagdVType AxisDir,
			    CagdRType Eps);
CagdBType SymbIsPlanarSrf(CagdSrfStruct *Srf,
			  PlaneType Plane,
			  CagdRType Eps);

/******************************************************************************
* Routines to handle injectivity testing of Bspline surfaces.         	      *
******************************************************************************/
CagdBType SymbCubicBspInjective(CagdRType x[4][4], CagdRType y[4][4]);

/******************************************************************************
* Routines to handle blending of Bspline surfaces.	         	      *
******************************************************************************/
CagdCrvStruct *SymbClipCrvToSrfDomain(CagdSrfStruct *Srf,
				      CagdCrvStruct *UVCrv);
CagdSrfStruct *SymbShapeBlendOnSrf(CagdSrfStruct *Srf,
				   CagdCrvStruct *UVCrv,
				   CagdCrvStruct *CrossSecShape,
				   CagdRType TanScale,
				   CagdRType Width,
				   CagdCrvStruct *WidthField,
				   CagdRType Height,
				   CagdCrvStruct *HeightField);
CagdSrfStruct *SymbShapeBlendSrf(CagdCrvStruct *Pos1Crv,
				 CagdCrvStruct *Pos2Crv,
				 CagdCrvStruct *Dir1Crv,
				 CagdCrvStruct *Dir2Crv,
				 CagdCrvStruct *CrossSecShape,
				 CagdCrvStruct *Normal);

/******************************************************************************
* Routines to handle bi arc approximations of curves.	         	      *
******************************************************************************/
SymbArcStruct *SymbCrvBiArcApprox(CagdCrvStruct *Crv,
				  CagdRType Tolerance,
				  CagdRType MaxAngle);
CagdCrvStruct *SymbArcs2Crvs(SymbArcStruct *Arcs);

/******************************************************************************
* Routines to compare the similarity of freeforms.	         	      *
******************************************************************************/
SymbCrvRelType SymbCrvsCompare(CagdCrvStruct *Crv1,
			       CagdCrvStruct *Crv2,
			       CagdRType Eps,
			       CagdRType *StartOvrlpPrmCrv1,
			       CagdRType *EndOvrlpPrmCrv1,
			       CagdRType *StartOvrlpPrmCrv2,
			       CagdRType *EndOvrlpPrmCrv2);
CagdCrvStruct *SymbCanonicBzrCrv(CagdCrvStruct *Crv, CagdRType Eps);
CagdCrvStruct *SymbBzrDegReduce(CagdCrvStruct *Crv, CagdRType Eps);

/******************************************************************************
* Routines to handle lower envelopes of curve arrangments.	              *
******************************************************************************/
CagdCrvStruct *SymbCrvsLowerEnvelop(CagdCrvStruct *Crvs,
				    CagdRType *Pt,
				    CagdRType Eps);
CagdCrvStruct *SymbSplitCrvsAtExtremums(CagdCrvStruct *Crvs,
					int Axis,
					CagdPType Pt,
					CagdRType Eps);

/******************************************************************************
* Error handling.							      *
******************************************************************************/
char *SymbDescribeError(SymbFatalErrorType ErrorNum);
void SymbFatalError(SymbFatalErrorType ErrID);


#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#endif /* SYMB_LIB_H */
