/*****************************************************************************
*   Main module of "Irit" - the 3d (not only polygonal) solid modeller.      *
******************************************************************************
* (C) Gershon Elber, Technion, Israel Institute of Technology                *
******************************************************************************
* Usage:								     *
*   Irit [-t] [-g] [-i] [-z] [file[.irt]]				     *
*									     *
* Written by:  Gershon Elber				Ver 3.0, Apr. 1990   *
*****************************************************************************/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "program.h"
#include "misc_lib.h"
#include "ctrl-brk.h"
#include "dosintr.h"
#include "inptprsg.h"
#include "iritprsr.h"
#include "objects.h"
#include "geom_lib.h"
#include "iritgrap.h"
#include "bool_lib.h"
#include "trim_lib.h"
#include "triv_lib.h"
#include "symb_lib.h"
#include "user_lib.h"
#ifdef USE_VARARGS
#include <varargs.h>
#else
#include <stdarg.h>
#endif /* USE_VARARGS */
#ifdef __WINNT__
#include <direct.h>
#endif /* __WINNT__ */

#ifdef DJGCC
#include "intr_lib.h"
#include "intr_gr.h"
#endif /* DJGCC */
#ifdef __UNIX__
#if defined (__FreeBSD__)
#include <term.h>
STATIC_DATA struct termios GlblOrigTermio;
#else
#include <termio.h>
STATIC_DATA struct termio GlblOrigTermio;
#endif /* !__FreeBSD__ */
#endif /* __UNIX__ */

#ifdef NO_CONCAT_STR
STATIC_DATA char
    *VersionStr = "Irit		Version 9.5, Gershon Elber,\n\
	(C) Copyright 1989-2005 Gershon Elber, Non commercial use only.";
#else
STATIC_DATA char
    *VersionStr = "Irit	        " IRIT_VERSION
	",	Gershon Elber,	" __DATE__ ",   " __TIME__ "\n"
	IRIT_COPYRIGHT ", Non commercial use only.";
#endif /* NO_CONCAT_STR */

STATIC_DATA char
     *CtrlStr = "Irit [-t] [-g] [-q] [-i] [-z] [file[.irt]]";

char *GlblHelpFileName = "irit.hlp";

STATIC_DATA char
    *GlblPrgmHeader = "                Irit - the not only polygonal solid modeller";
STATIC_DATA char
    *GlblCopyRight  =
#ifdef DJGCC
     IRIT_COPYRIGHT ",  DJGCC " IRIT_VERSION ",   " __DATE__;
#else
#ifdef OS2GCC
     IRIT_COPYRIGHT ",  OS2   " IRIT_VERSION ",   " __DATE__;
#else
#ifdef AMIGA
     IRIT_COPYRIGHT ", Amiga  " IRIT_VERSION ",   " __DATE__;
#else
#ifdef __WINNT__
     IRIT_COPYRIGHT ", WinNT,95/8,2000/XP " IRIT_VERSION ", " __DATE__;
#else
#ifdef NO_CONCAT_STR
     "(C) Copyright 1989-2005 Gershon Elber,  Unix  Version 9.5";
#else
     IRIT_COPYRIGHT ",  Unix  " IRIT_VERSION;
#endif /* NO_CONCAT_STR */
#endif /* __WINNT__ */
#endif /* AMIGA */
#endif /* OS2GCC */
#endif /* DJGCC */
STATIC_DATA char
    *GlblAuthorName = "                         Written by Gershon Elber";

STATIC_DATA int
    GlblKeepClientsCursorEvents = FALSE,
    GlblQuietMode = FALSE;

STATIC_DATA char GlblCrntWorkingDir[LINE_LEN];/* Init CWD to recover on exit.*/

STATIC_DATA IPObjectStruct
    *GlblLastClientCursorEvent = NULL;        /* Cursor events from clients. */

GLOBAL_DATA IPObjectStruct
    *GlblObjList = NULL;		   /* All objects defined on system. */

GLOBAL_DATA FILE
    *GlblLogFile = NULL;

GLOBAL_DATA jmp_buf GlblLongJumpBuffer;		  /* Used in error recovery. */

GLOBAL_DATA int
#ifdef DJGCC					   /* Defaults for intr_lib. */
    GlblWindowFrameWidth = 8,
    GlblViewFrameColor   = INTR_COLOR_RED,
    GlblViewBackColor    = INTR_COLOR_BLACK,
    GlblTransFrameColor  = INTR_COLOR_GREEN,
    GlblTransBackColor   = INTR_COLOR_BLACK,
    GlblStatusFrameColor = INTR_COLOR_MAGENTA,
    GlblStatusBackColor  = INTR_COLOR_BLACK,
    GlblInputFrameColor  = INTR_COLOR_YELLOW,
    GlblInputBackColor   = INTR_COLOR_BLACK,
    GlblDrawHeader       = FALSE,
    GlblSmoothTextScroll = TRUE,
    GlblIntrSaveMethod   = INTR_SAVE_DISK,
    GlblMouseSensitivity = 10,       /* Sensitivity control of mouse device. */
    GlblJoystickExists   = FALSE,
#endif /* DJGCC */
    GlblCurrentDisplay = -1,       /* Handlers to streams to display device. */
    GlblFlatLoadMode = FALSE,
    GlblHandleDependencies = 0,
    GlblStdinInteractive = FALSE,
    GlblInterpProd	 = TRUE,
    GlblDoGraphics       = TRUE,/* Control if running in graphics/text mode. */
    GlblGUIMode          = FALSE,		     /* Running under a GUI. */
    GlblFatalError       = FALSE, /* True if disaster in system - must quit! */
    GlblPrintLogFile     = FALSE,    /* If TRUE everything goes to log file. */
    GlblPointLenAux      = IG_POINT_DEFAULT_LENGTH;

GLOBAL_DATA RealType
    GlblPointLen = 0.02;		       /* Scaler for point if drawn. */

GLOBAL_DATA char
#ifdef DJGCC					   /* Defaults for intr_lib. */
    *GlblViewWndwPos   = "0.02, 0.02, 0.72, 0.66",
    *GlblTransWndwPos  = "0.75, 0.02, 0.98, 0.66",
    *GlblStatusWndwPos = "0.75, 0.02, 0.98, 0.66",
    *GlblInputWndwPos  = "0.04, 0.7,  0.98, 0.98",
    *GlblIntrSaveDisk  = "c:\\",
#endif /* DJGCC */
    *GlblLineEditControl = " 8,  4,  2,  6,  1,  5, 16, 14, 11,  9, 10",
    *GlblStartFileName = "",
    *GlblLogFileName = "",
#ifdef IRIT_DOUBLE
    *GlblFloatFormat = "%-16.14lg";
#else
    *GlblFloatFormat = "%-10.8g";
#endif /* IRIT_DOUBLE */

STATIC_DATA IritConfigStruct SetUp[] =
{
#ifdef DJGCC
  { "Joystick",		"",   (VoidPtr) &GlblJoystickExists, IC_BOOLEAN_TYPE },
  { "MouseSensitivity",	"",   (VoidPtr) &GlblMouseSensitivity,IC_INTEGER_TYPE },
  { "WndwWidth",	"",   (VoidPtr) &GlblWindowFrameWidth,IC_INTEGER_TYPE },
  { "WndwHeader",	"",   (VoidPtr) &GlblDrawHeader,     IC_BOOLEAN_TYPE },
  { "WndwSmthTxtScrl",	"",   (VoidPtr) &GlblSmoothTextScroll,IC_BOOLEAN_TYPE },
  { "WndwViewClr",	"",   (VoidPtr) &GlblViewFrameColor, IC_INTEGER_TYPE },
  { "WndwTransClr",	"",   (VoidPtr) &GlblTransFrameColor,IC_INTEGER_TYPE },
  { "WndwInputClr",	"",   (VoidPtr) &GlblInputFrameColor,IC_INTEGER_TYPE },
  { "WndwViewPos",	"",   (VoidPtr) &GlblViewWndwPos,    IC_STRING_TYPE },
  { "WndwTransPos",	"",   (VoidPtr) &GlblStatusWndwPos,  IC_STRING_TYPE },
  { "WndwInputPos",	"",   (VoidPtr) &GlblInputWndwPos,   IC_STRING_TYPE },
  { "WndwBackSave",	"",   (VoidPtr) &GlblIntrSaveMethod, IC_INTEGER_TYPE },
  { "WndwBackSavePath",	"",   (VoidPtr) &GlblIntrSaveDisk,   IC_STRING_TYPE },
#endif /* DJGCC */
  { "DoGraphics",	"-t", (VoidPtr) &GlblDoGraphics,     IC_BOOLEAN_TYPE },
  { "InterpProd",	"",   (VoidPtr) &GlblInterpProd,     IC_BOOLEAN_TYPE },
  { "PointLength",	"",   (VoidPtr) &GlblPointLenAux,    IC_INTEGER_TYPE },
  { "LineEdit",         "",   (VoidPtr) &GlblLineEditControl,IC_STRING_TYPE },
  { "StartFile",	"",   (VoidPtr) &GlblStartFileName,  IC_STRING_TYPE },
  { "LogFile",		"",   (VoidPtr) &GlblLogFileName,    IC_STRING_TYPE },
  { "FloatFrmt",	"",   (VoidPtr) &GlblFloatFormat,    IC_STRING_TYPE },
};
#define NUM_SET_UP	(sizeof(SetUp) / sizeof(IritConfigStruct))

static void Interact(void);
static void ValidateVariables(void);
static void PrintInptPrsrError(void);
static void ConvertData2Irt(IPObjectStruct *PObj);
static void IritBoolFatalError(BoolFatalErrorType ErrID);

#ifdef DEBUG
static void AllFatalErrorTrap(void);
#endif /* DEBUG */

/*****************************************************************************
* DESCRIPTION:                                                               M
* Main module of IRIT - Read command line and do what is needed...	     M
*                                                                            *
* PARAMETERS:                                                                M
*   argc, argv:  Command line.                                               M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   main                                                                     M
*****************************************************************************/
void main(int argc, char **argv)
{
    char *FullPathStartFileName;

#ifdef DEBUG_IRIT_MALLOC
    IritInitTestDynMemory();
#endif /* DEBUG_IRIT_MALLOC */

    IritCPUTime(TRUE);					 /* Reset the clock. */

    IritConfig("irit", SetUp, NUM_SET_UP);   /* Read config. file if exists. */

    while (argc >= 2) {
    	if (strncmp(argv[1], "-z", 2) == 0) {
	    fprintf(stderr, IRIT_EXP_STR("\n%s\n\nUsage: %s\n"),
		    VersionStr, CtrlStr);
	    IritConfigPrint(SetUp, NUM_SET_UP);
	    exit(0);
    	}
	else if (strncmp(argv[1], "-i", 2) == 0) {
	    GlblStdinInteractive = TRUE;
	}
	else if (strncmp(argv[1], "-t", 2) == 0) {
	    GlblDoGraphics = FALSE;
	}
	else if (strncmp(argv[1], "-g", 2) == 0) {
	    GlblGUIMode = TRUE;
	}
	else if (strncmp(argv[1], "-q", 2) == 0) {
	    GlblQuietMode = TRUE;
	}
#	ifdef DEBUG_IRIT_MALLOC
	    else if (strncmp(argv[1], "-m", 2) == 0) {
	        IritInitTestDynMemory2(atoi(argv[2]),
				       atoi(argv[3]),
				       atoi(argv[4]));
		argv += 3;
		argc -= 3;
	    }
#	endif /* DEBUG_IRIT_MALLOC */
	else {
	    break;
	}
    	argv++;
	argc--;
    }

    getcwd(GlblCrntWorkingDir, LINE_LEN - 1);

    SetUpCtrlBrk();	     /* Set up control break trap routine (int 1bh). */
    signal(SIGFPE, DefaultFPEHandler);	 /* Will trap floating point errors. */

    BspMultInterpFlag(GlblInterpProd);
    IPSetFloatFormat(GlblFloatFormat);
    IPSetFlattenObjects(FALSE);
    IPSetPolyListCirc(TRUE);

    /* Print some copyright messages: */
    if (GlblQuietMode) {
	InptPrsrSetEchoSource(FALSE);
    }
    else {
	IRIT_WNDW_PUT_STR(GlblPrgmHeader);
	IRIT_WNDW_PUT_STR(GlblCopyRight);
	IRIT_WNDW_PUT_STR(GlblAuthorName);
    }

    SetUpPredefObjects();		  /* Prepare the predefined objects. */

    BoolSetHandleCoplanarPoly(TRUE);		/* Handle coplanar polygons. */
    BoolSetFatalErrorFunc(IritBoolFatalError);

    /* Execute the file specified in the command line if was one: */
    if (argc == 2)
	InptPrsrFileInclude(argv[1]);

    /* Execute the start up file first by inserting it to the include stack. */
    if ((int) strlen(GlblStartFileName) > 0 &&
	(FullPathStartFileName = searchpath(GlblStartFileName)) != NULL)
	InptPrsrFileInclude(FullPathStartFileName);

#   ifdef __UNIX__
    {
#if defined (__FreeBSD__)
	struct termios Termio;
#else
	struct termio Termio;
#endif /* !__FreeBSD__ */

	/* We read stdin on a char by char basis with a 0.1 second timer so  */
	/* we could simultaneously handle other requests (for example, from  */
	/* display devices).						     */
#if defined (__FreeBSD__)
	tcgetattr(0, &GlblOrigTermio);
	tcgetattr(0, &Termio);
#else
	ioctl(0, TCGETA, &GlblOrigTermio);
	ioctl(0, TCGETA, &Termio);
#endif /* !__FreeBSD__ */
	Termio.c_lflag &= ~ICANON;	     /* Clear the canonical editing. */
	Termio.c_cc[VEOF] = 0;    /* MIN = 0, no minimal length to wait for. */
	Termio.c_cc[VEOL] = 1;    /* TIME - 1 tenth of a second as time out. */
#       ifdef VSUSP
	    Termio.c_cc[VSUSP] = 0;		   /* Disable ^Z suspension. */
#       endif /* VSUSP */
#if defined (__FreeBSD__)
	tcsetattr(0, TCSANOW, &Termio);
#else
	ioctl(0, TCSETA, &Termio);
#endif /* !__FreeBSD__ */
    }
#   endif /* __UNIX__ */

    IPSocSrvrInit();            /* Initialize the listen socket for clients. */

    Interact();				      /* Go and do some real work... */

    IritExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Interact - the main read/eval/print routine. This routine reads data from  *
* standart input and execute it "forever" (using Input Parser).		     *
*   Note exit from this program is controled by input parser itself.	     *
*                                                                            *
* PARAMETERS:                                                                *
*   None                                                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void Interact(void)
{
    /* setjmp return 0 on first install time. Will return 1 if error is      */
    /* recoverable, 2 if cannt continue - must quit the program now.	     */
    switch (setjmp(GlblLongJumpBuffer)) {         /* Used in error recovery. */
	case 1:
	    ValidateVariables();
	case 0:
	    while (TRUE) {
		if (!InptPrsrInputParser(NULL))	 /* Print the error message. */
		    PrintInptPrsrError();
		ValidateVariables();
	    }
	case 2:
	    IRIT_WNDW_PUT_STR("Press return to die...");
	    getchar();
	    break;
	case 3:
	    /* Optional - trapping stdin request. */
	    break;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Exec a line by injecting it into the input stream.                       M
*                                                                            *
* PARAMETERS:                                                                M
*   Line:     To execute for processing in input stream.                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   ExecOneLine                                                              M
*****************************************************************************/
void ExecOneLine(char *Line)
{
    InptPrsrQueueInputLine(Line);
    if (strchr(Line, ';') == NULL)
        InptPrsrQueueInputLine(";");

    /* Evaluate this expression. */
    if (!InptPrsrInputParser(NULL))
        PrintInptPrsrError();			 /* Print the error message. */
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Validates all the variables in the global list, scream and remove        *
* undefined variables.                                                       *
*                                                                            *
* PARAMETERS:                                                                *
*   None                                                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void ValidateVariables(void)
{
    IPObjectStruct *PObj, *PTmp;

    while (GlblObjList != NULL && IP_IS_UNDEF_OBJ(GlblObjList)) {
	IRIT_WNDW_FPRINTF2("Error: undefined object \"%s\" has been removed from global data base\n",
			  GlblObjList -> ObjName);

	PObj = GlblObjList -> Pnext;
	IPFreeObject(GlblObjList);
	GlblObjList = PObj;
    }

    if (GlblObjList) {
	for (PObj = GlblObjList; PObj -> Pnext != NULL; PObj = PObj -> Pnext) {
	    if (IP_IS_UNDEF_OBJ(PObj -> Pnext)) {
		IRIT_WNDW_FPRINTF2("Error: undefined object \"%s\" has been removed from global data base\n",
				  PObj -> Pnext -> ObjName);

		PTmp = PObj -> Pnext -> Pnext;
		IPFreeObject(PObj -> Pnext);
		PObj -> Pnext = PTmp;
	    }
	}
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Modifies the state of the IRIT solid modeller.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   Name:      Name of state variable.                                       M
*   Data:      New value of state variable.                                  M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  Old value                                             M
*                                                                            *
* KEYWORDS:                                                                  M
*   SetIritState                                                             M
*****************************************************************************/
IPObjectStruct *SetIritState(char *Name, IPObjectStruct *Data)
{
    IPObjectStruct
	*OldVal = NULL;

    if (stricmp(Name, "BoolPerturb") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(BoolSetPerturbAmount(Data -> U.R));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "CmpObjEps") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			  InptPrsrSetCmpObjEps(Data -> U.R));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "CnvxPl2Vrtcs") == 0) {
	if (IP_IS_NUM_OBJ(Data)) {
	    OldVal = IPGenNUMValObject(
			     GMConvexRaysToVertices(REAL_TO_INT(Data -> U.R)));
	}
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "Coplanar") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			  BoolSetHandleCoplanarPoly(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "CursorKeep") == 0) {
        OldVal = IPGenNUMValObject(GlblKeepClientsCursorEvents);
        GlblKeepClientsCursorEvents = REAL_TO_INT(Data -> U.R);
	if (GlblKeepClientsCursorEvents && GlblLastClientCursorEvent != NULL) {
	    IPFreeObject(GlblLastClientCursorEvent);
	    GlblLastClientCursorEvent = NULL;
	}
    }
    else if (stricmp(Name, "DebugFunc") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			     InptPrsrDebugFuncLevel(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "Dependency") == 0) {
	if (IP_IS_NUM_OBJ(Data)) {
	    OldVal = IPGenNUMValObject(GlblHandleDependencies);
	    GlblHandleDependencies = REAL_TO_INT(Data -> U.R);
	}
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "DumpLevel") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(SetDumpLevel(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "EchoSource") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
	      GlblQuietMode ? FALSE
                            : InptPrsrSetEchoSource(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "FlatLoad") == 0) {
        if (IP_IS_NUM_OBJ(Data)) {
	    OldVal = IPGenNUMValObject(GlblFlatLoadMode);
	    GlblFlatLoadMode = REAL_TO_INT(Data -> U.R);
	}
    }
    else if (stricmp(Name, "FloatFrmt") == 0) {
	if (IP_IS_STR_OBJ(Data)) {
	    GlblFloatFormat = IritStrdup(Data -> U.Str);
	    OldVal = IPGenSTRObject(IPSetFloatFormat(GlblFloatFormat));
	}
	else
	    IRIT_WNDW_PUT_STR("String state value expected");
    }
    else if (stricmp(Name, "FastPolys") == 0) {
	if (IP_IS_NUM_OBJ(Data)) {
	    OldVal = IPGenNUMValObject(
			        CagdSrf2PolygonFast(REAL_TO_INT(Data -> U.R)));
	}
	else
	    IRIT_WNDW_PUT_STR("String state value expected");
    }
    else if (stricmp(Name, "GMEpsilon") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			              GMBasicSetEps(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "InterCrv") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			    BoolSetOutputInterCurve(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "InterpProd") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			          BspMultInterpFlag(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "MvNConeTst") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(MvarMVsZerosNormalConeTest(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "LoadFont") == 0) {
	if (IP_IS_STR_OBJ(Data))
	    OldVal = IPGenNUMValObject(GMLoadTextFont(Data -> U.Str));
	else
	    IRIT_WNDW_PUT_STR("String state value expected");
    }
    else if (stricmp(Name, "PolySort") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			        BoolSetPolySortAxis(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "PrimSrfs") == 0) {
	if (IP_IS_NUM_OBJ(Data)) {
	    OldVal = IPGenNUMValObject(
		       !PrimSetPolygonalPrimitives(!REAL_TO_INT(Data -> U.R)));
	}
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "PrimRatSrfs") == 0) {
	if (IP_IS_NUM_OBJ(Data)) {
	    OldVal = IPGenNUMValObject(
		    PrimSetSurfacePrimitiveRational(REAL_TO_INT(Data -> U.R)));
	}
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "RandomInit") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    IritRandomInit(REAL_TO_INT(Data -> U.R));
	OldVal = IPGenNUMValObject(Data -> U.R);
    }
    else if (stricmp(Name, "TrimCrvs") == 0) {
	if (IP_IS_NUM_OBJ(Data)) {
	    TrimSetEuclidComposedFromUV(Data -> U.R <= 0);
	    TrimSetTrimCrvLinearApprox((int) (FABS(Data -> U.R)),
				       SYMB_CRV_APPROX_UNIFORM);
	}
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }
    else if (stricmp(Name, "UVBoolean") == 0) {
	if (IP_IS_NUM_OBJ(Data))
	    OldVal = IPGenNUMValObject(
			  BoolSetParamSurfaceUVVals(REAL_TO_INT(Data -> U.R)));
	else
	    IRIT_WNDW_PUT_STR("Numeric state value expected");
    }

    return OldVal;
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Routine to query (and print) the errors found by InputParser.		     *
*                                                                            *
* PARAMETERS:                                                                *
*   None                                                                     *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void PrintInptPrsrError(void)
{
    InptPrsrEvalErrType ErrorNum;
    char *ErrorMsg, *p;
    char Line[INPUT_LINE_LEN + LINE_LEN_LONG];

    if ((ErrorNum = InptPrsrParseError(&ErrorMsg)) != IPE_NO_ERR) {/*Prsr err*/
	sprintf(Line, "Parsing Error: ");
	p = &Line[strlen(Line)];
	switch (ErrorNum) {
	    case IR_ERR_WRONG_SYNTAX:
		sprintf(p, "Wrong syntax\n");
		break;
	    case IR_ERR_PARAM_EXPECT:
		sprintf(p, "Parameter Expected - %s\n", ErrorMsg);
		break;
	    case IR_ERR_ONE_OPERAND:
	    case IR_ERR_TWO_OPERAND:
		sprintf(p, "Wrong # of operands - %s\n", ErrorMsg);
		break;
	    case IR_ERR_STACK_OV:
		sprintf(p, "Internal Stack OverFlow at - %s\n", ErrorMsg);
		break;
	    case IR_ERR_PARAM_MATCH:
		sprintf(p, "Parenthesis mismatch - %s\n", ErrorMsg);
		break;
	    case IR_ERR_UNDEF_TOKEN:
		sprintf(p, "Undefined token - %s\n", ErrorMsg);
		break;
	    case IR_ERR_UNDEF_FUNC:
		sprintf(p, "Undefined function - %s\n", ErrorMsg);
		break;
	    case IR_ERR_NAME_TOO_LONG:
		sprintf(p, "Object name too long - %s\n", ErrorMsg);
		break;
	    case IR_ERR_PARAM_FUNC:
		sprintf(p, "Parameters expected in func %s\n", ErrorMsg);
		break;
	    case IR_ERR_NO_PARAM_FUNC:
		sprintf(p, "No Parameters expected in func %s\n", ErrorMsg);
		break;
	    case IR_ERR_STR_TOO_LONG:
		sprintf(p, "String too long - %s\n", ErrorMsg);
		break;
	    default:
		sprintf(p, "Undefined error %d", ErrorNum);
		break;
	}
	IRIT_WNDW_PUT_STR(Line);
	return;
    }

    if ((ErrorNum = InptPrsrEvalError(&ErrorMsg)) != IPE_NO_ERR) {/*Eval err.*/
	sprintf(Line, "Eval Error: ");
	p = &Line[strlen(Line)];
	switch (ErrorNum) {
	    case IE_ERR_FATAL_ERROR:
		sprintf(p, "Fatal error - %s\n", ErrorMsg);
		break;
	    case IE_ERR_DIV_BY_ZERO:
		sprintf(p, "Division by zero - %s\n", ErrorMsg);
		break;
	    case IE_ERR_NO_OBJ_METHOD:
		sprintf(p, "No such method for object - %s\n", ErrorMsg);
		break;
	    case IE_ERR_TYPE_MISMATCH:
		sprintf(p, "Parameter type mismatch - %s\n", ErrorMsg);
		break;
	    case IE_ERR_ASSIGN_LEFT_OP:
		sprintf(p, "Lval is not a parameter - %s\n", ErrorMsg);
		break;
	    case IE_ERR_MIXED_OBJ:
		sprintf(p, "Mixed types in expression - %s\n", ErrorMsg);
		break;
	    case IE_ERR_IP_OBJ_UNDEFINED:
		sprintf(p, "No such object defined - %s\n", ErrorMsg);
		break;
	    case IE_ERR_NO_ASSIGNMENT:
		sprintf(p, "Assignment was expected\n");
		break;
	    case IE_ERR_FP_ERROR:
		sprintf(p, "Floating Point Error - %s\n", ErrorMsg);
		break;
	    case IE_ERR_NUM_PRM_MISMATCH:
		sprintf(p, "Number of func. param. mismatch - %s\n", ErrorMsg);
		break;
	    case IE_ERR_MAT_POWER:
		sprintf(p, "Wrong range or not exists, operator - %s\n", ErrorMsg);
		break;
	    case IE_ERR_FREE_SIMPLE:
		sprintf(p, "Free only geometric objects - %s\n", ErrorMsg);
		break;
	    case IE_ERR_MODIF_ITER_VAR:
		sprintf(p, "Iteration var. type modified or freed - %s\n", ErrorMsg);
		break;
	    case IE_ERR_BOOLEAN_ERR:
		sprintf(p, "Geometric Boolean operation error - %s\n", ErrorMsg);
		break;
	    case IE_ERR_OUT_OF_RANGE:
		sprintf(p, "Out of range.\n");
		break;
	    case IE_ERR_DATA_PRSR_ERROR:
		sprintf(p, "%s", ErrorMsg);
		break;
	    case IE_ERR_USER_FUNC_NO_RETVAL:
		sprintf(p, "User defined function \"%s\" has no returned value\n",
			ErrorMsg);
		break;
	    case IE_ERR_INCOMPARABLE_TYPES:
		sprintf(p, "Incomparable object types found");
		break;
	    case IE_ERR_ONLYEQUALITY_TEST:
		sprintf(p, "Only equality or non equality test is valid for these objects");
		break;
	    case IE_ERR_IF_HAS_NO_COND:
		sprintf(p, "Condition of if clause is illegal");
		break;
	    case IE_ERR_IP_USERFUNC_DUP_VAR:
		sprintf(p, "Duplicated variable, %s", ErrorMsg);
		break;
	    case IE_ERR_IP_USERFUNC_TOO_MANY_PRMS:
		sprintf(p, "Too many parameters, %s", ErrorMsg);
		break;
	    case IE_ERR_UNDEF_INSTANCE:
		sprintf(p, "Undefined object reference in instance, %s",
			ErrorMsg);
		break;
	    default:
		sprintf(p, "Undefined error %d\n", ErrorNum);
		break;
	}
	IRIT_WNDW_PUT_STR(Line);
	return;
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Routine to handle reading one line from stdin into string Str, with        M
* maximum of length Length in the Input Window.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   Str:        Where input is placed at.                                    M
*   Length:     Maximum length of input to allow.                            M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritInputWindowGetStr                                                    M
*****************************************************************************/
void IritInputWindowGetStr(char *Str, int Length)
{
#ifdef DJGCC
    if (GlblDoGraphics) {
	Str[0] = 0;
    	IntrGetLineWindow(IGGlblInputWindowID, Str, Length);
    }
    else {
	WndwGetLineStdin(Str, IRIT_PROMPT, Length - 1);

	if (Str[strlen(Str) - 1] < ' ')
	    Str[strlen(Str) - 1] = 0;				/* No CR/LF. */
	puts(Str);
	fflush(stdout);
    } 
#else
    WndwGetLineStdin(Str, IRIT_PROMPT, Length - 1);

    if (Str[strlen(Str) - 1] < ' ') 
	Str[strlen(Str) - 1] = 0;				/* No CR/LF. */
    puts(Str);
    fflush(stdout);
#endif /* DJGCC */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* IRIT Reset routine. Initialize again all object as in the starting state.  M
*                                                                            *
* PARAMETERS:                                                                M
*   None                                                                     *
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritReset                                                                M
*****************************************************************************/
void IritReset(void)
{
    InptEvalFreeFunc(NULL);

    if (GlblObjList != NULL) {
	IPFreeObjectList(GlblObjList);
	GlblObjList = NULL;
    }

    SetUpPredefObjects();		  /* Prepare the predefined objects. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* IRIT Exit routine. Error code of zero.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   None                                                                     *
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritExit0                                                                M
*****************************************************************************/
void IritExit0(void)
{
    IritExit(0);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* IRIT Exit routine.                           				     M
*                                                                            *
* PARAMETERS:                                                                M
*   ExitCode:   Exit code.                                                   *
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritExit                                                                 M
*****************************************************************************/
void IritExit(int ExitCode)
{
    RealType
	R = IP_CLNT_BROADCAST_ALL_HANDLES;

    if (GlblDoGraphics) {
    	WndwViewExit();
    }

    ClientClose(&R, TRUE);

    chdir(GlblCrntWorkingDir);	  /* Recover original directory before exit. */

    if (GlblPrintLogFile)
	fclose(GlblLogFile);		      /* Close log file if was open. */

#   ifdef __UNIX__    
#if defined (__FreeBSD__)
	tcsetattr(0, TCSANOW, &GlblOrigTermio);
#else
	ioctl(0, TCSETA, &GlblOrigTermio);
#endif /* !__FreeBSD__ */
#   endif /* __UNIX__ */

#   ifdef __WINNT__
	_exit(ExitCode);
#   else
	exit(ExitCode);
#   endif /* __WINNT__ */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Geom_lib errors right here. Call back function of geom_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in geom_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   GeomFatalError                                                           M
*****************************************************************************/
void GeomFatalError(GeomFatalErrorType ErrID)
{
    char
        *ErrorMsg = GeomDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("GEOM_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Misc_lib errors right here. Call back function of misc_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in misc_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   MiscFatalError                                                           M
*****************************************************************************/
void MiscFatalError(MiscFatalErrorType ErrID)
{
    char
        *ErrorMsg = MiscDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("MISC_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               *
* Traps Bool_lib errors right here. Call back function of bool_lib.	     *
*                                                                            *
* PARAMETERS:                                                                *
*   ErrID:    Error number in bool_lib library.                              *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void IritBoolFatalError(BoolFatalErrorType ErrID)
{
    char
        *ErrorMsg = BoolDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("BOOL_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Cagd_lib errors right here. Call back function of cagd_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in cagd_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   CagdFatalError                                                           M
*****************************************************************************/
void CagdFatalError(CagdFatalErrorType ErrID)
{
    char
        *ErrorMsg = CagdDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("CAGD_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Symb_lib errors right here. Call back function of symb_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in symb_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   SymbFatalError                                                           M
*****************************************************************************/
void SymbFatalError(SymbFatalErrorType ErrID)
{
    char
	*ErrorMsg = SymbDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("SYMB_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Trim_lib errors right here. Call back function of trim_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in trim_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   TrimFatalError                                                           M
*****************************************************************************/
void TrimFatalError(TrimFatalErrorType ErrID)
{
    char
	*ErrorMsg = TrimDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("TRIM_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Triv_lib errors right here. Call back function of triv_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in triv_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   TrivFatalError                                                           M
*****************************************************************************/
void TrivFatalError(TrivFatalErrorType ErrID)
{
    char
	*ErrorMsg = TrivDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("TRIV_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps Trng_lib errors right here. Call back function of trng_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in trng_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   TrngFatalError                                                           M
*****************************************************************************/
void TrngFatalError(TrngFatalErrorType ErrID)
{
    char
	*ErrorMsg = TrngDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("TRNG_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps mvar_lib errors right here. Call back function of mvar_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in mvar_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   MvarFatalError                                                           M
*****************************************************************************/
void MvarFatalError(MvarFatalErrorType ErrID)
{
    char
	*ErrorMsg = MvarDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("MVAR_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Traps user_lib errors right here. Call back function of user_lib.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrID:    Error number in user_lib library.                              M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   UserFatalError                                                           M
*****************************************************************************/
void UserFatalError(UserFatalErrorType ErrID)
{
    char
	*ErrorMsg = UserDescribeError(ErrID);

    IRIT_WNDW_FPRINTF2("USER_LIB: %s", ErrorMsg);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Prints IRIT none fatal error message and go back to cursor mode.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   va_alist:    Format of error message to print out.			     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   IritNonFatalError                                                        M
*****************************************************************************/
#ifdef USE_VARARGS
void IritNonFatalError(char *va_alist, ...)
{
    char *Format, Line[LINE_LEN_LONG];
    int i;
    va_list ArgPtr;

    va_start(ArgPtr);
    Format = va_arg(ArgPtr, char *);
#else
void IritNonFatalError(char *Format, ...)
{
    char Line[LINE_LEN_LONG];
    va_list ArgPtr;

    va_start(ArgPtr, Format);
#endif /* USE_VARARGS */

    vsprintf(Line, Format, ArgPtr);
    IRIT_WNDW_PUT_STR(Line);

    va_end(ArgPtr);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Prints IRIT' warnings to stdout.                       		     M
*   This function just prints the given error message.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   Msg:       Error message to print.                                       M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritWarningError, error trap                                             M
*****************************************************************************/
void IritWarningError(char *Msg)
{
    if (Msg != NULL) {
	IRIT_WNDW_FPRINTF2("Irit Warning: %s", Msg);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Prints IRIT's fatal error message and go back to cursor mode.		     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrorMsg:    Error message to print out.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   IritFatalError, error trap                                               M
*****************************************************************************/
void IritFatalError(char *ErrorMsg)
{
    if (ErrorMsg != NULL) {
	IRIT_WNDW_PUT_STR("Fatal error occured, please report it:");
	IRIT_WNDW_PUT_STR(ErrorMsg);
    }

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
* Trap error and go back to cursor mode.				     M
*                                                                            *
* PARAMETERS:                                                                M
*   None		                                                     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   IritQuietError                                                           M
*****************************************************************************/
void IritQuietError(void)
{
    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Idle function that is invoked when no input is waiting from stdin/file.  M
* typically sleeps and/or checks display device's io.			     M
*                                                                            *
* PARAMETERS:                                                                M
*   MiliSeconds:  Period to sleep if nothing to do.                          M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritIdleFunction                                                         M
*****************************************************************************/
void IritIdleFunction(int MiliSeconds)
{
#ifdef DUMP_COUNTER
    STATIC_DATA int
	i = 0;

    fprintf(stderr, IRIT_EXP_STR("IritIdle: %d\r"), i++);
#endif /* DUMP_COUNTER */

    if (!IPSocSrvrListen() && MiliSeconds > 0)
        IritSleep(MiliSeconds);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Call back function of the server listening to clients.                   M
*                                                                            *
* PARAMETERS:                                                                M
*   Handler:    Client handler from which an event has been recieved.        M
*   PObj:       NULL if a new client has connected, object recieved from     M
*		Client otherwise.					     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* SEE ALSO:                                                                  M
*   IritIdleFunction                                                         M
*                                                                            *
* KEYWORDS:                                                                  M
*   IPSocHandleClientEvent                                                   M
*****************************************************************************/
void IPSocHandleClientEvent(int Handler, IPObjectStruct *PObj)
{
    int i;
    char *p;

    if (PObj == NULL) {
	if (GlblGUIMode)
	    IRIT_WNDW_FPRINTF2("<<IPC>> New client - handler %d <<CPI>>\n",
			       Handler);

	/* If no client connected so far, make sure we validate this one. */
	if (GlblCurrentDisplay < 0)
	    GlblCurrentDisplay = IP_CLNT_BROADCAST_ALL_HANDLES;
    }
    else if (GlblKeepClientsCursorEvents &&
	     stricmp(PObj -> ObjName, "_PickCrsr_") == 0) {
        /* Keep this object as the last cursor event. */
        if (GlblLastClientCursorEvent == NULL)
	    GlblLastClientCursorEvent = PObj;
	else
	    IPFreeObject(PObj);
    }
    else if ((stricmp(PObj -> ObjName, "_SubmitObject_") == 0 ||
	      stricmp(PObj -> ObjName, "_SubmitMat_") == 0) &&
	     (p = AttrGetObjectStrAttrib(PObj, "ObjName")) != NULL) {
        IP_SET_OBJ_NAME2(PObj, p);
	for (i = 0; PObj -> ObjName[i]; i++)
	    PObj -> ObjName[i] = islower(PObj -> ObjName[i]) ?
                              toupper(PObj -> ObjName[i]) : PObj -> ObjName[i];

	if (GlblGUIMode) {
	    IRIT_WNDW_FPRINTF3("<<IPC>> Client %d Start Submission \"%s\" <<CPI>>\n",
			       Handler, p);

	    ConvertData2Irt(PObj);
	    IRIT_WNDW_FPRINTF3("<<IPC>> Client %d End Submission \"%s\" <<CPI>>\n",
			       Handler, p);
	}
	else {
	    IRIT_WNDW_FPRINTF3("<<IPC>> Accepted submitted object \"%s\" from handler %d <<CPI>>\n",
			       p, Handler);
	}
	AttrFreeObjectAttribute(PObj, "ObjName");
	InsertIritObjectLast(PObj, TRUE);
    }
    else {
        if (stricmp(PObj -> ObjName, "_PickFail_") == 0) {
	    IRIT_WNDW_FPRINTF2("<<IPC>> Client %d failed to pick <<CPI>>\n",
			       Handler);
	}
	else if (stricmp(PObj -> ObjName, "_PickName_") == 0) {
	    IRIT_WNDW_FPRINTF3("<<IPC>> Client %d picked object name \"%s\" <<CPI>>\n",
			       Handler, PObj -> U.Str);
	}
	else {
	    /* It is complete geometry. */
	    if (GlblGUIMode) {
	        IRIT_WNDW_FPRINTF3("<<IPC>> Client %d Start Submission \"%s\" <<CPI>>\n",
				   Handler, PObj -> ObjName);
		ConvertData2Irt(PObj);
		IRIT_WNDW_FPRINTF3("<<IPC>> Client %d End Submission \"%s\" <<CPI>>\n",
				   Handler, PObj -> ObjName);
	    }
	    else
	        IRIT_WNDW_FPRINTF3("<<IPC>> Client %d picked complete geometry of \"%s\" <<CPI>>\n",
				   Handler, PObj -> ObjName);
	}
	IPFreeObject(PObj);
    }
}

/*****************************************************************************
* DESCRIPTION:                                                               *
*   Converts the given PObj to irit script (.irt) style and dump out.        *
* Mostly used by the GUI interface of irit.				     *
*                                                                            *
* PARAMETERS:                                                                *
*   PObj:        To convert given PObj object to .irt style.                 *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void ConvertData2Irt(IPObjectStruct *PObj)
{
    IPPrintFuncType
	OldPrintFunc = IPCnvSetPrintFunc(IritWndwPutStr2);

    IPCnvDataToIrit(PObj);

    IPCnvSetPrintFunc(OldPrintFunc);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*   Get next cursor event from one of the clients, and wait at most WaitTime M
* milliseconds.								     M
*                                                                            *
* PARAMETERS:                                                                M
*   RWaitTime:  Maximal time to wait for the cursor event, in miliseconds.   M
*		If zero, waits indefinitly.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   IPObjectStruct *:  The picked cursor event of a list object with one     M
*		string "_PickFail_" object of nothing occured.		     M
*                                                                            *
* KEYWORDS:                                                                  M
*   IritPickCrsrClientEvent                                                  M
*****************************************************************************/
IPObjectStruct *IritPickCrsrClientEvent(RealType *RWaitTime)
{
    int OrigWaitTime = REAL_PTR_TO_INT(RWaitTime),
	WaitTime = OrigWaitTime;
    IPObjectStruct *RetVal;

    while (WaitTime >= 0) {
        if (GlblLastClientCursorEvent != NULL) {
	    RetVal = GlblLastClientCursorEvent;
	    GlblLastClientCursorEvent = NULL;	
	    return RetVal;
	}

	if (OrigWaitTime > 0)
	    WaitTime -= 10;

	IritIdleFunction(10);
    }

    /* Return empty list object instead. */
    RetVal = IPGenListObject("_PickFail_", NULL, NULL);

    return RetVal;
}

#ifndef IRIT_QUIET_STRINGS

/*****************************************************************************
* DESCRIPTION:                                                               M
* Prints IRIT parser's fatal error message and go back to cursor mode.	     M
*                                                                            *
* PARAMETERS:                                                                M
*   ErrorMsg:    Error message to print out.				     M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   IPFatalError                                                             M
*****************************************************************************/
void IPFatalError(char *ErrorMsg)
{
    if (ErrorMsg != NULL) {
        IRIT_WNDW_FPRINTF2("PRSR_LIB: %s", ErrorMsg);
    }

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);	   /* Go back to main loop directly. */
}

#endif /* !IRIT_QUIET_STRINGS */

/*****************************************************************************
* DESCRIPTION:                                                               M
* Called from the floating point package in case of fatal floating point     M
* error.								     M
*   Prints error message and long jumps to main loop.			     M
*   Default FPE handler - must be reset after redirected to other module.    M
*                                                                            *
* PARAMETERS:                                                                M
*   Type:      Of floating point error.                                      M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            *
* KEYWORDS:                                                                  M
*   DefaultFPEHandler                                                        M
*****************************************************************************/
void DefaultFPEHandler(int Type)
{
    IRIT_WNDW_FPRINTF2("Floating point error %d.", Type);

    InptPrsrFlushToEndOfExpr(TRUE);

#   ifdef DEBUG
    AllFatalErrorTrap();
#   endif /* DEBUG */

    longjmp(GlblLongJumpBuffer, 1);
}

#ifdef DEBUG

/*****************************************************************************
* DESCRIPTION:                                                               *
*    Dummy function to link at debugging time.                               *
*                                                                            *
* PARAMETERS:                                                                *
*   None	                                                             *
*                                                                            *
* RETURN VALUE:                                                              *
*   void                                                                     *
*****************************************************************************/
static void AllFatalErrorTrap(void)
{
    IRIT_WNDW_PUT_STR("Trapped fatal error from external library");
    fflush(stderr);
    fflush(stdout);
}

/*****************************************************************************
* DESCRIPTION:                                                               M
*    Dummy function to link at debugging time.                               M
*                                                                            *
* PARAMETERS:                                                                M
*   None	                                                             M
*                                                                            *
* RETURN VALUE:                                                              M
*   void                                                                     M
*                                                                            M
* KEYWORDS:                                                                  M
*   DummyLinkCagdDebug                                                       M
*****************************************************************************/
void DummyLinkCagdDebug(void)
{
    IPDbg();
}

#endif /* DEBUG */
