#ifndef _TEXTURE_H
#define _TEXTURE_H

#include "rndr_loc.h"

#define TEXTURE_TYPE_NONE    0
#define TEXTURE_TYPE_PROC    1
#define TEXTURE_TYPE_RSTR    2
#define TEXTURE_TYPE_SRF    3
#define TEXTURE_TYPE_BUMP    8     /* Can be combined with any of the above. */

#define STEXTURE_FUNC_NONE    0
#define STEXTURE_FUNC_SQRT    1
#define STEXTURE_FUNC_ABS    2

#define PTEXTURE_UV_TYPE        0
#define PTEXTURE_SPHERICAL_TYPE        1
#define PTEXTURE_SPHERE_BIJECT_TYPE    2
#define PTEXTURE_CYLINDERICAL_TYPE    3
#define PTEXTURE_PLANAR_TYPE        4

/* Texture image: its size and 2D array of pixels. */
typedef struct ImageStruct {
    int xSize, ySize;
    IrtImgPixelStruct *Data;
} ImageStruct;

struct TextureStruct;

typedef void (*TextureFuncType)(struct TextureStruct *,
                PointType,
                NormalType,
                RealType *,
                IRndrColorType);

typedef struct TextureStruct {
    int Type;                                  /* Procedural/Raster/Surface. */
    char *TextureFileName;

    IPObjectStruct *OrigSrf;      /* Reference to original geometry surface. */
    CagdPType OrigSrfParamDomain[2];   /* Parametric domain of original srf. */

    /* Raster image texture. */
    ImageStruct *PrmImage;  /* Points to texture image if exists, NULL else. */
    RealType PrmUMin, PrmUMax, PrmVMin, PrmVMax;       /* Parametric domain. */
    RealType PrmUScale, PrmVScale, PrmWScale;    /* Scale in U/V of texture. */
    RealType PrmAngle;     /* Angle of rotation of texture around main axis. */
    int PrmTextureType;  /* Type of param. texture: regular, spherical, etc. */
    VectorType PrmDir;       /* Direction used in some parametric texturing. */
    VectorType PrmOrg;    /* Origin point used in some parametric texturing. */
    MatrixType PrmMat;          /* Parametric texture transformation matrix. */

    /* Surface style texture. */
    CagdSrfStruct *Srf;         /* A surface to evaluate for texture value.  */
    CagdPType SrfParamDomain[2];                /* Parametric domain of srf. */
    ImageStruct *SrfScale;          /* To map the value of srf into a color. */
    RealType SrfScaleMinMax[2];  /* Values to be mapped to srfScale extrema. */
    int SrfFunc;     /* If surface value should be piped through a function. */

    /* Curvature style textures. */
    RealType CrvtrSrfMax;         /* Bounding value on the curvature. */

    /* Procedure/volumetric type texture. */
    TextureFuncType vTexture;
    MatrixType Mat;                        /* Texture transformation matrix. */
    RealType Width;              /* Width used in some volumetric texturing. */
    RealType Depth;              /* Width used in some volumetric texturing. */
    RealType Brightness;    /* Brightness used in some volumetric texturing. */
    RealType Frequency;      /* Frequency used in some volumetric texturing. */
    IrtImgPixelStruct Color;     /* Color used in some volumetric texturing. */
    VectorType tScale;                          /* Volumetric texture scale. */
    VectorType Dir;          /* Direction used in some volumetric texturing. */
    VectorType Org;       /* Origin point used in some volumetric texturing. */
    /* Parameters specific for "wood" volumetric texture. */
    RealType CenterScale;
    RealType CenterFactor;
    RealType WaveNPoints;
    RealType WaveFactor;
    RealType FiberScale;
    RealType FiberFactor;
    /* Parameters specific for "marble" volumetric texture. */
    RealType TurbulenceScale;
    RealType TurbulenceFactor;
    /* Parameters specific for "checker" volumetric texture. */
    IRndrColorType CheckerColor[3];                     /* Checker's colors. */
    RealType CheckerPlane; /* Flag for whether it's a plane checker or a 3D. */
} TextureStruct;

IrtImgPixelStruct *TextureImageGetPixel(TextureStruct *Txtr,
                                 ImageStruct *i,
                                 PointType p,
                                 RealType v,
                                 RealType u,
                                 IPPolygonStruct *Poly);

/* Structures for noise generation, used for Solid Texture Mapping.          */

/* NoiseStruct is used for noise in a 3D space. */
typedef struct NoiseStruct {
    RealType Lattice[MAX_NOISE][MAX_NOISE][MAX_NOISE];
    RealType Scale;
} NoiseStruct;

/* SmoothNoiseStruct is used when a linear interpolation of noise values     */
/* isn't good enough. The values are interpolated by a cubic interpolation   */
/* but this is good only for 1D noise. (i.e. noise = f(x)   )                */
typedef struct SmoothNoiseStruct {
    int n;
    RealType Min, Max;
    RealType Coefs[MAX_NOISE - 1][4];
} SmoothNoiseStruct;

/* We make use of Texture name to Texture functions mapping table.           */
/* That data structure describes a single entry.                             */
typedef struct ProcTextureStruct {
    char *Name;             /* Procedural volumetric texture symbolic name. */
    TextureFuncType vTexture;
} ProcTextureStruct;

/*-----------------------------*/
void TextureMarble(TextureStruct *Txtr,
           PointType Point,
           NormalType Normal,
           RealType *Uv,
           IRndrColorType Color);
void TextureWood(TextureStruct *Txtr,
          PointType Point,
         NormalType Normal,
         RealType *Uv,
         IRndrColorType Color);
void TextureCamouf(TextureStruct *Txtr,
           PointType Point,
           NormalType Normal,
           RealType *Uv,
           IRndrColorType Color);
void TextureBumpOrange(TextureStruct *Txtr,
               PointType Point,
               NormalType Normal,
               RealType *Uv,
               IRndrColorType Color);
void TextureBumpChocolate(TextureStruct *Txtr,
              PointType Point,
              NormalType Normal,
              RealType *Uv,
              IRndrColorType Color);
void TextureChecker(TextureStruct *Txtr,
            PointType Point,
            NormalType Normal,
            RealType *Uv,
            IRndrColorType Color);
void TextureContour(TextureStruct *Txtr,
            PointType Point,
            NormalType Normal,
            RealType *Uv,
            IRndrColorType Color);
void TextureCurvature(TextureStruct *Txtr,
              PointType Point,
              NormalType Normal,
              RealType *Uv,
              IRndrColorType Color);
void TextureContourNormal(TextureStruct *Txtr,
               PointType Point,
              NormalType Normal,
              RealType *Uv,
              IRndrColorType Color);
void TexturePunky(TextureStruct *Txtr,
           PointType Point,
          NormalType Normal,
          RealType *Uv,
          IRndrColorType Color);
void TextureInitParameters( TextureStruct *Txtr, char *pString);

/* Mapping of predefined textures names to functions. */
extern GLOBAL_DATA ProcTextureStruct ProcTextures[];

#endif
