/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.setup;

import com.izforge.izpack.util.os.ShellLink;
import ja.centre.util.assertions.States;

import java.io.*;
import java.util.jar.JarOutputStream;
import java.util.jar.Pack200;
import java.util.zip.GZIPInputStream;

class JaLingoSetupHelper {
    private static final String RESOURCES = "resources/";

    private static final String JAR = "jalingo.jar";
    private static final String JAR_PACK_GZ = JAR + ".pack.gz";

    private static final String[] ICONS = {
            "jalingo.ico",
            "jalingo16x16.png", "jalingo32x32.png",
            "jalingo48x48.png", "jalingo64x64.png" };
    private static final String ICON_ICO = ICONS[0];

    private static OsDependent osDependent = OsDependent.create();

    public static String getDefaultDestination() {
        return osDependent.getDefaultDestination();
    }
    public static String getDefaultDirName() {
        return osDependent.getDefaultDirName();
    }

    public static boolean canCreateShortcuts() {
        return osDependent.canCreateShortcuts();
    }

    public static void install( String destination, boolean desktopShortcut, boolean startMenuShortcut ) throws Exception {
        File dirFile = new File( destination );

        if ( dirFile.exists() && !dirFile.isDirectory() ) {
            States.doThrow( "Destination path \"" + destination + "\" already exists. It is a file. Should be a directory." );
        }

        if ( !dirFile.exists() && !dirFile.mkdirs() ) {
            States.doThrow( "Could not create installation directory \"" + destination + "\"" );
        }

        // install jar
        try {
            installJar( destination );
        } catch ( IOException e ) {
            States.doThrow( "Could not install \""
                    + osDependent.calculateDestination( destination, JAR )
                    + "\": " + e.getMessage(), e );
        }
        // install icon
        try {
            installIcons( destination );
        } catch ( IOException e ) {
            States.doThrow( "Could not install \""
                    + osDependent.calculateDestination( destination, JAR )
                    + "\": " + e.getMessage(), e );
        }

        // create shortcuts
        try {
            if ( desktopShortcut ) {
                createShortcut( ShellLink.DESKTOP, destination );
            }
            if ( startMenuShortcut ) {
                createShortcut( ShellLink.START_MENU, destination );
            }
        } catch ( Exception e ) {
            // if attempt to create shourtcuts failed, just leave it and continue
            e.printStackTrace();
        }

        // system-dependent post-install
        try {
            osDependent.doPostinstall( destination, JAR );
        } catch ( Exception e ) {
            // just leave it and continue
            e.printStackTrace();
        }
    }

    public static void installJar( String destination ) throws IOException {
        // NOTE possible leak: streams stay open if exception was raised

        InputStream is = null;
        JarOutputStream os = null;
        try {
            is = new GZIPInputStream( getResource( JAR_PACK_GZ ) );
            os = new JarOutputStream( new FileOutputStream(
                    osDependent.calculateDestination( destination, JAR ) ) );

            Pack200.newUnpacker().unpack( is, os );
        } finally {
            if ( is != null ) {
                is.close();
            }
            if ( os != null ) {
                os.close();
            }
        }
    }

    public static void installIcons( String destination ) throws IOException {
        for ( int i = 0; i < ICONS.length; i++ ) {
            String icon = ICONS[i];

            InputStream is = getResource( icon );
            OutputStream os = new FileOutputStream( osDependent.calculateDestination( destination, icon ) );

            // NOTE not too efficient, but the icon is small
            int read;
            while ( (read = is.read()) != -1 ) {
                os.write( read );
            }
            is.close();
            os.close();
        }
    }

    private static void createShortcut( int linkType, String destination ) throws Exception {
        String arguments = osDependent.getJavaCmdArgs( destination, JAR );

        ShellLink link = new ShellLink();
        link.setLinkName( null );
        link.setLinkType( linkType );
        link.setUserType( ShellLink.CURRENT_USER );
        link.setIconLocation( osDependent.calculateDestination( destination, ICON_ICO ), 0 );
        //link.setIconLocation( "%SystemRoot%\\system32\\SHELL32.dll", 33 * 4 + 1 );
        link.setDescription( "JaLingo" );
        link.setLinkName( "JaLingo" );
        link.setShowCommand( ShellLink.NORMAL );
        link.setTargetPath( osDependent.getJavaCmd() );
        link.setArguments( arguments );
        link.setWorkingDirectory( destination );
        link.save();
    }

    private static InputStream getResource( String resourceName ) {
        InputStream is = JaLingoSetup.class.getClassLoader().getResourceAsStream( RESOURCES + resourceName );

        if ( is == null ) {
            throw new RuntimeException( "Could not get resource \"" + resourceName + "\"" );
        }

        return is;
    }

    // TODO implement run
    /*
        public static void run( final String destination ) throws Exception {
            new Thread( new Runnable() {
                public void run() {
                    try {
                        String destinationJar = calculateDestinationJar( destination );

                        URLClassLoader classLoader = new URLClassLoader( new URL[] {
                                new File( destinationJar ).toURL()
                        } );
                        Class loaderClass = classLoader.loadClass( "ja.lingo.client.JaLingo" );
                        Method mainMethod = loaderClass.getMethod( "main", String[].class );

                        mainMethod.invoke( null, (Object) new String[] { } );
                    } catch ( Exception e ) {
                        throw new RuntimeException( e );
                    }
                }
            } ).start();
        }
    */
}
