/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.engine;

import ja.JaTestCase;
import ja.centre.util.io.checksum.ChecksumUtil;
import ja.centre.util.io.Files;
import ja.centre.util.io.lock.LockedException;
import ja.lingo.engine.beans.IArticle;
import ja.lingo.engine.monitor.NullMonitor;
import ja.lingo.engine.reader.Readers;

import java.io.File;
import java.io.IOException;

public class EngineTest extends JaTestCase {
    private static final String TEMP_DIRECTORY = "temp";
    private static final String WORKING_DIRECTORY = "temp/engine";

    private static final String KOI8_R = "KOI8-R";

    private static final String PTKDICT_1 = "ptkdict_1.sql";
    private static final String PTKDICT_2 = "ptkdict_2.sql";

    private static final String MOVA = "mova.mova";
    private static final String MOVA_MODIFIED = "mova_modified.mova";

    private IEngine engine;

    protected void setUp() throws Exception {
        super.setUp();

        Files.deleteRecusrsively( TEMP_DIRECTORY );
        Files.makeDirs( TEMP_DIRECTORY );
        Files.makeDirs( WORKING_DIRECTORY );

        recreateEngine();
    }
    protected void tearDown() throws Exception {
        try {
            engine.close();
            Files.deleteRecusrsively( TEMP_DIRECTORY );
        } catch ( Exception e) {
            e.printStackTrace();
        } finally {
            super.tearDown();
        }
    }

    private void recreateEngine() throws LockedException, UnknownCacheVersionException, IOException {
        Files.close( engine );
        engine = new Engine( WORKING_DIRECTORY );
    }

    private static String inTemp( String file ) {
        return TEMP_DIRECTORY + File.separator + file;
    }

    public void test() throws IOException {
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );
        engine.addDictionary( PTKDICT_2, KOI8_R, Readers.PTKDICT );

        engine.compile( new NullMonitor() );

        engine.getFinder().get( 0 ).getTitle();

        engine.remove( engine.getInfos().get( 0 ) );
        engine.remove( engine.getInfos().get( 0 ) );

        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );

        engine.compile( new NullMonitor() );

        engine.getFinder().get( 0 ).getTitle();
    }

    public void testAddRemoveTheSame() throws IOException {
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );
        engine.compile( new NullMonitor() );

        engine.remove( engine.getInfos().get( 0 ) );
        assertTrue( engine.getInfos().isEmpty() );
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );

        engine.compile( new NullMonitor() );
    }

    public void testAddRemoveTheSame2() throws IOException, UnknownCacheVersionException, LockedException {
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );
        engine.compile( new NullMonitor() );
        engine.close();

        recreateEngine();

        engine.remove( engine.getInfos().get( 0 ) );
        assertTrue( engine.getInfos().isEmpty() );
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );

        engine.compile( new NullMonitor() );
    }

    public void testSwap() throws IOException {
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );
        engine.addDictionary( PTKDICT_2, KOI8_R, Readers.PTKDICT );

        engine.compile( new NullMonitor() );

        assertEquals( 3, engine.getInfos().get( 0 ).getCapacity() );
        assertEquals( 3, engine.getInfos().get( 1 ).getCapacity() );

        {
            IArticle article = engine.getFinder().get( 0 );
            assertEquals( "a", article.getTitle() );
            assertEquals( 2, article.size() );
            assertEquals( "-a from dictionary 1", article.getBody( 0 ) );
            assertEquals( "-a from dictionary 2", article.getBody( 1 ) );
        }

        engine.swapDictionaries( 0, 1 );

        engine.compile( new NullMonitor() );

        {
            IArticle article = engine.getFinder().get( 0 );
            assertEquals( "a", article.getTitle() );
            assertEquals( 2, article.size() );
            assertEquals( "-a from dictionary 2", article.getBody( 0 ) );
            assertEquals( "-a from dictionary 1", article.getBody( 1 ) );
        }
    }

    public void testModelIsSavedOnCompile() throws IOException {
        engine.addDictionary( PTKDICT_1, KOI8_R, Readers.PTKDICT );

        engine.compile( new NullMonitor() );

        EngineFileNamer fileNamer = new EngineFileNamer( WORKING_DIRECTORY );
        long modelChecksum = ChecksumUtil.calculateFull( fileNamer.getModelFileName() );

        engine.addDictionary( PTKDICT_2, KOI8_R, Readers.PTKDICT );
        engine.compile( new NullMonitor() );

        long modelChecksum2 = ChecksumUtil.calculateFull( fileNamer.getModelFileName() );

        assertFalse( modelChecksum == modelChecksum2 );
    }

    public void testDetectRemove() throws LockedException, UnknownCacheVersionException, IOException {
        Files.copy( PTKDICT_1, inTemp( PTKDICT_1 ) );

        engine.addDictionary( inTemp( PTKDICT_1 ), KOI8_R, Readers.PTKDICT );
        engine.compile( new NullMonitor() );

        assertEquals( 1, engine.getInfos().size() );

        engine.close();

        // TODO sometimes tests fail because this file is not deleted (or it looks so). Why?
        Files.delete( inTemp( PTKDICT_1 ) );
        //assertFalse( Files.exists( PTKDICT_1_TEMP ));

        recreateEngine();
        assertEquals( 0, engine.getInfos().size() );
    }

    public void testDetectChange() throws IOException, UnknownCacheVersionException, LockedException {
        Files.copy( MOVA, inTemp( MOVA ) );

        engine.addDictionary( inTemp( MOVA ), KOI8_R, Readers.MOVA );
        engine.compile( new NullMonitor() );

        assertEquals( 3, engine.getFinder().size() );

        engine.close();

        Files.copy( MOVA_MODIFIED, inTemp( MOVA ) );

        recreateEngine();
        assertEquals( 4, engine.getFinder().size() );
    }
}
