/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.readers.dsl;

import ja.centre.util.io.Files;
import ja.lingo.engine.beans.Info;
import ja.lingo.engine.dictionaryindex.builder.MockDictionaryIndexBuilder;
import ja.lingo.engine.dictionaryindex.reader.DataSource;
import ja.lingo.engine.dictionaryindex.reader.DictionaryIndex;
import ja.lingo.engine.dictionaryindex.reader.IIndexSource;
import ja.lingo.engine.dictionaryindex.reader.ReaderFormat;
import ja.lingo.engine.monitor.NullMonitor;
import ja.lingo.engine.reader.IParser;
import junit.framework.TestCase;

import java.io.IOException;

public class DslReaderTest extends TestCase {
    private DictionaryIndex index;
    private static final String FILE_SIMPLE = "simple.dsl";
    private static final String FILE_NONEWLINE = "nonewline.dsl";
    private static final String FILE_BR = "br.dsl";
    private static final String FILE_TILDE = "tilde.dsl";

    protected void tearDown() throws Exception {
        Files.closeQuietly( index );

        super.tearDown();
    }

    private void build( String fileName ) {
        Info info = new Info();
        info.setDataFileName( "formats/dsl/" + fileName );
        info.setDataFileEncoding( "UTF-16LE" );

        MockDictionaryIndexBuilder builder = new MockDictionaryIndexBuilder();

        try {
            IParser parser = new DslReader().createParser( info, new NullMonitor() );
            parser.buildIndex( builder );

            IIndexSource indexSource = builder.getIndexSource();

            index = new DictionaryIndex( builder.getInfo(),
                    new ReaderFormat( indexSource, new DataSource( info.getDataFileName() ) ) );
        } catch ( IOException e ) {
            new RuntimeException( e );
        }
    }

    public void testSimple() {
        build( FILE_SIMPLE );

        assertEquals( 6, index.size() );

        assertEquals( "a", index.getTitle( 0 ) );
        assertEquals( "b", index.getTitle( 1 ) );
        assertEquals( "c", index.getTitle( 2 ) );
        assertEquals( "cc", index.getTitle( 3 ) );
        assertEquals( "c c", index.getTitle( 4 ) );
        assertEquals( "d", index.getTitle( 5 ) );

        assertEquals( "a<br>aa<br>aaa", index.getBody( 0 ) );
        assertEquals( "b<br>bb<br>bbb", index.getBody( 1 ) );
        assertEquals( "c<br>cc<br>ccc", index.getBody( 2 ) );
        assertEquals( "c<br>cc<br>ccc", index.getBody( 3 ) );
        assertEquals( "c<br>cc<br>ccc", index.getBody( 4 ) );
        assertEquals( "d<br>dd<br>ddd", index.getBody( 5 ) );
    }

    public void testNoNewLine() {
        build( FILE_NONEWLINE );

        assertEquals( 1, index.size() );
    }

    public void testBr() {
        build( FILE_BR );

        assertEquals( 2, index.size() );

        assertEquals( "1<br>2<br>3", index.getBody( 0 ) );

        assertEquals(
                "<div class=\"m1\">1</div>" +
                "<div class=\"m1\">2</div>" +
                "<div class=\"m1\">3</div>",
                index.getBody( 1 ) );
    }

    public void testTilda() {
        build( FILE_TILDE );

        assertEquals( 1, index.size() );

        assertEquals( "lingo", index.getTitle( 0 ) );
        assertEquals( "lingo lango", index.getBody( 0 ) );
    }
}