/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.centre.util.io.linereader;

import ja.centre.util.arrays.ArrayUtil;
import ja.centre.util.io.ByteArray;
import junit.framework.TestCase;

import java.io.IOException;
import java.nio.ByteBuffer;

public class LineReaderTest extends TestCase {
    private static final String LINE_0 = "line 0";
    private static final String LINE_1 = "line 1";
    private static final String LINE_2 = "line 2";

    private static final String DATA_0 = "line 0\nline 1\r\nline 2";

    private static final String ENCODING_ISO = "ISO-8859-1";
    private static final String ENCODING_UTF_16LE = "UTF-16LE";
    //private static final String ENCODING_UTF_16BE = "UTF-16BE";
    //private static final String ENCODING_UTF8 = "UTF-8";

    private static final byte[] SIGNATURE_UTF_16LE = new byte[] { (byte) 0xFF, (byte) 0xFE };
    //private static final byte[] SIGNATURE_UTF_16BE = new byte[] { (byte) 0xFE, (byte) 0xFF };
    //private static final byte[] SIGNATURE_UTF8 = new byte[] { (byte) 0xEF, (byte) 0xBB, (byte) 0xBF };

    private byte[] DATA_ISO;
    private byte[] DATA_UTF16LE;
    //private byte[] DATA_UTF16BE;
    //private byte[] DATA_UTF8;

    protected void setUp() throws Exception {
        super.setUp();

        DATA_ISO = DATA_0.getBytes( ENCODING_ISO );
        DATA_UTF16LE = ArrayUtil.prepend( DATA_0.getBytes( ENCODING_UTF_16LE ), SIGNATURE_UTF_16LE );
        //DATA_UTF16BE = ArrayUtil.prepend( DATA_0.getBytes( ENCODING_UTF_16BE ), SIGNATURE_UTF_16BE );
        //DATA_UTF8 = ArrayUtil.prepend( DATA_0.getBytes( ENCODING_UTF8 ), SIGNATURE_UTF8 );
    }

    public void testIso() throws IOException {
        checkReader( DATA_ISO, ENCODING_ISO, false );
    }

    public void testUtf16LE() throws IOException {
        checkReader( DATA_UTF16LE, ENCODING_UTF_16LE, true );
    }

    /*
        // TODO FIXME!
        public void testUnicodeBigEndian() throws IOException {
            checkReader( DATA_UTF16BE, ENCODING_UTF_16BE, true );
        }
    */

    //public void testUnicodeUtf8() throws IOException {
    //    checkReader( DATA_UTF8, ENCODING_UTF8, false );
    //}

    private void checkReader( byte[] rawData, String encoding, boolean unicode ) throws IOException {
        ILineReader reader = new LineReader( ByteBuffer.wrap( rawData ) );
        assertEquals( 0, reader.getLastLineEnd() );

        int LINE_0_END = LINE_0.length();
        int LINE_1_END = LINE_0.length() + 1 + LINE_1.length();
        int LINE_2_END = LINE_0.length() + 1 + LINE_1.length() + 2 + LINE_2.length();

        if ( unicode ) {
            LINE_0_END = LINE_0_END * 2 + 2;
            LINE_1_END = LINE_1_END * 2 + 2;
            LINE_2_END = LINE_2_END * 2 + 2;
        }

        checkReadLine( reader, LINE_0, rawData,
                reader.getNextLineStart(),
                LINE_0_END, encoding );

        //System.out.println( "reader = " + reader );

        checkReadLine( reader, LINE_1, rawData,
                reader.getNextLineStart(),
                LINE_1_END, encoding );

        checkReadLine( reader, LINE_2, rawData,
                reader.getNextLineStart(),
                LINE_2_END, encoding );

        assertNull( reader.readLine() );
    }


    private void checkReadLine( ILineReader reader, String expectedLine, byte[] rawData, int start, int end, String encoding ) throws IOException {
        byte[] expectedBytes = new byte[end - start];
        System.arraycopy( rawData, start, expectedBytes, 0, expectedBytes.length );

        ByteArray actualBytes = reader.readLine();

        assertEquals( expectedBytes, actualBytes.getBytes(), expectedBytes.length );

        assertEquals( expectedLine, new String( expectedBytes, encoding ) );
        assertEquals( expectedLine, new String( actualBytes.getBytes(), 0, actualBytes.getLength(), encoding ) );
        assertEquals( "wrong end", end, reader.getLastLineEnd() );
    }

    /*
        private static byte[] readWholeFile( String fileName ) throws IOException {
            byte[] bytes = new byte[(int)Files.length( fileName )];
            FileInputStream fis = new FileInputStream( fileName );
            try {
                assertEquals( bytes.length, fis.read( bytes ) );
                return bytes;
            } finally {
                fis.close();
            }
        }

        public static void main( String[] args ) throws IOException {
            byte[] is = readWholeFile( TXT_ISO_8859_1 );
            byte[] le = readWholeFile( TXT_UNICODE_LE );
            byte[] be = readWholeFile( TXT_UNICODE_BE );
            byte[] u8 = readWholeFile( TXT_UNICODE_UTF8 );

            String sis = new String( is, ENCODING_ISO );
            String sle = new String( le, 2, le.length - 2, ENCODING_UTF_16LE );
            String sbe = new String( be, 2, be.length - 2, ENCODING_UTF_16BE );
            String su8 = new String( u8, 3, u8.length - 3, ENCODING_UTF8 );

            byte[] is2 = DATA_0.getBytes(ENCODING_ISO );
            byte[] le2 = DATA_1.getBytes(ENCODING_UTF_16LE);
            byte[] be2 = DATA_1.getBytes(ENCODING_UTF_16BE);
            byte[] u82 = DATA_1.getBytes(ENCODING_UTF8 );

            le2 = prepend( le2, SIGNATURE_UTF_16LE );
            be2 = prepend( be2, SIGNATURE_UTF_16BE );
            u82 = prepend( u82, SIGNATURE_UTF8 );

            assertEquals( is, is2 );
            assertEquals( le, le2 );
            assertEquals( be, be2 );
            assertEquals( u8, u82 );
        }
    */

    private static void assertEquals( byte[] expected, byte[] actual, int length ) {
        String expectedAndActual = "\nExpected: " + ArrayUtil.asList( expected )
                + "\nActual  : " + ArrayUtil.asList( actual, 0, length );

        if ( expected.length != actual.length ) {
            fail( "Different size. Expected: " + expected.length + ". Actual " + actual.length
                    + expectedAndActual );
        }
        for ( int i = 0; i < expected.length; i++ ) {
            if ( expected[i] != actual[i] ) {
                fail( "Difference in arrays at index " + i + ". Expected element: "
                        + expected[i] + ". Actual element: " + actual[i]
                        + expectedAndActual );
            }
        }
    }

    /*
        public static void main( String[] args ) throws IOException, InterruptedException {
            Thread.sleep( 2000 );

            TimeMeasurer measurer = new TimeMeasurer();

            LineReader reader = new LineReader( new FileInputStream( "english3.sql" ) );
            byte[] bytes;
            while ( (bytes = reader.readLine()) != null ) {
                ;
            }

            System.out.println( "measurer = " + measurer );
        }
    */
}