/*
 * JaLingo, http://jalingo.sourceforge.net/
 *
 * Copyright (c) 2002-2006 Oleksandr Shyshko
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package ja.lingo.readers.ptkdictmysql;

import ja.JaTestCase;
import ja.centre.util.io.Files;
import ja.centre.util.measurer.TimeMeasurer;
import ja.lingo.engine.beans.Info;
import ja.lingo.engine.dictionaryindex.builder.MockDictionaryIndexBuilder;
import ja.lingo.engine.monitor.NullMonitor;
import ja.lingo.engine.reader.IDictionaryReader;
import ja.lingo.readers.mova.MovaReader;

import java.io.IOException;

public class PtkDictMySqlParserTest extends JaTestCase {
    private PtkDictMySqlParser reader;

    private static final String INSERT_INTO_TITLE = "within \\'\"\" an inch";
    private static final String INSERT_INTO_VALUE = "within an \\'\"\"inch: of one\\'s \\',\\' .";
    private static final String INSERT_INTO = "INSERT INTO eng_rus_dicitonary1 VALUES (  854  ,  '" +
            INSERT_INTO_TITLE +
            "'  ,  '" +
            INSERT_INTO_VALUE +
            "'  );";
    private static final String INSERT_INTO_WRONG1 = "iNSERT INTO ";
    private static final String INSERT_INTO_WRONG2 = " INSERT INTO ";
    private static final String INSERT_INTO_WRONG3 = "INSERT INTO\t";

    private static final String INFO1 = "eng_rus_dicitonary1\\'\"\\nEnglish\\nl\\nRussian\\nr";
    private static final String INFO1_CONVERTED = "eng_rus_dicitonary1'\", English-Russian";

    private static final String INFO2 = "eng_rus_dictionary1\\'\"\\nEnglish\\nr\\nRussian\\nr";
    private static final String INFO2_CONVERTED = "eng_rus_dictionary1'\", English-Russian";

    protected void setUp() throws Exception {
        super.setUp();

        reader = new PtkDictMySqlParser();
    }

    public void testIsEmptyOrComment() {
        assertTrue( reader.isEmptyOrComment( "" ) );
        assertTrue( reader.isEmptyOrComment( "#" ) );
        assertTrue( reader.isEmptyOrComment( "# some comments" ) );
        assertFalse( reader.isEmptyOrComment( "some data" ) );
    }

    public void testIsInsert() {
        assertTrue( reader.isInsert( INSERT_INTO ) );
        assertFalse( reader.isInsert( INSERT_INTO_WRONG1 ) );
        assertFalse( reader.isInsert( INSERT_INTO_WRONG2 ) );
        assertFalse( reader.isInsert( INSERT_INTO_WRONG3 ) );
    }

    public void testExtractInsert() {
        String[] values = reader.extractInsertAsStrings( INSERT_INTO );
        assertEquals( INSERT_INTO_TITLE, values[0] );
        assertEquals( INSERT_INTO_VALUE, values[1] );
    }

    public void testIsInsertInfo() throws Exception {
        assertEquals( INFO1_CONVERTED, reader.convertInfoDecription( INFO1 ) );
        assertEquals( INFO2_CONVERTED, reader.convertInfoDecription( INFO2 ) );
    }

    public static void main( String[] args ) throws IOException, InterruptedException {
        System.out.println( "Warming JVM for 5 seconds..." );
        Thread.sleep( 5000 );
        //System.in.read();

        String fileName = "60mb.koi";
        IDictionaryReader reader = new MovaReader(); // 4.5 - 9.4 MB/s

        //String fileName = "english3.sql";
        //IDictionaryReader reader = new PtkDictMySqlReader(); // 5.1 - 8.6 MB/s

        TimeMeasurer measurer = new TimeMeasurer();

        reader.createParser( new Info( fileName, "KOI8-R" ), new NullMonitor() ).buildIndex( new MockDictionaryIndexBuilder() );

        System.out.println( "measurer = " + measurer );

        long length = Files.length( fileName ) / (1024 * 1024);
        double speed = length / measurer.getDeltaSeconds();
        System.out.println( "length = " + length + " MB" );
        System.out.println( "speed = " + speed + " MB/s" );

        //System.in.read();
    }
}
