/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.engine.scheduling.domain;

import com.jaspersoft.jasperserver.api.JasperServerAPI;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;

import java.io.Serializable;

/**
 * Contains attributes related to the generation of repository resources
 * for report job output files.
 * 
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ReportJobRepositoryDestination.java 19921 2010-12-11 14:52:49Z tmatyashovsky $
 * @since 1.0
 * @see ReportJob#getContentRepositoryDestination()
 */
@JasperServerAPI
public class ReportJobRepositoryDestination implements Serializable {

	private static final long serialVersionUID = 1L;

	private long id;
	private int version = ReportJob.VERSION_NEW;
	private String folderURI;
	private boolean sequentialFilenames;
	private boolean overwriteFiles;
	private String outputDescription;
	private String timestampPattern;

	/**
	 * Creates an empty object.
	 */
	public ReportJobRepositoryDestination() {
	}

	/**
	 * Returns the ID of this object.
	 * 
	 * <p>
	 * The ID is automatically generated when the job for which the notification
	 * is defined gets saved in the scheduler.
	 * </p>
	 * 
	 * @return the ID of this object
	 */
	public long getId() {
		return id;
	}

	/**
	 * Sets the ID of this object.
	 * 
	 * <p>
	 * The ID needs to be set when updating a report job, but not when saving
	 * a new job.
	 * </p>
	 * 
	 * @param id the object ID
	 */
	public void setId(long id) {
		this.id = id;
	}

	/**
	 * Returns the persistent version of this object.
	 * 
	 * <p>
	 * The version is automatically set when a repository destination object is retrieved
	 * (along with the job) from the scheduling service.
	 * </p>
	 * 
	 * @return the persistent version
	 * @see ReportJob#VERSION_NEW
	 */
	public int getVersion() {
		return version;
	}

	/**
	 * Sets the persistent version for this object.
	 * 
	 * <p>
	 * The version doesn't have to be set when saving a new report job,
	 * but it needs to be set when updating a report job (along with its 
	 * repository destination object).
	 * </p>
	 * 
	 * @param version the persistent repository destination version
	 */
	public void setVersion(int version) {
		this.version = version;
	}

	/**
	 * Returns the repository URI/path of the folder under which job output
	 * resources would be created.
	 * 
	 * @return the repository output folder
	 */
	public String getFolderURI() {
		return folderURI;
	}

	/**
	 * Returns the repository URI/path of the folder under which job output
	 * resources are to be be created.
	 * 
	 * <p>
	 * The job owner should have write permission on the output folder.
	 * </p>
	 * 
	 * @param folder the URI/path of the repository output folder
	 * @see ReportJob#setBaseOutputFilename(String)
	 */
	public void setFolderURI(String folder) {
		this.folderURI = folder;
	}
	
	/**
	 * Decides if a timestamp is to be added to the names of the job output 
	 * resources.
	 * 
	 * <p>
	 * The timestamp added to the output resource names are created from the 
	 * job execution time using the specified pattern.
	 * </p>
	 * 
	 * @return <code>true</code> if the job output resource names should include
	 * a timestamp
	 * @see ReportJob#getBaseOutputFilename()
	 * @see #getTimestampPattern()
	 */
	public boolean isSequentialFilenames() {
		return sequentialFilenames;
	}

	/**
	 * Specifies whether a timestamp is to be added to the names of the job 
	 * output resources. 
	 * 
	 * <p>
	 * This is usually required when a job occurs severa time and the output
	 * from each execution needs to be kept in the repository.
	 * </p>
	 * 
	 * @param sequentialFilenames <code>true</code> if the job output resource 
	 * names should include a timestamp
	 */
	public void setSequentialFilenames(boolean sequentialFilenames) {
		this.sequentialFilenames = sequentialFilenames;
	}

	/**
	 * Decides whether the scheduler can overwrite files in the repository when
	 * saving job output resources.
	 * 
	 * <p>
	 * If the flag is not set, the job would fail if the repository already
	 * contains a resource with the same name as one of the job output resources.
	 * If the flag is set and the job owner does not have the permission to 
	 * overwrite an existing resource, the job execution will also fail.
	 * </p>
	 * 
	 * @return <code>true</code> if the scheduler should overwrite repository
	 * resources that have the same names as the job output resources
	 */
	public boolean isOverwriteFiles() {
		return overwriteFiles;
	}

	/**
	 * Specifies whether the scheduler should overwrite files in the repository
	 * when saving job output resources.
	 * 
	 * @param overwriteFiles
	 * @see #isOverwriteFiles()
	 */
	public void setOverwriteFiles(boolean overwriteFiles) {
		this.overwriteFiles = overwriteFiles;
	}

	/**
	 * Returns a description to be used for job output resources.
	 * 
	 * <p>
	 * The description will be used as is for all output resources.
	 * </p>
	 * 
	 * @return the output resource description, or <code>null<code> if not set
	 * @since 3.0
	 * @see Resource#getDescription()
	 */
	public String getOutputDescription() {
		return outputDescription;
	}

	/**
	 * Sets the description that should be used for job output resources.
	 * 
	 * @param outputDescription the job output resources description
	 * @since 3.0
	 */
	public void setOutputDescription(String outputDescription) {
		this.outputDescription = outputDescription;
	}

	/**
	 * Returns the pattern to be used for the timestamp included in the job
	 * output resource names.
	 * 
	 * <p>
	 * If no pattern has been specified, <code>yyyyMMddHHmm</code> will be used.
	 * </p>
	 * 
	 * @return the timestamp pattern to use for the output resource names
	 * @since 3.0
	 * @see #isSequentialFilenames()
	 */
	public String getTimestampPattern() {
		return timestampPattern;
	}

	/**
	 * Sets a date pattern to be used for the timestamp included in job output
	 * resource names.
	 * 
	 * <p>
	 * The pattern should be a valid pattern as defined by 
	 * <code>java.text.SimpleDateFormat</code> and can only contain underscores,
	 * dots and dashes as token separators.
	 * </p>
	 * 
	 * @param timestampPattern
	 * @since 3.0
	 * @see #setSequentialFilenames(boolean)
	 */
	public void setTimestampPattern(String timestampPattern) {
		this.timestampPattern = timestampPattern;
	}
}
