/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.security.userdetails.jdbc;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.sql.DataSource;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.security.AccessDeniedException;
import org.springframework.security.Authentication;
import org.springframework.security.AuthenticationManager;
import org.springframework.security.BadCredentialsException;
import org.springframework.security.GrantedAuthority;
import org.springframework.security.GrantedAuthorityImpl;
import org.springframework.security.MockAuthenticationManager;
import org.springframework.security.PopulatedDatabase;
import org.springframework.security.TestDataSource;
import org.springframework.security.context.SecurityContextHolder;
import org.springframework.security.providers.UsernamePasswordAuthenticationToken;
import org.springframework.security.providers.dao.UserCache;
import org.springframework.security.userdetails.User;
import org.springframework.security.userdetails.UserDetails;
import org.springframework.security.userdetails.jdbc.JdbcUserDetailsManager;
import org.springframework.security.util.AuthorityUtils;

public class JdbcUserDetailsManagerTests {
    private static final String SELECT_JOE_SQL = "select * from users where username = 'joe'";
    private static final String SELECT_JOE_AUTHORITIES_SQL = "select * from authorities where username = 'joe'";
    private static final UserDetails joe = new User("joe", "password", true, true, true, true, AuthorityUtils.stringArrayToAuthorityArray((String[])new String[]{"A", "C", "B"}));
    private static TestDataSource dataSource;
    private JdbcUserDetailsManager manager;
    private MockUserCache cache;
    private JdbcTemplate template;

    @BeforeClass
    public static void createDataSource() {
        dataSource = new TestDataSource("jdbcusermgrtest");
    }

    @AfterClass
    public static void clearDataSource() throws Exception {
        dataSource.destroy();
        dataSource = null;
    }

    @Before
    public void initializeManagerAndCreateTables() {
        this.manager = new JdbcUserDetailsManager();
        this.cache = new MockUserCache();
        this.manager.setUserCache((UserCache)this.cache);
        this.manager.setDataSource((DataSource)((Object)dataSource));
        this.manager.setCreateUserSql("insert into users (username, password, enabled) values (?,?,?)");
        this.manager.setUpdateUserSql("update users set password = ?, enabled = ? where username = ?");
        this.manager.setUserExistsSql("select username from users where username = ?");
        this.manager.setCreateAuthoritySql("insert into authorities (username, authority) values (?,?)");
        this.manager.setDeleteUserAuthoritiesSql("delete from authorities where username = ?");
        this.manager.setDeleteUserSql("delete from users where username = ?");
        this.manager.setChangePasswordSql("update users set password = ? where username = ?");
        this.manager.initDao();
        this.template = this.manager.getJdbcTemplate();
        this.template.execute("create table users(username varchar(20) not null primary key,password varchar(20) not null, enabled boolean not null)");
        this.template.execute("create table authorities (username varchar(20) not null, authority varchar(20) not null, constraint fk_authorities_users foreign key(username) references users(username))");
        PopulatedDatabase.createGroupTables(this.template);
        PopulatedDatabase.insertGroupData(this.template);
    }

    @After
    public void dropTablesAndClearContext() {
        this.template.execute("drop table authorities");
        this.template.execute("drop table users");
        this.template.execute("drop table group_authorities");
        this.template.execute("drop table group_members");
        this.template.execute("drop table groups");
        SecurityContextHolder.clearContext();
    }

    @Test
    public void createUserInsertsCorrectData() {
        this.manager.createUser(joe);
        UserDetails joe2 = this.manager.loadUserByUsername("joe");
        Assert.assertEquals((Object)joe, (Object)joe2);
    }

    @Test
    public void deleteUserRemovesUserDataAndAuthoritiesAndClearsCache() {
        this.insertJoe();
        this.manager.deleteUser("joe");
        Assert.assertEquals((long)0L, (long)this.template.queryForList(SELECT_JOE_SQL).size());
        Assert.assertEquals((long)0L, (long)this.template.queryForList(SELECT_JOE_AUTHORITIES_SQL).size());
        Assert.assertFalse((boolean)this.cache.getUserMap().containsKey("joe"));
    }

    @Test
    public void updateUserChangesDataCorrectlyAndClearsCache() {
        this.insertJoe();
        User newJoe = new User("joe", "newpassword", false, true, true, true, AuthorityUtils.stringArrayToAuthorityArray((String[])new String[]{"D", "F", "E"}));
        this.manager.updateUser((UserDetails)newJoe);
        UserDetails joe = this.manager.loadUserByUsername("joe");
        Assert.assertEquals((Object)newJoe, (Object)joe);
        Assert.assertFalse((boolean)this.cache.getUserMap().containsKey("joe"));
    }

    @Test
    public void userExistsReturnsFalseForNonExistentUsername() {
        Assert.assertFalse((boolean)this.manager.userExists("joe"));
    }

    @Test
    public void userExistsReturnsTrueForExistingUsername() {
        this.insertJoe();
        Assert.assertTrue((boolean)this.manager.userExists("joe"));
        Assert.assertTrue((boolean)this.cache.getUserMap().containsKey("joe"));
    }

    @Test(expected=AccessDeniedException.class)
    public void changePasswordFailsForUnauthenticatedUser() {
        this.manager.changePassword("password", "newPassword");
    }

    @Test
    public void changePasswordSucceedsWithAuthenticatedUserAndNoAuthenticationManagerSet() {
        this.insertJoe();
        this.authenticateJoe();
        this.manager.changePassword("wrongpassword", "newPassword");
        UserDetails newJoe = this.manager.loadUserByUsername("joe");
        Assert.assertEquals((Object)"newPassword", (Object)newJoe.getPassword());
        Assert.assertFalse((boolean)this.cache.getUserMap().containsKey("joe"));
    }

    @Test
    public void changePasswordSucceedsWithIfReAuthenticationSucceeds() {
        this.insertJoe();
        Authentication currentAuth = this.authenticateJoe();
        this.manager.setAuthenticationManager((AuthenticationManager)new MockAuthenticationManager(true));
        this.manager.changePassword("password", "newPassword");
        UserDetails newJoe = this.manager.loadUserByUsername("joe");
        Assert.assertEquals((Object)"newPassword", (Object)newJoe.getPassword());
        Authentication newAuth = SecurityContextHolder.getContext().getAuthentication();
        Assert.assertEquals((Object)"joe", (Object)newAuth.getName());
        Assert.assertEquals((Object)currentAuth.getDetails(), (Object)newAuth.getDetails());
        Assert.assertEquals((Object)"newPassword", (Object)newAuth.getCredentials());
        Assert.assertFalse((boolean)this.cache.getUserMap().containsKey("joe"));
    }

    @Test
    public void changePasswordFailsIfReAuthenticationFails() {
        this.insertJoe();
        this.authenticateJoe();
        this.manager.setAuthenticationManager((AuthenticationManager)new MockAuthenticationManager(false));
        try {
            this.manager.changePassword("password", "newPassword");
            Assert.fail((String)"Expected BadCredentialsException");
        }
        catch (BadCredentialsException expected) {
            // empty catch block
        }
        UserDetails newJoe = this.manager.loadUserByUsername("joe");
        Assert.assertEquals((Object)"password", (Object)newJoe.getPassword());
        Assert.assertEquals((Object)"password", (Object)SecurityContextHolder.getContext().getAuthentication().getCredentials());
        Assert.assertTrue((boolean)this.cache.getUserMap().containsKey("joe"));
    }

    @Test
    public void findAllGroupsReturnsExpectedGroupNames() {
        ArrayList<String> groups = new ArrayList<String>(Arrays.asList(this.manager.findAllGroups()));
        Assert.assertEquals((long)4L, (long)groups.size());
        Collections.sort(groups);
        Assert.assertEquals((Object)"GROUP_0", groups.get(0));
        Assert.assertEquals((Object)"GROUP_1", groups.get(1));
        Assert.assertEquals((Object)"GROUP_2", groups.get(2));
        Assert.assertEquals((Object)"GROUP_3", groups.get(3));
    }

    @Test
    public void findGroupMembersReturnsCorrectData() {
        String[] groupMembers = this.manager.findUsersInGroup("GROUP_0");
        Assert.assertEquals((long)1L, (long)groupMembers.length);
        Assert.assertEquals((Object)"jerry", (Object)groupMembers[0]);
        groupMembers = this.manager.findUsersInGroup("GROUP_1");
        Assert.assertEquals((long)2L, (long)groupMembers.length);
    }

    @Test
    public void createGroupInsertsCorrectData() {
        this.manager.createGroup("TEST_GROUP", AuthorityUtils.stringArrayToAuthorityArray((String[])new String[]{"ROLE_X", "ROLE_Y"}));
        List roles = this.template.queryForList("select ga.authority from groups g, group_authorities ga where ga.group_id = g.id and g.group_name = 'TEST_GROUP'");
        Assert.assertEquals((long)2L, (long)roles.size());
    }

    @Test
    public void deleteGroupRemovesData() throws Exception {
        this.manager.deleteGroup("GROUP_0");
        this.manager.deleteGroup("GROUP_1");
        this.manager.deleteGroup("GROUP_2");
        this.manager.deleteGroup("GROUP_3");
        Assert.assertEquals((long)0L, (long)this.template.queryForList("select * from group_authorities").size());
        Assert.assertEquals((long)0L, (long)this.template.queryForList("select * from group_members").size());
        Assert.assertEquals((long)0L, (long)this.template.queryForList("select id from groups").size());
    }

    @Test
    public void renameGroupIsSuccessful() throws Exception {
        this.manager.renameGroup("GROUP_0", "GROUP_X");
        Assert.assertEquals((long)0L, (long)this.template.queryForInt("select id from groups where group_name = 'GROUP_X'"));
    }

    @Test
    public void addingGroupUserSetsCorrectData() throws Exception {
        this.manager.addUserToGroup("tom", "GROUP_0");
        Assert.assertEquals((long)2L, (long)this.template.queryForList("select username from group_members where group_id = 0").size());
    }

    @Test
    public void removeUserFromGroupDeletesGroupMemberRow() throws Exception {
        this.manager.removeUserFromGroup("jerry", "GROUP_1");
        Assert.assertEquals((long)1L, (long)this.template.queryForList("select group_id from group_members where username = 'jerry'").size());
    }

    @Test
    public void findGroupAuthoritiesReturnsCorrectAuthorities() throws Exception {
        GrantedAuthority[] authorities = this.manager.findGroupAuthorities("GROUP_0");
        Assert.assertEquals((Object)"ROLE_A", (Object)authorities[0].getAuthority());
    }

    @Test
    public void addGroupAuthorityInsertsCorrectGroupAuthorityRow() throws Exception {
        GrantedAuthorityImpl auth = new GrantedAuthorityImpl("ROLE_X");
        this.manager.addGroupAuthority("GROUP_0", (GrantedAuthority)auth);
        this.template.queryForObject("select authority from group_authorities where authority = 'ROLE_X' and group_id = 0", String.class);
    }

    @Test
    public void deleteGroupAuthorityRemovesCorrectRows() throws Exception {
        GrantedAuthorityImpl auth = new GrantedAuthorityImpl("ROLE_A");
        this.manager.removeGroupAuthority("GROUP_0", (GrantedAuthority)auth);
        Assert.assertEquals((long)0L, (long)this.template.queryForList("select authority from group_authorities where group_id = 0").size());
        this.manager.removeGroupAuthority("GROUP_2", (GrantedAuthority)auth);
        Assert.assertEquals((long)2L, (long)this.template.queryForList("select authority from group_authorities where group_id = 2").size());
    }

    private Authentication authenticateJoe() {
        UsernamePasswordAuthenticationToken auth = new UsernamePasswordAuthenticationToken((Object)"joe", (Object)"password", joe.getAuthorities());
        SecurityContextHolder.getContext().setAuthentication((Authentication)auth);
        return auth;
    }

    private void insertJoe() {
        this.template.execute("insert into users (username, password, enabled) values ('joe','password','true')");
        this.template.execute("insert into authorities (username, authority) values ('joe','A')");
        this.template.execute("insert into authorities (username, authority) values ('joe','B')");
        this.template.execute("insert into authorities (username, authority) values ('joe','C')");
        this.cache.putUserInCache(joe);
    }

    private class MockUserCache
    implements UserCache {
        private Map<String, UserDetails> cache = new HashMap<String, UserDetails>();

        private MockUserCache() {
        }

        public UserDetails getUserFromCache(String username) {
            return (User)this.cache.get(username);
        }

        public void putUserInCache(UserDetails user) {
            this.cache.put(user.getUsername(), user);
        }

        public void removeUserFromCache(String username) {
            this.cache.remove(username);
        }

        Map getUserMap() {
            return this.cache;
        }
    }
}

