/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.engine.scheduling.service;

import java.util.List;

import com.jaspersoft.jasperserver.api.JasperServerAPI;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJob;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobIdHolder;
import com.jaspersoft.jasperserver.api.engine.scheduling.domain.ReportJobSummary;

/**
 * A service that persists report job details.
 * 
 * <p>
 * This service would not be directly called by code that does report scheduling
 * in regular scenarios, such code would normally use {@link ReportSchedulingService}.
 * This persistence service is used internal by the central scheduling service.
 * </p>
 * 
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: ReportJobsPersistenceService.java 19921 2010-12-11 14:52:49Z tmatyashovsky $
 * @since 1.0
 * @see ReportSchedulingService
 * @see ReportJobsScheduler
 */
@JasperServerAPI
public interface ReportJobsPersistenceService {

	/**
	 * Saves the details of a newly created report job.
	 * 
	 * <p>
	 * An ID for the job will be automatically generated and returned as part of
	 * the saved job.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param job the job details to save
	 * @return the job details as saved
	 * @see ReportSchedulingService#scheduleJob(ExecutionContext, ReportJob)
	 */
	ReportJob saveJob(ExecutionContext context, ReportJob job);

	/**
	 * Updates the details of an existing report job.
	 * 
	 * <p>
     * The existing job is determined by the ID of the job argument.
     * </p>
	 * 
     * @param context the caller execution context
	 * @param job the job details to update 
	 * @return the job as updated
     * @throws ReportJobNotFoundException when a job with the specified ID is 
     * not found
     * @see ReportSchedulingService#updateScheduledJob(ExecutionContext, ReportJob)
	 */
	ReportJob updateJob(ExecutionContext context, ReportJob job)
			throws ReportJobNotFoundException;
	
	/**
	 * Loads the details of a report job.
	 * 
     * @param context the caller execution context
	 * @param jobId the ID of the job to load
	 * @return the details of the requested job, or <code>null</code> if no job
	 * with the specified ID was found
	 * @see ReportSchedulingService#getScheduledJob(ExecutionContext, long)
	 */
	ReportJob loadJob(ExecutionContext context, ReportJobIdHolder jobId);

	/**
	 * Removes a saved report job.
	 * 
	 * <p>
	 * If a job with the specified ID is not found, no operation is performed.
	 * </p>
	 * 
     * @param context the caller execution context
	 * @param jobId the ID of the job to delete
	 * @see ReportSchedulingService#removeScheduledJob(ExecutionContext, long)
	 */
	void deleteJob(ExecutionContext context, ReportJobIdHolder jobId);

	/**
     * Returns a list of jobs saved for a report.
     * 
     * @param context the caller execution context
     * @param reportUnitURI the repository URI/path of the report for which jobs
     * should be listed
     * @return a list of {@link ReportJobSummary} objects with no runtime
     * information
	 * @see ReportSchedulingService#getScheduledJobs(ExecutionContext, String)
	 */
	List listJobs(ExecutionContext context, String reportUnitURI);

	/**
     * Returns the list of all saved report jobs.
	 * 
     * @param context the caller execution context
     * @return a list of {@link ReportJobSummary} objects with no runtime
     * information
	 * @see ReportSchedulingService#getScheduledJobs(ExecutionContext)
	 */
	List listJobs(ExecutionContext context);

}
