/*
 * iReport - Visual Designer for JasperReports.
 * Copyright (C) 2002 - 2009 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com
 *
 * Unless you have purchased a commercial license agreement from Jaspersoft,
 * the following license terms apply:
 *
 * This program is part of iReport.
 *
 * iReport is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * iReport is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with iReport. If not, see <http://www.gnu.org/licenses/>.
 */


package com.jaspersoft.jasperserver.rest;

import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.OperationResult;
import com.jaspersoft.jasperserver.remote.AbstractService;
import java.io.BufferedInputStream;
import com.jaspersoft.jasperserver.remote.AbstractService;
import java.io.IOException;
import java.io.OutputStream;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.ServicesUtils;
import com.jaspersoft.jasperserver.ws.xml.Marshaller;
import java.io.PrintWriter;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.activation.DataSource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.multipart.MultipartResolver;
import org.springframework.web.multipart.commons.CommonsMultipartResolver;

/**
 *
 * @author gtoffoli
 */
public class RESTUtils {

    private final static Log log = LogFactory.getLog(RESTUtils.class);

    public static final String REQUEST_PARAMENTER_RD = "ResourceDescriptor";
    public static final String REQUEST_PARAMENTER_MOVE_TO = "moveTo";
    public static final String REQUEST_PARAMENTER_COPY_TO = "copyTo";


    /*
     * Convenient method to check if the current request is of type multipart.
     *
     * @param request
     * @return
     */
     public static boolean isMultipartContent(HttpServletRequest request) {
        if ( !(RESTAbstractService.HTTP_PUT.equals(request.getMethod().toLowerCase()) || RESTAbstractService.HTTP_POST.equals(request.getMethod().toLowerCase())) ) {
            return false;
        }
        String contentType = request.getContentType();
        if (contentType == null) {
            return false;
        }
        if (contentType.toLowerCase().startsWith("multipart/")) {
            return true;
        }
        return false;
    }

     
   /**
    * Extract the attachments from the request and put the in the list of
    * input attachments of the service.
    *
    * The method returns the currenst HttpServletRequest if the message is not
    * multipart, otherwise it returns a MultipartHttpServletRequest
    *
    *
    * @param context
    * @param hRequest
    */
    public static HttpServletRequest extractAttachments(AbstractService context, HttpServletRequest hRequest)
    {
          //Check whether we're dealing with a multipart request
          MultipartResolver resolver= new CommonsMultipartResolver();

          // Giulio: If the getContentLength is -1, avoid to consider this
          // message like a multipart request
          // resolver.isMultipart works only with POST, so we add our own check...

          if(isMultipartContent(hRequest) && hRequest.getContentLength() != -1)
          {
              MultipartHttpServletRequest mreq = resolver.resolveMultipart(hRequest);

              
              if(mreq!=null)
              {
                      Iterator iterator=mreq.getFileNames();
                      String fieldName=null;
                      while(iterator.hasNext()){
                              fieldName=(String)iterator.next();
                              MultipartFile file=mreq.getFile(fieldName);
                              if(file!=null){
                                  DataSource ds = new MultipartFileDataSource(file);
                                  context.getInputAttachments().put(fieldName, ds);
                              }
                      }

                  return mreq;
              }
            }

          return hRequest;
        
    }


    /**
     * Return the service name from the request.
     * The service name is the first item in the path, i.e.
     * /serviceName/somethingelse
     * 
     *
     *
     * @param path
     * @return The service name, or an empty string if the service name is empty
     */
    public static String extractServiceName(String path)
    {
        if (path == null || path.length() == 0) return "";
        if (path.startsWith("/"))
        {
            path = path.substring(1);
        }
        int startIndex = path.indexOf("/");
        if (startIndex > 0)
        {
            path = path.substring(0, startIndex);
        }

        return path;
    }

    /**
     * Return the repository uri from a pathInfo
     * The repository uri is the part of the url after the
     * service name
     *
     * /serviceName/this/it/the/uri
     *
     * The method looks for the first occurence of a / after
     * the service name.
     *
     * Any final / is removed (until the uri represent the root path).
     *
     * @param path
     * @return a path without the last / or just the root path (/) if a valid path is not found.
     */
    public static String extractRepositoryUri(String path)
    {
        if (path == null) return "/";

        if (path.startsWith("/"))
        {
            path = path.substring(1);
        }
        int startIndex = path.indexOf("/");
        if (startIndex > 0)
        {
            path = path.substring(startIndex);
        }

        while (path.length() > 1 && path.endsWith("/"))
        {
            path = path.substring(0,path.length()-1);
        }

        return path;
    }

    // removes \ and / from the string
    public static String getURIName(String uri){
        return (uri.replace("\\", "")).replace("//", "");
    }


   /**
    * Look for parameters provided by the client.
    *
    * @param req  HttpServletRequest
    * @return Map&lt;String,Object&gt;
    */
    public static Map<String,Object> extractParameters(HttpServletRequest req)
    {
        Map<String,Object> parameters = new HashMap<String,Object>();
        Enumeration penum = req.getParameterNames();
        while (penum.hasMoreElements())
        {
            String pname = (String) penum.nextElement();
            if (pname.startsWith("P_"))
            {
                parameters.put(pname.substring(2), req.getParameter(pname));
            }
            else if(pname.startsWith("PL_"))
            {
                parameters.put(pname.substring(3), Arrays.asList(req.getParameterValues(pname)));
            }
        }
        return parameters;
    }

    /**
     * Set the status of the response to the errorCode and send the message to the client
     *
     * @param errorCode the errorCode (see HttpServletResponse for common HTTP status codes)
     * @param response the HttpServletResponse
     * @param body if null, an empty string is sent.
     */
    public static void setStatusAndBody(int errorCode, HttpServletResponse response, String body)
    {
        response.setStatus( errorCode );
        // we can put an error in the output too (so avoid the webserver to display unwanted pages...)
        try {
            PrintWriter pw = response.getWriter();
            pw.print((body == null) ? "" : body);
        } catch (Exception ioEx)
        {
            // do nothing. If we have an I/O error, we just avoid to specify unuseful errors here.
            log.error("Error sending output a file",ioEx);
        }
    }


    public static void sendFile(DataSource ds, HttpServletResponse response)
    {
        response.setContentType( ds.getContentType() );

        if (ds.getName() != null && ds.getName().length() > 0)
        {
            response.addHeader("Content-Disposition", "attachment; filename=" + ds.getName());
        }

        OutputStream outputStream = null;
        BufferedInputStream bufferedInputStream = null;
        try {
             outputStream = response.getOutputStream();

             bufferedInputStream = new BufferedInputStream(ds.getInputStream());
             int readBytes = 0;

             while ((readBytes = bufferedInputStream.read()) != -1)
             {
                outputStream.write(readBytes);
             }


        } catch (IOException ex) {
            log.error("Error serving a file: " + ex.getMessage(), ex);
        } finally {
          if (outputStream != null) try { outputStream.close(); } catch (Exception ex) {}
          if (bufferedInputStream != null) try {  bufferedInputStream.close(); } catch (Exception ex) {}
        }
    }

    public static String getDetinationUri(HttpServletRequest req){ // for action that specify the destination of the action in the request
        if ( req.getParameterMap().containsKey(REQUEST_PARAMENTER_MOVE_TO) )
            return req.getParameter(REQUEST_PARAMENTER_MOVE_TO);

        if ( req.getParameterMap().containsKey(REQUEST_PARAMENTER_COPY_TO) )
            return req.getParameter(REQUEST_PARAMENTER_COPY_TO);

        return null;
    }
}
