/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.war.action;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.impl.JasperReportInputControlInformation;
import com.jaspersoft.jasperserver.api.engine.jasperreports.service.impl.ReportInputControlsInformationImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.DataTypeImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.InputControlImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.QueryImpl;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.war.BaseUnitTest;
import com.jaspersoft.jasperserver.war.dto.RuntimeInputControlWrapper;
import com.jaspersoft.jasperserver.war.util.DefaultCalendarFormatProvider;
import net.sf.jasperreports.engine.JRParameter;
import org.apache.commons.collections.OrderedMap;
import org.apache.commons.collections.map.LinkedMap;
import org.junit.Before;
import org.junit.Test;
import org.springframework.webflow.context.servlet.ServletExternalContext;
import org.springframework.webflow.core.collection.MutableAttributeMap;
import org.springframework.webflow.core.collection.SharedAttributeMap;
import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;
import org.unitils.mock.Mock;
import org.unitils.mock.MockUnitils;
import org.unitils.mock.PartialMock;
import org.unitils.mock.mockbehavior.MockBehavior;
import org.unitils.mock.proxy.ProxyInvocation;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.jaspersoft.jasperserver.war.action.ReportParametersActionTestQueryExecutor.execute;
import static com.jaspersoft.jasperserver.war.action.ReportParametersActionTestQueryExecutor.feedResults;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.CASCADE_ACCOUNT_TYPE;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.CASCADE_COUNTRY;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.CASCADE_INDUSTRY;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.CASCADE_NAME;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.CASCADE_STATE;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.TEXT_DATE;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.TEXT_DATE_TIME;
import static com.jaspersoft.jasperserver.war.action.ViewReportActionTest.Parameters.STRING;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * @author Vladimir Tsukur
 * @author Anton Fomin
 * @version $Id: ViewReportActionTest.java 20827 2011-08-10 13:36:14Z afomin $
 */
public class ViewReportActionTest extends BaseUnitTest {

    private PartialMock<ViewReportAction> viewReportActionMock;

    private Mock<RequestContext> requestContext;

    private Mock<MutableAttributeMap> mutableAttributeMap;

    private Mock<ServletExternalContext> externalContext;

    private Mock<SharedAttributeMap> sessionMap;

    private Mock<Map> simpleSessionMap;

    private Mock<Map> sessionUUIDMap;

    private Mock<HttpServletRequest> httpServletRequest;

    private static final String DEFAULT_BOOLEAN_PARAMETER_NAME = "boolean";

    private Mock<ExecutionContext> executionContext;

    private Parameters parameters;

    private Mock<ReportUnit> reportUnit;

    @Test
    public void parseRequestParametersForBooleanWhenValueIsNotNull() {
        List<RuntimeInputControlWrapper> wrappers = Arrays.asList(createBooleanInputControlWrapper());

        Map<String, String[]> parameterMap = new HashMap<String, String[]>();
        parameterMap.put(DEFAULT_BOOLEAN_PARAMETER_NAME, new String[] { "not-null-value" });

        programStandardParseRequestParametersMocking(wrappers, parameterMap);

        Event event = getAction().parseRequestParameters(requestContext.getMock());
        assertEquals(event.getId(), "success");
        assertEquals(wrappers.get(0).getValue(), true);
    }

    @Test
    public void parseRequestParametersForBooleanWhenValueIsNull() {
        List<RuntimeInputControlWrapper> wrappers = Arrays.asList(createBooleanInputControlWrapper());

        Map<String, String[]> parameterMap = new HashMap<String, String[]>();
        parameterMap.put(DEFAULT_BOOLEAN_PARAMETER_NAME, new String[] { null });

        programStandardParseRequestParametersMocking(wrappers, parameterMap);

        Event event = getAction().parseRequestParameters(requestContext.getMock());
        assertEquals(event.getId(), "success");
        assertEquals(wrappers.get(0).getValue(), false);
    }
    
    private RuntimeInputControlWrapper createBooleanInputControlWrapper() {
        InputControl inputControl = new InputControlImpl();
        inputControl.setName(DEFAULT_BOOLEAN_PARAMETER_NAME);
        inputControl.setType(InputControl.TYPE_BOOLEAN);
        return new RuntimeInputControlWrapper(inputControl);
    }

    private void programStandardParseRequestParametersMocking(List wrappers, Map parameterMap) {
        String wrappersFlowScopeUUID = "UUID";
        requestContext.returns(mutableAttributeMap).getFlowScope();
        mutableAttributeMap.returns(wrappersFlowScopeUUID).get(ReportParametersAction.WRAPPERS_FLOW_SCOPE_UUID);
        requestContext.returns(externalContext).getExternalContext();
        externalContext.returns(sessionMap).getSessionMap();
        sessionMap.returns(simpleSessionMap).asMap();
        simpleSessionMap.returns(sessionUUIDMap).get(ReportParametersAction.INPUTWRAPPERS_ATTR);
        sessionUUIDMap.returns(wrappers).get(wrappersFlowScopeUUID);

        mutableAttributeMap.returns(wrappers).get(ReportParametersAction.INPUTWRAPPERS_ATTR);
        externalContext.returns(httpServletRequest).getNativeRequest();
        httpServletRequest.returns(parameterMap).getParameterMap();
    }

    /**
     * Tests processParamValue
     * using createDefaultValueAsFirstListResult and isValuePresentInList
     */
    @Test
    public void processParamValue() {
        /* Create sample controls only with needed properties */
        InputControl singleControl = new InputControlImpl();
        singleControl.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);
        singleControl.setMandatory(true);
        InputControl multiControl = new InputControlImpl();
        multiControl.setType(InputControl.TYPE_MULTI_SELECT_QUERY);
        multiControl.setMandatory(true);

        /* Create sample wrappers */
        RuntimeInputControlWrapper singleWrapper = new RuntimeInputControlWrapper(singleControl);
        RuntimeInputControlWrapper multiWrapper = new RuntimeInputControlWrapper(multiControl);

        /* Use the same queryResults for wrappers, only just for test */
        Map queryResults = createQueryResults();
        singleWrapper.setQueryResults(queryResults);
        multiWrapper.setQueryResults(queryResults);

        /**
         * Test Multi Select control
         */

        /* Define test variables */
        List<String> valueList;
        Object methodResult;
        List expectedResult;

        /* All values are present in queryResults - return the same list */
        valueList = new ArrayList<String>();
        valueList.add("Galya");
        valueList.add("Sveta");
        valueList.add("Lena");
        methodResult = getAction().processParamValue(valueList, multiWrapper);
        assertEquals(methodResult, valueList);

        /* At least one value is present in queryResults - return the same list */
        valueList = new ArrayList<String>();
        valueList.add("Galya");
        valueList.add("Marina");
        methodResult = getAction().processParamValue(valueList, multiWrapper);
        assertEquals(methodResult, valueList);

        /* No values match the queryResults - return first value from queryResults */
        valueList = new ArrayList<String>();
        valueList.add("Ira");
        valueList.add("Marina");
        methodResult = getAction().processParamValue(valueList, multiWrapper);
        expectedResult = new ArrayList<String>();
        expectedResult.add("Galya");
        assertEquals(methodResult, expectedResult);

        /**
         * Test Single Select control
         */

        /* Define test variables */
        String value;

        /* Value is presented is queryResults list - return the same */
        value = "Natasha";
        methodResult = getAction().processParamValue(value, singleWrapper);
        assertEquals(methodResult, value);

        /* Value isn't present - return first one from queryResults */
        value = "Sasha";
        methodResult = getAction().processParamValue(value, singleWrapper);
        assertEquals(methodResult, "Galya");


        /* Check against null */
        assertNull(getAction().processParamValue("notNullValue", null));
        assertNull(getAction().processParamValue(null, singleWrapper));
        assertNull(getAction().processParamValue(null, null));

    }

    /**
     * Create sample queryResults for wrapper
     * @return
     *      Map which is instance of LinkedMap
     */
    private Map createQueryResults () {
        Map resultSet = new LinkedMap();
        resultSet.put("Galya", new String[] {"Galya", "Galya"});
        resultSet.put("Natasha", new String[] {"Natasha", "Natasha"});
        resultSet.put("Sveta", new String[] {"Sveta", "Sveta"});
        resultSet.put("Dasha", new String[] {"Dasha", "Dasha"});
        resultSet.put("Lena", new String[] {"Lena", "Lena"});
        return resultSet;
    }

    /**
     * Method ReportParametersAction.createWrappers() uses 3 types of parameter values:
     * - argument values - values coming as method argument,
     *                     in Spring Context these values are the same as default values taken somewhere before createWrappers,
     *                     but in Dashboard Runtime they are different from defaults (Dashboard frame values).
     * - default values -  default parameter values taken from InputControlInfo and other sources
     *                     but these are the same values which are defined in report's JRXML file;
     *                     Default values are available only when invoke using Spring context,
     *                     and in this case they override argument values.
     * - values from request - values passed in URL, it can be Dashboard frame or Drill link,
     *                     These values override argument and default values, sure if they're present :)
     *
     * We have few standard cases:
     * - report is running from Repository or AdHoc Runtime:
     *      - default values are used - they override argument values;
     * - report is running as Dashboard Frame or using a Drill Link:
     *      - Spring context isn't present;
     *      - values are present in URL, they override argument and default report parameter values;
     *          - values that aren't present in URL left default;
     *
     * There are variations around whether Default values are presented for parameters
     * and whether controls are Mandatory, these data are configurable in test.
     *
     * In this test we have 5 controls:
     * Country - multi select,
     * State - multi,
     * AccountType - single,
     * Industry - single,
     * Name - single.
     *
     * We assume that these controls are in cascade, we do not define queries for them in test,
     * we're using ReportParametersActionTestQueryExecutor instead
     * which knows that controls are in cascade and what values it should return for them.
     */

    /**
     * Webflow invocation
     * Parameters have defaults and are mandatory
     */
    @Test
    public void testCreateWrappersRuntime() {

        /* Argument values are overridden by default values when running in Spring context */
        /* parameters.setArgumentCascadeParameterValues(); */

        parameters.setDefaultParameterValues(list("USA"), list("CA"), "Distribution", "Engineering", "D & A Carmona Machinery Group");

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentCascadeParameterValues());

        assertWrappers(CASCADE_COUNTRY, list("USA"), list("USA", "Ukraine", "Canada"), wrappers);
        assertWrappers(CASCADE_STATE, list("CA"), list("CA", "OR", "WA"), wrappers);
        assertWrappers(CASCADE_ACCOUNT_TYPE, "Distribution", list("Consulting", "Distribution", "Manufactoring"), wrappers);
        assertWrappers(CASCADE_INDUSTRY, "Engineering", list("Communications", "Engineering", "Telecommunications"), wrappers);
        assertWrappers(CASCADE_NAME, "D & A Carmona Machinery Group", list("Dubois-Maestas Telecommunications, Ltd", "Azari-Dabit Telecommunications, Ltd", "D & A Carmona Machinery Group"), wrappers);
    }

    /**
     * Dashboard runtime invocation
     * Dashboard frames have values and controls are mandatory
     */
    @Test
    public void testCreateWrappersDashboard() {

        parameters.setArgumentCascadeParameterValues(list("USA"), list("CA"), "Distribution", "Engineering", "D & A Carmona Machinery Group");

        /* Defaults are ignored in Dashboard since requestContext is not available */
        /* parameters.setDefaultParameterValues(); */

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), null, reportUnit.getMock(), parameters.getArgumentCascadeParameterValues());

        assertWrappers(CASCADE_COUNTRY, list("USA"), list("USA", "Ukraine", "Canada"), wrappers);
        assertWrappers(CASCADE_STATE, list("CA"), list("CA", "OR", "WA"), wrappers);
        assertWrappers(CASCADE_ACCOUNT_TYPE, "Distribution", list("Consulting", "Distribution", "Manufactoring"), wrappers);
        assertWrappers(CASCADE_INDUSTRY, "Engineering", list("Communications", "Engineering", "Telecommunications"), wrappers);
        assertWrappers(CASCADE_NAME, "D & A Carmona Machinery Group", list("Dubois-Maestas Telecommunications, Ltd", "Azari-Dabit Telecommunications, Ltd", "D & A Carmona Machinery Group"), wrappers);
    }

    /**
     * Dashboard runtime invocation
     * Dashboard frames have all null values and controls are mandatory
     *
     * It's case when we're in dashboard using report with controls
     * which don't have default values and are not mandatory
     */
    @Test
    public void testCreateWrappersDashboardFrameValuesAreNull() {

        /* Frame values are all null */
        parameters.setArgumentCascadeParameterValues(null, null, null, null, null);

        /* Defaults are ignored in Dashboard since requestContext is not available */
        /* parameters.setDefaultParameterValues(); */

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), null, reportUnit.getMock(), parameters.getArgumentCascadeParameterValues());

        assertWrappers(CASCADE_COUNTRY, list("USA"), list("USA", "Ukraine", "Canada"), wrappers);
        assertWrappers(CASCADE_STATE, list("CA"), list("CA", "OR", "WA"), wrappers);
        assertWrappers(CASCADE_ACCOUNT_TYPE, "Consulting", list("Consulting", "Distribution", "Manufactoring"), wrappers);
        assertWrappers(CASCADE_INDUSTRY, "Engineering", list("Engineering", "Machinery"), wrappers);
        assertWrappers(CASCADE_NAME, "EngBureau, Ltd", list("EngBureau, Ltd", "BestEngineering, Inc", "SuperSoft, LLC"), wrappers);
    }

    /**
     * Dashboard runtime invocation
     * Dashboard frames have values and controls are mandatory
     *
     * We have one control here - single value type (non-query), class java.util.Date
     */
    @Test
    public void testCreateWrappersDashboardNonQueryControl() {

        /* This is default value from dashboard frame */
        parameters.setArgumentParameterValue(TEXT_DATE, "20110628000000");
        parameters.setArgumentParameterValue(TEXT_DATE_TIME, "20110628182200");

        /* Defaults are ignored in Dashboard since requestContext is not available */
        /* parameters.setDefaultParameterValues(); */

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), null, reportUnit.getMock(), parameters.getArgumentParameterValues(TEXT_DATE, TEXT_DATE_TIME));

        assertWrappers(TEXT_DATE, new GregorianCalendar(2011, 5, 28).getTime(), null, wrappers);
        assertWrappers(TEXT_DATE_TIME, new GregorianCalendar(2011, 5, 28, 18, 22).getTime(), null, wrappers);
    }

    /**
     * Run report with URL parameters of type Date with normal value
     */
    @Test
    public void testRequestDateValue() {

        /* This is default value from dashboard frame */
        /* parameters.setArgumentParameterValue(TEXT_DATE, "20110628000000"); */
        /* parameters.setArgumentParameterValue(TEXT_DATE_TIME, "20110628182200"); */

        /* Defaults are ignored in Dashboard since requestContext is not available */
        parameters.setDefaultParameterValue(TEXT_DATE, "20110628000000");
        parameters.setDefaultParameterValue(TEXT_DATE_TIME, "20110628182200");

        /* Set request parameter value */
        parameters.setRequestParameterValue(TEXT_DATE, "20120625000000");
        parameters.setRequestParameterValue(TEXT_DATE_TIME, "20120628182500");

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentParameterValues(TEXT_DATE, TEXT_DATE_TIME));

        assertWrappers(TEXT_DATE, new GregorianCalendar(2012, 5, 25).getTime(), null, wrappers);
        assertWrappers(TEXT_DATE_TIME, new GregorianCalendar(2012, 5, 28, 18, 25).getTime(), null, wrappers);
    }

    /**
     * Run report with URL parameters of type Date with empty value
     */
    @Test
    public void testRequestDateEmptyValue() {

        /* This is default value from dashboard frame */
        /* parameters.setArgumentParameterValue(TEXT_DATE, "20110628000000"); */
        /* parameters.setArgumentParameterValue(TEXT_DATE_TIME, "20110628182200"); */

        /* Defaults are ignored in Dashboard since requestContext is not available */
        parameters.setDefaultParameterValue(TEXT_DATE, "20110628000000");
        parameters.setDefaultParameterValue(TEXT_DATE_TIME, "20110628182200");

         /* Set request parameter value */
        parameters.setRequestParameterValue(TEXT_DATE, "");
        parameters.setRequestParameterValue(TEXT_DATE_TIME, "");

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentParameterValues(TEXT_DATE, TEXT_DATE_TIME));

        assertWrappers(TEXT_DATE, null, null, wrappers);
        assertWrappers(TEXT_DATE_TIME, null, null, wrappers);
    }

    /**
     * Run report with URL parameters of type String with normal value
     */
    @Test
    public void testRequestStringValue() {

        /* Argument value is overridden by default value when running in Spring context */
        /* parameters.setArgumentParameterValue(STRING, "defaultValue"); */

        /* Default value from jrxml */
        parameters.setDefaultParameterValue(STRING, "defaultValue");

        /* Set request parameter value */
        parameters.setRequestParameterValue(STRING, "valueFromRequest");

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentParameterValues(STRING));

        assertWrappers(STRING, "valueFromRequest", null, wrappers);
    }

    /**
     * Run report with URL parameters of type String with empty value
     */
    @Test
    public void testRequestStringEmptyValue() {

        /* Argument value is overridden by default value when running in Spring context */
        /* parameters.setArgumentParameterValue(STRING, "defaultValue"); */

        /* Default value from jrxml */
        parameters.setDefaultParameterValue(STRING, "defaultValue");

        /* Set request parameter value */
        parameters.setRequestParameterValue(STRING, "");

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentParameterValues(STRING));

        assertWrappers(STRING, "", null, wrappers);
    }

    /**
     * Run report with URL parameters of type String with null value
     */
    @Test
    public void testRequestStringNullValue() {

        /* Argument value is overridden by default value when running in Spring context */
        /* parameters.setArgumentParameterValue(STRING, "defaultValue"); */

        /* Default value from jrxml */
        parameters.setDefaultParameterValue(STRING, "defaultValue");

        /* Set request parameter value */
        parameters.setRequestParameterValue(STRING, null);

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentParameterValues(STRING));

        assertWrappers(STRING, null, null, wrappers);
    }

    /**
     * Run report with Default parameters of type String with null value
     */
    @Test
    public void testStringDefaultNullValue() {

        /* Argument value is overridden by default value when running in Spring context */
        /* parameters.setArgumentParameterValue(STRING, "defaultValue"); */

        /* Default value from jrxml */
        parameters.setDefaultParameterValue(STRING, null);

        List<RuntimeInputControlWrapper> wrappers = getAction()
                .createWrappers(executionContext.getMock(), requestContext.getMock(), reportUnit.getMock(), parameters.getArgumentParameterValues(STRING));

        assertWrappers(STRING, null, null, wrappers);
    }

    /**
     * Set up method
     * - creates Parameters helper object
     * - set behavior for needed ReportsParametersAction
     * - set behavior for requestContext and reportUnit entities
     */
    @Before
    public void initViewReportAction() {
        createParameters();
        overrideMethodsForCreateWrappers();
        requestContext.returns(mutableAttributeMap).getFlowScope();
        reportUnit.returns(parameters.getInputControlReferences()).getInputControls();
    }

    /**
     * Simple getter for action object
     * @return ViewReportAction
     */
    private ViewReportAction getAction() {
        return viewReportActionMock.getMock();
    }

    /**
     * Getter for PartialMock
     * @return PartialMock<ViewReportAction>
     */
    private PartialMock<ViewReportAction> getViewReportActionMock() {
        return viewReportActionMock;
    }

    /**
     * Set behavior for methods needed for testing createWrappers()
     */
    private void overrideMethodsForCreateWrappers() {
        overrideLoadInputControlsInformation(getViewReportActionMock());
        overrideGetWrapperValueFromRequest(getViewReportActionMock());
        overrideInitialValueProvider(getViewReportActionMock());
        overrideExecuteQuery(getViewReportActionMock());
        overrideGetCalendarFormatProvider(getViewReportActionMock());
        overrideGetStaticDatePattern(getViewReportActionMock());
    }

    /**
     * Simple wrapper for Arrays.asList()
     * @param values Array
     * @return List
     */
    private List<String> list(String... values) {
        return Arrays.asList(values);
    }

    /**
     * Set behavior for loadInputControlsInformation method for createWrappers test
     * @param actionMock partial action mock
     */
    private void overrideLoadInputControlsInformation(PartialMock<ViewReportAction> actionMock) {
        actionMock.performs(new MockBehavior() {
            public ReportInputControlsInformationImpl execute(ProxyInvocation invocation) {
                return parameters.getInputControlInfo();
            }
        }).loadInputControlsInformation(null, null, null);
    }

    /**
     * Set behavior for getWrapperValueFromRequest method for createWrappers test
     * @param actionMock partial action mock
     */
    private void overrideGetWrapperValueFromRequest(PartialMock<ViewReportAction> actionMock) {
        actionMock.returns(parameters.getRequestParameterValues()).getWrapperValueFromRequest(null, null, null);
    }

    /**
     * Set behavior for initialValueProvider method for createWrappers test
     * @param actionMock partial action mock
     */
    private void overrideInitialValueProvider(PartialMock<ViewReportAction> actionMock) {
        Mock<ReportParametersAction.InputValueProvider> provider = MockUnitils.createMock(ReportParametersAction.InputValueProvider.class);
        provider.performs(new MockBehavior() {
            public Object execute(ProxyInvocation invocation) {
                List<Object> args = invocation.getArguments();
                return parameters.getDefaultParameterValues().get(args.get(0));
            }
        }).getValue(null);

        actionMock.returns(provider.getMock()).initialValueProvider(null);
    }

    /**
     * Set behavior for initialValueProvider method for createWrappers test
     * @param actionMock partial action mock
     */
    private void overrideExecuteQuery(PartialMock<ViewReportAction> actionMock) {
        actionMock.performs(new MockBehavior() {
            public OrderedMap execute(ProxyInvocation mockInvocation) {
                List<Object> args = mockInvocation.getArguments();
                RuntimeInputControlWrapper wrapper = (RuntimeInputControlWrapper) args.get(2);
                String parameterName = wrapper.getInputControl().getName();
                return ReportParametersActionTestQueryExecutor.execute(parameterName, (Map<String, Object>) args.get(3));
            }
        }).executeQuery(null, null, null, null);
    }

    /**
     * Override CalendarFormatProvider taken from context
     * @param actionMock partial action mock
     */
    private void overrideGetCalendarFormatProvider(PartialMock<ViewReportAction> actionMock) {
        actionMock.returns(new DefaultCalendarFormatProvider()).getCalendarFormatProvider();
    }

    /**
     * Override date pattern taken from context
     * @param actionMock partial action mock
     */
    private void overrideGetStaticDatePattern(PartialMock<ViewReportAction> actionMock) {
        actionMock.returns("yyyyMMddHHmmss").getStaticDatePattern();
    }

    /**
     * Create Parameters helper object
     */
    private void createParameters() {
        if (parameters == null) {
            parameters = new Parameters();
            parameters.addParameterAndControlInfo(CASCADE_COUNTRY, "Billing Country", InputControl.TYPE_MULTI_SELECT_QUERY, Collection.class, null, list("USA"), true);
            parameters.addParameterAndControlInfo(CASCADE_STATE, "Billing State", InputControl.TYPE_MULTI_SELECT_QUERY, Collection.class, null, list("CA"), true);
            parameters.addParameterAndControlInfo(CASCADE_ACCOUNT_TYPE, "Account Type", InputControl.TYPE_SINGLE_SELECT_QUERY, String.class, null, "Consulting", true);
            parameters.addParameterAndControlInfo(CASCADE_INDUSTRY, "Industry", InputControl.TYPE_SINGLE_SELECT_QUERY, String.class, null, "Engineering", true);
            parameters.addParameterAndControlInfo(CASCADE_NAME, "Company Name", InputControl.TYPE_SINGLE_SELECT_QUERY, String.class, null, "EngBureau, Ltd", true);

            parameters.addParameterAndControlInfo(TEXT_DATE, "Order Date", InputControl.TYPE_SINGLE_VALUE, Date.class, DataType.TYPE_DATE, new GregorianCalendar(2011, 6, 28).getTime(), true);
            parameters.addParameterAndControlInfo(TEXT_DATE_TIME, "Order DateTime", InputControl.TYPE_SINGLE_VALUE, Date.class, DataType.TYPE_DATE_TIME, new GregorianCalendar(2011, 6, 28, 18, 22).getTime(), true);

            parameters.addParameterAndControlInfo(STRING, "String Parameter", InputControl.TYPE_SINGLE_VALUE, String.class, DataType.TYPE_TEXT, "defaultValue", true);
        }
    }

    /**
     * This helper class is encapsulation for entities and arguments needed for createWrappers test, it holds:
     * - JRParameter
     * - InputControl
     * - ReportInputControlsInformationImpl
     *
     * - argument parameter values
     * - default parameter values
     * - request URL parameter values
     */
    final class Parameters {
        /* Cascade query controls controls */
        public static final String CASCADE_COUNTRY = "country";
        public static final String CASCADE_STATE = "state";
        public static final String CASCADE_ACCOUNT_TYPE = "accountType";
        public static final String CASCADE_INDUSTRY = "industry";
        public static final String CASCADE_NAME = "name";

        /* Date single value (non-query) controls */
        public static final String TEXT_DATE = "date";
        public static final String TEXT_DATE_TIME = "dateTime";

        /* String single value (non-query) controls */
        public static final String STRING = "string";

        private Map<String, JRParameter> parameters;
        private Map<String, String> controlLabels;
        private Map<String, InputControl> inputControls;
        private ReportInputControlsInformationImpl infos;
        private Map<String, Object> argumentParameterValues;
        private Map<String, Object> requestParameterValues;

        Parameters() {
            parameters = new LinkedHashMap<String, JRParameter>();
            controlLabels = new LinkedHashMap<String, String>();
            inputControls = new LinkedHashMap<String, InputControl>();
            infos = new ReportInputControlsInformationImpl();
            argumentParameterValues = new HashMap<String, Object>();
            requestParameterValues = new HashMap<String, Object>();
        }

        public void addParameterAndControlInfo(String name, String label, byte controlType, Class clazz, Byte controlDataType, Object defaultValue, boolean mandatory) {
            parameters.put(name, createJRParameter(name, clazz));
            controlLabels.put(name, label);
            inputControls.put(name, createInputControl(name, label, mandatory, controlType, controlDataType));
            infos.setInputControlInformation(name, createCI(parameters.get(name), label, defaultValue));
        }

        public JRParameter getParameter(String name) {
            return parameters.get(name);
        }

        public InputControl getInputControl(String name) {
            return inputControls.get(name);
        }

        public ResourceReference getInputControlReference(String name) {
            return new ResourceReference(inputControls.get(name));
        }

        public List<ResourceReference> getInputControlReferences() {
            List<ResourceReference> list = new ArrayList<ResourceReference>();
            for (InputControl control : inputControls.values()) {
                list.add(new ResourceReference(control));
            }
            return list;
        }

        public String getInputControLabel(String name) {
            return controlLabels.get(name);
        }

        public ReportInputControlsInformationImpl getInputControlInfo() {
            return infos;
        }

        public Map<String, Object> getDefaultParameterValues() {
            Map<String, Object> defaults = new HashMap<String, Object>();
            for (String name : infos.getControlNames()) {
                defaults.put(name, infos.getInputControlInformation(name).getDefaultValue());
            }
            return defaults;
        }

        public void setDefaultParameterValues(List<String> country, List<String> state, String accountType, String industry, String name) {
            setDefaultParameterValue(CASCADE_COUNTRY, country);
            setDefaultParameterValue(CASCADE_STATE, state);
            setDefaultParameterValue(CASCADE_ACCOUNT_TYPE, accountType);
            setDefaultParameterValue(CASCADE_INDUSTRY, industry);
            setDefaultParameterValue(CASCADE_NAME, name);
        }

        private void setDefaultParameterValue(String parameterName, Object defaultValue) {
            ((JasperReportInputControlInformation) infos.getInputControlInformation(parameterName)).setDefaultValue(defaultValue);
        }

        public void setArgumentCascadeParameterValues(List<String> country, List<String> state, String accountType, String industry, String name) {
            argumentParameterValues.put(CASCADE_COUNTRY, country);
            argumentParameterValues.put(CASCADE_STATE, state);
            argumentParameterValues.put(CASCADE_ACCOUNT_TYPE, accountType);
            argumentParameterValues.put(CASCADE_INDUSTRY, industry);
            argumentParameterValues.put(CASCADE_NAME, name);
        }

        public void setArgumentParameterValue(String name, Object value) {
            argumentParameterValues.put(name, value);
        }

        public Map<String, Object> getAllArgumentParameterValues() {
            return argumentParameterValues;
        }

        public Map<String, Object> getArgumentCascadeParameterValues() {
            return getArgumentParameterValues(CASCADE_COUNTRY, CASCADE_STATE, CASCADE_ACCOUNT_TYPE, CASCADE_INDUSTRY, CASCADE_NAME);
        }

        public Map<String, Object> getArgumentParameterValues(String... names) {
            Map<String, Object> argumentsMap = new HashMap<String, Object>();
            for (String name : names) {
                argumentsMap.put(name, argumentParameterValues.get(name));
            }
            return argumentsMap;
        }

        public void setRequestParameterValues(List<String> country, List<String> state, String accountType, String industry, String name) {
            requestParameterValues = new HashMap<String, Object>();
            requestParameterValues.put(CASCADE_COUNTRY, country);
            requestParameterValues.put(CASCADE_STATE, state);
            requestParameterValues.put(CASCADE_ACCOUNT_TYPE, accountType);
            requestParameterValues.put(CASCADE_INDUSTRY, industry);
            requestParameterValues.put(CASCADE_NAME, name);
        }

        public void setRequestParameterValue(String name, Object value) {
            requestParameterValues.put(name, value);
        }

        public Map<String, Object> getRequestParameterValues() {
            return requestParameterValues;
        }

        /**
         * See whether parameter with given name is a Collection
         * @param parameterName
         *              Parameter name
         * @return boolean
         *              Multi or Single value
         */
        public boolean isMulti(String parameterName) {
            return getParameter(parameterName).getValueClass().equals(Collection.class);
        }

        public void setQueryForControl(String name, QueryImpl query) {
            inputControls.get(name).setQuery(query);
        }
    }

    /**
     * Create JRParameter entity
     * @param name
     *          Parameter name
     * @param valueClass
     *          Class of values: Collection, String or ohther type
     * @return
     *          Mocked JRParameter
     */
    private JRParameter createJRParameter(String name, Class valueClass) {
        Mock<JRParameter> pMock = MockUnitils.createMock(JRParameter.class);
        pMock.returns(name).getName();
        pMock.returns(valueClass.getName()).getValueClassName();
        pMock.returns(valueClass).getValueClass();
        return pMock.getMock();
    }

    /**
     * Create InputControl entity
     * @param name
     *          Parameter name
     * @param label
     *          Control label
     * @param mandatory
     *          Indicates whether control is mandatory
     * @param type
     *          Type of InputControl, see InputControl interface for possible types
     * @return
     *          Mocked InputControl
     */
    private InputControl createInputControl(String name, String label, boolean mandatory, byte type, Byte controlDataType) {
        InputControl control = new InputControlImpl();
        control.setURIString("/" + name);
        control.setName(name);
        control.setLabel(label);
        control.setMandatory(mandatory);
        control.setType(type);

        /* Control is query type */
        if (type == InputControl.TYPE_MULTI_SELECT_QUERY
         || type == InputControl.TYPE_MULTI_SELECT_QUERY_CHECKBOX
         || type == InputControl.TYPE_SINGLE_SELECT_QUERY
         || type == InputControl.TYPE_SINGLE_SELECT_QUERY_RADIO) {

            control.setQuery(new QueryImpl());
        /* Experimentally I have found that only TYPE_SINGLE_VALUE controls have dataType */
        } else if (type == InputControl.TYPE_SINGLE_VALUE) {

            if (controlDataType != null) {
                DataType dataType = new DataTypeImpl();
                dataType.setType(controlDataType);
                control.setDataType(dataType);
            }
        }
        return control;
    }

    /**
     * Create JasperReportInputControlInformation entity
     * @param parameter
     *          JRParameter entity
     * @param label
     *          The same label as for InputControl
     * @param defaultValue
     *          Default parameter value, it's handy to set it in this place
     * @return
     *          JasperReportInputControlInformation instance
     */
    private JasperReportInputControlInformation createCI(JRParameter parameter, String label, Object defaultValue) {
        JasperReportInputControlInformation info = new JasperReportInputControlInformation();
        info.setReportParameter(parameter);
        info.setPromptLabel(label);
        info.setDefaultValue(defaultValue);
        return info;
    }

    /**
     * Test for helper class ReportParametersActionTestQueryExecutor
     */
    @Test
    public void testReportParametersActionTestQueryExecutor() {
        Map<String, Object> parameterValues = new HashMap<String, Object>();
        OrderedMap expectedResults = new LinkedMap();
        OrderedMap actualResults;

        /* Country is a control without parameters so pass empty map */
        actualResults = execute(CASCADE_COUNTRY, parameterValues);
        
        expectedResults.clear();
        feedResults(expectedResults, "USA", "Ukraine", "Canada");
        assertResultSet(actualResults, expectedResults);

        /* Return states for USA */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, "USA");
        actualResults = execute(CASCADE_STATE, parameterValues);

        expectedResults.clear();
        feedResults(expectedResults, "CA", "OR", "WA");
        assertResultSet(actualResults, expectedResults);

        /* Return states for USA and Canada */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, Arrays.asList("USA", "Canada"));
        actualResults = execute(CASCADE_STATE, parameterValues);

        expectedResults.clear();
        feedResults(expectedResults, "CA", "OR", "WA", "BC");
        assertResultSet(actualResults, expectedResults);

        /* Return account types for USA, CA */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, "USA");
        parameterValues.put(CASCADE_STATE, "CA");
        actualResults = execute(CASCADE_ACCOUNT_TYPE, parameterValues);

        expectedResults.clear();
        feedResults(expectedResults, "Consulting", "Distribution", "Manufactoring");
        assertResultSet(actualResults, expectedResults);

        /* Return industries for Ukraine, Kyivska, Consulting */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, "Ukraine");
        parameterValues.put(CASCADE_STATE, "Kyivska");
        parameterValues.put(CASCADE_ACCOUNT_TYPE, "Consulting");
        actualResults = execute(CASCADE_INDUSTRY, parameterValues);

        expectedResults.clear();
        feedResults(expectedResults, "Construction", "Engineering", "Machinery", "Telecommunications");
        assertResultSet(actualResults, expectedResults);

        /* Return industries for Ukraine, Zakarpatska, Consulting */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, "Ukraine");
        parameterValues.put(CASCADE_STATE, "Zakarpatska");
        parameterValues.put(CASCADE_ACCOUNT_TYPE, "Consulting");
        actualResults = execute(CASCADE_INDUSTRY, parameterValues);

        expectedResults.clear();
        /* The result set is expected to be empty */
        assertResultSet(actualResults, expectedResults);

        /* Return industries for [Ukraine, Canada], [Kyivska, BC], [Distribution, Consulting] */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, Arrays.asList("Ukraine", "Canada"));
        parameterValues.put(CASCADE_STATE, Arrays.asList("Kyivska", "BC"));
        parameterValues.put(CASCADE_ACCOUNT_TYPE, Arrays.asList("Distribution", "Consulting"));
        actualResults = execute(CASCADE_INDUSTRY, parameterValues);

        expectedResults.clear();
        feedResults(expectedResults, "Engineering", "Telecommunications", "Construction", "Machinery", "Communications");
        assertResultSet(actualResults, expectedResults);

        /* Return names for USA, WA, Manufactoring, Machinery */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, "USA");
        parameterValues.put(CASCADE_STATE, "WA");
        parameterValues.put(CASCADE_ACCOUNT_TYPE, "Manufactoring");
        parameterValues.put(CASCADE_INDUSTRY, "Machinery");
        actualResults = execute(CASCADE_NAME, parameterValues);
        
        expectedResults.clear();
        feedResults(expectedResults,
                "Cook-Thompson Transportation Holdings",
                "Sutton-Mlincek Communications Associates");
        assertResultSet(actualResults, expectedResults);

        /* Return names for Ukraine, Zakarpatska, Distribution, [Communications, Telecommunications] */
        parameterValues.clear();
        parameterValues.put(CASCADE_COUNTRY, "Ukraine");
        parameterValues.put(CASCADE_STATE, "Zakarpatska");
        parameterValues.put(CASCADE_ACCOUNT_TYPE, "Distribution");
        parameterValues.put(CASCADE_INDUSTRY, Arrays.asList("Communications", "Telecommunications"));
        actualResults = execute(CASCADE_NAME, parameterValues);

        expectedResults.clear();
        feedResults(expectedResults,
                "J & B Heiser Telecommunications Company",
                "Romero-Hicks Construction, Ltd",
                "X & V Thorne Communications Partners",
                "Short-Swesey Engineering Corp",
                "Aguilar-Vanderbout Communications Associates");
        assertResultSet(actualResults, expectedResults);
    }


    /**
     * Assertion method for RuntimeInputControlWrapper queryResults
     * @param actualResults
     *                  Result set from wrapper has type OrderedMap
     * @param expectedResults
     *                  Expected value Map
     * TODO make assertion order sensitive
     */
    private void assertResultSet(Map actualResults, Map expectedResults) {
        if (actualResults != null && expectedResults != null) {
            if (actualResults.size() == expectedResults.size()) {
                boolean match = true;
                for (Object actualKey : actualResults.keySet()) {
                    if (!expectedResults.containsKey(actualKey)) {
                        match = false;
                        break;
                    }
                }
                assertTrue("Result sets don't match.", match);
            } else {
                fail("Result sets must have the same amount of records: " + "\n" + actualResults  + "\n" + "vs" + "\n" + expectedResults + "\n");
            }
        } else if (actualResults == null && expectedResults == null) {
            /* Cool, both result sets are null - we have non-query controls */
        } else {
            fail("One result set is null but another isn't.");
        }

    }

    /**
     * Assertion method for RuntimeInputControlWrapper value
     * @param orderedSet
     *              Wrapper value has type SetOrderedList
     * @param list
     *              Expected value List
     * TODO make assertion order sensitive
     */
    private void assertValueList(Set orderedSet, List list) {
        if (orderedSet != null && list != null) {
            if (orderedSet.size() == list.size()) {
                boolean match = true;
                for (Object object : orderedSet) {
                    if(!list.contains(object)) {
                        match = false;
                        break;
                    }
                }
                assertTrue("Value lists don't match: " + orderedSet.toString() + " vs " + list.toString(), match);
            } else {
                fail("Value lists must have the same amount of values.");
            }
        } else {
            fail("Value lists cannot be null.");
        }
    }

    /**
     * Assert RuntimeInputControlWrapper by value and queryResults
     * @param parameterName
     *              Parameter name
     * @param value
     *              Wrapper value resolved by tested method createWrappers()
     * @param queryResults
     *              Result set expected to be resolved, can be null for non-query controls
     * @param wrappers
     *              List of parameter returned by  createWrappers()
     */
    private void assertWrappers(String parameterName, Object value, List<String> queryResults, List<RuntimeInputControlWrapper> wrappers) {
        boolean wrapperPresentForParameter = false;
        for (RuntimeInputControlWrapper wrapper : wrappers) {
            if (wrapper.getInputControl().getName().equals(parameterName)) {
                if (parameters.isMulti(parameterName)) {
                    assertValueList((Set) wrapper.getValue(), (List) value);
                } else {
                    assertEquals(value, wrapper.getValue());
                }

                if (queryResults != null) {
                    Map<String, Object> expectedResults = new LinkedMap();
                    feedResults(expectedResults, (String[]) queryResults.toArray());
                    assertResultSet(wrapper.getQueryResults(), expectedResults);
                } else {
                    assertResultSet(wrapper.getQueryResults(), null);
                }
                wrapperPresentForParameter = true;
                break;
            }
        }
        if (!wrapperPresentForParameter) fail("There is no wrapper for parameter: " + parameterName);
    }
}
