/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.remote.handlers;

import com.jaspersoft.jasperserver.api.common.domain.ValidationError;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jaspersoft.jasperserver.api.common.domain.ValidationErrorFilter;
import com.jaspersoft.jasperserver.api.common.domain.ValidationErrors;
import com.jaspersoft.jasperserver.api.common.domain.impl.UniversalValidationErrorFilter;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.ResourceDescriptor;
import com.jaspersoft.jasperserver.remote.AbstractService;
import com.jaspersoft.jasperserver.remote.ManagementServices;
import com.jaspersoft.jasperserver.remote.ResourceActionResolver;
import com.jaspersoft.jasperserver.remote.ResourceHandler;
import com.jaspersoft.jasperserver.remote.ResourceResolverAdapter;
import com.jaspersoft.jasperserver.remote.ServiceException;
import java.util.List;

/**
 * This is a base class for a resource handler.
 *
 * @author Giulio Toffoli
 */
public abstract class AbstractResourceHandler implements ResourceHandler, ResourceResolverAdapter {

    private static final Log log = LogFactory.getLog(AbstractResourceHandler.class);
    private ResourceActionResolver resourceActionResolver;

    public void setResourceActionHandler(ResourceActionResolver resourceActionResolver) {
        this.resourceActionResolver = resourceActionResolver;
    }

    public ResourceActionResolver getResourceActionResolver()
    {
        return this.resourceActionResolver;
    }
    
    public ResourceDescriptor get(Resource resource, Map options) throws ServiceException {

        ResourceDescriptor descriptor = new ResourceDescriptor();
        setCommonAttributes(resource, descriptor);

        doGet(resource, descriptor, options);
        return descriptor;
    }

    /**
     * Set the common attributes of the resources:
     * uri, id (name), label, description, type, parent uri, version and creation date
     *
     * @param resource
     * @param descriptor
     */
    protected void setCommonAttributes(Resource resource,

        ResourceDescriptor descriptor) {
        descriptor.setUriString(resource.getURIString());
        descriptor.setDescription(resource.getDescription());
        descriptor.setLabel(resource.getLabel());
        descriptor.setName(resource.getName());
        descriptor.setResourceType(resource.getResourceType());
        descriptor.setParentFolder(resource.getParentFolder());
        descriptor.setVersion(resource.getVersion());
        descriptor.setCreationDate(resource.getCreationDate());
    }

    /**
     *
     * Performs the specific work to get a resource.
     *
     * @param resource
     * @param descriptor
     * @param options
     * @throws ServiceException
     */
    protected abstract void doGet(Resource resource,
            ResourceDescriptor descriptor, Map options) throws ServiceException;


    /**
     * Use the UniversalValidationErrorFilter to validate the resource.
     * Use the configured ResourceActionResolver to check if the resource can be written or not.
     *
     * @param resource
     * @throws ServiceException
     */
    protected void validate(Resource resource) throws ServiceException {
        ValidationErrorFilter filter = resource.isNew() ? UniversalValidationErrorFilter.getInstance() : null;
        RepositoryService repositoryService = AbstractService.getContext().getManagementServices().getRepository();



        ValidationErrors errors = null;

        if (resource instanceof Folder) errors = repositoryService.validateFolder(null, (Folder)resource, filter);
        else errors = repositoryService.validateResource(null, resource, filter);

         if (errors.isError()){

                String errorString = "";
                List errorsList = errors.getErrors();
                for (int i=0; i < errorsList.size(); ++i)
                {
                    ValidationError ve = (ValidationError)errorsList.get(i);
                    if (errorString.length() > 0) errorString += "\n";
                    errorString += AbstractService.getContext().getMessage(ve.getErrorCode(), ve.getErrorArguments(), ve.getDefaultMessage());
                }
                throw new ServiceException(ServiceException.RESOURCE_BAD_REQUEST, errorString);
         }
    }

    /**
     * Create or update a resource.
     * This methos is the same as update(descriptor, options,true);
     *
     * @param descriptor
     * @return
     * @throws ServiceException
     */

    public ResourceDescriptor update(ResourceDescriptor descriptor, Map options) throws ServiceException {
        return update(descriptor, options,true);
    }

    /**
     * Implementation of the update method. This default implementation perform most of the generic work, creating
     * a new instance of a resource if it does not exist.
     * The method update on existing resources is called.
     *
     * @param descriptor
     * @param options
     * @param save
     * @return
     * @throws ServiceException
     */
    public ResourceDescriptor update(ResourceDescriptor descriptor, Map options, boolean save) throws ServiceException {

        ManagementServices managementServices = AbstractService.getContext().getManagementServices();
        RepositoryService repositoryService = managementServices.getRepository();

        // Check if the resource must be saved or modified inside another resource...
        Resource parentResource = getParentResource(descriptor);
        ResourceContainer parentHandler = null;
        
        if (parentResource != null)
        {
            ResourceHandler pResourceHandler = (ResourceHandler) managementServices.getHandlerRegistry().getHandler(parentResource);
            if (pResourceHandler instanceof ResourceContainer)
            {
                parentHandler = (ResourceContainer)pResourceHandler;
            }
        }

        if (descriptor.getIsNew()) {

            if (parentHandler == null) {
                
                Resource resource = createResource(descriptor);
                validate(resource);
                if (save){
                    if (resource instanceof Folder){
                        repositoryService.saveFolder(null, (Folder)resource);
                    }
                    else{
                        repositoryService.saveResource(null, resource);
                    }
                }
                return managementServices.createResourceDescriptor(resource.getURIString());

            } else {
                
                Resource subResource = parentHandler.attachSubResource(parentResource, descriptor);

                if (subResource != null) {
                    validate(parentResource);

                    // We assume here that a folder does not have nested children...
                    if (save) repositoryService.saveResource(null, parentResource);
                    return managementServices.createResourceDescriptor(subResource);
                }
                else
                {
                    // The error here is actually something like: cannot create the resource.
                    throw new ServiceException(ServiceException.FORBIDDEN,
                                    AbstractService.getContext().getMessage("webservices.error.cannotCreateResource",
                                    new String[]{parentResource.getParentFolder()} ));
                }
            }
        } else { // We are modifying the resource...

            if (parentHandler == null) {

                log.debug("Put: modifying " + descriptor.getWsType());

                Class resourceType = getResourceType();

                Resource resource = null;
                if (ResourceDescriptor.TYPE_FOLDER.equals(descriptor.getWsType()))
                {
                    resource = managementServices.getRepository().getFolder(null, descriptor.getUriString());
                }
                else
                {
                   resource = managementServices.getRepository().getResource(null, descriptor.getUriString(), resourceType);
                }
                if (resource == null) {

                    throw new ServiceException(ServiceException.RESOURCE_NOT_FOUND, AbstractService.getContext().getMessage("webservices.error.resourceNotFound", new Object[]{}));

                } else {

                    updateResource(resource, descriptor, options);
                    validate(resource);

                    if (save){
                        if (resource instanceof Folder){
                            repositoryService.saveFolder(null, (Folder)resource);
                        }
                        else{
                            repositoryService.saveResource(null, resource);
                        }
                    }
                    return managementServices.createResourceDescriptor(resource);
                }
                
            } else { // We need to ask the parent to handle the change...

                    Resource subResource = parentHandler.attachSubResource(parentResource, descriptor);
                    if (subResource != null) {
                        // validate the sub resource before save it...
                        validate(subResource);
                        if (save) repositoryService.saveResource(null, parentResource);
                        return managementServices.createResourceDescriptor(subResource);
                    }
                    else
                    {
                        // The error here is actually something like: cannot update the resource.
                        throw new ServiceException(ServiceException.GENERAL_ERROR,
                                        AbstractService.getContext().getManagementServices().getMessage("webservices.error.cannotCreateResource",
                                        new String[]{parentResource.getParentFolder()}, AbstractService.getContext().getLocale() ));
                    }
            }
        } // End if/else is new resource.
    }


    /**
     * This method looks into the options map to see is the resource to update is actually a child of a particular report unit...
     *
     * @param options
     * @return
     * @throws ServiceException
     * @deprecated The server is able to check if the resource belogs to another resource automatically
     */
    protected Resource getModifyingParentResource(Map options) throws ServiceException {
        Resource resource = null;

        if (options!=null && options.containsKey(Argument.MODIFY_REPORTUNIT))
        {
            String reportUnitUrl = (String)options.get(Argument.MODIFY_REPORTUNIT);
            if (reportUnitUrl != null && reportUnitUrl.length() > 0) {
                log.debug("Put: adding/modifying resource in reportUnit " + reportUnitUrl);
                ReportUnit reportUnit = (ReportUnit) AbstractService.getContext().getManagementServices().getRepository().getResource(null, reportUnitUrl);
                if (reportUnit == null) {
                    throw new ServiceException(
                            ServiceException.RESOURCE_NOT_FOUND, AbstractService.getContext().getMessage(
                            "webservices.error.reportUnitNotFound"));
                }
                resource = reportUnit;
            }

        }
        // for now, only report units can act as parent resources
        
        
        return resource;
    }

    /**
     * Ask the repository to create a new resource of the type specified by this handler with the name and the
     * description specified by the descriptor.
     * The method uses updateResource() to fill the empty resource created.
     *
     *
     * @param descriptor
     * @return
     * @throws ServiceException
     */
    protected Resource createResource(ResourceDescriptor descriptor) throws ServiceException
    {
        ResourceDescriptor childRD;
        Class resourceType = getResourceType();
        Resource resource = AbstractService.getContext().getManagementServices().getRepository().newResource(null, resourceType);
        resource.setParentFolder(descriptor.getParentFolder());
        resource.setVersion(Resource.VERSION_NEW);
        resource.setName(descriptor.getName());
        resource.setURIString(descriptor.getUriString());
        updateResource(resource, descriptor, null);

        return resource;
    }

    /**
     * convenient method to create a new resource of generic type. This method looks for the wsType in the descriptor, looks for an handler of
     * this type of resource and create it.
     * If an handler is not found for this resource, or if the handler does not implements the AbstractResourceHandler class,
     * the method returns null.
     *
     *
     * @param descriptor
     * @return
     */
    protected Resource createChildResource(ResourceDescriptor descriptor) throws ServiceException {
            String childType = descriptor.getWsType();
            ResourceHandler handler = AbstractService.getContext().getManagementServices().getHandlerRegistry().getHandler(childType);

            if (handler == null || !(handler instanceof AbstractResourceHandler) )
            {
                log.warn(handler.getClass().getName() + " does not implement AbstractResourceHandler");
                return null;
            }
            return  ((AbstractResourceHandler)handler).createResource(descriptor);
    }

    
    /**
     * Specific code to update a resource, in particular to read the resource properties from the
     * resource descriptor, and store them into the passed resource implementation.
     *
     * The default implementation just set the label and the description.
     *
     * Attention: name, uri and wsType should not be modified. The porpuse of override this method
     * is to add code specific for the type of resource your are dealing with.
     *
     * @param resource
     * @param descriptor
     * @throws ServiceException
     */
    protected void updateResource(Resource resource, ResourceDescriptor descriptor, Map options)
    {
        resource.setLabel(descriptor.getLabel());
        resource.setDescription(descriptor.getDescription());
    }


    /**
     * Checks if the descriptor is some kind of data source
     *
     * @param rd
     * @return
     */
    public boolean isDataSource(ResourceDescriptor rd) {

        String type = rd.getWsType();

        ResourceHandler handler = (ResourceHandler) getManagementServices().getHandlerRegistry().getHandler(type);

        if (handler!=null){
            Class resourceType = handler.getResourceType();
            ResourceHandler parentHandler = (ResourceHandler) getManagementServices().getHandlerRegistry().getHandler(ResourceDescriptor.TYPE_DATASOURCE);

            Class parentResourceType = parentHandler.getResourceType();
            return parentResourceType.isAssignableFrom(resourceType);
        }
        return false;
    }




    /**
     * convenient method to return the management services from the current context.
     *
     * @return
     */
    protected ManagementServices getManagementServices()
    {
        return AbstractService.getContext().getManagementServices();
    }

    /**
     * Get the parent resource of this descriptor. A folder is not a parent resource,
     * it must be another resource...
     * Usually parent resources ends with _files, (but this may be not the rule we are doing a strong assumption here)...
     * so we have to do a couple of tests first...
     * @param descriptor
     * @return
     */
    private Resource getParentResource(ResourceDescriptor descriptor) {

        String resourceUri = descriptor.getUriString();

        // get the parent resource uri....
        int lastSeparatorPos = descriptor.getUriString().lastIndexOf(Folder.SEPARATOR);
        String parentResourceUri = (lastSeparatorPos == 0) ? null : resourceUri.substring(0, lastSeparatorPos);

        if (parentResourceUri == null) return null;
        if (parentResourceUri.endsWith("_files"))
        {
            parentResourceUri.substring(0, parentResourceUri.length() - "_files".length());
        }

        try {
            return AbstractService.getContext().getManagementServices().locateResource(parentResourceUri);
        } catch (Exception ex)
        {
            // resource not found, we can ignore that.
        }
        return null;
        
    }

    public void delete(ResourceDescriptor descriptor) throws ServiceException {

        AbstractService context = AbstractService.getContext();

        ResourceActionResolver resourceActionResolver = context.getManagementServices().getResourceActionResolver();
        if (resourceActionResolver.isResourceDeletable(descriptor.getUriString())) {

            if (descriptor.getWsType().equals(ResourceDescriptor.TYPE_FOLDER)){
                getManagementServices().getRepository().deleteFolder(null, descriptor.getUriString());
            }
            else{

                // Check if the resource must be saved or modified inside another resource...
                Resource parentResource = getParentResource(descriptor);
                
                if (parentResource != null)
                {
                    ResourceHandler pResourceHandler = (ResourceHandler) context.getManagementServices().getHandlerRegistry().getHandler(parentResource);
                    if (pResourceHandler instanceof ResourceContainer)
                    {
                        ((ResourceContainer)pResourceHandler).detachSubResource(parentResource, descriptor);
                        getManagementServices().getRepository().saveResource(null, parentResource);
                    }
                }

                getManagementServices().getRepository().deleteResource(null, descriptor.getUriString());
            }
            
        } else {
            throw new ServiceException(ServiceException.FORBIDDEN,
                    context.getMessage("webservices.error.notDeletableResource", null));
        }
    }

//    public void move(Request request, RepositoryServiceContext serviceContext) throws WSException {
//        String sourceURI = request.getResourceDescriptor().getUriString();
//        String destinationURI = getDestinationURI(request, serviceContext);
//
//        if (log.isDebugEnabled()) {
//            log.debug("Moving resource " + sourceURI + " to folder " + destinationURI);
//        }
//
//        if (!resourceActionResolver.canCreateResource(destinationURI)) {
//            throw new WSException(WSException.GENERAL_ERROR,
//                    serviceContext.getMessage("webservices.error.cannotCreateResource", new String[]{destinationURI}));
//        }
//
//        serviceContext.getRepository().moveResource(null, sourceURI, destinationURI);
//    }

//    public ResourceDescriptor copy(Request request, RepositoryServiceContext serviceContext) throws WSException {
//        String sourceURI = request.getResourceDescriptor().getUriString();
//        String destinationURI = getDestinationURI(request, serviceContext);
//
//        if (log.isDebugEnabled()) {
//            log.debug("Copying resource " + sourceURI + " to URI " + destinationURI);
//        }
//
//        if (!resourceActionResolver.canCreateResource(ResourceUtils.getParentFolder(destinationURI))) {
//            throw new WSException(WSException.GENERAL_ERROR,
//                    serviceContext.getMessage("webservices.error.cannotCreateResource",
//                    new String[]{ResourceUtils.getParentFolder(destinationURI)}));
//        }
//
//        // TODO multi resource copy?
//        Resource copy = serviceContext.getRepository().copyResource(null, sourceURI, destinationURI);
//        return describe(copy, getDefaultDescribeArguments(), serviceContext);
//    }

//    protected String getDestinationURI(Request request,
//            RepositoryServiceContext serviceContext) throws WSException {
//        String destinationURI = request.getArgumentValue(Argument.DESTINATION_URI);
//        if (destinationURI == null) {
//            throw new WSException(WSException.GENERAL_REQUEST_ERROR,
//                    serviceContext.getMessage("webservices.error.request.no.destination.URI", null));
//        }
//        return destinationURI;
//    }

//    public List listResources(Request request,
//            RepositoryServiceContext serviceContext) throws WSException {
//        Class resourceType = getResourceType();
//
//        String parentFolder = request.getArgumentValue(Argument.PARENT_DIRECTORY);
//        String ancestorFolder = request.getArgumentValue(Argument.START_FROM_DIRECTORY);
//
//        if (log.isDebugEnabled()) {
//            log.debug("Listing resources of type " + resourceType
//                    + (parentFolder != null
//                    ? ("from folder " + parentFolder)
//                    : (ancestorFolder != null
//                    ? (" starting from folder " + ancestorFolder)
//                    : "")));
//        }
//
//        FilterCriteria filter = FilterCriteria.createFilter(resourceType);
//
//        if (parentFolder != null) {
//            filter.addFilterElement(FilterCriteria.createParentFolderFilter(parentFolder));
//        } else if (ancestorFolder != null
//                // do not set ancestor folder filter if root
//                && !Folder.SEPARATOR.equals(ancestorFolder)) {
//            filter.addFilterElement(FilterCriteria.createAncestorFolderFilter(ancestorFolder));
//        }
//
//        FilterElement additionalCriteria = additionalListResourcesFilterCriteria(request);
//        if (additionalCriteria != null) {
//            filter.addFilterElement(additionalCriteria);
//        }
//
//        List resources = serviceContext.getRepository().loadClientResources(filter);
//        List descriptors;
//        if (resources == null || resources.isEmpty()) {
//            descriptors = new ArrayList(0);
//        } else {
//            descriptors = new ArrayList(resources.size());
//            for (Iterator it = resources.iterator(); it.hasNext();) {
//                Resource resource = (Resource) it.next();
//                descriptors.add(serviceContext.createResourceDescriptor(resource));
//            }
//        }
//        return descriptors;
//    }

//    protected FilterElement additionalListResourcesFilterCriteria(Request request) {
//        return null;
//    }
}
