/*
// $Id: //open/mondrian/src/main/mondrian/gui/QueryPanel.java#19 $
// This software is subject to the terms of the Common Public License
// Agreement, available at the following URL:
// http://www.opensource.org/licenses/cpl.html.
// Copyright (C) 2002-2008 Julian Hyde and others
// Copyright (C) 2006-2007 Cincom Systems, Inc.
// All Rights Reserved.
// You must accept the terms of that agreement to use this software.
*/
package mondrian.gui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.sql.Connection;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;
import javax.swing.AbstractButton;
import javax.swing.BorderFactory;
import javax.swing.ComboBoxModel;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JInternalFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTabbedPane;
import javax.swing.JTextPane;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.plaf.basic.BasicButtonUI;
import javax.swing.text.DefaultEditorKit;

import org.apache.log4j.Logger;
import org.olap4j.CellSet;
import org.olap4j.OlapConnection;
import org.olap4j.OlapStatement;
import org.olap4j.OlapWrapper;
import org.olap4j.query.CellSetFormatter;
import org.olap4j.query.RectangularCellSetFormatter;

/**
 *
 * @author  sean
 * @version $Id: //open/mondrian/src/main/mondrian/gui/QueryPanel.java#19 $
 */
public class QueryPanel extends javax.swing.JPanel {

    private static final Logger LOGGER = Logger.getLogger(QueryPanel.class);

    Connection connection;
    JMenuItem queryMenuItem;
    int windowMenuIndex;
    Map schemaWindowMap;    // map of schema frames and schema menu items

    Workbench workbench;
    SchemaExplorer schemaExplorer = null;

    /** Creates new form QueryPanel */
    public QueryPanel(Workbench workbench) {
        this.workbench = workbench;
        initComponents();
    }
    public QueryPanel(Workbench workbench, SchemaExplorer se) {
        this.workbench = workbench;
        this.schemaExplorer = se;
        initComponents();
    }

    public void setConnection(Connection c) {
        connection = c;
    }

    public Connection getConnection() {
        return connection;
    }

    //====================================================

    public void setMenuItem(JMenuItem mi) {
        this.queryMenuItem = mi;
    }

    public void setSchemaWindowMap(Map swindowMap) {
        this.schemaWindowMap = swindowMap;  // MAP OF FILE AND FRAME
        setCatalogs();
    }

    private void setCatalogs() {
        Vector v = new Vector();
        Iterator it = schemaWindowMap.values().iterator();  // schema menu items
        while (it.hasNext()) {
            JMenuItem elem = (JMenuItem) it.next();
            v.add(elem.getText());
        }
        ComboBoxModel cCatalogs = new DefaultComboBoxModel(v);
        schemaList.setModel(cCatalogs) ;
    }
    public void setWindowMenuIndex(int i) {
        this.windowMenuIndex = i;
    }

    /**
     * @return the workbench i18n converter
     */
    public I18n getResourceConverter() {
        return workbench.getResourceConverter();
    }

    public void initConnection(String smenutext) {
        schemaList.setSelectedItem(smenutext);
        connectButtonActionPerformed(null);
    }
    //=====================================================

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {
        jScrollPane3 = new javax.swing.JScrollPane();
        executeButton = new javax.swing.JButton();
        jSplitPane1 = new javax.swing.JSplitPane();
        jScrollPane1 = new javax.swing.JScrollPane();
        queryTextPane = new javax.swing.JTextPane();
        jScrollPane2 = new javax.swing.JScrollPane();
        //resultTextPane = new javax.swing.JTextPane();
        jTabbedPane = new javax.swing.JTabbedPane();

        connectButton = new javax.swing.JButton();
        jPopupMenu = new JPopupMenu();
        jPopupMenu.add(new DefaultEditorKit.CutAction());
        jPopupMenu.add(new DefaultEditorKit.CopyAction());
        jPopupMenu.add(new DefaultEditorKit.PasteAction());

        schemaScrollPane1 = new javax.swing.JScrollPane();
        schemaLabel = new javax.swing.JLabel();
        schemaList = new JComboBox(
                new String[] {getResourceConverter().getString(
                                            "common.join.title","Join"),
                              getResourceConverter().getString(
                                            "common.table.title","Table")});
        //schemaScrollPane1.setViewportView(schemaList);
        schemaPanel = new JPanel();
        //schemaPanel.setLayout(new BorderLayout(25,0));
        schemaPanel.setLayout(new FlowLayout(FlowLayout.LEFT, 10, 0));

        setLayout(new java.awt.BorderLayout());

        schemaLabel.setFont(new Font("Dialog", 1, 12));
        schemaLabel.setForeground((Color) UIManager.getDefaults().get(
                "CheckBoxMenuItem.acceleratorForeground"));
        schemaLabel.setHorizontalAlignment(SwingConstants.CENTER);
        schemaLabel.setText(getResourceConverter().getString(
                "common.schema.title","Schema"));
        //schemaLabel.setBorder(new EtchedBorder());

        schemaList.setBackground(Color.white);
        final JPanel qpanel = this;
        schemaList.addItemListener(
            new ItemListener() {
                public void itemStateChanged(ItemEvent e) {
                }
            });
        connectButton.setText(getResourceConverter().getString(
                "queryPanel.connect.title",
                "Connect"));
        connectButton.addActionListener(
            new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent evt) {
                    connectButtonActionPerformed(evt);
                }
            });

        schemaPanel.add(schemaLabel); //java.awt.BorderLayout.WEST
        schemaPanel.add(schemaList);
        schemaPanel.add(connectButton);

        // Only display the schema selection panel if it is launched
        // from the Workbench
        if (schemaExplorer == null) {
            add(schemaPanel, java.awt.BorderLayout.NORTH);
        }

        queryPanel = new JPanel();
        queryPanel.setLayout(new javax.swing.BoxLayout(queryPanel, javax.swing.BoxLayout.Y_AXIS));

        queryTextPane.setFont(new java.awt.Font("Courier New", 0, 12));
        queryTextPane.setText("");
        queryTextPane.addMouseListener(new MouseAdapter() {
            // From MouseAdapter javadoc:
            //
            // Popup menus are triggered differently
            // on different systems. Therefore, isPopupTrigger
            // should be checked in both mousePressed
            // and mouseReleased
            // for proper cross-platform functionality.

            public void mousePressed(MouseEvent e) {
                checkPopupTrigger(e);
            }

            public void mouseReleased(MouseEvent e) {
                checkPopupTrigger(e);
            }

            public void checkPopupTrigger(MouseEvent e) {
                if (e.isPopupTrigger()) {
                    int x = e.getX();
                    int y = e.getY();

                    jPopupMenu.show(queryTextPane, x, y);
                }
            }
        });

        jScrollPane1.setViewportView(queryTextPane);

        queryPanel.add(jScrollPane1, Component.CENTER_ALIGNMENT);

        executeButton.setText(getResourceConverter().getString(
                "queryPanel.execute.title",
                "Execute"));
        executeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                executeButtonActionPerformed(evt);
            }
        });

        queryPanel.add(executeButton, Component.CENTER_ALIGNMENT);

        // resultTextPane.setEditable(false);
        // resultTextPane.setFont(new java.awt.Font("Courier New", 0, 12));
        jScrollPane2.setViewportView(jTabbedPane);

        jSplitPane1.setDividerLocation(100);
        jSplitPane1.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);
        
        jSplitPane1.setTopComponent(queryPanel);

        jSplitPane1.setBottomComponent(jScrollPane2);

        add(jSplitPane1, java.awt.BorderLayout.CENTER);
    }

    private void executeButtonActionPerformed(java.awt.event.ActionEvent evt) {
        //run the query, and show the results.

        if (connection == null) {
            JOptionPane.showMessageDialog(this,
                    getResourceConverter().getString(
                        "queryPanel.noConnection.alert",
                        "No Mondrian connection. Select a Schema to connect."),
                    getResourceConverter().getString(
                        "common.alertDialog.title",
                        "Alert"),
                    JOptionPane.WARNING_MESSAGE);
            return;
        } //common.alertDialog.title

        jTabbedPane.removeAll();

        String allQueryStr = queryTextPane.getText();

        if (allQueryStr == null || allQueryStr.length() == 0) {
            return;
        }

        String[] queryStrs = allQueryStr.split(";");

        OlapConnection olapConnection = null;
        OlapStatement olapStatement = null;

        CellSetFormatter formatter = new RectangularCellSetFormatter(false);

        if (queryStrs.length > 0) {
            int i = 0;
            for (String queryStr : queryStrs) {

                queryStr = queryStr.trim();

                if (queryStr.length() == 0) {
                    continue;
                }

                java.io.StringWriter sw = new java.io.StringWriter();
                java.io.PrintWriter pw = new java.io.PrintWriter(sw);

                pw.println(queryStr);
                pw.println();
                pw.println("----------------------------------");
                pw.println();

                try {
                    olapConnection =
                            ((OlapWrapper) connection).unwrap(OlapConnection.class);

                    olapStatement = olapConnection.createStatement();

                    CellSet cellSet = olapStatement.executeOlapQuery(queryStr);

                    formatter.format(cellSet, pw);
                } catch (Exception ex) {
                    LOGGER.error(queryStr, ex);
                    Throwable e = ex;
                    while (e != null) {
                        pw.println(e.getLocalizedMessage());
                        Throwable prev = e;
                        e = e.getCause();
                        if (e == prev) {
                            break;
                        }
                        pw.println();
                    }
                }

                pw.flush();

                JTextPane newPane = new JTextPane();
                newPane.setEditable(false);
                newPane.setFont(new java.awt.Font("Courier New", 0, 12));
                newPane.setText(sw.getBuffer().toString());

                jTabbedPane.add("" + ++i, newPane);

                try {
                    if (olapStatement != null) {
                        olapStatement.close();
                    }
                } catch (Exception e) {

                }

//                try {
//                    if (olapConnection != null) {
//                        olapConnection.close();
//                    }
//                } catch (Exception e) {
//
//                }
            }
        }
    }

    private void connectButtonActionPerformed(java.awt.event.ActionEvent evt) {
        if (schemaExplorer == null) {
            connectButtonActionPerformedForFile(evt);
        } else {
            connectButtonActionPerformedForOpenSchema(evt);
        }
    }

    /**
     * Get the file from the schema
     * Test it
     * If ok, connect
     *
     * @param evt
     */
    private void connectButtonActionPerformedForOpenSchema(java.awt.event.ActionEvent evt) {
        Component o = this;
        while (o != null) {
            if (o.getClass() == JInternalFrame.class) {
                break;
            }
            o = o.getParent();
        }

        if (o == null) {
            return;
        }

        JInternalFrame jf = (JInternalFrame) o;

        if (schemaExplorer == null) {
            JOptionPane.showMessageDialog(this,
                    getResourceConverter().getString(
                        "queryPanel.schemaNotOpen.alert",
                        "Schema file is not open"),
                    getResourceConverter().getString(
                        "common.errorDialog.title",
                        "Error"),
                    JOptionPane.ERROR_MESSAGE);
            jf.dispose();
            return;
        }

        File sfile = schemaExplorer.createTempSchemaFile();

        if (sfile == null) {
            jf.dispose();
            return;
        }

        try {
            Connection con = workbench.checkSchemaFile(sfile);

            if (con != null) {
                connection = con;
                queryMenuItem.setText(getResourceConverter().getFormattedString("queryPanel.successfulConnection.menuItem",
                        "{0} MDX - {1}",
                        new String[] { Integer.toString(windowMenuIndex), schemaExplorer.getSchema().name }));
                o =  this;
                while (o != null) {
                    if (o.getClass() == JInternalFrame.class) {
                        ((JInternalFrame) o).setTitle(getResourceConverter().getFormattedString("queryPanel.successfulConnection.internalFrame.title",
                                "MDX Query - connected to {0}",
                                new String[] { schemaExplorer.getSchema().name }));
                        break;
                    }
                    o =  o.getParent();
                }
                JOptionPane.showMessageDialog(this, "Mondrian connection Successful.",
                        getResourceConverter().getString("common.informationDialog.title","Information") , JOptionPane.INFORMATION_MESSAGE);
            } else {
                JOptionPane.showMessageDialog(this, getResourceConverter().getFormattedString("queryPanel.unsuccessfulConnection.alert",
                        "Mondrian connection could not be done for - {0}",
                        new String[] { schemaExplorer.getSchema().name }),
                        getResourceConverter().getString("common.errorDialog.title","Error"), JOptionPane.ERROR_MESSAGE);
            }
        } catch (Exception ex) {
            LOGGER.error("Exception: " + ex.getMessage(), ex);
            JOptionPane.showMessageDialog(this, getResourceConverter().getFormattedString("queryPanel.unsuccessfulConnection.exception",
                    "Mondrian connection could not be done for - {0}",
                    new String[] {
                        sfile == null ?
                            getResourceConverter().getString("queryPanel.selectedSchema.alert",
                                "selected Schema") :
                            sfile.getName() }),
                        getResourceConverter().getString("common.errorDialog.title","Error"), JOptionPane.ERROR_MESSAGE);
        }
    }

    private void connectButtonActionPerformedForFile(java.awt.event.ActionEvent evt) {
        File sfile = null;
        try {
            JInternalFrame sf = null;

            String sfname = (String) schemaList.getSelectedItem();
            Iterator it = schemaWindowMap.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry elem = (Map.Entry) it.next();
                if (((JMenuItem) elem.getValue()).getText().equals(sfname)) {
                    sf = (JInternalFrame) elem.getKey();
                    break;
                }
            }

            if (sf == null) {
                // this case may arise when a schema file is opened,
                // mdx query is opened and the schema frame is closed
                JOptionPane.showMessageDialog(this,
                        getResourceConverter().getString(
                            "queryPanel.schemaNotOpen.alert",
                            "Schema file is not open"),
                        getResourceConverter().getString(
                            "common.errorDialog.title",
                            "Error"),
                        JOptionPane.ERROR_MESSAGE);
                return;
            }

            //JInternalFrame sf = (JInternalFrame) schemaWindowMap.get(sfile);
            SchemaExplorer se = (SchemaExplorer) sf.getContentPane().getComponent(0);
            if (se.isNewFile()) {
                JOptionPane.showMessageDialog(this,
                        getResourceConverter().getString(
                            "queryPanel.saveSchemaFirst.alert",
                            "You must first save the Schema to open a Mondrian connection"),
                        getResourceConverter().getString(
                            "common.alertDialog.title","Alert"),
                            JOptionPane.WARNING_MESSAGE);
                sf.setSelected(true);
                return;
            }
            sfile = se.getSchemaFile();

            Connection con = workbench.checkSchemaFile(sfile);

            if (con != null) {
                connection = con;
                queryMenuItem.setText(getResourceConverter().getFormattedString("queryPanel.successfulConnection.menuItem",
                        "{0} MDX - {1}",
                        new String[] { Integer.toString(windowMenuIndex), se.getSchemaFile().getName() }));
                Component o =  this;
                while (o != null) {
                    //System.out.println(""+o.getClass());
                    if (o.getClass() == JInternalFrame.class) {
                        ((JInternalFrame) o).setTitle(getResourceConverter().getFormattedString("queryPanel.successfulConnection.internalFrame.title",
                                "MDX Query - connected to {0}",
                                new String[] { se.getSchemaFile().getName() }));
                        break;
                    }
                    o =  o.getParent();
                }
                JOptionPane.showMessageDialog(this, "Mondrian connection Successful.",
                        getResourceConverter().getString("common.informationDialog.title","Information") , JOptionPane.INFORMATION_MESSAGE);
            } else {
                JOptionPane.showMessageDialog(this, getResourceConverter().getFormattedString("queryPanel.unsuccessfulConnection.alert",
                        "Mondrian connection could not be done for - {0}",
                        new String[] { se.getSchemaFile().getName() }),
                        getResourceConverter().getString("common.errorDialog.title","Error"), JOptionPane.ERROR_MESSAGE);
            }
        } catch (Exception ex) {
            LOGGER.error("Exception: " + ex.getMessage(), ex);
            JOptionPane.showMessageDialog(this, getResourceConverter().getFormattedString("queryPanel.unsuccessfulConnection.exception",
                    "Mondrian connection could not be done for - {0}",
                    new String[] { sfile == null ? getResourceConverter().getString("queryPanel.selectedSchema.alert","selected Schema") : sfile.getName() }),
                        getResourceConverter().getString("common.errorDialog.title","Error"), JOptionPane.ERROR_MESSAGE);
        }
    }

    protected void dispose() {
        if (connection != null) {
            try {
                connection.close();
            } catch (Exception e) {
                
            }
        }
    }
    
    // Variables declaration - do not modify
    private javax.swing.JScrollPane jScrollPane3;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JTabbedPane jTabbedPane;
    // private javax.swing.JTextPane resultTextPane;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JTextPane queryTextPane;
    private javax.swing.JSplitPane jSplitPane1;
    private javax.swing.JButton executeButton;
    private javax.swing.JComboBox schemaList;
    private JLabel schemaLabel;
    private JPanel schemaPanel;
    private JPanel queryPanel;
    private javax.swing.JScrollPane schemaScrollPane1;
    private javax.swing.JButton connectButton;
    private JPopupMenu jPopupMenu;

    // End of variables declaration

    /**
     * Component to be used as tabComponent;
     * Contains a JLabel to show the text and
     * a JButton to close the tab it belongs to
     */
    public class ButtonTabComponent extends JPanel {
        private final JTabbedPane pane;

        public ButtonTabComponent(final JTabbedPane pane) {
            //unset default FlowLayout' gaps
            super(new FlowLayout(FlowLayout.LEFT, 0, 0));
            if (pane == null) {
                throw new NullPointerException("TabbedPane is null");
            }
            this.pane = pane;
            setOpaque(false);

            //make JLabel read titles from JTabbedPane
            JLabel label = new JLabel() {
                public String getText() {
                    int i = pane.indexOfComponent(ButtonTabComponent.this);
                    if (i != -1) {
                        return pane.getTitleAt(i);
                    }
                    return null;
                }
            };

            add(label);
            //add more space between the label and the button
            label.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 5));
            //tab button
            JButton button = new TabButton(pane);
            add(button);
            //add more space to the top of the component
            setBorder(BorderFactory.createEmptyBorder(2, 0, 0, 0));
        }
    }

    private class TabButton extends JButton implements ActionListener {
        private final JTabbedPane pane;
        public TabButton(JTabbedPane pane) {
            this.pane = pane;

            int size = 17;
            setPreferredSize(new Dimension(size, size));
            setToolTipText("close this tab");
            //Make the button looks the same for all Laf's
            setUI(new BasicButtonUI());
            //Make it transparent
            setContentAreaFilled(false);
            //No need to be focusable
            setFocusable(false);
            setBorder(BorderFactory.createEtchedBorder());
            setBorderPainted(false);
            //Making nice rollover effect
            //we use the same listener for all buttons
            addMouseListener(buttonMouseListener);
            setRolloverEnabled(true);
            //Close the proper tab by clicking the button
            addActionListener(this);
        }

        public void actionPerformed(ActionEvent e) {
            int i = pane.indexOfComponent(this);
            if (i != -1) {
                pane.remove(i);
            }
        }

        //we don't want to update UI for this button
        public void updateUI() {
        }

        //paint the cross
        protected void paintComponent(Graphics g) {
            super.paintComponent(g);
            Graphics2D g2 = (Graphics2D) g.create();
            //shift the image for pressed buttons
            if (getModel().isPressed()) {
                g2.translate(1, 1);
            }
            g2.setStroke(new BasicStroke(2));
            g2.setColor(Color.BLACK);
            if (getModel().isRollover()) {
                g2.setColor(Color.MAGENTA);
            }
            int delta = 6;
            g2.drawLine(delta, delta, getWidth() - delta - 1, getHeight() - delta - 1);
            g2.drawLine(getWidth() - delta - 1, delta, delta, getHeight() - delta - 1);
            g2.dispose();
        }
    }

    private final static MouseListener buttonMouseListener = new MouseAdapter() {
        public void mouseEntered(MouseEvent e) {
            Component component = e.getComponent();
            if (component instanceof AbstractButton) {
                AbstractButton button = (AbstractButton) component;
                button.setBorderPainted(true);
            }
        }

        public void mouseExited(MouseEvent e) {
            Component component = e.getComponent();
            if (component instanceof AbstractButton) {
                AbstractButton button = (AbstractButton) component;
                button.setBorderPainted(false);
            }
        }
    };

}

// End QueryPanel.java