/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata;

import com.jaspersoft.jasperserver.api.common.domain.ValidationResult;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.MondrianConnection;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.MondrianXMLADefinition;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.OlapClientConnection;
import com.jaspersoft.jasperserver.api.metadata.olap.domain.OlapUnit;
import com.jaspersoft.jasperserver.api.metadata.olap.service.OlapConnectionService;
import com.jaspersoft.jasperserver.api.metadata.olap.service.impl.OlapConnectionServiceImpl;

import junit.framework.TestCase;

/**
 * @author jshih
 */

//
// todo: TK: The resources created in this unit-test have the old uri path of /olap.
//           The Standard Sample Data uses /analysis. So, are the resources created in
//           this unit-test used (or are these deleted somewhere else)?
//

public class OlapClientTest extends TestCase {

	protected final Log log = LogFactory.getLog(this.getClass());

	protected final String OLAP_SCHEMAS = "/analysis/schemas/";

	protected final String OLAP_DATASOURCES = "/analysis/datasources/";

	protected final String OLAP_CONNECTIONS = "/analysis/connections/";

	protected final String OLAP_DEFINITIONS = "/analysis/xmla/definitions/";

	protected final String OLAP_VIEWS = "/analysis/views/";

    protected final String MDX_QUERY =
    	"SELECT {[Measures].[Unit Sales]} ON COLUMNS, " +
    	"{[Time].[1997].[Q1]} ON ROWS FROM [Sales]";

    protected final String XML_SCHEMA = "/analysis/schemas/Foodmart";

    private final static String strSchemaFoodmart = "FoodmartSchema";

    private final static String strSchemaSugarCrm = "SugarCRMSchema";

    private final static String strDatasourceFoodmart = "FoodmartDataSourceJNDI";

    private final static String strDatasourceSugarCrm = "SugarCRMDataSource";

	static String strConnection  = "Foodmart";

	private OlapClientConnection connectionOriginal;

	private ClassPathXmlApplicationContext appContext;

	private Properties jdbcProps;

    private RepositoryService repositoryService;

    private OlapConnectionService connectionService;

	protected void setUp() throws Exception
	{

		loadJdbcProps();


		// metadata.additionalSettings=hibernateRepositoryAdditionalSettings.xml

		if (jdbcProps.getProperty("metadata.additionalSettings") == null) {
			appContext = new ClassPathXmlApplicationContext(
					new String[]{"hibernateConfig.xml", "viewService.xml", "userAuthorityService.xml"});
		} else {
			appContext = new ClassPathXmlApplicationContext(
					new String[]{"hibernateConfig.xml", jdbcProps.getProperty("metadata.additionalSettings"), "viewService.xml", "userAuthorityService.xml"});
		}

		setRepositoryService((RepositoryService) appContext.getBean("repositoryService"));

		setConnectionService((OlapConnectionService) appContext.getBean("olapConnectionService"));
	}

	protected void tearDown() {

        if (connectionOriginal != null) {
            connectionOriginal =
                (OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);

            // change schema back to Foodmart
            ((MondrianConnection) connectionOriginal).setSchemaReference(OLAP_SCHEMAS + strSchemaFoodmart);

            // change datasource back to Foodmart
            ((MondrianConnection) connectionOriginal).setDataSourceReference(OLAP_DATASOURCES + strDatasourceFoodmart);

            repositoryService.saveResource(null, connectionOriginal);
        }
		appContext.close();
	}

    // tests

    static FileResource schemaFoodmart;

    static FileResource schemaSugarCrm;

    public void testSchemasExist()
	{
    	schemaFoodmart =
    		(FileResource) repositoryService.getResource(null, OLAP_SCHEMAS + strSchemaFoodmart);

        assertNotNull("Foodmart schema does not exist", schemaFoodmart);

    	assertNotNull("no data in Foodmart schema", schemaFoodmart.getDataStream());

    	schemaSugarCrm =
    		(FileResource) repositoryService.getResource(null, OLAP_SCHEMAS + strSchemaSugarCrm);

        assertNotNull("SugarCRM schema does not exist", schemaSugarCrm);

    	assertNotNull("no data in SugarCRM schema", schemaSugarCrm.getDataStream());

	}

	static ReportDataSource datasourceFoodmart;

   	static ReportDataSource datasourceSugarCrm;

    public void testDatasourcesExist()
    {
    	datasourceFoodmart =
    		(ReportDataSource) repositoryService.getResource(null, OLAP_DATASOURCES + strDatasourceFoodmart);

        assertNotNull("Foodmart data source does not exist", datasourceFoodmart);

    	datasourceSugarCrm =
    		(ReportDataSource) repositoryService.getResource(null, OLAP_DATASOURCES + strDatasourceSugarCrm);

        assertNotNull("SugarCRM data source does not exist", datasourceSugarCrm);

//    	String strDsFoodmart = "FoodmartDS";
//        JdbcReportDataSource datasource =
//        	(JdbcReportDataSource) repositoryService.newResource( null, JdbcReportDataSource.class );
//        datasource.setName(strDsFoodmart);
//        datasource.setLabel(strDsFoodmart);
//        datasource.setDescription(strDsFoodmart);
//        datasource.setDriverClass(jdbcProps.getProperty("foodmart.jdbc.driverClassName"));
//        datasource.setConnectionUrl(jdbcProps.getProperty("foodmart.jdbc.url"));
//        datasource.setUsername(jdbcProps.getProperty("foodmart.jdbc.username"));
//        datasource.setPassword(jdbcProps.getProperty("foodmart.jdbc.password"));
////        getConnectionService().saveResource( null, "/olap/datasources/", datasource );
//        repositoryService.saveResource(null, datasource);
//
//    	datasourceFoodmart =
//    		(ReportDataSource) repositoryService.getResource(null, OLAP_DATASOURCES + strDsFoodmart);
//
//    	assertTrue(datasourceFoodmart.getURIString().equals("/olap/datasources/FoodmartDS"));
    }

    private void setupTestMondrianConnection() {

    	connectionOriginal =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);

    	assertNotNull(connectionOriginal);

    	assertTrue(connectionOriginal instanceof MondrianConnection);

    	assertTrue(((MondrianConnection) connectionOriginal).getSchema().getReferenceLookup().getName()
    			.equals("FoodmartSchema"));

    	assertTrue(((MondrianConnection) connectionOriginal).getSchema().getReferenceURI()
    			.equals(OLAP_SCHEMAS + "FoodmartSchema"));
    }

	public void testUpdateMondrianConnectionSchema() {

		setupTestMondrianConnection();

    	// change schema to SugarCRM
    	((MondrianConnection) connectionOriginal).setSchemaReference(OLAP_SCHEMAS + strSchemaSugarCrm);

    	repositoryService.saveResource(null, connectionOriginal);

    	OlapClientConnection connectionNew =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);

    	assertNotNull(connectionNew);

    	assertTrue(connectionNew instanceof MondrianConnection);

    	assertTrue(((MondrianConnection) connectionNew).getSchema().getReferenceLookup().getName()
    			.equals("SugarCRMSchema"));

    	assertTrue(((MondrianConnection) connectionNew).getSchema().getReferenceURI()
    			.equals(OLAP_SCHEMAS + "SugarCRMSchema"));

    }

    public void testUpdateMondrianConnectionDatasource() {

    	setupTestMondrianConnection();

    	// new datasource
    	((MondrianConnection) connectionOriginal).setDataSourceReference(OLAP_DATASOURCES + strDatasourceSugarCrm);

    	repositoryService.saveResource(null, connectionOriginal);

    	OlapClientConnection connectionNew =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strConnection);

    	assertNotNull(connectionNew);

    	assertTrue(connectionNew instanceof MondrianConnection);

    	assertTrue(((MondrianConnection) connectionNew).getDataSource().getReferenceLookup().getName()
    			.equals("SugarCRMDataSource"));

    	assertTrue(((MondrianConnection) connectionNew).getDataSource().getReferenceURI()
    			.equals(OLAP_DATASOURCES + "SugarCRMDataSource"));

    }

    public void testXmlaDefinitionExists()
    {
    	String strDefinition = "FoodmartXmlaDefinition";

    	MondrianXMLADefinition definition =
    		(MondrianXMLADefinition) repositoryService.getResource(null, OLAP_DEFINITIONS + strDefinition);

    	assertNotNull(definition.getURIString()); // null
    }

    public void testXmlaConnectionExists()
    {
    	String strXMLAConnection = "FoodmartXmlaConnection";

    	OlapClientConnection connection =
    		(OlapClientConnection) repositoryService.getResource(null, OLAP_CONNECTIONS + strXMLAConnection);

    	assertNotNull(connection);
    }

    public void testOlapConnectionlParsesQuery()
    {
    	String olapViewRef = OLAP_CONNECTIONS + "Foodmart";

    	String olapQuery = MDX_QUERY;

        mondrian.olap.Connection mondrianConnection =
        	((OlapConnectionServiceImpl) getConnectionService()).getMondrianConnection(null, olapViewRef);

        mondrian.olap.Query mondrianQuery = mondrianConnection.parseQuery(olapQuery);

        assertNotNull(mondrianQuery);

        assertTrue(mondrianQuery.getCube().toString().equals("Sales"));
    }

    public void testOlapViewQuery()
    {
    	String strOlapView = "Foodmart_sample";

    	OlapUnit olapView =
    		(OlapUnit) repositoryService.getResource(null, OLAP_VIEWS + strOlapView);

    	assertNotNull(olapView.getOlapClientConnection());

    	assertTrue(olapView.getMdxQuery()
    			.equals(OlapConnectionTest.SAMPLE_FOODMART_MDX_QUERY));
    }

    // helpers

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException
	{
		jdbcProps = new Properties();

		String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");

		BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));

		jdbcProps.load(is);

		is.close();

		return jdbcProps;
	}

    public RepositoryService getRepositoryService()
    {
        return repositoryService;
    }

    public void setRepositoryService(RepositoryService repository)
    {
        this.repositoryService = repository;
    }

    public OlapConnectionService getConnectionService()
    {
        return connectionService;
    }

    public void setConnectionService(OlapConnectionService connection)
    {
        this.connectionService = connection;
    }
}
