/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.springframework.test.AbstractDependencyInjectionSpringContextTests;

import com.jaspersoft.jasperserver.api.metadata.common.domain.DataType;
import com.jaspersoft.jasperserver.api.metadata.common.domain.FileResource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.InputControl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ListOfValues;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ListOfValuesItem;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Query;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.DataTypeImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.ListOfValuesItemImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.PasswordCipherer;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.BeanReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.JndiJdbcReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportDataSource;
import com.jaspersoft.jasperserver.api.metadata.jasperreports.domain.ReportUnit;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterCriteria;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterElementDisjunction;
import com.jaspersoft.jasperserver.api.metadata.view.domain.FilterElementOr;

/**
 * @author Lucian Chirita (lucianc@users.sourceforge.net)
 * @version $Id: HibernateRepositoryTest.java 15275 2009-03-13 20:11:55Z andy21ca $
 */
public class HibernateRepositoryTest extends AbstractDependencyInjectionSpringContextTests {

    private Properties jdbcProps;
    private RepositoryService repositoryService;

    public RepositoryService getRepositoryService() {
		return repositoryService;
	}

	public void setRepositoryService(RepositoryService repositoryService) {
		this.repositoryService = repositoryService;
	}

	public HibernateRepositoryTest(String name) {
        super(name);
		setAutowireMode(AUTOWIRE_BY_NAME);
    }

    protected String[] getConfigLocations() {
		try {
			loadJdbcProps();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		// metadata.additionalSettings=hibernateRepositoryAdditionalSettings.xml

		if (jdbcProps.getProperty("metadata.additionalSettings") == null) {
			return
					new String[]{"hibernateConfig.xml", "viewService.xml", "userAuthorityService.xml", "methodAndObjectSecurity.xml"};
		} else {
			return
					new String[]{"hibernateConfig.xml", jdbcProps.getProperty("metadata.additionalSettings"), "viewService.xml", "userAuthorityService.xml", "methodAndObjectSecurity.xml"};
		}
    }

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException {
        jdbcProps = new Properties();
        String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
        BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
        jdbcProps.load(is);
        is.close();
        return jdbcProps;
    }

    // TODO disabled for now
    public void disabled_testDeleteFolder() {
        assertNotNull("Repo not null", repositoryService);

        Folder tmpFolder = new FolderImpl();
        setCommon(tmpFolder, "tmp_foo");
        repositoryService.saveFolder(null, tmpFolder);

        Folder subtmpFolder = new FolderImpl();
        setCommon(subtmpFolder, "sub");
        subtmpFolder.setParentFolder(tmpFolder);
        repositoryService.saveFolder(null, subtmpFolder);

        JndiJdbcReportDataSource ds = (JndiJdbcReportDataSource) repositoryService.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(ds, "dummyDS");
        ds.setJndiName("foo");
        ds.setParentFolder(tmpFolder);
        repositoryService.saveResource(null, ds);

        JndiJdbcReportDataSource subds = (JndiJdbcReportDataSource) repositoryService.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(subds, "subDS");
        subds.setJndiName("subfoo");
        subds.setParentFolder(subtmpFolder);
        repositoryService.saveResource(null, subds);

        Resource dsResource = repositoryService.getResource(null, "/tmp_foo/dummyDS");
        assertNotNull(dsResource);
        assertTrue(dsResource instanceof JndiJdbcReportDataSource);
        ds = (JndiJdbcReportDataSource) dsResource;
        assertEquals("foo", ds.getJndiName());

        Resource subdsResource = repositoryService.getResource(null, "/tmp_foo/sub/subDS");
        assertNotNull(subdsResource);
        assertTrue(subdsResource instanceof JndiJdbcReportDataSource);
        subds = (JndiJdbcReportDataSource) subdsResource;
        assertEquals("subfoo", subds.getJndiName());

        repositoryService.deleteFolder(null, "/tmp_foo");

        List folders = repositoryService.getSubFolders(null, "/");
        for (Iterator it = folders.iterator(); it.hasNext();) {
            Folder folder = (Folder) it.next();
            assertFalse("tmp_foo".equals(folder.getName()));
        }

        dsResource = repositoryService.getResource(null, "/tmp_foo/dummyDS");
        assertNull(dsResource);

        subdsResource = repositoryService.getResource(null, "/tmp_foo/dummyDS");
        assertNull(subdsResource);
    }

    // TODO disabled for now
    public void disabled_testDeleteRollback() {
        assertNotNull("Repo not null", repositoryService);

        Folder tmp1 = new FolderImpl();
        setCommon(tmp1, "tmp1");
        repositoryService.saveFolder(null, tmp1);

        Folder tmp2 = new FolderImpl();
        setCommon(tmp2, "tmp2");
        repositoryService.saveFolder(null, tmp2);

        tmp1 = repositoryService.getFolder(null, "/tmp1");
        assertNotNull(tmp1);
        assertEquals("tmp1", tmp1.getName());

        tmp2 = repositoryService.getFolder(null, "/tmp2");
        assertNotNull(tmp2);
        assertEquals("tmp2", tmp2.getName());

        boolean exception = false;
        try {
            repositoryService.delete(null, null, new String[] { "/tmp1", "/tmp_zzz" });
        } catch (Exception e) {
            exception = true;
        }
        assertTrue(exception);


        tmp1 = repositoryService.getFolder(null, "/tmp1");
        assertNotNull(tmp1);
        assertEquals("tmp1", tmp1.getName());

        repositoryService.delete(null, null, new String[] { "/tmp1", "/tmp2" });

        tmp1 = repositoryService.getFolder(null, "/tmp1");
        assertNull(tmp1);

        tmp2 = repositoryService.getFolder(null, "/tmp2");
        assertNull(tmp2);
    }

    public void testFolderUpdate() {
        assertNotNull("Repo not null", repositoryService);

        Folder folder = new FolderImpl();
        setCommon(folder, "test_update");
        repositoryService.saveFolder(null, folder);

        JndiJdbcReportDataSource ds = (JndiJdbcReportDataSource) repositoryService.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(ds, "fooDS");
        ds.setJndiName("foo");
        ds.setParentFolder(folder);
        repositoryService.saveResource(null, ds);

        folder = repositoryService.getFolder(null, "/test_update");
        assertNotNull(folder);
        assertEquals("test_update", folder.getName());
        assertEquals("test_update_label", folder.getLabel());

        folder.setLabel("updated");
        repositoryService.saveFolder(null, folder);

        folder = repositoryService.getFolder(null, "/test_update");
        assertNotNull(folder);
        assertEquals("test_update", folder.getName());
        assertEquals("updated", folder.getLabel());

        Resource resource = repositoryService.getResource(null, "/test_update/fooDS");
        assertNotNull(resource);
        assertTrue(resource instanceof JndiJdbcReportDataSource);
        ds = (JndiJdbcReportDataSource) resource;
        assertEquals("fooDS", ds.getName());
        assertEquals("foo", ds.getJndiName());

        repositoryService.deleteFolder(null, "/test_update");
        folder = repositoryService.getFolder(null, "/test_update");
        assertNull(folder);
    }

    public void testRepo() {
        assertNotNull("Repo not null", repositoryService);

        write();

        read();

        readFolders();

        update();

        optimisticLocking();

        list();

        resources();

        // TODO disabled to allow build to run
        //deleteRollback();
    }

    private void deleteRollback() {
        boolean exception = false;
        try {
            repositoryService.delete(null, new String[]{"/datasources/JServerJdbcDS"}, null);
        } catch (Exception e) {
            exception = true;
        }
        assertTrue(exception);

        Resource resource = repositoryService.getResource(null, "/datasources/JServerJdbcDS");
        assertNotNull(resource);
        assertTrue(resource instanceof JdbcReportDataSource);
    }

    private void write() {
        Folder dsFolder = new FolderImpl();
        dsFolder.setName("datasources");
        dsFolder.setLabel("Data Sources");
        dsFolder.setDescription("Data Sources used by reports");
        repositoryService.saveFolder(null, dsFolder);

        createJndiDS();
        createJdbcDS();
        createBeanDS();
        createTableModelDS();

        Folder reportsFolder = new FolderImpl();
        reportsFolder.setName("reports");
        reportsFolder.setLabel("Reports");
        reportsFolder.setDescription("Reports");
        repositoryService.saveFolder(null, reportsFolder);

        Folder samplesFolder = new FolderImpl();
        samplesFolder.setName("samples");
        samplesFolder.setLabel("Samples");
        samplesFolder.setDescription("Samples");
        samplesFolder.setParentFolder(reportsFolder);
        repositoryService.saveFolder(null, samplesFolder);

        createAllAccounts(samplesFolder);
        createSalesByMonth(samplesFolder);
        createCustomDSReportFileResource(samplesFolder);
        createCustomDSReport(samplesFolder);
        createTableModelDSReport(samplesFolder);
        createEmployeeAccounts(samplesFolder);
        createEmployees(samplesFolder);
        createParamMany(samplesFolder);
        createAllCharts(samplesFolder);
        createOFCReport(samplesFolder);

        createImage();

        Folder olapFolder = new FolderImpl();
        olapFolder.setName("analysis");
        olapFolder.setLabel("Analysis Components");
        olapFolder.setDescription("Analysis Components");
        repositoryService.saveFolder(null, olapFolder);

        Folder connectionsFolder = new FolderImpl();
        connectionsFolder.setName("connections");
        connectionsFolder.setLabel("Analysis Connections");
        connectionsFolder.setDescription("Connections used by Analysis");
        connectionsFolder.setParentFolder(olapFolder);
        repositoryService.saveFolder(null, connectionsFolder);

        Folder schemasFolder = new FolderImpl();
        schemasFolder.setName("schemas");
        schemasFolder.setLabel("Analysis Schemas");
        schemasFolder.setDescription("Schemas used by Analysis");
        schemasFolder.setParentFolder(olapFolder);
        repositoryService.saveFolder(null, schemasFolder);

        Folder olapDsFolder = new FolderImpl();
        olapDsFolder.setName("datasources");
        olapDsFolder.setLabel("Analysis Data Sources");
        olapDsFolder.setDescription("Data sources used by Analysis");
        olapDsFolder.setParentFolder(olapFolder);
        repositoryService.saveFolder(null, olapDsFolder);

        Folder olapViewsFolder = new FolderImpl();
        olapViewsFolder.setName("views");
        olapViewsFolder.setLabel("Analysis Views");
        olapViewsFolder.setDescription("Analysis Views");
        olapViewsFolder.setParentFolder(olapFolder);
        repositoryService.saveFolder(null, olapViewsFolder);
    }

    private void createImage() {
        Folder folder = new FolderImpl();
        folder.setName("images");
        folder.setLabel("Images");
        folder.setDescription("Folder containing reusable images");
        repositoryService.saveFolder(null, folder);

        FileResource image = (FileResource) repositoryService.newResource(null, FileResource.class);
        image.setFileType(FileResource.TYPE_IMAGE);
        image.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        image.setName("JRLogo");
        image.setLabel("JR logo");
        image.setDescription("JR logo");
        image.setParentFolder(folder);

        repositoryService.saveResource(null, image);
    }

    private void createJndiDS() {
        JndiJdbcReportDataSource datasource = (JndiJdbcReportDataSource) repositoryService.newResource(null, JndiJdbcReportDataSource.class);
        datasource.setName("JServerJNDIDS");
        datasource.setLabel("JServer JNDI data source");
        datasource.setDescription("JServer JNDI data source");
        datasource.setJndiName(jdbcProps.getProperty("test.jndi"));
        datasource.setParentFolder("/datasources");

        repositoryService.saveResource(null, datasource);
    }

    private void createJdbcDS() {
        JdbcReportDataSource datasource = (JdbcReportDataSource) repositoryService.newResource(null, JdbcReportDataSource.class);
        datasource.setName("JServerJdbcDS");
        datasource.setLabel("JServer Jdbc data source");
        datasource.setDescription("JServer Jdbc data source");
        datasource.setParentFolder("/datasources");

        datasource.setDriverClass(jdbcProps.getProperty("test.jdbc.driverClassName"));
        datasource.setConnectionUrl(jdbcProps.getProperty("test.jdbc.url"));
        datasource.setUsername(jdbcProps.getProperty("test.jdbc.username"));
        datasource.setPassword(jdbcProps.getProperty("test.jdbc.password"));

        repositoryService.saveResource(null, datasource);
    }

    private void createBeanDS() {
        BeanReportDataSource datasource = (BeanReportDataSource) repositoryService.newResource(null, BeanReportDataSource.class);
        datasource.setName("CustomDSFromBean");
        datasource.setLabel("Custom data source from a bean");
        datasource.setDescription("A custom data source through a bean");
        datasource.setParentFolder("/datasources");

        datasource.setBeanName("customDataSourceService");

        repositoryService.saveResource(null, datasource);
    }

    private void createTableModelDS() {
        BeanReportDataSource datasource = (BeanReportDataSource) repositoryService.newResource(null, BeanReportDataSource.class);
        datasource.setName("CustomTableModelDS");
        datasource.setLabel("Custom data source from a table model");
        datasource.setDescription("A custom data source through a table model");
        datasource.setParentFolder("/datasources");

        datasource.setBeanName("customDataSourceServiceFactory");
        datasource.setBeanMethod("tableModelDataSource");

        repositoryService.saveResource(null, datasource);
    }

    private void createSalesByMonth(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("SalesByMonthReport");
        reportRes.setLabel("Sales By Month Jasper Report");
        reportRes.setDescription("Sales By Month Jasper Report");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/SalesByMonth.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("SalesByMonth");
        unit.setLabel("Sales By Month Report");
        unit.setDescription("Sales By Month Report");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        FileResource jar = (FileResource) repositoryService.newResource(null, FileResource.class);
        jar.setFileType(FileResource.TYPE_JAR);
        jar.readData(getClass().getResourceAsStream("/jars/scriptlet.jar"));
        setCommon(jar, "Scriptlet");
        unit.addResource(jar);

        FileResource img = (FileResource) repositoryService.newResource(null, FileResource.class);
        img.setFileType(FileResource.TYPE_IMAGE);
        img.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        setCommon(img, "Logo");
        unit.addResource(img);

        FileResource subrep = (FileResource) repositoryService.newResource(null, FileResource.class);
        subrep.setFileType(FileResource.TYPE_JRXML);
        subrep.readData(getClass().getResourceAsStream("/reports/jasper/SalesByMonthDetail.jrxml"));
        setCommon(subrep, "SalesByMonthDetail");
        unit.addResource(subrep);

        FileResource resBdl = (FileResource) repositoryService.newResource(null, FileResource.class);
        resBdl.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
        resBdl.readData(getClass().getResourceAsStream("/resource_bundles/sales.properties"));
        setCommon(resBdl, "sales.properties");
        unit.addResource(resBdl);

        FileResource resBdl_ro = (FileResource) repositoryService.newResource(null, FileResource.class);
        resBdl_ro.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
        resBdl_ro.readData(getClass().getResourceAsStream("/resource_bundles/sales_ro.properties"));
        setCommon(resBdl_ro, "sales_ro.properties");
        unit.addResource(resBdl_ro);

        InputControl textInputCtrl = (InputControl) repositoryService.newResource(null, InputControl.class);
        setCommon(textInputCtrl, "TextInputControl");
        textInputCtrl.setType(InputControl.TYPE_SINGLE_VALUE);
        textInputCtrl.setMandatory(false);
        textInputCtrl.setReadOnly(false);
        textInputCtrl.setVisible(true);
        //FIXME textInputCtrl.setSize(new Integer(30));
        textInputCtrl.setLabel("Text Input Control");
        textInputCtrl.setName("TextInput");
        DataType dataType = new DataTypeImpl();
        dataType.setName("test");
        dataType.setLabel("test");
        dataType.setType(DataType.TYPE_NUMBER);
        textInputCtrl.setDataType(dataType);
        unit.addInputControl(textInputCtrl);

        InputControl checkboxInputControl = (InputControl) repositoryService.newResource(null, InputControl.class);
        setCommon(checkboxInputControl, "CheckboxInputControl");
        checkboxInputControl.setType(InputControl.TYPE_BOOLEAN);
        checkboxInputControl.setMandatory(true);
        checkboxInputControl.setReadOnly(false);
        checkboxInputControl.setVisible(true);
        checkboxInputControl.setLabel("Checkbox Input Control");
        checkboxInputControl.setName("CheckboxInput");
        unit.addInputControl(checkboxInputControl);

        InputControl listInputControl = (InputControl) repositoryService.newResource(null, InputControl.class);
        setCommon(listInputControl, "ListInputControl");
        listInputControl.setType(InputControl.TYPE_SINGLE_SELECT_LIST_OF_VALUES);
        listInputControl.setMandatory(true);
        listInputControl.setReadOnly(false);
        listInputControl.setVisible(true);
        listInputControl.setLabel("List Input Control");
        listInputControl.setName("ListInput");

        ListOfValues values = (ListOfValues) repositoryService.newResource(null, ListOfValues.class);
        values.setName("List_of_values");
        values.setLabel("List of values label");
        values.setDescription("List of values description");
        ListOfValuesItem item = new ListOfValuesItemImpl();
        item.setLabel("An item");
        item.setValue("1");
        values.addValue(item);
        item = new ListOfValuesItemImpl();
        item.setLabel("Another item");
        item.setValue("2");
        values.addValue(item);
        item = new ListOfValuesItemImpl();
        item.setLabel("Yet another item");
        item.setValue("3");
        values.addValue(item);
        listInputControl.setListOfValues(values);

        dataType = new DataTypeImpl();
        dataType.setName("test");
        dataType.setLabel("test");
        dataType.setType(DataType.TYPE_TEXT);
        listInputControl.setDataType(dataType);

        unit.addInputControl(listInputControl);

        createDateDatatype();

        InputControl dateInputCtrl = (InputControl) repositoryService.newResource(null, InputControl.class);
        setCommon(dateInputCtrl, "DateInput");
        dateInputCtrl.setType(InputControl.TYPE_SINGLE_VALUE);
        dateInputCtrl.setMandatory(false);
        dateInputCtrl.setReadOnly(false);
        dateInputCtrl.setVisible(true);
        dateInputCtrl.setDataTypeReference("/datatypes/date");
        unit.addInputControl(dateInputCtrl);

        InputControl queryInputCtrl = (InputControl) repositoryService.newResource(null, InputControl.class);
        setCommon(queryInputCtrl, "QueryInput");
        queryInputCtrl.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);
        queryInputCtrl.setMandatory(false);
        queryInputCtrl.setReadOnly(false);
        queryInputCtrl.setVisible(true);
        queryInputCtrl.setQueryValueColumn("user_name");
        queryInputCtrl.addQueryVisibleColumn("first_name");
        queryInputCtrl.addQueryVisibleColumn("last_name");
        Query query = (Query) repositoryService.newResource(null, Query.class);
        setCommon(query, "testQuery");
        query.setLanguage("sql");
        query.setSql("select user_name, first_name, last_name from users");
        queryInputCtrl.setQuery(query);
        unit.addInputControl(queryInputCtrl);

        repositoryService.saveResource(null, unit);
    }

    private void createEmployees(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        setCommon(reportRes, "EmployeesJRXML");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/Employees.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("Employees");
        unit.setLabel("Employee List");
        unit.setDescription("Employee List");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        repositoryService.saveResource(null, unit);
    }

    private void createEmployeeAccounts(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        setCommon(reportRes, "EmployeeAccountsJRXML");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/EmployeeAccounts.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("EmployeeAccounts");
        unit.setLabel("Employee Accounts");
        unit.setDescription("List of Accounts per Employee");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        InputControl empIC = (InputControl) repositoryService.newResource(null, InputControl.class);
        empIC.setName("EmployeeID");
        empIC.setLabel("Employee");
        empIC.setMandatory(true);

        Query empQuery = (Query) repositoryService.newResource(null, Query.class);
        empQuery.setName("EmployeeQuery");
        empQuery.setLabel("Employee Query");
        empQuery.setLanguage("sql");
        empQuery.setSql("SELECT id, user_name FROM users WHERE employee_status = 'Active'");

        empIC.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);
        empIC.setQuery(empQuery);
        empIC.setQueryValueColumn("id");
        empIC.addQueryVisibleColumn("user_name");

        unit.addInputControl(empIC);

        repositoryService.saveResource(null, unit);
    }

    private void createParamMany(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        setCommon(reportRes, "ParametersJRXML");

        InputStream jrxml = getClass().getResourceAsStream("/reports/jasper/ParamMany.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("Freight");
        unit.setLabel("Freight Report");
        unit.setDescription("Freight Report with Saved Parameters");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        // Country
        InputControl countryIC = (InputControl) repositoryService.newResource(null, InputControl.class);
        countryIC.setName("Country");
        countryIC.setLabel("Country");
        countryIC.setDescription("Country");
        countryIC.setMandatory(true);
        countryIC.setType(InputControl.TYPE_SINGLE_SELECT_QUERY);

        Query countryQ = (Query) repositoryService.newResource(null, Query.class);
        countryQ.setName("CountryQuery");
        countryQ.setLabel("CountryQuery");
        countryQ.setLanguage("sql");
        countryQ.setDataSourceReference("/datasources/JServerJNDIDS");
        countryQ.setSql("select distinct SHIPCOUNTRY from ORDERS");
        countryIC.setQuery(countryQ);

        countryIC.setQueryValueColumn("SHIPCOUNTRY");
        countryIC.addQueryVisibleColumn("SHIPCOUNTRY");

        unit.addInputControl(countryIC);

        // Request Date
        InputControl requestDateIC = (InputControl) repositoryService.newResource(null, InputControl.class);
        requestDateIC.setName("RequestDate");
        requestDateIC.setLabel("RequestDate");
        requestDateIC.setDescription("RequestDate");
        requestDateIC.setMandatory(false);
        requestDateIC.setType(InputControl.TYPE_SINGLE_VALUE);

        DataType dateDT = (DataType) repositoryService.newResource(null, DataType.class);
        setCommon(dateDT, "Date");
        dateDT.setType(DataType.TYPE_DATE);
        requestDateIC.setDataType(dateDT);

        unit.addInputControl(requestDateIC);

        // Order Id
        InputControl orderIdIC = (InputControl) repositoryService.newResource(null, InputControl.class);
        orderIdIC.setName("OrderId");
        orderIdIC.setLabel("OrderId");
        orderIdIC.setDescription("OrderId");
        orderIdIC.setMandatory(false);
        orderIdIC.setType(InputControl.TYPE_SINGLE_VALUE);

        DataType numberDT = (DataType) repositoryService.newResource(null, DataType.class);
        setCommon(numberDT, "Number");
        numberDT.setType(DataType.TYPE_NUMBER);
        orderIdIC.setDataType(numberDT);

        unit.addInputControl(orderIdIC);

        repositoryService.saveResource(null, unit);
    }

    protected void createDateDatatype() {
        Folder folder = new FolderImpl();
        folder.setName("datatypes");
        folder.setLabel("Input data types");
        repositoryService.saveFolder(null, folder);

        DataType dateDataType = new DataTypeImpl();
        setCommon(dateDataType, "date");
        dateDataType.setType(DataType.TYPE_DATE);
        dateDataType.setParentFolder(folder);
        repositoryService.saveResource(null, dateDataType);
    }

    private void setCommon(Resource res, String id) {
        res.setName(id);
        res.setLabel(id + "_label");
        res.setDescription(id + " description");
    }

    private void createAllAccounts(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("AllAccountsReport");
        reportRes.setLabel("All Accounts Jasper Report");
        reportRes.setDescription("All Accounts Jasper Report");
        reportRes.setParentFolder(folder);

        InputStream jrxml = getClass().getResourceAsStream(
                "/reports/jasper/AllAccounts.jrxml");
        reportRes.readData(jrxml);

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("AllAccounts");
        unit.setLabel("All Accounts Report");
        unit.setDescription("All Accounts Report");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        FileResource res1 = (FileResource) repositoryService.newResource(null, FileResource.class);
        res1.setFileType(FileResource.TYPE_IMAGE);
        res1.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        res1.setName("AllAccounts_Res1");
        res1.setLabel("AllAccounts_Res1");
        res1.setDescription("AllAccounts_Res1");
        unit.addResource(res1);

        FileResource res2 = (FileResource) repositoryService.newResource(null, FileResource.class);
        res2.setFileType(FileResource.TYPE_IMAGE);
        res2.readData(getClass().getResourceAsStream("/images/logo.jpg"));
        res2.setName("AllAccounts_Res2");
        res2.setLabel("AllAccounts_Res2");
        res2.setDescription("AllAccounts_Res2");
        unit.addResource(res2);

        repositoryService.saveResource(null, unit);
    }


    private void createAllCharts(Folder folder) 
    {
    	{
            FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
            reportRes.setFileType(FileResource.TYPE_JRXML);
            reportRes.setName("AllChartsReport");
            reportRes.setLabel("All Charts Report");
            reportRes.setDescription("All Charts Report");
            reportRes.setParentFolder(folder);

            InputStream jrxml = getClass().getResourceAsStream(
                    "/reports/jasper/AllChartsReport.jrxml");
            reportRes.readData(jrxml);

            ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
            unit.setName("AllChartsReport");
            unit.setLabel("All Charts Report");
            unit.setDescription("All Charts Report");
            unit.setParentFolder(folder);

            unit.setDataSourceReference("/datasources/JServerJNDIDS");
            unit.setMainReport(reportRes);

            FileResource resBdl = (FileResource) repositoryService.newResource(null, FileResource.class);
            resBdl.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
            resBdl.readData(getClass().getResourceAsStream("/resource_bundles/allCharts.properties"));
            setCommon(resBdl, "allCharts.properties");
            unit.addResource(resBdl);

            FileResource resBdl_ro = (FileResource) repositoryService.newResource(null, FileResource.class);
            resBdl_ro.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
            resBdl_ro.readData(getClass().getResourceAsStream("/resource_bundles/allCharts_ro.properties"));
            setCommon(resBdl_ro, "allCharts_ro.properties");
            unit.addResource(resBdl_ro);
            
            repositoryService.saveResource(null, unit);
    	}


    	{
            FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
            reportRes.setFileType(FileResource.TYPE_JRXML);
            reportRes.setName("AllChartsAegeanReport");
            reportRes.setLabel("All Charts Aegean Report");
            reportRes.setDescription("All Charts Aegean Report");
            reportRes.setParentFolder(folder);

            InputStream jrxml = getClass().getResourceAsStream(
                    "/reports/jasper/AllChartsAegeanReport.jrxml");
            reportRes.readData(jrxml);

            ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
            unit.setName("AllChartsAegeanReport");
            unit.setLabel("All Charts Aegean Report");
            unit.setDescription("All Charts Aegean Report");
            unit.setParentFolder(folder);

            unit.setDataSourceReference("/datasources/JServerJNDIDS");
            unit.setMainReport(reportRes);

            FileResource resBdl = (FileResource) repositoryService.newResource(null, FileResource.class);
            resBdl.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
            resBdl.readData(getClass().getResourceAsStream("/resource_bundles/allCharts.properties"));
            setCommon(resBdl, "allCharts.properties");
            unit.addResource(resBdl);

            FileResource resBdl_ro = (FileResource) repositoryService.newResource(null, FileResource.class);
            resBdl_ro.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
            resBdl_ro.readData(getClass().getResourceAsStream("/resource_bundles/allCharts_ro.properties"));
            setCommon(resBdl_ro, "allCharts_ro.properties");
            unit.addResource(resBdl_ro);
            
            repositoryService.saveResource(null, unit);
    	}

    	{
            FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
            reportRes.setFileType(FileResource.TYPE_JRXML);
            reportRes.setName("AllChartsEyeCandyReport");
            reportRes.setLabel("All Charts Eye Candy Report");
            reportRes.setDescription("All Charts Eye Candy Report");
            reportRes.setParentFolder(folder);

            InputStream jrxml = getClass().getResourceAsStream(
                    "/reports/jasper/AllChartsEyeCandyReport.jrxml");
            reportRes.readData(jrxml);

            ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
            unit.setName("AllChartsEyeCandyReport");
            unit.setLabel("All Charts Eye Candy Report");
            unit.setDescription("All Charts Eye Candy Report");
            unit.setParentFolder(folder);

            unit.setDataSourceReference("/datasources/JServerJNDIDS");
            unit.setMainReport(reportRes);

            FileResource resBdl = (FileResource) repositoryService.newResource(null, FileResource.class);
            resBdl.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
            resBdl.readData(getClass().getResourceAsStream("/resource_bundles/allCharts.properties"));
            setCommon(resBdl, "allCharts.properties");
            unit.addResource(resBdl);

            FileResource resBdl_ro = (FileResource) repositoryService.newResource(null, FileResource.class);
            resBdl_ro.setFileType(FileResource.TYPE_RESOURCE_BUNDLE);
            resBdl_ro.readData(getClass().getResourceAsStream("/resource_bundles/allCharts_ro.properties"));
            setCommon(resBdl_ro, "allCharts_ro.properties");
            unit.addResource(resBdl_ro);
            
            repositoryService.saveResource(null, unit);
    	}
    }


    private void createOFCReport(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("OpenFlashChartsJRXML");
        reportRes.setLabel("Open Flash Charts JRXML");
        reportRes.setDescription("JRXML that contains Open Flash Chart components.\nThis report uses Open Flash Chart V2 Beta 1, (C) 2007 John Glazebrook and released under the LGPL License.\nhttp://teethgrinder.co.uk/open-flash-chart-2/");

        reportRes.readData(getClass().getResourceAsStream("/reports/jasper/OpenFlashChartsReport.jrxml"));

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("OpenFlashChartsReport");
        unit.setLabel("Open Flash Charts Report");
        unit.setDescription("A report that displays Open Flash Charts");
        unit.setParentFolder(folder);

        unit.setDataSourceReference("/datasources/JServerJNDIDS");
        unit.setMainReport(reportRes);

        repositoryService.saveResource(null, unit);
    }


    private void createCustomDSReportFileResource(Folder folder) {
        FileResource reportRes = (FileResource) repositoryService.newResource(null, FileResource.class);
        reportRes.setFileType(FileResource.TYPE_JRXML);
        reportRes.setName("DataSourceReportTemplate");
        reportRes.setLabel("Report showing Custom Data Source");
        reportRes.setDescription("Report showing use of Custom Data Source via a bean");
        reportRes.setParentFolder(folder);

        InputStream jrxml = getClass().getResourceAsStream(
                "/reports/jasper/DataSourceReport.jrxml");
        reportRes.readData(jrxml);

        repositoryService.saveResource(null, reportRes);
    }

    private void createCustomDSReport(Folder folder) {

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("DataSourceReport");
        unit.setLabel("Report showing Custom Data Source");
        unit.setDescription("Report showing use of Custom Data Source via a bean");
        unit.setParentFolder(folder);

        unit.setMainReportReference("/reports/samples/DataSourceReportTemplate");
        unit.setDataSourceReference("/datasources/CustomDSFromBean");

        repositoryService.saveResource(null, unit);
    }

    private void createTableModelDSReport(Folder folder) {

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        unit.setName("DataSourceTableModel");
        unit.setLabel("Table Model Data Source");
        unit.setDescription("Report showing use of Custom Data Source via table model");
        unit.setParentFolder(folder);

        unit.setMainReportReference("/reports/samples/DataSourceReportTemplate");
        unit.setDataSourceReference("/datasources/CustomTableModelDS");
        unit.setMainReportReference("/reports/samples/DataSourceReportTemplate");

        repositoryService.saveResource(null, unit);
    }

    private void update() {
        ReportUnit unit = (ReportUnit) repositoryService.getResource(null, "/reports/samples/AllAccounts");
        assertEquals("AllAccounts", unit.getName());
        assertEquals("/reports/samples/AllAccounts", unit.getURIString());

        unit.setLabel("Accounts Report");

        FileResource removed = unit.removeResourceLocal("AllAccounts_Res1");
        assertNotNull(removed);

        FileResource res3 = (FileResource) repositoryService.newResource(null, FileResource.class);
        res3.setFileType(FileResource.TYPE_IMAGE);
        res3.readData(getClass().getResourceAsStream("/images/jasperreports.gif"));
        res3.setName("AllAccounts_Res3");
        res3.setLabel("AllAccounts_Res3");
        res3.setDescription("AllAccounts_Res3");
        unit.addResource(res3);

        FileResource res4 = (FileResource) repositoryService.newResource(null, FileResource.class);
        res4.setFileType(FileResource.TYPE_IMAGE);
        res4.setReferenceURI("/images/JRLogo");
        setCommon(res4, "LogoLink");
        unit.addResource(res4);

        repositoryService.saveResource(null, unit);

        unit = (ReportUnit) repositoryService.getResource(null, "/reports/samples/AllAccounts");
        assertTrue(unit.getLabel().startsWith("Accounts "));

        List resources = unit.getResources();
        assertNotNull(resources);
        assertEquals(3, resources.size());
    }

    private void read() {
        ReportUnit unit = (ReportUnit) repositoryService.getResource(null, "/reports/samples/AllAccounts");
        assertNotNull(unit);
        assertEquals(ReportUnit.class.getName(), unit.getResourceType());
        assertEquals("AllAccounts", unit.getName());

        ResourceReference dsRef = unit.getDataSource();
        assertNotNull(dsRef);
        assertFalse(dsRef.isLocal());
        assertEquals("/datasources/JServerJNDIDS", dsRef.getReferenceURI());

        ResourceReference mainReportRef = unit.getMainReport();
        assertNotNull(mainReportRef);
        assertTrue(mainReportRef.isLocal());
        Resource mainReport = mainReportRef.getLocalResource();
        assertTrue(mainReport instanceof FileResource);
        assertEquals("AllAccountsReport", mainReport.getName());

        List resources = unit.getResources();
        assertNotNull(resources);
        assertEquals(2, resources.size());

        FileResource img = (FileResource) repositoryService.getResource(null, "/images/JRLogo");
        assertNotNull(img);
        assertEquals("JRLogo", img.getName());

        datasources();
        parentFolder();
        propertyFilter();
        disjunctionFilter();
        orFilter();
    }

    private void datasources() {
        FilterCriteria criteria = FilterCriteria.createFilter(ReportDataSource.class);
        ResourceLookup[] lookups = repositoryService.findResource(null, criteria);
        assertNotNull(lookups);
        assertTrue(2 <= lookups.length);
        for (int i = 0; i < lookups.length; i++) {
            Resource res = repositoryService.getResource(null, lookups[i].getURIString());
            assertTrue(res instanceof ReportDataSource);
        }
    }

    private void parentFolder() {
        FilterCriteria criteria = FilterCriteria.createFilter();
        criteria.addFilterElement(FilterCriteria.createParentFolderFilter("/datasources"));
        ResourceLookup[] folderResources = repositoryService.findResource(null, criteria);
        assertNotNull(folderResources);
        assertTrue(folderResources.length > 0);
        for (int i = 0; i < folderResources.length; i++) {
            ResourceLookup lookup = folderResources[i];
            assertEquals("/datasources", lookup.getParentFolder());
            if (lookup.getName().equals("JServerJdbcDS")) {
                assertEquals(JdbcReportDataSource.class.getName(), lookup.getResourceType());
            } else if (lookup.getName().equals("JServerJNDIDS")) {
                assertEquals(JndiJdbcReportDataSource.class.getName(), lookup.getResourceType());
            }
        }
    }

    private void propertyFilter() {
        FilterCriteria criteria = FilterCriteria.createFilter(FileResource.class);
        criteria.addFilterElement(FilterCriteria.createPropertyEqualsFilter("fileType", FileResource.TYPE_IMAGE));
        ResourceLookup[] resources = repositoryService.findResource(null, criteria);
        assertNotNull(resources);
        assertTrue(resources.length > 0);
        for (int i = 0; i < resources.length; i++) {
            Resource res = repositoryService.getResource(null, resources[i].getURIString());
            assertTrue(res instanceof FileResource);
            FileResource fileRes = (FileResource) res;
            assertEquals(FileResource.class.getName(), fileRes.getResourceType());
            assertEquals(FileResource.TYPE_IMAGE, fileRes.getFileType());
        }
    }

    private void disjunctionFilter() {
        FilterCriteria criteria = FilterCriteria.createFilter();
        FilterElementDisjunction disjunction = criteria.addDisjunction();
        disjunction.addFilterElement(FilterCriteria.createPropertyLikeFilter("name", "%JdbcDS"));
        disjunction.addNegatedFilterElement(FilterCriteria.createParentFolderFilter("/datasources"));

        ResourceLookup[] resources = repositoryService.findResource(null, criteria);
        assertNotNull(resources);
        assertTrue(resources.length > 0);
        for (int i = 0; i < resources.length; i++) {
            ResourceLookup lookup = resources[i];
            assertTrue(lookup.getName().endsWith("JdbcDS") || !lookup.getParentFolder().equals("/datasources"));
        }
    }

    private void orFilter() {
        FilterCriteria criteria = FilterCriteria.createFilter();
        FilterElementOr or = criteria.addOr();
        or.setLeftHandSide(FilterCriteria.createParentFolderFilter("/images"));
        or.setRightHandSide(FilterCriteria.createParentFolderFilter("/datasources"));

        ResourceLookup[] resources = repositoryService.findResource(null, criteria);
        assertNotNull("Null resources found for or filter", resources);
        assertTrue("No resources found for or filter", resources.length > 0);
        boolean imagesFound = false;
        boolean dataSourcesFound = false;
        for (int i = 0; i < resources.length && (!imagesFound || !dataSourcesFound); i++) {
            ResourceLookup lookup = resources[i];
            if (lookup.getParentFolder().equals("/images")) {
                imagesFound = true;
            } else if (lookup.getParentFolder().equals("/datasources")) {
                dataSourcesFound = true;
            }
        }
        assertTrue("No images found for or filter", imagesFound);
        assertTrue("No data sources found for or filter", dataSourcesFound);
    }

    private void readFolders() {
        List folders = repositoryService.getAllFolders(null);
        assertNotNull(folders);
        assertTrue(5 <= folders.size());

        Iterator it = folders.iterator();
        Folder folder = (Folder) it.next();
        assertEquals("/", folder.getURIString());

        Set folderURIs = new HashSet();
        while (it.hasNext()) {
            folder = (Folder) it.next();
            folderURIs.add(folder.getURIString());
        }
        assertTrue(folderURIs.contains("/datasources"));
        assertTrue(folderURIs.contains("/images"));
        assertTrue(folderURIs.contains("/reports"));
        assertTrue(folderURIs.contains("/reports/samples"));

        List subFolders = repositoryService.getSubFolders(null, "/reports");
        assertNotNull(subFolders);
        assertEquals(1, subFolders.size());
        folder = (Folder) subFolders.get(0);
        assertEquals("/reports/samples", folder.getURIString());
    }

    private void list() {
        ResourceLookup[] units = repositoryService.findResource(null, FilterCriteria.createFilter(ReportUnit.class));
        assertNotNull(units);
        assertTrue(units.length >= 2);
    }

    private void resources() {
        Resource[] resources = repositoryService.findResource(null, null);
        assertNotNull(resources);
        assertTrue(resources.length >= 2);
    }

    private void optimisticLocking() {
        JndiJdbcReportDataSource ds1 = (JndiJdbcReportDataSource) repositoryService.getResource(null, "/datasources/JServerJNDIDS");
        ds1.setLabel(ds1.getLabel() + " Updated 1");

        JndiJdbcReportDataSource ds2 = (JndiJdbcReportDataSource) repositoryService.getResource(null, "/datasources/JServerJNDIDS");
        ds2.setLabel(ds1.getLabel() + " Updated 2");

        repositoryService.saveResource(null, ds1);

        boolean failed = false;
        try {
            repositoryService.saveResource(null, ds2);
        } catch (Exception e) {
            failed = true;
        }
        assertTrue(failed);
    }


    public void testLocalResourceReplace() {
        assertNotNull("Repo not null", repositoryService);

        Folder folder = new FolderImpl();
        setCommon(folder, "testLocalResourceReplace");
        folder.setParentFolder("/");
        repositoryService.saveFolder(null, folder);

        ReportUnit unit = (ReportUnit) repositoryService.newResource(null, ReportUnit.class);
        setCommon(unit, "unit");
        unit.setParentFolder(folder);

        FileResource rep = (FileResource) repositoryService.newResource(null, FileResource.class);
        setCommon(rep, "report");
        rep.readData(getClass().getResourceAsStream("/reports/jasper/SalesByMonth.jrxml"));
        unit.setMainReport(rep);

        JdbcReportDataSource jdbcDS = (JdbcReportDataSource) repositoryService.newResource(null, JdbcReportDataSource.class);
        setCommon(jdbcDS, "ds");
        jdbcDS.setConnectionUrl("jdbc:mysql://localhost:3306/jasperserver");
        jdbcDS.setDriverClass("com.mysql.jdbc.Driver");
        jdbcDS.setUsername("user");
        jdbcDS.setPassword("passwd");
        unit.setDataSource(jdbcDS);

        repositoryService.saveResource(null, unit);

        Resource res = repositoryService.getResource(null, "/testLocalResourceReplace/unit");
        assertNotNull(unit);
        assertTrue(res instanceof ReportUnit);
        unit = (ReportUnit) res;
        ResourceReference dsRef = unit.getDataSource();
        assertNotNull(dsRef);
        assertTrue(dsRef.isLocal());
        Resource ds = dsRef.getLocalResource();
        assertTrue(ds instanceof JdbcReportDataSource);
        assertEquals("ds", ds.getName());

        JndiJdbcReportDataSource jndiDS = (JndiJdbcReportDataSource) repositoryService.newResource(null, JndiJdbcReportDataSource.class);
        setCommon(jndiDS, "ds");
        jndiDS.setJndiName("jdbc/jserver");
        unit.setDataSource(jndiDS);

        repositoryService.saveResource(null, unit);

        res = repositoryService.getResource(null, "/testLocalResourceReplace/unit");
        assertNotNull(unit);
        assertTrue(res instanceof ReportUnit);
        unit = (ReportUnit) res;

        dsRef = unit.getDataSource();
        assertNotNull(dsRef);
        assertTrue(dsRef.isLocal());
        Resource dsLocal = dsRef.getLocalResource();
        assertEquals("ds", dsLocal.getName());
        assertTrue(dsLocal instanceof JndiJdbcReportDataSource);
        jndiDS = (JndiJdbcReportDataSource) dsLocal;
        assertEquals("jdbc/jserver", jndiDS.getJndiName());

        repositoryService.deleteFolder(null, "/testLocalResourceReplace");
    }
}
