package com.jaspersoft.jasperserver.war.action;

import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONObject;
import org.json.JSONArray;
import org.acegisecurity.Authentication;
import org.acegisecurity.context.SecurityContextHolder;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;
import com.jaspersoft.jasperserver.api.metadata.user.service.TenantService;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.domain.Tenant;
import com.jaspersoft.jasperserver.api.metadata.user.domain.TenantQualified;
import com.jaspersoft.jasperserver.api.metadata.user.domain.impl.hibernate.RepoTenant;
import com.jaspersoft.jasperserver.api.metadata.common.service.PaginatedOperationResult;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.PaginationHelper;
import com.jaspersoft.jasperserver.war.common.ConfigurationBean;
import com.jaspersoft.jasperserver.war.helper.UserJsonHelper;
import com.jaspersoft.jasperserver.war.helper.RoleJsonHelper;

import java.util.List;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Date;

/**
 * @author schubar
 */
public class UserManagerAction extends BaseManagerAction {

    public static final String UM_DEFAULT_USER = "defaultUser";
    public static final String UM_CONFIGURATION = "configuration";
    public static final String UM_TENANT_ID = "tenantId";
    public static final String UM_USER_DETAILS = "userDetails";
    public static final String UM_USER_NAME = "userName";
    public static final String UM_ROLE_NAME = "roleName";
    public static final String UM_USER_ROLES = "userRoles";
    public static final String UM_FIRST_RESULT = "firstResult";
    public static final String AJAX_RESPONSE_MODEL = "ajaxResponseModel";

    public static final String USER_NAME_SEPARATOR = "userNameSeparator";
    public static final String USER_NAME_NOT_SUPPORTED_SYMBOLS = "userNameNotSupportedSymbols";
    public static final String USER_DEFAULT_ROLE = "userDefaultRole";
    public static final String USER_PASSWORD_MASK = "passwordMask";

    protected final Log log = LogFactory.getLog(this.getClass());


    private UserAuthorityService userService;
    private MessageSource messages;

    private UserJsonHelper userHelper;
    private RoleJsonHelper roleHelper;

    public void setUserService(UserAuthorityService userService) {
        this.userService = userService;
    }

    public void setMessages(MessageSource messages) {
        this.messages = messages;
    }

    public void setUserHelper(UserJsonHelper userHelper) {
        this.userHelper = userHelper;
    }

    public void setRoleHelper(RoleJsonHelper roleHelper) {
        this.roleHelper = roleHelper;
    }

    private String getTenantId(RequestContext context) {
        String tenantId = getDecodedRequestParameter(context, UM_TENANT_ID);

        return (tenantId != null && tenantId.length() > 0) ? tenantId : null;
    }

    private String getUserJson(RequestContext context) {
        String json = context.getRequestParameters().get(UM_USER_DETAILS);

        return json;
    }

    private String getUserName(RequestContext context) {

        return getUserName(context, true);
    }

    private String getUserName(RequestContext context, boolean decode) {
        String name;
        if (decode) {
            name = getDecodedRequestParameter(context, UM_USER_NAME);
        } else {
            name = context.getRequestParameters().get(UM_USER_NAME);
        }

        return (name != null) ? name : "";
    }

    private String getRoleName(RequestContext context) {
        String name = getDecodedRequestParameter(context, UM_ROLE_NAME);

        return (name != null) ? name : "";
    }

    private Set getUserRoles(RequestContext context) throws Exception {
        String userRolesParam = getDecodedRequestParameter(context, UM_USER_ROLES);

        Set userRoles = new HashSet();

        if (userRolesParam != null && userRolesParam.length() > 0) {

            JSONArray userRolesArray = new JSONArray(userRolesParam);

            for (int i = 0; i < userRolesArray.length(); i ++) {

                userRoles.add(userRolesArray.getString(i));
            }
        }

        return userRoles;
    }

    private int getFirstResult(RequestContext context) {
        String firstResult = getDecodedRequestParameter(context, UM_FIRST_RESULT);

        return (firstResult != null) ? Integer.valueOf(firstResult).intValue() : 0;
    }

    public Event initEvent(RequestContext context) throws Exception {

        JSONObject conf = new JSONObject();
        conf.put(USER_NAME_NOT_SUPPORTED_SYMBOLS, configuration.getUserNameNotSupportedSymbols());
        conf.put(USER_NAME_SEPARATOR, configuration.getUserNameSeparator());
        conf.put(USER_DEFAULT_ROLE, configuration.getDefaultRole());
        conf.put(USER_PASSWORD_MASK, configuration.getPasswordMask());
        conf.put(SUPERUSER_ROLE, ROLE_SUPERUSER);
        conf.put(ADMIN_ROLE, ROLE_ADMINISTRATOR);

        context.getFlowScope().put(UM_CONFIGURATION, conf.toString());
        context.getFlowScope().put(USER_NAME_NOT_SUPPORTED_SYMBOLS, configuration.getUserNameNotSupportedSymbols());

        context.getFlowScope().put(UM_DEFAULT_USER, getUserName(context));
        context.getFlowScope().put(UM_CURRENT_USER, getCurrentUser(context));
        context.getFlowScope().put(UM_CURRENT_USER_ROLES,
                roleHelper.convertRoleListToJson(getCurrentUserRoles(context), null).toString());

        log.info("=== Default user : " + getUserName(context));

        return success();
    }

    public Event loadUsers(RequestContext context) throws Exception {
        final String userName = getUserName(context);
        final String tenantId = getTenantId(context);
        final int firstResult = getFirstResult(context);
        final int maxResults = this.configuration.getUserItemsPerPage();

        String responseModel;

        try {
            final Set tenantIdSet = getTenantsCriteriaSet(tenantId);

            PaginatedOperationResult result = PaginationHelper.paginatedGetOperationResult(firstResult, maxResults,
                    new PaginationHelper.JasperService() {
                            public List getResultList(int firstResult, int maxResults) {
                                return userService.getTenantUsers(null, tenantIdSet, userName, firstResult, maxResults);
                            }

                            public int getResultCount() {
                                return userService.getTenantUsersCount(null, tenantIdSet, userName);
                            }
            });

            List users = result.getResult();
            int usersCount = result.getTotalResults();

            if (users != null && !users.isEmpty()) {

                JSONObject usersJson = userHelper.createUsersResponseJson(users, firstResult, maxResults, usersCount);
                responseModel = userHelper.createDataResponseModel(usersJson);
            } else {

                responseModel = userHelper.createDataResponseModel(userHelper.createEmptyUserListResponseJson());
            }
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event getUserDetails(RequestContext context) throws Exception {
        String userName = getUserName(context);

        String responseModel;

        User user;
        try {

            user = this.userService.getUser(null, userName);

            if (user != null) {
//                user.setPassword(hidePassword(user.getPassword()));

                JSONObject usersJson = userHelper.convertUserToJson(user, null);
                responseModel = userHelper.createDataResponseModel(usersJson);
            } else {

                throw new IllegalArgumentException("Cannot find user with username : " + userName);
            }
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event isUserExist(RequestContext context) throws Exception {
        String userName = getUserName(context);

        String responseModel;

        User user;
        try {

            user = this.userService.getUser(null, userName);

            JSONObject isUserExistJson = userHelper.createIsUserExistJson((user != null));
            responseModel = userHelper.createDataResponseModel(isUserExistJson);
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event deleteUser(RequestContext context) throws Exception {
        String userName = getUserName(context, false);

        String responseModel = userHelper.createSuccessResponseModel();

        try {

            if (userName.length() > 0) {

                this.userService.deleteUser(null, userName);
            } else {

                throw new IllegalArgumentException("Username is empty.");
            }

        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event updateUser(RequestContext context) throws Exception {
        String userName = getUserName(context, false);
        String userJson = getUserJson(context);

        String responseModel = userHelper.createSuccessResponseModel();

        try {

            User user = userHelper.convertJsonToUser(userJson);

            if (user != null && user.getUsername().trim().length() > 0) {

                String newPassword = user.getPassword();
                String oldPassword = getOldUserPassword(userName);
//                if (newPassword == null || newPassword.equals(hidePassword(oldPassword))) {
//
//                    user.setPassword(oldPassword);
//                }

                if (newPassword == null) {

                    user.setPassword(oldPassword);
                } else if (!newPassword.equals(oldPassword)) {

                    user.setPreviousPasswordChangeTime(new Date());
                }


                this.userService.updateUser(null, userName, user);
            } else {

                throw new IllegalArgumentException("Error when updating user details.");
            }

        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    private String getOldUserPassword(String userName) {

        User user = this.userService.getUser(null, userName);

        return user.getPassword();
    }

    public Event createUser(RequestContext context) throws Exception {
        String userJson = getUserJson(context);

        String responseModel = userHelper.createSuccessResponseModel();

        try {

            if (userJson != null && userJson.length() > 0) {

                User user = userHelper.convertJsonToUser(userJson);

                if (user.getTenantId() != null) {

                    Tenant tenant = getTenantService().getTenant(null, user.getTenantId());

                    if (tenant == null) {

                        throw new IllegalArgumentException("Cannot find organization with id : " + user.getTenantId());
                    }
                }

                user.setPreviousPasswordChangeTime(new Date());

                this.userService.putUser(null, user);
            } else {

                throw new IllegalAccessException("Error when creating user");
            }

        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public Event getAvailableRoles(RequestContext context) throws Exception {
        final String userName = getUserName(context);
        final String roleName = getRoleName(context);
        final Set userRoles = getUserRoles(context);
        final int firstResult = getFirstResult(context);
        int maxResults = 10;

        String responseModel/* = createResponseModel("[]", null)*/;

        try {

            PaginatedOperationResult result = PaginationHelper.paginatedGetOperationResult(firstResult, maxResults,
                    new PaginationHelper.JasperService() {
                            public List getResultList(int firstResult, int maxResults) {
                                return userService.getAvailableRoles(null, roleName, userRoles, userName, firstResult, maxResults);
                            }

                            public int getResultCount() {
                                return userService.getAvailableRolesCount(null, roleName, userRoles, userName);
                            }
            });

            List roles = result.getResult();
            int rolesCount = result.getTotalResults();

            if (roles != null && !roles.isEmpty()) {

                JSONObject rolesJson =
                        roleHelper.createRolesResponseJson(roles, result.getFirstResult(), maxResults, rolesCount);
                responseModel = userHelper.createDataResponseModel(rolesJson);
            } else {

                responseModel = userHelper.createDataResponseModel(roleHelper.createEmptyRoleListResponseJson());
            }
        } catch (Exception e) {

            responseModel = createUnexpectedExceptionResponseModel(e.getMessage());
        }

        context.getRequestScope().put(AJAX_RESPONSE_MODEL, responseModel);

        return success();
    }

    public String hidePassword(String password) {

        return (password == null) ? password : password.replaceAll(".", configuration.getPasswordMask());
    }

    private String createUnexpectedExceptionResponseModel(String desc) throws Exception {

        String message = messages.getMessage("jsp.userManager.unexpectedException", new Object[0],
                LocaleContextHolder.getLocale());

        JSONObject exceptionJson = userHelper.createUnexpectedExceptionJson("", message, desc);

        return userHelper.createErrorResponseModel(exceptionJson);
    }

}
