/* 
* Copyright (C) 2005 - 2007 JasperSoft Corporation. All rights reserved.
* http://www.jaspersoft.com.
* Licensed under commercial JasperSoft Subscription License Agreement
*/

package com.jaspersoft.hibernate.dialect;

import java.sql.Types;

import org.hibernate.dialect.PostgreSQLDialect;

import com.jaspersoft.hibernate.ByteWrappingBlobType;

/**
 * This dialect maps blobs to binary streams for PostgreSQL.
 * This is necessary because PostgreSQL requires blob access to happen within a transaction.
 * That means we have to turn defaultAutoCommit off which has some weird side effects.
 * Using the "bytea" type should work better.
 * 
 * There are a few aspects to implementing this:
 * - Generate the "bytea" type for columns with Hibernate type "blob"
 * - Set the "mapBlobsToBinaryType" flag, which will cause ByteWrappingBlobType.set() 
 *   to use ResultSet.setBinaryStream() instead of ResultSet.setBlob()
 * - Use ByteWrappingBlobType (a subclass of BlobType) instead of "blob" in the Hibernate mapping files.
 *   This overrides BlobType.get() to call ResultSet.getObject() instead of getBlob, and then turns
 *   whatever it gets into a Blob if it isn't one already.
 *   
 *   I'm not sure why we have to use blobs, and they're only used in 4 places. However, I do know that
 *   Oracle has a brain-dead binary type (LONG RAW) that we should stay away from, and if we've already
 *   tested with blobs on other databases, then just let it be.
 */
public class PostgresqlNoBlobDialect extends PostgreSQLDialect {

    public PostgresqlNoBlobDialect() {
        super();
        // use "bytea" to map blob types
        registerColumnType( Types.BLOB, "bytea" );
        // turn on blob mapping
        getDefaultProperties().setProperty(ByteWrappingBlobType.MAP_BLOBS_TO_BINARY_TYPE, String.valueOf(true));
    }

}