/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.api.metadata.view.service.impl;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import java.util.Properties;

import javax.naming.NamingException;

import junit.textui.TestRunner;

import org.acegisecurity.GrantedAuthority;
import org.acegisecurity.userdetails.UserDetails;
import org.acegisecurity.userdetails.UserDetailsService;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.test.AbstractDependencyInjectionSpringContextTests;

import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.service.TenantService;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.PasswordCipherer;

/**
 * @author swood
 * @version $Id: UserAuthorityServiceTest.java 15275 2009-03-13 20:11:55Z andy21ca $
 */
public class UserAuthorityServiceTest extends AbstractDependencyInjectionSpringContextTests {

	Properties jdbcProps;
	UserAuthorityService userAuthorityService;
    TenantService tenantService;

    protected void onSetUp() throws Exception {
        super.onSetUp();
        
        tenantService = (TenantService) this.getApplicationContext().getBean("tenantService");
    }

    public UserAuthorityService getUserAuthorityService() {
		return userAuthorityService;
	}

	public void setUserAuthorityService(UserAuthorityService userAuthorityService) {
		this.userAuthorityService = userAuthorityService;
	}

	public UserAuthorityServiceTest(String name) {
		super(name);
		setAutowireMode(AUTOWIRE_BY_NAME);
	}

	public static void main(String[] args) {
		TestRunner.run(UserAuthorityServiceTest.class);
	}

    protected String[] getConfigLocations() {
		try {
			loadJdbcProps();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		// metadata.additionalSettings=hibernateRepositoryAdditionalSettings.xml

		if (jdbcProps.getProperty("metadata.additionalSettings") == null) {
			return
					new String[] {"hibernateConfig.xml", "userAuthorityService.xml", "viewService.xml", "methodAndObjectSecurity.xml"};
		} else {
			return
					new String[] {"hibernateConfig.xml", jdbcProps.getProperty("metadata.additionalSettings"), "userAuthorityService.xml", "viewService.xml", "methodAndObjectSecurity.xml"};
		}
    }

    protected Properties loadJdbcProps() throws IOException, FileNotFoundException, NamingException {
        jdbcProps = new Properties();
        String jdbcPropFile = System.getProperty("test.hibernate.jdbc.properties");
        BufferedInputStream is = new BufferedInputStream(new FileInputStream(jdbcPropFile));
        jdbcProps.load(is);
        is.close();
        return jdbcProps;
    }

	public void onTearDown() {
		System.out.println("Tear down");
		/*
		 *  Leave entries in the database
		 *
		 */

		// get rid of objects we don't want in samples

		userAuthorityService.deleteUser(null, "newUser");

		userAuthorityService.deleteRole(null, "newRole");
		userAuthorityService.deleteRole(null, "anotherRole");

		// temp
		userAuthorityService.deleteUser(null, "TestUser");
		userAuthorityService.deleteRole(null, "TestRole0");
		userAuthorityService.deleteRole(null, "TestRole1");
		userAuthorityService.deleteRole(null, "TestRole2");
		userAuthorityService.deleteRole(null, "TestRole3");
		userAuthorityService.deleteRole(null, "TestRole4");
	}

	public Role addRole(User user, String roleName) {
		System.out.println("addRole");

		Role role = userAuthorityService.getRole(null, roleName);

		if (role == null) {
			role = userAuthorityService.newRole(null);
			//System.out.println("role class: " + role.getClass().getName());
			role.setRoleName(roleName);
			role.setExternallyDefined(false);

			userAuthorityService.putRole(null, role);
		}

		userAuthorityService.addRole(null, user, role);
		return role;
	}

	private User findOrCreateUser(String username, String password, String fullName) {
		User workingUser = userAuthorityService.getUser(null, username);
		if (workingUser == null) {
			workingUser = userAuthorityService.newUser(null);
			workingUser.setUsername(username);
			workingUser.setPassword((password == null ? username : password));
			workingUser.setFullName(fullName);
			workingUser.setEnabled(true);
			workingUser.setExternallyDefined(false);

			userAuthorityService.putUser(null, workingUser);
		}

		return workingUser;
	}

	public void testNewUser() {
		System.out.println("testNewUser");
		User newUser = findOrCreateUser("newUser", "newPassword", "newUser");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "newRole");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

	}

	public void testOrdinaryUser() {
		System.out.println("testOrdinaryUser");
		User newUser = findOrCreateUser("joeuser", "joeuser", "Joe User");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);
	}

	public void testOrdinaryETLUser() {
		System.out.println("testOrdinaryETLUser");
		User newUser = findOrCreateUser("etluser", "etluser", "etluser");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ETL_ADMIN");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);
	}

	public void testAdminETLUser() {
		System.out.println("testAdminETLUser");
		User newUser = findOrCreateUser("etladmin", "etladmin", "etladmin");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ADMINISTRATOR");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);

		addRole(newUser, "ROLE_ETL_ADMIN");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 3);

	}

	public void testAdminUser() {
		System.out.println("testAdminUser");
		User newUser = findOrCreateUser("admin", null, "admin");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ADMINISTRATOR");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);

	}

	public void testJasperadminUser() {
		System.out.println("testJasperadminUser");
		User newUser = findOrCreateUser("jasperadmin", "jasperadmin", "jasperadmin");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_USER");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 1);

		addRole(newUser, "ROLE_ADMINISTRATOR");
		assertTrue(newUser.getRoles() != null || newUser.getRoles().size() == 2);
	}

	public void testAnonymousUser() {
		System.out.println("testAnonymousUser");
		User newUser = findOrCreateUser("anonymousUser", "", "anonymousUser");
		// No roles added yet
		assertTrue(newUser.getRoles() == null || newUser.getRoles().isEmpty());

		addRole(newUser, "ROLE_ANONYMOUS");
		assertTrue("newUser.getRoles() wrong size", newUser.getRoles() != null || newUser.getRoles().size() == 1);
		//assertTrue("newRole.getUsers() wrong size", newRole.getUsers() != null || newRole.getUsers().size() == 1);

		UserDetailsService userDetailsService = (UserDetailsService) userAuthorityService;

		UserDetails details = userDetailsService.loadUserByUsername(newUser.getUsername());

		assertTrue("load by Acegi interface", details != null);

		GrantedAuthority[] authorities = details.getAuthorities();
		assertTrue("authorities wrong size", authorities != null || authorities.length == 1);
		assertTrue("right authority", authorities[0].getAuthority().equalsIgnoreCase("ROLE_ANONYMOUS"));

	}

	public void testUpdateUser() {
		System.out.println("testUpdateUser");
		User newUser = findOrCreateUser("newUser", "", "newUser");
		assertNotNull(newUser);
		assertTrue(newUser.isEnabled());
		userAuthorityService.disableUser(null, newUser.getUsername());
		newUser = findOrCreateUser("newUser", "", "newUser");
		assertTrue("error: user still disabled", !newUser.isEnabled());

		addRole(newUser, "anotherRole");

		Role r = userAuthorityService.getRole(null, "anotherRole");

		assertTrue("error: 'anotherRole' does not exist", r != null);
		assertTrue("error: newUser does not have anotherRole", newUser.getRoles().contains(r));

		userAuthorityService.putUser(null, newUser);

		newUser.removeRole(r);
		assertTrue("error: newUser still contains anotherRole", !newUser.getRoles().contains(r));

		userAuthorityService.putUser(null, newUser);

		newUser = userAuthorityService.getUser(null, "newUser");
		assertTrue("error: after retrieval - newUser REALLY still contains anotherRole", !newUser.getRoles().contains(r));

	}

	public void testGetUsersAndRoles() {
		System.out.println("testGetUsersAndRoles");
		// Depending on the order of test classes that are run, we could have more users and roles
		// than we created here.

		List results = userAuthorityService.getUsers(null, null);
		assertTrue("getUsers right size: expected at least 3, got " + results.size(), results.size() >= 3);
		results = userAuthorityService.getRoles(null, null);
		assertTrue("getRoles right size: expected at least 4, got " + results.size(), results.size() >= 4);
	}

/*	public void testObjectPermissionMaintenance() {

	}
*/
}
