/*
 * Copyright (C) 2005 - 2007 JasperSoft Corporation.  All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased a commercial license agreement from JasperSoft,
 * the following license terms apply:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed WITHOUT ANY WARRANTY; and without the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see http://www.gnu.org/licenses/gpl.txt
 * or write to:
 *
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330,
 * Boston, MA  USA  02111-1307
 */
package com.jaspersoft.jasperserver.war.action.tree;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.MessageSource;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.webflow.action.MultiAction;
import org.springframework.webflow.execution.Event;
import org.springframework.webflow.execution.RequestContext;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.war.common.JasperServerUtil;
import com.jaspersoft.jasperserver.war.model.TreeDataProvider;
import com.jaspersoft.jasperserver.war.model.TreeDataProviderFactory;
import com.jaspersoft.jasperserver.war.model.TreeHelper;
import com.jaspersoft.jasperserver.war.model.TreeNode;

/**
 * Spring Action class for Tree flow.
 * @author asokolnikov
 */
public class TreeAction extends MultiAction {
	
	public static final String AJAX_REPORT_MODEL = "ajaxResponseModel";
	public static final String PROVIDER = "provider";
	public static final String URI = "uri";
	public static final String URIS = "uris";
	public static final String DEPTH = "depth";
    public static final String PREFETCH = "prefetch";
    public static final String MESSAGE_ID = "messageId";
	
	protected final Log log = LogFactory.getLog(this.getClass());

	private TreeDataProviderFactory treeDataProviderFactory;
	private MessageSource messageSource;
	
	// Actions

	/**
	 * Returns internationalized message for tree client
	 */
    public Event getMessage(RequestContext context) {
    	
    	String messageId = (String) context.getRequestParameters().get(MESSAGE_ID);
    	String message = messageSource.getMessage(messageId, null, LocaleContextHolder.getLocale());
    	
    	context.getRequestScope().put(AJAX_REPORT_MODEL, message);

    	return success();
    }
    
	/**
	 * Gets the data, builds tree model and returns serialized tree data 
	 */
    public Event getNode(RequestContext context) {
    	
    	String providerId = (String) context.getRequestParameters().get(PROVIDER);
    	String uri = (String) context.getRequestParameters().get(URI);
    	String depth = (String) context.getRequestParameters().get(DEPTH);
        String prefetchNodesList = context.getRequestParameters().get(PREFETCH);
    	
    	int d = 0;
    	if (depth != null && depth.length() > 0) {
	    	try {
	    		d = Integer.parseInt(depth);
	    		if (d < 0) {
	    			d = 0;
	    		}
	    	} catch (Exception e) {
	            log.error("Invalid parameter : depth : " + depth, e);
	        }
    	}
        
        List prefetchList = null;
        if (prefetchNodesList != null && prefetchNodesList.length() > 0) {
            String[] ss = prefetchNodesList.split(",");
            prefetchList = new ArrayList();
            for (int i = 0; i < ss.length; i++) {
                prefetchList.add(ss[i]);
            }
        }
    	
    	TreeDataProvider treeDataProvider = findProvider(context, providerId);
    	
        TreeNode treeNode;
        if (prefetchList == null) {
            treeNode = treeDataProvider.getNode(exContext(context), uri, d);
        } else {
            treeNode = TreeHelper.getSubtree(exContext(context), treeDataProvider, uri, prefetchList, d);
        }
    	
    	String model = "";
    	if (treeNode != null) {
    	    StringBuffer sb = new StringBuffer();
    	    sb.append("<div id='treeNodeText'>");
    	    sb.append(treeNode.toJSONString());
    	    sb.append("</div>");
    	    model = sb.toString();
    	}
    	
    	context.getRequestScope().put(AJAX_REPORT_MODEL, model);
    	
    	return success();
    }
    
    /**
     * Gets childern for specified tree node
     * @param context
     * @return
     */
    public Event getChildren(RequestContext context) {
    	
    	String providerId = (String) context.getRequestParameters().get(PROVIDER);
    	String uri = uriDecode((String) context.getRequestParameters().get(URI));
    	
    	TreeDataProvider treeDataProvider = findProvider(context, providerId);
    	
    	TreeNode treeNode = treeDataProvider.getNode(exContext(context), uri, 1);
    	
    	String model = "";
    	if (treeNode != null) {
    	    StringBuffer sb = new StringBuffer();
    	    sb.append("<div id='treeNodeText'>");
    	    //sb.append(treeNode.toJSONString());
    	    sb.append('[');
    	    for (Iterator i = treeNode.getChildren().iterator(); i.hasNext(); ) {
    	        TreeNode n = (TreeNode) i.next();
    	        sb.append(n.toJSONString());
    	        if (i.hasNext()) {
    	            sb.append(',');
    	        }
    	    }
    	    sb.append(']');
    	    sb.append("</div>");
    	    model = sb.toString();
    	}
    	
    	context.getRequestScope().put(AJAX_REPORT_MODEL, model);
    	
    	return success();
    }
    
    /**
     * Gets childern for specified tree node
     * @param context
     * @return
     */
    public Event getMultipleChildren(RequestContext context) {
    	
    	String providerId = (String) context.getRequestParameters().get(PROVIDER);
    	String uriParam = uriDecode((String) context.getRequestParameters().get(URIS));
    	String[] uris = uriParam.split(",");
    	
    	TreeDataProvider treeDataProvider = findProvider(context, providerId);
    	
    	String model = "";
	    StringBuffer sb = new StringBuffer();
	    sb.append("<div id='treeNodeText'>");
	    sb.append('[');

	    boolean empty = true;
    	for (int k = 0; k < uris.length; k++) {
    	
	    	TreeNode treeNode = treeDataProvider.getNode(exContext(context), uris[k], 1);
	    	
	    	if (treeNode != null) {
	    		
	    		if (empty) {
	    			empty = false;
	    		} else {
	    			sb.append(',');
	    		}
	    		
	    		sb.append("{\"parentUri\":\"").append(treeNode.getUriString()).append("\",\"children\":[");
	    	    for (Iterator i = treeNode.getChildren().iterator(); i.hasNext(); ) {
	    	        TreeNode n = (TreeNode) i.next();
	    	        sb.append(n.toJSONString());
	    	        if (i.hasNext()) {
	    	            sb.append(',');
	    	        }
	    	    }
	    	    sb.append("]}");
	    	}
    	
    	}
	    sb.append(']');
	    sb.append("</div>");
	    model = sb.toString();
    	
    	context.getRequestScope().put(AJAX_REPORT_MODEL, model);
    	
    	return success();
    }
    
    private TreeDataProvider findProvider(RequestContext context, String providerId) {
    	TreeDataProvider treeDataProvider = null;
    	// First, try to find data provider in session scope
    	treeDataProvider = (TreeDataProvider) context.getExternalContext().getSessionMap().get(providerId);
    	// Then, try to find it in the factory
		if (treeDataProvider == null) {
    		treeDataProvider =  treeDataProviderFactory.getDataProvider(providerId);
		}
		// Fail if not found
		if (treeDataProvider == null) {
			log.error("Cannot find tree data provider with id : " + providerId);
			throw new IllegalArgumentException("Cannot find tree data provider with id : " + providerId);
		}
		
    	return treeDataProvider;
    }
    
	// Getters and Setters
	
    public TreeDataProviderFactory getTreeDataProviderFactory() {
        return treeDataProviderFactory;
    }
    
    public void setTreeDataProviderFactory(
            TreeDataProviderFactory treeDataProviderFactory) {
        this.treeDataProviderFactory = treeDataProviderFactory;
    }

	private ExecutionContext exContext(RequestContext rContext) {
		return JasperServerUtil.getExecutionContext(rContext);
	}

	public MessageSource getMessageSource() {
		return messageSource;
	}

	public void setMessageSource(MessageSource messageSource) {
		this.messageSource = messageSource;
	}

    /**
     * Do URI decoding of a single argument.
     * The AJAX stuff is double-encoded because Tomcat (and possibly other web servers)
     * will assume ISO-8859-1 encoding of anything in %xx format, but we are UTF-8 encoding
     * in order to support unicode.
     *
     * @param s
     *      String decoded string
     * @return String
     */
    public static String uriDecode(String s) {
        if (s == null) {
            return null;
        }
        StringBuffer sbuf = new StringBuffer();
        int l = s.length();
        int ch = -1;
        int b, sumb = 0;
        for (int i = 0, more = -1; i < l; i++) {
            /* Get next byte b from URL segment s */
            switch (ch = s.charAt(i)) {
            case '%':
                ch = s.charAt(++i);
                int hb = (Character.isDigit((char) ch) ? ch - '0'
                        : 10 + Character.toLowerCase((char) ch) - 'a') & 0xF;
                ch = s.charAt(++i);
                int lb = (Character.isDigit((char) ch) ? ch - '0'
                        : 10 + Character.toLowerCase((char) ch) - 'a') & 0xF;
                b = (hb << 4) | lb;
                break;
            case '+':
                b = ' ';
                break;
            default:
                b = ch;
            }
            /* Decode byte b as UTF-8, sumb collects incomplete chars */
            if ((b & 0xc0) == 0x80) { // 10xxxxxx (continuation byte)
                sumb = (sumb << 6) | (b & 0x3f); // Add 6 bits to sumb
                if (--more == 0)
                    sbuf.append((char) sumb); // Add char to sbuf
            } else if ((b & 0x80) == 0x00) { // 0xxxxxxx (yields 7 bits)
                sbuf.append((char) b); // Store in sbuf
            } else if ((b & 0xe0) == 0xc0) { // 110xxxxx (yields 5 bits)
                sumb = b & 0x1f;
                more = 1; // Expect 1 more byte
            } else if ((b & 0xf0) == 0xe0) { // 1110xxxx (yields 4 bits)
                sumb = b & 0x0f;
                more = 2; // Expect 2 more bytes
            } else if ((b & 0xf8) == 0xf0) { // 11110xxx (yields 3 bits)
                sumb = b & 0x07;
                more = 3; // Expect 3 more bytes
            } else if ((b & 0xfc) == 0xf8) { // 111110xx (yields 2 bits)
                sumb = b & 0x03;
                more = 4; // Expect 4 more bytes
            } else /* if ((b & 0xfe) == 0xfc) */{ // 1111110x (yields 1 bit)
                sumb = b & 0x01;
                more = 5; // Expect 5 more bytes
            }
            /* We don't test if the UTF-8 encoding is well-formed */
        }
        return sbuf.toString();
    }

}

