/*
 * Copyright (C) 2005 - 2012 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public  License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.api.security.externalAuth.processors;

import com.jaspersoft.jasperserver.api.JSException;
import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.impl.hibernate.PersistentObjectResolver;
import com.jaspersoft.jasperserver.api.metadata.user.domain.Role;
import com.jaspersoft.jasperserver.api.metadata.user.domain.User;
import com.jaspersoft.jasperserver.api.metadata.user.service.UserAuthorityService;
import com.jaspersoft.jasperserver.api.metadata.user.service.impl.ExternalUserService;
import com.jaspersoft.jasperserver.api.security.externalAuth.ExternalAuthProperties;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.Predicate;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;
import org.springframework.security.GrantedAuthority;
import org.springframework.security.userdetails.UserDetails;

import java.util.*;

import static com.jaspersoft.jasperserver.api.security.externalAuth.processors.ProcessorData.Key.*;


/**
 * User: dlitvak
 * Date: 8/22/12
 */
public class ExternalUserSetupProcessor extends AbstractExternalUserProcessor {
    private static final Logger logger = LogManager.getLogger(ExternalUserSetupProcessor.class);
	private static final String ROLE_SUFFIX = "|*";

    // roles that will be created automatically for each user once he is authenticated.
    private List defaultInternalRoles;
	private ExternalAuthProperties externalAuthProperties = new ExternalAuthProperties();
	private Map<String, String> organizationRoleMap = Collections.emptyMap();
	private String permittedExternalRoleNameRegex = "[A-Za-z0-9_]+";
	private List<String> adminUsernames;
	private List<String> defaultAdminRoles;
	private String conflictingExternalInternalRoleNameSuffix = "EXT";

	@Override
    public void afterPropertiesSet() throws Exception {
//        Assert.notNull(this.defaultInternalRoles, "Please specify non-null internal default role");
//        Assert.notEmpty(this.defaultInternalRoles, "Please specify at least one internal default role");
        super.afterPropertiesSet();
    }

    protected User getUser(){
        ProcessorData processorData = ProcessorData.getInstance();
        UserDetails userDetails = (UserDetails) processorData.getData(EXTERNAL_AUTH_DETAILS);
        return getUserAuthorityService().getUser(new ExecutionContextImpl(), userDetails.getUsername());
    }

    @Override
	public void process() {
		ProcessorData processorData = ProcessorData.getInstance();
		UserDetails userDetails = (UserDetails) processorData.getData(EXTERNAL_AUTH_DETAILS);

		try {
			String userName = userDetails.getUsername();

			logger.debug("Setting up external user: " + userName);

			User user = getUser();
			String logoutUrl = externalAuthProperties != null ? externalAuthProperties.getLogoutUrl() : null;
			if ( user==null ) {
				user = createNewExternalUser(userName);
			}
			else if (!user.isEnabled()) {
				throw new JSException("External user " + user.getUsername() + " was disabled on jasperserver. Please contact an admin user to re-enable. " +
						(logoutUrl != null && logoutUrl.length() > 0 ?  "Click <a href=\"" + logoutUrl + "\">logout</a> to exit from external system." : ""));
			}
			else if (!user.isExternallyDefined()) {
				throw new JSException("Internally defined user " + user.getUsername() + " already exists. Please contact an admin user to resolve the issue. " +
						(logoutUrl != null && logoutUrl.length() > 0 ?  "Click <a href=\"" + logoutUrl + "\">logout</a> to exit from external system." : ""));
			}

			GrantedAuthority[] grantedAuthorities = (GrantedAuthority[]) processorData.getData(EXTERNAL_AUTHORITIES);
			final String tenantId = (String) processorData.getData(EXTERNAL_JRS_USER_TENANT_ID);

			Set<Role> externalRoles = convertGrantedAuthoritiesToRoles(grantedAuthorities, tenantId);
			user.setTenantId(tenantId);

			Set roles = persistRoles(externalRoles);
			alignInternalAndExternalUser(roles, user);

            ((ExternalUserService)getUserAuthorityService()).makeUserLoggedIn(user);
		}
		catch (RuntimeException e) {
			String userName = (userDetails != null ? userDetails.getUsername() : "");
			logger.error("Error processing external user " + userName + ": " + e.getMessage());
			throw e;
		}
	}

    /**
     * New user created from given authentication details. No password is set or needed.
     * Roles are set elsewhere.
     *
     * @param userName
     * @return created User
     */
    protected User createNewExternalUser(String userName) {
        User user = getUserAuthorityService().newUser(new ExecutionContextImpl());
        user.setUsername(userName);
        // If it is externally authenticated, no save of password
        //user.setPassword(userDetails.getPassword());
        user.setFullName(userName); // We don't know the real name
        user.setExternallyDefined(true);
        user.setEnabled(true);
		logger.warn("Created new external user: " + user.getUsername());
        return user;
    }

    protected Set persistRoles(Set<Role> roles) {
        Set<Role> persistedRoles = new HashSet<Role>();
        for (Iterator<Role> iter = roles.iterator(); iter.hasNext(); ) {
            Role r = iter.next();
            persistedRoles.add(getOrCreateRole(r));
        }
        return persistedRoles;
    }

    /**
	 * TODO refactor this old code, clean up
	 *
     * Ensure the external user has the right roles. Roles attached to the userDetails are the definitive list
     * of externally defined roles.
     *
     * @param externalRoles
     * @param user
     */
    protected void alignInternalAndExternalUser(Set externalRoles, User user) {

        final Predicate externallyDefinedRoles = new Predicate() {
            public boolean evaluate(Object input) {
                if (!(input instanceof Role)) {
                    return false;
                }
                return ((Role) input).isExternallyDefined();
            }
        };

        Set currentRoles = user.getRoles();

        // we may have a new user, so always persist them
        boolean persistUserNeeded = (currentRoles.size() == 0);

		Collection currentExternalRoles = CollectionUtils.select(user.getRoles(), externallyDefinedRoles);
        if (logger.isDebugEnabled()) {
            logger.debug("Login of external User: " + user.getUsername() );
            logger.debug("Roles from authentication:\n" + roleCollectionToString(externalRoles));
            logger.debug("Current roles from metadata:\n" + roleCollectionToString(user.getRoles()));
            logger.debug("Current external roles for user from metadata: " + user.getUsername() + "\n" + roleCollectionToString(currentExternalRoles));
        }

        /*
           * If we have new external roles, we want to add them
           */
        Collection newExternalRoles = CollectionUtils.subtract(externalRoles, currentExternalRoles);

        if (newExternalRoles.size() > 0) {
            currentRoles.addAll(newExternalRoles);
            if (logger.isDebugEnabled()) {
                logger.warn("Added following external roles to: " + user.getUsername() + "\n" + roleCollectionToString(newExternalRoles));
            }
            persistUserNeeded = true;
        }

        /*
           * If external roles have been removed, we need to remove them
           */
        Collection rolesNeedingRemoval = CollectionUtils.subtract(currentExternalRoles, externalRoles);

        if (rolesNeedingRemoval.size() > 0) {
            currentRoles.removeAll(rolesNeedingRemoval);
            if (logger.isDebugEnabled()) {
                logger.warn("Removed following external roles from: " + user.getUsername() + "\n" + roleCollectionToString(rolesNeedingRemoval));
            }
            persistUserNeeded = true;
        }

        /*
           * If we have new default internal roles, we want to add them
           */
        Collection defaultInternalRolesToAdd = CollectionUtils.subtract(getNewDefaultInternalRoles(user.getUsername()), currentRoles);

        if (defaultInternalRolesToAdd.size() > 0) {
            if (logger.isDebugEnabled()) {
                logger.debug("Default internal roles: " + roleCollectionToString(getNewDefaultInternalRoles(user.getUsername())));
            }
            currentRoles.addAll(defaultInternalRolesToAdd);
            if (logger.isDebugEnabled()) {
                logger.warn("Added following new default internal roles to: " + user.getUsername() + "\n" + roleCollectionToString(defaultInternalRolesToAdd));
            }
            persistUserNeeded = true;
        }

        if (persistUserNeeded) {
            if (logger.isDebugEnabled()) {
                logger.warn("Updated user: " + user.getUsername() + ". Roles are now:\n" + roleCollectionToString(currentRoles));
            }
            user.setRoles(currentRoles);
            // persist user and roles
            getUserAuthorityService().putUser(new ExecutionContextImpl(), user);
            if (logger.isDebugEnabled()) {
                logger.warn("Updated user: " + user.getUsername() + ". Roles are now:\n" + roleCollectionToString(currentRoles));
            }
        }

    }

    /**
     * Get a set of roles based on the given GrantedAuthority[]. Roles are created
     * in the metadata if they do not exist.
     *
     *
	 * @param authorities from authenticated user
	 * @param tenantId
	 * @return Set of externally defined Roles
	 *
	 * protected scope for unit testing
     */
    protected Set<Role> convertGrantedAuthoritiesToRoles(GrantedAuthority[] authorities, String tenantId) {
        Set<Role> set = new HashSet<Role>();

        if (authorities == null || authorities.length == 0)
            return set;

		final UserAuthorityService userAuthorityService = getUserAuthorityService();
        for (GrantedAuthority auth : authorities) {
           	String authorityName = auth.getAuthority();
			if (authorityName == null)
				continue;
			if (!authorityName.matches(permittedExternalRoleNameRegex)) {
				ProcessorData processorData = ProcessorData.getInstance();
				UserDetails userDetails = (UserDetails) processorData.getData(EXTERNAL_AUTH_DETAILS);
				logger.warn("External role " + authorityName + " has forbidden characters " +
						"according to permittedExternalRoleNameRegex: " + permittedExternalRoleNameRegex +
						".\nSkipping this role for the user " +
						(userDetails != null && userDetails.getUsername() != null ? userDetails.getUsername() : ""));
				continue;
			}
            authorityName = authorityName.toUpperCase().trim();

			Role role = userAuthorityService.newRole(new ExecutionContextImpl());
			String internalRoleName = organizationRoleMap.get(authorityName);
			if (internalRoleName != null) {
				if (internalRoleName.endsWith(ROLE_SUFFIX)) {
					internalRoleName = internalRoleName.substring(0, internalRoleName.length() - ROLE_SUFFIX.length());
					role.setTenantId(tenantId);
				}
				role.setRoleName(internalRoleName);
				role.setExternallyDefined(false);  //role is mapped to internal
			}
			else {
				role.setRoleName(authorityName);
				role.setExternallyDefined(true);
				role.setTenantId(tenantId);
			}

            set.add(role);
        }
        return set;
    }

    private String roleCollectionToString(Collection coll) {
        Iterator it = coll.iterator();
        StringBuffer rolesPrint = new StringBuffer();
        while (it.hasNext()) {
            String s = ((Role) it.next()).getRoleName();
            rolesPrint.append(s).append("\n");
        }
        return rolesPrint.toString();
    }

    /**
     * Get a set of roles that are the defaults for a new external user. Roles are created
     * in the metadata if they do not exist.
     *
     * @return Set of internally defined Roles
     */
    private Set getNewDefaultInternalRoles(String username) {
        ExecutionContext executionContext = new ExecutionContextImpl();
        Set<Role> set = new HashSet<Role>();

		List<String> internalRoles = Collections.<String>emptyList();
        if (this.adminUsernames != null && this.adminUsernames.contains(username)
				&& this.defaultAdminRoles != null && this.defaultAdminRoles.size() > 0)
			internalRoles = this.defaultAdminRoles;
        else if (this.defaultInternalRoles != null && this.defaultInternalRoles.size() > 0)
			internalRoles = this.defaultInternalRoles;
		else
            return set;

        for (String roleName : internalRoles) {
            Role role = getUserAuthorityService().getRole(executionContext, roleName);
            if (role == null) {
                role = getUserAuthorityService().newRole(executionContext);
                role.setRoleName(roleName);
                role.setExternallyDefined(false);
                getUserAuthorityService().putRole(executionContext, role);
            }

            set.add(role);
        }
        return set;
    }

	private Role getOrCreateRole(Role role) {
		Role existingRole = null;
		UserAuthorityService userAuthorityService = getUserAuthorityService();

		if (userAuthorityService instanceof PersistentObjectResolver)
			existingRole = (Role) ((PersistentObjectResolver) userAuthorityService).getPersistentObject(role);

		//when internal role name&tenantId coincide with the external role, modify the external
		// role name in order to avoid overwriting the roles (bug 31324).
		if (existingRole != null && role.isExternallyDefined() && !existingRole.isExternallyDefined() ) {
			role.setRoleName(role.getRoleName() + "_" + this.conflictingExternalInternalRoleNameSuffix);
			existingRole = null;
		}

		//role does not exist.  Need to create it.
		if (existingRole == null)
			userAuthorityService.putRole(new ExecutionContextImpl(), role);
		return role;
    }

    public List getDefaultInternalRoles() {
        return defaultInternalRoles;
    }

    public void setDefaultInternalRoles(List defaultInternalRoles) {
        this.defaultInternalRoles = defaultInternalRoles;
    }

	public void setExternalAuthProperties(ExternalAuthProperties externalAuthProperties) {
		this.externalAuthProperties = externalAuthProperties;
	}

	public ExternalAuthProperties getExternalAuthProperties() {
		return externalAuthProperties;
	}

	public String getPermittedExternalRoleNameRegex() {
		return permittedExternalRoleNameRegex;
	}

	public void setPermittedExternalRoleNameRegex(String permittedExternalRoleNameRegex) {
		this.permittedExternalRoleNameRegex = permittedExternalRoleNameRegex;
	}

	public void setOrganizationRoleMap(Map<String, String> organizationRoleMapParam) {
		if (!this.organizationRoleMap.isEmpty()) {
			logger.warn("Duplicate organizationRoleMap initialization.");
			return;
		}

		this.organizationRoleMap = new HashMap<String, String>();
		for (Map.Entry<String,String> rolePair : organizationRoleMapParam.entrySet()) {
			final String rolePairKey = rolePair.getKey();
			final String rolePairValue = rolePair.getValue();
			if (rolePairValue != null) {
				String roleNameToValidate = rolePairValue.trim();
				roleNameToValidate = roleNameToValidate.endsWith(ROLE_SUFFIX) ?
						roleNameToValidate.substring(0, roleNameToValidate.length() - ROLE_SUFFIX.length()) : roleNameToValidate;
				if (!roleNameToValidate.matches(permittedExternalRoleNameRegex)) {
					logger.warn("organizationRoleMap's internal role " + rolePairValue + " has forbidden characters " +
							"according to permittedExternalRoleNameRegex: " + permittedExternalRoleNameRegex +
							".\nSkipping this role in organizationRoleMap.");
					continue;
				}

				this.organizationRoleMap.put(rolePairKey.toUpperCase().trim(),
					rolePairValue.toUpperCase().trim());
			}
		}
	}

	/**
	 * Names of external users that are converted into admins
	 * @param adminUsernames
	 */
	public void setAdminUsernames(List<String> adminUsernames) {
		this.adminUsernames = adminUsernames;
	}

	public List<String> getAdminUsernames() {
		return adminUsernames;
	}

	/**
	 * Default admin roles that are assigned to the users in {@link #adminUsernames}
	 * @param defaultAdminRoles
	 */
	public void setDefaultAdminRoles(List<String> defaultAdminRoles) {
		this.defaultAdminRoles = defaultAdminRoles;
	}

	public List<String> getDefaultAdminRoles() {
		return defaultAdminRoles;
	}

	public void setConflictingExternalInternalRoleNameSuffix(String conflictingExternalInternalRoleNameSuffix) {
		this.conflictingExternalInternalRoleNameSuffix = conflictingExternalInternalRoleNameSuffix;
	}

	public String getConflictingExternalInternalRoleNameSuffix() {
		return conflictingExternalInternalRoleNameSuffix;
	}
}
