/*
 * Copyright (C) 2005 - 2012 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.remote.exception.AccessDeniedException;
import com.jaspersoft.jasperserver.remote.exception.FolderNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceAlreadyExistsException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.services.SingleRepositoryService;
import org.springframework.security.SpringSecurityException;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.regex.Pattern;

@Component("singleRepositoryService")
@Transactional(rollbackFor = Exception.class)
public class SingleRepositoryServiceImpl implements SingleRepositoryService {
    private Pattern nameWithNumber = Pattern.compile("^.*_\\d+$",Pattern.CASE_INSENSITIVE);

    @javax.annotation.Resource(name = "concreteRepository")
    protected RepositoryService repositoryService;

    @Override
    public Resource getResource(String uri) throws ResourceNotFoundException {
        return getResourceInternal(uri, false);
    }

    @Override
    public void deleteResource(String uri) throws IllegalParameterValueException, AccessDeniedException  {
        if (uri == null || "".equals(uri) || Folder.SEPARATOR.equals(uri)) {
            throw new IllegalParameterValueException("uri", uri);
        }

        try {
            Resource resource = repositoryService.getResource(null, uri);
            if (resource != null) {
                repositoryService.deleteResource(null, uri);
            } else {
                resource = repositoryService.getFolder(null, uri);
                if (resource != null) {
                    repositoryService.deleteFolder(null, uri);
                }
            }
        } catch (SpringSecurityException sse) {
            throw new AccessDeniedException(uri);
        }
    }

    @Override
    public Resource createResource(Resource serverResource, String parentUri, String accept, boolean createFolders) throws RemoteException {
        if (createFolders) {
            ensureFolderUri(parentUri);
        } else if (!repositoryService.folderExists(null, parentUri)) {
            throw new FolderNotFoundException(parentUri);
        }

        serverResource.setParentFolder(parentUri);
        serverResource.setName(generateName(parentUri, serverResource.getLabel()));

        if (serverResource instanceof Folder) {
            repositoryService.saveFolder(null, (Folder) serverResource);
        } else {
            repositoryService.saveResource(null, serverResource);
        }

        return getResource(serverResource.getURIString());
    }

    @Override
    public void copyResource(String sourceUri, String destinationUri, boolean createFolders, boolean overwrite) throws ResourceNotFoundException, AccessDeniedException, ResourceAlreadyExistsException, IllegalParameterValueException {
        Resource resource = prepareToOperation(sourceUri, destinationUri, createFolders, overwrite);
        // underlying service requires full uri, not just parent uri
        // after setting new parent uri we obtain the new uri of resource
        resource.setParentFolder(destinationUri);
        if (resource instanceof Folder) {
            repositoryService.copyFolder(null, sourceUri, resource.getURIString());
        } else {
            repositoryService.copyResource(null, sourceUri, resource.getURIString());
        }
    }

    @Override
    public void moveResource(String sourceUri, String destinationUri, boolean createFolders, boolean overwrite) throws ResourceNotFoundException, AccessDeniedException, ResourceAlreadyExistsException, IllegalParameterValueException {
        Resource resource = prepareToOperation(sourceUri, destinationUri, createFolders, overwrite);
        // underlying service requires full uri, not just parent uri
        // after setting new parent uri we obtain the new uri of resource
        resource.setParentFolder(destinationUri);
        if (resource instanceof Folder) {
            repositoryService.moveFolder(null, sourceUri, resource.getURIString());
        } else {
            repositoryService.moveResource(null, sourceUri, resource.getURIString());
        }
    }

    protected Resource getResourceInternal(String uri, boolean tolerateAbsence) throws ResourceNotFoundException {
        Resource resource = repositoryService.getResource(null, uri);
        if (resource == null) {
            resource = repositoryService.getFolder(null, uri);
        }
        if (resource == null && !tolerateAbsence){
            throw new ResourceNotFoundException(uri);
        }
        return resource;
    }

    private Resource prepareToOperation(String sourceUri, String destinationUri, boolean createFolders, boolean overwrite) throws ResourceNotFoundException, AccessDeniedException, ResourceAlreadyExistsException, IllegalParameterValueException {
        if (sourceUri == null || "".equals(sourceUri)) {
            throw new IllegalParameterValueException("sourceUri", sourceUri);
        }

        Resource resource = getResourceInternal(sourceUri, false), existing = null;
        String newUri = (destinationUri.endsWith(Folder.SEPARATOR)?  destinationUri : destinationUri + Folder.SEPARATOR ) + resource.getName();
        existing = getResourceInternal(newUri, true);

        if (existing != null) {
            if (overwrite) {
                deleteResource(newUri);
            } else {
                throw new ResourceAlreadyExistsException(newUri);
            }
        }

        if (createFolders) {
            ensureFolderUri(destinationUri);
        } else if (!repositoryService.folderExists(null, destinationUri)) {
            throw new FolderNotFoundException(destinationUri);
        }

        return resource;
    }

    private Folder ensureFolderUri(String uri) throws AccessDeniedException, ResourceAlreadyExistsException {
        try {
            Folder folder = repositoryService.getFolder(null, uri);
            if (folder == null) {
                if (repositoryService.getResource(null, uri) != null) {
                    throw new ResourceAlreadyExistsException(uri);
                }

                int lastSeparator = uri.lastIndexOf(Folder.SEPARATOR);
                String label = uri.substring(lastSeparator + 1, uri.length());

                folder = new FolderImpl();
                folder.setParentFolder(ensureFolderUri(uri.substring(0, lastSeparator)));
                folder.setName(generateName(uri.substring(0, lastSeparator), label.toLowerCase().replace("\\w", "_")));
                folder.setLabel(label);
                folder.setCreationDate(new Date());

                repositoryService.saveFolder(null, folder);
            }
            return folder;
        } catch (SpringSecurityException spe) {
            throw new AccessDeniedException("Access denied", uri);
        }
    }

    private String generateName(String parentUri, String label){
        String uri = parentUri + Folder.SEPARATOR + label;
        Resource resource = repositoryService.getResource(null, uri);
        if (resource == null) {
            resource = repositoryService.getFolder(null, uri);
        }
        if (resource != null){
            if (nameWithNumber.matcher(label).matches()){
                int divider = label.lastIndexOf("_");
                Integer number = Integer.parseInt(label.substring(divider + 1)) + 1;
                label = label.substring(0, divider + 1) + number.toString();
            }
            else {
                label = label.concat("_1");
            }
            label = generateName(parentUri, label);
        }
        return label;
    }
}