/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.jaxrs.resources;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.client.FolderImpl;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceListWrapper;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceLookup;
import com.jaspersoft.jasperserver.jaxrs.common.RestConstants;
import com.jaspersoft.jasperserver.remote.exception.AccessDeniedException;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.resources.GenericParametersReflectionHelper;
import com.jaspersoft.jasperserver.remote.resources.converters.LookupResourceConverter;
import com.jaspersoft.jasperserver.remote.resources.converters.ResourceConverterProviderImpl;
import com.jaspersoft.jasperserver.remote.resources.converters.ToServerConverter;
import com.jaspersoft.jasperserver.remote.services.BatchRepositoryService;
import com.jaspersoft.jasperserver.remote.services.SingleRepositoryService;
import com.jaspersoft.jasperserver.search.common.ResourceDetails;
import com.jaspersoft.jasperserver.search.service.RepositorySearchCriteria;
import com.jaspersoft.jasperserver.search.service.RepositorySearchService;
import com.jaspersoft.jasperserver.war.common.JasperServerHttpConstants;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.core.Response;
import java.util.ArrayList;
import java.util.List;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyListOf;
import static org.mockito.Matchers.isNull;
import static org.mockito.Mockito.*;
import static org.testng.Assert.*;

/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: SearchResourcesJaxrsServiceTest.java 29699 2013-03-14 14:23:46Z ztomchenco $
 */
public class SearchResourcesJaxrsServiceTest {

    @InjectMocks
    private SearchResourcesJaxrsService searchService = new SearchResourcesJaxrsService();
    @Mock
    private RepositorySearchService repositorySearchService;
    @Mock
    private LookupResourceConverter lookupResourceConverter;
    @Mock
    private ResourceConverterProviderImpl resourceConverterProvider;
    @Mock
    private BatchRepositoryService batchRepositoryService;
    @Mock
    private SingleRepositoryService singleRepositoryService;
    @Mock
    private RepositoryService repositoryService;

    private ArgumentCaptor<RepositorySearchCriteria> repositorySearchCriteriaArgumentCaptor;
    private ArgumentCaptor<RepositorySearchCriteria> repositorySearchCriteriaArgumentCaptorForCount;
    private List<ResourceDetails> expectedResourceDetailsList = new ArrayList<ResourceDetails>();
    private final Integer expectedResourceDetailsListSize = 10;
    private final String uri = "/test";

    @BeforeClass
    public void init(){
        MockitoAnnotations.initMocks(this);
    }

    @BeforeMethod
    public void prepareSearchService(){
        expectedResourceDetailsList.clear();
        repositorySearchCriteriaArgumentCaptor = ArgumentCaptor.forClass(RepositorySearchCriteria.class);
        when(repositorySearchService.getResults(isNull(ExecutionContext.class), repositorySearchCriteriaArgumentCaptor.capture())).thenReturn(expectedResourceDetailsList);
        when(repositorySearchService.getResultsCount(isNull(ExecutionContext.class), any(RepositorySearchCriteria.class))).thenReturn(expectedResourceDetailsListSize);
        reset(batchRepositoryService, singleRepositoryService, repositoryService);
        when(repositoryService.getFolder(isNull(ExecutionContext.class), any(String.class))).thenReturn(new FolderImpl());
    }

    @Test(expectedExceptions = IllegalParameterValueException.class)
    public void getResources_validateFolderUri_invalidFormat() throws ResourceNotFoundException, IllegalParameterValueException {
        searchService.getResources(null, "not.valid.uri", null, null, null, null, null, null, null);
    }

    @Test(expectedExceptions = ResourceNotFoundException.class)
    public void getResources_validateFolderUri_folderDoesntExist() throws ResourceNotFoundException, IllegalParameterValueException {
        reset(repositoryService);
        searchService.getResources(null, "/doesnt/exist", null, null, null, null, null, null, null);
    }

    @Test(groups = {"GET"})
    public void getResources_allParameterAreForwardedCorrectly() throws IllegalParameterValueException, ResourceNotFoundException {
        final String q = "testQuery";
        final String folderUri = "/test/folder/uri";
        final String type = "clientType";
        final String serverType = "serverType";
        final Integer start = 10;
        final Integer limit = 20;
        final Boolean recursive = false;
        final Boolean showHidden = false;
        final Boolean forceTotalCount = false;
        final String sortBy = "testSortBy";
        ToServerConverter converterMock = mock(ToServerConverter.class);
        when(converterMock.getServerResourceType()).thenReturn(serverType);
        when(resourceConverterProvider.getToServerConverter(type)).thenReturn(converterMock);
        searchService.getResources(q, folderUri, type, start, limit, recursive, showHidden, forceTotalCount, sortBy);
        final RepositorySearchCriteria value = repositorySearchCriteriaArgumentCaptor.getValue();
        assertEquals(value.getSearchText(), q);
        assertEquals(value.getFolderUri(), folderUri);
        assertEquals(value.getResourceTypes().get(0), serverType);
        assertEquals(value.getStartIndex(), start.intValue());
        assertEquals(value.getMaxCount(), limit.intValue());
        assertEquals(value.getSortBy(), sortBy);
    }

    @Test(groups = {"GET"})
    public void getResources_resourceLookup_204() throws IllegalParameterValueException, ResourceNotFoundException {
        IllegalParameterValueException exception = null;
        final String resourceLookupType = GenericParametersReflectionHelper.extractClientType(ClientResourceLookup.class);
        when(resourceConverterProvider.getToServerConverter(resourceLookupType)).thenReturn((ToServerConverter)new LookupResourceConverter());
        try {
            final Response resources = searchService.getResources(null, null, resourceLookupType, null, null, null, null, null, null);
            assertEquals(resources.getStatus(), HttpServletResponse.SC_NO_CONTENT);
            final ClientResourceListWrapper entity = (ClientResourceListWrapper) resources.getEntity();
            assertNull(entity);
        } catch (IllegalParameterValueException e) {
            exception = e;
        }
        assertNull(exception);
    }

    @Test(groups = {"GET"})
    public void getResources_defaultValues() throws IllegalParameterValueException, ResourceNotFoundException {
        searchService.getResources(null, null, null, null, null, null, null, null, null);
        final RepositorySearchCriteria value = repositorySearchCriteriaArgumentCaptor.getValue();
        assertEquals(value.getStartIndex(), 0);
        assertEquals(value.getMaxCount(), 100);
        assertEquals(value.getFolderUri(), Folder.SEPARATOR);
        assertFalse(value.isShowHidden());
    }

    @Test(groups = {"GET"})
    public void getResources_toClientConversion() throws IllegalParameterValueException, ResourceNotFoundException {
        ResourceDetails details = new ResourceDetails();
        final int resultsCount = 10;
        for(int i = 0; i < resultsCount; i++){
            expectedResourceDetailsList.add(details);
        }
        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, null, null,null, null, null, null);
        assertNotNull(result);

        List<ClientResourceLookup> entity = ((ClientResourceListWrapper)result.getEntity()).getResourceLookups();

        assertEquals(entity.size(), resultsCount);
        for(ClientResourceLookup currentResultItem : entity){
            assertSame(currentResultItem, clientResourceLookup);
        }
    }

    @Test(groups = {"GET"},dependsOnMethods = "getResources_toClientConversion")
    public void getResources_ResponseCode_200() throws IllegalParameterValueException, ResourceNotFoundException {
        ResourceDetails details = new ResourceDetails();
        final int resultsCount = 10;
        for (int i = 0; i < resultsCount; i++) {
            expectedResourceDetailsList.add(details);
        }
        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, null, null, null, null, null, null);

        assertEquals(result.getStatus(), HttpServletResponse.SC_OK);
    }

    @Test(groups = {"GET"}, dependsOnMethods = "getResources_toClientConversion")
    public void getResources_ResponseCode_200_TotalCount() throws IllegalParameterValueException, ResourceNotFoundException {
        ResourceDetails details = new ResourceDetails();
        final int resultsCount = 10;
        for (int i = 0; i < resultsCount; i++) {
            expectedResourceDetailsList.add(details);
        }
        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, null, null, null, null, null, null);

        assertEquals(result.getStatus(), HttpServletResponse.SC_OK);
        assertEquals(result.getMetadata().getFirst(RestConstants.HEADER_TOTAL_COUNT), 10);
    }

    @Test(groups = {"GET"}, dependsOnMethods = "getResources_toClientConversion")
    public void getResources_ResponseCode_200_TotalCount_BiggerThenSize() throws IllegalParameterValueException, ResourceNotFoundException {
        ResourceDetails details = new ResourceDetails();
        final int resultsCount = 5;
        for (int i = 0; i < resultsCount; i++) {
            expectedResourceDetailsList.add(details);
        }

        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, null, 1, null, null, null, null);

        assertEquals(result.getStatus(), HttpServletResponse.SC_OK);
        assertEquals(result.getMetadata().getFirst(RestConstants.HEADER_TOTAL_COUNT), expectedResourceDetailsListSize);
    }

    @Test(groups = {"GET"}, dependsOnMethods = "getResources_toClientConversion")
    public void getResources_ResponseCode_200_TotalCount_Absent() throws IllegalParameterValueException, ResourceNotFoundException {
        ResourceDetails details = new ResourceDetails();
        final int resultsCount = 5;
        for (int i = 0; i < resultsCount; i++) {
            expectedResourceDetailsList.add(details);
        }

        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, 5, null, null, null, null, null);

        assertEquals(result.getStatus(), HttpServletResponse.SC_OK);
        assertNull(result.getMetadata().getFirst(RestConstants.HEADER_TOTAL_COUNT));
    }

    @Test(groups = {"GET"}, dependsOnMethods = "getResources_ResponseCode_200_TotalCount_Absent")
    public void getResources_ResponseCode_200_TotalCount_Forced() throws IllegalParameterValueException, ResourceNotFoundException {
        ResourceDetails details = new ResourceDetails();
        final int resultsCount = 5;
        for (int i = 0; i < resultsCount; i++) {
            expectedResourceDetailsList.add(details);
        }

        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, 5, null, null, null, true, null);

        assertEquals(result.getStatus(), HttpServletResponse.SC_OK);
        assertEquals(result.getMetadata().getFirst(RestConstants.HEADER_TOTAL_COUNT), expectedResourceDetailsListSize);
    }

    @Test(groups = {"GET"}, dependsOnMethods = "getResources_toClientConversion")
    public void getResources_ResponseCode_204() throws IllegalParameterValueException, ResourceNotFoundException {
        ClientResourceLookup clientResourceLookup = new ClientResourceLookup();
        when(lookupResourceConverter.toClient(any(ResourceDetails.class))).thenReturn(clientResourceLookup);
        final Response result = searchService.getResources(null, null, null, null, null,null, null, null, null);

        assertEquals(expectedResourceDetailsList.size(), 0);
        assertEquals(result.getStatus(), HttpServletResponse.SC_NO_CONTENT);
    }

    @Test(groups = {"DELETE"})
    public void deleteResources() throws Exception {
        searchService.deleteResources(new ArrayList<String>());

        verify(batchRepositoryService).deleteResources(anyListOf(String.class));
    }

    @Test(groups = {"DELETE"}, expectedExceptions = {IllegalParameterValueException.class})
    public void deleteResources_Null() throws Exception {
        searchService.deleteResources(null);
    }

    @Test(groups = {"DELETE"})
    public void deleteResources_statusCode() throws Exception {
        assertEquals(searchService.deleteResources(new ArrayList<String>()).getStatus(), HttpServletResponse.SC_NO_CONTENT) ;
    }

    @Test(groups = {"DELETE"}, expectedExceptions = {ResourceNotFoundException.class})
    public void deleteResources_notFound() throws Exception {
        doThrow(new ResourceNotFoundException("some")).when(batchRepositoryService).deleteResources(anyListOf(String.class));
        searchService.deleteResources(new ArrayList<String>());
    }

    @Test(groups = {"DELETE"}, expectedExceptions = {AccessDeniedException.class})
    public void deleteResources_accessDenied() throws Exception {
        doThrow(new AccessDeniedException("some")).when(batchRepositoryService).deleteResources(anyListOf(String.class));
        searchService.deleteResources(new ArrayList<String>());
    }
    @Test(groups = "COPY")
    public void testCopyResource_copy() throws Exception {
        Response response = searchService.copyResource(uri, true, true);

        verify(singleRepositoryService).copyResource(uri, Folder.SEPARATOR, true, true);
    }

    @Test(groups = "COPY")
    public void testCopyResource_status() throws Exception {
        Response response = searchService.copyResource(uri, true, true);

        assertEquals(response.getStatus(), HttpServletResponse.SC_NO_CONTENT);
    }

    @Test(groups = "COPY", expectedExceptions = {ResourceNotFoundException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testCopyResource_notFound() throws Exception {
        doThrow(ResourceNotFoundException.class).when(singleRepositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = searchService.copyResource(uri, true, true);
    }

    @Test(groups = "COPY", expectedExceptions = {AccessDeniedException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testCopyResource_accessDenied() throws Exception {
        doThrow(AccessDeniedException.class).when(singleRepositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = searchService.copyResource(uri, true, true);
    }

    @Test(groups = "MOVE")
    public void testMoveResource_move() throws Exception {
        Response response = searchService.copyResource(uri, true, true);

        verify(singleRepositoryService).copyResource(uri, Folder.SEPARATOR, true, true);
    }

    @Test(groups = "MOVE")
    public void testMoveResource_status() throws Exception {
        Response response = searchService.copyResource(uri, true, true);

        assertEquals(response.getStatus(), HttpServletResponse.SC_NO_CONTENT);
    }

    @Test(groups = "MOVE", expectedExceptions = {ResourceNotFoundException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testMoveResource_notFound() throws Exception {
        doThrow(ResourceNotFoundException.class).when(singleRepositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = searchService.copyResource(uri,true, true);
    }

    @Test(groups = "MOVE", expectedExceptions = {AccessDeniedException.class}, dependsOnMethods = {"testCopyResource_copy"})
    public void testMoveResource_accessDenied() throws Exception {
        doThrow(AccessDeniedException.class).when(singleRepositoryService).copyResource(anyString(), anyString(), anyBoolean(), anyBoolean());

        Response response = searchService.copyResource(uri, true, true);
    }


}

