/*
* Copyright (C) 2005 - 2009 Jaspersoft Corporation. All rights  reserved.
* http://www.jaspersoft.com.
*
* Unless you have purchased  a commercial license agreement from Jaspersoft,
* the following license terms  apply:
*
* This program is free software: you can redistribute it and/or  modify
* it under the terms of the GNU Affero General Public License  as
* published by the Free Software Foundation, either version 3 of  the
* License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Affero  General Public License for more details.
*
* You should have received a copy of the GNU Affero General Public  License
* along with this program.&nbsp; If not, see <http://www.gnu.org/licenses/>.
*/
package com.jaspersoft.jasperserver.remote.resources.converters;

import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceReference;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.dto.resources.ClientFile;
import com.jaspersoft.jasperserver.dto.resources.ClientReference;
import com.jaspersoft.jasperserver.dto.resources.ClientReferenceable;
import com.jaspersoft.jasperserver.dto.resources.ClientResourceLookup;
import com.jaspersoft.jasperserver.dto.resources.ClientUriHolder;
import com.jaspersoft.jasperserver.remote.exception.IllegalParameterValueException;
import com.jaspersoft.jasperserver.remote.exception.MandatoryParameterNotFoundException;
import com.jaspersoft.jasperserver.remote.resources.GenericParametersReflectionHelper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;


/**
 * <p></p>
 *
 * @author Yaroslav.Kovalchyk
 * @version $Id: ResourceReferenceConverter.java 28864 2013-02-21 11:27:18Z ykovalchyk $
 */
public class ResourceReferenceConverter<T extends ClientReferenceable> {
    protected final ResourceConverterProvider resourceConverterProvider;
    protected final RepositoryService repositoryService;
    protected final List<ClientReferenceRestriction> restrictions = new ArrayList<ClientReferenceRestriction>();

    public ResourceReferenceConverter(ResourceConverterProvider resourceConverterProvider,
            RepositoryService repositoryService, ClientReferenceRestriction... restriction) {
        this.repositoryService = repositoryService;
        this.resourceConverterProvider = resourceConverterProvider;
        if (restriction != null) {
            restrictions.addAll(Arrays.asList(restriction));
        }
    }

    public ResourceReferenceConverter<T> addReferenceRestriction(ClientReferenceRestriction restriction) {
        restrictions.add(restriction);
        return this;
    }

    /**
     * Converts ResoruceReferrence to client type object.
     *
     * @param serverObject ResourceReference instance
     * @return client referenceable object
     * @throws ClassCastException if server side local resource of given ResourceReference is of wrong type.
     */
    @SuppressWarnings("unchecked")
    public T toClient(ResourceReference serverObject) throws ClassCastException {
        ClientUriHolder result = null;
        if (serverObject != null) {
            if (serverObject.isLocal()) {
                final Resource localResource = serverObject.getLocalResource();
                result = resourceConverterProvider.getToClientConverter(localResource).toClient(localResource);
            } else {
                result = new ClientReference(serverObject.getReferenceURI());
            }
        }
        return (T) result;
    }

    public ResourceReference toServer(T clientObject) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        return toServer(clientObject, null);
    }

    public ResourceReference toServer(T clientObject, ResourceReference resultToUpdate)
            throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        final ResourceReference resourceReference;
        if (clientObject == null) {
            resourceReference = null;
        } else if (clientObject.getClass() == ClientReference.class) {
            resourceReference = toServerReference(clientObject, resultToUpdate);
        } else if (clientObject instanceof ClientResourceLookup) {
            resourceReference = toServerLocalResource((ClientResourceLookup) clientObject, resultToUpdate);
        } else {
            // shouldn't happen
            throw new IllegalParameterValueException("References of type "
                    + GenericParametersReflectionHelper.extractClientType(clientObject.getClass()) + " are not supported");
        }
        return resourceReference;
    }

    protected ResourceReference toServerReference(T clientObject, ResourceReference resultToUpdate) throws IllegalParameterValueException {
        ResourceReference result;
        final String uriFromClient = clientObject.getUri();
        if (resultToUpdate != null && !resultToUpdate.isLocal() && resultToUpdate.getReferenceURI().equals(uriFromClient)) {
            // no update is needed. Existing ResourceReference points to the same URI
            result = resultToUpdate;
        } else {
            validateReference(uriFromClient);
            if (resultToUpdate == null) {
                result = new ResourceReference(uriFromClient);
            } else {
                resultToUpdate.setReference(uriFromClient);
                result = resultToUpdate;
            }
        }
        return result;
    }

    protected void validateReference(String referenceUri) throws IllegalParameterValueException {
        if (referenceUri == null) {
            throw new IllegalParameterValueException("resourceLookup.uri", "null");
        }
        // we need to update reference
        final Resource resource = repositoryService.getResource(null, referenceUri);
        if (resource == null) {
            // resource with such URI doesn't exist
            throw new IllegalParameterValueException("Referenced resource doesn't exist", "resourceLookup.uri", referenceUri);
        } else if (!restrictions.isEmpty()) {
            final ClientResourceLookup clientTargetObject = resourceConverterProvider.getToClientConverter(resource).toClient(resource);
            for (ClientReferenceRestriction restriction : restrictions) {
                restriction.validateReference(clientTargetObject);
            }
        }
    }

    protected ResourceReference toServerLocalResource(ClientResourceLookup clientObject, ResourceReference resultToUpdate) throws IllegalParameterValueException, MandatoryParameterNotFoundException {
        ResourceReference result;
        Resource localResource = resourceConverterProvider.getToServerConverter(clientObject).toServer(clientObject);
        if (localResource.getName() == null) {
            localResource.setName(clientObject.getLabel());
        }
        if (resultToUpdate != null) {
            result = resultToUpdate;
            result.setLocalResource(localResource);
        } else {
            result = new ResourceReference(localResource);
        }
        return result;
    }

    public static class ReferenceClassRestriction implements ClientReferenceRestriction {
        protected final Class<? extends ClientReferenceable> targetClientClass;

        public ReferenceClassRestriction(Class<? extends ClientReferenceable> targetClientClass) {
            this.targetClientClass = targetClientClass;
        }

        @Override
        public void validateReference(ClientResourceLookup clientResource) throws IllegalParameterValueException {
            if (!targetClientClass.isAssignableFrom(clientResource.getClass())) {
                throw new IllegalParameterValueException("Reference target is of wrong type",
                        "resourceLookup.uri",
                        clientResource.getUri(),
                        GenericParametersReflectionHelper.extractClientType(clientResource.getClass()));
            }
        }
    }

    public static class FileTypeRestriction implements ClientReferenceRestriction{
        private final ClientFile.FileType fileType;
        private final String fieldName;

        public FileTypeRestriction(ClientFile.FileType fileType){
            this(fileType, null);
        }

        public FileTypeRestriction(ClientFile.FileType fileType, String fieldName){
            this.fileType = fileType;
            this.fieldName = fieldName;
        }

        @Override
        public void validateReference(ClientResourceLookup clientResource) throws IllegalParameterValueException {
            if(clientResource instanceof ClientFile && fileType != ((ClientFile)clientResource).getType()){
                throw new IllegalParameterValueException("Reference target is of wrong type",
                        fieldName != null ? fieldName : "file.type",
                        clientResource.getUri(),
                        ((ClientFile)clientResource).getType().name());
            }
        }
    }
}
