/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.jaspersoft.jasperserver.remote.services.impl;

import com.jaspersoft.jasperserver.api.JSValidationException;
import com.jaspersoft.jasperserver.api.engine.common.service.EngineService;
import com.jaspersoft.jasperserver.api.engine.common.service.ReportExecutionStatusInformation;
import com.jaspersoft.jasperserver.api.engine.common.service.SchedulerReportExecutionStatusSearchCriteria;
import com.jaspersoft.jasperserver.api.engine.jasperreports.domain.impl.ReportUnitResult;
import com.jaspersoft.jasperserver.api.metadata.xml.domain.impl.Argument;
import com.jaspersoft.jasperserver.remote.ServiceException;
import com.jaspersoft.jasperserver.remote.ServicesUtils;
import com.jaspersoft.jasperserver.remote.exception.RemoteException;
import com.jaspersoft.jasperserver.remote.exception.ResourceNotFoundException;
import com.jaspersoft.jasperserver.remote.exception.xml.ErrorDescriptor;
import com.jaspersoft.jasperserver.remote.exporters.HtmlExporter;
import com.jaspersoft.jasperserver.remote.services.ExecutionStatus;
import com.jaspersoft.jasperserver.remote.services.ExportExecution;
import com.jaspersoft.jasperserver.remote.services.ExportExecutionOptions;
import com.jaspersoft.jasperserver.remote.services.ReportExecution;
import com.jaspersoft.jasperserver.remote.services.ReportExecutionOptions;
import com.jaspersoft.jasperserver.remote.services.ReportExecutor;
import com.jaspersoft.jasperserver.remote.services.ReportOutputPages;
import com.jaspersoft.jasperserver.remote.services.ReportOutputResource;
import com.jaspersoft.jasperserver.remote.services.RunReportService;
import com.jaspersoft.jasperserver.remote.utils.AuditHelper;
import com.jaspersoft.jasperserver.war.action.ReportParametersUtils;
import com.jaspersoft.jasperserver.war.cascade.CascadeResourceNotFoundException;
import com.jaspersoft.jasperserver.war.cascade.InputControlsLogicService;
import com.jaspersoft.jasperserver.war.cascade.InputControlsValidationException;
import com.jaspersoft.jasperserver.war.dto.InputControlState;
import net.sf.jasperreports.engine.JRExporterParameter;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.export.GenericElementReportTransformer;
import net.sf.jasperreports.engine.export.JRHtmlExporterParameter;
import net.sf.jasperreports.engine.util.JRSaver;
import net.sf.jasperreports.engine.util.JRTypeSniffer;
import net.sf.jasperreports.web.servlets.JasperPrintAccessor;
import net.sf.jasperreports.web.servlets.ReportExecutionStatus;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Run Report service
 * Run a report unit using the passing in parameters and options
 *
 * @author gtoffoli
 * @version $Id: RunReportServiceImpl.java 28947 2013-02-26 15:02:08Z vsabadosh $
 */
@Service("runReportService")
@Scope(value = "session", proxyMode = ScopedProxyMode.TARGET_CLASS)
public class RunReportServiceImpl implements RunReportService {
    private final static Log log = LogFactory.getLog(RunReportServiceImpl.class);
    @javax.annotation.Resource
    private AuditHelper auditHelper;
    @javax.annotation.Resource
    private ServicesUtils servicesUtils;
    @javax.annotation.Resource(name = "concreteEngineService")
    private EngineService engine;
    @javax.annotation.Resource
    private InputControlsLogicService inputControlsLogicService;
    @javax.annotation.Resource
    private ReportExecutor reportExecutor;

    private final Map<String, ReportExecution> executions = new ConcurrentHashMap<String, ReportExecution>();

    public ReportExecution getReportExecution(String requestId) throws ResourceNotFoundException {
        final ReportExecution execution = executions.get(requestId);
        final ReportUnitResult reportUnitResult = execution != null ? execution.getReportUnitResult() : null;
        if (reportUnitResult != null) {
            ExecutionStatus executionStatus = null;
            final ReportExecutionStatus reportStatus = reportUnitResult.getJasperPrintAccessor().getReportStatus();
            final ReportExecutionStatus.Status status = reportStatus.getStatus();
            switch (status) {
                case CANCELED:
                    executionStatus = ExecutionStatus.cancelled;
                    break;
                case ERROR: {
                    executionStatus = ExecutionStatus.failed;
                    execution.setErrorDescriptor(new ErrorDescriptor(reportStatus.getError()));
                }
                break;
                case FINISHED: {
                    executionStatus = ExecutionStatus.ready;
                    execution.setTotalPages(reportStatus.getTotalPageCount());
                }
                break;
                case RUNNING: {
                    executionStatus = ExecutionStatus.execution;
                    execution.setCurrentPage(reportStatus.getCurrentPageCount());
                }
                break;
            }
            execution.setStatus(executionStatus);
            return execution;
        } else {
            throw new ResourceNotFoundException(requestId);
        }
    }

    @Override
    public ReportExecution getReportExecutionFromRawParameters(String reportUnitURI, Map<String, String[]> rawParameters,
            ReportExecutionOptions inputOptions, ExportExecutionOptions exportOptions) throws RemoteException, JSValidationException {
        if (exportOptions == null || exportOptions.getOutputFormat() == null) {
            throw new RemoteException(new ErrorDescriptor.Builder().setErrorCode("export.parameters.missing")
                    .setMessage("Output format must be specified").getErrorDescriptor());
        }
        final ReportExecutionOptions options = inputOptions != null ? inputOptions : new ReportExecutionOptions();
        // convert parameters from raw strings to objects
        Map<String, Object> convertedParameters;
        try {
            convertedParameters = executeInputControlsCascadeWithRawInput(reportUnitURI, rawParameters);
            // forward parameters without input control defined
            for (String currentKey : rawParameters.keySet()) {
                String[] currentValue = rawParameters.get(currentKey);
                if (currentValue != null && !convertedParameters.containsKey(currentKey)) {
                    Object valueToForward;
                    if (currentValue.length == 1) {
                        // forward as single value
                        valueToForward = currentValue[0];
                    } else {
                        // forward as collection
                        Collection<String> collection = new ArrayList<String>();
                        collection.addAll(Arrays.asList(currentValue));
                        valueToForward = collection;
                    }
                    convertedParameters.put(currentKey, valueToForward);
                }
            }
        } catch (CascadeResourceNotFoundException e) {
            throw new ResourceNotFoundException("URI:" + e.getResourceUri() + " Type:" + e.getResourceType());
        } catch (InputControlsValidationException e) {
            throw new JSValidationException(e.getErrors());
        }
        final ReportUnitResult reportUnitResult = reportExecutor.runReport(reportUnitURI, convertedParameters, options);
        final String requestId = reportUnitResult.getRequestId();
        final ReportExecution execution = new ReportExecution();
        execution.setReportUnitResult(reportUnitResult);
        executions.put(requestId, execution);
        execution.setRawParameters(rawParameters);
        execution.setOptions(options);
        final ExportExecution exportExecution = executeExport(exportOptions, execution);
        if (!options.isAsync()) {
            // wait till export is complete
            exportExecution.getFinalOutputResource();
        }
        return getReportExecution(requestId);
    }

    public ExportExecution executeExport(final String executionId, final ExportExecutionOptions exportOptions) throws RemoteException {
        final ReportExecution execution = executions.get(executionId);
        if (execution == null) throw new ResourceNotFoundException(executionId);
        return executeExport(exportOptions, execution);
    }

    protected ExportExecution executeExport(ExportExecutionOptions exportOptions, final ReportExecution reportExecution) throws RemoteException {
        ExportExecution runningExport = reportExecution.getExports().get(exportOptions);
        if (runningExport == null) {
            Boolean startExport = false;
            synchronized (reportExecution) {
                runningExport = reportExecution.getExports().get(exportOptions);
                if (runningExport == null) {
                    runningExport = new ExportExecution();
                    runningExport.setStatus(ExecutionStatus.queued);
                    runningExport.setOptions(exportOptions);
                    reportExecution.getExports().put(exportOptions, runningExport);
                    startExport = true;
                }
            }
            if (startExport) {
                final ExportExecution executionClosure = runningExport;
                final ReportExecutionOptions options = reportExecution.getOptions();
                String attachmentsPrefix = exportOptions.getAttachmentsPrefix() != null ?
                        exportOptions.getAttachmentsPrefix() : options.getDefaultAttachmentsPrefixTemplate();
               if (attachmentsPrefix != null) {
                   attachmentsPrefix = attachmentsPrefix
                            .replace(CONTEXT_PATH_ATTACHMENTS_PREFIX_TEMPLATE_PLACEHOLDER, options.getContextPath() != null ? options.getContextPath() : "")
                            .replace(REPORT_EXECUTION_ID_ATTACHMENTS_PREFIX_TEMPLATE_PLACEHOLDER, reportExecution.getRequestId())
                            .replace(EXPORT_OPTIONS_ATTACHMENTS_PREFIX_TEMPLATE_PLACEHOLDER, exportOptions.toString());
                }
                final String attachmentsPrefixClosure = attachmentsPrefix;
                new Thread(new Runnable() {
                    @Override
                    public void run() {
                        executeExport(executionClosure, reportExecution.getReportURI(), reportExecution.getReportUnitResult().getJasperPrintAccessor(),
                                attachmentsPrefixClosure, reportExecution.getRawParameters(), options);
                    }
                }).start();
            }
        }
        return runningExport;
    }

    protected void executeExport(ExportExecution exportExecution, String reportUnitURI, JasperPrintAccessor jasperPrintAccessor,
            String imagesUri, Map<String, String[]> rawParameters, ReportExecutionOptions options) {
        try {
            ExportExecutionOptions exportOptions = exportExecution.getOptions();
            Integer exportStartPage = null;
            if (exportOptions.getPages() != null) {
                exportStartPage = exportOptions.getPages().getPage() != null ? exportOptions.getPages().getPage() : exportOptions.getPages().getEndPage();
            }
            JasperPrint jasperPrint;
            if (exportStartPage != null) {
                // convert to page to pageIndex (0-based)
                exportStartPage = exportStartPage - 1;
                // in case if jasperPrintAccessor is AsyncJasperPrintAccessor and pageStatus() method is invoked,
                // then current thread waits till requested page is generated.
                jasperPrintAccessor.pageStatus(exportStartPage, null);
                jasperPrint = jasperPrintAccessor.getJasperPrint();
            } else {
                jasperPrint = jasperPrintAccessor.getFinalJasperPrint();
            }
            exportExecution.setStatus(ExecutionStatus.execution);
            generateReportOutput(reportUnitURI, jasperPrint, exportExecution.getOptions().getOutputFormat(), imagesUri,
                    exportExecution, rawParameters, exportExecution.getOptions().getPages(), options);
            exportExecution.setStatus(ExecutionStatus.ready);
        } catch (RemoteException ex) {
            exportExecution.setStatus(ExecutionStatus.failed);
            exportExecution.setErrorDescriptor(ex.getErrorDescriptor());
        } catch (Exception ex) {
            log.error("Unexpected error occurs during export", ex);
            exportExecution.setStatus(ExecutionStatus.failed);
            exportExecution.setErrorDescriptor(new ErrorDescriptor(ex));
        }
    }

    public ReportOutputResource getOutputResource(String executionId, ExportExecutionOptions exportExecutionOptions) throws RemoteException {
        final ReportExecution execution = executions.get(executionId);
        if (execution == null) throw new ResourceNotFoundException(executionId);
        return executeExport(exportExecutionOptions, execution).getFinalOutputResource();
    }

    public ReportOutputResource getAttachment(String executionId, ExportExecutionOptions exportOptions, String attachmentName) throws ResourceNotFoundException {
        ReportOutputResource outputResource = getExportExecution(executionId, exportOptions).getAttachments().get(attachmentName);
        if (outputResource == null) throw new ResourceNotFoundException(attachmentName);
        return outputResource;
    }

    public ExportExecution getExportExecution(String executionId, ExportExecutionOptions exportOptions) throws ResourceNotFoundException {
        final ReportExecution execution = executions.get(executionId);
        if (execution == null) throw new ResourceNotFoundException(executionId);
        final ExportExecution exportExecution = execution.getExports().get(exportOptions);
        if (exportExecution == null) throw new ResourceNotFoundException(exportOptions.toString());
        return exportExecution;
    }

    @Override
    public ReportOutputResource getReportOutputFromRawParameters(String reportUnitURI, Map<String, String[]> rawParameters,
            ReportExecutionOptions executionOptions, ExportExecutionOptions exportOptions) throws RemoteException {
        final ReportExecution execution = getReportExecutionFromRawParameters(reportUnitURI, rawParameters, executionOptions, exportOptions);
        return getOutputResource(execution.getRequestId(), exportOptions);
    }

    public ReportOutputResource getReportOutputFromRawParameters(
            String reportUnitURI,
            String outputFormat,
            Boolean ignorePagination,
            String attachmentsPrefix,
            Integer page,
            String transformerKey,
            Map<String, String[]> rawParameters,
            Boolean avoidCache,
            Boolean freshData,
            Boolean saveDataSnapshot,
            Boolean interactive) throws RemoteException {
        ReportExecutionOptions options = new ReportExecutionOptions()
                .setIgnorePagination(ignorePagination != null ? ignorePagination : false)
                .setTransformerKey(transformerKey)
                .setFreshData(freshData != null ? freshData : false)
                .setSaveDataSnapshot(saveDataSnapshot != null ? saveDataSnapshot : false)
                .setInteractive(interactive != null ? interactive : false);
        final ExportExecutionOptions exportOptions = new ExportExecutionOptions().setPages(new ReportOutputPages().setPage(page)).setAttachmentsPrefix(attachmentsPrefix);
        return getReportOutputFromRawParameters(reportUnitURI, rawParameters, options, exportOptions);
    }

    protected Map<String, Object> executeInputControlsCascadeWithRawInput(String reportUnitUri, Map<String, String[]> rawInputParameters) throws CascadeResourceNotFoundException, InputControlsValidationException {
        final List<InputControlState> valuesForInputControls = inputControlsLogicService.getValuesForInputControls(reportUnitUri, null, rawInputParameters);
        final Map<String, String[]> inputControlFormattedValues = ReportParametersUtils.getValueMapFromInputControlStates(valuesForInputControls);
        return inputControlsLogicService.getTypedParameters(reportUnitUri, inputControlFormattedValues);
    }

    /**
     * Report output generation or export.
     *
     * @param reportUnitURI   - URI of the report
     * @param jasperPrint     - filled with data jasper print object
     * @param rawOutputFormat - output format in raw format
     * @param imagesURI       - images URI prefix
     * @param exportExecution - export execution model
     */
    protected void generateReportOutput(String reportUnitURI, JasperPrint jasperPrint, String rawOutputFormat,
            String imagesURI, ExportExecution exportExecution, Map<String, String[]> rawParameters, ReportOutputPages pages,
            ReportExecutionOptions reportExecutionOptions) throws RemoteException {
        long currentTime = System.currentTimeMillis();
        auditHelper.createAuditEvent("runReport");
        try {
            String outputFormat = rawOutputFormat != null ? rawOutputFormat.toUpperCase() : Argument.RUN_OUTPUT_FORMAT_PDF;
            // Export...
            ByteArrayOutputStream bos = new ByteArrayOutputStream();

            if (outputFormat.equals(Argument.RUN_OUTPUT_FORMAT_JRPRINT)) {
                if (log.isDebugEnabled())
                    log.debug("Returning JasperPrint");
                if (reportExecutionOptions.getTransformerKey() != null) {
                    if (log.isDebugEnabled())
                        log.debug("Transforming JasperPrint generic element for key " + reportExecutionOptions.getTransformerKey());
                    GenericElementReportTransformer.transformGenericElements(reportExecutor.getJasperReportsContext(reportExecutionOptions.isInteractive()), jasperPrint, reportExecutionOptions.getTransformerKey());
                }
                JRSaver.saveObject(jasperPrint, bos);
                exportExecution.setOutputResource(new ReportOutputResource("application/octet-stream", bos.toByteArray()));
            } else {
                HashMap<String, Object> exportParameters = new HashMap<String, Object>(rawParameters);
                if (pages != null) exportParameters.put(Argument.RUN_OUTPUT_PAGES, pages);
                if (imagesURI != null) exportParameters.put(Argument.RUN_OUTPUT_IMAGES_URI, imagesURI);
                exportParameters.put(HtmlExporter.CONTEXT_PATH_PARAM_NAME, reportExecutionOptions.getContextPath() != null ? reportExecutionOptions.getContextPath() : "");
                Map<JRExporterParameter, Object> exporterParams;
                try {
                    exporterParams = reportExecutor.exportReport(reportUnitURI, jasperPrint, outputFormat, bos, exportParameters);
                    if (log.isDebugEnabled())
                        log.debug("Exporter params: " + Arrays.asList(exporterParams.keySet().toArray()));
                } catch (Exception e) {
                    log.error("Error exporting report", e);
                    throw new RemoteException(
                            new ErrorDescriptor.Builder()
                                    .setErrorCode("webservices.error.errorExportingReportUnit").setParameters(e.getMessage())
                                    .getErrorDescriptor(), e);
                } finally {
                    if (bos != null) {
                        try {
                            bos.close();
                        } catch (IOException ex) {
                            log.error("caught exception: " + ex.getMessage(), ex);
                        }
                    }
                }
                exportExecution.setOutputResource(new ReportOutputResource(reportExecutor.getContentType(outputFormat), bos.toByteArray()));
                if (Argument.RUN_OUTPUT_FORMAT_HTML.equals(outputFormat))
                    putImages(exporterParams, exportExecution.getAttachments());
            }
        } catch (RemoteException e) {
            auditHelper.addExceptionToAllAuditEvents(e);
            throw e;
        } catch (ServiceException e) {
            log.error("caught exception: " + e.getMessage(), e);
            auditHelper.addExceptionToAllAuditEvents(e);
        } catch (Throwable e) {
            log.error("caught Throwable exception: " + e.getMessage(), e);
            auditHelper.addExceptionToAllAuditEvents(e);
        }
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionStartTime", new Date(currentTime));
        auditHelper.addPropertyToAuditEvent("runReport", "reportExecutionTime", System.currentTimeMillis() - currentTime);
    }

    /**
     * Place images to output container.
     *
     * @param exportParameters - export result, contains images
     * @param outputContainer  - output container to fill with images
     * @throws RemoteException if any error occurs
     */
    protected void putImages(Map<JRExporterParameter, Object> exportParameters, Map<String, ReportOutputResource> outputContainer) throws RemoteException {
        try {
            // cast is safe because of known parameter key
            @SuppressWarnings("unchecked")
            Map<String, byte[]> imagesMap = (Map<String, byte[]>) exportParameters.get(JRHtmlExporterParameter.IMAGES_MAP);
            if (imagesMap != null && !imagesMap.isEmpty()) {
                if (log.isDebugEnabled()) {
                    log.debug("imagesMap : " + Arrays.asList(imagesMap.keySet().toArray()));
                }
                for (String name : imagesMap.keySet()) {
                    byte[] data = imagesMap.get(name);
                    if (log.isDebugEnabled()) {
                        log.debug("Adding image for HTML: " + name);
                    }
                    outputContainer.put(name, new ReportOutputResource(JRTypeSniffer.getImageTypeValue(data).getMimeType(), data, name));
                }
            }
        } catch (Throwable e) {
            log.error(e);
            throw new RemoteException(new ErrorDescriptor.Builder()
                    .setErrorCode("webservices.error.errorAddingImage").setParameters(e.getMessage()).getErrorDescriptor(), e);
        }
    }

    /**
     * @param searchCriteria - search criteria
     * @return set of currently running report's information
     */
    public Set<ReportExecutionStatusInformation> getCurrentlyRunningReports(SchedulerReportExecutionStatusSearchCriteria searchCriteria) {
        Set<ReportExecutionStatusInformation> result = null;
        List<ReportExecutionStatusInformation> reportExecutionStatusList = searchCriteria != null ?
                engine.getSchedulerReportExecutionStatusList(searchCriteria) : engine.getReportExecutionStatusList();
        if (reportExecutionStatusList != null && !reportExecutionStatusList.isEmpty()) {
            result = new HashSet<ReportExecutionStatusInformation>();
            result.addAll(reportExecutionStatusList);
        }
        return result;
    }

    public Boolean cancelReportExecution(String requestId) throws RemoteException {
        return engine.cancelExecution(requestId);
    }
}
