/*
 * Copyright (C) 2005 - 2011 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com.
 *
 * Unless you have purchased  a commercial license agreement from Jaspersoft,
 * the following license terms  apply:
 *
 * This program is free software: you can redistribute it and/or  modify
 * it under the terms of the GNU Affero General Public License  as
 * published by the Free Software Foundation, either version 3 of  the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero  General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public  License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.jaspersoft.jasperserver.search.service.impl;

import com.jaspersoft.jasperserver.api.common.domain.ExecutionContext;
import com.jaspersoft.jasperserver.api.common.domain.impl.ExecutionContextImpl;
import com.jaspersoft.jasperserver.api.logging.diagnostic.domain.DiagnosticAttribute;
import com.jaspersoft.jasperserver.api.logging.diagnostic.helper.DiagnosticAttributeBuilder;
import com.jaspersoft.jasperserver.api.logging.diagnostic.service.Diagnostic;
import com.jaspersoft.jasperserver.api.logging.diagnostic.service.DiagnosticCallback;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Folder;
import com.jaspersoft.jasperserver.api.metadata.common.domain.Resource;
import com.jaspersoft.jasperserver.api.metadata.common.domain.ResourceLookup;
import com.jaspersoft.jasperserver.api.metadata.common.service.RepositoryService;
import com.jaspersoft.jasperserver.api.search.SearchCriteriaFactory;
import com.jaspersoft.jasperserver.api.search.SearchFilter;
import com.jaspersoft.jasperserver.api.search.SearchSorter;
import com.jaspersoft.jasperserver.api.search.TransformerFactory;
import com.jaspersoft.jasperserver.search.common.CustomFilter;
import com.jaspersoft.jasperserver.search.common.CustomSorter;
import com.jaspersoft.jasperserver.search.common.RepositorySearchConfiguration;
import com.jaspersoft.jasperserver.search.common.ResourceDetails;
import com.jaspersoft.jasperserver.search.mode.SearchMode;
import com.jaspersoft.jasperserver.search.mode.SearchModeSettingsResolver;
import com.jaspersoft.jasperserver.search.service.ChildrenLoaderService;
import com.jaspersoft.jasperserver.search.service.RepositorySearchCriteria;
import com.jaspersoft.jasperserver.search.service.RepositorySearchService;
import com.jaspersoft.jasperserver.search.service.ResourceService;
import com.jaspersoft.jasperserver.search.sorter.ByLabelSorter;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Implementation of {@link RepositorySearchService}.
 *
 * @author Yuriy Plakosh
 * @version $Id: RepositorySearchServiceImpl.java 30161 2013-03-22 19:20:15Z inesterenko $
 */
@Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
public class RepositorySearchServiceImpl implements RepositorySearchService, Diagnostic {
    protected RepositoryService repositoryService;
    private ResourceService resourceService;
    protected TransformerFactory transformerFactory;
    private Map<String, ChildrenLoaderService> childrenLoaders;
    protected SearchCriteriaFactory defaultSearchCriteriaFactory;
    private SearchModeSettingsResolver searchModeSettingsResolver;
    protected Map<String, List<String>> filterOptionToResourceTypes;

    public void setSearchModeSettingsResolver(SearchModeSettingsResolver searchModeSettingsResolver) {
        this.searchModeSettingsResolver = searchModeSettingsResolver;
    }

    public void setDefaultSearchCriteriaFactory(SearchCriteriaFactory defaultSearchCriteriaFactory) {
        this.defaultSearchCriteriaFactory = defaultSearchCriteriaFactory;
    }

    public void setRepositoryService(RepositoryService repositoryService) {
        this.repositoryService = repositoryService;
    }

    public void setResourceService(ResourceService resourceService) {
        this.resourceService = resourceService;
    }

    public void setTransformerFactory(TransformerFactory transformerFactory) {
        this.transformerFactory = transformerFactory;
    }

    public void setFilterOptionToResourceTypes(Map<String, List<String>> filterOptionToResourceTypes) {
        this.filterOptionToResourceTypes = filterOptionToResourceTypes;
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public List<ResourceDetails> getResults(ExecutionContext context, SearchCriteriaFactory searchCriteriaFactory,
                                            List<SearchFilter> filters, SearchSorter sorter, int current, int max) {
        List<ResourceLookup> resources = repositoryService.getResources(context, searchCriteriaFactory,
                filters, sorter, transformerFactory, current, max);

        return getResourceDetailsList(resources);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public List<ResourceDetails> getResults(ExecutionContext context, RepositorySearchCriteria criteria) {
        context = putCriteriaToContext(context, criteria);
        // SearchMode.SEARCH is used by default
        final RepositorySearchConfiguration configuration = getConfiguration(criteria.getSearchMode() != null ? criteria.getSearchMode() : SearchMode.SEARCH);
        final List<SearchFilter> allFiltersList = createAllFiltersList(configuration);
        if (criteria.getCustomFilters() != null && !criteria.getCustomFilters().isEmpty())
            allFiltersList.addAll(criteria.getCustomFilters());
        SearchCriteriaFactory factory = defaultSearchCriteriaFactory;
        final List<String> resourceTypes = criteria.getResourceTypes();
        if (resourceTypes != null && resourceTypes.size() == 1) {
            // if the only resource type is requested, then build criteria for the only type.
            // It is possible via specific SearchCriteriaFactory
            final String singleResourceType = resourceTypes.get(0);
            factory = defaultSearchCriteriaFactory.newFactory(singleResourceType);
            if(Folder.class.getName().equals(singleResourceType)){
                // folder object don't have persistent field resourceType
                // single type searchCriteriaFactory is enough to restrict search results to folders only
                resourceTypes.clear();
            }

        }
        return getResults(context, factory, allFiltersList, getSorter(configuration, criteria.getSortBy()), criteria.getStartIndex(), criteria.getMaxCount());
    }

    public int getResultsCount(ExecutionContext context, RepositorySearchCriteria criteria) {
        context = putCriteriaToContext(context, criteria);
        // SearchMode.SEARCH is used by default
        final RepositorySearchConfiguration configuration = getConfiguration(criteria.getSearchMode() != null ? criteria.getSearchMode() : SearchMode.SEARCH);
        final List<SearchFilter> allFiltersList = createAllFiltersList(configuration);
        if (criteria.getCustomFilters() != null && !criteria.getCustomFilters().isEmpty())
            allFiltersList.addAll(criteria.getCustomFilters());
        SearchCriteriaFactory factory = defaultSearchCriteriaFactory;
        final List<String> resourceTypes = criteria.getResourceTypes();
        if (resourceTypes != null && resourceTypes.size() == 1) {
            // if the only resource type is requested, then build criteria for the only type.
            // It is possible via specific SearchCriteriaFactory
            final String singleResourceType = resourceTypes.get(0);
            factory = defaultSearchCriteriaFactory.newFactory(singleResourceType);
            if(Folder.class.getName().equals(singleResourceType)){
                // folder object don't have persistent field resourceType
                // single type searchCriteriaFactory is enough to restrict search results to folders only
                resourceTypes.clear();
            }

        }
        return getResultsCount(context, factory, allFiltersList, getSorter(configuration, criteria.getSortBy()));
    }

    @Override
    public Map<DiagnosticAttribute, DiagnosticCallback> getDiagnosticData() {
        final RepositorySearchCriteria repositorySearchCriteria = new RepositorySearchCriteriaImpl.Builder().setFolderUri("/")
                .setSearchText(null).setStartIndex(0).setMaxCount(0)
                .setSearchMode(SearchMode.SEARCH).setSortBy("name").getCriteria();
        final RepositorySearchConfiguration configuration = getConfiguration(repositorySearchCriteria.getSearchMode());
        final SearchCriteriaFactory factory = defaultSearchCriteriaFactory.newFactory(Resource.class.getCanonicalName());

        return new DiagnosticAttributeBuilder()
            .addDiagnosticAttribute(DiagnosticAttributeBuilder.TOTAL_REPORTS_COUNT, new DiagnosticCallback<Integer>() {
                @Override
                public Integer getDiagnosticAttributeValue() {
                    final List<SearchFilter> allFiltersList = new ArrayList<SearchFilter>(createAllFiltersList(configuration));
                    repositorySearchCriteria.setResourceTypes(filterOptionToResourceTypes.get("resourceTypeFilter-reports"));
                    return repositoryService.getResourcesCount(putCriteriaToContext(null, repositorySearchCriteria), factory, allFiltersList,
                            new ByLabelSorter(), transformerFactory);
                }
            })
            .addDiagnosticAttribute(DiagnosticAttributeBuilder.TOTAL_REPORT_OUTPUTS_COUNT, new DiagnosticCallback<Integer>() {
                @Override
                public Integer getDiagnosticAttributeValue() {
                    final List<SearchFilter> allFiltersList = new ArrayList<SearchFilter>(createAllFiltersList(configuration));
                    repositorySearchCriteria.setResourceTypes(filterOptionToResourceTypes.get("resourceTypeFilter-reportOutput"));
                    return repositoryService.getResourcesCount(putCriteriaToContext(null, repositorySearchCriteria), factory, allFiltersList,
                            new ByLabelSorter(), transformerFactory);
                }
            })
            .addDiagnosticAttribute(DiagnosticAttributeBuilder.TOTAL_FOLDERS_COUNT, new DiagnosticCallback<Integer>() {
                @Override
                public Integer getDiagnosticAttributeValue() {
                    return repositoryService.getAllFolders(null).size();
                }
            })
            .addDiagnosticAttribute(DiagnosticAttributeBuilder.TOTAL_OLAP_VIEWS_COUNT, new DiagnosticCallback<Integer>() {
                @Override
                public Integer getDiagnosticAttributeValue() {
                    final List<SearchFilter> allFiltersList = new ArrayList<SearchFilter>(createAllFiltersList(configuration));
                    repositorySearchCriteria.setResourceTypes(filterOptionToResourceTypes.get("resourceTypeFilter-view"));
                    return repositoryService.getResourcesCount(putCriteriaToContext(null, repositorySearchCriteria), factory, allFiltersList,
                            new ByLabelSorter(), transformerFactory);
                }
            })
            .addDiagnosticAttribute(DiagnosticAttributeBuilder.TOTAL_DATA_SOURCES_COUNT, new DiagnosticCallback<Integer>() {
                @Override
                public Integer getDiagnosticAttributeValue() {
                    final List<SearchFilter> allFiltersList = new ArrayList<SearchFilter>(createAllFiltersList(configuration));
                    repositorySearchCriteria.setResourceTypes(filterOptionToResourceTypes.get("resourceTypeFilter-dataSources"));
                    return repositoryService.getResourcesCount(putCriteriaToContext(null, repositorySearchCriteria), factory, allFiltersList,
                            new ByLabelSorter(), transformerFactory);
                }
            }).build();
    }
    
    /**
     * @param context  - execution context
     * @param criteria - search criteria
     * @deprecated criteria shouldn't be in context, temporary solution
     */
    protected ExecutionContext putCriteriaToContext(ExecutionContext context, RepositorySearchCriteria criteria) {
        ExecutionContext nullSafeContext = context != null ? context : ExecutionContextImpl.getRuntimeExecutionContext();
        if (nullSafeContext.getAttributes() == null)
            nullSafeContext.setAttributes(new ArrayList());
        nullSafeContext.getAttributes().add(criteria);
        return nullSafeContext;
    }

    public SearchSorter getSorter(RepositorySearchConfiguration configuration, String sortBy) {
        SearchSorter result = null;
        SearchSorter defaultSorter = null;
        for (CustomSorter sorter : configuration.getCustomSorters()) {
            if (sorter.getId().equals(sortBy)) {
                result = sorter.getSearchSorter();
                break;
            } else if(sorter.isDefault()){
                defaultSorter = sorter.getSearchSorter();
            }
        }
        return result != null ? result : defaultSorter;
    }

    protected RepositorySearchConfiguration getConfiguration(SearchMode searchMode) {
        return searchModeSettingsResolver.getSettings(searchMode).getRepositorySearchConfiguration();
    }

    protected List<SearchFilter> createAllFiltersList(RepositorySearchConfiguration configuration) {
        List<SearchFilter> filterList = new ArrayList<SearchFilter>();
        filterList.addAll(configuration.getSystemFilters());
        filterList.addAll(getRestrictionsFilters(configuration));
        return filterList;
    }

    protected List<SearchFilter> getRestrictionsFilters(RepositorySearchConfiguration configuration) {
        Set<String> customFilerIds = new HashSet<String>(configuration.getCustomFiltersMap().keySet());
        List<SearchFilter> filters = new ArrayList<SearchFilter>();
        if (customFilerIds != null && !customFilerIds.isEmpty())
            for (CustomFilter filter : configuration.getCustomFilters()) {
                if (customFilerIds.contains(filter.getId())) {
                    filters.add(filter.getFilter());
                }
            }
        return filters;
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public int getResultsCount(ExecutionContext context, SearchCriteriaFactory searchCriteriaFactory,
                               List<SearchFilter> filters, SearchSorter sorter) {
        return repositoryService.getResourcesCount(context, searchCriteriaFactory, filters, sorter, transformerFactory);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public List<ResourceDetails> getResourceChildren(String type, String resourceUri) {
        ChildrenLoaderService childrenLoaderService = childrenLoaders.get(type);

        if (childrenLoaderService != null) {
            return getResourceDetailsList(childrenLoaderService.getChildren(resourceUri));
        }

        return Collections.emptyList();
    }

    private List<ResourceDetails> getResourceDetailsList(List<ResourceLookup> resources) {
        List<ResourceDetails> list = new ArrayList<ResourceDetails>();

        for (Resource resource : resources) {
            ResourceDetails resourceDetails = resourceService.getResourceDetails(resource);

            list.add(resourceDetails);
        }

        return list;
    }

    public void setChildrenLoaders(Map<String, ChildrenLoaderService> childrenLoaders) {
        this.childrenLoaders = childrenLoaders;
    }

}
