//Copyright 2002-2003 Erwin Bolwidt. All rights reserved.
//See the file LICENSE.txt in this package for information about licensing.
package org.jaxup.tests;

import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.util.StringTokenizer;

import org.jaxen.FunctionCallException;
import org.jaxen.JaxenException;
import org.jaxen.Navigator;
import org.jaxup.UpdateException;
import org.jaxup.dom.DOMDocumentUpdater;
import org.jaxup.dom.DOMWriter;
import org.jaxup.dom4j.Dom4JDocumentUpdater;
import org.jaxup.dom4j.Dom4JWriter;
import org.jaxup.jdom.JDOMDocumentUpdater;
import org.jaxup.jdom.JDOMWriter;
import org.saxpath.SAXPathException;

public class TestDriver
{
    private final static String TEST_FILE_PATH = "tests/tests.txt";
    private final static String INPUT_DOC_PATH = "tests/input/";
    private final static String UPDATE_DOC_PATH = "tests/update/";
    private final static String RESULT_DOC_PATH = "tests/result/";

    private final static DocumentModelTestData[] MODELS =
        {
            new DocumentModelTestData("W3C", new DOMDocumentUpdater(), new DOMWriter()),
            new DocumentModelTestData("JDom", new JDOMDocumentUpdater(), new JDOMWriter()),
            new DocumentModelTestData("DOM4J", new Dom4JDocumentUpdater(), new Dom4JWriter())};

    public static void main(String[] args) throws IOException
    {
        new TestDriver().doTests();
    }

    private void doTests() throws IOException
    {
        LineNumberReader tests = new LineNumberReader(new FileReader(TEST_FILE_PATH));
        try
        {
            String line;
            while ((line = tests.readLine()) != null)
            {
                if (line.startsWith("#") || line.trim().length() == 0)
                {
                    continue;
                }
                StringTokenizer tokens = new StringTokenizer(line, ":, \t");
                if (tokens.countTokens() != 3)
                {
                    System.err.println("Bad test declaration on line " + tests.getLineNumber() + ": " + line);
                    continue;
                }

                String testName = tokens.nextToken();
                String inputDocName = tokens.nextToken();
                String updateDocName = tokens.nextToken();

                doTest(testName, inputDocName, updateDocName);
            }
        }
        finally
        {
            tests.close();
        }
    }

    /** 
     * Runs a test with a given input document and update document through all
     * the updaters that are configured in this TestDriver.
     */
    private void doTest(String testName, String inputDocName, String updateDocName)
    {

        for (int i = 0; i < MODELS.length; i++)
        {
            try
            {
                doTest(MODELS[i], testName, inputDocName, updateDocName);
                System.err.println("Test " + testName + " succesful for updater " + MODELS[i].getName());
            }
            catch (TestFailedException e)
            {
                System.err.println(
                    "Test "
                        + testName
                        + " failed for updater "
                        + MODELS[i].getName()
                        + ", reason: "
                        + e.getMessage());
                if (e.getCause() != null)
                    System.err.println("Cause: " + e.getCause());
                //e.printStackTrace();
            }
        }
    }

    /** 
     * Runs a test with a given input document and update document through
     * one given updater.
     */
    private void doTest(
        DocumentModelTestData model,
        String testName,
        String inputDocName,
        String updateDocName)
        throws TestFailedException
    {

        Navigator navigator = model.getUpdater().getNavigator();
        Object modifyDoc;
        Object updateDoc;
        Object goodResultDoc;

        try
        {
            modifyDoc = navigator.getDocument(INPUT_DOC_PATH + inputDocName);
            updateDoc = navigator.getDocument(UPDATE_DOC_PATH + updateDocName);
        }
        catch (FunctionCallException e)
        {
            throw new TestFailedException("Loading test documents", e);
        }

        try
        {
            goodResultDoc = navigator.getDocument(RESULT_DOC_PATH + testName + ".xml");
        }
        catch (FunctionCallException e)
        {
            try
            {
                goodResultDoc = model.getUpdater().createDocument();
            }
            catch (UpdateException eu)
            {
                throw new TestFailedException("Creating empty document", eu);
            }
        }

        try
        {
            model.getXUpdater().runUpdateDocument(modifyDoc, updateDoc);
        }
        catch (JaxenException e)
        {
            throw new TestFailedException("In the jaxen navigator", e);
        }
        catch (SAXPathException e)
        {
            throw new TestFailedException("Parsing an xpath expression", e);
        }
        catch (UpdateException e)
        {
            throw new TestFailedException("Performing update operations", e);
        }

        try
        {
            if (!model.getComparator().compare(modifyDoc, goodResultDoc))
            {
                try
                {
                    model.getWriter().write(modifyDoc, new OutputStreamWriter(System.err));
                }
                catch (IOException e)
                {
                }
                throw new TestFailedException("The modified document is not equal to the expected result of the test");
            }
        }
        catch (JaxenException e)
        {
            throw new TestFailedException("Comparing results", e);
        }
    }

}
