// Copyright 2001-2003 Erwin Bolwidt. All rights reserved.
// See the file LICENSE.txt in this package for information about licensing.
package org.jaxup.jdom;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.jaxen.Navigator;
import org.jaxen.jdom.DocumentNavigator;
import org.jaxen.jdom.XPathNamespace;
import org.jaxup.InvalidContextException;
import org.jaxup.UpdateException;
import org.jaxup.Updater;
import org.jdom.Attribute;
import org.jdom.Comment;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.ProcessingInstruction;
import org.jdom.Text;

/**
 * XML document creating and updating methods for the JDOM document object model.
 *
 * @author Erwin Bolwidt
 */
public class JDOMDocumentUpdater implements Updater
{
    private final static Navigator modifiedNavigator = new DocumentNavigator()
    {
        public Iterator getChildAxisIterator(Object contextNode)
        {
            if (contextNode instanceof Document)
            {
                Document doc = (Document) contextNode;
                if (!doc.hasRootElement())
                {
                    return Collections.EMPTY_LIST.iterator();
                }
            }
            return super.getChildAxisIterator(contextNode);
        }
    };

    public Object createDocument() throws UpdateException
    {
        return new Document(Collections.EMPTY_LIST);
    }

    public static Navigator getModifiedNavigator()
    {
        return modifiedNavigator;
    }

    public Navigator getNavigator()
    {
        return modifiedNavigator;
    }

    public Object createComment(Object contextNode, String comment) throws InvalidContextException
    {
        return new Comment(comment);
    }

    public Object createText(Object contextNode, String text) throws InvalidContextException
    {
        return new Text(text);
    }

    public Object createElement(Object contextNode, String uri, String qname) throws InvalidContextException
    {
        int colon = qname.indexOf(':');
        if (colon == -1)
        {
            // qname is unprefixed
            Namespace ns = Namespace.getNamespace(uri);
            return new Element(qname, ns);
        }
        else
        {
            String prefix = qname.substring(0, colon);
            String lname = qname.substring(colon + 1);
            Namespace ns = Namespace.getNamespace(prefix, uri);
            return new Element(lname, ns);
        }
    }

    public Object createNamespace(Object contextNode, String prefix, String uri)
        throws InvalidContextException
    {
        Namespace ns = Namespace.getNamespace(prefix, uri);
        return new XPathNamespace(ns);
    }

    public Object createAttribute(Object contextNode, String uri, String qname, String value)
        throws InvalidContextException
    {
        if (uri == null)
        {
            return new Attribute(qname, value);
        }

        int colon = qname.indexOf(':');
        if (colon == -1)
        {
            // qname is unprefixed
            Namespace ns = Namespace.getNamespace(uri);
            return new Attribute(qname, value, ns);
        }
        else
        {
            String prefix = qname.substring(0, colon);
            String lname = qname.substring(colon + 1);
            Namespace ns = Namespace.getNamespace(prefix, uri);

            return new Attribute(lname, value, ns);
        }
    }

    public Object createProcessingInstruction(Object contextNode, String target, String data)
        throws InvalidContextException
    {
        return new ProcessingInstruction(target, data);
    }

    protected Element getParent(Object node)
    {
        if (node instanceof Document)
        {
            return null;
        }
        else if (node instanceof Element)
        {
            return ((Element) node).getParent();
        }
        else if (node instanceof Attribute)
        {
            return ((Attribute) node).getParent();
        }
        else if (node instanceof XPathNamespace)
        {
            return ((XPathNamespace) node).getJDOMElement();
        }
        else if (node instanceof ProcessingInstruction)
        {
            return ((ProcessingInstruction) node).getParent();
        }
        else if (node instanceof Text)
        {
            return ((Text) node).getParent();
        }
        else if (node instanceof Comment)
        {
            return ((Comment) node).getParent();
        }
        else
        {
            return null;
        }
    }

    protected Document getDocument(Object node)
    {
        if (node instanceof Document)
        {
            return (Document) node;
        }
        else if (node instanceof Element)
        {
            return ((Element) node).getDocument();
        }
        else if (node instanceof Attribute)
        {
            return ((Attribute) node).getDocument();
        }
        else if (node instanceof XPathNamespace)
        {
            return ((XPathNamespace) node).getJDOMElement().getDocument();
        }
        else if (node instanceof ProcessingInstruction)
        {
            return ((ProcessingInstruction) node).getDocument();
        }
        else if (node instanceof Text)
        {
            return ((Text) node).getDocument();
        }
        else if (node instanceof Comment)
        {
            return ((Comment) node).getDocument();
        }
        else
        {
            return null;
        }
    }

    protected Object getParentOrDocument(Object node)
    {
        Element parent = getParent(node);
        if (parent == null)
        {
            return getDocument(node);
        }
        else
        {
            return parent;
        }
    }

    public void insertBefore(Object refNode, Object node) throws UpdateException
    {
        Object parent = getParentOrDocument(refNode);
        if (parent == null)
        {
            throw new InvalidContextException("refNode has no parent");
        }

        List siblings;
        if (parent instanceof Element)
        {
            siblings = ((Element) parent).getContent();
        }
        else
        {
            siblings = ((Document) parent).getContent();
        }

        int refIndex = siblings.indexOf(refNode);
        if (refIndex == -1)
        {
            throw new UpdateException("refNode is not a sibling of node");
        }
        siblings.add(refIndex, node);
    }

    public void insertAfter(Object refNode, Object node) throws UpdateException
    {
        Object parent = getParentOrDocument(refNode);
        if (parent == null)
        {
            throw new InvalidContextException("refNode has no parent");
        }

        List siblings;
        if (parent instanceof Element)
        {
            siblings = ((Element) parent).getContent();
        }
        else
        {
            siblings = ((Document) parent).getContent();
        }

        int refIndex = siblings.indexOf(refNode);
        if (refIndex == -1)
        {
            throw new UpdateException("refNode is not a sibling of node");
        }
        siblings.add(refIndex + 1, node);
    }

    public void appendChild(Object element, Object child, int position) throws UpdateException
    {
        if (element instanceof Element)
        {
            List siblings = ((Element) element).getContent();
            if (position == -1)
            {
                siblings.add(child);
            }
            else
            {
                siblings.add(position, child);
            }
        }
        else if (element instanceof Document)
        {
            Document doc = (Document) element;

            if (child instanceof Comment)
            {
                doc.addContent((Comment) child);
            }
            else if (child instanceof ProcessingInstruction)
            {
                doc.addContent((ProcessingInstruction) child);
            }
            else if (child instanceof Element)
            {
                doc.setRootElement((Element) child);
            }
            else if (child instanceof Text)
            {
                Text t = (Text) child;
                if (t.getTextTrim().length() == 0)
                {
                    return;
                }
                else
                {
                    throw new IllegalArgumentException(
                        "Can't add non-whitespace text to a document: " + t.getTextTrim());
                }
            }
            else if (child == null)
            {
                throw new NullPointerException("child");
            }
            else
            {
                throw new IllegalArgumentException(
                    "Can't add this type of child to a Document: " + child.getClass());
            }
        }
        else
        {
            throw new IllegalArgumentException(
                "Argument is neither an Element nor a Document as expected: " + element.getClass().getName());
        }
    }

    public void remove(Object node) throws UpdateException
    {
        Object parent = getParentOrDocument(node);
        if (parent == null)
        {
            throw new InvalidContextException("node has no parent");
        }

        if (node instanceof Attribute)
        {
            ((Element) parent).removeAttribute((Attribute) node);
        }
        else
        {

            List siblings;
            if (parent instanceof Element)
            {
                siblings = ((Element) parent).getContent();
            }
            else
            {
                siblings = ((Document) parent).getContent();
            }

            int index = siblings.indexOf(node);
            if (index == -1)
            {
                return;
            }
            else
            {
                siblings.remove(index);
            }
        }
    }

    public void setAttribute(Object element, Object attribute) throws UpdateException
    {
        if (!(element instanceof Element) || !(attribute instanceof Attribute))
        {
            throw new InvalidContextException("Invalid JDOM node types");
        }
        Attribute jdomAttribute = (Attribute) attribute;
        Element jdomElement = (Element) element;
        jdomElement.setAttribute(jdomAttribute);
    }

    public void setNamespace(Object element, Object namespace) throws UpdateException
    {
        if (!(element instanceof Element)
            || (!(namespace instanceof Namespace) && !(namespace instanceof XPathNamespace)))
        {
            throw new InvalidContextException("Invalid JDOM node types");
        }

        Element jdomElement = (Element) element;
        Namespace jdomNamespace;

        if (namespace instanceof XPathNamespace)
        {
            XPathNamespace xpathNamespace = (XPathNamespace) namespace;
            jdomNamespace = xpathNamespace.getJDOMNamespace();
            xpathNamespace.setJDOMElement(jdomElement);
        }
        else
        {
            jdomNamespace = (Namespace) namespace;
        }

        jdomElement.addNamespaceDeclaration(jdomNamespace);
    }

    public void setAttributeValue(Object attribute, String value) throws UpdateException
    {
        if (!(attribute instanceof Attribute))
        {
            throw new InvalidContextException("Invalid JDOM node type");
        }
        Attribute jdomAttribute = (Attribute) attribute;
        jdomAttribute.setValue(value);
    }
}
