/*
 * Decompiled with CFR 0.152.
 */
package jgnash.engine;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import jgnash.engine.Account;
import jgnash.engine.AccountGroup;
import jgnash.engine.CurrencyNode;
import jgnash.engine.Engine;
import jgnash.engine.InvestmentTransaction;
import jgnash.engine.MathConstants;
import jgnash.engine.SecurityNode;
import jgnash.engine.Transaction;

public class InvestmentPerformanceSummary {
    private Account account;
    private Date startDate;
    private Date endDate;
    private Map<SecurityNode, SecurityPerformanceData> performanceData = new TreeMap<SecurityNode, SecurityPerformanceData>();
    private List<Transaction> transactions;
    private CurrencyNode baseCurrency;

    public InvestmentPerformanceSummary(Account account, boolean recursive) {
        this(account, null, null, recursive);
    }

    private InvestmentPerformanceSummary(Account account, Date startDate, Date endDate, boolean recursive) {
        if (account == null) {
            throw new IllegalArgumentException("Account may not be null");
        }
        if (!account.memberOf(AccountGroup.INVEST)) {
            throw new IllegalArgumentException("Not a valid account type");
        }
        this.baseCurrency = account.getCurrencyNode();
        this.account = account;
        if (startDate == null || endDate == null) {
            this.setStartDate(new Date(1L));
            this.setEndDate(new Date());
        } else {
            this.setStartDate(startDate);
            this.setEndDate(endDate);
        }
        this.transactions = account.getTransactions(this.getStartDate(), this.getEndDate());
        if (recursive && account.getChildCount() > 0) {
            this.collectSubAccountTransactions(account, this.transactions);
        }
        Collections.sort(this.transactions);
        this.runCalculations(recursive);
    }

    private void collectSubAccountTransactions(Account account, List<Transaction> transactions) {
        for (Account child : account.getChildren()) {
            transactions.addAll(child.getTransactions(this.getStartDate(), this.getEndDate()));
            if (child.getChildCount() <= 0) continue;
            this.collectSubAccountTransactions(child, transactions);
        }
    }

    private void collectSubAccountSecurities(Account account, Set<SecurityNode> securities) {
        for (Account child : account.getChildren()) {
            securities.addAll(child.getSecurities());
            if (child.getChildCount() <= 0) continue;
            this.collectSubAccountSecurities(child, securities);
        }
    }

    public SecurityPerformanceData getPerformanceData(SecurityNode node) {
        return this.performanceData.get(node);
    }

    public List<SecurityNode> getSecurities() {
        return new ArrayList<SecurityNode>(this.performanceData.keySet());
    }

    private void calculateCostBasis(SecurityPerformanceData data, List<Transaction> transactions) {
        SecurityNode node = data.getNode();
        BigDecimal totalShares = BigDecimal.ZERO;
        BigDecimal totalCost = BigDecimal.ZERO;
        for (Transaction transaction : transactions) {
            InvestmentTransaction t;
            if (!(transaction instanceof InvestmentTransaction) || !(t = (InvestmentTransaction)transaction).getSecurityNode().equals(node)) continue;
            BigDecimal rate = this.baseCurrency.getExchangeRate(t.getInvestmentAccount().getCurrencyNode());
            BigDecimal fees = t.getFees().multiply(rate);
            BigDecimal quantity = t.getQuantity();
            BigDecimal price = t.getPrice().multiply(rate);
            switch (t.getTransactionType()) {
                case BUYSHARE: 
                case REINVESTDIV: {
                    totalShares = totalShares.add(quantity);
                    totalCost = totalCost.add(price.multiply(quantity).add(fees));
                    break;
                }
                case SPLITSHARE: {
                    totalShares = totalShares.add(quantity);
                    break;
                }
                case MERGESHARE: {
                    totalShares = totalShares.subtract(quantity);
                    break;
                }
                case DIVIDEND: {
                    break;
                }
            }
        }
        if (totalShares.compareTo(BigDecimal.ZERO) != 0) {
            data.setCostBasisShares(totalShares);
            data.setCostBasisPerShare(totalCost.divide(totalShares, MathConstants.mathContext));
        }
    }

    private void calculateRealizedGains(SecurityPerformanceData data, List<Transaction> transactions) {
        SecurityNode node = data.getNode();
        BigDecimal totalSharesSold = BigDecimal.ZERO;
        BigDecimal totalSales = BigDecimal.ZERO;
        for (Transaction transaction : transactions) {
            InvestmentTransaction t;
            if (!(transaction instanceof InvestmentTransaction) || !(t = (InvestmentTransaction)transaction).getSecurityNode().equals(node)) continue;
            BigDecimal rate = this.baseCurrency.getExchangeRate(t.getInvestmentAccount().getCurrencyNode());
            BigDecimal fees = t.getFees().multiply(rate);
            BigDecimal quantity = t.getQuantity();
            BigDecimal price = t.getPrice().multiply(rate);
            switch (t.getTransactionType()) {
                case SELLSHARE: {
                    totalSharesSold = totalSharesSold.add(quantity);
                    totalSales = totalSales.add(price.multiply(quantity).subtract(fees));
                    break;
                }
                case DIVIDEND: {
                    totalSales = totalSales.add(t.getTotal(t.getInvestmentAccount()).multiply(rate));
                    break;
                }
                case REINVESTDIV: {
                    totalSales = totalSales.add(t.getTotal(t.getInvestmentAccount()).multiply(rate)).subtract(fees);
                    break;
                }
            }
        }
        if (totalSharesSold.compareTo(BigDecimal.ZERO) != 0) {
            data.setAvgSalePrice(totalSales.divide(totalSharesSold, MathConstants.mathContext));
            data.setRealizedGains(data.getAvgSalePrice().subtract(data.getCostBasisPerShare()).multiply(totalSharesSold));
        } else if (totalSales.compareTo(BigDecimal.ZERO) != 0) {
            data.setRealizedGains(totalSales);
        }
    }

    private void calculateUnrealizedGains(SecurityPerformanceData data) {
        if (data.getSharesHeld().compareTo(BigDecimal.ZERO) != 0) {
            data.setUnrealizedGains(data.getPrice().subtract(data.getCostBasisPerShare()).multiply(data.getSharesHeld()));
        }
    }

    private void calculateTotalGains(SecurityPerformanceData data) {
        data.setTotalGains(data.getUnrealizedGains().add(data.getRealizedGains()));
        if (data.getTotalCostBasis().compareTo(BigDecimal.ZERO) != 0) {
            data.setTotalGainsPercentage(data.getTotalGains().divide(data.getTotalCostBasis(), MathConstants.mathContext));
        }
    }

    private void calculateShares(SecurityPerformanceData data, List<Transaction> transactions) {
        SecurityNode node = data.getNode();
        BigDecimal shares = BigDecimal.ZERO;
        for (Transaction transaction : transactions) {
            InvestmentTransaction t;
            if (!(transaction instanceof InvestmentTransaction) || !(t = (InvestmentTransaction)transaction).getSecurityNode().equals(node)) continue;
            switch (t.getTransactionType()) {
                case BUYSHARE: 
                case REINVESTDIV: 
                case SPLITSHARE: 
                case ADDSHARE: {
                    shares = shares.add(t.getQuantity());
                    break;
                }
                case MERGESHARE: 
                case SELLSHARE: 
                case REMOVESHARE: {
                    shares = shares.subtract(t.getQuantity());
                    break;
                }
            }
        }
        data.setSharesHeld(data.getSharesHeld().add(shares));
    }

    private void calculatePercentPortfolio() {
        BigDecimal marketValue = BigDecimal.ZERO;
        for (SecurityPerformanceData data : this.performanceData.values()) {
            marketValue = marketValue.add(data.getMarketValue(this.account.getCurrencyNode()));
        }
        for (SecurityPerformanceData data : this.performanceData.values()) {
            if (data.getMarketValue().compareTo(BigDecimal.ZERO) == 0) continue;
            BigDecimal percentage = data.getMarketValue(this.account.getCurrencyNode()).divide(marketValue, MathConstants.mathContext);
            data.setPercentPortfolio(percentage);
        }
    }

    private void runCalculations(boolean recursive) {
        Set<SecurityNode> nodes = this.account.getSecurities();
        if (recursive) {
            this.collectSubAccountSecurities(this.account, nodes);
        }
        for (SecurityNode node : nodes) {
            SecurityPerformanceData data = new SecurityPerformanceData(node);
            data.setPrice(this.getMarketPrice(node, this.getEndDate()));
            this.performanceData.put(node, data);
            this.calculateShares(data, this.transactions);
            this.calculateCostBasis(data, this.transactions);
            this.calculateRealizedGains(data, this.transactions);
            this.calculateUnrealizedGains(data);
            this.calculateTotalGains(data);
        }
        this.calculatePercentPortfolio();
    }

    private BigDecimal getMarketPrice(SecurityNode node, Date date) {
        return Engine.getMarketPrice(this.transactions, node, this.baseCurrency, date);
    }

    public String toString() {
        StringBuilder b = new StringBuilder();
        NumberFormat percentageFormat = NumberFormat.getPercentInstance();
        percentageFormat.setMinimumFractionDigits(2);
        for (SecurityPerformanceData data : this.performanceData.values()) {
            b.append(data.getNode().getSymbol()).append('\n');
            b.append("sharesHeld: ").append(data.getSharesHeld().toPlainString()).append('\n');
            b.append("price: ").append(data.getPrice().toPlainString()).append('\n');
            b.append("costBasisPerShare: ").append(data.getCostBasisPerShare().toPlainString()).append('\n');
            b.append("costBasisShares: ").append(data.getCostBasisShares().toPlainString()).append('\n');
            b.append("totalCostBasis: ").append(data.getTotalCostBasis().toPlainString()).append('\n');
            b.append("heldCostBasis: ").append(data.getHeldCostBasis().toPlainString()).append('\n');
            b.append("marketValue: ").append(data.getMarketValue().toPlainString()).append('\n');
            b.append("unrealizedGains: ").append(data.getUnrealizedGains().toPlainString()).append('\n');
            b.append("realizedGains: ").append(data.getRealizedGains().toPlainString()).append('\n');
            b.append("totalGains: ").append(data.getTotalGains().toPlainString()).append('\n');
            b.append("totalGainsPercentage: ").append(percentageFormat.format(data.getTotalGainsPercentage())).append('\n');
            b.append("sharesSold: ").append(data.getSharesSold().toPlainString()).append('\n');
            b.append("avgSalePrice: ").append(data.getAvgSalePrice().toPlainString()).append('\n');
            b.append("percentPortfolio: ").append(percentageFormat.format(data.getPercentPortfolio())).append('\n');
            b.append('\n');
        }
        return b.toString();
    }

    private void setStartDate(Date startDate) {
        this.startDate = startDate;
    }

    final Date getStartDate() {
        return this.startDate;
    }

    private void setEndDate(Date endDate) {
        this.endDate = endDate;
    }

    final Date getEndDate() {
        return this.endDate;
    }

    public class SecurityPerformanceData {
        private SecurityNode node;
        private BigDecimal sharesHeld = BigDecimal.ZERO;
        private BigDecimal costBasisPerShare = BigDecimal.ZERO;
        private BigDecimal costBasisShares = BigDecimal.ZERO;
        private BigDecimal avgSalePrice = BigDecimal.ZERO;
        private BigDecimal price = BigDecimal.ZERO;
        private BigDecimal realizedGains = BigDecimal.ZERO;
        private BigDecimal unrealizedGains = BigDecimal.ZERO;
        private BigDecimal totalGains = BigDecimal.ZERO;
        private BigDecimal totalGainsPercentage = BigDecimal.ZERO;
        private BigDecimal percentPortfolio = BigDecimal.ZERO;

        SecurityPerformanceData(SecurityNode node) {
            this.setNode(node);
        }

        public BigDecimal getCostBasisPerShare() {
            return this.costBasisPerShare;
        }

        public BigDecimal getCostBasisShares() {
            return this.costBasisShares;
        }

        public BigDecimal getMarketValue() {
            return this.getPrice().multiply(this.getSharesHeld(), MathConstants.mathContext);
        }

        public BigDecimal getMarketValue(CurrencyNode currencyNode) {
            return this.getPrice(currencyNode).multiply(this.getSharesHeld(), MathConstants.mathContext);
        }

        public SecurityNode getNode() {
            return this.node;
        }

        public BigDecimal getSharesHeld() {
            return this.sharesHeld;
        }

        public BigDecimal getPrice(CurrencyNode currencyNode) {
            return this.price.multiply(InvestmentPerformanceSummary.this.account.getCurrencyNode().getExchangeRate(currencyNode));
        }

        public BigDecimal getPrice() {
            return this.price;
        }

        public BigDecimal getRealizedGains() {
            return this.realizedGains;
        }

        public BigDecimal getTotalCostBasis() {
            return this.getCostBasisPerShare().multiply(this.getCostBasisShares(), MathConstants.mathContext);
        }

        public BigDecimal getHeldCostBasis() {
            return this.getCostBasisPerShare().multiply(this.getSharesHeld(), MathConstants.mathContext);
        }

        public BigDecimal getTotalGains() {
            return this.totalGains;
        }

        public BigDecimal getTotalGainsPercentage() {
            return this.totalGainsPercentage;
        }

        public BigDecimal getUnrealizedGains() {
            return this.unrealizedGains;
        }

        void setCostBasisPerShare(BigDecimal costBasis) {
            this.costBasisPerShare = costBasis;
        }

        void setCostBasisShares(BigDecimal costBasisShares) {
            this.costBasisShares = costBasisShares;
        }

        private void setNode(SecurityNode node) {
            this.node = node;
        }

        void setSharesHeld(BigDecimal sharesHeld) {
            this.sharesHeld = sharesHeld;
        }

        void setPrice(BigDecimal price) {
            this.price = price;
        }

        void setAvgSalePrice(BigDecimal avgSalePrice) {
            this.avgSalePrice = avgSalePrice;
        }

        void setUnrealizedGains(BigDecimal unrealizedGains) {
            this.unrealizedGains = unrealizedGains;
        }

        void setRealizedGains(BigDecimal realizedGains) {
            this.realizedGains = realizedGains;
        }

        void setTotalGains(BigDecimal totalGains) {
            this.totalGains = totalGains;
        }

        void setTotalGainsPercentage(BigDecimal totalGainsPercentage) {
            this.totalGainsPercentage = totalGainsPercentage;
        }

        public BigDecimal getAvgSalePrice() {
            return this.avgSalePrice;
        }

        public BigDecimal getSharesSold() {
            return this.getCostBasisShares().subtract(this.getSharesHeld());
        }

        public BigDecimal getPercentPortfolio() {
            return this.percentPortfolio;
        }

        public void setPercentPortfolio(BigDecimal percentPortfolio) {
            this.percentPortfolio = percentPortfolio;
        }
    }
}

