/*
 * Jitsi, the OpenSource Java VoIP and Instant Messaging client.
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package net.java.sip.communicator.impl.gui.main.presence;

import java.awt.*;
import java.awt.image.*;

import javax.swing.*;

import net.java.sip.communicator.impl.gui.*;
import net.java.sip.communicator.impl.gui.event.*;
import net.java.sip.communicator.impl.gui.lookandfeel.*;
import net.java.sip.communicator.impl.gui.main.*;
import net.java.sip.communicator.impl.gui.main.presence.avatar.*;
import net.java.sip.communicator.impl.gui.utils.*;
import net.java.sip.communicator.service.gui.*;
import net.java.sip.communicator.service.gui.Container;
import net.java.sip.communicator.service.protocol.*;
import net.java.sip.communicator.service.protocol.event.*;
import net.java.sip.communicator.util.*;
import net.java.sip.communicator.util.skin.*;

import net.java.sip.communicator.plugin.desktoputil.*;
import net.java.sip.communicator.plugin.desktoputil.SwingWorker;
import net.java.sip.communicator.plugin.desktoputil.TransparentPanel;

import org.jitsi.util.*;

/**
 * The panel shown on the top of the contact list. It contains user name,
 * current status menu and the avatar of the user.
 *
 * @author Yana Stamcheva
 * @author Adam Netocny
 */
public class AccountStatusPanel
    extends TransparentPanel
    implements  RegistrationStateChangeListener,
                ServerStoredDetailsChangeListener,
                PluginComponentListener,
                AvatarListener,
                Skinnable
{
    /**
     * Class id key used in UIDefaults.
     */
    private static final String uiClassID =
        AccountStatusPanel.class.getName() +  "OpaquePanelUI";

    /**
     * Adds the ui class to UIDefaults.
     */
    static
    {
        UIManager.getDefaults().put(uiClassID,
            SIPCommOpaquePanelUI.class.getName());
    }

    /**
     * The desired height of the avatar.
     */
    private static final int AVATAR_ICON_HEIGHT = 40;

    /**
     * The desired width of the avatar.
     */
    private static final int AVATAR_ICON_WIDTH = 40;

    /**
     * The image object storing the avatar.
     */
    private final FramedImage accountImageLabel;

    /**
     * The label showing the name of the user.
     */
    private final JLabel accountNameLabel
        = new EmphasizedLabel(
                GuiActivator
                    .getResources().getI18NString("service.gui.ACCOUNT_ME"));

    /**
     * The background color property.
     */
    private Color bgColor;

    /**
     * The background image property.
     */
    private Image logoBgImage;

    /**
     * The combo box containing status menu.
     */
    private final GlobalStatusSelectorBox statusComboBox;

    /**
     * TexturePaint used to paint background image.
     */
    private TexturePaint texture;

    /**
     * The tool bar plug-in container.
     */
    private final TransparentPanel toolbarPluginPanel;

    /**
     * The south plug-in container.
     */
    private final TransparentPanel southPluginPanel;

    private static byte[] currentImage;

    private String currentFirstName;

    private String currentLastName;

    private String currentDisplayName;

    private String globalDisplayName;

    /**
     * Property to disable auto answer menu.
     */
    private static final String GLOBAL_DISPLAY_NAME_PROP =
        "net.java.sip.communicator.impl.gui.main.presence.GLOBAL_DISPLAY_NAME";

    /**
     * Keep reference to plugin container or it will loose its
     * listener.
     */
    private final PluginContainer southPluginContainer;

    /**
     * Keep reference to plugin container or it will loose its
     * listener.
     */
    private final PluginContainer mainToolbarPluginContainer;

    /**
     * Creates an instance of <tt>AccountStatusPanel</tt> by specifying the
     * main window, where this panel is added.
     * @param mainFrame the main window, where this panel is added
     */
    public AccountStatusPanel(MainFrame mainFrame)
    {
        super(new BorderLayout(10, 0));

        FramedImageWithMenu imageWithMenu
            = new FramedImageWithMenu(
                    mainFrame,
                    new ImageIcon(
                            ImageLoader
                                .getImage(ImageLoader.DEFAULT_USER_PHOTO)),
                    AVATAR_ICON_WIDTH,
                    AVATAR_ICON_HEIGHT);
        imageWithMenu.setPopupMenu(new SelectAvatarMenu(imageWithMenu));
        this.accountImageLabel = imageWithMenu;

        accountNameLabel.setFont(
            accountNameLabel.getFont().deriveFont(12f));
        accountNameLabel.setOpaque(false);

        statusComboBox = new GlobalStatusSelectorBox(mainFrame);
        // Align status combo box with account name field.
        statusComboBox.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 0));

        TransparentPanel statusToolsPanel
            = new TransparentPanel(new BorderLayout(0, 0));

        SIPCommMenuBar statusMenuBar = new SIPCommMenuBar();
        statusMenuBar.add(statusComboBox);
        statusToolsPanel.add(statusMenuBar, BorderLayout.WEST);

        toolbarPluginPanel
            = new TransparentPanel(new FlowLayout(FlowLayout.RIGHT, 0, 0));

        mainToolbarPluginContainer = new PluginContainer(toolbarPluginPanel,
                            Container.CONTAINER_MAIN_TOOL_BAR);

        statusToolsPanel.add(toolbarPluginPanel, BorderLayout.EAST);

        TransparentPanel rightPanel = new TransparentPanel();
        rightPanel.setLayout(new BorderLayout(0, 0));
        rightPanel.add(accountNameLabel, BorderLayout.NORTH);
        rightPanel.add(statusToolsPanel, BorderLayout.SOUTH);

        this.add(accountImageLabel, BorderLayout.WEST);
        this.add(rightPanel, BorderLayout.CENTER);

        southPluginPanel = new TransparentPanel(new BorderLayout());

        southPluginContainer = new PluginContainer(
            southPluginPanel,
            Container.CONTAINER_ACCOUNT_SOUTH);

        this.add(southPluginPanel, BorderLayout.SOUTH);

        loadSkin();

        GuiActivator.getUIService().addPluginComponentListener(this);

        globalDisplayName = GuiActivator.getConfigurationService().getString(
            GLOBAL_DISPLAY_NAME_PROP, null);

        if(!StringUtils.isNullOrEmpty(globalDisplayName))
            accountNameLabel.setText(globalDisplayName);
    }

    /**
     * Adds the account given by <tt>protocolProvider</tt> in the contained
     * status combo box.
     * @param protocolProvider the <tt>ProtocolProviderService</tt>
     * corresponding to the account to add
     */
    public void addAccount(final ProtocolProviderService protocolProvider)
    {
        if(!SwingUtilities.isEventDispatchThread())
        {
            SwingUtilities.invokeLater(new Runnable()
            {
                public void run()
                {
                    addAccount(protocolProvider);
                }
            });
            return;
        }

        statusComboBox.addAccount(protocolProvider);

        protocolProvider.addRegistrationStateChangeListener(this);
    }

    /**
     * Removes the account given by <tt>protocolProvider</tt> from the contained
     * status combo box.
     * @param protocolProvider the <tt>ProtocolProviderService</tt>
     * corresponding to the account to remove
     */
    public void removeAccount(final ProtocolProviderService protocolProvider)
    {
        if(!SwingUtilities.isEventDispatchThread())
        {
            SwingUtilities.invokeLater(new Runnable()
            {
                public void run()
                {
                    removeAccount(protocolProvider);
                }
            });
            return;
        }

        if (containsAccount(protocolProvider))
        {
            statusComboBox.removeAccount(protocolProvider);
            protocolProvider.removeRegistrationStateChangeListener(this);
        }
    }

    /**
     * Checks if an account corresponding to the given <tt>protocolProvider</tt>
     * is contained in the contained status combo box.
     * @param protocolProvider the <tt>ProtocolProviderService</tt>
     * corresponding to the account to check for
     * @return <tt>true</tt> to indicate that an account corresponding to the
     * given <tt>protocolProvider</tt> is contained in the status box,
     * <tt>false</tt> - otherwise
     */
    public boolean containsAccount(ProtocolProviderService protocolProvider)
    {
        return statusComboBox.containsAccount(protocolProvider);
    }

    /**
     * Updates the current status of the <tt>protocolProvider</tt> with the
     * <tt>newStatus</tt>. If status is null uses the current status.
     * @param protocolProvider the <tt>ProtocolProviderService</tt> to update
     * @param newStatus the new status to set
     */
    public void updateStatus(final ProtocolProviderService protocolProvider,
                             final PresenceStatus newStatus)
    {
        if(!SwingUtilities.isEventDispatchThread())
        {
            SwingUtilities.invokeLater(new Runnable()
            {
                public void run()
                {
                    updateStatus(protocolProvider, newStatus);
                }
            });
            return;
        }

        if(newStatus != null)
            statusComboBox.updateStatus(protocolProvider, newStatus);
        else
            statusComboBox.updateStatus(protocolProvider);
    }

    /**
     * Updates the current status of the <tt>protocolProvider</tt>.
     * @param protocolProvider the <tt>ProtocolProviderService</tt> to update
     */
    public void updateStatus( ProtocolProviderService protocolProvider)
    {
        updateStatus(protocolProvider, null);
    }

    /**
     * Updates the image that is shown.
     * @param img the new image.
     */
    public void updateImage(ImageIcon img)
    {
        accountImageLabel.setImageIcon(img.getImage());
        accountImageLabel.setMaximumSize(
            new Dimension(AVATAR_ICON_WIDTH, AVATAR_ICON_HEIGHT));
        revalidate();
        repaint();
    }

    /**
     * Starts connecting user interface for the given <tt>protocolProvider</tt>.
     * @param protocolProvider the <tt>ProtocolProviderService</tt> to start
     * connecting for
     */
    public void startConnecting(final ProtocolProviderService protocolProvider)
    {
        if(!SwingUtilities.isEventDispatchThread())
        {
            SwingUtilities.invokeLater(new Runnable()
            {
                public void run()
                {
                    startConnecting(protocolProvider);
                }
            });
            return;
        }

        statusComboBox.startConnecting(protocolProvider);
    }

    /**
     * Stops connecting user interface for the given <tt>protocolProvider</tt>.
     * @param protocolProvider the <tt>ProtocolProviderService</tt> to stop
     * connecting for
     */
    public void stopConnecting(final ProtocolProviderService protocolProvider)
    {
        if(!SwingUtilities.isEventDispatchThread())
        {
            SwingUtilities.invokeLater(new Runnable()
            {
                public void run()
                {
                    stopConnecting(protocolProvider);
                }
            });
            return;
        }

        statusComboBox.stopConnecting(protocolProvider);
    }

    /**
     * Returns <tt>true</tt> if there are selected status selector boxes,
     * otherwise returns <tt>false</tt>.
     * @return <tt>true</tt> if there are selected status selector boxes,
     * otherwise returns <tt>false</tt>
     */
    public boolean hasSelectedMenus()
    {
        return statusComboBox.hasSelectedMenus();
    }

    /**
     * Updates account information when a protocol provider is registered.
     * @param evt the <tt>RegistrationStateChangeEvent</tt> that notified us
     * of the change
     */
    public void registrationStateChanged(RegistrationStateChangeEvent evt)
    {
        ProtocolProviderService protocolProvider = evt.getProvider();

        // There is nothing we can do when account is registering,
        // will set only connecting state later.
        // While dispatching the registering if the state of the provider
        // changes to registered we may end with client logged off
        // this may happen if registered is coming too quickly after registered
        // Dispatching registering is doing some swing stuff which
        // is scheduled in EDT and so can be executing when already registered
        if (!evt.getNewState().equals(RegistrationState.REGISTERING))
            this.updateStatus(protocolProvider);

        if (evt.getNewState().equals(RegistrationState.REGISTERED))
        {
            /*
             * Check the support for OperationSetServerStoredAccountInfo prior
             * to starting the Thread because only a couple of the protocols
             * currently support it and thus starting a Thread that is not going
             * to do anything useful can be prevented.
             */
            OperationSetServerStoredAccountInfo accountInfoOpSet
                = protocolProvider.getOperationSet(
                        OperationSetServerStoredAccountInfo.class);

            if (accountInfoOpSet != null)
            {
                /*
                 * FIXME Starting a separate Thread for each
                 * ProtocolProviderService is uncontrollable because the
                 * application is multi-protocol and having multiple accounts is
                 * expected so one is likely to end up with a multitude of
                 * Threads. Besides, it not very clear when retrieving the first
                 * and last name is to stop so one ProtocolProviderService being
                 * able to supply both the first and the last name may be
                 * overwritten by a ProtocolProviderService which is able to
                 * provide just one of them.
                 */
                new UpdateAccountInfo(protocolProvider, accountInfoOpSet)
                    .start();
            }

            OperationSetAvatar avatarOpSet
                = protocolProvider.getOperationSet(OperationSetAvatar.class);
            if (avatarOpSet != null)
                avatarOpSet.addAvatarListener(this);

            OperationSetServerStoredAccountInfo serverStoredAccountInfo
                = protocolProvider.getOperationSet(
                    OperationSetServerStoredAccountInfo.class);
            if (serverStoredAccountInfo != null)
                serverStoredAccountInfo.addServerStoredDetailsChangeListener(
                        this);
        }
        else if (evt.getNewState().equals(RegistrationState.UNREGISTERING)
                || evt.getNewState().equals(RegistrationState.CONNECTION_FAILED))
        {
            OperationSetAvatar avatarOpSet
                = protocolProvider.getOperationSet(OperationSetAvatar.class);
            if (avatarOpSet != null)
                avatarOpSet.removeAvatarListener(this);

            OperationSetServerStoredAccountInfo serverStoredAccountInfo
                = protocolProvider.getOperationSet(
                    OperationSetServerStoredAccountInfo.class);
            if (serverStoredAccountInfo != null)
                serverStoredAccountInfo.removeServerStoredDetailsChangeListener(
                        this);
        }
        else if (evt.getNewState().equals(RegistrationState.REGISTERING))
        {
            startConnecting(protocolProvider);
        }
    }

    /**
     * Paints this component.
     * @param g the <tt>Graphics</tt> object used for painting
     */
    public void paintComponent(Graphics g)
    {
        super.paintComponent(g);

        if (logoBgImage != null)
        {
            Graphics2D g2 = (Graphics2D) g;

            g.setColor(bgColor);
            g2.setPaint(texture);
            g2.fillRect(0, 0, this.getWidth(), this.getHeight());

            g.drawImage(
                logoBgImage,
                this.getWidth() - logoBgImage.getWidth(null),
                0,
                null);
        }
    }

    /**
     * Indicates that a plug-in component is registered to be added in a
     * container. If the plug-in component in the given event is registered for
     * this container then we add it.
     * @param event <tt>PluginComponentEvent</tt> that notified us
     */
    public void pluginComponentAdded(PluginComponentEvent event)
    {
        PluginComponent pluginComponent = event.getPluginComponent();
        Container containerID = pluginComponent.getContainer();
        /*
        // avoid early creating of components by calling getComponent
        Object component = pluginComponent.getComponent();

        if (!(component instanceof Component))
            return;
        */

        if (containerID.equals(Container.CONTAINER_MAIN_TOOL_BAR)
           || containerID.equals(Container.CONTAINER_ACCOUNT_SOUTH))
        {
            this.revalidate();
            this.repaint();
        }
    }

    /**
     * Indicates that a plug-in component is registered to be removed from a
     * container. If the plug-in component in the given event is registered for
     * this container then we remove it.
     * @param event <tt>PluginComponentEvent</tt> that notified us
     */
    public void pluginComponentRemoved(PluginComponentEvent event)
    {
        PluginComponent pluginComponent = event.getPluginComponent();
        Container pluginContainer = pluginComponent.getContainer();
        /*Object component = pluginComponent.getComponent();

        if (!(component instanceof Component))
            return;
        */

        if (pluginContainer.equals(Container.CONTAINER_MAIN_TOOL_BAR)
            || pluginContainer.equals(Container.CONTAINER_ACCOUNT_SOUTH))
        {
            this.revalidate();
            this.repaint();
        }
    }

    /**
     * Returns the global account image currently shown on the top of the
     * application window.
     * @return the global account image
     */
    public static byte[] getGlobalAccountImage()
    {
        return currentImage;
    }

    /**
     * Called whenever a new avatar is defined for one of the protocols that we
     * have subscribed for.
     *
     * @param event the event containing the new image
     */
    public void avatarChanged(AvatarEvent event)
    {
        currentImage = event.getNewAvatar();
        // If there is no avatar image set, then displays the default one.
        if(currentImage == null)
        {
            currentImage = ImageUtils.toByteArray(
                    ImageLoader.getImage(ImageLoader.DEFAULT_USER_PHOTO));
        }

        AvatarCacheUtils.cacheAvatar(
            event.getSourceProvider(), currentImage);

        accountImageLabel.setImageIcon(currentImage);
    }

    /**
     * Loads images for the account status panel.
     */
    public void loadSkin()
    {
        bgColor
            = new Color(GuiActivator.getResources()
                .getColor("service.gui.LOGO_BAR_BACKGROUND"));

        logoBgImage
            = ImageLoader.getImage(ImageLoader.WINDOW_TITLE_BAR);

        // texture
        BufferedImage bgImage
            = ImageLoader.getImage(ImageLoader.WINDOW_TITLE_BAR_BG);
        texture
            = new TexturePaint(
                    bgImage,
                    new Rectangle(
                            0,
                            0,
                            bgImage.getWidth(null),
                            bgImage.getHeight(null)));

        GuiActivator.getUIService().addPluginComponentListener(this);

        if(currentImage == null)
            accountImageLabel.setImageIcon(ImageLoader
                .getImage(ImageLoader.DEFAULT_USER_PHOTO));
    }

    /**
     * Registers a ServerStoredDetailsChangeListener with the operation sets
     * of the providers, if a provider change its name we use it in the UI.
     *
     * @param evt the <tt>ServerStoredDetailsChangeEvent</tt>
     * the event for name change.
     */
    public void serverStoredDetailsChanged(ServerStoredDetailsChangeEvent evt)
    {
        if(!StringUtils.isNullOrEmpty(globalDisplayName))
            return;

        if(evt.getNewValue() instanceof
                ServerStoredDetails.DisplayNameDetail
            && (evt.getEventID() == ServerStoredDetailsChangeEvent.DETAIL_ADDED
                || evt.getEventID()
                    == ServerStoredDetailsChangeEvent.DETAIL_REPLACED))
        {
            String accountName =
                    ((ServerStoredDetails.DisplayNameDetail)evt.getNewValue())
                        .getName();

            if (accountName != null && accountName.length() > 0)
                accountNameLabel.setText(accountName);
        }
    }

    /**
     * Returns the name of the L&F class that renders this component.
     *
     * @return the string "TreeUI"
     * @see JComponent#getUIClassID
     * @see UIDefaults#getUI
     */
    public String getUIClassID()
    {
        if(ConfigurationUtils.isTransparentWindowEnabled())
            return uiClassID;
        else
            return super.getUIClassID();
    }

    /**
     * Queries the operations sets to obtain names and display info.
     * Queries are done in separate thread.
     */
    private class UpdateAccountInfo
        extends SwingWorker
    {
        /**
         * The protocol provider.
         */
        private ProtocolProviderService protocolProvider;

        /**
         * The account info operation set to query.
         */
        private OperationSetServerStoredAccountInfo accountInfoOpSet;

        /**
         * Constructs with provider and opset to use.
         * @param protocolProvider the provider.
         * @param accountInfoOpSet the opset.
         */
        UpdateAccountInfo(
            ProtocolProviderService protocolProvider,
            OperationSetServerStoredAccountInfo accountInfoOpSet)
        {
            this.protocolProvider = protocolProvider;
            this.accountInfoOpSet = accountInfoOpSet;
        }

        /**
         * Worker thread method.
         * @return
         * @throws Exception
         */
        @Override
        protected Object construct()
            throws
            Exception
        {
            if (currentImage == null)
            {
                currentImage
                    = AvatarCacheUtils
                        .getCachedAvatar(protocolProvider);

                if (currentImage == null)
                {
                    byte[] accountImage
                        = AccountInfoUtils
                            .getImage(accountInfoOpSet);

                    // do not set empty images
                    if ((accountImage != null)
                            && (accountImage.length > 0))
                    {
                        currentImage = accountImage;

                        AvatarCacheUtils.cacheAvatar(
                            protocolProvider, accountImage);
                    }
                }
            }

            if(!StringUtils.isNullOrEmpty(globalDisplayName))
                return null;

            if (currentFirstName == null)
            {
                String firstName = AccountInfoUtils
                    .getFirstName(accountInfoOpSet);

                if (!StringUtils.isNullOrEmpty(firstName))
                {
                    currentFirstName = firstName;
                }
            }

            if (currentLastName == null)
            {
                String lastName = AccountInfoUtils
                    .getLastName(accountInfoOpSet);

                if (!StringUtils.isNullOrEmpty(lastName))
                {
                    currentLastName = lastName;
                }
            }

            if (currentFirstName == null && currentLastName == null)
            {
                String displayName = AccountInfoUtils
                    .getDisplayName(accountInfoOpSet);

                if (displayName != null)
                    currentDisplayName = displayName;
            }

            return null;
        }

        /**
         * Called on the event dispatching thread (not on the worker thread)
         * after the <code>construct</code> method has returned.
         */
        protected void finished()
        {
            if ((currentImage != null) && (currentImage.length > 0))
            {
                accountImageLabel.setImageIcon(currentImage);
            }

            String accountName = null;
            if (!StringUtils.isNullOrEmpty(currentFirstName))
            {
                accountName = currentFirstName;
            }

            if (!StringUtils.isNullOrEmpty(currentLastName))
            {
                /*
                 * If accountName is null, don't use += because
                 * it will make the accountName start with the
                 * string "null".
                 */
                if ((accountName == null)
                        || (accountName.length() == 0))
                    accountName = currentLastName;
                else
                    accountName += " " + currentLastName;
            }

            if (currentFirstName == null && currentLastName == null)
            {
                if (currentDisplayName != null)
                    accountName = currentDisplayName;
            }

            if (accountName != null && accountName.length() > 0)
                    accountNameLabel.setText(accountName);
        }
    }
}
