/***************************************************************************
                             KlassModelerDoc.cpp
                             -------------------
    begin                : Thu April 3, 2003
    copyright            : (C) 2003 by James Wells
    email                : james@wells.net

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 ***************************************************************************/
#include <wx/wxprec.h>
#ifdef __BORLANDC__
#pragma hdrstop
#endif
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif
#include "KlassModelerDoc.h"
#include "KlassModelerApp.h"
#include "KlassModelerView.h"
#include "ModelCanvas.h"
#include "MainFrame.h"
#include "Log.h"
#include <fcntl.h>
#ifdef __WXMSW__
#include <io.h>
#else
#include <unistd.h>
#include <cctype>
#include <errno.h>
#endif
#include <sys/stat.h>
#include <algorithm>
#include <wx/filename.h>
#include <wx/clipbrd.h>
#include <wx/dataobj.h>

#define DOC_VERSION_STRING_0_1		"DOC_VERSION 0.1"
#define DOC_VERSION_STRING_0_2		"DOC_VERSION 0.2"
#define DOC_VERSION_STRING_0_3 		"DOC_VERSION 0.3"
#define DOC_VERSION_STRING_0_4		"DOC_VERSION 0.4"
#define DOC_VERSION_STRING_0_5		"DOC_VERSION 0.5"
#define DOC_VERSION_STRING		 	"DOC_VERSION 0.6"

extern CLog MasterLog;

IMPLEMENT_DYNAMIC_CLASS(CKlassModelerDoc,wxDocument)

CKlassModelerDoc::CKlassModelerDoc()
	: wxDocument()
{
	m_Types.push_back( "void" );
	m_Types.push_back( "int" );
	m_Types.push_back( "float" );
	m_Types.push_back( "double" );
	m_Types.push_back( "char" );
	m_Types.push_back( "unsigned char" );
	m_Types.push_back( "unsigned short" );
	m_Types.push_back( "unsigned long" );
	m_Types.push_back( "short" );
	m_Types.push_back( "long" );
	m_Types.push_back( "unsigned int" );
	m_Types.push_back( "bool" );
	m_Types.push_back( "string" );
}

CKlassModelerDoc::~CKlassModelerDoc()
{
	ClassVector::iterator	itVect;

	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect )
		delete *itVect;
}

bool CKlassModelerDoc::OnNewDocument()
{
	if( wxDocument::OnNewDocument() ){
		wxGetApp().GetMainFrame()->DrawTree( this, true );
		return true;
	}

	return false;
}

bool CKlassModelerDoc::OnOpenDocument( const wxString &FileName )
{
	wxList				AllViews;
	wxNode				*pNode;
	CKlassModelerView	*pView;
	
	if( !wxDocument::OnOpenDocument( FileName ) )
		return false;
	wxGetApp().GetMainFrame()->DrawTree( this, true );
	AllViews = GetViews();
	pNode = AllViews.GetFirst();
	while( pNode ){
		pView = (CKlassModelerView*)pNode->GetData();
		pView->GetCanvas()->ComputeSizes();		
		pNode = pNode->GetNext();
	}
		
	return true;
}

void CKlassModelerDoc::AddClass( CClass *pClass )
{
	m_Classes.push_back( pClass );
}

void CKlassModelerDoc::RemoveClass( CClass *pClass )
{
	ClassVector::iterator	itVect;
	
	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		if( *itVect == pClass ){
			m_Classes.erase( itVect );
			return;
		}
	}
}

wxInputStream &CKlassModelerDoc::LoadObject( wxInputStream &Stream )
{
	char		VersionInfo[512];
	bool		CorrectVersion=false;

	wxDocument::LoadObject( Stream );

	for( int i=0; i<512; i++ ){
		Stream.Read( &(VersionInfo[i]), 1 );
		if( VersionInfo[i]==0 ){

			CorrectVersion = true;
			break;
		}
	}
	if( !CorrectVersion )
		return Stream;
	if( !strcmp( VersionInfo, DOC_VERSION_STRING_0_1 ) ){
		m_DocVersion = 0.1;
	} else if( !strcmp( VersionInfo, DOC_VERSION_STRING_0_2 ) ){
		m_DocVersion = 0.2;
	} else if( !strcmp( VersionInfo, DOC_VERSION_STRING_0_3 ) ){
		m_DocVersion = 0.3;
	} else if( !strcmp( VersionInfo, DOC_VERSION_STRING_0_4 ) ){
		m_DocVersion = 0.4;
	} else if( !strcmp( VersionInfo, DOC_VERSION_STRING_0_5 ) ){
		m_DocVersion = 0.5;

	} else if( !strcmp( VersionInfo, DOC_VERSION_STRING ) ){
		m_DocVersion = 0.6;
	} else {
		return Stream;
	}
	if( !ReadClasses( Stream ) )
		return Stream;
	if( !ReadInheritence( Stream ) )
		return Stream;
	if( m_DocVersion >= 0.6 ){
		if( !ReadRelation( Stream ) )
			return Stream;
	}
	Modify( false );

	return Stream;
}

bool CKlassModelerDoc::ReadClasses( wxInputStream &Stream )
{
	unsigned long		ClassCount;
	CClass				*pClass;
	string				Name;
	string				Doc;
	string				Header;
	string				HTML;

	Stream.Read( &ClassCount, sizeof(unsigned long) );
	for( unsigned long i=0; i<ClassCount; i++ ){
		pClass = new CClass( *this );
		Name = ReadString( Stream );
		Doc = ReadString( Stream );
		if( m_DocVersion == 0.1 ){
			int			X, Y;
			Stream.Read( &X, sizeof(int) );
			Stream.Read( &Y, sizeof(int) );
			pClass->SetPosition( (float)X, (float)Y );
		} else {
			float			fX;
			float			fY;
			Stream.Read( &fX, sizeof(float) );
			Stream.Read( &fY, sizeof(float) );
			pClass->SetPosition( fX, fY );
		}
		Header = ReadString( Stream );
		pClass->SetName( Name );
		pClass->SetDoc( Doc );
		pClass->SetHeaderFile( Header );
		if( m_DocVersion >= 0.4 ){
			HTML = ReadString( Stream );
			pClass->SetHTMLFile( HTML );
		}
		if( m_DocVersion >= 0.5 ){
			bool	bIsQtObject;
			Stream.Read( &bIsQtObject, sizeof(bool) );
			pClass->SetIsQObject( bIsQtObject );
		}
		if( m_DocVersion >= 0.6 ){
			bool	b;
			Stream.Read( &b, sizeof(bool) );
			pClass->SetExcludeHeader( b );
			Stream.Read( &b, sizeof(bool) );
			pClass->SetExcludeHTML( b );
		}
		if( !ReadMethods( Stream, pClass ) )
			return false;
		if( !ReadVariables( Stream, pClass ) )
			return false;
		if( m_DocVersion >= 0.6 ){
			if( !ReadEnums( Stream, pClass ) )
				return false;
		}
		m_Classes.push_back( pClass );
	}

	return true;
}

string CKlassModelerDoc::ReadString( wxInputStream &Stream )
{
	string	RetString;
	char		Letter;

	Stream.Read( &Letter, 1 );
	while( Letter!=0 ){
		RetString += Letter;
		Stream.Read( &Letter, 1 );
	}

	return RetString;
}

bool CKlassModelerDoc::ReadMethods( wxInputStream &Stream, CClass *pParent )
{
	CMethod			*pMethod;
	unsigned long	Count;
	string			Name;
	string			ReturnType;
	string			Doc;
	AccessType		Access;
	bool			bIsPureVirtual=false;
	bool			bIsQtSignal=false;
	bool			bIsQtSlot=false;
	bool			bIsConst=false;
	string			InlineCode;

	Stream.Read( &Count, sizeof(unsigned long) );
	for( unsigned long i=0; i<Count; i++ ){
		Name = ReadString( Stream );
		ReturnType = ReadString( Stream );

		Doc = ReadString( Stream );
		Stream.Read( &Access, sizeof(AccessType) );
		if( m_DocVersion >= 0.3 )
			Stream.Read( &bIsPureVirtual, sizeof(bool) );
		if( m_DocVersion >= 0.5 ){
			Stream.Read( &bIsQtSignal, sizeof(bool) );
			Stream.Read( &bIsQtSlot, sizeof(bool) );
			InlineCode = ReadString( Stream );
		}
		if( m_DocVersion >= 0.6 )
			Stream.Read( &bIsConst, sizeof(bool) );
		pMethod = new CMethod( *this );
		pMethod->SetName( Name );
		pMethod->SetReturnType( ReturnType );
		pMethod->SetDoc( Doc );
		pMethod->SetAccessType( Access );
		pMethod->SetPureVirtual( bIsPureVirtual );
		pMethod->SetIsConst( bIsConst );
		pMethod->SetIsQtSignal( bIsQtSignal );
		pMethod->SetIsQtSlot( bIsQtSlot );
		pMethod->SetInlineCode( InlineCode );
		if( !ReadVariables( Stream, pMethod ) )
			return false;
		pParent->AddMethod( pMethod );
	}

	return true;
}

bool CKlassModelerDoc::ReadVariables( wxInputStream &Stream, CClass *pParent )
{
	CVariable			*pVariable;
	unsigned long	Count;

	Stream.Read( &Count, sizeof(unsigned long) );
	for( unsigned long i=0; i<Count; i++ ){
		if( !ReadVariable( Stream, pVariable ) )
			return false;
		pParent->AddVariable( pVariable );
	}

	return true;
}

bool CKlassModelerDoc::ReadVariables( wxInputStream &Stream, CMethod *pParent )
{
	CVariable		*pVariable;
	unsigned long	Count;

	Stream.Read( &Count, sizeof(unsigned long) );
	for( unsigned long i=0; i<Count; i++ ){
		if( !ReadVariable( Stream, pVariable ) )
			return false;
		pParent->AddArg( pVariable );
	}

	return true;
}

bool CKlassModelerDoc::ReadVariable( wxInputStream &Stream, CVariable *&pVar )
{
	string 			Name;
	string			Type;
	string			Doc;
	AccessType		Access;

	Name = ReadString( Stream );
	Type = ReadString( Stream );
	Doc = ReadString( Stream );
	Stream.Read( &Access, sizeof(AccessType) );
	pVar = new CVariable( *this );
	pVar->SetName( Name );
	pVar->SetDoc( Doc );
	pVar->SetType( Type );
	pVar->SetAccessType( Access );

	return true;
}

bool CKlassModelerDoc::ReadEnums( wxInputStream &Stream, CClass *pParent )
{
	CEnum			*pEnum;
	unsigned long	Count;

	Stream.Read( &Count, sizeof(unsigned long) );
	for( unsigned long i=0; i<Count; i++ ){
		if( !ReadEnum( Stream, pEnum ) )
			return false;
		pParent->AddEnum( pEnum );
	}

	return true;
}

bool CKlassModelerDoc::ReadEnum( wxInputStream &Stream, CEnum *&pEnum )
{
	string 			Name;
	string			Text;
	AccessType		Access;

	Name = ReadString( Stream );
	Text = ReadString( Stream );
	Stream.Read( &Access, sizeof(AccessType) );

	pEnum = new CEnum( *this );
	pEnum->SetName( Name );
	pEnum->SetText( Text );
	pEnum->SetAccessType( Access );

	return true;
}

bool CKlassModelerDoc::ReadInheritence( wxInputStream &Stream )
{
	unsigned long		Count;
	string				Parent;
	string				Child;
	CClass				*pClass;
	CInheritence		*pInheritence;

	Stream.Read( &Count, sizeof(unsigned long) );
	for( unsigned long i=0; i<Count; i++ ){
      pInheritence = new CInheritence;
		Parent = ReadString( Stream );
		Child = ReadString( Stream );
		if( m_DocVersion == 0.1 ){
			int 	Pos;
			Stream.Read( &Pos, sizeof(int) );
			pInheritence->SetHorizontal( Pos );
		} else {
         float Horz;
			Stream.Read( &Horz, sizeof(float) );
			pInheritence->SetHorizontal( Horz );
		}
		pClass = FindClassNamed( Parent );
		if( !pClass )
         pInheritence->SetParentName( Parent );
      else
         pInheritence->SetParent( pClass );
		pClass = FindClassNamed( Child );
		if( !pClass ){
      	delete pInheritence;
			return false;
   	}
      pClass->AddInheritence( pInheritence );
	}

	return true;
}

bool CKlassModelerDoc::ReadRelation( wxInputStream &Stream )
{
	unsigned long		Count;
	string				Parent;
	string				Child;
	CClass				*pParent;
	CClass				*pChild;
	CRelation			*pRelation;
	bool				bIsHasA;

	Stream.Read( &Count, sizeof(unsigned long) );
	for( unsigned long i=0; i<Count; i++ ){
		Parent = ReadString( Stream );
		Child = ReadString( Stream );
		Stream.Read( &bIsHasA, sizeof(bool) );
		pParent = FindClassNamed( Parent );
		if( !pParent )
			return false;
		pChild = FindClassNamed( Child );
		if( !pChild )
			return false;
		pRelation = new CRelation;
		pRelation->SetChild( pChild );
		pRelation->SetHasA( bIsHasA );
		pParent->AddRelation( pRelation );
	}

	return true;
}

CClass *CKlassModelerDoc::FindClassNamed( string Name )
{
	ClassVector::iterator	itVect;

	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		if( (*itVect)->GetName() == Name )
			return *itVect;
	}

	return NULL;
}

bool CKlassModelerDoc::NewClassesFromHeader( string FileName, ClassVector &NewClasses )
{
	char			*pFile;
	int			fd;
	struct stat	Stat;
	bool			RetVal;
	wxString		Stripper;
	char			CRLF[3] = { 0xd, 0xa, 0 };

	fd = open( FileName.c_str(), O_RDONLY );
	if( fd == -1 ){
		wxMessageBox( _T("Couldn't open file"), _T("Verbose Error Info:") );
		return false;
	}
	fstat( fd, &Stat );
	pFile = new char[Stat.st_size+1];
	if( read( fd, pFile, Stat.st_size ) == -1 ){
		close( fd );	
		delete pFile;
		wxMessageBox( _T("Couldn't read file"), _T("Verbose Error Info:") );
		return false;
	}
	close( fd );
	pFile[Stat.st_size] = 0;
	Stripper = pFile;
	Stripper.Replace( CRLF, "\n" );
	RetVal = NewClassesFromString( (char*)Stripper.c_str(), Stat.st_size, FileName, NewClasses );
	delete pFile;

	return RetVal;
}

bool CKlassModelerDoc::NewClassesFromClipboard( ClassVector &NewClasses )
{
	if( !wxTheClipboard->Open() ){
		wxMessageBox( _T("Couldn't open clipboard.  I have no idea why"), _T("Error") );
		return false;
	}
	if( !wxTheClipboard->IsSupported( wxDF_TEXT ) ){
		wxMessageBox( _T("Clipboard data does not appear to be a header"), _T("Error") );
		return false;
	}
	wxTextDataObject	data;
 	wxString			TheText;
	char				CRLF[3] = { 0xd, 0xa, 0 };

	wxTheClipboard->GetData( data );
	TheText = data.GetText();
	TheText.Replace( CRLF, "\n" );
	wxTheClipboard->Close();

	return NewClassesFromString( (char*)TheText.c_str(), TheText.Length(), "", NewClasses );
}

bool CKlassModelerDoc::NewClassesFromString( char *pString, int Length, string FileName, ClassVector &NewClasses )
{
	string			CleanFile;
	int				pos;
	AccessType		Access;
	bool			bIsQtSignal, bIsQtSlot;
	bool			bEndOfClass;
	int				CheckPos;
	bool			bFoundAClass;
	CClass			*pNewClass;
	int				BytesRead;
	bool			bIsAccessChange;

	if( !Preprocessor( pString, Length, CleanFile ) ){
		wxMessageBox( _T("Preprocessor failed"), _T("Verbose Error Info:") );
		return false;
	}
	bFoundAClass = false;
	pos = 0;
	while( pos < CleanFile.length() ){
		pNewClass = new CClass( *this );
		if( !FileName.empty() ){
			if( GetBasePath() == FileName.substr( 0, GetBasePath().length() ) )
				pNewClass->SetHeaderFile( FileName.substr( GetBasePath().length() ) );
			else
				pNewClass->SetHeaderFile( FileName );
		}
		// At this point CleanFile is the same file, only stripped of
		// #directives and comments.
		if( !ReadClassNameAndInheritence( CleanFile, pos, pNewClass ) ){
			delete pNewClass;
			if( !bFoundAClass )
				wxMessageBox( _T("Couldn't find the class in the file"), _T("Verbose Error Info:") );
			return bFoundAClass;				// most likely this will be the exits point if 2 classes in a file
		}
		bFoundAClass = true;
		pos++;
		// let's see if this is a Q_OBJECT
		if( !StripWhiteSpaces( CleanFile, pos ) ){
			delete pNewClass;
			wxMessageBox( _T("failed to strip white spaces"), _T("Verbose Error Info:") );
			return false;
		}
		if( CleanFile.substr( pos, 8 ) == "Q_OBJECT" ){
			pos += 8;
			pNewClass->SetIsQObject( true );
		}
		Access = Private;
		bEndOfClass = false;
		while( !bEndOfClass ){
			CheckPos = pos;
			if( pos > CleanFile.length() ){
				delete pNewClass;
				wxMessageBox( _T("Somehow went past the end of the file"), _T("Verbose Error Info:") );
				return false;				// premature end of file
			}
			if( !StripWhiteSpaces( CleanFile, pos ) ){
				delete pNewClass;
				wxMessageBox( _T("Failed to strip white spaces (case 2)"), _T("Verbose Error Info:") );
				return false;
			}
			if( !CheckForAccessType( CleanFile, pos, Access, bIsQtSignal, bIsQtSlot, bIsAccessChange ) ){
				delete pNewClass;
				wxMessageBox( _T("Couldn't find access type"), _T("Verbose Error Info:") );
				return false;
			}
			// for those folks who like to have lots of empty access
			// statements, let's keep looking for them.  
			while( bIsAccessChange ){
				if( !CheckForAccessType( CleanFile, pos, Access, bIsQtSignal, bIsQtSlot, bIsAccessChange ) ){
					delete pNewClass;
					wxMessageBox( _T("Couldn't find access type"), _T("Verbose Error Info:") );
					return false;
				}
			}
			// now the four kinds of things I'm looking for.
			// - enum reader
			// - I don't handle friends correctly, so do a specific check
			// - Check for the end of the class
			// - Check for a variable or method
			if( CleanFile.substr( pos, 5 ) == "enum " ){
				if( !ReadEnum( CleanFile, pos, Access, *pNewClass ) ){
					delete pNewClass;
					wxMessageBox( _T("error handling enum"), _T("Verbose Error Info:") );
					return false;
				}
			} else if( CleanFile.substr( pos, 7 ) == "friend " ){
				pos = CleanFile.find( ";", pos );
				if( pos == CleanFile.npos ){
					delete pNewClass;
					wxMessageBox( _T("error handling friend"), _T("Verbose Error Info:") );
					return false;
				}
				pos++;
			} else if( CleanFile.substr( pos, 2 ) == "};" ){
				bEndOfClass = true;
				pos += 2;
			} else {
				if( !ReadVariableOrMethod( CleanFile, pos, Access, *pNewClass, bIsQtSignal, bIsQtSlot ) ){
					delete pNewClass;
					wxMessageBox( _T("error reading variable or method"), _T("Verbose Error Info:") );
					return false;
				}
			}
			// to make sure progress is actually being made and we're not in an infenite loop
			// check to see if pos has been updated at all.  If not then it's an error.
			if( CheckPos == pos ){
				delete pNewClass;
				wxMessageBox( _T("passed through main loop with no progress, exiting to stop infinite loop"), _T("Verbose Error Info:") );
				return false;
			}
		}
		NewClasses.push_back( pNewClass );
	}

	return true;
}

bool CKlassModelerDoc::Preprocessor( char *pFile, int FileSize, string &CleanFile )
{
	char	*pHold = pFile;
	char	*pCheckHold;
	
	// read the file line by line.
	while( *pHold != NULL ){
		pCheckHold = pHold;
		if( !strncmp( pHold, "/*", 2 ) ){
			pHold = strstr( pHold, "*/" );
			if( pHold == NULL ){
				wxMessageBox( _T("found the /*, but couldn't find the */"), _T("Verbose Error Info:") );
				return false;
			}
			pHold += 2;
		} else if( !strncmp( pHold, "//", 2 ) ){
			pHold = strchr( pHold, '\n' );
			if( pHold == NULL ){
				wxMessageBox( _T("found the //, but couldn't find the end of line"), _T("Verbose Error Info:") );
				return false;
			}
			pHold++;
		} else if( *pHold == '#' ){
			pHold = strchr( pHold, '\n' );
			// different handling because you don't NEED a \n
			if( pHold != NULL )
				pHold++;
			else
				// I think the only way to get here is if you have
				// a #something that ends the file with no \n in it.
				// So if we get here the file is over.
				return true;
		} else {
			CleanFile += *pHold;
			pHold++;
		}
		// if pHold has not been updated then it's an error
		if( pHold == pCheckHold ){
			wxMessageBox( _T("No progress after preprocessor loop, exiting to avoid infinite loop"), _T("Verbose Error Info:") );
			return false;
		}
	}
		
	return true;
}

// upon succesful return the class name and inheritence will be read into the pClass.
// pos will be update to point to the first { in the class.
bool CKlassModelerDoc::ReadClassNameAndInheritence( string &File, int &Pos, CClass *pClass )
{
	string			ClassName;
	int				InheritencePos;
	int				SquigglePos;
	CInheritence	*pInheritence;

	if( !FindClass( File, Pos ) )
		return false;
	if( !FindWhiteSpaces( File, Pos ) )
		return false;
	if( !StripWhiteSpaces( File, Pos ) )
		return false;
	if( !ReadToken( File, ClassName, Pos ) )
		return false;
	pClass->SetName( ClassName );
	SquigglePos = File.find( "{", Pos );
	if( SquigglePos == File.npos ){
		wxMessageBox( _T("Couldn't find a {"), _T("Verbose Error Info:") );
		return false;
	}
	InheritencePos = File.find( ":", Pos );
	if( InheritencePos == File.npos ){
		Pos = SquigglePos;
		return true;
	}
	if( InheritencePos > SquigglePos ){
		Pos = SquigglePos;
		return true;
	}
	// if we get this far then there is inheritence info to figure out
	while( Pos < SquigglePos ){
		InheritencePos++;
		ClassName = "";
		if( !StripWhiteSpaces( File, InheritencePos ) )
			return false;
		// now we're on the inheritence type, public, protected, or private
		if( !FindWhiteSpaces( File, InheritencePos ) )
			return false;
		if( !StripWhiteSpaces( File, InheritencePos ) )
			return false;
		// now we're on the parent		
		if( !ReadToken( File, ClassName, InheritencePos ) )
			return false;
		pInheritence = new CInheritence;
		pInheritence->SetParentName( ClassName );
		pClass->AddInheritence( pInheritence );
		InheritencePos = File.find( ",", InheritencePos );
		if( InheritencePos == File.npos )
			Pos = SquigglePos;
		else if( InheritencePos < SquigglePos )
			Pos = InheritencePos;
		else
			Pos = SquigglePos;
	}

	return true;
}

// checks to see if the next set of tokens represents a change in access type
// if it does not then there is no change in variables.  If it does then Pos will
// come back pointing to the next token
bool CKlassModelerDoc::CheckForAccessType( string &File, int &Pos, AccessType &Access, bool &bIsQtSignal, bool &bIsQtSlot, bool &bFoundAccess )
{
	bFoundAccess = false;

	if( File.substr( Pos, 6 ) == "public" ){
		Access = Public;
		Pos += 6;
		StripWhiteSpaces( File, Pos );		// doesn't matter if it fails
		bFoundAccess = true;
	} else if( File.substr( Pos, 9 ) == "protected" ){
		Access = Protected;
		Pos += 9;
		StripWhiteSpaces( File, Pos );		// doesn't matter if it fails
		bFoundAccess = true;
	} else if( File.substr( Pos, 7 ) == "private" ){
		Access = Private;
		Pos += 7;
		StripWhiteSpaces( File, Pos );		// doesn't matter if it fails
		bFoundAccess = true;
	}
	// at this point we may or may not have found the access type
	// If we have then Pos is either on :, signals, or slots
	// if we have not the Pos is unmoved
	if( File[Pos] == ':' ){
		Pos++;
		bIsQtSignal = false;
		bIsQtSlot = false;
		if( !StripWhiteSpaces( File, Pos ) )
			return false;
		return bFoundAccess;				// if we haven't found Access type then this function failed
	} else if( File.substr( Pos, 8 ) == "signals:" ){
		bIsQtSignal = true;
		bIsQtSlot = false;
		if( !FindWhiteSpaces( File, Pos ) )
			return false;
		if( !StripWhiteSpaces( File, Pos ) )
			return false;
		if( !bFoundAccess )
			Access = Public;				// you can say just "signals:" without saying "public signals:"
		return true;
	} else if( File.substr( Pos, 6 ) == "slots:" ){
		bIsQtSignal = false;
		bIsQtSlot = true;
		if( !FindWhiteSpaces( File, Pos ) )
			return false;
		if( !StripWhiteSpaces( File, Pos ) )
			return false;
		if( !bFoundAccess )
			Access = Public;				// you can say just "slots:" without saying "public slots:"
		return true;
	}

	// if we get this far that just means that this line was NOT a access change line
	// it is probably a variable or method decleration
	return true;
}

// upon successful return, Pos is the position of the c in class
bool CKlassModelerDoc::FindClass( string &File, int &Pos )
{
	int i, j, k;

	i = 0;
	while( i != File.npos ){
		i = File.find( "class ", Pos );
		if( i == File.npos ){
// will keep trying to find a class.  So it may find one and if 
// that's the only one it will call this anyway.  So don't popup
// an error because that might be correct (to only find 1)
//			wxMessageBox( _T("Couldn't find \"class\" token"), _T("Verbose Error Info:") );
			return false;
		}
		j = File.find( ";", i );
		if( i == File.npos ){
			wxMessageBox( _T("Couldn't find a closing ; for this class"), _T("Verbose Error Info:") );
			return false;
		}
		k = File.find( "{", i );
		if( i == File.npos ){
			wxMessageBox( _T("Couldn't find the opening { for this class"), _T("Verbose Error Info:") );
			return false;
		}
		// check to make sure this is the real class and not a forward define
		if( k < j ){
			Pos = i;
			return true;
		} else
			Pos = j;
	}	
	wxMessageBox( _T("Could not find a \"class\", a \"{\", and a \";\" in this file"), _T("Verbose Error Info:") );
	
	return false;
}

bool CKlassModelerDoc::ReadToken( string &Src, string &Dest, int &Pos )
{
	while( (Src[Pos]!=' ') && (Src[Pos]!='\n') && (Src[Pos]!='\t') &&
			(Src[Pos]!=',') && (Src[Pos]!=';') && (Src[Pos]!='(') &&
			(Src[Pos]!=')') && (Src[Pos]!='{') && (Src[Pos]!='}') ){
		// the = is an annoying special case.  If this is a pure virtual:
		// void Func()=0;
		// then the token is done.  If this is an operator:
		// void Operator=();
		// void Operator!=();
		// then this token is NOT done.
		if( Src[Pos] == '=' ){
			if( Pos>0 ){
				if( Src[Pos-1] == ')' )
					break;
			}
		}
		// the : represents a special case.  On its own, during inheritence, it ends a token
		// however when paired :: it represents scope.
		if( Src[Pos] == ':' ){
			if( (Src[Pos-1]!=':') && (Src[Pos+1]!=':') )
				return true;
		}
		Dest += Src[Pos];
		Pos++;
		if( Pos >= Src.length() ){
			wxMessageBox( _T("Failed to read a token"), _T("Verbose Error Info:") );
			return false;
		}
	}

	return true;
}

bool CKlassModelerDoc::StripWhiteSpaces( string &Src, int &Pos )
{
	while( (Src[Pos]==' ') || (Src[Pos]=='\n') || (Src[Pos]=='\t') ){
		Pos++;
		if( Pos >= Src.length() ){
			wxMessageBox( _T("failed at stripping white spaces"), _T("Verbose Error Info:") );
			return false;
		}
	}

	return true;
}

bool CKlassModelerDoc::FindWhiteSpaces( string &Src, int &Pos )
{
	while( (Src[Pos]!=' ') && (Src[Pos]!='\n') && (Src[Pos]!='\t') ){
		Pos++;
		if( Pos >= Src.length() ){
			wxMessageBox( _T("Failed at finding white spaces"), _T("Verbose Error Info:") );
			return false;
		}
	}

	return true;
}

bool CKlassModelerDoc::ReadVariableOrMethod( string &Src, int &Pos, AccessType Access, CClass &Class, bool bIsQtSignal, bool bIsQtSlot )
{
	int		Semi;
	int		Paren;
	int		Squigle;

	Semi = Src.find( ";", Pos );
	if( Semi == Src.npos ){
		wxMessageBox( _T("trying to read variable or method, but couldn't find a ;"), _T("Verbose Error Info:") );
		return false;
	}
	Paren = Src.find( "(", Pos );
	if( Paren == Src.npos )
		Paren = Semi+1;
	Squigle = Src.find( "{", Pos );
	if( Squigle != Src.npos ){
		// watch out for methods with bodies
		if( Squigle < Semi )
			return ReadMethod( Src, Pos, Access, Class, bIsQtSignal, bIsQtSlot );
	}
	if( Paren < Semi )
		return ReadMethod( Src, Pos, Access, Class, bIsQtSignal, bIsQtSlot );
	if( bIsQtSignal || bIsQtSlot ){
		wxMessageBox( _T("Looks like I found a variable that is trying to be a Qt signal or slot"), _T("Verbose Error Info:") );
		return false;					// can't have a variable that is a signal or slot
	}

	return ReadVariable( Src, Pos, Access, Class );
}

bool CKlassModelerDoc::ReadVariable( string &Src, int &Pos, AccessType Access, CClass &Class )
{
	int			Semi;
	string		Type;
	string		Name;		
	string		Token;
	int			PosHold;

	if( !StripWhiteSpaces( Src, Pos ) )
		return false;
	Semi = Src.find( ";", Pos );
	while( Pos<Semi ){
		PosHold = Pos;
		if( Token.length() > 0 ){
			if( Type.length() > 0 )
				Type += " ";
			Type += Token;
			Token = "";
		}

		if( !ReadToken( Src, Token, Pos ) )
			return false;
		if( !StripWhiteSpaces( Src, Pos ) )
			return false;

		if( PosHold == Pos ){
			wxMessageBox( _T("Exiting in ReadVariable to avoid infinite loop."), _T("Verbose Error") );
			return false;
		}
	}

	if( Token[0] == '*' ){
		Type += '*';
		Name = Token.substr( 1 );
	} else if( Token[0] == '&' ){
		Type += '&';
		Name = Token.substr( 1 );
	} else
		Name = Token;
	Pos = Semi+1;
	CVariable	*pNewVar = new CVariable( *this );
	pNewVar->SetAccessType( Access );
	pNewVar->SetName( Name );
	pNewVar->SetType( Type );
	Class.AddVariable( pNewVar );

	return true;
}

// on entry pos points to the first e in enum.  On exit it points
// to the first char after the semicolon.
bool CKlassModelerDoc::ReadEnum( string &Src, int &pos, AccessType Access, CClass &Class )
{
	int			Semi;
	string		Name;		
	string		EnumText;
	CEnum		*pEnum;

	Semi = Src.find( ";", pos );
	// pos now points to the word enum.
	pos += 4;	// get past the word enum
	if( !StripWhiteSpaces( Src, pos ) )
		return false;
	if( !ReadToken(Src, Name, pos ) )
		return false;
	if( !StripWhiteSpaces( Src, pos ) )
		return false;
	// pos now points to the first { in the enum
	EnumText = Src.substr( pos, Semi-pos+1 );
	// no save it all
	pEnum = new CEnum( *this );
	pEnum->SetName( Name );
	pEnum->SetText( EnumText );
	pEnum->SetAccessType( Access );
	Class.AddEnum( pEnum );
	pos = Semi+1;

	return true;
}

bool CKlassModelerDoc::ReadMethod( string &Src, int &Pos, AccessType Access, CClass &Class, bool bIsQtSignal, bool bIsQtSlot )
{
	int				Paren;
	int				Coma;
	int				PosHold;
	string			Type;
	string			Name;		
	string			Token;
	CMethod			*pMethod;
	CVariable		*pArg;

	// find name and return type first
	Paren = Src.find( "(", Pos );
	if( !StripWhiteSpaces( Src, Pos ) )
		return false;
	while( Pos<Paren ){
		PosHold = Pos;
		if( Token.length() > 0 ){
			if( Type.length() > 0 )
				Type += " ";
			Type += Token;
			Token = "";
		}
		if( !ReadToken( Src, Token, Pos ) )
			return false;
		if( !StripWhiteSpaces( Src, Pos ) )
			return false;
		if( PosHold == Pos ){
			wxMessageBox( _T("Exiting in ReadMethod to avoid infinite loop."), _T("Verbose Error") );
			return false;
		}
	}
	if( Token[0] == '*' ){
		Type += '*';
		Name = Token.substr( 1 );
	} else if( Token[0] == '&' ){
		Type += '&';
		Name = Token.substr( 1 );
	} else
		Name = Token;
	pMethod = new CMethod( *this );
	pMethod->SetAccessType( Access );
	pMethod->SetName( Name );
	pMethod->SetReturnType( Type );
	Pos = Paren;
	Token = "";
	// then work on variables
	Paren = Src.find( ")", Pos );
	if( Paren == Src.npos ){
		wxMessageBox( _T("Trying to read a method, but couldn't find a )"), _T("Verbose Error Info:") );
		return false;
	}
	Pos++;
	if( !StripWhiteSpaces( Src, Pos ) )
		return false;
	if( Pos != Paren ){
		// only do this part if there are arguments
		while( Pos<Paren ){
			PosHold = Pos;
			Type = "";
			Coma = Src.find( ",", Pos );
			if( (Coma == Src.npos) || (Coma>Paren) )
				Coma = Paren;
			if( !StripWhiteSpaces( Src, Pos ) )
				return false;
			while( Pos < Coma ){
				if( Token.length()>0 ){
					// watch out for default variables
					if( (Token[0] != '=') && ((Token[0]>'9')||(Token[0]<'0')) ){
						if( Type.length() > 0 )
							Type += " ";
						Type += Token;
					}
					Token = "";
				}
				if( !ReadToken( Src, Token, Pos ) )
					return false;
/*
				// special case, if you have an argument that is of a type from
				// another class, ie (CSomeClass::MyType	*pVariable)
				// then we need to catch that.
				if( (Src[Pos]==':') && (Src[Pos+1]==':' ) ){
					string	SecondPart;
					Pos += 2;
					if( !ReadToken( Src, SecondPart, Pos ) )
						return false;
					Token += "::" + SecondPart;
				}
*/
				if( !StripWhiteSpaces( Src, Pos ) )
					return false;
				if( Src[Pos] == '=' ){
					Token += Src.substr( Pos, Coma-Pos );
					Pos = Coma;
				}
			}
			if( Token[0] == '*' ){
				Type += '*';
				Name = Token.substr( 1 );
			} else if( Token[0] == '&' ){
				Type += '&';
				Name = Token.substr( 1 );
			} else
				Name = Token;
			pArg = new CVariable( *this );
			pArg->SetName( Name );
			pArg->SetType( Type );

			pMethod->AddArg( pArg );
			Token = "";
			Type = "";
			if( Src[Pos] == ',' )
				Pos++;
			if( PosHold == Pos ){
				wxMessageBox( _T("After one pass of trying to find a method there was no progress.  Exiting to avoid infinite loop"), _T("Verbose Error Info:") );
				return false;
			}
		}
	}
	Pos++;
	if( !StripWhiteSpaces( Src, Pos ) )
		return false;
	// at this point Pos is pointing to the first character after the )
	// finally, watch out for implementation, pure virtuals, or const functions
	if( Src[Pos] == '=' ){
		pMethod->SetPureVirtual( true );
		Pos = Src.find( ";", Pos );
		if( Pos == Src.npos ){
			wxMessageBox( _T("Looks like a pure virtual method, but no ; found"), _T("Verbose Error Info:") );
			return false;
		}
	} else if( Src[Pos] == '{' ){
		int		EndPos;
		string	InlineCode;

		EndPos = Src.find( "}", Pos );
		if( EndPos == Src.npos ){
			wxMessageBox( _T("Looks like there's an implementation but no } found"), _T("Verbose Error Info:") );
			return false;
		}
		InlineCode = Src.substr( Pos, EndPos-Pos+1 );	
		pMethod->SetInlineCode( InlineCode );
		Pos = EndPos;
	} else if( Src[Pos] == 'c' ){
		Pos = Src.find( ";", Pos );
		if( Pos == Src.npos ){
			wxMessageBox( _T("Looks like this is a const method, but couldn't find the ;"), _T("Verbose Error Info:") );
			return false;
		}
		pMethod->SetIsConst( true );
	}
	Pos++;
	pMethod->SetIsQtSignal( bIsQtSignal );
	pMethod->SetIsQtSlot( bIsQtSlot );
	Class.AddMethod( pMethod );

	return true;
}

wxOutputStream &CKlassModelerDoc::SaveObject( wxOutputStream &Stream )
{
	ClassVector::iterator			itVect;
	unsigned long					ClassCount;
	unsigned long					InheritenceCount;
	unsigned long					RelationCount;
	InheritenceVector				*pInheritence;
	InheritenceVector::iterator		itInheritence;
	RelationVector					*pRelation;
	RelationVector::iterator		itRelation;
	
	wxDocument::SaveObject( Stream );
	Stream.Write( DOC_VERSION_STRING, strlen(DOC_VERSION_STRING)+1 );
	ClassCount = m_Classes.size();
	Stream.Write( &ClassCount, sizeof(unsigned long) );
	InheritenceCount = 0;
	RelationCount = 0;
	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		if( !SaveClass( Stream, *itVect ) )
			return Stream;
		pInheritence = (*itVect)->GetInheritence();
		for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ++itInheritence )
			InheritenceCount++;
		pRelation = (*itVect)->GetRelation();
		for( itRelation=pRelation->begin(); itRelation!=pRelation->end(); ++itRelation )
			RelationCount++;
	}	
	Stream.Write( &InheritenceCount, sizeof(unsigned long) );
	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		pInheritence = (*itVect)->GetInheritence();
		for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ++itInheritence ){
			if( !SaveInheritence( Stream, (*itVect)->GetName(), *itInheritence) )
				return Stream;
		}
	}
	Stream.Write( &RelationCount, sizeof(unsigned long) );
	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		pRelation = (*itVect)->GetRelation();
		for( itRelation=pRelation->begin(); itRelation!=pRelation->end(); ++itRelation ){
			if( !SaveRelation( Stream, (*itVect)->GetName(), *itRelation) )
				return Stream;
		}
	}
	Modify( false );

	return Stream;
}

bool CKlassModelerDoc::SaveClass( wxOutputStream &Stream, CClass *pClass )
{
	float						Position;
	MethodVector::iterator		itMethod;
	VariableVector::iterator	itVariable;
	EnumVector::iterator		itEnum;
	unsigned long				Count;
	bool						bIsQObject;
	bool						bExclude;
	
	Stream.Write( pClass->GetName().c_str(), pClass->GetName().length()+1 );
	Stream.Write( pClass->GetDoc().c_str(), pClass->GetDoc().length()+1 );
	Position = pClass->GetX();
	Stream.Write( &Position, sizeof(float) );
	Position = pClass->GetY();
	Stream.Write( &Position, sizeof(float) );
	Stream.Write( pClass->GetHeaderFile().c_str(), pClass->GetHeaderFile().length()+1 );
	Stream.Write( pClass->GetHTMLFile().c_str(), pClass->GetHTMLFile().length()+1 );
	bIsQObject = pClass->IsQObject();
	Stream.Write( &bIsQObject, sizeof(bool) );
	bExclude = pClass->GetExcludeHeader();
	Stream.Write( &bExclude, sizeof(bool) );
	bExclude = pClass->GetExcludeHTML();
	Stream.Write( &bExclude, sizeof(bool) );
	// methods
	Count = pClass->GetMethods()->size();
	Stream.Write( &Count, sizeof(unsigned long) );
	for( itMethod=pClass->GetMethods()->begin(); itMethod!=pClass->GetMethods()->end(); ++itMethod ){
		if( !SaveMethod( Stream, *itMethod ) )
			return false;
	}
	// variables
	Count = pClass->GetVariables()->size();
	Stream.Write( &Count, sizeof(unsigned long) );
	for( itVariable=pClass->GetVariables()->begin(); itVariable!=pClass->GetVariables()->end(); ++itVariable ){
		if( !SaveVariable( Stream, *itVariable ) )
			return false;
	}
	// enums
	Count = pClass->GetEnums()->size();
	Stream.Write( &Count, sizeof(unsigned long) );
	for( itEnum=pClass->GetEnums()->begin(); itEnum!=pClass->GetEnums()->end(); ++itEnum ){
		if( !SaveEnum( Stream, *itEnum ) )
			return false;
	}
	
	return true;
}

bool CKlassModelerDoc::SaveMethod( wxOutputStream &Stream, CMethod *pMethod )
{
	AccessType					Type;
	VariableVector::iterator	itVect;
	unsigned long				Count;
	bool						bIsPureVirtual;
	bool						bIsSignal;
	bool						bIsSlot;
	bool						bIsConst;
	
	Stream.Write( pMethod->GetName().c_str(), pMethod->GetName().length()+1 );
	Stream.Write( pMethod->GetReturnType().c_str(), pMethod->GetReturnType().length()+1 );
	Stream.Write( pMethod->GetDoc().c_str(), pMethod->GetDoc().length()+1 );
	Type = pMethod->GetAccess();
	Stream.Write( &Type, sizeof(AccessType) );
	bIsPureVirtual = pMethod->IsPureVirtual();
	Stream.Write( &bIsPureVirtual, sizeof(bool) );
	bIsSignal = pMethod->IsQtSignal();
	bIsSlot = pMethod->IsQtSlot();
	Stream.Write( &bIsSignal, sizeof(bool) );
	Stream.Write( &bIsSlot, sizeof(bool) );
	Stream.Write( pMethod->GetInlineCode().c_str(), pMethod->GetInlineCode().length()+1 );
	bIsConst = pMethod->IsConst();
	Stream.Write( &bIsConst, sizeof(bool) );
	Count = pMethod->GetArgs()->size();
	Stream.Write( &Count, sizeof(unsigned long) );
	for( itVect=pMethod->GetArgs()->begin(); itVect!=pMethod->GetArgs()->end(); ++itVect ){
		if( !SaveVariable( Stream, *itVect ) )
			return false;
	}
	
	return true;
}

bool CKlassModelerDoc::SaveVariable( wxOutputStream &Stream, CVariable *pVariable )
{
	AccessType		Type;
	
	Stream.Write( pVariable->GetName().c_str(), pVariable->GetName().length()+1 );
	Stream.Write( pVariable->GetType().c_str(), pVariable->GetType().length()+1 );
	Stream.Write( pVariable->GetDoc().c_str(), pVariable->GetDoc().length()+1 );
	Type = pVariable->GetAccess();
	Stream.Write( &Type, sizeof(AccessType) );
	
	return true;
}

bool CKlassModelerDoc::SaveEnum( wxOutputStream &Stream, CEnum *pEnum )
{
	AccessType		Type;

	Stream.Write( pEnum->GetName().c_str(), pEnum->GetName().length()+1 );
	Stream.Write( pEnum->GetText().c_str(), pEnum->GetText().length()+1 );
	Type = pEnum->GetAccess();
	Stream.Write( &Type, sizeof(AccessType) );

	return true;
}

bool CKlassModelerDoc::SaveInheritence( wxOutputStream &Stream, string ChildName, CInheritence *pInheritence )
{
   float Horz;

   Stream.Write( pInheritence->GetParentName().c_str(), pInheritence->GetParentName().length()+1 );
   Stream.Write( ChildName.c_str(),ChildName.length()+1 );
   Horz = pInheritence->GetHorizontal();
   Stream.Write( &Horz, sizeof(float) );
	
   return true;
}

bool CKlassModelerDoc::SaveRelation( wxOutputStream &Stream, string ParentName, CRelation *pRelation )
{
	bool	bIsHasA;

	Stream.Write( ParentName.c_str(), ParentName.length()+1 );
	Stream.Write( pRelation->GetChild()->GetName().c_str(), pRelation->GetChild()->GetName().length()+1 );
	bIsHasA = pRelation->IsHasA();
	Stream.Write( &bIsHasA, sizeof(bool) );

	return true;
}

// this function returns the base path leading to the
// opened file name.  This is useful for doc and header
// generation where the paths in the model are relative 
// to the path of the model.
string CKlassModelerDoc::GetBasePath()
{
	string		FileName;
	string		FullPath;
	char		Seperator;

#ifdef __WXMSW__
	Seperator = '\\';
#else
	Seperator = '/';
#endif
	FileName = GetFilename();
	if( FileName.substr( 0, 7 ) == "unnamed" ){
		FullPath = wxGetCwd();
		FullPath += Seperator;
		return FullPath;
	}
	if( (FileName[1] != ':') && (FileName[0] != Seperator) ){
		FullPath = wxGetCwd() + Seperator;
		FullPath += FileName;
	} else
		FullPath = FileName;

	FullPath = wxPathOnly( FullPath.c_str() ); 
	FullPath += Seperator;

	return FullPath;
}

bool CKlassModelerDoc::GenerateCode()
{
	ClassVector::iterator	itVect;
	string					ErrMsg;

	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		if( !GenerateCodeFor( *itVect, true ) ){
			ErrMsg = "Error creating header for ";
			ErrMsg += (*itVect)->GetName();
			ErrMsg += ".  Stopping code generation";
			wxMessageBox( wxString((wxChar*)ErrMsg.c_str()), _T("Error") );
			return false;
		}
	}
	
	return true;
}

bool CKlassModelerDoc::GenerateCodeFor( CClass *pClass, bool bFullModel )
{
	if( bFullModel && pClass->GetExcludeHeader() )
		return true;
	int		fd;
	FILE	*pFile;
	string	Header;
	string	FullPath;
	string	JustPath;

	Header = pClass->GetHeaderFile();
	if( (Header[1]==':') || (Header[0]=='/') || (Header[0]=='\\') )
		FullPath = Header;
	else 
		FullPath = GetBasePath() + pClass->GetHeaderFile();
	JustPath = wxPathOnly( FullPath.c_str() );
	if( !DirectoryExists( FullPath ) ){
		if( !CreateDirStructure( JustPath ) )
			return false;
	}
	fd = open( FullPath.c_str(), O_CREAT | O_EXCL, 0666 );
	if( fd == -1 ){
		if( errno != EEXIST ){
			return false;
		}
		// update file
		fd = open( FullPath.c_str(), O_RDWR );
		if( fd == -1 ){
			return false;
		}
		return AlterExistingHeader( fd, pClass, FullPath );
	} else {
		close( fd );
		pFile = fopen( FullPath.c_str(), "w" );
		if( pFile == NULL ){
			return false;
		}
		return WriteNewHeader( pFile, pClass );
	}
}

// there's a bug in wxFileExists (or at least in the docs for
// it) where it won't return true if the string is a directory.
// so I'm copying out the code directly and changing it.
bool CKlassModelerDoc::DirectoryExists( string FullPath )
{
#ifdef __WXMSW__
	return ::GetFileAttributes(FullPath.c_str()) != ((DWORD)-1);
#else
    wxStructStat st;
	return wxStat(FullPath.c_str(), &st) == 0;
#endif
	return false;
}

bool CKlassModelerDoc::CreateDirStructure( string FullPath )
{
	if( FullPath.empty() )
		return true;
	if( DirectoryExists( FullPath ) )
		return true;

	string	Parent;
	Parent = wxPathOnly( FullPath.c_str() );
	if( !CreateDirStructure( Parent ) )
		return false;

	return wxMkdir( FullPath.c_str() );
}

bool CKlassModelerDoc::WriteNewHeader( FILE *pFile, CClass *pClass )
{
	string		HeaderDef;
	int			Pos;
	string		DefUpper;
	
	HeaderDef = pClass->GetHeaderFile();
	Pos = HeaderDef.rfind( '/' );
	if( Pos != HeaderDef.npos )
		HeaderDef = HeaderDef.substr( Pos+1 );
	else {
		// what if they're using the \ seperator?
		Pos = HeaderDef.rfind( '\\' );
		if( Pos != HeaderDef.npos )
			HeaderDef = HeaderDef.substr( Pos+1 );
	}
	DefUpper = HeaderDef;
	transform( DefUpper.begin(), DefUpper.end(), DefUpper.begin(), toupper );
	HeaderDef = DefUpper;
	HeaderDef[HeaderDef.length()-2] = '_';
		
	fprintf( pFile, "#ifndef _%s_\n#define _%s_\n\n", HeaderDef.c_str(), HeaderDef.c_str() );	
	if( !WriteClass( pFile, pClass ) ){
		fclose( pFile );
		return false;
	}
	fprintf( pFile, "\n#endif\n" );
	fclose( pFile );
	
	return true;
}

bool CKlassModelerDoc::WriteClass( FILE *pFile, CClass *pClass )
{
	MethodVector::iterator		itMethods;
	MethodVector				*pMethods;
	VariableVector::iterator	itVariables;
	VariableVector				*pVariables;
	EnumVector::iterator		itEnums;
	EnumVector					*pEnums;
	InheritenceVector::iterator itInheritence;
	InheritenceVector           *pInheritence;
	bool						bSignalSlot;
	
	fprintf( pFile, "class %s", pClass->GetName().c_str() );
	pInheritence = pClass->GetInheritence();
	itInheritence = pInheritence->begin();
	if( itInheritence!=pInheritence->end() ){
		fprintf( pFile, " : public %s", (*itInheritence)->GetParentName().c_str() );
		++itInheritence;
		for( ; itInheritence!=pInheritence->end(); ++itInheritence ){
			fprintf( pFile, ",\n\tpublic %s", (*itInheritence)->GetParentName().c_str() );
		}
	}
	fprintf( pFile, "\n{\n" );
	if( pClass->IsQObject() )
		fprintf( pFile, "Q_OBJECT\n" );
	pMethods = pClass->GetMethods();
	pVariables = pClass->GetVariables();
	pEnums = pClass->GetEnums();
	// first do all the public methods and variables
	fprintf( pFile, "public:\n" );
	for( itEnums=pEnums->begin(); itEnums!=pEnums->end(); ++itEnums ){
		if( (*itEnums)->GetAccess() == Public )
			fprintf( pFile, "\tenum %s %s\n", (*itEnums)->GetName().c_str(), (*itEnums)->GetText().c_str() );
	}
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( ((*itMethods)->GetAccess() == Public) && !((*itMethods)->IsQtSignal() || (*itMethods)->IsQtSlot()) )
			WriteMethod( pFile, *itMethods );
	}
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Public )
			fprintf( pFile, "\t%s;\n", (*itVariables)->GetFullText().c_str() );
	}
	// next do all the public signals and slots
	if( pClass->IsQObject() ){
		bSignalSlot = false;
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Public) && (*itMethods)->IsQtSlot() ){
				if( !bSignalSlot ){
					fprintf( pFile, "public slots:\n" );
					bSignalSlot = true;
				}
				WriteMethod( pFile, *itMethods );
			}
		}
		bSignalSlot = false;
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Public) && (*itMethods)->IsQtSignal() ){
				if( !bSignalSlot ){
					fprintf( pFile, "public signals:\n" );
					bSignalSlot = true;
				}
 				WriteMethod( pFile, *itMethods );
			}
		}
	}
	// next do all the protected methods and variables
	fprintf( pFile, "protected:\n" );
	for( itEnums=pEnums->begin(); itEnums!=pEnums->end(); ++itEnums ){
		if( (*itEnums)->GetAccess() == Protected )
			fprintf( pFile, "\tenum %s %s\n", (*itEnums)->GetName().c_str(), (*itEnums)->GetText().c_str() );
	}
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( ((*itMethods)->GetAccess() == Protected) && !((*itMethods)->IsQtSignal() || (*itMethods)->IsQtSlot())  )
			WriteMethod( pFile, *itMethods );
	}
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Protected )
			fprintf( pFile, "\t%s;\n", (*itVariables)->GetFullText().c_str() );
	}
	// next do all the protected method slots and signals
	if( pClass->IsQObject() ){
		bSignalSlot = false;
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Protected) && (*itMethods)->IsQtSlot() ){
				if( !bSignalSlot ){
					fprintf( pFile, "protected slots:\n" );
					bSignalSlot = true;
				}
				WriteMethod( pFile, *itMethods );
			}
		}
		bSignalSlot = false;
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Protected) && (*itMethods)->IsQtSignal() ){
				if( !bSignalSlot ){

					fprintf( pFile, "protected signals:\n" );
					bSignalSlot = true;
				}
 				WriteMethod( pFile, *itMethods );
			}
		}
	}
	// next do all the private methods
	fprintf( pFile, "private:\n" );
	for( itEnums=pEnums->begin(); itEnums!=pEnums->end(); ++itEnums ){
		if( (*itEnums)->GetAccess() == Private )
			fprintf( pFile, "\tenum %s %s\n", (*itEnums)->GetName().c_str(), (*itEnums)->GetText().c_str() );
	}
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( ((*itMethods)->GetAccess() == Private) && !((*itMethods)->IsQtSignal() || (*itMethods)->IsQtSlot())  )
			WriteMethod( pFile, *itMethods );
	}
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Private )
			fprintf( pFile, "\t%s;\n", (*itVariables)->GetFullText().c_str() );
	}
	// next do all the private signals and slots
	if( pClass->IsQObject() ){
		bSignalSlot = false;
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Private) && (*itMethods)->IsQtSlot() ){
				if( !bSignalSlot ){
					fprintf( pFile, "private slots:\n" );
					bSignalSlot = true;
				}
				WriteMethod( pFile, *itMethods );
			}
		}
		bSignalSlot = false;
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Private) && (*itMethods)->IsQtSignal() ){
				if( !bSignalSlot ){
					fprintf( pFile, "private signals:\n" );
					bSignalSlot = true;
				}
 				WriteMethod( pFile, *itMethods );
			}
		}
	}
	fprintf( pFile, "};\n" );

	
	return true;
}

void CKlassModelerDoc::WriteMethod( FILE *pFile, CMethod *pMethod )
{
	fprintf( pFile, "\t%s", pMethod->GetFullText().c_str() );
	if( pMethod->GetInlineCode() != "" )
		fprintf( pFile, "%s\n", pMethod->GetInlineCode().c_str() );
	else
		fprintf( pFile, ";\n" );
}

bool CKlassModelerDoc::AlterExistingHeader( int fd, CClass *pClass, string FullPath )
{
	string		StrLine;
	string		ClassHeader;
	int			Pos;
	FILE		*pTempFile;
	string		ClassFooter;
	bool		bReadingClass = false;
			
	pTempFile = fopen( "KlassModelerTempHeader", "w" );
	ClassHeader = "class ";
	ClassHeader += pClass->GetName();
	ClassFooter = "};";
	while( ReadLine(fd, StrLine) ){
		Pos = StrLine.find( ClassHeader );
		if( (Pos != StrLine.npos) && ((StrLine[Pos+ClassHeader.length()] == ' ')||(StrLine[Pos+ClassHeader.length()] == '\n')||(StrLine[Pos+ClassHeader.length()] == '\r')) ){
			bReadingClass = true;
			if( !WriteClass( pTempFile, pClass ) ){
				close( fd );
				fclose( pTempFile );
				return false;
			}
		} else {
		  	if( bReadingClass ){
		  		Pos = StrLine.find( ClassFooter );
				if( Pos != StrLine.npos )
	  				bReadingClass = false;
		  	} else {
				fprintf( pTempFile, "%s", StrLine.c_str() );
			}
		}
		StrLine = "";
	}
	close( fd );
	fclose( pTempFile );
#ifdef WIN32
	if( !MoveFileEx( "KlassModelerTempHeader", FullPath.c_str(), MOVEFILE_REPLACE_EXISTING ) ){
		wxMessageBox( _T("Failed to move temporary header to new header.  Perhaps you have it opened and locked?"), _T("Error") );
		return false;
	}
#else
	if( remove( FullPath.c_str() ) ){
		char	ErrMsg[512];
		sprintf( ErrMsg, "Failed to delete %s.  Perhaps you have it open and locked?", FullPath.c_str() );
		wxMessageBox( ErrMsg, _T("Error!") );
		return false;
	}
	if( rename( "KlassModelerTempHeader", FullPath.c_str() ) ){
		wxMessageBox( _T("Failed to rename temporary header to new header.  Your header may be deleted.  Please see KlassModelerTempHeader in current working dircetory for a backup"), _T("Big time error!") );
		return false;
	}
#endif
	
	return true;
}

bool CKlassModelerDoc::ReadLine( int fd, string &StrLine )
{
	char	SingleChar;
	
	do {
		if( read( fd, &SingleChar, 1 ) == 0 )
	  		return false;
		StrLine += SingleChar;
	} while( SingleChar != '\n' );
	
	return true;
}

bool CKlassModelerDoc::GenerateHTML( CModelCanvas *pCanvas )
{
	ClassVector::iterator	itVect;
	FILE					*pFile;

	// create the class image
	if( !pCanvas->GenerateJPGImage( (GetBasePath()+"KlassModelerImage.png").c_str(), wxBITMAP_TYPE_PNG ) ){
		MasterLog.Log( LVL_HIGH, "CKlassModelerDoc::GenerateHTML failed to create PNG\n" );
		return false;
	}
	// create the image html file
	pFile = fopen( (GetBasePath()+"KlassModelerImage.html").c_str(), "w" );
	if( pFile == NULL ){
		MasterLog.Log( LVL_HIGH, "CKlassModelerDoc::GenerateHTML failed to open KlassModelerImage.html\n" );
		return false;
	}
	fprintf( pFile, "<html><body><img src=\"KlassModelerImage.png\"><hr>Generated by the <a href=\"http://www.jameswells.net\" target=\"top\">KlassModeler</a> by <a href=\"mailto:james@wells.net\">James Wells</a>.</body></html>" );
	fclose( pFile );
	// create the index.html file
	pFile = fopen( (GetBasePath()+"index.html").c_str(), "w" );
	if( pFile == NULL ){
		MasterLog.Log( LVL_HIGH, "CKlassModelerDoc::GenerateHTML failed to create index.html\n" );
		return false;
	}
	fprintf( pFile, "<html><frameset cols=\"20%%,80%%\">\n" );
	fprintf( pFile, "<frame src=\"KlassModelerMenu.html\" name=\"MenuFrame\">\n" );
	fprintf( pFile, "<frame src=\"KlassModelerImage.html\" name=\"ContentFrame\">\n" );
	fprintf( pFile, "</frameset></html>" );
	fclose( pFile );
	// reorder the class vector
	sort( m_Classes.begin(), m_Classes.end(), ClassVectorLess() );
	// now create the menu file
	pFile = fopen( (GetBasePath()+"KlassModelerMenu.html").c_str(), "w" );
	fprintf( pFile, "<a href=\"KlassModelerImage.html\" target=\"ContentFrame\">KlassModeler Image</a>\n" );
	fprintf( pFile, "<html>\n<head>\n<title>Class List</title>\n</head>\n<body>\n<ul>\n" );
	for( itVect=m_Classes.begin(); itVect!=m_Classes.end(); ++itVect ){
		if( !(*itVect)->GetExcludeHTML() ){
			fprintf( pFile, "<li><a href=\"%s\" target=\"ContentFrame\">%s</a>\n", (*itVect)->GetHTMLFile().c_str(), (*itVect)->GetName().c_str() );
			if( !GenerateHTMLFor( *itVect ) ){
				string	ErrMsg;
				ErrMsg = "Error generating html for ";
				ErrMsg += (*itVect)->GetName();
				wxMessageBox( wxString((wxChar*)ErrMsg.c_str()), _T("Error") );
				MasterLog.Log( LVL_HIGH, "CKlassModelerDoc::GenerateHTML failed to generate HTML for %s\n", (*itVect)->GetName().c_str() );
				return false;
			}
		}
	}
	fprintf( pFile, "</ul>\n<hr>Generated by the <a href=\"http://www.jameswells.net\" target=\"top\">KlassModeler</a> by <a href=\"mailto:james@wells.net\">James Wells</a>.</body></html>" );
	fclose( pFile );

	return true;
}

bool CKlassModelerDoc::GenerateHTMLFor( CClass *pClass )
{
	FILE								*pFile;
	MethodVector::iterator			itMethods;
	MethodVector					*pMethods;
	VariableVector::iterator		itVariables;
	VariableVector					*pVariables;
	InheritenceVector				*pInheritence;

	InheritenceVector::iterator		itInheritence;
	bool							bFirstParent;
	CClass							*pParent;
	string							FullPath;
	string							JustPath;

	FullPath = GetBasePath() + pClass->GetHTMLFile();
	JustPath = wxPathOnly( FullPath.c_str() );
	if( !DirectoryExists( JustPath ) ){
		if( !CreateDirStructure( JustPath ) )
			return false;
	}
	pFile = fopen( FullPath.c_str(), "w" );
	if( pFile == NULL )
		return false;
	fprintf( pFile, "<html>\n<head>\n<title>%s</title>\n</head>\n<body>\n", pClass->GetName().c_str() );
	fprintf( pFile, "<center><h1>%s</h1></center><br>\n", pClass->GetName().c_str() );
	if( pClass->GetDoc().length() > 0 )
		fprintf( pFile, "%s<br><br>\n", pClass->GetDoc().c_str() );
	fprintf( pFile, "<b>class %s</b>", pClass->GetName().c_str() );
	pInheritence = pClass->GetInheritence();
	bFirstParent = true;
	for( itInheritence=pInheritence->begin(); itInheritence!=pInheritence->end(); ++itInheritence ){
		pParent = (*itInheritence)->GetParent();
		if( bFirstParent ){
			bFirstParent = false;
			fprintf( pFile, "<br>\n&nbsp;&nbsp;&nbsp;: public " );
		} else 
			fprintf( pFile, "<br>\n&nbsp;&nbsp;&nbsp;, public " );
		if( pParent )
			fprintf( pFile, "<a href=\"%s\" target=\"ContentFrame\">%s</a>\n", pParent->GetHTMLFile().c_str(), (*itInheritence)->GetParentName().c_str() );
		else
			fprintf( pFile, "%s\n", (*itInheritence)->GetParentName().c_str() );
	}
	fprintf( pFile, "<br>\n{<br>\n" );
	if( pClass->IsQObject() )
		fprintf( pFile, "Q_OBJECT<br>\n" );
	pMethods = pClass->GetMethods();
	pVariables = pClass->GetVariables();
	// public methods
	fprintf( pFile, "<b>public:</b><br>\n" );
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( ((*itMethods)->GetAccess() == Public) && !((*itMethods)->IsQtSignal() || (*itMethods)->IsQtSlot()) )
			fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
	}
	// public variables
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Public )
			fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itVariables)->GetFullText()).c_str(), (*itVariables)->GetFullText().c_str() );
	}
	// public signals and slots
	if( pClass->IsQObject() ){
		fprintf( pFile, "<b>public signals:</b><br>\n" );
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Public) && (*itMethods)->IsQtSignal() )
				fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
		}
		fprintf( pFile, "<b>public slots:</b><br>\n" );
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Public) && (*itMethods)->IsQtSlot() )
				fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
		}
	}
	// protected methods
	fprintf( pFile, "<b>protected:</b><br>\n" );
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( ((*itMethods)->GetAccess() == Protected) && !((*itMethods)->IsQtSignal() || (*itMethods)->IsQtSlot()) )
			fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
	}
	// protected variables
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Protected )
			fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itVariables)->GetFullText()).c_str(), (*itVariables)->GetFullText().c_str() );
	}
	// protected signals and slots
	if( pClass->IsQObject() ){
		fprintf( pFile, "<b>protected signals:</b><br>\n" );
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Protected) && (*itMethods)->IsQtSignal() )
				fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
		}
		fprintf( pFile, "<b>protected slots:</b><br>\n" );
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Protected) && (*itMethods)->IsQtSlot() )
				fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
		}
	}
	// private methods
	fprintf( pFile, "<b>private:</b><br>\n" );
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		if( ((*itMethods)->GetAccess() == Private) && !((*itMethods)->IsQtSignal() || (*itMethods)->IsQtSlot()) )
			fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
	}
	// private variables
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		if( (*itVariables)->GetAccess() == Private )
			fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itVariables)->GetFullText()).c_str(), (*itVariables)->GetFullText().c_str() );
	}
	// private signals and slots
	if( pClass->IsQObject() ){
		fprintf( pFile, "<b>private signals:</b><br>\n" );
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Private) && (*itMethods)->IsQtSignal() )
				fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
		}
		fprintf( pFile, "<b>private slots:</b><br>\n" );
		for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
			if( ((*itMethods)->GetAccess() == Private) && (*itMethods)->IsQtSlot() )
				fprintf( pFile, "&nbsp;&nbsp;&nbsp;<a href=\"#%s\">%s;</a><br>\n", MangleForHTMLName((*itMethods)->GetFullText()).c_str(), (*itMethods)->GetFullText().c_str() );
		}
	}
	fprintf( pFile, "}<br>\n" );
	// now do the actual docs
	fprintf( pFile, "<hr><h2>Methods:</h2>\n" );
	for( itMethods=pMethods->begin(); itMethods!=pMethods->end(); ++itMethods ){
		WriteMethodHTML( pFile, *itMethods );
	}
	fprintf( pFile, "<hr><h2>Members:</h2>\n" );
	for( itVariables=pVariables->begin(); itVariables!=pVariables->end(); ++itVariables ){
		WriteVariableHTML( pFile, *itVariables );
	}
	fprintf( pFile, "<hr>Generated by the <a href=\"http://www.jameswells.net\" target=\"top\">KlassModeler</a> by <a href=\"mailto:james@wells.net\">James Wells</a>.\n" );
	fprintf( pFile, "</body>\n</html>\n" );
	fclose( pFile );

	return true;
}

bool CKlassModelerDoc::WriteMethodHTML( FILE *pFile, CMethod *pMethod )
{
	VariableVector::iterator	itVect;
	VariableVector				*pArgs;

	fprintf( pFile, "<b><a name=\"%s\">%s</a></b><br>\n", MangleForHTMLName(pMethod->GetFullText().c_str()).c_str(), pMethod->GetFullText().c_str() );
	if( pMethod->GetDoc().length() > 0 )
		fprintf( pFile, "%s<br>\n", pMethod->GetDoc().c_str() );
	pArgs = pMethod->GetArgs();
	fprintf( pFile, "<ul>\n" );
	for( itVect=pArgs->begin(); itVect!=pArgs->end(); ++itVect ){
		if( (*itVect)->GetDoc().length() > 0 )
			fprintf( pFile, "<li><i>%s</i> - %s</br>\n", (*itVect)->GetName().c_str(), (*itVect)->GetDoc().c_str() );
	}
	fprintf( pFile, "</ul>\n<br>\n" );

	return true;
}

bool CKlassModelerDoc::WriteVariableHTML( FILE *pFile, CVariable *pVariable )

{
	fprintf( pFile, "<b><a name=\"%s\">%s</a></b><br>\n", MangleForHTMLName(pVariable->GetFullText()).c_str(), pVariable->GetFullText().c_str() );
	if( pVariable->GetDoc().length()>0 )
		fprintf( pFile, "%s<br>\n", pVariable->GetDoc().c_str() );
	fprintf( pFile, "<br>" );

	return true;
}

string CKlassModelerDoc::MangleForHTMLName( const string &Src )
{
	string Output;

	for( int i=0; i<Src.length(); i++ ){
		if( (Src[i]!=' ') && (Src[i]!='\t')&& (Src[i]!='\n')&& (Src[i]!='=')&& (Src[i]!=',') )
			Output += Src[i];
	}

	return Output;
}

void CKlassModelerDoc::AddNewType( string Type )
{
	StringVector::iterator	itVect;

	for( itVect=m_Types.begin(); itVect!=m_Types.end(); ++itVect ){
		if( *itVect == Type )
			return;
	}
	m_Types.push_back( Type );
}

