#
# @file    TestKineticLaw.rb
# @brief   SBML KineticLaw unit tests
#
# @author  Akiya Jouraku (Ruby conversion)
# @author  Ben Bornstein 
#
# $Id: TestKineticLaw.rb 10068 2009-08-21 18:54:20Z ajouraku $
# $HeadURL: https://sbml.svn.sourceforge.net/svnroot/sbml/branches/libsbml-4/src/bindings/ruby/test/sbml/TestKineticLaw.rb $
#
# This test file was converted from src/sbml/test/TestKineticLaw.c
# with the help of conversion sciprt (ctest_converter.pl).
#
#<!---------------------------------------------------------------------------
# This file is part of libSBML.  Please visit http://sbml.org for more
# information about SBML, and the latest version of libSBML.
#
# Copyright 2005-2009 California Institute of Technology.
# Copyright 2002-2005 California Institute of Technology and
#                     Japan Science and Technology Corporation.
# 
# This library is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation.  A copy of the license agreement is provided
# in the file named "LICENSE.txt" included with this software distribution
# and also available online as http://sbml.org/software/libsbml/license.html
#--------------------------------------------------------------------------->*/
require 'test/unit'
require 'libSBML'

class TestKineticLaw < Test::Unit::TestCase

  def setup
    @@kl = LibSBML::KineticLaw.new(2,4)
    if (@@kl == nil)
    end
  end

  def teardown
    @@kl = nil
  end

  def test_KineticLaw_addParameter
    p = LibSBML::Parameter.new(2,4)
    p.setId( "p")
    @@kl.addParameter(p)
    assert( @@kl.getNumParameters() == 1 )
    p = nil
  end

  def test_KineticLaw_create
    assert( @@kl.getTypeCode() == LibSBML::SBML_KINETIC_LAW )
    assert( @@kl.getMetaId() == "" )
    assert( @@kl.getNotes() == nil )
    assert( @@kl.getAnnotation() == nil )
    assert( @@kl.getFormula() == "" )
    assert( @@kl.getMath() == nil )
    assert( @@kl.getTimeUnits() == "" )
    assert( @@kl.getSubstanceUnits() == "" )
    assert_equal false, @@kl.isSetFormula()
    assert_equal false, @@kl.isSetMath()
    assert_equal false, @@kl.isSetTimeUnits()
    assert_equal false, @@kl.isSetSubstanceUnits()
    assert( @@kl.getNumParameters() == 0 )
  end

  def test_KineticLaw_createWithNS
    xmlns = LibSBML::XMLNamespaces.new()
    xmlns.add( "http://www.sbml.org", "testsbml")
    sbmlns = LibSBML::SBMLNamespaces.new(2,1)
    sbmlns.addNamespaces(xmlns)
    object = LibSBML::KineticLaw.new(sbmlns)
    assert( object.getTypeCode() == LibSBML::SBML_KINETIC_LAW )
    assert( object.getMetaId() == "" )
    assert( object.getNotes() == nil )
    assert( object.getAnnotation() == nil )
    assert( object.getLevel() == 2 )
    assert( object.getVersion() == 1 )
    assert( object.getNamespaces() != nil )
    assert( object.getNamespaces().getLength() == 2 )
    object = nil
  end

  def test_KineticLaw_free_NULL
  end

  def test_KineticLaw_getParameter
    k1 = LibSBML::Parameter.new(2,4)
    k2 = LibSBML::Parameter.new(2,4)
    k1.setId( "k1")
    k2.setId( "k2")
    k1.setValue(3.14)
    k2.setValue(2.72)
    @@kl.addParameter(k1)
    @@kl.addParameter(k2)
    k1 = nil
    k2 = nil
    assert( @@kl.getNumParameters() == 2 )
    k1 = @@kl.getParameter(0)
    k2 = @@kl.getParameter(1)
    assert ((  "k1" == k1.getId() ))
    assert ((  "k2" == k2.getId() ))
    assert( k1.getValue() == 3.14 )
    assert( k2.getValue() == 2.72 )
  end

  def test_KineticLaw_getParameterById
    k1 = LibSBML::Parameter.new(2,4)
    k2 = LibSBML::Parameter.new(2,4)
    k1.setId( "k1")
    k2.setId( "k2")
    k1.setValue(3.14)
    k2.setValue(2.72)
    @@kl.addParameter(k1)
    @@kl.addParameter(k2)
    k1 = nil
    k2 = nil
    assert( @@kl.getNumParameters() == 2 )
    k1 = @@kl.getParameter( "k1")
    k2 = @@kl.getParameter( "k2")
    assert ((  "k1" == k1.getId() ))
    assert ((  "k2" == k2.getId() ))
    assert( k1.getValue() == 3.14 )
    assert( k2.getValue() == 2.72 )
  end

  def test_KineticLaw_removeParameter
    o1 = @@kl.createParameter()
    o2 = @@kl.createParameter()
    o3 = @@kl.createParameter()
    o3.setId("test")
    assert( @@kl.removeParameter(0) == o1 )
    assert( @@kl.getNumParameters() == 2 )
    assert( @@kl.removeParameter(0) == o2 )
    assert( @@kl.getNumParameters() == 1 )
    assert( @@kl.removeParameter("test") == o3 )
    assert( @@kl.getNumParameters() == 0 )
    o1 = nil
    o2 = nil
    o3 = nil
  end

  def test_KineticLaw_setBadFormula
    formula =  "k1 X0";
    @@kl.setFormula(formula)
    assert_equal false, @@kl.isSetFormula()
    assert_equal false, @@kl.isSetMath()
  end

  def test_KineticLaw_setFormula
    formula =  "k1*X0";
    @@kl.setFormula(formula)
    assert (( formula == @@kl.getFormula() ))
    assert_equal true, @@kl.isSetFormula()
    if (@@kl.getFormula() == formula)
    end
    @@kl.setFormula(@@kl.getFormula())
    assert (( formula == @@kl.getFormula() ))
    @@kl.setFormula("")
    assert_equal false, @@kl.isSetFormula()
    if (@@kl.getFormula() != nil)
    end
  end

  def test_KineticLaw_setFormulaFromMath
    math = LibSBML::parseFormula("k1 * X0")
    assert_equal false, @@kl.isSetMath()
    assert_equal false, @@kl.isSetFormula()
    @@kl.setMath(math)
    assert_equal true, @@kl.isSetMath()
    assert_equal true, @@kl.isSetFormula()
    assert ((  "k1 * X0" == @@kl.getFormula() ))
    math = nil
  end

  def test_KineticLaw_setMath
    math = LibSBML::parseFormula("k3 / k2")
    @@kl.setMath(math)
    math1 = @@kl.getMath()
    assert( math1 != nil )
    formula = LibSBML::formulaToString(math1)
    assert( formula != nil )
    assert ((  "k3 / k2" == formula ))
    assert( @@kl.getMath() != math )
    assert_equal true, @@kl.isSetMath()
    @@kl.setMath(@@kl.getMath())
    math1 = @@kl.getMath()
    assert( math1 != nil )
    formula = LibSBML::formulaToString(math1)
    assert( formula != nil )
    assert ((  "k3 / k2" == formula ))
    assert( @@kl.getMath() != math )
    @@kl.setMath(nil)
    assert_equal false, @@kl.isSetMath()
    if (@@kl.getMath() != nil)
    end
    math = nil
  end

  def test_KineticLaw_setMathFromFormula
    formula =  "k3 / k2";
    assert_equal false, @@kl.isSetMath()
    assert_equal false, @@kl.isSetFormula()
    @@kl.setFormula(formula)
    assert_equal true, @@kl.isSetMath()
    assert_equal true, @@kl.isSetFormula()
    formula = LibSBML::formulaToString(@@kl.getMath())
    assert ((  "k3 / k2" == formula ))
  end

end
