/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.memory;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.common.collect.Maps;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.engine.EngineClosedException;
import org.elasticsearch.index.engine.FlushNotAllowedEngineException;
import org.elasticsearch.index.service.IndexService;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.service.IndexShard;
import org.elasticsearch.index.shard.service.InternalIndexShard;
import org.elasticsearch.index.translog.Translog;
import org.elasticsearch.indices.IndicesLifecycle;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.threadpool.ThreadPool;

public class IndexingMemoryController
extends AbstractLifecycleComponent<IndexingMemoryController> {
    private final ThreadPool threadPool;
    private final IndicesService indicesService;
    private final ByteSizeValue indexingBuffer;
    private final ByteSizeValue minShardIndexBufferSize;
    private final ByteSizeValue maxShardIndexBufferSize;
    private final TimeValue inactiveTime;
    private final TimeValue interval;
    private final Listener listener = new Listener();
    private final Map<ShardId, ShardIndexingStatus> shardsIndicesStatus = Maps.newHashMap();
    private volatile ScheduledFuture scheduler;
    private final Object mutex = new Object();

    @Inject
    public IndexingMemoryController(Settings settings, ThreadPool threadPool, IndicesService indicesService) {
        super(settings);
        ByteSizeValue indexingBuffer;
        this.threadPool = threadPool;
        this.indicesService = indicesService;
        String indexingBufferSetting = this.componentSettings.get("index_buffer_size", "10%");
        if (indexingBufferSetting.endsWith("%")) {
            double percent = Double.parseDouble(indexingBufferSetting.substring(0, indexingBufferSetting.length() - 1));
            indexingBuffer = new ByteSizeValue((long)((double)JvmInfo.jvmInfo().mem().heapMax().bytes() * (percent / 100.0)));
            ByteSizeValue minIndexingBuffer = this.componentSettings.getAsBytesSize("min_index_buffer_size", new ByteSizeValue(48L, ByteSizeUnit.MB));
            ByteSizeValue maxIndexingBuffer = this.componentSettings.getAsBytesSize("max_index_buffer_size", null);
            if (indexingBuffer.bytes() < minIndexingBuffer.bytes()) {
                indexingBuffer = minIndexingBuffer;
            }
            if (maxIndexingBuffer != null && indexingBuffer.bytes() > maxIndexingBuffer.bytes()) {
                indexingBuffer = maxIndexingBuffer;
            }
        } else {
            indexingBuffer = ByteSizeValue.parseBytesSizeValue(indexingBufferSetting, null);
        }
        this.indexingBuffer = indexingBuffer;
        this.minShardIndexBufferSize = this.componentSettings.getAsBytesSize("min_shard_index_buffer_size", new ByteSizeValue(4L, ByteSizeUnit.MB));
        this.maxShardIndexBufferSize = this.componentSettings.getAsBytesSize("max_shard_index_buffer_size", new ByteSizeValue(512L, ByteSizeUnit.MB));
        this.inactiveTime = this.componentSettings.getAsTime("shard_inactive_time", TimeValue.timeValueMinutes(30L));
        this.interval = this.componentSettings.getAsTime("interval", TimeValue.timeValueSeconds(30L));
        this.logger.debug("using index_buffer_size [{}], with min_shard_index_buffer_size [{}], max_shard_index_buffer_size [{}], shard_inactive_time [{}]", this.indexingBuffer, this.minShardIndexBufferSize, this.maxShardIndexBufferSize, this.inactiveTime);
    }

    @Override
    protected void doStart() throws ElasticSearchException {
        this.indicesService.indicesLifecycle().addListener(this.listener);
        this.scheduler = this.threadPool.scheduleWithFixedDelay(new ShardsIndicesStatusChecker(), this.interval);
    }

    @Override
    protected void doStop() throws ElasticSearchException {
        this.indicesService.indicesLifecycle().removeListener(this.listener);
        if (this.scheduler != null) {
            this.scheduler.cancel(false);
            this.scheduler = null;
        }
    }

    @Override
    protected void doClose() throws ElasticSearchException {
    }

    private void calcAndSetShardIndexingBuffer(String reason2) {
        int shardsCount = this.countShards();
        if (shardsCount == 0) {
            return;
        }
        ByteSizeValue shardIndexingBufferSize = this.calcShardIndexingBuffer(shardsCount);
        if (shardIndexingBufferSize == null) {
            return;
        }
        if (shardIndexingBufferSize.bytes() < this.minShardIndexBufferSize.bytes()) {
            shardIndexingBufferSize = this.minShardIndexBufferSize;
        }
        if (shardIndexingBufferSize.bytes() > this.maxShardIndexBufferSize.bytes()) {
            shardIndexingBufferSize = this.maxShardIndexBufferSize;
        }
        this.logger.debug("recalculating shard indexing buffer (reason={}), total is [{}] with [{}] active shards, each shard set to [{}]", reason2, this.indexingBuffer, shardsCount, shardIndexingBufferSize);
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                ShardIndexingStatus status2 = this.shardsIndicesStatus.get(indexShard.shardId());
                if (status2 != null && status2.inactiveIndexing) continue;
                try {
                    ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(shardIndexingBufferSize);
                }
                catch (EngineClosedException e) {
                }
                catch (FlushNotAllowedEngineException e) {
                }
                catch (Exception e) {
                    this.logger.warn("failed to set shard [{}][{}] index buffer to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), shardIndexingBufferSize);
                }
            }
        }
    }

    private ByteSizeValue calcShardIndexingBuffer(int shardsCount) {
        return new ByteSizeValue(this.indexingBuffer.bytes() / (long)shardsCount);
    }

    private int countShards() {
        int shardsCount = 0;
        for (IndexService indexService : this.indicesService) {
            for (IndexShard indexShard : indexService) {
                ShardIndexingStatus status2 = this.shardsIndicesStatus.get(indexShard.shardId());
                if (status2 != null && status2.inactiveIndexing) continue;
                ++shardsCount;
            }
        }
        return shardsCount;
    }

    static class ShardIndexingStatus {
        long translogId = -1L;
        int translogNumberOfOperations = -1;
        boolean inactiveIndexing = false;
        long time = -1L;

        ShardIndexingStatus() {
        }
    }

    class Listener
    extends IndicesLifecycle.Listener {
        Listener() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void afterIndexShardCreated(IndexShard indexShard) {
            Object object = IndexingMemoryController.this.mutex;
            synchronized (object) {
                IndexingMemoryController.this.calcAndSetShardIndexingBuffer("created_shard[" + indexShard.shardId().index().name() + "][" + indexShard.shardId().id() + "]");
                IndexingMemoryController.this.shardsIndicesStatus.put(indexShard.shardId(), new ShardIndexingStatus());
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void afterIndexShardClosed(ShardId shardId, boolean delete2) {
            Object object = IndexingMemoryController.this.mutex;
            synchronized (object) {
                IndexingMemoryController.this.calcAndSetShardIndexingBuffer("removed_shard[" + shardId.index().name() + "][" + shardId.id() + "]");
                IndexingMemoryController.this.shardsIndicesStatus.remove(shardId);
            }
        }
    }

    class ShardsIndicesStatusChecker
    implements Runnable {
        ShardsIndicesStatusChecker() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            Object object = IndexingMemoryController.this.mutex;
            synchronized (object) {
                boolean activeInactiveStatusChanges = false;
                ArrayList<IndexShard> activeToInactiveIndexingShards = Lists.newArrayList();
                ArrayList<IndexShard> inactiveToActiveIndexingShards = Lists.newArrayList();
                for (IndexService indexService : IndexingMemoryController.this.indicesService) {
                    for (IndexShard indexShard : indexService) {
                        long time = IndexingMemoryController.this.threadPool.estimatedTimeInMillis();
                        Translog translog = ((InternalIndexShard)indexShard).translog();
                        ShardIndexingStatus status2 = (ShardIndexingStatus)IndexingMemoryController.this.shardsIndicesStatus.get(indexShard.shardId());
                        if (status2 == null) continue;
                        if (status2.translogId == translog.currentId() && translog.estimatedNumberOfOperations() == 0) {
                            if (status2.time == -1L) {
                                status2.time = time;
                            }
                            if (!status2.inactiveIndexing && time - status2.time > IndexingMemoryController.this.inactiveTime.millis() && indexShard.mergeStats().current() == 0L) {
                                activeToInactiveIndexingShards.add(indexShard);
                                status2.inactiveIndexing = true;
                                activeInactiveStatusChanges = true;
                                IndexingMemoryController.this.logger.debug("marking shard [{}][{}] as inactive (inactive_time[{}]) indexing wise, setting size to [{}]", indexShard.shardId().index().name(), indexShard.shardId().id(), IndexingMemoryController.this.inactiveTime, Engine.INACTIVE_SHARD_INDEXING_BUFFER);
                            }
                        } else {
                            if (status2.inactiveIndexing) {
                                inactiveToActiveIndexingShards.add(indexShard);
                                status2.inactiveIndexing = false;
                                activeInactiveStatusChanges = true;
                                IndexingMemoryController.this.logger.debug("marking shard [{}][{}] as active indexing wise", indexShard.shardId().index().name(), indexShard.shardId().id());
                            }
                            status2.time = -1L;
                        }
                        status2.translogId = translog.currentId();
                        status2.translogNumberOfOperations = translog.estimatedNumberOfOperations();
                    }
                }
                for (IndexShard indexShard : activeToInactiveIndexingShards) {
                    try {
                        ((InternalIndexShard)indexShard).engine().updateIndexingBufferSize(Engine.INACTIVE_SHARD_INDEXING_BUFFER);
                    }
                    catch (EngineClosedException e) {
                    }
                    catch (FlushNotAllowedEngineException e) {}
                }
                if (activeInactiveStatusChanges) {
                    IndexingMemoryController.this.calcAndSetShardIndexingBuffer("shards became active/inactive (indexing wise)");
                }
            }
        }
    }
}

