/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.jvm;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ScheduledFuture;
import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.common.collect.ImmutableCollection;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.collect.ImmutableSet;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.monitor.dump.DumpGenerator;
import org.elasticsearch.monitor.dump.DumpMonitorService;
import org.elasticsearch.monitor.jvm.DeadlockAnalyzer;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.monitor.jvm.JvmStats;
import org.elasticsearch.threadpool.ThreadPool;

public class JvmMonitorService
extends AbstractLifecycleComponent<JvmMonitorService> {
    private final ThreadPool threadPool;
    private final DumpMonitorService dumpMonitorService;
    private final boolean enabled;
    private final TimeValue interval;
    private final ImmutableMap<String, GcThreshold> gcThresholds;
    private volatile ScheduledFuture scheduledFuture;

    @Inject
    public JvmMonitorService(Settings settings, ThreadPool threadPool, DumpMonitorService dumpMonitorService) {
        super(settings);
        this.threadPool = threadPool;
        this.dumpMonitorService = dumpMonitorService;
        this.enabled = this.componentSettings.getAsBoolean("enabled", true);
        this.interval = this.componentSettings.getAsTime("interval", TimeValue.timeValueSeconds(1L));
        MapBuilder<String, GcThreshold> gcThresholds = MapBuilder.newMapBuilder();
        Map<String, Settings> gcThresholdGroups = this.componentSettings.getGroups("gc");
        for (Map.Entry<String, Settings> entry : gcThresholdGroups.entrySet()) {
            String name2 = entry.getKey();
            TimeValue warn2 = entry.getValue().getAsTime("warn", null);
            TimeValue info = entry.getValue().getAsTime("info", null);
            TimeValue debug = entry.getValue().getAsTime("debug", null);
            if (warn2 == null || info == null || debug == null) {
                this.logger.warn("ignoring gc_threshold for [{}], missing warn/info/debug values", name2);
                continue;
            }
            gcThresholds.put(name2, new GcThreshold(name2, warn2.millis(), info.millis(), debug.millis()));
        }
        if (!gcThresholds.containsKey("ParNew")) {
            gcThresholds.put("ParNew", new GcThreshold("ParNew", 1000L, 700L, 400L));
        }
        if (!gcThresholds.containsKey("ConcurrentMarkSweep")) {
            gcThresholds.put("ConcurrentMarkSweep", new GcThreshold("ConcurrentMarkSweep", 10000L, 5000L, 2000L));
        }
        if (!gcThresholds.containsKey("default")) {
            gcThresholds.put("default", new GcThreshold("default", 10000L, 5000L, 2000L));
        }
        this.gcThresholds = gcThresholds.immutableMap();
        this.logger.debug("enabled [{}], last_gc_enabled [{}], interval [{}], gc_threshold [{}]", this.enabled, JvmStats.isLastGcEnabled(), this.interval, this.gcThresholds);
    }

    @Override
    protected void doStart() throws ElasticSearchException {
        if (!this.enabled) {
            return;
        }
        this.scheduledFuture = this.threadPool.scheduleWithFixedDelay(new JvmMonitor(), this.interval);
    }

    @Override
    protected void doStop() throws ElasticSearchException {
        if (!this.enabled) {
            return;
        }
        this.scheduledFuture.cancel(true);
    }

    @Override
    protected void doClose() throws ElasticSearchException {
    }

    private class JvmMonitor
    implements Runnable {
        private JvmStats lastJvmStats = JvmStats.jvmStats();
        private long seq = 0L;
        private final Set<DeadlockAnalyzer.Deadlock> lastSeenDeadlocks = new HashSet<DeadlockAnalyzer.Deadlock>();

        @Override
        public void run() {
            this.monitorLongGc();
        }

        private synchronized void monitorLongGc() {
            ++this.seq;
            JvmStats currentJvmStats = JvmStats.jvmStats();
            for (int i2 = 0; i2 < currentJvmStats.gc().collectors().length; ++i2) {
                long avgCollectionTime;
                long collectionTime;
                JvmStats.GarbageCollector gc2 = currentJvmStats.gc().collectors()[i2];
                JvmStats.GarbageCollector prevGc = this.lastJvmStats.gc.collectors[i2];
                long collections = gc2.collectionCount - prevGc.collectionCount;
                if (collections == 0L || (collectionTime = gc2.collectionTime - prevGc.collectionTime) == 0L) continue;
                GcThreshold gcThreshold = (GcThreshold)JvmMonitorService.this.gcThresholds.get(gc2.name());
                if (gcThreshold == null) {
                    gcThreshold = (GcThreshold)JvmMonitorService.this.gcThresholds.get("default");
                }
                if (gc2.lastGc() != null && prevGc.lastGc() != null) {
                    JvmStats.GarbageCollector.LastGc lastGc = gc2.lastGc();
                    if (lastGc.startTime == prevGc.lastGc().startTime() || lastGc.duration().hoursFrac() > 1.0) continue;
                    if (lastGc.duration().millis() > gcThreshold.warnThreshold) {
                        JvmMonitorService.this.logger.warn("[last_gc][{}][{}][{}] duration [{}], collections [{}], total [{}]/[{}], reclaimed [{}], leaving [{}][{}]/[{}]", gc2.name(), this.seq, gc2.getCollectionCount(), lastGc.duration(), collections, TimeValue.timeValueMillis(collectionTime), gc2.collectionTime(), lastGc.reclaimed(), lastGc.afterUsed(), lastGc.max());
                    } else if (lastGc.duration().millis() > gcThreshold.infoThreshold) {
                        JvmMonitorService.this.logger.info("[last_gc][{}][{}][{}] duration [{}], collections [{}], total [{}]/[{}], reclaimed [{}], leaving [{}]/[{}]", gc2.name(), this.seq, gc2.getCollectionCount(), lastGc.duration(), collections, TimeValue.timeValueMillis(collectionTime), gc2.collectionTime(), lastGc.reclaimed(), lastGc.afterUsed(), lastGc.max());
                    } else if (lastGc.duration().millis() > gcThreshold.debugThreshold && JvmMonitorService.this.logger.isDebugEnabled()) {
                        JvmMonitorService.this.logger.debug("[last_gc][{}][{}][{}] duration [{}], collections [{}], total [{}]/[{}], reclaimed [{}], leaving [{}]/[{}]", gc2.name(), this.seq, gc2.getCollectionCount(), lastGc.duration(), collections, TimeValue.timeValueMillis(collectionTime), gc2.collectionTime(), lastGc.reclaimed(), lastGc.afterUsed(), lastGc.max());
                    }
                }
                if ((avgCollectionTime = collectionTime / collections) > gcThreshold.warnThreshold) {
                    JvmMonitorService.this.logger.warn("[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}", gc2.name(), this.seq, gc2.collectionCount(), TimeValue.timeValueMillis(collectionTime), collections, TimeValue.timeValueMillis(currentJvmStats.timestamp() - this.lastJvmStats.timestamp()), TimeValue.timeValueMillis(collectionTime), gc2.collectionTime(), this.lastJvmStats.mem().heapUsed(), currentJvmStats.mem().heapUsed(), JvmInfo.jvmInfo().mem().heapMax(), this.buildPools(this.lastJvmStats, currentJvmStats));
                    continue;
                }
                if (avgCollectionTime > gcThreshold.infoThreshold) {
                    JvmMonitorService.this.logger.info("[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}", gc2.name(), this.seq, gc2.collectionCount(), TimeValue.timeValueMillis(collectionTime), collections, TimeValue.timeValueMillis(currentJvmStats.timestamp() - this.lastJvmStats.timestamp()), TimeValue.timeValueMillis(collectionTime), gc2.collectionTime(), this.lastJvmStats.mem().heapUsed(), currentJvmStats.mem().heapUsed(), JvmInfo.jvmInfo().mem().heapMax(), this.buildPools(this.lastJvmStats, currentJvmStats));
                    continue;
                }
                if (avgCollectionTime <= gcThreshold.debugThreshold || !JvmMonitorService.this.logger.isDebugEnabled()) continue;
                JvmMonitorService.this.logger.debug("[gc][{}][{}][{}] duration [{}], collections [{}]/[{}], total [{}]/[{}], memory [{}]->[{}]/[{}], all_pools {}", gc2.name(), this.seq, gc2.collectionCount(), TimeValue.timeValueMillis(collectionTime), collections, TimeValue.timeValueMillis(currentJvmStats.timestamp() - this.lastJvmStats.timestamp()), TimeValue.timeValueMillis(collectionTime), gc2.collectionTime(), this.lastJvmStats.mem().heapUsed(), currentJvmStats.mem().heapUsed(), JvmInfo.jvmInfo().mem().heapMax(), this.buildPools(this.lastJvmStats, currentJvmStats));
            }
            this.lastJvmStats = currentJvmStats;
        }

        private String buildPools(JvmStats prev, JvmStats current2) {
            StringBuilder sb = new StringBuilder();
            for (JvmStats.MemoryPool currentPool : current2.mem()) {
                JvmStats.MemoryPool prevPool = null;
                for (JvmStats.MemoryPool pool : prev.mem()) {
                    if (!pool.getName().equals(currentPool.getName())) continue;
                    prevPool = pool;
                    break;
                }
                sb.append("{[").append(currentPool.name()).append("] [").append(prevPool == null ? "?" : prevPool.used()).append("]->[").append(currentPool.used()).append("]/[").append(currentPool.getMax()).append("]}");
            }
            return sb.toString();
        }

        private void monitorDeadlock() {
            Object[] deadlocks = DeadlockAnalyzer.deadlockAnalyzer().findDeadlocks();
            if (deadlocks != null && deadlocks.length > 0) {
                ImmutableCollection asSet = ((ImmutableSet.Builder)new ImmutableSet.Builder().add(deadlocks)).build();
                if (!((ImmutableSet)asSet).equals(this.lastSeenDeadlocks)) {
                    DumpGenerator.Result genResult = JvmMonitorService.this.dumpMonitorService.generateDump("deadlock", null, "summary", "thread");
                    StringBuilder sb = new StringBuilder("Detected Deadlock(s)");
                    for (DeadlockAnalyzer.Deadlock deadlock : asSet) {
                        sb.append("\n   ----> ").append(deadlock);
                    }
                    sb.append("\nDump generated [").append(genResult.location()).append("]");
                    JvmMonitorService.this.logger.error(sb.toString(), new Object[0]);
                    this.lastSeenDeadlocks.clear();
                    this.lastSeenDeadlocks.addAll(asSet);
                }
            } else {
                this.lastSeenDeadlocks.clear();
            }
        }
    }

    static class GcThreshold {
        public final String name;
        public final long warnThreshold;
        public final long infoThreshold;
        public final long debugThreshold;

        GcThreshold(String name2, long warnThreshold, long infoThreshold, long debugThreshold) {
            this.name = name2;
            this.warnThreshold = warnThreshold;
            this.infoThreshold = infoThreshold;
            this.debugThreshold = debugThreshold;
        }

        public String toString() {
            return "GcThreshold{name='" + this.name + '\'' + ", warnThreshold=" + this.warnThreshold + ", infoThreshold=" + this.infoThreshold + ", debugThreshold=" + this.debugThreshold + '}';
        }
    }
}

