/*
 *   Elliptic Curve Digital Signature Algorithm (ECDSA)
 *
 *
 *   This program verifies the signature given to a <file> in
 *   <file>.ecs generated by program ecsign
 * 
 *   The curve is y^2=x^3+Ax+B mod p
 *
 *   The file common.ecs is presumed to exist, and to contain the domain
 *   information {p,A,B,q,x,y}, where A and B are curve parameters, (x,y) are
 *   a point of order q, p is the prime modulus, and q is the order of the 
 *   point (x,y). In fact normally q is the prime number of points counted
 *   on the curve. 
 *
 *   This program is written for static mode.
 *   For a 160-bit modulus p, MR_STATIC could be defined as 5 in mirdef.h
 *   for a 32-bit processor, or 10 for a 16-bit processor.
 *   The system parameters can be found in the file common.ecs
 *   Assumes MR_GENERIC_MT is defined in mirdef.h
 *
 *
 *   Copyright (c) 1997-2005 Shamus Software Ltd.
 */

#include <stdio.h>
#include "miracl.h"
#include <stdlib.h>
#include <string.h>

void strip(char *name)
{ /* strip off filename extension */
    int i;
    for (i=0;name[i]!='\0';i++)
    {
        if (name[i]!='.') continue;
        name[i]='\0';
        break;
    }
}

static void hashing(miracl *mip,FILE *fp,big hash)
{ /* compute hash function */
    char h[20];
    sha sh;
    int len,ch;
    shs_init(&sh);
    while ((ch=fgetc(fp))!=EOF) shs_process(&sh,ch);
    shs_hash(&sh,h);
    len=(MIRACL*MR_STATIC)/8;
    if (len>20) len=20;
    bytes_to_big(mip,len,h,hash);
}

int main()
{
    FILE *fp;
    int bits,ep;
    epoint *g,*public;
    char ifname[50],ofname[50];
    big a,b,p,q,x,y,v,u1,u2,r,s,hash;
    miracl instance;
    miracl *mip=&instance;
    char mem[MR_BIG_RESERVE(12)];           /* reserve space on the stack for 12 bigs */
    char mem1[MR_ECP_RESERVE(2)];           /* and two elliptic curve points         */
    memset(mem,0,MR_BIG_RESERVE(12));
    memset(mem1,0,MR_ECP_RESERVE(2));

/* get public data */
    fp=fopen("common.ecs","rt");
    if (fp==NULL)
    {
        printf("file common.ecs does not exist\n");
        return 0;
    }
    fscanf(fp,"%d\n",&bits);

    mirsys(mip,bits/4,16);   /* Use Hex Internally */
    a=mirvar_mem(mip,mem,0);
    b=mirvar_mem(mip,mem,1);
    p=mirvar_mem(mip,mem,2);
    q=mirvar_mem(mip,mem,3);
    x=mirvar_mem(mip,mem,4);
    y=mirvar_mem(mip,mem,5);
    v=mirvar_mem(mip,mem,6);
    u1=mirvar_mem(mip,mem,7);
    u2=mirvar_mem(mip,mem,8);
    s=mirvar_mem(mip,mem,9);
    r=mirvar_mem(mip,mem,10);
    hash=mirvar_mem(mip,mem,11);

    innum(mip,p,fp);
    innum(mip,a,fp);
    innum(mip,b,fp);
    innum(mip,q,fp);
    innum(mip,x,fp);
    innum(mip,y,fp);
    
    fclose(fp);

    ecurve_init(mip,a,b,p,MR_PROJECTIVE);  /* initialise curve */
    g=epoint_init_mem(mip,mem1,0);
    epoint_set(mip,x,y,0,g); /* initialise point of order q */

/* get public key of signer */
    fp=fopen("public.ecs","rt");
    if (fp==NULL)
    {
        printf("file public.ecs does not exist\n");
        return 0;
    }
    fscanf(fp,"%d",&ep);
    innum(mip,x,fp);
    fclose(fp);

    public=epoint_init_mem(mip,mem1,1);
    epoint_set(mip,x,x,ep,public);  /* decompress */

/* get message */
    printf("signed file = ");
    gets(ifname);
    strcpy(ofname,ifname);
    strip(ofname);
    strcat(ofname,".ecs");
    if ((fp=fopen(ifname,"rb"))==NULL)
    { /* no message */
        printf("Unable to open file %s\n",ifname);
        return 0;
    }
    hashing(mip,fp,hash);
    fclose(fp);
    fp=fopen(ofname,"rt");
    if (fp==NULL)
    { /* no signature */
        printf("signature file %s does not exist\n",ofname);
        return 0;
    }
    innum(mip,r,fp);
    innum(mip,s,fp);
    fclose(fp);
    if (compare(r,q)>=0 || compare(s,q)>=0)
    {
        printf("Signature is NOT verified\n");
        return 0;
    }
    xgcd(mip,s,q,s,s,s);
    mad(mip,hash,s,s,q,q,u1);
    mad(mip,r,s,s,q,q,u2);

    ecurve_mult2(mip,u2,public,u1,g,g);
    epoint_get(mip,g,v,v);
    divide(mip,v,q,q);
    if (compare(v,r)==0) printf("Signature is verified\n");
    else                 printf("Signature is NOT verified\n");

    memset(mem,0,MR_BIG_RESERVE(12));
    memset(mem1,0,MR_ECP_RESERVE(2));
 
    return 0;
}

