//---------------------------------------------------------------------------
// Copyright (C) 1999 Dallas Semiconductor Corporation, All Rights Reserved.
// 
// Permission is hereby granted, free of charge, to any person obtaining a 
// copy of this software and associated documentation files (the "Software"), 
// to deal in the Software without restriction, including without limitation 
// the rights to use, copy, modify, merge, publish, distribute, sublicense, 
// and/or sell copies of the Software, and to permit persons to whom the 
// Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included 
// in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
// MERCHANTABILITY,  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
// IN NO EVENT SHALL DALLAS SEMICONDUCTOR BE LIABLE FOR ANY CLAIM, DAMAGES 
// OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR 
// OTHER DEALINGS IN THE SOFTWARE.
// 
// Except as contained in this notice, the name of Dallas Semiconductor 
// shall not be used except as stated in the Dallas Semiconductor 
// Branding Policy. 
//---------------------------------------------------------------------------
//
//  thermodl.c - This utility uses to download the results of the
//               current mission of a DS1921 Thermochron iButton.
//
//  Version:   1.03
//
//

#define TMEXUTIL

#include <stdio.h>
#include <stdlib.h>
#include "mlan.h"   
#include "thermo.h"

// external function prototypes 
extern int  RunThermoScript(ThermoState script[],FILE *);
extern void MissionStatusToString(MissionStatus *, int, char *);
extern void InterpretStatus(MissionStatus *);
extern void InterpretHistogram(Histogram *);
extern void HistogramToString(Histogram *, int, char *);
extern void InterpretAlarms(TempAlarmEvents *, MissionStatus *);
extern void AlarmsToString(TempAlarmEvents *, char *);
extern void InterpretLog(Log *, MissionStatus *);
extern void LogToString(Log *, int, char *);
extern int  Aquire1WireNet(char *, char *);
extern void Release1WireNet(char *);

// local function prototypes
void PrintResults(FILE *,int);

//----------------------------------------------------------------------
//  This is the Main routine for thermodl.
//
void main(short argc, char **argv)
{
   int Fahrenheit=FALSE,filenum;
   FILE *fp;
   char return_msg[128];

   // check arguments to see if request instruction with '?' or too many
   if ((argc < 2) || (argc > 4) || ((argc > 1) && (argv[1][0] == '?' || argv[1][1] == '?')))
   {
       printf("\nusage: thermodl 1wire_net_name <output_filename> </Fahrenheit>\n"
              "  - Thermochron download on the 1-Wire Net port\n"
              "  - 1wire_net_port required port name\n"
              "    example: \"COM1\" (Win32 DS2480),\"/dev/cua0\" \n"
              "    (Linux DS2480),\"1\" (Win32 TMEX)\n"
              "  - <output_filename> optional output filename\n"
              "  - </Fahrenheit> optional Fahrenheit mode (default Celsius)\n"
              "  - version 1.03\n");
       exit(1);
   }

   // attempt to aquire the 1-Wire Net
   if (!Aquire1WireNet(argv[1], return_msg))
   {  
      printf("%s",return_msg);
      exit(1);
   }

   // success
   printf("%s",return_msg);

   //----------------------------------------
   // Introduction
   printf("\n/---------------------------------------------\n");
   printf("  Find and download a DS1921 Thermochron iButton\n" 
          "  Version 1.03\n\n");

   // check arguments for temperature conversion and filename
   Fahrenheit = FALSE;
   filenum = 0;
   if (argc >= 3)
   {
      if (argv[2][0] != '/')
         filenum = 2;
      else if ((argv[2][1] == 'F') || (argv[2][1] == 'f'))
         Fahrenheit = TRUE;

      if (argc == 4)
      {    
         if (argv[3][0] != '/')
            filenum = 3;
         else if ((argv[3][1] == 'F') || (argv[3][1] == 'f'))
            Fahrenheit = TRUE;
      }   
   }

   // open the output file  
   fp = NULL;
   if (filenum > 0)
   {
      fp = fopen(argv[filenum],"w+");
      if(fp == NULL)
      {    
         printf("ERROR, Could not open output file!\n");
         exit(1);
      }
      else
         printf("File '%s' opened to write mission results.\n",
                 argv[filenum]);
   }

   // run the script to find and download thermochron
   if (!RunThermoScript(DownloadThermo,stdout))
   { 
      printf("Thermochon not downloaded, end program\n");
      exit(1);
   }
   
   // print the output
   PrintResults(fp,Fahrenheit);

   // close opened file
   if (fp != NULL)
   {
      printf("File '%s' closed.\n",
              argv[filenum]);
      fclose(fp);
   }

   // release the 1-Wire Net
   Release1WireNet(return_msg);
   printf("%s",return_msg);
   printf("End program normally\n");
   exit(0);
}



//--------------------------------------------------------------------------
//  Prints the mission data optionaly to a file or standard out
//
void PrintResults(FILE *fp, int ConvertToF)
{
   char *str;  

   // check if need to use standard out
   if (fp == NULL)
      fp = stdout;

   // get big block to use as a buffer
   str = malloc(80000);   
   if (str == NULL)
   {
      printf("Insufficient memory available to print!\n"); 
      return;
   }

   // mission status 
   InterpretStatus(&MissStat);
   MissionStatusToString(&MissStat, ConvertToF, &str[0]);
   fprintf(fp,"\n%s\n",str);

   // alarm events
   InterpretAlarms(&AlarmData, &MissStat);
   AlarmsToString(&AlarmData, &str[0]);
   fprintf(fp,"%s\n",str);

   // histogram
   InterpretHistogram(&HistData);
   HistogramToString(&HistData, ConvertToF, &str[0]);
   fprintf(fp,"%s\n",str);

   // log data
   InterpretLog(&LogData, &MissStat);
   LogToString(&LogData, ConvertToF, &str[0]);
   fprintf(fp,"%s\n",str);

   // free the memory block used
   free(str);
}
