/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt. 
  * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */ 

 /*
 * SVGWaitScreen.java
 *
 * Created on June 19, 2006, 4:32 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.netbeans.microedition.svg;

import javax.microedition.lcdui.Command;
import javax.microedition.lcdui.CommandListener;
import javax.microedition.lcdui.Display;
import javax.microedition.m2g.SVGEventListener;
import javax.microedition.m2g.SVGImage;
import org.netbeans.microedition.util.CancellableTask;

/**
 * This component suits as a wait screen, which let the user to execute a blocking
 * background task (e.g. a network communication) and waits for it until finished.
 * During the execution of the task, an SVG image/animation is being shown
 * on the screen.
 * <p/>
 * The background task is being started immediately prior the component is being
 * shown on the screen.
 * <p/>
 * When the background task is finished, this component calls commandAction method
 * on assigned CommandListener object. In the case of success, the commandAction method
 * is called with SUCCESS_COMMAND as parameter, in the case of failure, the commandAction
 * method is called with FAILURE_COMMAND as parameter.
 * <p/>
 * @author breh
 */
public class SVGWaitScreen extends SVGAnimatorWrapper {
     
    /**
     * Command fired when the background task was finished succesfully
     */
    public static final Command SUCCESS_COMMAND = new Command("Success",Command.OK,0);
    
    /**
     * Command fired when the background task failed (threw exception)
     */
    public static final Command FAILURE_COMMAND = new Command("Failure",Command.OK,0);

    
    // background task 
    private CancellableTask task;
    // background thread task executor
    private Thread backgroundExecutor;
    
    
    /**
     * 
     * Creates a new instance of SVGWaitScreen. It requires instance of SVGImage, which will
     * be used for animation and display. 
     * <p/> Please note, supplied SVGImage shouldn't be reused in other SVGAnimator.
     */
    public SVGWaitScreen(SVGImage svgImage, Display display) throws IllegalArgumentException {
        super(svgImage, display);
        setSVGEventListener(new WaitScreenSvgEventListener());
    }
    
    
    /**
     * Sets the task to be run on the background.
     * @param task task to be executed
     */
    public void setTask(CancellableTask task) {
        this.task = task;
    }
    
    
    /**
     * Gets the background task.
     * @return task being executed in background while this component is being shown
     * on the screen
     */
    public CancellableTask getTask() {
        return task;
    }
    
    
    // private stuff
    private void doAction() {
        CommandListener cl = getCommandListener();
        if (cl != null) {            
            if ((task != null) && (task.hasFailed())) {
                cl.commandAction(FAILURE_COMMAND,this);
            } else {
                // task didn't failed - success !!!
                cl.commandAction(SUCCESS_COMMAND,this);
            }
        }
        
    }
    
    
    
    /**
     * BackgroundExecutor task
     */
    private class BackgroundExecutor implements Runnable {
        
        private CancellableTask task;
        
        public BackgroundExecutor(CancellableTask task)	throws IllegalArgumentException {
            if (task == null) throw new IllegalArgumentException("Task parameter cannot be null");
            this.task = task;
        }
        
        public void run() {
            try {
                task.run();
            } finally {
                SVGWaitScreen.this.backgroundExecutor = null;
                doAction();
            }
        }
    }    
    
    
    // svg event listener listening on key presses
    private class WaitScreenSvgEventListener implements SVGEventListener {

        public void keyPressed(int i) {
        }

        public void keyReleased(int i) {
        }

        public void pointerPressed(int i, int i0) {
        }

        public void pointerReleased(int i, int i0) {
        }

        public void hideNotify() {
        }

        public void showNotify() {
            if (task != null) {
                if (backgroundExecutor == null) {
                    backgroundExecutor = new Thread(new BackgroundExecutor(task));
                    backgroundExecutor.start();
                }
            }  else {                
                // switch to next displayable immediatelly - no task was assigned
                // do it when the task is repainted - on some devices there are
                // some race-conditions
                getDisplay().callSerially(new Runnable() {
                    public void run() {
                        doAction();
                    }
                });
        }
        }

        public void sizeChanged(int i, int i0) {
        }
        
    }    
    
}
