/**
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at
 * https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at https://woodstock.dev.java.net/public/CDDLv1.0.html.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 */


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.body");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.common");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");

/**
 * @class This class contains functions common to HTML elements.
 * @static
 */
webui.suntheme4_2.common = {
    /**
     * Variables needed when submitting form so timeout will work properly.
     * @private
     */
    _formToSubmit: null,
    _submissionComponentId: null,

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // String functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Replace occurences of delimiter with the escapeChar and the
     * delimiter. For example replace "," with "/," if delimiter == "," and
     * escapeChar is "/".
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The escaped string.
     */
    escapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // Escape occurrences of delimiter with 
        // escapeChar and the delimiter.
        //
        // First escape the escape char.
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        var rx = new RegExp(escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar + escapeChar);
        
        rx = new RegExp(delimiter, "g");
        return s1.replace(rx, escapeChar + delimiter);
    },
    
    /**
     * Replace occurences of a sequence of 2 instances of delimiter 
     * with 1 instance of the delimiter. For example replace ",," with "," if delimiter == ","
     *
     * @param {String} s The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {String} The unescaped string.
     */
    unescapeString: function(s, delimiter, escapeChar) {
        if (s == null) {
            return null;
        }
        if (delimiter == null) {
            return s;
        }
        if (escapeChar == null) {
            return null;
        }
        
        // UnEscape occurrences of delimiter with 
        // single instance of the delimiter
        //
        var escape_escapeChar = escapeChar;
        if (escapeChar == "\\") {
            escape_escapeChar = escapeChar + escapeChar;
        }
        
        // First unescape the escape char.
        //
        var rx = new RegExp(escape_escapeChar + escape_escapeChar, "g");
        var s1 = s.replace(rx, escapeChar);
        
        // Now replace escaped delimters
        //
        rx = new RegExp(escape_escapeChar + delimiter, "g");
        return s1.replace(rx, delimiter);
    },
    
    /**
     * Return an array of unescaped strings from escapedString
     * where the escaped character is delimiter.
     * If delimiter is "," escapedString might have the form
     * <p><pre>
     * XX\,XX,MM\,MM
     *
     * where "\" is the escape char.
     * 
     * and is returned as an array
     * array[0] == "XX,XX"
     * array[1] == "MM,MM"
     * </pre><p>
     *
     * @param {String} escapedString The string to escape.
     * @param {String} delimiter The character to replace.
     * @param {String} escapeChar The character used for the escape.
     * @return {Array} An array of unescaped strings.
     */
    unescapeStrings: function(escapedString, delimiter, escapeChar) {
        if (escapedString == null || escapedString == "") {
            return null;
        }
        if (delimiter == null || delimiter == "") {
            return escapedString;
        }
        if (escapeChar == null || escapeChar == "") {
            return null;
        }
        
        // Need to do this character by character.
        var selections = new Array();
        var index = 0;
        var escseen = 0;
        var j = 0;
        
        for (var i = 0; i < escapedString.length; ++i) {
            if (escapedString.charAt(i) == delimiter) {
                if (escseen % 2 == 0) {
                    selections[index++] = escapedString.slice(j, i);
                    j = i + 1;
                }
            }
            if (escapedString.charAt(i) == escapeChar) {
                ++escseen;
                continue;
            } else {
                escseen = 0;
            }
        }
        
        // Capture the last split.
        selections[index] = escapedString.slice(j);
        
        // Now unescape each selection
        var unescapedArray = new Array();
        
        // Now replace escaped delimiters
        // i.e.  "\," with ","
        for (i = 0; i < selections.length; ++i) {
            unescapedArray[i] = webui.suntheme4_2.common.unescapeString(
            selections[i], delimiter, escapeChar);
        }
        return unescapedArray;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Style functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function add any styleClass to an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to add
     * @return {boolean} true if successful; otherwise, false.
     */
    addStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null) {
            return false;
        }
        
        // handle easy case first
        if (element.className == null) {
            element.className = styleClass;
            return true;
        }
        
        // break out style classes into an array  
        var classes = webui.suntheme4_2.common.splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each element of classes, compare it to styleClass.
        // If it is not in the array, append it to the end.
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                return true;
            }
        }
        element.className = element.className + " " + styleClass;
        return true;
    },
    
    /**
     * Use this function to check if an array has a style class
     *
     * @param {Array} styleArray of style classes to check
     * @param {String} styleClass the styleClass to check
     * @return {Array} An array of classes.
     */
    checkStyleClasses: function(styleArray, styleClass) {
        if (styleArray == null || styleClass == null) {
            return false;
        }
        for (var i = 0; i < styleArray.length; i++) {
            if (styleArray[i] != null && styleArray[i] == styleClass) {
                return true;
            }
        }   
        return false;
    },
    
    /**
     * Use this function to get array of style classes
     *
     * @param {Node} element the dom html tag element
     * @return {Array} An array of classes.
     */
    splitStyleClasses: function(element) {
        if (element != null && element.className != null) {
            return element.className.split(" ");
        } else {
            return null;
        }
    },
    
    /**
     * Use this function remove any styleClass for an html tag
     *
     * @param {Node} element the dom html tag element
     * @param {String} styleClass the name of the class to remove
     * @return {boolean} true if successful; otherwise, false.
     */
    stripStyleClass: function(element, styleClass) {
        // routine protection in javascript
        if (element == null || styleClass == null || element.className == null) {
            return false;
        }
        
        // break out style classes into an array  
        var classes = webui.suntheme4_2.common.splitStyleClasses(element);
        if (classes == null) {
            return false;
        }
        
        // For each styleClass, check if it's hidden and remove otherwise write
        // it back out to the class
        for (var i = 0; i < classes.length; i++) {
            if (classes[i] != null && classes[i] == styleClass) {
                classes.splice(i,1);  	
            }
        }
        element.className = classes.join(" ");
        return true;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Submit functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to insert a hidden field element in the page.
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {String} elementValue The value of the html tag.
     * @param {Node} parentForm The parent form of the html tag.
     * @return {boolean} true if successful; otherwise, false.
     */
    insertHiddenField: function(elementId, elementValue, parentForm) {
        // We have to assume that there is only one element
        // with elementId. document.getElementById, returns
        // the first one it finds, which appears to be the 
        // first one created dynamically, if more than one 
        // element is created dynamically with the same id.
        //
        // appendChild just appends even if there is an element
        // with the same id that exists.
        //
        // The assumption is that there should only be 
        // one element in the document with such an id.
        //
        // If the elementId exists just modifiy its value
        // instead of creating and appending.
        //
        
        var element = document.forms[parentForm.id].elements[elementId];
        if (element != null) {
            element.value = elementValue;            
            return true;            
        } 
        
        var newElement = document.createElement('input');
        newElement.type = 'hidden';
        newElement.id = elementId;
        newElement.value = elementValue;
        newElement.name = elementId;
        parentForm.appendChild(newElement);

        return true;
    },
    
    /**
     * Use this function to submit a virtual form.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    submitForm: function() {
        // "formToSubmit" is a literal (not virtual) form.
        // "submissionComponentId" is a component id (not client id).
        // the virtual form implementation uses _submissionComponentId
        // to determine which virtual form (if any) was submitted.
        if (webui.suntheme4_2.common._formToSubmit == null) {
            return false;
        }
        if (webui.suntheme4_2.common._submissionComponentId != null &&
                webui.suntheme4_2.common._submissionComponentId.length > 0) {
            webui.suntheme4_2.common.insertHiddenField('_submissionComponentId', 
                webui.suntheme4_2.common._submissionComponentId,
                webui.suntheme4_2.common._formToSubmit);
        }
        webui.suntheme4_2.common._formToSubmit.submit();
        return false;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    timeoutSubmitForm: function(form, submissionComponentId) {
        webui.suntheme4_2.common._formToSubmit = form;
        webui.suntheme4_2.common._submissionComponentId = submissionComponentId;
        setTimeout('webui.suntheme4_2.common.submitForm()', 0);
        return true;
    },
    
    /**
     * Helper function to submit a virtual form.
     *
     * @param {Node} form The HTML form element to submit.
     * @param {String} submissionComponentId The Id of the component submitting the form.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    leaveSubmitterTrace: function(form, submissionComponentId) {
        // This function only needs to be called in the onclick handler of 
        // an ActionSource component that appears within a -standard- table.
        // Under those circumstances, if this function is not called, then when
        // the component is clicked, the virtual form implementation will have 
        // no way of knowing that a virtual form was submitted.
        if (form != null && submissionComponentId != null && 
                submissionComponentId.length > 0) {
            webui.suntheme4_2.common.insertHiddenField('_submissionComponentId',
            submissionComponentId, form);
        }
        return true;
    },
    
    /**
     * delete a previously created element by createSubmittableArray.
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    deleteSubmittableArray: function(name, parentForm) {
        try {
            var submittableArray  = document.getElementById(name);
            if (submittableArray != null) {
                parentForm.removeChild(submittableArray);
            }
        } catch (e) {}
        return true;
    },
    
    /**
     * This method creates a hidden "select" element with id 
     * and name attributes set name, values taken from the values
     * array argument, and display labels from the labels array.
     * It adds the element to the parentForm argument.
     * <p>
     * The pupose of this method is to create an array of values
     * that can be decoded using "name" as the key from a FacesContext
     * external context's "getRequestParameterValuesMap" as an
     * array of Strings. This reduces the need of rendering hidden input
     * field and delimiting several strings so that a multiple selection
     * can be returned.
     * </p><p>
     * The labels array provides an additional piece of data
     * for use on the client, but it is not contained in the submit.
     * All values added to the select are selected so that the
     * values will be submitted.
     * </p>
     *
     * @param {String} name The element ID of the html tag 
     * @param {Node} form The HTML form element to submit.
     * @param {Array} labels
     * @param {Array} values
     * @return {Node} The newly created select element.
     * @private
     */
    createSubmittableArray: function(name, parentForm, labels, values) {
        // An attempt is made to remove a possibly previously created element
        // by this name. It always deletes an element of name from parentForm.
        webui.suntheme4_2.common.deleteSubmittableArray(name, parentForm);
        
        if (values == null || values.length <= 0) {
            return null;
        }
        
        var selections = document.createElement('select');
        selections.className = webui.suntheme4_2.theme.common.getClassName("HIDDEN");
        selections.name = name;
        selections.id = name;
        selections.multiple = true;
        
        // Start from the end of the array because
        // add puts things in at the head.
        //
        for (var i = 0; i < values.length; ++i) {
            var opt = document.createElement('option');
            opt.value = values[i];
            if (labels != null) {
                opt.label = labels[i];
            }
            opt.defaultSelected = true;
            selections.add(opt, null);
        }
        parentForm.appendChild(selections);
        return selections;
    },
    
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Visible functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Use this function to test if the specified element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {String} elementId The element ID of the html tag 
     * @return {boolean} true if visible; otherwise, false
     */
    isVisible: function(elementId) {
        if (elementId == null) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return webui.suntheme4_2.common.isVisibleElement(element);
    },
    
    /**
     * Use this function to test if the given element is visible (i.e., it
     * does not contain the hidden className).
     *
     * @param {Node} element The HTML element
     * @return {boolean} true if visible; otherwise, false
     */
    isVisibleElement: function(element) {
        if (element == null) {
            return false;
        }
        // Test for the hidden style class.
        var styleClasses = webui.suntheme4_2.common.splitStyleClasses(element); 
        return !webui.suntheme4_2.common.checkStyleClasses(styleClasses,
            webui.suntheme4_2.theme.common.getClassName("HIDDEN"));
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {String} elementId The element ID of the html tag 
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     */
    setVisible: function(elementId, visible) {
        if (elementId == null || visible == null ) {
            return false;
        }
        // Get element.
        var element = document.getElementById(elementId);
        return webui.suntheme4_2.common.setVisibleElement(element, visible);
    },
    
    /**
     * Use this function to show or hide any html element in the page
     *
     * @param {Node} element The HTML element
     * @param {boolean} visible true to make the element visible, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     */
    setVisibleElement: function(element, visible) {
        if (element == null || visible == null) {
            return false;
        }
        if (visible) {
            return webui.suntheme4_2.common.stripStyleClass(element,
                webui.suntheme4_2.theme.common.getClassName("HIDDEN"));
        } else {
            return webui.suntheme4_2.common.addStyleClass(element,
                webui.suntheme4_2.theme.common.getClassName("HIDDEN"));
        }
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.cookie");

/**
 * @class This class contains functions to manipulate cookies.
 * @static
 */
webui.suntheme4_2.cookie = {
    /**
     * This function will get the cookie value.
     *
     * @return {String} The cookie value.
     */
    get: function() {
        // Get document cookie.
        var cookie = document.cookie;

        // Parse cookie value.
        var pos = cookie.indexOf(this.$cookieName + "=");
        if (pos == -1) {
            return null;
        }

        var start = pos + this.$cookieName.length + 1;
        var end = cookie.indexOf(";", start);
        if (end == -1) {
            end = cookie.length;
        }

        // return cookie value
        return cookie.substring(start, end);
    },

    /**
     * This function will load the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    load: function() {
        // Get document cookie.
        var cookieVal = this.get();
        if (cookieVal == null) {
            return false;
        }

        // Break cookie into names and values.
        var a = cookieVal.split('&');

        // Break each pair into an array.
        for (var i = 0; i < a.length; i++) {
            a[i] = a[i].split(':');
        }

        // Set name and values for this object.
        for (i = 0; i < a.length; i++) {
            this[a[i][0]] = unescape(a[i][1]);
        }
        return true;
    },

    /**
     * This function will reset the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    reset: function() {
        // Clear cookie value.
        document.cookie = this.$cookieName + "=";
        return true;
    },

    /**
     * This function will store the cookie value.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    store: function() {
        // Create cookie value by looping through object properties
        var cookieVal = "";

        // Since cookies use the equals and semicolon signs as separators,
        // we'll use colons and ampersands for each variable we store.
        for (var prop in this) {
            // Ignore properties that begin with '$' and methods.
            if (prop.charAt(0) == '$' || typeof this[prop] == 'function') {
                continue;
            }
            if (cookieVal != "") {
                cookieVal += '&';
            }
            cookieVal += prop + ':' + escape(this[prop]);
        }
        var cookieString = this.$cookieName + "=" + cookieVal;
        if (this.$path != null) {
            cookieString += ";path=" + this.$path;
        }
        // Store cookie value.
        document.cookie = cookieString;
        return true;
    }
};

/**
 * @class This class contains functionality for scrolling.
 * @constructor This function is used to construct a javascript object for
 * maintaining scroll position via cookie.
 * @param {String} viewId
 * @param {String} path
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.scrollCookie = function(viewId, path) {    
    // All predefined properties of this object begin with '$' because
    // we don't want to store these values in the cookie.
    this.$cookieName = viewId;
    this.$path = path;

    // Default properties.
    this.left = "0";
    this.top  = "0";

    // This function will load the cookie and restore scroll position.
    this.restore = function() {
        // Load cookie value.
        this.load();
        scrollTo(this.left, this.top);
        return true;
    };

    // This function will set the cookie value.
    this.set = function() {
        var documentElement = window.document.documentElement;
        if (documentElement && documentElement.scrollTop) {
            this.left = documentElement.scrollLeft;
            this.top  = documentElement.scrollTop;
        } else {
            this.left = window.document.body.scrollLeft;
            this.top  = window.document.body.scrollTop;
        }
        // if the left and top scroll values are still null
        // try to extract it assuming the browser is IE
        if (this.left == null && this.top == null) {
            this.left = window.pageXOffset;
            this.top = window.pageYOffset;
        }
        // Store cookie value.
        this.store();
        return true;
    };
    return true;
};

// Inherit cookie properties.
webui.suntheme4_2.scrollCookie.prototype = webui.suntheme4_2.cookie;

/**
 * @class This class contains functions used to maintain focus and scroll position.
 * <p>
 * There can be an initial focus element and a default focus element. The
 * initial focus element is identifed by the "focusElementId" argument.
 * This argument is typically null on the first display of the page. If
 * the Body component is not preserving the focus then it may also be null,
 * at other times, since it represents the element id that last received
 * the focus before the request.
 * </p><p>
 * Whenever the page is displayed and "focusElementId" is null
 * "defaultFocusElementId" will receive the focus if it is not null. This id is
 * defined by the application using the Body "focus" attribute. If the
 * application wants to control the focus in all cases then it should set
 * the Body component "preserveFocus" attribute to "false". The application then
 * explicitly sets the Body "focus" attribute to the element id to receive
 * focus on every request/response.
 * </p><p>
 * In order to preserve focus across requests, the "focusElementFieldId"
 * element is used to preserve the id of the element that receives the focus
 * last. It is a hidden field that is submitted in the
 * request. See the "com.sun.webui.jsf.util.FocusManager" class for
 * information on how the focus is managed. This field exists in all
 * forms, since that it is the only way to guarantee that the last
 * element that received the focus is sent to the server. Which form
 * is submitted can never be known.
 * </p>
 * @constructor This function is used to construct a body object.
 * @param {String} viewId Used to name the scroll cookie
 * @param {String} path A member of the scroll cookie
 * @param {String} defaultFocusElementId The HTML element id that will receive focus.
 * @param {String} focusElementId The id of the element to receive the initial focus.
 * @param {String} focusElementFieldId The id of a hidden field to maintain
 * the id of the last element to have the focus.
 * @param {boolean} preserveScroll if true or not defined the scroll position is 
 * maintained else scroll is not maintained.
 */
webui.suntheme4_2.body = function(viewId, path, defaultFocusElementId, 
	focusElementId, focusElementFieldId, preserveScroll)  {
    /**
     * The id of the HTML element to receive the focus, if the
     * element identified in focusElementFieldId cannot receive the focus.
     */
    this.defaultFocusId = defaultFocusElementId;

    /**
     * The id of a hidden input element whose value is the id
     * of the HTML element to receive the focus. It is set by
     * the focusListener and calls to setFocusBy{Id,Element}.
     */
    this.focusElementFieldId = focusElementFieldId;

    /**
     * The element id to receive the preserved, or initial focus.
     * This member should not be referenced once the onload listener
     * has been invoked. After that point the hidden field will have
     * have the element with the focus. We do this so that two locations
     * do not have to be kept in sync. Developers can just set the focus
     * to the element itself and the focus handler will manage the
     * focus persisitence.
     */
    this.focusElementId = focusElementId;

    // "==" also handles "null"
    //
    this.preserveScroll = (preserveScroll == null)
        ? true : new Boolean(preserveScroll).valueOf();

    /**
     * Create the scroll cookie object.
     */
    if (this.preserveScroll == true) {
	this.scrollCookie = new webui.suntheme4_2.scrollCookie(viewId, path);
    }

    /**
     * According to HTML spec only these elements have
     * "onfocus" which we will equate to "focus".
     * A, AREA, BUTTON, INPUT, LABEL, SELECT, TEXTAREA
     * However just check for a non null "focus" or 
     * catch the exception when trying to reference it.
     * Returns true if the element has a "focus" method, is not
     * disabled, and isVisible, else false.
     *
     * @param {Node} element The DOM node to have focus.
     * @return {boolean} true if DOM Node can have focus.
     */
    this.canAcceptFocus = function(element) {
	var result = false;
	try {
	    result = element != null && element.focus && !element.disabled
		&& element.type != "hidden"
		&& webui.suntheme4_2.common.isVisible(element.id);
	} catch(err) {}
	return result;
    };

    /**
     * Record the id of the element that has just receivied focus.
     * This is called whenever an element receives the focus.
     * This is set on the document so that the cursor entering the
     * window does not trigger this listener.
     *
     * @param {Event} event The object generated by the focus event.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.focusListener = function(event) {
	// If it's not an element node just return
	//
	var node = null;
	var isElementNode = false;
	
	// is IE 
	//
	if (document.attachEvent) {
	    node = event.srcElement;
	
	    // We have to hard code "1" as the Node.ELEMENT_NODE in
	    // ie, because ie does not make the constant accessible.
	    //
	    isElementNode = (node == null ? false : node.nodeType == 1);
	} else {
	    node = event.target;
	    isElementNode = node.nodeType == Node.ELEMENT_NODE;
	}

	if (isElementNode) {
	    // Note that there is no reliable way to set
	    // focus to some other element if the event element
	    // deemed to receive the focus can't accept the focus.
	    //
	    this.updateFocusElementField(node);
	}
	return true;
    };

    /**
     * Set the initial focus and the scroll position.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    this.onLoadListener = function() {
	// register the focus listener first.
	// Then call "setDefaultFocus" using "setTimeout" to
	// allow javascript processing to complete, probably
	// to allow "onload" to complete. The focus listener
	// will update the hidden focus fields as a result
	// of the call to "focus" in setDefaultFocus.
	//

	// Add the focus listener, in the onload to prevent
	// recursive calls from calling setDefaultFocus.
	//
        if (webui.suntheme4_2.browser.isIe()) {
            webui.suntheme4_2.dojo.connect(document, "onfocusin", this, "focusListener");
        } else {
            webui.suntheme4_2.dojo.connect(window, "onfocus", this, "focusListener");
        }

        // Rely on the focus listener to update the focus
        // hidden fields by catching the 'element.focus()' in
        // setDefaultFocus
        //
        this.setInitialFocus();

	// Set up the scroll position after the focus has been
	// restored. Need to make sure that this takes into
	// account the default focus that was just set.
	//
	return this.setDefaultScrollPosition();
    };

    /**
     * Update the page's scroll position.
     *
     * @param {Event} event The object generated by the onUnload event.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.onUnloadListener = function(event) {
	return this.storeScrollPosition();
    };

    /**
     * Set the default focus to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this.defaultFocusId" is not null it will receive the focus; 
     * otherwise, no focus is set.
     * </p>
     * @return {boolean} false if a default focus cannot be established, else true.
     */
    this.setDefaultFocus = function() {
        // HTML elements may not have been added, yet.
        if (this.defaultFocusId != null) {
            var domNode = document.getElementById(this.defaultFocusId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this.setDefaultFocus(); }, 10);
            }

            // Focus not set try the default.
            //
            if (this.setFocusById(this.defaultFocusId)) {
                return true;
            }
        }

	/* For now it doesn't sound like a good idea to ever set
	 * a "heuristic" default focus. It is better for screen readers to start
	 * from the top of the page which we will assume that that
	 * browser starts from there when focus is not set explicitly.
	 * This code can be removed, but left it in case we want to
	 * for some reason.

	// No previously set focus element and no default.
	// Find the first focusable element in the first available form.
	//
	for each (var f in window.document.forms) {
	    for each (var e in f.elements) {
		if (this.setFocusByElement(e)) {
		    return true;
		}
	    }
	}
	// If there is no form, set on the first available link
	//
	for each (var l in window.document.links) {
	    if (this.setFocusByElement(l)) {
		return true;
	    }
	}
	*/
	return false;
    };

    /**
     * This method is invoked in the onload listener, body.onLoadListener.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    this.setDefaultScrollPosition = function() {
	if (!this.preserveScroll) {
	    return false;
	}
	// # char found, anchor being used. forego scrolling.
	// CR 6342635. 
	//
        if (window.location.href.indexOf('#') == -1) {
	    this.scrollCookie.restore(); 
	} else {
	    // destroy the recent scroll data
	    //
	    this.scrollCookie.reset();
	}
        return true;
    };

    /**
     * Set the initial focus by restoring the focus from a previous
     * request or to the application's chosen default focus element.
     * This method should only be called once to prevent recursive
     * calls since it calls "focus()" on the focus element.
     * Called currently from the onload listener.
     * <p>
     * If "this.focusElementId" is not null it will receive the focus.
     * If that element can't receive the focus then the application defined 
     * "this.defaultFocusId" receives the focus. If that element cannot receive 
     * the focus, no focus is set.
     * </p>
     * @return {boolean} false if focus cannot be established, else true.
     */
    this.setInitialFocus = function() {
        // HTML elements may not have been added, yet.
        if (this.focusElementId != null) {
            var domNode = document.getElementById(this.focusElementId);
            if (domNode == null) {
                var _this = this; // Closure magic.
                return setTimeout(function() { _this.setInitialFocus(); }, 10);
            }

            // Try to set focus to "this.focusElementId". If this fails
            // fallback to the app defined default 
            // "this.defaultFocusElementId", if there is one.
            //
            if (this.setFocusById(this.focusElementId)) {
                return true;
            }
        }
        return this.setDefaultFocus();
    };

    /**
     * Set the focus on "focusElement".
     * If focus can be set returns true else false.
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.setFocusByElement = function(focusElement) {
	if (focusElement == null || !this.canAcceptFocus(focusElement)) {
	    return false;
	}

	// canAcceptFocus tests the existence of the "focus" handler.
	// So it is safe to call it outside of a try/catch statement.
	// This should trigger the focus listener.
        try {
            // Still need try/catch because canAcceptFocus doesn't account for 
            // when parent is invisible. For example, the table's sort panel 
            // closes during page submit making focus element invisible.
            focusElement.focus();
        } catch(err) {}

	// Assume that this update is performed by the 
	// focus listener. This policy was changed in order to 
	// call "setDefaultFocus" using "setTimeout" in order for
	// javascript to have time to be evaluated, probably for
	// on load processing to complete.
	//this.updateFocusElementField(focusElement);
	return true;
    };

    /**
     * Set the focus on element with id "fid".
     * If focus can be set returns true else false.
     *
     * @param {String} fid The id of the DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.setFocusById = function(fid) {
	if (fid == null || fid.length == 0) {
	    return false;
	}
	return this.setFocusByElement(document.getElementById(fid));
    };

    /**
     * This method is invoked in the onunload event listener
     * body.onUnloadListener
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    this.storeScrollPosition = function() {
	if (!this.preserveScroll) {
	    return false;
	}
	try {
	    this.scrollCookie.set(); 
	} catch (e) {
	}
        return true; 
    };

    /** 
     * Update the hidden field that maintains the last element to 
     * receive the focus. If the body has multiple forms every form's
     * hidden field is updated with the "focusElement".
     *
     * @param {Node} focusElement The DOM node to have focus.
     * @return {boolean} true if successful; otherwise, false.
     */
    this.updateFocusElementField = function(focusElement) {
	// Don't know if we'll have issues if multiple forms contain
	// an element with the same id. I know getElementById gets
	// confused.
	//

        if (focusElement == null) {
	    return false;
	}
	// Get the form that contains the focus element.
	//
	for (var i = 0;  i < document.forms.length; ++i) {
	    var form = document.forms[i];
            var field = null;

	    // Get the hidden field that maintains the focus element id.
	    // If it exists return it. We know its name is the same
	    // as its id.
	    //
	    try {
		field = form.elements[this.focusElementFieldId];
		if (field != null) {
		    field.value = focusElement.id;
		    continue;
		}
	    } catch (e) {
		// the array access of a non existent element
		// probably threw exception so create the field.
	    }
		
	    // If it doesn't exist create it.
	    // and add it to the form.
	    //
	    field = document.createElement('input');
	    field.type = 'hidden';
	    field.id = this.focusElementFieldId;
	    field.name = this.focusElementFieldId;
	    field.value = focusElement.id;
	    form.appendChild(field);
	}
	return true;
    };

    // The focus listener is set on the document so that the cursor 
    // entering the window does not trigger this listener. 
    this.onLoadListener();

    // If we are not preserving scroll don't add the unload listener.
    if (this.preserveScroll == true) {
        webui.suntheme4_2.dojo.addOnUnload(this, "onUnloadListener");
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.table");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.formElements");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// button functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for button components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.button
 */
webui.suntheme4_2.button = {
    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The element id.
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See webui.suntheme4_2.widget.button
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize button.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var widget = webui.suntheme4_2.dijit.byId(props.id);
        if (widget == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set functions
        widget.domNode.isSecondary = webui.suntheme4_2.button.isSecondary;
        widget.domNode.setSecondary = webui.suntheme4_2.button.setSecondary;
        widget.domNode.isPrimary = webui.suntheme4_2.button.isPrimary;
        widget.domNode.setPrimary = webui.suntheme4_2.button.setPrimary;
        widget.domNode.isMini = webui.suntheme4_2.button.isMini;
        widget.domNode.setMini = webui.suntheme4_2.button.setMini;
        widget.domNode.getDisabled = webui.suntheme4_2.button.getDisabled;
        widget.domNode.setDisabled = webui.suntheme4_2.button.setDisabled;
        widget.domNode.getVisible = webui.suntheme4_2.button.getVisible;
        widget.domNode.setVisible = webui.suntheme4_2.button.setVisible;
        widget.domNode.getText = webui.suntheme4_2.button.getText;
        widget.domNode.setText = webui.suntheme4_2.button.setText;
        widget.domNode.doClick = webui.suntheme4_2.button.click;

        return true;
    },

    /**
     * Simulate a mouse click in a button. 
     *
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).click();
     */
    click: function() {
        return this.click();
    },

    /**
     * Get the textual label of a button. 
     *
     * @return {String} The element value.
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getText: function() {
        return this.getProps().value;
    },

    /**
     * Set the textual label of a button. 
     *
     * @param {String} text The element value
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setText: function(text) {
        return this.setProps({value: text});
    },

    /**
     * Use this function to show or hide a button. 
     *
     * @param {boolean} show true to show the element, false to hide the element
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({visible: boolean});
     */
    setVisible: function(show) {
        if (show == null) {
            return null;
        }
        return this.setProps({visible: show});
    },

    /**
     * Use this function to find whether or not this is visible according to our
     * spec.
     *
     * @return {boolean} true if visible; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().visible;
     */
    getVisible: function() {
        return this.getProps().visible;
    },

    /**
     * Test if button is set as "primary".
     *
     * @return {boolean} true if primary; otherwise, false for secondary
     * @deprecated Use document.getElementById(id).getProps().primary;
     */
    isPrimary: function() {
        return this.getProps().primary;
    },

    /**
     * Set button as "primary".
     *
     * @param {boolean} primary true for primary, false for secondary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: boolean});
     */
    setPrimary: function(primary) {
        if (primary == null) {
            return null;
        }
        return this.setProps({primary: primary});
    },

    /**
     * Test if button is set as "secondary".
     *
     * @return {boolean} true if secondary; otherwise, false for primary
     * @deprecated Use !(document.getElementById(id).getProps().primary);
     */
    isSecondary: function() {
        return !(this.getProps().primary);
    },

    /**
     * Set button as "secondary".
     *
     * @param {boolean} secondary true for secondary, false for primary
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({primary: false});
     */
    setSecondary: function(secondary) {
        if (secondary == null) {
            return null;
        }
        return this.setProps({primary: !secondary});
    },

    /**
     * Test if button is set as "mini".
     *
     * @return {boolean} true if mini; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().mini;
     */
    isMini: function() {
        return this.getProps().mini;
    },

    /**
     * Set button as "mini".
     *
     * @param {boolean} mini true for mini, false for standard button
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({mini: boolean});
     */
    setMini: function(mini) {
        if (mini == null) {
            return null;
        }
        return this.setProps({mini: mini});
    },

    /**
     * Test disabled state of button.
     *
     * @return {boolean} true if disabled; otherwise, false
     * @deprecated Use document.getElementById(id).getProps().disabled;
     */
    getDisabled: function() {
        return this.getProps().disabled;
    },

    /**
     * Test disabled state of button.
     *
     * @param {boolean} disabled true if disabled; otherwise, false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(disabled) {
        if (disabled == null) {
            return null;
        }
        return this.setProps({disabled: disabled});
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// checkbox functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for checkbox components.
 * @static
 * 
 * @deprecated See webui.suntheme4_2.widget.checkbox
 */
webui.suntheme4_2.checkbox = {
    /**
     * Set the disabled state for the given checkbox element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {
        return webui.suntheme4_2.rbcb.setDisabled(elementId, disabled,
            "checkbox", "Cb", "CbDis");
    },

    /** 
     * Set the disabled state for all the checkboxes in the check box
     * group identified by controlName. If disabled
     * is set to true, the check boxes are shown with disabled styles.
     *
     * @param {String} controlName The checkbox group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return webui.suntheme4_2.rbcb.setGroupDisabled(controlName,
            disabled, "checkbox", "Cb", "CbDis");
    },

    /**
     * Set the checked property for a checkbox with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return webui.suntheme4_2.rbcb.setChecked(elementId, checked,
            "checkbox");
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// dropdown functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for dropDown components.
 * @static
 * 
 * @deprecated See webui.suntheme4_2.widget.dropDown
 */
webui.suntheme4_2.dropDown = {
    /**
     * Use this function to access the HTML select element that makes up
     * the dropDown.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the dropDown).
     * @return {Node} a reference to the select element. 
     * @deprecated Use document.getElementById(elementId).setSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * This function is invoked by the choice onselect action to set the
     * selected, and disabled styles.
     *
     * Page authors should invoke this function if they set the 
     * selection using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed();
     */
    changed: function(elementId) {         
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    },

    /**
     * Set the disabled state for given dropdown element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a dropdown using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({ disabled: disabled});
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the dropDown. If no option is selected, this
     * function returns null.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is
     * selected. 
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// field functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for field components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.field
 */
webui.suntheme4_2.field = {
    /**
     * Use this function to get the HTML input or textarea element
     * associated with a TextField, PasswordField, HiddenField or TextArea
     * component.
     *
     * @param {String} elementId The element ID of the field 
     * @return {Node} the input or text area element associated with the field component
     * @deprecated Use document.getElementById(elementId).getInputElement()
     */
    getInputElement: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getInputElement();
        }
        return null;
    },

    /**
     * Use this function to get the value of the HTML element 
     * corresponding to the Field component.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} the value of the HTML element corresponding to the Field component 
     * @deprecated Use document.getElementById(id).getProps().value;
     */
    getValue: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getProps().value;
        }
        return null;
    },

    /**
     * Use this function to set the value of the HTML element 
     * corresponding to the Field component
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newValue The new value to enter into the input element Field component 
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({value: "text"});
     */
    setValue: function(elementId, newValue) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({value: newValue});
        }
        return null;
    },

    /** 
     * Use this function to get the style attribute for the field. 
     * The style retrieved will be the style on the span tag that 
     * encloses the (optional) label element and the input element.
     *
     * @param {String} elementId The element ID of the Field component
     * @return {String} The style property of the field.
     * @deprecated Use document.getElementById(id).getProps().style;
     */
    getStyle: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getProps().style;
        }
        return null;
    },

    /**
     * Use this function to set the style attribute for the field. 
     * The style will be set on the <span> tag that surrounds the field.
     *
     * @param {String} elementId The element ID of the Field component
     * @param {String} newStyle The new style to apply
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({style: newStyle});
     */
    setStyle: function(elementId, newStyle) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({style: newStyle});
        }
        return null;
    },

    /**
     * Use this function to disable or enable a field. As a side effect
     * changes the style used to render the field. 
     *
     * @param {String} elementId The element ID of the field 
     * @param {boolean} newDisabled true to disable the field, false to enable the field
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, newDisabled) {  
        if (newDisabled == null) {
            return null;
        }
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: newDisabled});
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// hyperlink functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for hyperlink components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.hyperlink
 */
webui.suntheme4_2.hyperlink = {
    /**
     * This function is used to submit a hyperlink.
     * <p>
     * Note: Params are name value pairs but all one big string array so 
     * params[0] and params[1] form the name and value of the first param.
     * </p>
     *
     * @params {Object} hyperlink The hyperlink element
     * @params {String} formId The form id
     * @params {Object} params Name value pairs
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated See webui.suntheme4_2.widget.hyperlink
     */
    submit: function(hyperlink, formId, params) {
        // Need to test widget for tab and common task components. If a widget 
        // does not exist, fall back to the old code.
	var widget = webui.suntheme4_2.dijit.byId(hyperlink.id);
	if (widget == null) {
            // If a widget does not exist, we shall create one in order to call
            // the submit function directly.
            webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.hyperlink");
            widget = new webui.suntheme4_2.widget.hyperlink({id: hyperlink.id});
	}
        return widget.submitFormData(formId, params);
    },

    /**
     * Use this function to access the HTML img element that makes up
     * the icon hyperlink.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the outter most tag enclosing the HTML img element).
     * @return {Node} The HTML image element.
     * @deprecated Use document.getElementById(elementId).getProps().enabledImage;
     */
    getImgElement: function(elementId) {
        // Need to test widget for alarmStatus, jobstatus, and notification phrase
        // components. If a widget does not exist, fall back to the old code.
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        var props = (widget) ? widget.getProps() : null;
        if (props && props.enabledImage) {
            var imgWidget = webui.suntheme4_2.dijit.byId(props.enabledImage.id);
            if (imgWidget != null) {
                return imgWidget.domNode;    
            }
        }

        // Image hyperlink is now a naming container and the img element id 
        // includes the ImageHyperlink parent id.
        if (elementId != null) {
            var parentid = elementId;
            var colon_index = elementId.lastIndexOf(":");
            if (colon_index != -1) {
                parentid = elementId.substring(colon_index + 1);
            }
            return document.getElementById(elementId + ":" + parentid + "_image");
        }
        return document.getElementById(elementId + "_image");
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// jumpDropDown functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for jumpDropDown components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.dropDown
 */
webui.suntheme4_2.jumpDropDown = {
    /**
     * This function is invoked by the jumpdropdown onchange action to set the
     * form action and then submit the form.
     *
     * Page authors should invoke this function if they set the selection using 
     * JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed()
     */
    changed: function(elementId) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// listbox functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for listbox components.
 * @static
 * 
 * @deprecated See webui.suntheme4_2.widget.listbox
 */
webui.suntheme4_2.listbox = {
    /**
     * Use this function to access the HTML select element that makes up
     * the list. 
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * assigned to the span tag enclosing the HTML elements that make up
     * the list).
     * @return {Node} The HTML select element.
     * @deprecated Use document.getElementById(elementId).getSelectElement()
     */
    getSelectElement: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectElement();
        }
        return null;
    },

    /**
     * This function is invoked by the list onselect action to set the selected, 
     * and disabled styles.
     *
     * Page authors should invoke this function if they set the selection
     * using JavaScript.
     *
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).changed();
     */
    changed: function(elementId) {         
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.changed();
        }
        return false;
    },

    /**
     * Invoke this JavaScript function to set the enabled/disabled state
     * of the listbox component. In addition to disabling the list, it
     * also changes the styles used when rendering the component. 
     *
     * Page authors should invoke this function if they dynamically
     * enable or disable a list using JavaScript.
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(elementId).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the value of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The value of the selected option, or null if none is
     * selected.
     * @deprecated Use document.getElementById(elementId).getSelectedValue();
     */
    getSelectedValue: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedValue();
        }
        return null;
    },

    /**
     * Invoke this JavaScript function to get the label of the first
     * selected option on the listbox. If no option is selected, this
     * function returns null. 
     * 
     * @param {String} elementId The component id of the JSF component (this id is
     * rendered in the div tag enclosing the HTML elements that make up
     * the list).
     * @return {String} The label of the selected option, or null if none is selected.
     * @deprecated Use document.getElementById(elementId).getSelectedLabel();
     */
    getSelectedLabel: function(elementId) { 
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.getSelectedLabel();
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Generic checkbox and radio button functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for rbcbGroup components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.rbcbGroup
 */
webui.suntheme4_2.rbcb = {
    /**
     * 
     * @param {String} elementId The element Id.
     * @param {boolean} checked true or false
     * @param {String} type
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */ 
    setChecked: function(elementId, checked, type) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({checked: checked});
        }
        return null; 
    },

    /**
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @param {String} type
     * @param {String} enabledStyle
     * @param {String} disabledStyle
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */ 
    setDisabled: function(elementId, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null; 
    },

    /** 
     * Set the disabled state for all radio buttons with the given controlName.
     * If disabled is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id
     * @param {String} formName The name of the form containing the element
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled, type, enabledStyle,
            disabledStyle) {
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        if (widget) {
            return widget.setProps({disabled: disabled});
        }
        return null;
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// radiobutton functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for radioButton components.
 * @static
 *
 * @deprecated See webui.suntheme4_2.widget.radioButton
 */
webui.suntheme4_2.radiobutton = {
    /**
     * Set the disabled state for the given radiobutton element Id. If the disabled 
     * state is set to true, the element is shown with disabled styles.
     *
     * @param {String} elementId The element Id.
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setDisabled: function(elementId, disabled) {    
        return webui.suntheme4_2.rbcb.setDisabled(elementId, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the disabled state for all the radio buttons in the radio button
     * group identified by controlName. If disabled
     * is set to true, the check boxes are displayed with disabled styles.
     *
     * @param {String} controlName The radio button group control name
     * @param {boolean} disabled true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({disabled: boolean});
     */
    setGroupDisabled: function(controlName, disabled) {    
        return webui.suntheme4_2.rbcb.setGroupDisabled(controlName, disabled, 
            "radio", "Rb", "RbDis");
    },

    /**
     * Set the checked property for a radio button with the given element Id.
     *
     * @param {String} elementId The element Id
     * @param {boolean} checked true or false
     * @return {boolean} true if successful; otherwise, false.
     * @deprecated Use document.getElementById(id).setProps({checked: boolean});
     */
    setChecked: function(elementId, checked) {
        return webui.suntheme4_2.rbcb.setChecked(elementId, checked, "radio");
    }
};

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// upload functions
// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

/**
 * @class This class contains functions for upload components.
 * @static
 */
webui.suntheme4_2.upload = {
    /**
     * Use this function to get the HTML input element associated with the
     * Upload component.  
     * @param {String} elementId The client id of the Upload component
     * @return {Node} the input element associated with the Upload component
     * else null if elementId is null or "".
     */
    getInputElement: function(elementId) { 
        if (elementId == null || elementId == "") {
	    return null;
	}

	// The upload component MUST always render the input element
	// with the following suffix on the id 
	// "_com.sun.webui.jsf.upload".
	// This "binds" this version of the component to this theme
	// version.
	// This will change when "field" becomes a widget.
	//
        var element = document.getElementById(elementId + 
            "_com.sun.webui.jsf.upload");
        if (element && element.tagName == "INPUT") { 
            return element; 
        } else {
	    return null;
	}
    },

    /**
     * Use this function to disable or enable a upload. As a side effect
     * changes the style used to render the upload. 
     *
     * @param {String} elementId The client id of the upload component.
     * @param {boolean} disabled true to disable the upload, false to enable the upload
     * @return {boolean} true if successful; otherwise, false.
     */
    setDisabled: function(elementId, disabled) {  

        if (elementId == null || elementId == "" || 
		disabled == null || disabled == "") {
            // must supply an elementId && state
            return false;
        }
        var input = webui.suntheme4_2.upload.getInputElement(elementId); 
        if (input == null) {
            // specified elementId not found
            return false;
        }
        input.disabled = disabled;
	return true;
    },

    /**
     * Set the encoding type of the form to "multipart/form-data".
     * 
     *
     * @param {String} elementId The client id of the upload component.
     * @return {boolean} true if encoding type can be set, else false.
     */
    setEncodingType: function(elementId) { 
	if (elementId == null || elementId == "") {
	    return false;
	}

        var upload = webui.suntheme4_2.upload.getInputElement(elementId); 
        var form = upload != null ? upload.form : null;
	if (form != null) {

            // form.enctype does not work for IE, but works Safari
            // form.encoding works on both IE and Firefox
	    //
            if (webui.suntheme4_2.browser.isSafari()) {
                form.enctype = "multipart/form-data";
            } else {
                form.encoding = "multipart/form-data";
            }
	    return true;
        }
	return false;
    },

    /**
     * Create a hidden field with id "preservePathId" and add a listener
     * to the upload's input element, "uploadId". The listener is
     * is added for the onchange event of the upload's input field,
     * see preservePathListener.
     *
     * @param {String} uploadId The client id of the upload component.
     * @param {String} preservePathId
     * @return {boolean} true if the hidden element is created and a listener is
     * added, else false.
     */
    preservePath: function(uploadId, preservePathId) {
	if (uploadId == null || uploadId == "" ||
		preservePathId == null || preservePathId == "") {
	    return false;
	}

	// If there is no upload component, don't do anything.
	// I'm not sure if there is a timing issue here.
	//
	var uploadElement = webui.suntheme4_2.upload.getInputElement(uploadId);
	if (uploadElement == null) {
	    return false;
	}
	var theForm = uploadElement.form;

	// Create the change listener.
	// The event target/srcElement is the upload input element
	// its value is the changed value, save it in the 
	// preservePath hidden field.
	//
	var onChangeListener = function(evt) {

	    // Is IE
	    if (document.attachEvent) {
		node = evt.srcElement;
	    } else {
		node = evt.target;
	    }
	    // node is the upload input element
	    //
	    var preservePath = null;
	    try {
		preservePath = theForm.elements[preservePathId];
	    } catch (e) {
	    }

	    // If the hidden field isn't there create it and assign
	    // the node's value
	    //
	    if (preservePath != null) {
		preservePath.value = node.value;
	    } else {
		webui.suntheme4_2.common.insertHiddenField(preservePathId, 
			node.value, theForm);
	    }
	    return true;
	};

	if (uploadElement.addEventListener) {
	    uploadElement.addEventListener('change', onChangeListener, true);
	} else {
	    uploadElement.attachEvent('onchange', onChangeListener);
	}
	return true;
    }
};
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");

/** 
 * @class This class contains functions for table components.
 * <p>
 * Once the table is rendered, you will be able to invoke functions directly on 
 * the HTML element. For example:
 * </p><p><pre>
 * var table = document.getElementById("form1:table1");
 * var count = table.getAllSelectedRowsCount();
 * </pre></p><p>
 * Note: It is assumed that formElements.js has been included in the page. In
 * addition, all given HTML element IDs are assumed to be the outter most tag
 * enclosing the component.
 * </p>
 * @static
 */
webui.suntheme4_2.table = {
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Public functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected
     * by a delete action. This functionality requires the selectId property of
     * the tableColumn component and hiddenSelectedRows property of the
     * tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmDeleteSelectedRows: function() {
        return this.confirmSelectedRows(this.deleteSelectionsMsg);
    },

    /**
     * This function is used to confirm the number of selected components (i.e., 
     * checkboxes or radiobuttons used to de/select rows of the table), affected by
     * an action such as edit, archive, etc. This functionality requires the 
     * selectId property of the tableColumn component and hiddenSelectedRows
     * property of the tableRowGroup component to be set.
     * <p>
     * If selections are hidden from view, the confirmation message indicates the
     * number of selections not displayed in addition to the total number of
     * selections. If selections are not hidden, the confirmation message indicates
     * only the total selections.
     * </p>
     *
     * @param {String} message The confirmation message (e.g., Archive all selections?).
     * @return {boolean} A value of true if the user clicks the "OK" button, or 
     * false if the user clicks the "Cancel" button.
     */
    confirmSelectedRows: function(message) {
        // Get total selections message.
        var totalSelections = this.getAllSelectedRowsCount();
        var totalSelectionsArray = this.totalSelectionsMsg.split("{0}");
        var totalSelectionsMsg = totalSelectionsArray[0] + totalSelections;

        // Append hidden selections message.
        var hiddenSelections = this.getAllHiddenSelectedRowsCount();
        if (hiddenSelections > 0) {
            // Get hidden selections message.
            var hiddenSelectionsArray = this.hiddenSelectionsMsg.split("{0}");
            var hiddenSelectionsMsg = hiddenSelectionsArray[0] + hiddenSelections;

            totalSelectionsMsg = hiddenSelectionsMsg + totalSelectionsMsg;
        }
        return (message != null)
            ? confirm(totalSelectionsMsg + message)
            : confirm(totalSelectionsMsg);
    },

    /**
     * This function is used to toggle the filter panel from the filter menu. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     * <p>
     * If the "Custom Filter" option has been selected, the table filter panel is 
     * toggled. In this scenario, false is returned indicating the onChange event,
     * generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * If the "Custom Filter Applied" option has been selected, no action is taken.
     * Instead, the is selected filter menu is reverted back to the "Custom Filter" 
     * selection. In this scenario, false is also returned indicating the onChange 
     * event, generated by the table filter menu, should not be allowed to continue.
     * </p><p>
     * For all other selections, true is returned indicating the onChange event, 
     * generated by the table filter menu, should be allowed to continue.
     * </p>
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    filterMenuChanged: function() {
        // Validate panel IDs.
        if (this.panelToggleIds == null || this.panelToggleIds.length == 0) {
            return false;
        }

        // Get filter menu.
        var menu = webui.suntheme4_2.dropDown.getSelectElement(
            this.panelToggleIds[this.FILTER]);
        if (menu == null) {
            return true;
        }

        // Test if table filter panel should be opened.
        if (menu.options[menu.selectedIndex].value == this.customFilterOptionValue) {
            this.toggleFilterPanel();
            return false;
        } else if (menu.options[menu.selectedIndex].
                value == this.customFilterAppliedOptionValue) {
            // Set selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].value == this.customFilterOptionValue) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            return false;
        }
        return true;
    },

    /**
     * This function is used to get the number of selected components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select 
     * rows of the table). This functionality requires the selectId property of the
     * tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @return {int} The number of components selected in the current page.
     */
    getAllSelectedRowsCount: function() {
        return this.getAllHiddenSelectedRowsCount() +
            this.getAllRenderedSelectedRowsCount();
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently hidden from view. This functionality requires 
     * the selectId property of the tableColumn component and hiddenSelectedRows
     * property of the table component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllHiddenSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupHiddenSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table), currently rendered. This functionality requires 
     * the selectId property of the tableColumn component to be set.
     *
     * @return {int} The number of selected components hidden from view.
     */
    getAllRenderedSelectedRowsCount: function() {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return count;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            count = count + this._getGroupRenderedSelectedRowsCount(this.groupIds[i]);
        }
        return count;
    },

    /**
     * This function is used to initialize all rows displayed in the table when the
     * state of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    initAllRows: function() {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._initGroupRows(this.groupIds[i]);
        }
        return true;
    },

    /**
     * This function is used to toggle the filter panel open or closed. This
     * functionality requires the filterId of the table component to be set. In 
     * addition, the selected value must be set as well to restore the default
     * selected value when the embedded filter panel is closed.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    toggleFilterPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle filter panel.
        this._togglePanel(this.panelIds[this.FILTER],
        this.panelFocusIds[this.FILTER], this.panelToggleIds[this.FILTER]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to toggle the preferences panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     */
    togglePreferencesPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Toggle preferences panel.
        this._togglePanel(this.panelIds[this.PREFERENCES],
        this.panelFocusIds[this.PREFERENCES], this.panelToggleIds[this.PREFERENCES]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Private functions
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * This function is used to initialize HTML element properties with Object
     * literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {String} id The HTML element ID for the component.
     *
     * // Panel Properties
     * @config {Array} panelIds An array of embedded panel IDs.</li>
     * @config {Array} panelFocusIds An array of IDs used to set focus for open panels.</li>
     * @config {Array} panelToggleIds An array of IDs used to toggle embedded panels.</li>
     * @config {Array} panelToggleIconsOpen An array of toggle icons for open panels.</li>
     * @config {Array} panelToggleIconsClose An array of toggle icons for closed panels.</li>
     *
     * // Filter Properties
     * @config {String} basicFilterStyleClass The style class for basic or no filters.</li>
     * @config {String} customFilterStyleClass The style class for custom filters.</li>
     * @config {String} customFilterOptionValue The custom filter menu option value.</li>
     * @config {String} customFilterAppliedOptionValue The custom filter applied menu option value.</li>
     *
     * // Sort Panel Properties
     * @config {Array} sortColumnMenuIds An array of HTML element IDs for sort column menu components.</li>
     * @config {Array} sortOrderMenuIds An array of HTML element IDs for sort order menu components.</li>
     * @config {Array} sortOrderToolTips An array of tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsAscending An array of ascending tool tips used for sort order menus.</li>
     * @config {Array} sortOrderToolTipsDescending An array of descending tool tips used for sort order menus.</li>
     * @config {String} duplicateSelectionMsg The message displayed for duplicate menu selections.</li>
     * @config {String} missingSelectionMsg The message displayed for missing menu selections.</li>
     * @config {String} selectSortMenuOptionValue The sort menu option value for the select column.</li>
     * @config {boolean} hiddenSelectedRows Flag indicating that selected rows might be currently hidden from view.</li>
     * @config {boolean} paginated Flag indicating table is in pagination mode.</li>
     *
     * // Group Properties
     * @config {String} selectRowStylClass The style class for selected rows.</li>
     * @config {Array} selectIds An arrary of component IDs used to select rows of the table.</li>
     * @config {Array} groupIds An array of TableRowGroup IDs rendered for the table.</li>
     * @config {Array} rowIds An array of row IDs for rendered for each TableRowGroup.</li>
     * @config {Array} hiddenSelectedRowCounts An array of selected row counts hidden from view.</li>
     * @config {String} hiddenSelectionsMsg The hidden selections message for confirm dialog.</li>
     * @config {String} totalSelectionsMsg The total selections message for confirm dialog.</li>
     * @config {String} deleteSelectionsMsg The delete selections message for confirm dialog.</li>
     *
     * // Group Panel Properties
     * @param {String} columnFooterId ID for column footer.</li>
     * @param {String} columnHeaderId ID for column header.</li>
     * @param {String} tableColumnFooterId ID for table column footer.</li>
     * @param {String} groupFooterId ID for group footer.</li>
     * @param {String} groupPanelToggleButtonId ID for group panel toggle button.</li>
     * @param {String} groupPanelToggleButtonToolTipOpen tool tip for open row group.</li>
     * @param {String} groupPanelToggleButtonToolTipClose tool tip for closed row group.</li>
     * @param {String} groupPanelToggleIconOpen The toggle icon for open row group.</li>
     * @param {String} groupPanelToggleIconClose The toggle icon for closed row group.</li>
     * @param {String} warningIconId ID for warning icon.</li>
     * @param {String} warningIconOpen The warning icon for open row group.</li>
     * @param {String} warningIconClosed The warning icon for closed row group.</li>
     * @param {String} warningIconToolTipOpen The warning icon tool tip for open row group.</li>
     * @param {String} warningIconToolTipClose The warning icon tool tip for closed row group.</li>
     * @param {String} collapsedHiddenFieldId ID for collapsed hidden field.</li>
     * @param {String} selectMultipleToggleButtonId ID for select multiple toggle button.</li>
     * @param {String} selectMultipleToggleButtonToolTip The select multiple toggle button tool tip.</li>
     * @param {String} selectMultipleToggleButtonToolTipSelected The select multiple toggle button tool tip when selected.</li>
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        var message = "Cannot initialize table.";
        if (props == null || props.id == null) {
            console.debug(message); // See Firebug console.
            return false;
        }
        var domNode = document.getElementById(props.id);
        if (domNode == null) {
            console.debug(message); // See Firebug console.
            return false;
        }

        // Set given properties on domNode.
        webui.suntheme4_2.prototypejs.extend(domNode, props, false);

        // Misc properties.
        domNode.SEPARATOR = ":";   // NamingContainer separator.

        // Panel toggle keys.
        domNode.SORT        = 0;
        domNode.PREFERENCES = 1;
        domNode.FILTER      = 2;

        // Sort keys.
        domNode.PRIMARY   = 0;
        domNode.SECONDARY = 1;
        domNode.TERTIARY  = 2;

        // Replace extra backslashes, JSON escapes new lines (e.g., \\n).
	domNode.hiddenSelectionsMsg = props.hiddenSelectionsMsg.replace(/\\n/g, "\n");
        domNode.totalSelectionsMsg = props.totalSelectionsMsg.replace(/\\n/g, "\n");
	domNode.missingSelectionMsg = props.missingSelectionMsg.replace(/\\n/g, "\n");
	domNode.duplicateSelectionMsg = props.duplicateSelectionMsg.replace(/\\n/g, "\n");
        domNode.deleteSelectionsMsg = props.deleteSelectionsMsg.replace(/\\n/g, "\n");

        // Private functions.
        domNode._getGroupSelectedRowsCount = webui.suntheme4_2.table._getGroupSelectedRowsCount;
        domNode._getGroupHiddenSelectedRowsCount = webui.suntheme4_2.table._getGroupHiddenSelectedRowsCount;
        domNode._getGroupRenderedSelectedRowsCount = webui.suntheme4_2.table._getGroupRenderedSelectedRowsCount;
        domNode._initGroupRows = webui.suntheme4_2.table._initGroupRows;
        domNode._initPrimarySortOrderMenu = webui.suntheme4_2.table._initPrimarySortOrderMenu;
        domNode._initPrimarySortOrderMenuToolTip = webui.suntheme4_2.table._initPrimarySortOrderMenuToolTip;
        domNode._initSecondarySortOrderMenu = webui.suntheme4_2.table._initSecondarySortOrderMenu;
        domNode._initSecondarySortOrderMenuToolTip = webui.suntheme4_2.table._initSecondarySortOrderMenuToolTip;
        domNode._initSortColumnMenus = webui.suntheme4_2.table._initSortColumnMenus;
        domNode._initSortOrderMenu = webui.suntheme4_2.table._initSortOrderMenu;
        domNode._initSortOrderMenus = webui.suntheme4_2.table._initSortOrderMenus;
        domNode._initSortOrderMenuToolTip = webui.suntheme4_2.table._initSortOrderMenuToolTip;
        domNode._initTertiarySortOrderMenu = webui.suntheme4_2.table._initTertiarySortOrderMenu;
        domNode._initTertiarySortOrderMenuToolTip = webui.suntheme4_2.table._initTertiarySortOrderMenuToolTip;
        domNode._resetFilterMenu = webui.suntheme4_2.table._resetFilterMenu;
        domNode._selectAllRows = webui.suntheme4_2.table._selectAllRows;
        domNode._selectGroupRows = webui.suntheme4_2.table._selectGroupRows;
        domNode._toggleGroupPanel = webui.suntheme4_2.table._toggleGroupPanel;
        domNode._togglePanel = webui.suntheme4_2.table._togglePanel;
        domNode._toggleSortPanel = webui.suntheme4_2.table._toggleSortPanel;
        domNode._validateSortPanel = webui.suntheme4_2.table._validateSortPanel;

        // Public functions.
        domNode.confirmDeleteSelectedRows = webui.suntheme4_2.table.confirmDeleteSelectedRows;
        domNode.confirmSelectedRows = webui.suntheme4_2.table.confirmSelectedRows;
        domNode.filterMenuChanged = webui.suntheme4_2.table.filterMenuChanged;
        domNode.getAllSelectedRowsCount = webui.suntheme4_2.table.getAllSelectedRowsCount;
        domNode.getAllHiddenSelectedRowsCount = webui.suntheme4_2.table.getAllHiddenSelectedRowsCount;
        domNode.getAllRenderedSelectedRowsCount = webui.suntheme4_2.table.getAllRenderedSelectedRowsCount;
        domNode.initAllRows = webui.suntheme4_2.table.initAllRows;
        domNode.toggleFilterPanel = webui.suntheme4_2.table.toggleFilterPanel;
        domNode.togglePreferencesPanel = webui.suntheme4_2.table.togglePreferencesPanel;

        return true;
    },

    /**
     * This function is used to get the number of selected components for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table).
     * This functionality requires the selectId property of the tableColumn component
     * and the hiddenSelectedRows property of the table component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupSelectedRowsCount: function(groupId) {
        return this._getGroupHiddenSelectedRowsCount(groupId) + 
            this._getGroupRenderedSelectedRowsCount(groupId);
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently hidden from view. This functionality requires the selectId property
     * of the tableColumn component and hiddenSelectedRows property of the table
     * component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of selected components hidden from view.
     * @private
     */
    _getGroupHiddenSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Find the given group Id in the groupIds array.
        for (var i = 0; i < this.groupIds.length; i++) {
            // Get selectId and rowIds array associated with groupId.
            if (groupId == this.groupIds[i]) {
                count = this.hiddenSelectedRowCounts[i];
                break;
            }
        }
        return count;
    },

    /**
     * This function is used to get the number of selected components, for the given row 
     * group (i.e., checkboxes or radiobuttons used to de/select rows of the table),
     * currently rendered. This functionality requires the selectId property of the
     * tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {int} The number of components selected in the current page.
     * @private
     */
    _getGroupRenderedSelectedRowsCount: function(groupId) {
        var count = 0;

        // Validate group IDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return count;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select != null && select.getProps().checked) {
                count++;
            }
        }
        return count;
    },

    /**
     * This function is used to initialize rows for the given groupwhen the state
     * of selected components change (i.e., checkboxes or radiobuttons used to
     * de/select rows of the table). This functionality requires the selectId
     * property of the tableColumn component to be set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initGroupRows: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        var checked = true; // Checked state of multiple select button.
        var selected = false; // At least one component is selected.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + 
                this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }

            // Set row style class.
            var row = document.getElementById(this.groupIds[i] + 
                this.SEPARATOR + rowIds[k]);
            var props = select.getProps();
            if (select.getProps().checked == true) {
                webui.suntheme4_2.common.addStyleClass(row, 
                    this.selectRowStyleClass);
                selected = true;
            } else {
                webui.suntheme4_2.common.stripStyleClass(row, 
                    this.selectRowStyleClass);
                checked = false;
            }
        }

        // Set multiple select button state.
        var title;
        var checkbox = document.getElementById(groupId + this.SEPARATOR + 
            this.selectMultipleToggleButtonId);
        if (checkbox != null) {
            title = (checked) 
                ? this.selectMultipleToggleButtonToolTipSelected
                : this.selectMultipleToggleButtonToolTip;
            checkbox.setProps({
                "checked": checked,
                "title": title
            });
        }

        // Get flag indicating groupis collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !webui.suntheme4_2.common.isVisible(prefix + rowIds[0]);

        // Set next warning image.
        var image = document.getElementById(prefix + this.warningIconId);
        if (image != null) {
            var src = this.warningIconOpen;
            title = this.warningIconToolTipOpen;

            // Don't show icon when multiple select is checked.
            if (collapsed && selected && !checked) {
                src = this.warningIconClose;
                title = this.warningIconToolTipClose;
            }
            image.setProps({
               "src": src,
               "title": title
            });
        }
        return true;
    },

    /**
     * This function is used to initialize the primary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },  

    /**
     * This function is used to initialize the primary sort order menu tool tips 
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initPrimarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortOrderMenuIds[this.PRIMARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.PRIMARY], 
            this.sortOrderMenuIds[this.PRIMARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the secondary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSecondarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortOrderMenuIds[this.SECONDARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.SECONDARY], 
            this.sortOrderMenuIds[this.SECONDARY]);
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort column menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortColumnMenus: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get sort column menu.
            var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortColumnMenuIds[i]);
            if (sortColumnMenu == null) {
                continue;
            }

            // Set default selected value.
            sortColumnMenu.selectedIndex = 0;

            // Set selected option.
            for (var k = 0; k < sortColumnMenu.options.length; k++) {
                if (sortColumnMenu.options[k].defaultSelected == true) {
                    sortColumnMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values are updated.
            webui.suntheme4_2.dropDown.changed(this.sortColumnMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize the primary, secondary, and tertiary 
     * sort order menus used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenus: function() {
        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Set initial selected option for all sort column menus.
        for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
            // Get sort order menu.
            var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortOrderMenuIds[i]);
            if (sortOrderMenu == null) {
                continue;
            }

            // Set default selected value.
            sortOrderMenu.selectedIndex = 0;

            // Get sort column menu.
            var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortColumnMenuIds[i]);
            if (sortColumnMenu != null) {
                // If the table is paginated and there are no hidden selected rows, the select
                // column cannot be sorted descending.
                if (sortColumnMenu.options[sortColumnMenu.selectedIndex].
                        value == this.selectSortMenuOptionValue
                        && !this.hiddenSelectedRows && this.paginated) {
                    sortOrderMenu.options[1].disabled = true;
                } else {
                    sortOrderMenu.options[1].disabled = false;
                }
            }

            // Set selected option.
            for (var k = 0; k < sortOrderMenu.options.length; k++) {
                if (sortOrderMenu.options[k].defaultSelected == true) {
                    sortOrderMenu.options[k].selected = true;
                    break;
                }
            }
            // Ensure hidden filed values and styles are updated.
            webui.suntheme4_2.dropDown.changed(this.sortOrderMenuIds[i]);

            // Initialize tool tip.
            this._initSortOrderMenuToolTip(this.sortColumnMenuIds[i], this.sortOrderMenuIds[i]);
        }
        return true;
    },

    /**
     * This function is used to initialize sort order menus used in the
     * sort panel. When a sort column menu changes, the given sort order 
     * menu is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenu: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Validate sort order menu IDs.
        if (this.sortOrderMenuIds == null || this.sortOrderMenuIds.length == 0) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Reset selected option.
        sortOrderMenu.selectedIndex = 0; // Default ascending.

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // If the table is paginated and there are no hidden selected rows, the select
        // column cannot be sorted descending.
        if (sortColumnMenu.options[selectedIndex].value == this.selectSortMenuOptionValue
                && !this.hiddenSelectedRows && this.paginated) {
            sortOrderMenu.options[1].disabled = true;
        } else {
            sortOrderMenu.options[1].disabled = false;
        }

        // Attempt to match the selected index of the given sort column menu with the
        // default selected value from either sort column menu. If a match is found, the 
        // default selected value of the associated sort order menu is retrieved. This
        // default selected value is set as the current selection of the given sort 
        // order menu.
        for (var i = 0; i < this.sortColumnMenuIds.length; i++) {
            // Get the current sort column menu to test the default selected value.
            var currentSortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
                this.sortColumnMenuIds[i]);
            if (currentSortColumnMenu == null) {
                continue;
            }

            // Find default selected value for the current sort column menu.
            var defaultSelected = null;
            for (var k = 0; k < currentSortColumnMenu.options.length; k++) {
                if (currentSortColumnMenu.options[k].defaultSelected == true) {
                    defaultSelected = currentSortColumnMenu.options[k].value;
                    break;
                }
            }

            // Match default selected value with selected index value.
            if (defaultSelected != null && defaultSelected ==
                    sortColumnMenu.options[selectedIndex].value) {
                // Get current sort order menu to test the default selected value.
                var currentSortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
                    this.sortOrderMenuIds[i]);
                if (currentSortOrderMenu == null) {
                    continue;
                }

                // Find default selected index for the current sort order menu.
                var defaultSelectedIndex = -1;
                for (var c = 0; c < currentSortOrderMenu.options.length; c++) {
                    if (currentSortOrderMenu.options[c].defaultSelected == true) {
                        defaultSelectedIndex = c;
                        break;
                    }
                }

                // Set selected value for given sort order menu.
                if (defaultSelectedIndex > -1) {
                    sortOrderMenu.options[defaultSelectedIndex].selected = true;
                } else {
                    sortOrderMenu.options[0].selected = true; // Default ascending.
                }
                break;
            }
        }
        // Ensure hidden field values and styles are updated.
        webui.suntheme4_2.dropDown.changed(sortOrderMenuId);

        // Set sort order menu tool tip.
        return this._initSortOrderMenuToolTip(sortColumnMenuId, sortOrderMenuId);
    },

    /**
     * This function is used to initialize sort order menu tool tips used in the 
     * sort panel. When a sort column menu changes, the given sort order 
     * menu tool tip is initialized based on the the selected value of the given
     * sort column menu.
     *
     * @param {String} sortColumnMenuId The HTML element ID for the sort column menu component.
     * @param {String} sortOrderMenuId The HTML element ID for the sort order menu component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initSortOrderMenuToolTip: function(sortColumnMenuId, sortOrderMenuId) {
        if (sortColumnMenuId == null || sortOrderMenuId == null) {
            return false;
        }

        // Get sort column menu.
        var sortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(sortColumnMenuId);
        if (sortColumnMenu == null) {
            return false;
        }

        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(sortOrderMenuId);
        if (sortOrderMenu == null) {
            return false;
        }

        // Get tool tip associated with given sort order menu.
        var toolTip = "";
        if (this.sortOrderToolTips != null && this.sortOrderToolTips.length != 0
                && this.sortOrderMenuIds != null) {
            for (var i = 0; i < this.sortOrderMenuIds.length; i++) {
                // The tool tip is at index zero, after splitting the message.
                if (sortOrderMenuId == this.sortOrderMenuIds[i]) {
                    toolTip = this.sortOrderToolTips[i].split("{0}")[0];
                    break;
                }
            }
        }

        // Get sort column menu selected index.            
        var selectedIndex = (sortColumnMenu.selectedIndex > -1)
            ? sortColumnMenu.selectedIndex : 0; // Default to first option.

        // Set tool tip.
        if (sortOrderMenu.options[sortOrderMenu.selectedIndex].value == "true") {
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsDescending[selectedIndex];
        } else {
            // Default ascending.
            sortOrderMenu.title = toolTip + this.sortOrderToolTipsAscending[selectedIndex];
        }
        return true;
    },

    /**
     * This function is used to initialize the tertiary sort order menus used in the 
     * table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenu: function() {
        return this._initSortOrderMenu(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is used to initialize the tertiary sort order menu tool tips
     * used in the table sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _initTertiarySortOrderMenuToolTip: function() {
        // Get sort order menu.
        var sortOrderMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortOrderMenuIds[this.TERTIARY]);
        if (sortOrderMenu != null) {
            // IE hack so disabled option is not selected -- bugtraq #6269683.
            if (sortOrderMenu.options[sortOrderMenu.selectedIndex].disabled == true) {
                sortOrderMenu.selectedIndex = 0;
            }
        }
        return this._initSortOrderMenuToolTip(this.sortColumnMenuIds[this.TERTIARY], 
            this.sortOrderMenuIds[this.TERTIARY]);
    },

    /**
     * This function is used to reset filter drop down menu. This functionality 
     * requires the filterId of the table component to be set. In addition,
     * the selected value must be set as well to restore the default selected
     * value when the embedded filter panel is closed.
     *
     * @param {String} filterId The HTML element ID of the filter menu.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _resetFilterMenu: function(filterId) {
        if (filterId == null) {
            return false;
        }

        // Get filter menu.
        var menu = webui.suntheme4_2.dropDown.getSelectElement(filterId);
        if (menu == null) {
            return true;
        }

        // Get div element associated with the filter panel.
        var div = document.getElementById(this.panelIds[this.FILTER]);
        if (div == null) {
            return false;
        }

        // Set selected style.
        if (webui.suntheme4_2.common.isVisibleElement(div)) {
            webui.suntheme4_2.common.stripStyleClass(menu, this.basicFilterStyleClass);
            webui.suntheme4_2.common.addStyleClass(menu, this.customFilterStyleClass);
        } else {
            // Reset default selected option.
            menu.selectedIndex = 0;
            for (var i = 0; i < menu.options.length; i++) {
                if (menu.options[i].defaultSelected == true) {
                    menu.options[i].selected = true;
                    break;
                }
            }
            webui.suntheme4_2.common.stripStyleClass(menu, this.customFilterStyleClass);
            webui.suntheme4_2.common.addStyleClass(menu, this.basicFilterStyleClass);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components in all row groups
     * displayed in the table (i.e., checkboxes or radiobuttons used to de/select
     * rows of the table). This functionality requires the selectId property of
     * the tableColumn component to be set.
     *
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectAllRows: function(selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0) {
            return false;
        }

        // For each group, get the row and select id.
        for (var i = 0; i < this.groupIds.length; i++) {
            this._selectGroupRows(this.groupIds[i], selected);
        }
        return true;
    },

    /**
     * This function is used to set the selected state components for the given row group
     * (i.e., checkboxes or radiobuttons used to de/select rows of the table). This 
     * functionality requires the selectId property of the tableColumn component to be
     * set.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @param {boolean} selected Flag indicating components should be selected.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _selectGroupRows: function(groupId, selected) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get selectId and rowIds array associated with groupId.
        var selectId = null;
        var rowIds = null;
        for (var i = 0; i < this.groupIds.length; i++) {
            if (groupId == this.groupIds[i]) {
                selectId = this.selectIds[i];
                rowIds = this.rowIds[i];
                break;
            }
        }

        // If selectId or rowIds could not be found, do not continue.
        if (selectId == null || rowIds == null) {
            return false;
        }

        // Update the select component for each row.
        for (var k = 0; k < rowIds.length; k++) {
            var select = document.getElementById(
                this.groupIds[i] + this.SEPARATOR + rowIds[k] + this.SEPARATOR + selectId);
            if (select == null) {
                continue;
            }
            select.setProps({checked: new Boolean(selected).valueOf()});
        }
        return this._initGroupRows(groupId); // Set row highlighting.
    },

    /**
     * This function is used to toggle row group panels open or closed.
     *
     * @param {String} groupId The HTML element ID for the tableRowGroup component.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleGroupPanel: function(groupId) {
        // Validate groupIDs.
        if (this.groupIds == null || this.groupIds.length == 0
                || groupId == null) {
            return false;
        }

        // Get rowIds array associated with groupId.
        var rowIds = null;
        for (var c = 0; c < this.groupIds.length; c++) {
            if (groupId == this.groupIds[c]) {
                rowIds = this.rowIds[c];
                break;
            }
        }

        // If row IDs could not be found, do not continue.
        if (rowIds == null) {
            return false;
        }

        // Get flag indicating group is collapsed.
        var prefix = groupId + this.SEPARATOR;
        var collapsed = !webui.suntheme4_2.common.isVisible(prefix + rowIds[0]);

        // Get the number of column headers and table column footers for all 
        // TableRowGroup children.
        var _prefix;
        var columnHeaderId;
        var tableColumnFooterId;
        var columnHeadersCount = 0;
        var tableColumnFootersCount = 0;
        for (var i = 0; i < this.groupIds.length; i++) {
            // Only need to test first nested column header/footer; thus, index 0 is used.        
            _prefix = this.groupIds[i] + this.SEPARATOR;
            columnHeaderId = _prefix + this.columnHeaderId + this.SEPARATOR + "0";
            tableColumnFooterId = _prefix + this.tableColumnFooterId + this.SEPARATOR + "0";
            if (document.getElementById(columnHeaderId) != null) {
                columnHeadersCount++;
            }
            if (document.getElementById(tableColumnFooterId) != null) {
                tableColumnFootersCount++;
            }
        }

        // Toggle nested column footer.
        var rowIndex = 0;
        var columnFooterId;
        while (true) {
            columnFooterId = prefix + this.columnFooterId + 
                this.SEPARATOR + rowIndex++;
            if (document.getElementById(columnFooterId) == null) {
                break;
            }
            webui.suntheme4_2.common.setVisible(columnFooterId, collapsed);
        }

        // Toggle column header only if multiple column headers are shown.
        if (columnHeadersCount > 1) {
            rowIndex = 0;
            while (true) {
                columnHeaderId = prefix + this.columnHeaderId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(columnHeaderId) == null) {
                    break;
                }            
                webui.suntheme4_2.common.setVisible(columnHeaderId, collapsed);
            }
        }

        // Toggle table column footer only if multiple column footers are shown.
        if (tableColumnFootersCount > 1) {
            rowIndex = 0;
            while (true) {
                tableColumnFooterId = prefix + this.tableColumnFooterId + 
                    this.SEPARATOR + rowIndex++;
                if (document.getElementById(tableColumnFooterId) == null) {
                    break;
                }
                webui.suntheme4_2.common.setVisible(tableColumnFooterId, collapsed);
            }
        }

        // Toggle group rows.
        var rowId;
        for (var k = 0; k < rowIds.length; k++) {
            rowId = prefix + rowIds[k];
            webui.suntheme4_2.common.setVisible(rowId, collapsed);
        }

        // Toggle group footers.
        webui.suntheme4_2.common.setVisible(prefix + this.groupFooterId, collapsed);

        // Set next toggle button image.
        var groupPanelToggleButtonId = prefix + this.groupPanelToggleButtonId;
        var hyperlink = document.getElementById(groupPanelToggleButtonId);
        var image = webui.suntheme4_2.hyperlink.getImgElement(groupPanelToggleButtonId);
        if (hyperlink != null && image != null) {
            image.style.cssText = null; // Need to clear clipped, theme image.
            if (collapsed) {
                // Need to provide themed icon key?
                image.src = this.groupPanelToggleIconOpen;
                hyperlink.setProps({title: this.groupPanelToggleButtonToolTipOpen});
            } else {
                image.src = this.groupPanelToggleIconClose;
                hyperlink.setProps({title: this.groupPanelToggleButtonToolTipClose});
            }
        }

        // Set collapsed hidden field.
        var hiddenField = document.getElementById(prefix + this.collapsedHiddenFieldId);
        if (hiddenField != null) {
            hiddenField.value = !collapsed;
        }
        return this._initGroupRows(groupId); // Set next warning image.
    },

    /**
     * This function is used to toggle embedded panels.
     *
     * @param {String} panelId The panel ID to toggle.
     * @param {String} panelFocusIdOpen The ID used to set focus when panel is opened.
     * @param {String} panelFocusIdClose The ID used to set focus when panel is closed.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _togglePanel: function(panelId, panelFocusIdOpen, panelFocusIdClose) {
        if (panelId == null) {
            return false;
        }

        // Toggle the given panel, hide all others.
        for (var i = 0; i < this.panelIds.length; i++) {
            // Get div element associated with the panel.
            var div = document.getElementById(this.panelIds[i]);
            if (div == null) {
                continue;
            }

            // Set display value. Alternatively, we could set div.style.display
            // equal to "none" or "block" (i.e., hide/show).
            if (this.panelIds[i] == panelId) {
                // Set focus when panel is toggled -- bugtraq 6316565.
                var focusElement = null;

                if (webui.suntheme4_2.common.isVisibleElement(div)) {
                    webui.suntheme4_2.common.setVisibleElement(div, false); // Hide panel.
                    focusElement = document.getElementById(panelFocusIdClose);
                } else {
                    webui.suntheme4_2.common.setVisibleElement(div, true); // Show panel.
                    focusElement = document.getElementById(panelFocusIdOpen);
                }

                // Set focus.
                if (focusElement != null) {
                    focusElement.focus();
                }
            } else {
                // Panels are hidden by default.
                webui.suntheme4_2.common.setVisibleElement(div, false);
            }

            // Get image from icon hyperlink component.
            var image = webui.suntheme4_2.hyperlink.getImgElement(this.panelToggleIds[i]);
            if (image == null) {
                continue; // Filter panel uses a drop down menu.
            }

            // Set image.
            if (webui.suntheme4_2.common.isVisibleElement(div)) {
                image.src = this.panelToggleIconsOpen[i];
            } else {
                image.src = this.panelToggleIconsClose[i];
            }
        }
        return true;
    },

    /**
     * This function is used to toggle the sort panel open or closed. This
     * functionality requires the filterId of the table component to be set.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _toggleSortPanel: function() {
        // Validate panel IDs.
        if (this.panelIds == null || this.panelIds.length == 0) {
            return false;
        }

        // Initialize sort column and order menus.
        this._initSortColumnMenus(); 
        this._initSortOrderMenus();

        // Toggle sort panel.
        this._togglePanel(this.panelIds[this.SORT], 
        this.panelFocusIds[this.SORT], this.panelToggleIds[this.SORT]);
        return this._resetFilterMenu(this.panelToggleIds[this.FILTER]); // Reset filter menu.
    },

    /**
     * This function is used to validate sort column menu selections 
     * for the sort panel.
     *
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _validateSortPanel: function() {
        // Validate sort column menu IDs.
        if (this.sortColumnMenuIds == null || this.sortColumnMenuIds.length == 0) {
            return false;
        }

        // Get sort column menus.
        var primarySortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortColumnMenuIds[this.PRIMARY]);
        var secondarySortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortColumnMenuIds[this.SECONDARY]);
        var tertiarySortColumnMenu = webui.suntheme4_2.dropDown.getSelectElement(
            this.sortColumnMenuIds[this.TERTIARY]);

        // Test primary and secondary menu selections.
        if (primarySortColumnMenu != null && secondarySortColumnMenu != null) {
            // Test if secondary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && secondarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                  && primarySortColumnMenu.selectedIndex == secondarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test primary and tertiary menu selections.
        if (primarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but primary is not.
            if (primarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (primarySortColumnMenu.selectedIndex > 0
                    && primarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }

        // Test secondary and tertiary menu selections.
        if (secondarySortColumnMenu != null && tertiarySortColumnMenu != null) {
            // Test if tertiary sort is set, but secondary is not.
            if (secondarySortColumnMenu.selectedIndex < 1 
                    && tertiarySortColumnMenu.selectedIndex > 0) {
                alert(this.missingSelectionMsg);
                return false;
            }
            // If a selection has been made, test for duplicate.
            if (secondarySortColumnMenu.selectedIndex > 0
                    && secondarySortColumnMenu.selectedIndex == tertiarySortColumnMenu.selectedIndex) {
                alert(this.duplicateSelectionMsg);
                return false;
            }
        }
        return true;
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.anchor");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.anchorBase");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.widgetBase");
 
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.prototypejs");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.eventBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.config");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.dijit._Widget"); 
webui.suntheme4_2.dojo.require("webui.suntheme4_2.dijit._Templated");

/**
 * @name webui.suntheme4_2.widget.eventBase
 * @extends webui.suntheme4_2.dijit._Widget, webui.suntheme4_2.dijit._Templated
 * @class This class contains functions for widgets that extend eventBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.eventBase", [
    webui.suntheme4_2.dijit._Widget, webui.suntheme4_2.dijit._Templated]);

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.eventBase.event =
        webui.suntheme4_2.widget.eventBase.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: null,

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: null
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: null,

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: null
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: null,

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: null
    }
};

/**
 * Initialize public events and functions.
 * <p>
 * Note: If this.event.<eventName> is not null, a public function shall be added
 * to the DOM node. To avoid name clashes, do not create private functions with
 * the names; refresh, stateChanged, or submit.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.initEvents = function () {
    if (this.event == null) {
        return false;
    }

    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Flag to add subscribe function.
    var subscribe = false;

    // Add event topics.
    this.domNode.event = this.event;

    // Refresh.
    if (this.event.refresh != null) {
        // Set public function.
        this.domNode.refresh = function(execute) {
            return webui.suntheme4_2.dijit.byId(_id).refresh(execute);
        };
        subscribe = true;
    } else {
        this.domNode.event.refresh = null; // clean.
    }

    // Submit.
    if (this.event.submit != null) {
        // Set public function.
        this.domNode.submit = function(execute) {
            return webui.suntheme4_2.dijit.byId(_id).submit(execute);    
        };
        subscribe = true;
    } else {
        this.domNode.event.submit = null; // clean.
    }

    // State.
    if (this.event.state == null) {
        // Remove prototyped function.
        this.stateChanged = null;
        this.domNode.event.state = null; // clean.
    }
    
    // Subscribe.
    if (subscribe == true) {
        this.domNode.subscribe = function(topic, obj, func) {
            return webui.suntheme4_2.dijit.byId(_id).subscribe(topic, obj, func);
        };
    }
    return true;
};

/**
 * Publish an event topic.
 *
 * @param {String} topic The event topic to publish.
 * @param {Object} props Key-Value pairs of properties. This will be applied
 * to each topic subscriber.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.publish = function(topic, props) {
    // Obtain the Ajax module associated with this widget.
    var config = webui.suntheme4_2.config;
    if (config.ajax.module) {
        webui.suntheme4_2.dojo.require(config.ajax.module + "." + this.widgetName);
    }
    return webui.suntheme4_2.widget.eventBase.prototype._publish(topic, props);
};

/**
 * Publish an event topic.
 * <p>
 * Note: In order to obtain Ajax modules dynamically, this function shall be 
 * overridden by a custom AJAX implementation.
 * </p>
 * @param {String} topic The event topic to publish.
 * @param {Object} props Key-Value pairs of properties. This will be applied
 * to each topic subscriber.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.eventBase.prototype._publish = function(topic, props) {
    // Publish an event for custom AJAX implementations to listen for.
    webui.suntheme4_2.dojo.publish(topic, props);
    return true;
};

/**
 * Subscribe to an event topic.
 *
 * @param {String} topic The event topic to subscribe to.
 * @param {Object} obj The object in which a function will be invoked, or
 * null for default scope.
 * @param {String|Function} func The name of a function in context, or a 
 * function reference to invoke when topic is published. 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.subscribe = function(topic, obj, func) {
    webui.suntheme4_2.dojo.subscribe(topic, obj, func);
    return true;
};

/**
 * Process refresh event.
 * <p>
 * Note: If this.event.refresh is not null, an event is published for custom
 * Ajax implementations to listen for. If event topics are not implemented for 
 * this widget, the function returns and a message is output to the console.
 * </p>
 * @param {String} execute The string containing a comma separated list 
 * of client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.refresh = function(execute) {
    if (this.event.refresh == null) {
        console.debug("Error: Refresh event topics not implemented for " + 
            this.widgetName); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this.publish(this.event.refresh.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.refresh.endTopic
    }]);
    return true;
};

/**
 * Process state event.
 * <p>
 * Note: If this.event.state is not null, an event is published for custom
 * Ajax implementations to listen for. If event topics are not implemented for 
 * this widget, the function returns and a message is output to the console.
 * </p>
 * @param {Object} props Key-Value pairs of widget properties to update.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.stateChanged = function(props) {
    if (this.event.state == null) {
        console.debug("Error: State event topics not implemented for " + 
            this.widgetName); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this.publish(this.event.state.beginTopic, [{
        id: this.id,
        endTopic: this.event.state.endTopic,
        props: props
    }]);
    return true;
};

/**
 * Process submit event.
 * <p>
 * Note: If this.event.submit is not null, an event is published for custom
 * Ajax implementations to listen for. If event topics are not implemented for 
 * this widget, the function returns and a message is output to the console.
 * </p>
 * @param {String} execute Comma separated string containing a list of 
 * client ids against which the execute portion of the request 
 * processing lifecycle must be run.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.eventBase.prototype.submit = function(execute) {
    if (this.event.submit == null) {
        console.debug("Error: Submit event topics not implemented for " + 
            this.widgetName); // See Firebug console.
        return false;
    }

    // Publish an event for custom AJAX implementations to listen for.
    this.publish(this.event.submit.beginTopic, [{
        id: this.id,
        execute: execute,
        endTopic: this.event.submit.endTopic
    }]);
    return true;
};

/**
 * @name webui.suntheme4_2.widget.widgetBase
 * @extends webui.suntheme4_2.widget.eventBase
 * @class This class contains functions used for base functionality in all 
 * widgets. 
 * <p>
 * The widgetBase class inherits from webui.suntheme4_2.dijit._Widget and 
 * webui.suntheme4_2.dijit._Templated. The webui.suntheme4_2.dijit._Widget class is 
 * responsible for calling the buildRendering() and postCreate() functions in 
 * that order. The dijit_Templated function overrides the buildRendering() 
 * functon to fill in template properties.
 * <p></p>
 * The postCreate() function is responible for initializing CSS selectors, 
 * events, and public functions. Commonly used functions (e.g., getProps(), 
 * setProps(), and refresh() are set on the outermost DOM node via the 
 * "superclass" function of widgetBase. This inherited function is also 
 * responsible for invoking the private _setProps() function. 
 * <p></p>
 * The private _setProps() function is used to set widget properties that can be
 * updated by a web app developer. This helps encapsolate functionality and 
 * brand changes while providing a common API for all widgets. In addition, the 
 * widget is selctively updated (i.e., if and only if a key-value pair has been 
 * given). Saving given properties is deferred to the public setProps() function
 * which allows _setProps() to be used during initialization.
 * <p></p>
 * The private _setProps() function is also responsible for invoking 
 * setCommonProps() and setEventProps(). These properties may not always be set 
 * on the outermost DOM node; however, core (i.e., id, class, style, etc.) 
 * properties are. Core properties are set on the DOM via the "superclass" 
 * function of widgetBase which invokes the setCoreProps() function.
 * <p></p>
 * The getClassName() function is responsible for obtaining the selector that
 * will be set on the outermost DOM node. The private _setProps() function 
 * of widgetBase ensures that the getClassName() function is called prior to 
 * invoking setCoreProps(). In most cases, this function will be overridded in
 * order to apply widget specific selectors. However, selectors should be 
 * concatinated in order of precedence (e.g., the user's className property is 
 * always appended last).
 * <p></p>
 * The public setProps() function is responsible for extending the widget class
 * with properties so they can be used during later updates. After extending the
 * widget, the private _setProps() function is called. In some cases, the public
 * setProps() function may be overridden. For example, the label clears the
 * contents property from the widget because that is something we do not want to
 * extend.
 * <p></p>
 * The startup() function is typically called after the widget has been 
 * instantiated. For example, a progressBar might start a timer to periodically
 * refresh. 
 * <p></p>
 * Warning: It's not possible to append HTML elements from script that is 
 * not a direct child of the BODY element. If there is any Javascript
 * running inside the body that is a direct child of body, IE will throw
 * an "Internet Explorer cannot open the Internet site" error. For example,
 * webui.suntheme4_2.dijit._Templated._createNodesFromText generates such an error by calling
 * appendChild(). Therefore, widget creation must be deferred to the
 * window.onLoad event. See http://trac.dojotoolkit.org/ticket/4631
 * </p>
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.widgetBase", webui.suntheme4_2.widget.eventBase, {
    // Note: If your class contains arrays or other objects, they should be
    // declared in the constructor function so that each instance gets it's own
    // copy. Simple types (literal strings and numbers) are fine to declare in 
    // the class directly.
    constructor: function() {
        // Do nothing...
    },

    // Set defaults.
    common: webui.suntheme4_2.common, // Common utils.
    dojo: webui.suntheme4_2.dojo, // Dojo utils.
    prototypejs: webui.suntheme4_2.prototypejs, // Prototype utils.
    theme: webui.suntheme4_2.theme.common, // Theme utils.
    widget: webui.suntheme4_2.widget.common // Widget utils. 
});

/**
 * This function is used to render the widget from a template.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.buildRendering = function () {
    // Get default templates.
    if (this.templatePath == null && this.templateString == null) {
        this.templatePath = this.widget.getTemplatePath(this.widgetName);
        this.templateString = this.widget.getTemplateString(this.widgetName);
    }

    // The templatePath should have precedence. Therefore, in order for the 
    // templatePath to be used, templateString must be null.
    if (this.templatePath != null) {
        this.templateString = null;
    }

    // Template must be set prior to calling "superclass".
    return this.inherited("buildRendering", arguments);
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getClassName = function() {
    return this.className;
};

/**
 * This function is used to get common properties from the widget. Please see
 * the setCommonProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getCommonProps = function() {
    var props = {};

    // Set properties.
    if (this.accessKey) { props.accessKey = this.accessKey; }
    if (this.dir) { props.dir = this.dir; }
    if (this.lang) { props.lang = this.lang; }
    if (this.tabIndex) { props.tabIndex = this.tabIndex; }
    if (this.title) { props.title = this.title; }

    return props;
};

/**
 * This function is used to get core properties from the widget. Please see
 * the setCoreProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getCoreProps = function() {
    var props = {};

    // Set properties.
    if (this.className) { props.className = this.className; }
    if (this.id) { props.id = this.id; }
    if (this.style) { props.style = this.style; }
    if (this.visible != null) { props.visible = this.visible; }

    return props;
};

/**
 * This function is used to get event properties from the widget. Please
 * see the setEventProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getEventProps = function() {
    var props = {};

    // Set properties.
    if (this.onBlur) { props.onBlur = this.onBlur; }
    if (this.onChange) { props.onChange = this.onChange; }
    if (this.onClick) { props.onClick = this.onClick; }
    if (this.onDblClick) { props.onDblClick = this.onDblClick; }
    if (this.onFocus) { props.onFocus = this.onFocus; }
    if (this.onKeyDown) { props.onKeyDown = this.onKeyDown; }
    if (this.onKeyPress) { props.onKeyPress = this.onKeyPress; }
    if (this.onKeyUp) { props.onKeyUp = this.onKeyUp; }
    if (this.onMouseDown) { props.onMouseDown = this.onMouseDown; }
    if (this.onMouseOut) { props.onMouseOut = this.onMouseOut; }
    if (this.onMouseOver) { props.onMouseOver = this.onMouseOver; }
    if (this.onMouseUp) { props.onMouseUp = this.onMouseUp; }
    if (this.onMouseMove) { props.onMouseMove = this.onMouseMove; }
    if (this.onSelect) { props.onSelect = this.onSelect; }

    return props;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.widgetBase.prototype.getProps = function() {
    var props = {};

    // Set properties.
    this.prototypejs.extend(props, this.getCommonProps(), false);
    this.prototypejs.extend(props, this.getCoreProps(), false);
    this.prototypejs.extend(props, this.getEventProps(), false);

    return props;
};

/**
 * This function is used to test if widget has been initialized.
 * <p>
 * Note: It is assumed that an HTML element is used as a place holder for the
 * document fragment.
 * </p>
 * @return {boolean} true if widget is initialized.
 */
webui.suntheme4_2.widget.widgetBase.prototype.isInitialized = function() {
    // Testing if the outermost DOM node has been added to the document and
    // ensuring a Dojo attach point exists works fine for JSP. However, the 
    // following code always returns null for facelets.
    //
    // var domNode = document.getElementById(this.id);
    // if (domNode && domNode.getAttribute("dojoattachpoint")) {
    if (this.initialized == true) {
        return true;
    }
    return false;
};

/**
 * This is called after the buildRendering() function.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.postCreate = function () {
    this.inherited("postCreate", arguments);

    // In order to register widgets properly, the DOM node id must be set prior 
    // to creating any widget children. Otherwise, widgets may not be destroyed.
    this.domNode.id = this.id;

    // Since the anchor id and name must be the same on IE, we cannot obtain the
    // widget using the DOM node ID via the public functions below. Therefore, 
    // we need to set the widget id via closure magic.
    var _id = this.id;

    // Set public functions.
    this.domNode.getProps = function() { return webui.suntheme4_2.dijit.byId(_id).getProps(); };
    this.domNode.setProps = function(props, notify) { return webui.suntheme4_2.dijit.byId(_id).setProps(props, notify); };

    // Initialize public events and functions.
    this.initEvents();

    // Set properties.
    this._setProps(this.getProps());

    // All widget properties have been set.
    return this.initialized = true;
};

/**
 * This function is used to set common properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey Shortcut key.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setCommonProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.accessKey) { 
        domNode.accessKey = props.accessKey;
    }
    if (props.dir) {
        domNode.dir = props.dir;
    }
    if (props.lang) {
        domNode.lang = props.lang;
    }
    if (props.tabIndex > -1 && props.tabIndex < 32767) {
        domNode.tabIndex = props.tabIndex;
    }
    if (props.title) {
        domNode.title = props.title;
    }
    return true;
};

/**
 * This function is used to set core properties for the given domNode. These
 * properties are typically set on the outermost element.
 *
 * Note: The className is typically provided by a web app developer. If 
 * the widget has a default className, it should be added to the DOM node
 * prior to calling this function. For example, the "myCSS" className would
 * be appended to the existing "Tblsun4" className (e.g., "Tbl_sun4 myCSS").
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setCoreProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }
    if (props.className) {
        domNode.className = props.className;
    }
    if (props.id) { 
        domNode.id = props.id;
    }
    if (props.style) { 
        domNode.style.cssText = props.style;
    }
    if (props.visible != null) {
        this.common.setVisibleElement(domNode, 
            new Boolean(props.visible).valueOf());
    }
    return true;
};

/**
 * This function is used to set event properties for the given domNode.
 *
 * @param {Node} domNode The DOM node to assign properties to.
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Element value changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Element text selected.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setEventProps = function(domNode, props) {
    if (domNode == null || props == null) {
        return false;
    }

    // Note: JSON strings are not recognized as JavaScript. In order for
    // events to work properly, an anonymous function must be created.
    if (props.onBlur) { 
        domNode.onblur = (typeof props.onBlur == 'string')
            ? new Function("event", props.onBlur)
            : props.onBlur;
    }
    if (props.onClick) {
        domNode.onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick)
            : props.onClick;
    }
    if (props.onChange) {
        domNode.onchange = (typeof props.onChange == 'string')
            ? new Function("event", props.onChange)
            : props.onChange;
    }
    if (props.onDblClick) {
        domNode.ondblclick = (typeof props.onDblClick == 'string')
            ? new Function("event", props.onDblClick)
            : props.onDblClick;
    }
    if (props.onFocus) {
        domNode.onfocus = (typeof props.onFocus == 'string')
            ? new Function("event", props.onFocus)
            : props.onFocus;
    }
    if (props.onKeyDown) {
        domNode.onkeydown = (typeof props.onKeyDown == 'string')
            ? new Function("event", props.onKeyDown)
            : props.onKeyDown;
    }
    if (props.onKeyPress) {
        domNode.onkeypress = (typeof props.onKeyPress == 'string')
            ? new Function("event", props.onKeyPress)
            : props.onKeyPress;
    }
    if (props.onKeyUp) {
        domNode.onkeyup = (typeof props.onKeyUp == 'string')
            ? new Function("event", props.onKeyUp)
            : props.onKeyUp;
    }
    if (props.onMouseDown) {
        domNode.onmousedown = (typeof props.onMouseDown == 'string')
            ? new Function("event", props.onMouseDown)
            : props.onMouseDown;
    }
    if (props.onMouseOut) {
        domNode.onmouseout = (typeof props.onMouseOut == 'string')
            ? new Function("event", props.onMouseOut)
            : props.onMouseOut;
    }
    if (props.onMouseOver) {
        domNode.onmouseover = (typeof props.onMouseOver == 'string')
            ? new Function("event", props.onMouseOver)
            : props.onMouseOver;
    }
    if (props.onMouseUp) {
        domNode.onmouseup = (typeof props.onMouseUp == 'string')
            ? new Function("event", props.onMouseUp)
            : props.onMouseUp;
    }
    if (props.onMouseMove) {
        domNode.onmousemove = (typeof props.onMouseMove == 'string')
            ? new Function("event", props.onMouseMove)
            : props.onMouseMove;
    }
    if (props.onSelect) {
        domNode.onselect = (typeof props.onSelect == 'string')
            ? new Function("event", props.onSelect)
            : props.onSelect;
    }
    return true;
};

/**
 * This function is used to set widget properties.
 *
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 *
 * Note: If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} style Specify style rules inline.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Extend widget object for later updates.
    this.prototypejs.extend(this, props);

    // Set properties.
    this._setProps(props);

    // Notify listeners state has changed.
    if (new Boolean(notify).valueOf() == true &&
            typeof this.stateChanged == "function") {
        this.stateChanged(props);
    }
    return true;
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.widgetBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set style class -- must be set before calling setCoreProps().
    props.className = this.getClassName();

    // Set more properties.
    return this.setCoreProps(this.domNode, props);
};

/**
 * This function is used to "start" the widget, after the widget has been
 * instantiated.
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.widgetBase.prototype.startup = function () {
    if (this._started) {
        return false;
    }
    this.inherited("startup", arguments);
    return this._started = true;
};

/**
 * @name webui.suntheme4_2.widget.anchorBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend anchorBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.anchorBase", webui.suntheme4_2.widget.widgetBase);

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchorBase.prototype.addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this.widget.removeChildNodes(this.domNode);

    // Add contents.
    for (i = 0; i < props.contents.length; i++) {
        this.widget.addFragment(this.domNode, props.contents[i], "last");
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.anchorBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.hrefLang) { props.hrefLang = this.hrefLang; }
    if (this.target) { props.target = this.target; }
    if (this.type) { props.type = this.type; }
    if (this.rev) { props.rev = this.rev; }
    if (this.rel) { props.rel = this.rel; }
    if (this.shape) { props.shape = this.shape; }
    if (this.coords) { props.coords = this.coords; }
    if (this.charset) { props.charset = this.charset; }
    if (this.accessKey) { props.accesskey = this.accessKey; }
    if (this.href) { props.href = this.href; }
    if (this.name) { props.name = this.name; } 
    if (this.contents) { props.contents = this.contents; }
    if (this.disabled != null) { props.disabled = this.disabled; }

    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchorBase.prototype.onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this.domNode._onclick)
        ? this.domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchorBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.anchorBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Add contents.
    this.addContents(props);

    // Set properties.
    if (props.accessKey) { this.domNode.accesskey = props.accessKey; }
    if (props.charset) { this.domNode.charset = props.charset; }
    if (props.coords) { this.domNode.coords = props.coords; }
        if (props.href) {
                
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.
            if (this.prefix) {
                props.href = 
                    webui.suntheme4_2.widget.common.appendPrefix(this.prefix, props.href);
            }
            this.domNode.href = props.href; 
        }
    if (props.hrefLang) { this.domNode.hrefLang =  props.hrefLang; }
    if (props.name) { this.domNode.name = props.name; }
    if (props.rev) { this.domNode.rev = props.rev; }
    if (props.rel) { this.domNode.rel = props.rel; }
    if (props.shape) { this.domNode.shape = props.shape; }
    if (props.target) { this.domNode.target = props.target; }
    if (props.type) { this.domNode.type = props.type; }

    // Set id -- anchors must have the same id and name on IE.
    if (props.name) {
        props.id = props.name;
    }

    // A web app devleoper could return false in order to cancel the 
    // submit. Thus, we will handle this event via the onClick call back.
    if (props.onClick) {
        // Set private function scope on DOM node.
        this.domNode._onclick = (typeof props.onClick == 'string')
            ? new Function("event", props.onClick) : props.onClick;

        // Must be cleared before calling setEventProps() below.
        props.onClick = null;
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.anchor
 * @extends webui.suntheme4_2.widget.anchorBase
 * @class This class contains functions for the anchor widget.
 * @constructor This function is used to construct an anchor widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.anchor", webui.suntheme4_2.widget.anchorBase, {
    // Set defaults.
    widgetName: "anchor" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.anchor.event =
        webui.suntheme4_2.widget.anchor.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_anchor_event_refresh_begin",
        
        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_anchor_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_anchor_event_state_begin",
        
        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_anchor_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.anchor.prototype.getClassName = function() {
    // Set default style.
    var className = (this.href && this.disabled == false)
        ? this.widget.getClassName("ANCHOR","")
        : this.widget.getClassName("ANCHOR_DISABLED","");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.anchor.prototype.postCreate = function () {
    // Create callback function for onclick event.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    return this.inherited("postCreate", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.button");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");
 
/**
 * @name webui.suntheme4_2.widget.button
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the button widget.
 * @constructor This function is used to construct a button widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.button", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.escape = true;
        this.mini = false;
        this.primary = true;
    },
    widgetName: "button" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.button.event =
        webui.suntheme4_2.widget.button.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_button_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_button_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_button_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_button_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.button.prototype.getClassName = function() {
    var key = null;

    if (this.mini == true && this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_MINI_DISABLED" // primaryMiniDisabledClassName
            : "BUTTON1_MINI";         // primaryMiniClassName;
    } else if (this.mini == true) {
        key = (this.disabled == true)
            ? "BUTTON2_MINI_DISABLED" // secondaryMiniDisabledClassName
            : "BUTTON2_MINI";         // secondaryMiniClassName;
    } else if (this.primary == true) {
        key = (this.disabled == true)
            ? "BUTTON1_DISABLED"      // primaryDisabledClassName
            : "BUTTON1";              // primaryClassName
    } else {
        key = (this.disabled == true)
            ? "BUTTON2_DISABLED"      // secondaryDisabledClassName
            : "BUTTON2";	      // secondaryClassName
    }

    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to obtain the outermost HTML element class name during
 * an onFocus or onMouseOver event.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.button.prototype.getHoverClassName = function() {
    var key = null;

    if (this.mini == true && this.primary == true) {
        key = "BUTTON1_MINI_HOVER"; 	// primaryMiniHovClassName;
    } else if (this.mini == true) {
        key = "BUTTON2_MINI_HOVER"; 	// secondaryMiniHovClassName;
    } else if (this.primary == true) {
        key = "BUTTON1_HOVER"; 		// primaryHovClassName;
    } else {
        key = "BUTTON2_HOVER";		// secondaryHovClassName;
    }

    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};
    
/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.button.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.alt) { props.alt = this.alt; }
    if (this.align) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.escape != null) { props.escape = this.escape; }
    if (this.mini != null) { props.mini = this.mini; }
    if (this.primary != null) { props.primary = this.primary; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * Helper function to create callback for onBlur event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.onBlurCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this.domNode.className = this.getClassName();
    } catch (err) {}
    return true;
};

/**
 * Helper function to create callback for onFocus event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.onFocusCallback = function(event) {
    if (this.disabled == true) {
        return true;
    }
    // Prevent errors during page submit, when modules have not been loaded.
    try {
        // Set style class.
        this.domNode.className = this.getHoverClassName();
    } catch (err) {}
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.domNode.name = this.id;
    }

    // Initialize deprecated public functions. 
    // 
    // Note: Although we now have a setProps function to update properties,
    // these functions were previously added to the DOM node; thus, we must
    // continue to be backward compatible.
    this.domNode.isSecondary = function() { return !(webui.suntheme4_2.dijit.byId(this.id).getProps().primary); };
    this.domNode.setSecondary = function(secondary) { return webui.suntheme4_2.dijit.byId(this.id).setProps({primary: !secondary}); };
    this.domNode.isPrimary = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().primary; };
    this.domNode.setPrimary = function(primary) { return webui.suntheme4_2.dijit.byId(this.id).setProps({primary: primary}); };
    this.domNode.isMini = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().mini; };
    this.domNode.setMini = function(mini) { return webui.suntheme4_2.dijit.byId(this.id).setProps({mini: mini}); };
    this.domNode.getDisabled = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().disabled; };
    this.domNode.setDisabled = function(disabled) { return webui.suntheme4_2.dijit.byId(this.id).setProps({disabled: disabled}); };
    this.domNode.getVisible = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().visible; };
    this.domNode.setVisible = function(show) { return webui.suntheme4_2.dijit.byId(this.id).setProps({visible: show}); };
    this.domNode.getText = function() { return webui.suntheme4_2.dijit.byId(this.id).getProps().value; };
    this.domNode.setText = function(text) { return webui.suntheme4_2.dijit.byId(this.id).setProps({value: text}); };
    this.domNode.doClick = this.domNode.click;

    // Set events.
    this.dojo.connect(this.domNode, "onblur", this, "onBlurCallback");
    this.dojo.connect(this.domNode, "onfocus", this, "onFocusCallback");
    this.dojo.connect(this.domNode, "onmouseout", this, "onBlurCallback");
    this.dojo.connect(this.domNode, "onmouseover", this, "onFocusCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} mini Set button as mini if true.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} primary Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.button.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.button.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.alt) { this.domNode.alt = props.alt; }
    if (props.align) { this.domNode.align = props.align; }

    // Set disabled.
    if (props.disabled != null) { 
        this.domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set value (i.e., button text).
    if (props.value) {
        // If escape is true, we want the text to be displayed literally. To 
        // achieve this behavior, do nothing.
        //
        // If escape is false, we want any special sequences in the text 
        // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
        this.domNode.value = (new Boolean(this.escape).valueOf() == false)
            ? this.prototypejs.unescapeHTML(props.value)
            : props.value;
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkbox");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkedBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.checkedBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend checkedBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkedBase", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    idSuffix: "" // Overridden by subclass
});

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getImageClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getInputClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the chekcbox.
 *
 * @return {Node} The HTML input element. 
 */
webui.suntheme4_2.widget.checkedBase.prototype.getInputElement = function() {
    return this.inputNode;
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getLabelClassName = function() {
    return null; // Overridden by subclass.
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.checkedBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.  
    if (this.align) { props.align = this.align; }
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.image) { props.image = this.image; }
    if (this.label) { props.label = this.label; }
    if (this.name) { props.name = this.name; }        
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.value) { props.value = this.value; }

    // After widget has been initialized, get user's input.
    if (this.isInitialized() == true && this.inputNode.checked != null) {
        props.checked = this.inputNode.checked;
    } else if (this.checked != null) {
        props.checked = this.checked;
    }
    return props;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedBase.prototype.onClickCallback = function(event) {
    if (this.readOnly == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the request.
    var result = (this.domNode._onclick)
        ? this.domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    return true;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.inputNode.id = this.id + this.idSuffix;
        this.imageContainer.id = this.id + "_imageContainer";
        this.labelContainer.id = this.id + "_labelContainer";

        // If null, use HTML input id.
        if (this.name == null) {
            this.name = this.inputNode.id;
        }
    }

    // Set public functions.
    this.domNode.getInputElement = function() { return webui.suntheme4_2.dijit.byId(this.id).getInputElement(); }
    
    // Create callback function for onclick event.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.checkedBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.value) { 
        this.inputNode.value = props.value;
    }
    if (props.readOnly != null) { 
        this.inputNode.readOnly = new Boolean(props.readOnly).valueOf();       
    }
    if (props.disabled != null) {
        this.inputNode.disabled = new Boolean(props.disabled).valueOf();        
    }
    
    // Set HTML input element class name.
    this.inputNode.className = this.getInputClassName();
    
    if (props.name) { 
        this.inputNode.name = props.name;
    }

    if (props.checked != null) {
        var checked = new Boolean(props.checked).valueOf();

        // Dynamically setting the checked attribute on IE 6 does not work until
        // the HTML input element has been added to the DOM. As a work around, 
        // we shall use a timeout to set the property during initialization.
        if (this.isInitialized() != true &&
                webui.suntheme4_2.browser.isIe()) {
            var _id = this.id;
            setTimeout(function() {
                // New literals are created every time this function
                // is called, and it's saved by closure magic.
                var widget = webui.suntheme4_2.dijit.byId(_id);
                widget.inputNode.checked = checked;
            }, 0); // (n) milliseconds delay.
        } else {
            this.inputNode.checked = checked;
        }
    }

    // Set image properties.
    if (props.image || props.disabled != null && this.image) {     
        // Ensure property exists so we can call setProps just once.
        if (props.image == null) {
            props.image = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.image.className = this.getImageClassName();

        // Update/add fragment.
        this.widget.updateFragment(this.imageContainer, this.image.id, props.image);
    } 

    // Set label properties.
    if (props.label || props.disabled != null && this.label) {     
        // Ensure property exists so we can call setProps just once.
        if (props.label == null) {
            props.label = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.label.className = this.getLabelClassName();

        // Update/add fragment.
        this.widget.updateFragment(this.labelContainer, this.label.id, props.label);
    }

    // Set more properties.
    this.setCommonProps(this.inputNode, props);
    this.setEventProps(this.inputNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.checkbox
 * @extends webui.suntheme4_2.widget.checkedBase
 * @class This class contains functions for the checkbox widget.
 * @constructor This function is used to construct a checkbox widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkbox", webui.suntheme4_2.widget.checkedBase, {
    // Set defaults.
    idSuffix: "_cb",
    widgetName: "checkbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.checkbox.event =
        webui.suntheme4_2.widget.checkbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkbox_event_refresh_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getClassName = function() {
    // Set default style.
    var className = (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_SPAN_DISABLED", "")
        : this.widget.getClassName("CHECKBOX_SPAN", ""); 

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getImageClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_IMAGE_DISABLED", "")
        : this.widget.getClassName("CHECKBOX_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getInputClassName = function() {
    // readOnly style.
    if (this.readOnly == true) {
        return this.widget.getClassName("CHECKBOX_READONLY", "");        
    }

    // disabled style.
    return (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_DISABLED", "")
        : this.widget.getClassName("CHECKBOX", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 */
webui.suntheme4_2.widget.checkbox.prototype.getLabelClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("CHECKBOX_LABEL_DISABLED", "")
        : this.widget.getClassName("CHECKBOX_LABEL", "");  
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkbox.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkboxGroup");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.checkedGroupBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.checkedGroupBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend checkedGroupBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkedGroupBase", webui.suntheme4_2.widget.widgetBase);

/**
 * Helper function to add elements with Object literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled 
 * @config {Array} columns 
 * @config {Array} contents 
 * @config {Object} label
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.addContents = function(props) {   
    if (props == null) {
        return false;
    }
    
    if (props.contents) {
        this.widget.removeChildNodes(this.tbodyContainer);
        var rowContainerCloneNode = this.rowContainer.cloneNode(false);
        this.tbodyContainer.appendChild(rowContainerCloneNode);
        if (props.label) {
            var rowNodeClone = this.rowNode.cloneNode(false);
            rowContainerCloneNode.appendChild(rowNodeClone);
            var labelContainerClone = this.labelContainer.cloneNode(false);
            rowNodeClone.appendChild(labelContainerClone);
            this.widget.addFragment(labelContainerClone, props.label, "last");            
              
        }
        var itemN = 0;
        var length = props.contents.length;
        var columns = (props.columns <= 0) ? 1 : props.columns;
        var rows = (length + (columns-1))/columns;
        var propsdisabledvalue = props.disabled == null ? false : props.disabled;
        for (var row = 0; row <= rows; row++) {
            for (var column = 0; column < columns; column++) {
                if (itemN < length) {
                    // Clone < td> node.
                    var contentsRowNodeClone = this.contentsRowNode.cloneNode(false);
                    rowContainerCloneNode.appendChild(contentsRowNodeClone);
                    // Set disabled.                   
                    props.contents[itemN].disabled = propsdisabledvalue;
                   
                    // Add child to the group.
                    this.widget.addFragment(contentsRowNodeClone, props.contents[itemN], "last");
                    itemN++;
                }
            }
            if (row + 1 <= rows) {
                rowContainerCloneNode = this.rowContainer.cloneNode(false);
                this.tbodyContainer.appendChild(rowContainerCloneNode);
                // This check is required here. Else the elements won't be
                // aligned properly when there is no label.
                if (props.label != null) {
                    var contentsRowNodeClone = this.contentsRowNode.cloneNode(false);
                    rowContainerCloneNode.appendChild(contentsRowNodeClone);
                }
              
            }
        }
    } else {
        // Update the disabled property client side
        if (props.disabled != null && this.contents) {
            for (var i = 0; i < this.contents.length; i++) {
                var contentWidget = webui.suntheme4_2.dijit.byId(this.contents[i].id);
                if (contentWidget) {
                    contentWidget.setProps({disabled: props.disabled});
                }
            }
        }
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.     
    if (this.columns) { props.columns = this.columns; }
    if (this.contents) { props.contents = this.contents; }    
    if (this.disabled != null) { props.disabled = this.disabled; }   
    if (this.id) { props.id = this.id; }
    if (this.label) { props.label = this.label; }    
    if (this.name) { props.name = this.name; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }  

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {                    
        this.contentsRowNode.id = this.id + "_contentsRowNode";
        this.divContainer.id = this.id + "_divContainer";
        this.labelContainer.id = this.id + "_labelContainer";                            
        this.rowContainer.id = this.id + "_rowContainer";
        this.rowNode.id = this.id + "_rowNode";
        this.tableContainer.id = this.id + "_tableContainer";   
        this.tbodyContainer.id = this.id + "_tbodyContainer";     
    }

    // Show label.
    if (this.label) {
        this.common.setVisibleElement(this.rowNode, true);
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {int} columns 
 * @config {Array} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {boolean} readOnly Set button as primary if true.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.checkedGroupBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set label properties.
    if (props.label) {
        // Update/add fragment.
        this.widget.updateFragment(this.labelContainer, this.label.id, props.label);
    }

    // Set contents.    
    if (props.contents || props.disabled != null) {              
        this.addContents(props);   
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.checkboxGroup
 * @extends webui.suntheme4_2.widget.checkedGroupBase
 * @class This class contains functions for the checkboxGroup widget.
 * @constructor This function is used to construct a checkboxGroup widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.checkboxGroup", webui.suntheme4_2.widget.checkedGroupBase, {
    // Set defaults.
    widgetName: "checkboxGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.checkboxGroup.event =
        webui.suntheme4_2.widget.checkboxGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkboxGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkboxGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_checkboxGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_checkboxGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.checkboxGroup.prototype.getClassName = function() {
    // Set default style.
    var className = (this.columns > 1)
        ? this.widget.getClassName("CBGRP_HORIZ", "")
        : this.widget.getClassName("CBGRP_VERT", "");

    return (this.className)
        ? className + " " + this.className
        : className;
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.dropDown");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.selectBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.browser");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.label");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.label
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the label widget.
 * @constructor This function is used to construct a label widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.label", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.level = webui.suntheme4_2.widget.common.getMessage("label.level", null, 2);
        this.required = false;
        this.valid = true;
    },
    widgetName: "label" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.label.event =
        webui.suntheme4_2.widget.label.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_label_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_label_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_label_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_label_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.label.prototype.getClassName = function() {
    var key = "LABEL_LEVEL_TWO_TEXT";

    if (this.valid == false) {
        key = "CONTENT_ERROR_LABEL_TEXT";
    } else if (this.level == 1) {
        key = "LABEL_LEVEL_ONE_TEXT";
    } else if (this.level == 3) {
        key = "LABEL_LEVEL_THREE_TEXT";
    }

    // Get theme property.
    var className = this.theme.getClassName(key);
    if (className == null || className.length == 0) {
	return this.className;
    }
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to process notification events with Object
 * literals.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} detail Message detail text.
 * @config {boolean} valid Flag indicating validation state.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.label.prototype.notify = function(props) {
    if (props == null) {
        return false;
    }
    return this.setProps({
        valid: props.valid,
        errorImage: {
            title: props.detail
        }
    });
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.label.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.contents) { props.contents = this.contents; }
    if (this.errorImage) { props.errorImage = this.errorImage; }
    if (this.htmlFor) { props.htmlFor = this.htmlFor; }
    if (this.level != null) { props.level = this.level; }
    if (this.required != null) { props.required = this.required; }
    if (this.requiredImage) { props.requiredImage = this.requiredImage; }
    if (this.valid != null) { props.valid = this.valid; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.label.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.contentsContainer.id = this.id + "_contentsContainer";
        this.errorImageContainer.id = this.id + "_errorImageContainer";
        this.requiredImageContainer.id = this.id + "_requiredImageContainer";
        this.valueContainer.id = this.id + "_valueContainer";
    }

    // If errorImage or requiredImage are null, create images from the theme.
    // When the _setProps() function is called, image widgets will be
    // instantiated via the props param. 
    if (this.errorImage == null) {
	this.errorImage = this.widget.getWidgetProps("image", {
            icon: "LABEL_INVALID_ICON",
            id: this.id + "_error",
	    className: this.widget.getClassName("LABEL_INVALID_IMAGE", null)
        });
    }
    if (this.requiredImage == null) {
	this.requiredImage = this.widget.getWidgetProps("image", {
            icon: "LABEL_REQUIRED_ICON",
            id: this.id + "_required",
	    className: this.widget.getClassName("LABEL_REQUIRED_IMAGE", null)
        });
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey
 * @config {String} className CSS selector.
 * @config {String} contents 
 * @config {String} dir Specifies the directionality of text.
 * @config {Object} errorImage 
 * @config {String} htmlFor 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} level 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} primary Set button as primary if true.
 * @config {boolean} required
 * @config {Object} requiredImage
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.label.prototype.setProps = function(props, notify) {
    if (props == null) {
        return false;
    }

    // Replace contents -- do not extend.
    if (props.contents) {
        this.contents = null;
    }

    // Extend widget object for later updates.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.label.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.htmlFor) { this.domNode.htmlFor = props.htmlFor; }
    if (props.valid != null) { this.valid = new Boolean(props.valid).valueOf(); }
    if (props.required != null) { this.required = new Boolean(props.required).valueOf(); }
    if (props.value) { this.widget.addFragment(this.valueContainer, props.value); }

    // Set error image properties.
    if (props.errorImage || props.valid != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.errorImage == null) {
            props.errorImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.errorImage.visible = !this.valid;

        // Update/add fragment.
        this.widget.updateFragment(this.errorImageContainer, this.errorImage.id, 
            props.errorImage);
    }

    // Set required image properties.
    if (props.requiredImage || props.required != null) {
        // Ensure property exists so we can call setProps just once.
        if (props.requiredImage == null) {
            props.requiredImage = {}; // Avoid updating all props using "this" keyword.
        }

        // Set properties.
        props.requiredImage.visible = this.required;

        // Update/add fragment.
        this.widget.updateFragment(this.requiredImageContainer, 
            this.requiredImage.id, props.requiredImage);
    }

    // Set contents.
    if (props.contents) {
        // Remove child nodes.
        this.widget.removeChildNodes(this.contentsContainer);

	for (var i = 0; i < props.contents.length; i++) {
            this.widget.addFragment(this.contentsContainer, props.contents[i], "last");
        }
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.selectBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class defines functions and properties for
 * widgets based on the "select" HTML element.
 *
 * <h3>Dojo attach points</h3>
 * A <code>selectBase</code>  subclass's template must
 * define the following attach point identifiers.
 * <ul>
 * <li><code>listContainer</code> - the <code>select</code> element.</li>
 * <li><code>brNode</code> - the element that controls the label postion.
 * <li><code>optionNode</code> - the element to clone for an
 * <code>option</code> element.
 * </li>
 * <li><code>optionGroupContainer</code> - the element to clone for an
 * <code>optGroup</code> element.</li>
 * <li><code>memberOptionNode</code> - the element to clone for an
 * <code>option</code> in an <code>optGroup</code> element.</li>
 * </ul>
 * <h3>The <code>label</code> object property</h3>
 * The <code>label</code> property is an object that defines the properties
 * for a widget that is rendered by the <code>selectBase</code>
 * class to represent a label. Minimally, only the <code>value</code> 
 * property of the label object property must be non null for 
 * the <code>selectBase</code> widget to render a label. If only the
 * <code>value</code> property is specified the following default
 * values will be used to create an instance of 
 * <code>webui.suntheme4_2.widget.label</code>.
 * <p>
 * <ul>
 * <li><code>widgetType</code> -
 * <code>webui.suntheme4_2.widget.label</code></li>
 * <li><code>id</code> - this.id + "_label"</li>
 * <li><code>htmlFor</code> - this.listContainer.id</li>
 * </ul>
 * <p>See <code>postCreate</code> and <code>getLabelProps</code>
 * </p>
 * <h3>The <code>options</code> array property</h3>
 * The <code>options</code> array property defines the contents of the
 * <code>select</code> HTML element. The contents of the array are 
 * translated into <code>option</code> and <code>optGroup</code> HTML
 * elements. Each element of the <code>options</code> array property
 * is considered an object with the following properties. Note that
 * "NA" in the following table means "not applicable", meaning there
 * is no HTML element property counterpart.
 * <p>
 * <table border="1px">
 * <tr><th>Property</th><th>Type</th><th>Description</th>
 * <th>HTML option or optGroup element property</th></tr>
 * <tr><td>label</td><td>String</td><td>
 * The text that appears as the choice in the rendered <code>option</code>
 * HTML element. (See note[1] below)
 * </td><td>label</td></tr>
 * <tr><td>value</td><td>String</td><td>
 * The value for this option. It is submitted in a request if this option is
 * selected.</td><td>value</td></tr>
 * <tr><td>separator</td><td>boolean</td><td>
 * If true this option represents a separator and cannot be selected.
 * </td><td>NA</td></tr>
 * <tr><td>group</td><td>boolean</td><td>
 * If true this option represents a group and cannot be selected. An
 * <code>optGroup</code> HTML element is rendered to represent this option.
 * </td><td>NA</td></tr>
 * <tr><td>escape</td><td>boolean</td><td>
 * If false the label string will be evaluated as HTML markup.
 * </td><td>NA</td></tr>
 * <tr><td>selected</td><td>boolean</td><td>
 * If true this option will be initially selected. 
 * </td><td>selected</td></tr>
 * <tr><td>disabled</td><td>boolean</td><td>
 * If true this option will be initially disabled and cannot be selected.
 * </td><td>disabled</td></tr>
 * <tr><td>title</td><td>String</td><td>
 * The HTML title attribute text.</td><td>title</td></tr>
 * </table>
 * </p>
 * <p>
 * The option object may also define the javascript event properties
 * suitable for an <code>option</code> or <code>optGroup</code> HTML
 * element and they will be assigned to the element's DOM node.
 * </p>
 * <p>
 * <ul>
 * <li>Note[1] Browser runtime behavior, contrary to the HTML and DOM
 * specifications, indicates that the DOM <code>label</code> property
 * is not rendered and only the <code>text</code> property is rendered.
 * This means that the <code>label</code> property of an
 * <code>options</code> array element will
 * be assigned to an <code>option</code> or <code>optGroup</code>
 * DOM node's <code>text</code> property.</li>
 * </ul>
 * </p>
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.selectBase", 
	webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.required = false;
        this.valid = true;

        // Flag to remember that last label style class.
        this._lastLabelOnTopClassName = null; // never been set

        // Flag to prevent blank entries in the drop down for the original 
        // empty dojo attach point nodes -- see "setOptions".
        this._alreadyRemoved = false;
    }
});

/**
 * This function is called by the <code>_onChangeCallback</code>
 * function to set the option element's className attribute with appropriate
 * selected and disabled selectors.
 * <p>
 * This method calls <code>this.getOptionClassName</code> once for each
 * option. Subclasses should implement <code>getOptionClassName</code>
 * in order to assign the appropriate CSS class name, to reflect a disabled or
 * selected option and any other properties due to an onChange event.
 * <p>
 * See <code>getOptionClassName</code> for the default behavior.
 * </p>
 * <p>
 * Before this method calls <code>this.getOptionClassName</code>, it has
 * addressed an IE issue where disabled options can be selected, and has
 * deslected, selected disabled options.
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.changed = function(ev) {
    var options = this.listContainer.options;

    // IE allows disabled options to be selected. Ensure that
    // disabled options never appear as selected options.
    //
    if (webui.suntheme4_2.browser.isIe()) {
	for (var i = 0; i < options.length; ++i) {
	    if (options[i].disabled == true && options[i].selected == true) {
		if (this.listContainer.multiple == true) {
		    options[i].selected = false;
		} else {
		    // Don't we need to set the disabled option's
		    // selected to false too ?
		    //
		    this.listContainer.selectedIndex = -1;
		}
	    }
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this.getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	} 
    } else {
	for (var i = 0; i < options.length; ++i) { 
	    // Only set the option if its value is different
	    // than what is returned. Note that if this method
	    // returns null, the option may not visually reflect
	    // the true state.
	    //
	    var cn = this.getOptionClassName(options[i]);
	    if (cn != null && options[i].className != cn) {
		options[i].className = cn;
	    }
	}
    }
    return true;
};

/**
 * This method copies option properties from <code>fromOption</code> to 
 * <code>toOption</code>. Note that <code>fromOption</code> is assumed to
 * be a DOM <code>option</code> element. There is not a one to one mapping
 * of properties by name from a DOM <code>option</code> element to a
 * widget's <code>options</code> array <code>option</code> array element.
 * The properties that are set in <code>setOptionProps</code> and
 * <code>setGroupOptionProps</code> are copied.
 * @param {Object} toOption An <code>Object</code> to receive
 * <code>selectBase</code> option properties.
 * @param {Object} fromOption An <code>option</code> or <code>optgroup</code>
 * element DOM node to copy <code>selectBase</code> option properties from.
 * @return {boolean} Returns true.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._copyOption = function(toOption, fromOption) {
    var domhandlers = [ "onblur", "onchange", "onclick", "ondblclick",
	"onfocus", "onkeydown", "onkeypress", "onkeyup", "onmousedown",
	"onmouseout", "onmouseover", "onmouseup", "onmousemove", "onresize"];

    var widgethandlers = [ "onBlur", "onChange", "onClick", "onDblClick",
	"onFocus", "onKeyDown", "onKeyPress", "onKeyUp", "onMouseDown",
	"onMouseOut", "onMouseOver", "onMouseUp", "onMouseMove", "onResize"];

    for (var i = 0; i < domhandlers.length; ++i) {
	if (fromOption[domhandlers[i]]) {
	    toOption[widgethandlers[i]] = fromOption[domhandlers[i]];
	}
    }
    toOption.label = fromOption.label;
    toOption.value = fromOption.value;
    toOption.escape = fromOption.escape;
    toOption.disabled = fromOption.disabled;
    toOption.defaultSelected = fromOption.defaultSelected;
    toOption.selected = fromOption.selected;
    toOption.title = fromOption.title;
    toOption.className = fromOption.className;
    toOption.group = fromOption.group;
    if (toOption.group == true) {
	toOption.options = this._copyOptions(fromOption);
    }
    return true;
};

/**
 * Returns an <code>options</code> array of <code>option</code> objects
 * as defined by <code>selectBase</code> (see selectBase.setOptions).
 * <code>domNode</code> is assumed to be a <code>select</code> element
 * or an <code>optgroup</code>.
 * <p>
 * There is not a one to one mapping of properties by name from a DOM
 * <code>option</code> or <code>optgroup</code> element to a
 * widget's <code>options</code> array <code>option</code> object array element.
 * The properties that are set in <code>setOptionProps</code> and 
 * <code>setGroupOptionProps</code> are copied to <code>Object</code>s
 * and returned as the elements of the returned array. The returned
 * array is equivalent to the <code>options</code> array used to create
 * a selectBase subclass, like listbox or dropdown.
 * @param {Object} domNode A <code>select</code> element DOM node or an
 * <code>optgroup</code> DOM node.
 * @return {Object} An array of <code>selectBase</code> options. If 
 * <code>domNode</code> has no child nodes, an empty array is returned.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._copyOptions = function(domNode) {
    var newoptions = [];
    if (!domNode.hasChildNodes()) {
	return newoptions;
    }
    var len = domNode.childNodes.length;
    for (var j = 0; j < len; ++j) {
	newoptions[j] = new Object();
	if (domNode.childNodes != null) {
	    this._copyOption(newoptions[j], domNode.childNodes[j]);
	}
    }
    return newoptions;
};

/**
 * This function returns the CSS class name for an HTML <code>option</code> or
 * <code>optgroup</code> element.
 * <p>
 * This implementation returns <code>option.className</code>
 * This method should be overridden by subclasses.
 * </p>
 *
 * @param {Object} option Key-Value pairs of properties.
 * @return {String} The HTML option element class name.
 */
webui.suntheme4_2.widget.selectBase.prototype.getOptionClassName = function(option) {
    // Make sure that if a subclass does not implement this method
    // that it causes no change to option.
    //
    return option.className;
};

/**
 * This function replaces all <code>option</code> and <code>optgroup</code>
 * elements in the HTML <code>select</code> element,
 * <code>this.listContainer</code>.
 * <p>
 * All options are replaced with the options specified in <code>props</code>.
 * If <code>props</code> is null, <code>false</code>
 * is returned and no change to the <code>select</code> element occurs. If
 * <code>props</code> contains no properties all options in the 
 * <code>select</code> element are removed.
 * </p>
 * @param {Object} props Key-Value pairs of option properties.
 * @config {String} label The text that appears as the choice in the 
 * rendered <code>option</code>.
 * @config {String} value The value for this option, which is submitted
 * in a request if this option is selected.
 * @config {boolean} separator If true this option represents a separator and
 * cannot be selected.
 * @config {boolean} group If true this option represents a group and
 * cannot be selected.
 * @config {boolean} escape If false the label string will be evaluated as
 * HTML markup.
 * @config {boolean} selected If true this option will be initially
 * selected.
 * @config {boolean} disabled If true this option will be initially 
 * disabled and cannot be selected.
 * @config {String} title The HTML title attribute text.
 * @return {boolean} true if successful; otherwise, false.
 * </p>
 * <p>
 * The option object may also define javascript event properties suitable for
 * an <code>option</code> or <code>optGroup</code> HTML element.
 * </p>
 */
webui.suntheme4_2.widget.selectBase.prototype.setOptions = function(props) {
    if (props == null) {
	return false;
    }

    // Remove all existing options
    //
    // Note: this has to be done. Otherwise, you'll get blank entries in the 
    // drop down for the original empty dojo attach point nodes.
    //

    if (!this._alreadyRemoved) {
        this.listContainer.removeChild(this.optionNode); 
        this.optionGroupContainer.removeChild(this.memberOptionNode);
        this.listContainer.removeChild(this.optionGroupContainer);
        this._alreadyRemoved = true;
    }

    // Cleaning up the old options
    //
    while (this.listContainer.firstChild) {
        this.listContainer.removeChild(this.listContainer.firstChild);
    }

    var thisNode;
    for (var i = 0; i < props.options.length; i++) {

	var pOption = props.options[i];

	var isie = webui.suntheme4_2.browser.isIe();
	if (pOption.group == null || pOption.group == false) {

	    // For some reason, ie is prone to painting problems (esp. after a 
	    // style change) when using the DOM to populate options, but 
	    // apparently not when the options are in a group
	    //
	    // There is no working "clone" on "option" nodes in IE
	    // Manually get the attributes and then the "text"
	    // DOM attribute. I don't think an option can have HTML
	    // markup in the body, at least literally in the template
	    // so this strategy should be be sufficient enough to get
	    // the template's option attributes
	    //
	    if (isie) {
		thisNode = new Option();
		var len = this.optionNode.attributes.length;
		for (var j = 0; j < len; ++j) {
		    thisNode.setAttribute(this.optionNode.attributes[j].name,
			    this.optionNode.attributes[j].value);
		}
		// Need to manually do text, although this is likely null
		// in the template.
		//
		thisNode.text = this.optionNode.text;

	    } else {
		thisNode = this.optionNode.cloneNode(true);
	    }

	    // Set the properties on this option element
	    this.setOptionProps(thisNode, pOption);
	    
	    // Would it be better to create all the nodes and then
	    // add them to the "options" array or append then ?
	    // Granted this would mean iterating twice.
	    // But it might be better if something fails
	    // and if so leave the original list alone.
	    //

	    // Append this option node to the select
	    //
	    if (isie) {
		var idx = this.listContainer.options.length;
		var isSelected = thisNode.selected;
		this.listContainer.options[idx] = thisNode;

		// VERIFY THAT WE STILL NEED THIS
		//

		// explicitly set the selected property again!
		// this is necessary to work around a defect in 
		// some versions of IE6
		//
		this.listContainer.options[idx].selected = isSelected;
	    } else {
		this.listContainer.appendChild(thisNode);
	    }
	} else {
	    // group option optgroup
	    //
	    thisNode = this.optionGroupContainer.cloneNode(true);
	    
	    // Set the properties on this optgroup element
	    //
	    this.setGroupOptionProps(thisNode, pOption);
	    
	    // Add the option elements to this group
	    // Supports only one level
	    //
	    var thisSubNode;
	    for (var ix = 0; ix < pOption.options.length; ix++) {
		thisSubNode = this.memberOptionNode.cloneNode(true);
		this.setOptionProps(thisSubNode, pOption.options[ix]);
		thisNode.appendChild(thisSubNode); 
	    }

	    // Append this optgroup node to the select element
	    // only after constructing its option children
	    //
	    this.listContainer.appendChild(thisNode);
	}
    }
    return true;
};

/**
 * This function is used to get widget properties. Please see 
 * subclass's setProps() function for a complete list of supported properties.
 * <br/>
 * This base class handles the following properties.
 * <ul>
 * <li><code>disabled</code> - If true the select element is disabled.</li>
 * <li><code>label</code> - This object defines the widget and properties
 * for a label.</li>
 * <li><code>labelOnTop</code> - If true the label appears above the select
 * element.</li>
 * <li><code>options</code> - This array object contains the select element 
 * options and their attributes. See <code>setOptions</code> and the 
 * <code>selectBase</code> overview for details on the <code>options</code> 
 * array.</li>
 * <li><code>required</code> - If true the a selection is required.</li>
 * <li><code>valid</code> - If true the widget has a valid selection.</li>
 * <li><code>width</code> - This value will be assigned to the 
 * <code>listContainer.style.width</code> attribute to set the width of the
 * select element.</li>
 * </ul>
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.selectBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Get properties.
    // Should we return the default theme value for labelOnTop
    // if "this.labelOnTop" has not been set ?
    //
    if (this.labelOnTop != null) { props.labelOnTop = this.labelOnTop; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.label) { props.label = this.label; }
    if (this.required != null) { props.required = this.required; }
    if (this.valid != null) { props.valid = this.valid; }
    if (this.width != null) { props.width = this.width; }

    // After widget has been initialized, get actual select element state
    // Note that the options that exist on the select element will
    // not include the original "options" props that are not
    // HTML option or HTML optgroup attributes.
    //
    if (this.isInitialized() == true && this.listContainer != null) {
	// We need to copy the options from this.listContainer
	// or else they will be destroyed if passed back to addProps.
	//
	var opts = this._copyOptions(this.listContainer);
        props.options = opts;
	// It's not clear if we should do this.
	//
	// this.options = opts;
    } else if (this.options != null) {
        props.options = this.options;
    }
    return props;
};

/**
 * This function is used to obtain the underlying HTML select element.
 *
 * @return {Node} The HTML select element.
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectElement = function() {
    return this.listContainer;
};

/**
 * This function is used to obtain the index 
 * of the selected option.
 *
 * @return {Integer} The selected index of underlying select element
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectedIndex = function() { 
    return this.listContainer.selectedIndex; 
};

/**
 * This function is used to directly set selected index on the underlying select box
 * of the selected option.
 *
 * @return {Boolean} true
 */
webui.suntheme4_2.widget.selectBase.prototype.setSelectedIndex = function(index) { 
    if (index >=0 && index < this.listContainer.options.length) {
        this.listContainer.selectedIndex = index;
    }
    return true;
};

/**
 * This function is used to obtain the <code>label</code> attribute of
 * the selected option.
 * <p>
 * If no option is selected, this function returns null.
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code> attribute is true,
 * this method returns the first selected option's label.
 * </p>
 * 
 * @return The label of the selected option, or null if none is selected. 
 * @return {String} The label attribute of the selected option.
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectedLabel = function() { 
    var index = this.listContainer.selectedIndex; 

    if (index == -1) { 
        return null; 
    } else { 
	return this.listContainer.options[index].label;
    }
};

/**
 * This function is used to obtain the <code>value</code> attribute
 * of the selected option.
 * <p>
 * If no option is selected, this function returns null. 
 * </p>
 * <p>
 * If the underlying select element's <code>multiple</code>" attribute is true,
 * this method returns the first selected option's value.
 * </p>
 *
 * @return {String} The selected option value or null if none is selected. 
 */
webui.suntheme4_2.widget.selectBase.prototype.getSelectedValue = function() { 
    var index = this.listContainer.selectedIndex; 
    if (index == -1) { 
        return null; 
    } else { 
        return this.listContainer.options[index].value; 
    }
};

/**
 * This function is invoked for the select element's <code>onchange</code>
 * event.
 * <p>
 * If <code>this.disabled</code> is true, return false.<br/>
 * This method calls the handler defined by the <code>props.onChange</code>
 * property. If that handler returns false, false is returned. If the
 * <code>props.onChanged</code> handler returns true, then
 * <code>this.changed</code> is called and its return value is returned.
 * </p>
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._onChangeCallback = function(event) {
    if (this.disabled == true) {
        return false;
    }

    // If function returns false, we must prevent the auto-submit.
    //
    var result = (this.listContainer._onchange)
        ? this.listContainer._onchange(event) : true;
    if (result == false) {
        return false;
    }

    // Set style classes.
    return this.changed(event);
};

/**
 * This function is used to fill in remaining template properties, after the
 * <code>buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.labelContainer.id = this.id + "_label";
        this.listContainer.id = this.id + "_list";
	this.listContainer.name = this.listContainer.id;
    }

    // Subclasses have set this property
    // make the br node visible else hide it.
    //
    if (this.labelOnTop != null) {
	webui.suntheme4_2.common.setVisibleElement(this.brNode, this.labelOnTop);
	this._lastLabelOnTopClassName = this.getLabelClassName(null);
    }

    if (this.label && this.label.value != null &&
	    !this.widget.isFragment(this.label)) {

	this.label = this.widget.getWidgetProps("label", this.getLabelProps());
	this.label.id = this.id + "_label";
    }

    if (this.label != null) {
	if (this.required != null && this.required == true) {
	    this.label.required = true;
	}
	if (this.valid != null && this.valid != true) {
	    this.label.valid = false;
	}
	if (this._lastLabelOnTopClassName != null) {
	    webui.suntheme4_2.common.addStyleClass(this.label, 
		this._lastLabelOnTopClassName);
	}
    }

    // Set public functions.
    this.domNode.getSelectedValue = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectedValue(); };
    this.domNode.getSelectedLabel = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectedLabel(); };
    this.domNode.getSelectedIndex = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectedIndex(); };
    this.domNode.setSelectedIndex = function(index) { 
	return webui.suntheme4_2.dijit.byId(this.id).setSelectedIndex(index); };
    this.domNode.getSelectElement = function() { 
	return webui.suntheme4_2.dijit.byId(this.id).getSelectElement(); };

    // Set events.
    this.dojo.connect(this.listContainer, "onchange", this, 
	"_onChangeCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * Return an Object Literal of label properties desired
 * by the selectBase widget.
 * <p>
 * This implementation returns null. This method should be implemented
 * in subclasses to return label properties desired by the subclass.
 * </p>
 * @return {Object} This implementation returns null;
 */
webui.suntheme4_2.widget.selectBase.prototype.getLabelProps = function() {
    return null;
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>optgroup</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this.setEventProps</code>
 * and then calls <code>this.getOptionClassName</code>. 
 * Subclasses should implement <code>getOptionClassName</code> if the default
 * behavior of <code>selectBase.getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific 
 * CSS class names and properties. For all the properties defined in the 
 * option array see <code>setOptions</code>.
 * </p>
 * @param {Node} element The optgroup DOM node
 * @param {Object} option Key-Value pairs of properties for the optgroup node.
 * @config {boolean} disabled If true the optgroup is disabled.
 * @config {String} label The optgroup choice text.
 * @config {String} title The optgrp HTML title attribute value.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.setGroupOptionProps =
	function(element, option) {
    element.label = option.label;
  
    if (option.disabled != null) {
        element.disabled = option.disabled;
    }

    if (option.title != null) {
	element.title = option.title;
    }

    // optGroup elements can have events.
    this.setEventProps(element, option);

    // Non HTML properties
    //
    if (option.escape != null) {
	element.escape = option.escape;
    }
    element.group = true;

    // Do this last so that all other properties will have been set.
    // Note that if this does return null, which it shouldn't
    // then the visual appearance may not reflect the 
    // actual state.
    //
    var cn =  this.getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function is used to set the <code>option</code> properties on the
 * <code>option</code> HTML DOM element, <code>element</code>.
 * <p>
 * This method assigns the <code>option.label</code>,
 * <code>option.disabled</code>, <code>option.title</code> properties
 * to <code>element</code>, non HTML properties defined in 
 * <code>option</code>, calls <code>this.setEventProps</code>
 * and then calls <code>this.getOptionClassName</code>. 
 * Subclasses should implement <code>getOptionClassName</code> if the default
 * behavior of <code>selectBase.getOptionClassName</code> is not appropriate.
 * Subclasses usually subclass this method to provide subclass specific CSS
 * class names and properties. For all the properties defined in the option
 * array see <code>setOptions</code>
 * </p>
 * @param {Node} element The <code>option</code> HTML element DOM node
 * @param {Object} option Key-Value pairs of properties for the option node.
 * @config {boolean} disabled If true the option is disabled.
 * @config {String} label The option choice text.
 * @config {String} title The value of the option's HTML <code>title</code>
 * attribute.
 * @config {boolean} escape If false the label string will be evaluated
 * as HTML markup.
 * @config {boolean} separator If true this option acts as a separator.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.selectBase.prototype.setOptionProps =
        function(element, option) {
    element.value = option.value;

    // If option.escape is true, we want the text to be displayed
    // literally. To achieve this behavior, do nothing.
    // If option.escape is false, we want any special sequences in the text 
    // (e.g., "&nbsp;") to be displayed as evaluated (i.e., unescaped).
    //
    // This is because it is assumed that the value of the property has 
    // already been escaped. This may be true if the value was set in the 
    // HTML response. This is why we call "unescapeHTML".
    //
    // However it may not be true if the text was set in javascript.
    //

    // Note that the HTML and ECMA JS DOM language binding indicates
    // that the "label" property should be used. However browsers do not
    // respect those specs. In fact label does not even work at all.
    // Setting the string on label results in no string being displayed.
    //
    var textToUse = option.label;
    if (option.escape != null && option.escape == false) {
	// Prototype method.
	element.text = this.prototypejs.unescapeHTML(textToUse);
    } else {
	element.text = textToUse;
    }
    // We must record the state of the original option
    //
    element.label = option.label;

    // Never allow a disabled option to be spedified as selected
    // This is only a problem if an option is initially selected
    // and disbled on most browsers since once disabled it cannot
    // be selected except for IE. For IE it is ensured that a
    // disabled option cannot be "selected" in the
    // "change" method.
    //
    if (option.disabled != null && option.disabled == true) {
	option.selected = false;
    }

    element.defaultSelected = option.selected != null ? option.selected : false;

    // When creating new options, defaultSelected should also be set.
    // Actually only "defaultSelected" should be set and set only once
    // since it is supposed to be the "initial" value.
    // However all options are always recreated in this implementation
    // when this method is called by setOptions.
    //
    element.selected = element.defaultSelected;
	
    element.disabled = option.disabled != null ? option.disabled : false;

    if (option.title != null) {
	element.title = option.title;
    }

    // option elements can have events.
    this.setEventProps(element, option);

    // In order for subclasses to return the appropriate
    // selector all the properties in the option properties array must
    // be placed on the HTML element option and optgroup instances.
    //
    // This is because "getOptionClassName" is called in two different
    // contexts. It is called here where both the HTML option or
    // optgroup element is available and the original option properties 
    // are known, and during the "onchange" event where only the 
    // HTML option element is known. Therefore if the choice of selector 
    // is based on whether or not an option is a "separator" then, 
    // the separator state must be available from the HTML option or optgroup
    // element instance.
    //
    // This issue could be mitigated by using descendant selectors
    // vs. literal selectors. In other words, if every time an option
    // became disabled, the className attribute would be set (or merged) with 
    // Dis. A selector could be defined as ".Lst option.Dis" which would
    // implicitly match. If attribute selectors worked we could use
    // that instead. Either strategy would eliminate the need for
    // methods like "getOptionClassName".
    //
    // Only set non HTML element props if we have to.
    //
    if (option.separator != null && option.separator == true) {
	element.separator = option.separator;
	element.disabled = true;
    }
    if (option.escape != null && option.escape == false) {
	element.escape = option.escape;
    }
    element.group = false;

    // Call getOptionClassName only after setting HTML props
    // and option props on element.  Note that if this method
    // returns null, the option may not visually reflect
    // the true state.
    //
    var cn = this.getOptionClassName(element);
    if (cn != null) {
	element.className = cn;
    }
    return true;
};

/**
 * This function is used to set widget properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled If true the select element is disabled.
 * @config {Object} label The properties for a label widget.
 * @config {boolean} labelOnTop If false the label will appear to the left
 * of the <code>select</code> element, aligned with the first list option. 
 * @config {Object} options An array of Objects that represent the 
 * <code>select</code> element's <code>option</code> and 
 * <code>optgroup</code> elements. @see #setOptions.
 * @config {boolean} required If true the a selection is required.
 * @config {boolean} valid If true the widget has a valid selection.
 * @config {String} width This value will be assigned to the 
 * <code>listContainer.style.width</code> attribute to set the width of the
 * select element.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype.setProps = function(props) {
    // Always call inherited setProps
    //
    if (props == null) {
	return null;
    }

    // Set a flag indicating there is a label
    //
    var havelabel = this.label != null && this.label.id != null;

    // Always update the brNode state, even if there isn't a label
    // but only if it is changing.
    //
    var labelProps = null;
    if (props.labelOnTop != null && props.labelOnTop != this.labelOnTop) {

	// The ontop state is changing
	// If the state is "ontop" then make the br node 
	// visible else hide it.
	//
	webui.suntheme4_2.common.setVisibleElement(this.brNode, props.labelOnTop);

	// Remember the new ontop selector.
	//
	this._lastLabelOnTopClassName = this.getLabelClassName(
	    props.labelOnTop);

	// If we have a label remove the last ontop selector from the label.
	// We must strip it from the dom node.
	//
	if (havelabel) {

	    var labelnode = document.getElementById(this.label.id);
	    webui.suntheme4_2.common.stripStyleClass(labelnode,
		this._lastLabelOnTopClassName);

	    // If we are toggling ontop for an existing label or the
	    // application is updating or creating a label, add the
	    // correct ontop selector to props.
	    //
	    var labelProps = props.label != null 
		? props.label 
		: (props.label = {});

	} else if (props.label && this._lastLabelOnTopClassName != null) {
	    labelProps = props.label;
	    webui.suntheme4_2.common.addStyleClass(labelProps, 
		this._lastLabelOnTopClassName);
	}
    }

    var togglerequired = props.required != null && 
	props.required != this.required;
    var togglevalid = props.valid != null && props.valid != this.valid;

    // Update the label required and valid properties if necessary.
    // If labelProps is not null, then the labelOnTop property is changing.
    // If labelProps is null, then if there is an existing label and
    // the required or valid attributes is changing, make sure label.props
    // is updated with the new required and valid states.
    //
    if (labelProps == null && havelabel && (togglerequired || togglevalid)) {
	labelProps = props.label != null ? props.label : (props.label = {});
    }

    if (labelProps != null) {
	if (togglerequired) {
	    labelProps.required = props.required;
	}
	if (togglevalid) {
	    labelProps.valid = props.valid;
	}
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {Object} label The properties for a label widget.
 * @config {Object} options An array of Objects that represent the 
 * <code>select</code> element's <code>option</code> and 
 * <code>optgroup</code> elements. @see #setOptions.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.selectBase.prototype._setProps = function(props) {
    // Always call inherited _setProps
    //
    if (props == null) {
	return null;
    }

    // A web app devleoper could return false in order to cancel the 
    // auto-submit. Thus, we will handle this event via the onChange
    // call back.
    //
    if (props.onChange) {
        // Set private function scope on DOM node.
        this.listContainer._onchange = (typeof props.onChange == 'string')
	    ? new Function("event", props.onChange) 
	    : props.onChange;

        // Must be cleared before calling setEventProps() below.
        props.onChange = null;
    }

    if (props.disabled != null) {
	if (this.listContainer.disabled != props.disabled) {
	    this.listContainer.disabled = props.disabled;
	}
    }

    // Add option and optgroup elements in the select element
    //
    if (props.options) {
        this.setOptions(props);
    }

    if (props.width != null && props.width != "") {
	this.listContainer.style.width = props.width;
    }


    // If _setProps is called during initializat then we will be
    // creating the label and props.label == this.label.
    // If _setProps is called from setProps, then we are updating the
    // label. The label needs to be updated if the labelOnTop or
    // or required or valid properties are changed.
    // The application may also be creating a label after the
    // selectBase widget was created.
    //
    if (props.label) {
	// Now update or create the label.
	// If we don't have an existing label, this.label.id == null
	// then call addFragment in case the application is
	// creating the label after the selectBase widget was created.
	//
	if (this.label != null && this.label.id != null) {
	    this.widget.updateFragment(this.labelContainer, this.label.id,
		props.label);
	} else {
	    this.widget.addFragment(this.labelContainer, props.label);
	}
    }

    // Set more properties.
    this.setCommonProps(this.listContainer, props);
    this.setEventProps(this.listContainer, props);

    // Set remaining properties.
    //
    return this.inherited("_setProps", arguments);
};

/**
 * @name webui.suntheme4_2.widget.dropDown
 * @extends webui.suntheme4_2.widget.selectBase
 * @class This class contains functions for the dropDown widget.
 * @constructor This function is used to construct a dropDown widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.dropDown", webui.suntheme4_2.widget.selectBase, {
    // Set defaults.
    constructor: function() {
        this.labelOnTop = webui.suntheme4_2.widget.common.getMessageBoolean(
	    "dropDown.labelOnTop", false);
        this.submitForm = false;
        this.width = webui.suntheme4_2.theme.common.getMessage("dropDown.width", null);
    },
    widgetName: "dropDown" // Required for theme properties.
});

/**
 * This function is called by the <code>selectBase::onChangeCallback</code>
 * method in respsonse to an onchange event.
 * <p>
 * If this instance is configured as a "jump" menu,
 * <code>submitForm</code> is true, the containing form is submitted.
 * </p>
 * <p>
 * If this instance is configured as a standard "dropdown" menu, then
 * the superclass's <code>changed</code> method is called.
 * See <code>selectBase.changed</code>.<br/>
 * See <code>selectBase.onChangeCallback</code>
 * </p>
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dropDown.prototype.changed = function() {

    // A null submitForm is the same as a standard menu.
    //
    if (this.submitForm != true) {
        // Drop down changed.
        return this.inherited("changed", arguments);
    } else {

        // Jump drop down changed.
        var jumpDropdown = this.listContainer; 

        // Find the <form> for this drop down
        var form = jumpDropdown.form; 

        if (typeof form != "undefined" && form != null) { 
            this.submitterHiddenNode.value = "true";

            // Set style classes.
            var options = jumpDropdown.options;
            for (var i = 0; i < options.length; i++) { 
                options[i].className = this.getOptionClassName(options[i]);
            }
            form.submit();
        }
    }
    return true; 
};

/**
 * The dropDown defines the following event topics. The submit topics
 * are only valid if the <code>submitForm</code> property is true.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.dropDown.event =
        webui.suntheme4_2.widget.dropDown.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "webui_suntheme4_2_widget_dropDown_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "webui_suntheme4_2_widget_dropDown_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_dropDown_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_dropDown_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        beginTopic: "webui_suntheme4_2_widget_dropDown_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for.
	 * This topic is only valid if the <code>submitForm</code>
	 * property is <code>true</code>
	 */
        endTopic: "webui_suntheme4_2_widget_dropDown_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.dropDown.prototype.getClassName = function() {

    var cn = this.widget.getClassName("DROPDOWN", "");
    return (this.className) ? cn + " " + this.className : cn;
};

/**
 * Return an Object Literal of label properties desired
 * by the dropdown widget.
 * <p>
 * This implementation sets
 * the <code>dropDown.labelLevel</code> and <code>dropDown.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>listContainer.id</code>.
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 */
webui.suntheme4_2.widget.dropDown.prototype.getLabelProps = function() {

    var cn = this.getLabelClassName(null);
    var lvl = this.widget.getMessage("dropDown.labelLevel", null, 2);
    var props = {};
    if (cn != null) {
       props.className = cn;
    }
    props.labelLevel = lvl;
    props.htmlFor = this.listContainer.id;
    this.prototypejs.extend(props, this.label);
    return props;
};

/**
 * Return a CSS selector for the dropDown label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property and the whether or not the
 * dropdown is a "jump" dropDown. If <code>labelOnTop</code>
 * property is <code>true</code> null is returned. If the 
 * <code>labelOnTop</code> property is true, then the value of the
 * the of the theme keys, <code>MENU_JUMP_LABEL_ALIGN</code> or 
 * <code>MENU_STANDARD_LABEL_ALIGN</code> from the 
 * <code>styles</code> category are returned if the drop down is
 * a "jump" dropDown or plain dropDown, respectively.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the dropDown label.
 */
webui.suntheme4_2.widget.dropDown.prototype.getLabelClassName = function(ontop) {

    var labelontop = ontop != null ? ontop : this.labelOnTop;
    if (labelontop == true) {
	return null;
    }
    // Jumpmenu ?
    //
    return (this.submitForm != null && this.submitForm == true)
	? this.widget.getClassName("MENU_JUMP_LABEL_ALIGN", null)
        : this.widget.getClassName("MENU_STANDARD_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> property and whether or not the dropDown
 * is a "jump" dropDown. The returned selector is the value of a 
 * property defined in the <code>styles</code> category, as shown 
 * in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>jump dropDown</th></tr>
 * <tr>
 * <td>MENU_STANDARD</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_STANDARD_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>MENU_JUMP_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the listContainer HTML element.
 * @private
 */
webui.suntheme4_2.widget.dropDown.prototype._getListContainerClassName =
	function(disabled, jumpmenu) {

    var key = "MENU_STANDARD";
    if (disabled == true) {
	if (jumpmenu == true) {
	    key = "MENU_JUMP_DISABLED";
	} else {
	    key = "MENU_STANDARD_DISABLED";
	}
    } else if (jumpmenu == true) {
	key = "MENU_JUMP";
    }
    return this.widget.getClassName(key, null);
};

/**
 * This function is used to get widget properties.
 * <p>See <code>setProps</code> for the list of supported properties,
 * </p>
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.dropDown.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Get properties.
    if (this.submitForm != null) { props.submitForm = this.submitForm; }

    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name defined by 
 * the following theme keys from the <code>styles</code> theme category
 * is returned for the given states, based on 
 * the properties in <code>element</code>.
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>*_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>*_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>*_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td>
 * <td>*_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>*_OPTION</td>
 * </tr>
 * </table><br/>
 * Note:  * is <code>MENU_JUMP</code> for a jump menu and
 * <code>MENU_STANDARD</code> if not.
 * </p>
 * <p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "MENU_STANDARD_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 */
webui.suntheme4_2.widget.dropDown.prototype.getOptionClassName = function(element) {

    var jumpmenu = this.submitForm != null && this.submitForm == true;

    var key = null;
    if (jumpmenu) {
	key = "MENU_JUMP_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_JUMP_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_JUMP_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_JUMP_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_JUMP_OPTION_DISABLED";
	    }
	}
    } else {
	key = "MENU_STANDARD_OPTION";
	if (element != null) {
	    if (element.separator == true) {
		key = "MENU_STANDARD_OPTION_SEPARATOR";
	    } else
	    if (element.group == true) {
		key = "MENU_STANDARD_OPTION_GROUP";
	    } else
	    if (element.selected == true) {
		key = "MENU_STANDARD_OPTION_SELECTED";
	    }

	    // Disabled wins
	    //
	    if (element.disabled == true) {
		key = "MENU_STANDARD_OPTION_DISABLED";
	    }
	}
    }
    return this.widget.getClassName(key);
};


/*
 * This function is used to fill in remaining template properties, after the
 * <code>buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dropDown.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.submitterHiddenNode.id = this.id + "_submitter";
    }

    // Set the CSS class name for the select element.
    //
    var jumpmenu = this.submitForm != null && this.submitForm == true;
    var disabled = this.disabled != null && this.disabled == true;

    webui.suntheme4_2.common.addStyleClass(this.listContainer, 
	this._getListContainerClassName(disabled, jumpmenu));
    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties defined in the
 * Object literal <code>props</code>
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for
 * a label.
 * @config {boolean} labelOnTop If true the label appears above the dropdown. 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See <code>selectBase.setOptions</code> or
 * <code>selectBase</code> overview for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm If true the dropDown performs like a
 * "jump menu" and submits the entire form when an option is selected.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title The a HTML title attribue for the <code>select</code>
 * element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.dropDown.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this.inherited("setProps", arguments);
    }

	
    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    // We need to clear the selector of the listContainer
    // when state changes from disabled to non disabled state
    // or from submitForm to standard dropdown.
    //
    var isjumpmenu = this.submitForm == true;
    var toggleJumpmenu = props.submitForm != null && 
	props.submitForm != isjumpmenu;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null && 
	props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleJumpmenu || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, isjumpmenu);
	webui.suntheme4_2.common.stripStyleClass(this.listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleJumpmenu ? props.jumpmenu == true : isjumpmenu);
	webui.suntheme4_2.common.addStyleClass(this.listContainer, cn);
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.dropDown.prototype._setProps = function(props) {
    // If props is null, create one to pass default theme values
    // to the superClass via props.
    //
    if (props == null) {
	return this.inherited("_setProps", arguments);
	//props = {};
    }

    var jumpmenu = props.submitForm != null && props.submitForm == true;

    // Add attributes to the hidden input for jump drop down.
    //
    if ((jumpmenu && this.submitterHiddenNode.name == null) || (jumpmenu &&
	    this.submitterHiddenNode.name != this.submitterHiddenNode.id)) {

	this.submitterHiddenNode.name = this.submitterHiddenNode.id;
	this.submitterHiddenNode.value = "false";
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.hiddenField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.hiddenField
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the hiddenField widget.
 * @constructor This function is used to construct a hiddenField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.hiddenField", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
    },
    widgetName: "hiddenField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.hiddenField.event =
        webui.suntheme4_2.widget.hiddenField.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hiddenField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hiddenField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hiddenField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hiddenField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hiddenField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hiddenField_event_submit_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.hiddenField.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.name) { props.name = this.name; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} name
 * @config {String} value Value of input.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hiddenField.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.hiddenField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.name) { this.domNode.name = props.name; }
    if (props.value) { this.domNode.value = props.value; }
    if (props.disabled != null) { 
        this.domNode.disabled = new Boolean(props.disabled).valueOf();
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.hyperlink");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.anchorBase");

/**
 * @name webui.suntheme4_2.widget.hyperlink
 * @extends webui.suntheme4_2.widget.anchorBase
 * @class This class contains functions for the hyperlink widget.
 * @constructor This function is used to construct a hyperlink widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.hyperlink", webui.suntheme4_2.widget.anchorBase, {
    // Set defaults.
    widgetName: "hyperlink" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.hyperlink.event =
        webui.suntheme4_2.widget.hyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_hyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_hyperlink_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.hyperlink.prototype.getClassName = function() {
    // Set default style.
    var className = (this.disabled == true)
        ? this.widget.getClassName("HYPERLINK_DISABLED","")
        : this.widget.getClassName("HYPERLINK","");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * Helper function to create callback for onClick event.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hyperlink.prototype.onClickCallback = function(event) {
    if (this.disabled == true) {
        event.preventDefault();
        return false;
    }

    // If function returns false, we must prevent the submit.
    var result = (this.domNode._onclick)
        ? this.domNode._onclick(event) : true;
    if (result == false) {
        event.preventDefault();
        return false;
    }
    if (this.href) {
        return false;
    }
    event.preventDefault();
    
    // If a form id isnt provided, use the utility function to
    // obtain the form id.
    if (this.formId == null) {
        var form = this.widget.getForm(this.domNode);
        this.formId = (form) ? form.id : null;
    }
    return this.submitFormData(this.formId, this.params);
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hyperlink.prototype.postCreate = function () {
    // If the href attribute does not exist, set "#" as the default value of the
    // DOM node.
    this.domNode.href = "#";

    // Create callback function for onClick event.
    this.dojo.connect(this.domNode, "onclick", this, "onClickCallback");

    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} formId The id of the HTML form element.
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {Array} params The parameters to be passed during request.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.hyperlink.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function submits the hyperlink if the hyperlink is enabled.
 *
 * @param {String} formId The id of the HTML form element.
 * @param {Array} params The parameters to be passed during request.
 * @return {boolean} false to cancel the JavaScript event.
 */
webui.suntheme4_2.widget.hyperlink.prototype.submitFormData = function (formId, params) {
    var theForm = document.getElementById(formId);
    var oldTarget = theForm.target;
    var oldAction = theForm.action;

    // Obtain HTML element for tab and common task components.
    var link = document.getElementById(this.id);
    if (link == null) {
        link = this.domNode;
    }

    // Set new action URL.
    var prefix;
    (theForm.action.indexOf("?") == -1) ? prefix = "?" : prefix = "&";
    theForm.action += prefix + link.id + "_submittedLink=" + link.id;               
        
    // Set new target.
    if (link.target && link.target.length > 0) {
        theForm.target = link.target;
    } else {
        theForm.target = "_self";
    }

    // Append query params to new action URL.
    if (params != null) {
        var x;
        for (var i = 0; i < params.length; i++) {
            x = params[i];
            theForm.action += "&" + params[i] + "=" + params[i + 1]; 
            i++;
        }
    }

    // Submit form.
    theForm.submit(); 

    // Restore target and action URL.
    if (link.target != null) {
        theForm.target = oldTarget;
        theForm.action = oldAction;
    }
    return false;        
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.image");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.image
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the image widget.
 * @constructor This function is used to construct a image widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.image", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.border = 0;
    },
    widgetName: "image" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.image.event =
        webui.suntheme4_2.widget.image.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_image_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_image_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_image_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_image_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.image.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.alt) { props.alt = this.alt; }
    if (this.icon) { props.icon = this.icon; }
    if (this.align) { props.align = this.align; }
    if (this.border != null) { props.border = this.border; }
    if (this.height) { props.height = this.height; }
    if (this.hspace) { props.hspace = this.hspace; }
    if (this.longDesc) { props.longDesc = this.longDesc; }
    if (this.src) { props.src = this.src; }
    if (this.vspace) { props.vspace = this.vspace; }
    if (this.width) { props.width = this.width; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} border
 * @config {String} className CSS selector.
 * @config {String} prefix The application context path of image
 * @config {String} dir Specifies the directionality of text.
 * @config {String} height 
 * @config {String} hspace 
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} longDesc 
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} src 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @config {String} vspace
 * @config {String} width
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.image.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.image.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Style must be set first (e.g., for absolute positioning) -- some style
    // properties may be overridden later by the combined image.
    this.setCommonProps(this.domNode, props);

    // Clear style properties if icon was previously set.
    if (props.src && this.icon) {
        this.domNode.style.border = "";
        this.domNode.style.backgroundImage = "";
        this.domNode.style.backgroundPosition = "";
        this.domNode.style.height = "";
        this.domNode.style.width = "";
    }

    // Set properties.
    if (props.icon) {
        // IE6 has issues with "png" images. IE6 png issue can be fixed but that
        // needs an outermost <span> tag. 
        //
        // <span style="overflow: hidden; width:13px;height:13px; padding: 0px;zoom: 1";>
        // <img src="dot.gif"
        //  style="filter:progid:DXImageTransform.Microsoft.AlphaImageLoader(src='testImage.png',sizingMethod='crop');
        //  margin-left:-0px;
        //  margin-top:-26px; border: none; height:39px;width:13px;"/>
        // </span>
        //
        // For now, skipping the combined image approach for IE6. Also need fix 
        // for Safari.
        var iconProps = webui.suntheme4_2.theme.common.getImage(props.icon);
        var mapKey = iconProps['map_key'];
        if (mapKey != null && !webui.suntheme4_2.browser.isIe6()
                && !webui.suntheme4_2.widget.common.isHighContrastMode()) {
            // Note: Comparing height/width against "actual" properties is not a
            // valid test -- DOT images don't have a default size, for example.
            if (iconProps['top'] != null && iconProps['actual_height'] != null 
                    && iconProps['actual_width'] != null) {               
                var transImage = webui.suntheme4_2.theme.common.getImage("DOT");
                var combinedImage = webui.suntheme4_2.theme.common.getImage(mapKey);

                // Set style properties.
                this.domNode.style.border = "0";
                this.domNode.style.backgroundImage = "url(" + combinedImage['src'] + ")";
                this.domNode.style.backgroundPosition = "0px " + iconProps['top'] + "px";
                this.domNode.style.height = iconProps['actual_height'] + "px";
                this.domNode.style.width = iconProps['actual_width'] + "px";

                // Set transparent image.
                iconProps.src = transImage['src'];
            }
        }
        // Assign icon properties, even if combined image is not used.
        if (iconProps.alt) { this.domNode.alt = iconProps.alt; }
        if (iconProps.height) { this.domNode.height = iconProps.height; }
        if (iconProps.src) { this.domNode.src = iconProps.src; }
        if (iconProps.width) { this.domNode.width = iconProps.width; }
    } else {
        // Icon properties take precedence.
        if (props.alt) { this.domNode.alt = props.alt; }
        if (props.height) { this.domNode.height = props.height; }
        if (props.src) {
                
            // If context path is provided, then check whether the image has
            // context path already appended and if not, append it.
            if (this.prefix) {
                props.src = 
                    webui.suntheme4_2.widget.common.appendPrefix(this.prefix, props.src);                
            }
            this.domNode.src = props.src;  
        } 
        if (props.width) { this.domNode.width = props.width; }
    }
    if (props.align) { this.domNode.align = props.align; }
    if (props.border != null) { this.domNode.border = props.border; }
    if (props.hspace) { this.domNode.hspace = props.hspace; }
    if (props.longDesc) { this.domNode.longDesc = props.longDesc; }    
    if (props.vspace) { this.domNode.vspace = props.vspace; }

    // Set more properties.
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.imageButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.button");

/**
 * @name webui.suntheme4_2.widget.imageButton
 * @extends webui.suntheme4_2.widget.button
 * @class This class contains functions for the imageButton widget.
 * @constructor This function is used to construct a imageButton widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.imageButton", webui.suntheme4_2.widget.button, {
    // Set defaults.
    widgetName: "imageButton"  // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.imageButton.event =
        webui.suntheme4_2.widget.imageButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageButton_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.imageButton.prototype.getClassName = function() {
    // If it is an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy, vs by theme.
    var key = (this.disabled == true)
	? "BUTTON3_DISABLED"
	: "BUTTON3";

    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to obtain the outermost HTML element class name during
 * an onFocus or onMouseOver event.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.imageButton.prototype.getHoverClassName = function() {
    // If it is an image button, only the BUTTON3 selectors are used.
    // Note that the "mini" and "primary" values can still be set but
    // have no effect on image buttons by policy, vs by theme.
    var key = "BUTTON3_HOVER";
    var className = this.widget.getClassName(key, "");
    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.imageButton.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.src) { props.src = this.src; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {String} className CSS selector.
 * @config {String} prefix The application context path of image 
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} escape HTML escape button text (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} src Source for image.
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageButton.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.imageButton.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.src) {
        // If context path is provided, then check whether the image has
        // context path already appended and if not, append it.
        if (this.prefix) {
            props.src = 
                webui.suntheme4_2.widget.common.appendPrefix(this.prefix, props.src);                
        }
        this.domNode.src = props.src; 
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.imageHyperlink");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.hyperlink");

/**
 * @name webui.suntheme4_2.widget.imageHyperlink
 * @extends webui.suntheme4_2.widget.hyperlink
 * @class This class contains functions for the imageHyperlink widget.
 * @constructor This function is used to construct a imageHyperlink widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.imageHyperlink", webui.suntheme4_2.widget.hyperlink, {
    // Set defaults.
    widgetName: "imageHyperlink" // Required for theme properties.
});

/**
 * Helper function to add children.
 *
 * @param props Key-Value pairs of properties.
 * @config {Array} contents The contents of the anchor body.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.addContents = function(props) {
    if (props.contents == null) {
        return false;
    }

    // Remove child nodes.
    this.widget.removeChildNodes(this.leftContentsContainer);
    this.widget.removeChildNodes(this.rightContentsContainer);

    // Add contents.
    for (i = 0; i <props.contents.length; i++) {
        this.widget.addFragment(this.leftContentsContainer, props.contents[i], "last");
        this.widget.addFragment(this.rightContentsContainer, props.contents[i], "last");
    }
    return true;
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.imageHyperlink.event =
        webui.suntheme4_2.widget.imageHyperlink.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageHyperlink_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageHyperlink_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_imageHyperlink_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_imageHyperlink_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.enabledImage) { props.enabledImage = this.enabledImage; }
    if (this.disabledImage) { props.disabledImage = this.disabledImage; }
    if (this.imagePosition) { props.imagePosition = this.imagePosition; }

    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.enabledImageContainer.id = this.id + "_enabled";
        this.disabledImageContainer.id = this.id + "_disabled";
        this.leftContentsContainer.id = this.id + "_leftContents";
        this.rightContentsContainer.id = this.id + "_rightContents";
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accessKey
 * @config {String} charset
 * @config {String} className CSS selector.
 * @config {Array} contents
 * @config {String} coords
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {Object} disabledImage
 * @config {Object} enabledImage
 * @config {String} href
 * @config {String} hrefLang
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} imagePosition
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} rel
 * @config {String} rev
 * @config {String} shape
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.imageHyperlink.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.imageHyperlink.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Show en/disabled images.
    if (props.disabled != null) {
        var disabled = new Boolean(props.disabled).valueOf();

        // We need to hide/show images only when the disabed image is specified.
        if (this.disabledImage) { 
            this.common.setVisibleElement(this.enabledImageContainer, !disabled);
            this.common.setVisibleElement(this.disabledImageContainer, disabled);
        }
    }

    // Add enabled image.
    if (props.enabledImage) {
        this.widget.addFragment(this.enabledImageContainer, props.enabledImage);
    }

    // Add disabled image.
    if (props.disabledImage) {
        this.widget.addFragment(this.disabledImageContainer, props.disabledImage);
    }

    // Set image position.
    if (props.imagePosition) {
        var left = (props.imagePosition == "left");
        this.common.setVisibleElement(this.leftContentsContainer, !left);
        this.common.setVisibleElement(this.rightContentsContainer, left);    
    }

    // Add contents.
    this.addContents(props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.anchor");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.common");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.json");

//
// Copyright (c) 2005 JSON.org
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The Software shall be used for Good, not Evil.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//

/**
 * @class This class contains functions for parsing JSON.
 * @static
 */
webui.suntheme4_2.json = {
    /**
     * JSON escape chars.
     * @private
     */
    _m: {
        '\b': '\\b',
        '\t': '\\t',
        '\n': '\\n',
        '\f': '\\f',
        '\r': '\\r',
        '"' : '\\"',
        '\\': '\\\\'
    },

    /**
     * JSON parsor.
     * @private
     */
    _s: {
        'boolean': function (x) {
            return String(x);
        },
        number: function (x) {
            return isFinite(x) ? String(x) : null;
        },
        string: function (x) {
            if (/["\\\x00-\x1f]/.test(x)) {
                x = x.replace(/([\x00-\x1f\\"])/g, function(a, b) {
                    var c = webui.suntheme4_2.json._m[b];
                    if (c) {
                        return c;
                    }
                    c = b.charCodeAt();
                    return '\\u00' +
                        Math.floor(c / 16).toString(16) +
                        (c % 16).toString(16);
                });
            }
            return '"' + x + '"';
        },
        object: function (x) {
            if (x) {
                var a = [], b, f, i, l, v;
                if (x instanceof Array) {
                    a[0] = '[';
                    l = x.length;
                    for (i = 0; i < l; i += 1) {
                        v = x[i];
                        f = webui.suntheme4_2.json._s[typeof v];
                        if (f) {
                            v = f(v);
                            if (typeof v == 'string') {
                                if (b) {
                                    a[a.length] = ',';
                                }
                                a[a.length] = v;
                                b = true;
                            }
                        }
                    }
                    a[a.length] = ']';
                } else if (typeof x.hasOwnProperty === 'function') {
                    a[0] = '{';
                    for (i in x) {
                        if (x.hasOwnProperty(i)) {
                            v = x[i];
                            f = webui.suntheme4_2.json._s[typeof v];
                            if (f) {
                                v = f(v);
                                if (typeof v == 'string') {
                                    if (b) {
                                        a[a.length] = ',';
                                    }
                                    a.push(webui.suntheme4_2.json._s.string(i), ':', v);
                                    b = true;
                                }
                            }
                        }
                    }
                    a[a.length] = '}';
                } else {
                    return null;
                }
                return a.join('');
            }
            return null;
        }
    },

    /**
     * Stringify a JavaScript value, producing JSON text. 
     *
     * @param {Object} v A non-cyclical JSON object.
     * @return {boolean} true if successful; otherwise, false.
     */
    stringify: function (v) {
        var f = webui.suntheme4_2.json._s[typeof v];
        if (f) {
            v = f(v);
            if (typeof v == 'string') {
                return v;
            }
        }
        return null;
    },

    /**
     * Parse a JSON text, producing a JavaScript object.
     *
     * @param {String} text The string containing JSON text.
     * @return {boolean} true if successful; otherwise, false.
     */
    parse: function (text) {
        try {
            return !(/[^,:{}\[\]0-9.\-+Eaeflnr-u \n\r\t]/.test(
                text.replace(/"(\\.|[^"\\])*"/g, ''))) && eval('(' + text + ')');
        } catch (e) {
            return false;
        }
    }
};


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.dynaFaces");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.config");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.theme.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.eventBase");

/**
 * @class This class contains functions to obtain JSF Extensions resources.
 * @static
 */
webui.suntheme4_2.widget.jsfx.dynaFaces = {
    /**
     * This function is used to initialize the environment with Object literals.
     *
     * @param {Object} props Key-Value pairs of properties.
     * @config {Object} ajax Key-Value pairs of Ajax properties.     
     * @config {boolean} ajax.isJsfx Flag indicating to include JSF Extensions.
     * @config {boolean} webuiJsfx Flag indicating to include default Ajax functionality.
     * @return {boolean} true if successful; otherwise, false.
     * @private
     */
    _init: function(props) {
        if (props == null) {
            console.debug("Cannot initialize JSF Extensions."); // See Firebug console.
            return false;
        }

        // Don't load JSF Extensions.
        if (props.ajax && props.ajax.isJsfx != null) {
            if (new Boolean(props.ajax.isJsfx).valueOf() == false) {
                return false;
            }
        }
        
        // Load JSF Extensions immediately.
        if (new Boolean(props.webuiJsfx).valueOf() == true) {
            webui.suntheme4_2.widget.jsfx.dynaFaces.loadJsfx();
        } else {
            // Override default publish functionality to lazily load JSFX.
            webui.suntheme4_2.widget.eventBase.prototype._publish = 
                function(topic, props) {
                    // Load JSF Extensions and publish event via a callback, 
                    // ensuring all resources have been loaded.
                    webui.suntheme4_2.widget.jsfx.dynaFaces.loadJsfx(function() {

                    // Publish an event for custom AJAX implementations to listen for.
                    webui.suntheme4_2.dojo.publish(topic, props);
                    return true;
                });
            };
        }
        return true;
    },

    /**
     * Load JSF Extensions.
     *
     * @param {Function} callback The JavaScript function to invoke on load.
     * @return {boolean} true if successful; otherwise, false.
     */
    loadJsfx: function(callback) {
        if (typeof DynaFaces != "undefined") {
            return callback();
        }
        var bootstrap = webui.suntheme4_2.bootstrap;
        var theme = webui.suntheme4_2.theme.common;
        var isDebug = new Boolean(webui.suntheme4_2.config.isDebug).valueOf();

        // Get script URLs.
        var pUrl = theme.getJavaScript((isDebug == true)
            ? "prototypeUncompressed" : "prototype");
        var jUrl = theme.getJavaScript((isDebug == true)
            ? "jsfxUncompressed" : "jsfx");

        // Ensure Prototype is loaded first using a callback.
        var jsfxCallback = function() {
            bootstrap.loadScript(jUrl, callback);
        };

        // Load Prototype and DynaFaces.
        //
        // Note: Prototype must be added to the global scope. Therefore,
        // dojo._loadPath() and/or dojo._loadUri() will not work here.
        if (typeof Prototype == "undefined") {
            bootstrap.loadScript(pUrl, jsfxCallback);
        } else {
            jsfxCallback();
        }
        return true;
    }
};

// Initialize the environment.
webui.suntheme4_2.widget.jsfx.dynaFaces._init(webui.suntheme4_2.config);

/**
 * @class This class contains functions to obtain data asynchronously using JSF
 * Extensions as the underlying transfer protocol.
 * @static
 */
webui.suntheme4_2.widget.jsfx.common = {
    /**
     * This function is used to process refresh events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @return {boolean} true if successful; otherwise, false.
     */
    processRefreshEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);

        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: (props.execute) ? props.execute : "none",
            render: props.id,
            replaceElement: webui.suntheme4_2.widget.jsfx.common.refreshCallback,
            xjson: {
                id: props.id,
                endTopic: props.endTopic,
                event: "refresh"
            }
        });
        return true;
    },

    /**
     * This function is used to process state change events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {Object} props Key-Value pairs of widget properties to update.
     * @return {boolean} true if successful; otherwise, false.
     */
    processStateEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);

        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            render: props.id,
            replaceElement: webui.suntheme4_2.widget.jsfx.common.stateCallback,
            xjson: {
                id: props.id,
                endTopic: props.endTopic,
                event: "state",
                props: props.props // Widget properties to update.
            }
        });
        return true;
    },

    /**
     * This function is used to process submit events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @config {String} endTopic The event topic to publish.
     * @config {String} execute The string containing a comma separated list 
     * of client ids against which the execute portion of the request 
     * processing lifecycle must be run.
     * @return {boolean} true if successful; otherwise, false.
     */
    processSubmitEvent: function(props) {
        if (props == null) {
            return false;
        }

        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);

        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: (props.execute) ? props.execute : props.id,
            render: props.id,
            replaceElement: webui.suntheme4_2.widget.jsfx.common.submitCallback,
            xjson: {
                id: props.id,
                endTopic: props.endTopic,
                event: "submit"
            }
        });
        return true;
    }, 
   
    /**
     * This function is used to refresh widgets.
     *
     * @param {String} elementId The HTML element Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to DynaFaces.fireAjaxTransaction.
     * @param {Object} xjson The xjson argument provided to DynaFaces.fireAjaxTransaction.
     * @return {boolean} true if successful; otherwise, false.
     */
    refreshCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = webui.suntheme4_2.json.parse(content);

        // Add rows.
        var widget = webui.suntheme4_2.dijit.byId(id);
        widget.setProps(props);

        // Publish an event for custom AJAX implementations to listen for.
        if (xjson.endTopic) {
            webui.suntheme4_2.dojo.publish(xjson.endTopic, [props]);
        }
        return true;
    },

    /**
     * This function is a callback to respond to the end of state request.
     * It will only publish submit end event without updating the widget itself.
     *
     * @param {String} elementId The HTML element Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to DynaFaces.fireAjaxTransaction.
     * @param {Object} xjson The xjson argument provided to DynaFaces.fireAjaxTransaction.
     * @return {boolean} true if successful; otherwise, false.
     */
    stateCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = webui.suntheme4_2.json.parse(content);
            
        // Publish an event for custom AJAX implementations to listen for.
        if (xjson.endTopic) {
            webui.suntheme4_2.dojo.publish(xjson.endTopic, [props]);
        }
        return true;
    },

    /**
     * This function is a callback to respond to the end of submit request.
     * It will only publish submit end event without updating the widget itself.
     *
     * @param {String} elementId The HTML element Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to DynaFaces.fireAjaxTransaction.
     * @param {Object} xjson The xjson argument provided to DynaFaces.fireAjaxTransaction.
     * @return {boolean} true if successful; otherwise, false.
     */
    submitCallback: function(id, content, closure, xjson) {
        if (id == null || content == null) {
            return false;
        }

        // Parse JSON text.
        var props = webui.suntheme4_2.json.parse(content);
            
        // Publish an event for custom AJAX implementations to listen for.
        if (xjson.endTopic) {
            webui.suntheme4_2.dojo.publish(xjson.endTopic, [props]);
        }
        return true;
    }
};
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.anchor");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.anchor.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.button");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.button");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.button.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.checkbox");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.checkbox");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.checkbox.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.checkbox.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.checkboxGroup");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.checkboxGroup");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.checkboxGroup.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.dropDown");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.dropDown");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.dropDown.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.dropDown.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.hiddenField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.hiddenField");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.hiddenField.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.hiddenField.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.hyperlink");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.hyperlink");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.hyperlink.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.image");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.image");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.image.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.imageButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.imageButton");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.imageButton.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.imageHyperlink");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.imageHyperlink");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.imageHyperlink.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.label");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.label");

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.label.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.listbox");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.listbox");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.selectBase");

/**
 * @name webui.suntheme4_2.widget.listbox
 * @extends webui.suntheme4_2.widget.selectBase
 * @class This class contains functions for the listbox widget.
 * @constructor This function is used to construct a listbox widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.listbox", 
	webui.suntheme4_2.widget.selectBase, {
    // Set defaults.
    constructor: function() {
        // Or defer to setProps 
        this.labelOnTop = webui.suntheme4_2.widget.common.getMessageBoolean(
            "listbox.labelOnTop", false);
        this.monospace = webui.suntheme4_2.widget.common.getMessageBoolean(
            "listbox.monospace", false);
        this.multiple = webui.suntheme4_2.widget.common.getMessageBoolean(
            "listbox.multiple", false);
        this.size = webui.suntheme4_2.widget.common.getMessage("listbox.size", null, 12);
        this.width = webui.suntheme4_2.theme.common.getMessage("listbox.width", null);
    },
    widgetName: "listbox" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.listbox.event =
        webui.suntheme4_2.widget.listbox.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for.*/
        beginTopic: "webui_suntheme4_2_widget_listbox_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for.*/
        endTopic: "webui_suntheme4_2_widget_listbox_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_listbox_event_state_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_listbox_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_listbox_event_submit_begin",

        /** event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_listbox_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element CSS class name.
 * <p>
 * Note: Selectors should be concatenated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.listbox.prototype.getClassName = function() {
    var cn = this.widget.getClassName("LISTBOX", "");
    return (this.className) ? cn + " " + this.className : cn;
};

/**
 * Return an Object Literal of label properties desired
 * by the listbox widget.
 * <p>
 * This implementation sets
 * the <code>listbox.labelLevel</code> and <code>listbox.labelAlign</code>
 * theme values from the <code>messages</code> and <code>styles</code>
 * theme categories to the
 * label's <code>level</code> and <code>className</code> properties 
 * respectively. It sets the <code>htmlFor</code> attribute to 
 * <code>listContainer.id</code>
 * </p>
 * <p>
 * These properties are extended with <code>this.label</code> and the
 * resulting properties are returned.
 * </p>
 * @return {Object} label properties.
 */
webui.suntheme4_2.widget.listbox.prototype.getLabelProps = function() {
    var cn = this.getLabelClassName(null);
    var lvl = this.widget.getMessage("listbox.labelLevel", null, 2);
    var props = {};
    if (cn != null) {
       props.className = cn;
    }
    props.level = lvl;
    props.htmlFor = this.listContainer.id;
    this.prototypejs.extend(props, this.label);
    return props;
};

/**
 * Return a CSS selector for the listbox label.
 * The chosen CSS selector is a function of 
 * the <code>labelOnTop</code> property. If <code>labelOnTop</code>
 * property is <code>false</code> the value of the theme key 
 * <code>LISTBOX_LABEL_ALIGN</code> from the 
 * <code>styles</code> category is returned, else <code>null</code>.
 * @param {boolean} ontop The label on top state. In some contexts this state
 * must be passed to the method, for example when called from the 
 * <code>selectBase.setProps</code> method.
 * @return {String} A CSS selector for the listbox label.
 */
webui.suntheme4_2.widget.listbox.prototype.getLabelClassName = function(ontop) {
    var labelontop = ontop != null ? ontop : this.labelOnTop;
    return labelontop == true
	? null
	: this.widget.getClassName("LISTBOX_LABEL_ALIGN", null);
};

/**
 * Return a CSS selector for listContainer HTML element.
 * The chosen CSS selector is a function of 
 * the <code>disabled</code> and <code>monospace</code>properties.
 * The returned selector is the value of a property defined in the
 * <code>styles</code> category, as shown in the following table.
 * <table border="1px">
 * <tr><th>key</th><th>disabled</th><th>monospace</th></tr>
 * <tr>
 * <td>LIST</td>
 * <td>false</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_DISABLED</td>
 * <td>true</td>
 * <td>false</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE</td>
 * <td>false</td>
 * <td>true</td>
 * </tr>
 * <tr>
 * <td>LIST_MONOSPACE_DISABLED</td>
 * <td>true</td>
 * <td>true</td>
 * </tr>
 * </table>
 * @return {String} A CSS selector for the listContainer HTML element.
 * @private
 */
webui.suntheme4_2.widget.listbox.prototype._getListContainerClassName =
	function(disabled, monospace) {
    var key = "LIST";
    if (disabled == true) {
	if (monospace == true) {
	    key = "LIST_MONOSPACE_DISABLED";
	} else {
	    key = "LIST_DISABLED";
	}
    } else if (monospace == true) {
	key = "LIST_MONOSPACE";
    }
    return this.widget.getClassName(key, null);
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.listbox.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Get properties.
    //
    if (this.size != null) {
	props.size = this.size;
    }
    if (this.multiple != null) {
	props.multiple = this.multiple;
    }
    if (this.monospace != null) {
	props.monospace = this.monospace;
    }
    return props;
};

/**
 * This function returns the CSS class name for an HTML option element,
 * based on the state of <code>element</code>. The CSS class name for 
 * the following theme keys is returned for the given states, based on 
 * the properties
 * in "option".
 * <p>
 * <table border="1px">
 * <tr><th>state</th><th>key</th></tr>
 * <tr><td>
 * separator == true</td><td>LIST_OPTION_SEPARATOR</td>
 * </tr><tr><td>
 * group == true</td><td>LIST_OPTION_GROUP</td>
 * </tr><tr><td>
 * disabled == true</td><td>LIST_OPTION_DISABLED</td>
 * </tr><tr><td>
 * selected == true and disabled == false</td><td>LIST_OPTION_SELECTED</td>
 * </tr><tr><td>
 * separator == false and group false<br/> 
 * 	and disabled == false<br/>
 *	and selected == false</td><td>LIST_OPTION</td>
 * </tr>
 * </table>
 * </p>
 * <p>
 * If <code>element</code> is null, the CSS class name for the theme
 * key "LIST_OPTION" is returned.
 * </p>
 * @param {Object} element An HTML option DOM node instance.
 * @config {boolean} separator If true, element is a option separator.
 * @config {boolean} group If true the element is a optGroup instance.
 * @config {boolean} disabled If true the element is disabled.
 * @config {boolean} selected If true the element is selected.
 * @return {String} The HTML option element CSS class name.
 */
webui.suntheme4_2.widget.listbox.prototype.getOptionClassName = function(element) {
    var key = "LIST_OPTION";
    if (element == null) {
	return this.widget.getClassName(key, null);
    }

    if (element.separator == true) {
        key = "LIST_OPTION_SEPARATOR";
    } else
    if (element.group == true) {
        key = "LIST_OPTION_GROUP";
    } else
    if (element.selected == true) {
        key = "LIST_OPTION_SELECTED";
    }

    // disabled wins
    //
    if (element.disabled == true) {
	key = "LIST_OPTION_DISABLED";
    }
    return this.widget.getClassName(key, null);
};

/*
 * This function is used to fill in remaining template properties, after the
 * <code>buildRendering</code> function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.listbox.prototype.postCreate = function () {
    // Don't trash the template.
    //
    webui.suntheme4_2.common.addStyleClass(this.listContainer,
	this._getListContainerClassName(
		this.disabled == true, this.monospace == true));
    return this.inherited("postCreate", arguments);
};


/**
 * This function is used to set widget properties defined in the
 * <code>props</code> Object literal.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 * <p>
 * See <code>selectBase.setOptions</code> or overview for the properties
 * of option objects in the <code>options</code> array property.
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} label Defines the widget and its properties to use for a
 * label.
 * @config {String} labelOnTop If true the label appears above the listbox.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace If true use monospace font styling.
 * @config {String} onBlur Element lost focus.
 * @config {String} onChange Option selection is changed.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect Option text is highlighted.
 * @config {Array} options See selectBase.setOptions or selectBase overview
 * for details on the elements of this array.
 * @config {int} size The number of option rows to display.
 * @config {String} style Specify style rules inline. 
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.listbox.prototype.setProps = function(props, notify) {
    if (props == null) {
	return this.inherited("setProps", arguments);
    }

    // We are trying to deal with a state change which requires
    // knowing what the current state of the widget is and the
    // state as defined in props. In order to compare the states
    // it must be done in setProps before the "props" are extended
    // onto the widget. At that point there is no difference between
    // "props" and "this".
    //
    var ismonospace = this.monospace == true;
    var toggleMonospace = props.monospace != null && 
	props.monospace != ismonospace;

    var isdisabled = this.disabled == true;
    var toggleDisabled = props.disabled != null && 
	props.disabled != isdisabled;

    // Get current state of the classsname, and strip it 
    // and then add the classname for the new state.
    //
    if (toggleMonospace || toggleDisabled) {
	var cn = this._getListContainerClassName(isdisabled, ismonospace);
	webui.suntheme4_2.common.stripStyleClass(this.listContainer, cn);

	cn = this._getListContainerClassName(
	    toggleDisabled ? props.disabled == true : isdisabled,
	    toggleMonospace ? props.monospace == true : ismonospace);
	webui.suntheme4_2.common.addStyleClass(this.listContainer, cn);
    }
    return this.inherited("setProps", arguments);
};

/**
 * This function sets properties specific to the listbox, the superclass
 * will set properties specific to the <code>select</code> HTML element
 * and handle <code>labelOnTop</code> and the label subcomponent.
 * <p>
 * If the <code>size</code>, <code>multiple</code>, or <code>monospace</code>
 * properties are not set in <code>props</code> or <code>this</code>,
 * the values are obtained from the theme using the keys 
 * <code>listbox.size</code>, <code>listbox.multiple</code>,
 * <code>listbox.monospace</code>, and <code>listbox.labelOnTop</code>
 * respectively, from the theme "messages" category. If the theme does 
 * not define these values then the values <code>12</code>,
 * <code>false</code>, <code>false</code>, and <code>false</code>
 * are used, respectively.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {boolean} multiple If true allow multiple selections
 * @config {boolean} monospace  If true the <code>LIST_MONOSPACE</code> and
 * <code>LIST_MONOSPACE_DISABLED</code>CSS class names are used, else
 * <code>LIST</code> and <code>LIST_DISABLED</code>.
 * @config {int} size The number of rows to display.
 * @config {boolean} labelOnTop If true render the label on top of the
 * list box, else to the upper left.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.listbox.prototype._setProps = function(props) {
    // If props == null create one so that the superclass can 
    // handle "labelOnTop" or anything else that needs to be
    // set in props and handled in the superclass
    //
    if (props == null) {
	return this.inherited("_setProps", arguments);
	//props = {};
    }

    if (props.size != null) {
	var size = props.size;
	if (size < 1) {
	    size = this.theme.getMessage("listbox.size", null, 12);
	}
	if (this.listContainer.size != size) {
	    this.listContainer.size = size;
	}
    }

    if (props.multiple != null && 
	    this.listContainer.multiple != props.multiple) {
	this.listContainer.multiple = props.multiple;
    }
    return this.inherited("_setProps", arguments);
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.listbox.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.listbox.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.radioButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.radioButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.checkedBase");

/**
 * @name webui.suntheme4_2.widget.radioButton
 * @extends webui.suntheme4_2.widget.checkedBase
 * @class This class contains functions for the radioButton widget.
 * @constructor This function is used to construct a radioButton widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.radioButton", webui.suntheme4_2.widget.checkedBase, {
    // Set defaults.
    idSuffix: "_rb",
    widgetName: "radioButton" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.radioButton.event =
        webui.suntheme4_2.widget.radioButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButton_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButton_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButton_event_submit_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getClassName = function() {
    // Set default style.
    var className = (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_SPAN_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON_SPAN", "");

    return (this.className)
        ? className + " " + this.className
        : className;
};

/**
 * Helper function to obtain image class names.
 *
 * @return {String} The HTML image element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getImageClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_IMAGE_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON_IMAGE", "");  
};

/**
 * Helper function to obtain input class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getInputClassName = function() {
    // Set readOnly style.
    if (this.readOnly == true) {
        return this.widget.getClassName("RADIOBUTTON_READONLY", "");
    }

    // Disabled style.
    return (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON", "");  
};

/**
 * Helper function to obtain label class names.
 *
 * @return {String} The HTML label element class name.
 */
webui.suntheme4_2.widget.radioButton.prototype.getLabelClassName = function() {
    return (this.disabled == true)
        ? this.widget.getClassName("RADIOBUTTON_LABEL_DISABLED", "")
        : this.widget.getClassName("RADIOBUTTON_LABEL", "");  
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} alt Alternate text for image input.
 * @config {String} align Alignment of image input.
 * @config {boolean} checked 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {Object} image 
 * @config {String} label 
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} name 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} onSelect 
 * @config {boolean} readOnly 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.radioButton.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.radioButton.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    if (props.name) {
        // IE does not support the name attribute being set dynamically as 
        // documented at:
        //
        // http://msdn.microsoft.com/workshop/author/dhtml/reference/properties/name_2.asp
        //
        // In order to create an HTML element with a name attribute, the name
        // and value must be provided when using the inner HTML property or the
        // document.createElement() function. As a work around, we shall set the
        // attribute via the HTML template using name="${this.name}". In order
        // to obtain the correct value, the name property must be provided to 
        // the widget. Although we're resetting the name below, as the default,
        // this has no affect on IE. 
        this.inputNode.name = props.name;
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.radioButton.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.radioButton.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.radioButtonGroup");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.radioButtonGroup");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.checkedGroupBase");

/**
 * @name webui.suntheme4_2.widget.radioButtonGroup
 * @extends webui.suntheme4_2.widget.checkedGroupBase
 * @class This class contains functions for the radioButtonGroup widget.
 * @constructor This function is used to construct a radioButtonGroup widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.radioButtonGroup", webui.suntheme4_2.widget.checkedGroupBase, {
    // Set defaults.
    widgetName: "radioButtonGroup" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.radioButtonGroup.event =
        webui.suntheme4_2.widget.radioButtonGroup.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */        
        endTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_radioButtonGroup_event_state_end"
    }
};

/**
 * This function is used to obtain the outermost HTML element class name.
 * <p>
 * Note: Selectors should be concatinated in order of precedence (e.g., the 
 * user's className property is always appended last).
 * </p>
 * @return {String} The outermost HTML element class name.
 */
webui.suntheme4_2.widget.radioButtonGroup.prototype.getClassName = function() {
    // Set default style.
    var className = (this.columns > 1)
        ? this.widget.getClassName("RBGRP_HORIZ", "")
        : this.widget.getClassName("RBGRP_VERT", "");

    return (this.className)
        ? className + " " + this.className
        : className;
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.radioButtonGroup.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.resetButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.resetButton");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.button");

/**
 * @name webui.suntheme4_2.widget.resetButton
 * @extends webui.suntheme4_2.widget.button
 * @class This class contains functions for the resetButton widget.
 * @constructor This function is used to construct a resetButton widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.resetButton", webui.suntheme4_2.widget.button, {
    // Set defaults.
    widgetName: "resetButton"  // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.resetButton.event =
        webui.suntheme4_2.widget.resetButton.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_resetButton_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_resetButton_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_resetButton_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_resetButton_event_state_end"
    }
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.resetButton.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.staticText");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.staticText");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.staticText
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for the staticText widget.
 * @constructor This function is used to construct a staticText widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.staticText", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.escape = true;
    },
    widgetName: "staticText" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.staticText.event =
        webui.suntheme4_2.widget.staticText.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_staticText_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_staticText_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_staticText_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_staticText_event_state_end"
    }
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.staticText.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.escape) { props.escape = this.escape; }
    if (this.value) { props.value = this.value; }

    return props;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {String} escape HTML escape value (default).
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {String} style Specify style rules inline.
 * @config {String} title Provides a title for element.
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.staticText.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.staticText.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
      
    // Set text value.
    if (props.value) {
        // NOTE: If you set this value manually, text must be HTML escaped.
        this.widget.addFragment(this.domNode, props.value, null, this.escape);
    }

    // Set more properties.
    this.setCommonProps(this.domNode, props);
    this.setEventProps(this.domNode, props);

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.staticText.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.textArea");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.textArea");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.textField");



webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.fieldBase");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.widgetBase");

/**
 * @name webui.suntheme4_2.widget.fieldBase
 * @extends webui.suntheme4_2.widget.widgetBase
 * @class This class contains functions for widgets that extend fieldBase.
 * @static
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.fieldBase", webui.suntheme4_2.widget.widgetBase, {
    // Set defaults.
    constructor: function() {
        this.disabled = false;
        this.required = false;
        this.size = 20;
    },
    valid: true
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.fieldBase.prototype.getInputClassName = function() {   
    return null; // Overridden by subclass.
};

/**
 * Returns the HTML input element that makes up the text field.
 *
 * @return {Node} The HTML input element.
 */
webui.suntheme4_2.widget.fieldBase.prototype.getInputElement = function() {
    return this.fieldNode;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.fieldBase.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.alt) { props.alt = this.alt; }
    if (this.disabled != null) { props.disabled = this.disabled; }
    if (this.label) { props.label= this.label; }
    if (this.maxLength > 0) { props.maxLength = this.maxLength; }    
    if (this.notify) { props.notify = this.notify; }
    if (this.submitForm != null) { props.submitForm = this.submitForm; }
    if (this.text != null) { props.text = this.text; }
    if (this.title != null) { props.title = this.title; }
    if (this.type) { props.type= this.type; }
    if (this.readOnly != null) { props.readOnly = this.readOnly; }
    if (this.required != null) { props.required = this.required; }
    if (this.size > 0) { props.size = this.size; }
    if (this.style != null) { props.style = this.style; }
    if (this.valid != null) { props.valid = this.valid; }
    
    // After widget has been initialized, get user's input.
    if (this.isInitialized() == true && this.fieldNode.value != null) {
        props.value = this.fieldNode.value;
    } else if (this.value != null) {
        props.value = this.value;
    }
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.fieldBase.prototype.postCreate = function () {
    // Set ids.
    if (this.id) {
        this.fieldNode.id = this.id + "_field";
        this.fieldNode.name = this.id + "_field";
        this.labelContainer.id = this.id + "_label";
    }
    
    //initialize label
    if (this.label && this.label.value != null &&
	    !this.widget.isFragment(this.label)) {

	this.label = this.widget.getWidgetProps("label", this.label);
	this.label.id = this.labelContainer.id;
    }

    
    // Set public functions.
    this.domNode.getInputElement = function() { return webui.suntheme4_2.dijit.byId(this.id).getInputElement(); };
    
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.fieldBase.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.
    if (props.submitForm == false || props.submitForm == true) { 
        // connect the keyPress event
        this.dojo.connect(this.fieldNode, "onkeypress", this, "submitFormData");
    }
    if (props.maxLength > 0) { this.fieldNode.maxLength = props.maxLength; }
    if (props.size > 0) { this.fieldNode.size = props.size;  }
    if (props.value != null) { this.fieldNode.value = props.value; }
    if (props.title != null) { this.fieldNode.title = props.title; }   
    if (props.disabled != null) { 
        this.fieldNode.disabled = new Boolean(props.disabled).valueOf();
    }
    if (props.valid != null) { 
        this.valid = new Boolean(props.valid).valueOf();
        if (props.label == null) props.label = {};
        props.label.valid = this.valid;
    }
    if (props.required != null) { 
        this.required = new Boolean(props.required).valueOf();
        if (props.label == null) props.label = {};
        props.label.required = this.required;
    }
    if (props.readOnly != null) { 
        this.fieldNode.readOnly = new Boolean(props.readOnly).valueOf();
    }
    
    // Set label properties.  
    // If _setProps is called during initializat then we will be
    // creating the label and props.label == this.label.
    // If _setProps is called from setProps, then we are updating the
    // label. The label needs to be updated if 
    // required or valid properties are changed.
    // The application may also be creating a label after the
    // widget was created.
    //
    if (props.label) {
	// Now update or create the label.
	// If we don't have an existing label, this.label.id == null
	// then call addFragment in case the application is
	// creating the label after the selectBase widget was created.
	//
	if (this.label != null && this.label.id != null) {
	    this.widget.updateFragment(this.labelContainer, this.label.id,
		props.label);
	} else {
	    this.widget.addFragment(this.labelContainer, props.label);
	}
    }
      
    
    // Set HTML input element class name.
    this.fieldNode.className = this.getInputClassName();
    
    // Set more properties.
    this.setCommonProps(this.fieldNode, props);
    this.setEventProps(this.fieldNode, props);
    
    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.fieldBase.prototype.submitFormData = function(event) {
    if (event == null) {
        return false;
    }
    if (event.keyCode == event.KEY_ENTER) {               
        if (this.submitForm == false) {
            // Disable form submission.
            if (window.event) {
                event.cancelBubble = true;
                event.returnValue = false;
            } else{
                event.preventDefault();
                event.stopPropagation();
            }
            return false;
        } else {
            // Submit the form                    
            if (event.currentTarget.form) {
                event.currentTarget.form.submit();
            }
        }
    }
    return true;    
};

/**
 * @name webui.suntheme4_2.widget.textField
 * @extends webui.suntheme4_2.widget.fieldBase
 * @class This class contains functions for the textField widget.
 * @constructor This function is used to construct a textField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.textField", webui.suntheme4_2.widget.fieldBase, {
    // Set defaults.
    constructor: function() {
        // Array of list values; may be empty; if null - then no autocomplete 
        // functionality is provided
        this.autoCompleteOptions = null; 
        this.autoCompleteSize = 15;
        this.autoCompleteCloseDelayTime = 100;
    },                  
    widgetName: "textField" // Required for theme properties.
});

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.textField.event =
        webui.suntheme4_2.widget.textField.prototype.event = {
  /**
   * This object contains filter event topics.
   * @ignore
   */
    autoComplete: {
        /** Filter event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_autoComplete_begin",

        /** Filter event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_autoComplete_end"
    },
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_submit_end"
    },

    /**
     * This object contains validation event topics.
     * @ignore
     */
    validation: {
        /** Validation event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textField_event_validation_begin",

        /** Validation event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textField_event_validation_end"
    }
};

/**
 * Utility function to adjust input and list widths to match
 * the one of surrounding domNode node    
 *
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.adjustListGeometry = function () {
    this.listContainer.style.width = this.fieldNode.offsetWidth;
    this.listContainer.style.left = this.fieldNode.offsetLeft;
    this.listContainer.style.top = this.fieldNode.offsetTop + this.fieldNode.offsetHeight;
    this.listNode.style.width = this.fieldNode.offsetWidth;
    this.listContainer.style.zIndex = "999";
    return true;
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.textField.prototype.getInputClassName = function() {          
    // Set readOnly style.
    if (this.fieldNode.readOnly) {
        return this.widget.getClassName("TEXT_FIELD_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("TEXT_FIELD_INVALID", "")
        : " " + this.widget.getClassName("TEXT_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this.widget.getClassName("TEXT_FIELD_DISABLED", "") 
        : this.widget.getClassName("TEXT_FIELD", "") + validStyle;
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.textField.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);

    // Set properties.
    if (this.autoValidate != null) { props.autoValidate = this.autoValidate; }
    if (this.autoComplete != null) { props.autoComplete = this.autoComplete; } 
    if (this.autoCompleteOptions != null) { props.autoCompleteOptions = this.autoCompleteOptions; } //TODO clone array?
        
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.postCreate = function () {
    // Set events.
    if (this.autoValidate == true) {
        // Generate the following event ONLY when 'autoValidate' == true.
        this.dojo.connect(this.fieldNode, "onblur", this, "validate");
    }
    return this.inherited("postCreate", arguments);;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoValidate
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * Helper function to create callback for autoComplete list closing event.
 *
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.textField.prototype.createCloseListCallback = function() {
    var _id = this.id;
    return function(event) { 
        var widget = webui.suntheme4_2.dijit.byId(_id);
        if (widget == null) {
            return false;
        }
        widget.showAutoComplete = false;
        widget.updateListView();
        return true;
    };
};

/**
 * Publishes event to filter the options list according to specified 
 * filter string. Note that in default implementation such options will be 
 * updated remotely by Ajax call and then are automatically refreshed on the page.
 *
 * @param {String} filter
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.filterOptions = function() {    
    // Publish the event for custom AJAX implementations to listen for.
    // The implementation of this Ajax call will retrieve the value of the filter
    // and will obtain an updated lookup list ( either locally, or by submit to the server)
    // Data returned from Ajax call will be pushed into this.listWidget 
    //
    // @see javascript.widget.jsfx.autoComplete for default Ajax implementation
    this.dojo.publish(webui.suntheme4_2.widget.textField.event.autoComplete.beginTopic, [{
        id: this.id
    }]);
    return true;        
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.textField.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }
    
    //we can receive updated autoComplete flag from ajax call back
    if (props.autoComplete != null) { this.autoComplete = new Boolean(props.autoComplete).valueOf();  }
    
    //initialize list, if required    
    //autocomplete list may be initiated only on widget start, and cannot 
    //be introduced as part of dynamic options on setProps
    if (this.autoComplete && this.listWidget == null) {
        //create and populate props for listbox
        this.listWidgetProps = this.widget.getWidgetProps("listbox", {
           id: this.id + "_list",
           onFocus: "webui.suntheme4_2.dijit.byId('" + this.id + "').processFocusEvent(this.event);", 
           onBlur: "webui.suntheme4_2.dijit.byId('" + this.id + "').processBlurEvent(this.event);"
        });         
        //?? use of event registration as in following disables field processing keys 
        //onChange: "webui.suntheme4_2.dijit.byId('" + this.id + "').processListChange(this.event);"

        this.widget.addFragment(this.listContainer, this.listWidgetProps);
        
        //store reference to the list
        this.listWidget = webui.suntheme4_2.dijit.byId(this.listWidgetProps.id);
        this.listNode = this.listWidget.getSelectElement();
        
        //since original list box is created empty, make sure it is not shown
        this.updateListView();
  
         //disable browser autocomplete
         
         //we assume here that once the field is armed for autocomplete,
         //this mode will not get disabled. Thus there is no logic provided
         //for restoration of "autocomplete" attribute
         //
        this.fieldNode.setAttribute("autocomplete", "off");
        
        //use focus event to open the list
        this.dojo.connect(this.fieldNode, "onfocus", this, "processFocusEvent");
 
        //use blur events to close the list
        this.dojo.connect(this.fieldNode, "onblur", this, "processBlurEvent");        
         
        // onChange event of the list will change the content of the input field
        this.dojo.connect(this.listNode, "onchange", this, "processListChange");
 
        //onclick will allow to reopen options list after it has been closed with ESC
        this.dojo.connect(this.fieldNode, "onclick", this, "processFocusEvent");

        // input field changes will trigger updates to the autoComplete list options
        this.dojo.connect(this.fieldNode, "onkeyup", this, "processFieldKeyUpEvent");
        
        //additional logic applied to ENTER, ESCAPE, ARROWs on keydown in order to cancel the event bubble
        this.dojo.connect(this.fieldNode, "onkeydown", this, "processFieldKeyDownEvent");
    }        
    
    if (this.autoComplete && props.autoCompleteOptions != null && this.listWidget != null ) {
        //autoComplete param has changed
        
        //we can receive new options from ajax call
        this.autoCompleteOptions = props.autoCompleteOptions;
        
        // Push properties into listbox.  
        var listProps = {};
        listProps.options = props.autoCompleteOptions;
        
        //change list box size up to autoCompleteSize elem
        if (this.autoCompleteOptions.length > this.autoCompleteSize) {
            listProps.size = this.autoCompleteSize;
        } else if (this.autoCompleteOptions.length == 1) {
            //provide extra space for one-line
            listProps.size = 2;
        } else {
            listProps.size = this.autoCompleteOptions.length;
        }

        this.listWidget.setProps(listProps);
        
        /* // display list on initiation
        this.showAutoComplete = true;        
        */
        this.updateListView();
    }   

    // Set remaining properties.
    var ret = this.inherited("_setProps", arguments);
    
    if (props.autoComplete && props.autoCompleteOptions != null && this.listWidget != null ) {
        this.adjustListGeometry();  //even if autocomplete options are not defined in this set of props
    }
    return ret;
};

/**
 * Process the blur  event - activate delayed timer to close the list.
 * Such timer will be canceled if either list of text field receive 
 * the focus back within a delay period.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processBlurEvent = function(event) {    
    //clear timeout for list closing, thereby preventing list from being closed    
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    this.closingTimerId = setTimeout( 
        this.createCloseListCallback(), this.autoCompleteCloseDelayTime);   

    return true;
};

/**
 * Process keyPress events on the field, which enforces/disables 
 * submitForm behavior.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processFieldKeyDownEvent = function(event) {
    event = this.widget.getEvent(event);
    if (event == null) {
        return false;
    }

    if (event.keyCode == this.widget.keyCodes.ENTER && this.autoCompleteIsOpen) {               
        // Disable form submission. Note that form submission is disabled
        // only once when listbox is open. If it closed, form will be submitted
        // according to submitForm flag
        if (window.event) {
            event.cancelBubble = true;
            event.returnValue = false;
        } else{
            event.preventDefault();
            event.stopPropagation();
        } 
        this.fieldNode.value = this.listWidget.getSelectedValue();

        //force close the list box
        this.showAutoComplete = false;
        this.updateListView();    
        
        return true;
    }   

    //close the list in case of ESCAPE pressed
    if (event.keyCode == this.widget.keyCodes.ESCAPE  ) {               
        this.fieldNode.value = ""; //Warning: IE empties all fields on the page on 2nd ESC independently of setting value here
        this.showAutoComplete = false;
        this.updateListView();     
        this.filterOptions();
        return true;
    }
      
    //even when the text field is in focus, we want arrow keys ( up and down)
    //navigate through options in the select list
    if (event.keyCode == this.widget.keyCodes.DOWN_ARROW && this.listWidget.getSelectedIndex() < this.listNode.options.length) {               
        try {
            this.showAutoComplete = true;

            if (!this.autoCompleteIsOpen || this.listNode.options.length <=1) {
                this.filterOptions();
                this.listWidget.setSelectedIndex(0) ;
            } else {     
                //list already open
                this.listWidget.setSelectedIndex(this.listWidget.getSelectedIndex() + 1) ;
                this.updateListView();     
                
            }
            this.processListChange(event);
            return true;
       } catch (doNothing) {}
    }
    if (event.keyCode == this.widget.keyCodes.UP_ARROW && this.listWidget.getSelectedIndex() > 0) {               
        try {
            this.showAutoComplete = true;

                this.listWidget.setSelectedIndex(this.listWidget.getSelectedIndex() - 1) ;
            this.processListChange(event);
            return true;
        } catch (doNothing) {}
    }
    return true;
};

/**
 * Process keyPress events on the filter, which chages the filter
 * and submits a request for updated list.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processFieldKeyUpEvent = function(event) {    
    event = this.widget.getEvent(event);

    if (event != null &&
        ( event.keyCode == this.widget.keyCodes.ESCAPE ||
          event.keyCode == this.widget.keyCodes.ENTER ||
          event.keyCode == this.widget.keyCodes.DOWN_ARROW ||
          event.keyCode == this.widget.keyCodes.UP_ARROW  ||
          event.keyCode == this.widget.keyCodes.SHIFT  ||
          event.keyCode == this.widget.keyCodes.TAB
        )) {
        //these special keys are processed somewhere else - no filtering
        return false; 
    }
    this.showAutoComplete = true;
    this.filterOptions();
    return true;
};

/**
 * Process the focus event - turn on the flag for autoComplete list display
 * <p>
 * Displays autoComplete box on focus 
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processFocusEvent = function(event) {
    //clear timeout for list closing, thereby preventing list from being closed
    if (this.closingTimerId) {
        clearTimeout(this.closingTimerId);
        this.closingTimerId = null;
    }
    return true;
};

/**
 * Process onChange event on the select list, which results in filter being
 * updated with the new value. Note that no data submission is initiated here.
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.processListChange = function(event) {    
    event = this.widget.getEvent(event);

    if (event.type == "change") {               
        try {
            this.fieldNode.value = this.listWidget.getSelectedValue();
            
            //close the list
            this.showAutoComplete = false;
            this.updateListView();  
            this.fieldNode.focus(); 
                           
            return true;
       } catch (doNothing) {}
    }    

    // else - usually from selection movement with the arrow keys 
    this.fieldNode.value = this.listWidget.getSelectedValue();
    this.fieldNode.focus(); //keep the focus on filter field so that user can incrementally type additional data
    return true;
};

/**
 * Helper function to update the view of the component ( what is commonly known
 * in UI world as /refresh/ - but renamed not to be confused with Ajax refresh)
 *  - if size of the list box >=1, shows autocomplete list box.
 *  - if size of the list box <1, hides autocomplete list box.
 * 
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.updateListView = function() {
    if ( this.showAutoComplete == true && this.autoCompleteOptions.length >= 1) {
        //TODO a place for optimization here - not to adjust geometry each time
        this.adjustListGeometry();    

        //optionally we could add check for this.listNode.options.length >0
        this.listNode.className = this.widget.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "");

        // the following is preferred way of setting class, but it does not work
        //this.listWidget.setProps({className: this.widget.getClassName("TEXT_FIELD_AUTO_COMPLETE_LIST", "")}) ;
        
        //this.autoCompleteIsOpen flag indicates whether list box is open or not
        this.autoCompleteIsOpen = true;
    } else {
        this.listNode.className = this.theme.getClassName("HIDDEN");
        //this.listWidget.setProps(visible: 'false');
        this.autoCompleteIsOpen = false;
    }
    return true;
};

/**
 * Process validation event.
 * <p>
 * This function interprets an event (one of onXXX events, such as onBlur,
 * etc) and extracts data needed for subsequent Ajax request generation. 
 * Specifically, the widget id that has generated the event. If widget
 * id is found, publishBeginEvent is called with extracted data.
 * </p>
 *
 * @param {Event} event The JavaScript event.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textField.prototype.validate = function(event) {
    // Publish an event for custom AJAX implementations to listen for.
    this.publish(webui.suntheme4_2.widget.textField.event.validation.beginTopic, [{
        id: this.id
    }]);
    return true;
};

/**
 * @name webui.suntheme4_2.widget.textArea
 * @extends webui.suntheme4_2.widget.textField
 * @class This class contains functions for the textArea widget.
 * @constructor This function is used to construct a textArea widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.textArea", webui.suntheme4_2.widget.textField, {
    // Set defaults.
    constructor: function() {
        this.autoSave = 0;
        this.cols = 20;
        this.rows = 3;
    },
    widgetName: "textArea" // Required for theme properties.
});

/**
 * Helper function to create callback for timer event.
 *
 * @return {Function} The callback function.
 */
webui.suntheme4_2.widget.textArea.prototype.createSubmitCallback = function() {
    var _id = this.id;

    // New literals are created every time this function
    // is called, and it's saved by closure magic.
    return function(event) { 
        var widget = webui.suntheme4_2.dijit.byId(_id);
        if (widget == null) {
            return false;
        }
        //Create a submit request only if field has been modified
        if (widget.lastSaved != widget.fieldNode.value) {
            widget.lastSaved = widget.fieldNode.value;
            widget.submit();
        }
        return true;
    };
};

/**
 * This object contains event topics.
 * <p>
 * Note: Event topics must be prototyped for inherited functions. However, these
 * topics must also be available statically so that developers may subscribe to
 * events.
 * </p>
 * @ignore
 */
webui.suntheme4_2.widget.textArea.event =
        webui.suntheme4_2.widget.textArea.prototype.event = {
    /**
     * This object contains refresh event topics.
     * @ignore
     */
    refresh: {
        /** Refresh event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textArea_event_refresh_begin",

        /** Refresh event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textArea_event_refresh_end"
    },

    /**
     * This object contains state event topics.
     * @ignore
     */
    state: {
        /** State event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textArea_event_state_begin",

        /** State event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textArea_event_state_end"
    },

    /**
     * This object contains submit event topics.
     * @ignore
     */
    submit: {
        /** Submit event topic for custom AJAX implementations to listen for. */
        beginTopic: "webui_suntheme4_2_widget_textArea_event_submit_begin",

        /** Submit event topic for custom AJAX implementations to listen for. */
        endTopic: "webui_suntheme4_2_widget_textArea_event_submit_end"
    }
};

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.textArea.prototype.getInputClassName = function() {
    // Set readOnly style
    if (this.fieldNode.readOnly) {
        return this.widget.getClassName("TEXT_AREA_READONLY", "");
    }

    // Apply invalid style.
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("TEXT_AREA_INVALID", "")
        : " " + this.widget.getClassName("TEXT_AREA_VALID", "");

    // Set default style.    
    return (this.disabled == true)
        ? this.widget.getClassName("TEXT_AREA_DISABLED", "") 
        : this.widget.getClassName("TEXT_AREA", "") + validStyle;    
};

/**
 * This function is used to get widget properties. Please see the 
 * setProps() function for a list of supported properties.
 *
 * @return {Object} Key-Value pairs of properties.
 */
webui.suntheme4_2.widget.textArea.prototype.getProps = function() {
    var props = this.inherited("getProps", arguments);
    
    // Set properties.
    if (this.cols > 0 ) { props.cols = this.cols; }
    if (this.rows > 0) { props.rows = this.rows; }
    if (this.autoSave > 0 ) { props.autoSave = this.autoSave; }
   
    return props;
};

/**
 * This function is used to fill in remaining template properties, after the
 * buildRendering() function has been processed.
 * <p>
 * Note: Unlike Dojo 0.4, the DOM nodes don't exist in the document, yet. 
 * </p>
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textArea.prototype.postCreate = function () {
    // Set events.                
    if (this.autoSave > 0) {
        this.autoSaveTimerId = setInterval(this.createSubmitCallback(), 
            this.autoSave);  
    }
    return this.inherited("postCreate", arguments);
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {boolean} autoSave 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} rows 
 * @config {String} style Specify style rules inline.
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @param {boolean} notify Publish an event for custom AJAX implementations to listen for.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.textArea.prototype.setProps = function(props, notify) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};

/**
 * This function is used to set widget properties. Please see the setProps() 
 * function for a list of supported properties.
 * <p>
 * Note: This function should only be invoked through setProps().
 * </p>
 * @param {Object} props Key-Value pairs of properties.
 * @return {boolean} true if successful; otherwise, false.
 * @private
 */
webui.suntheme4_2.widget.textArea.prototype._setProps = function(props) {
    if (props == null) {
        return false;
    }

    // Set properties.   
    if (props.cols > 0) { this.fieldNode.cols = props.cols; }
    if (props.rows > 0) { this.fieldNode.rows = props.rows; }
    
    // Cancel autosave if it has been changed to <=0
    if (props.autoSave <= 0 && this.autoSaveTimerId && this.autoSaveTimerId != null ) {
        clearTimeout(this.autoSaveTimerId);
        this.autoSaveTimerId = null;
    }

    // Set label className -- must be set before calling superclass.
    if (props.label) {
        props.label.className = (props.label.className)
            ? this.widget.getClassName("TEXT_AREA_TOPLABELALIGN", "")  + " " + props.label.className
            : this.widget.getClassName("TEXT_AREA_TOPLABELALIGN", "") ;
    }

    // Set remaining properties.
    return this.inherited("_setProps", arguments);
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textArea.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textArea.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.jsfx.textField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.json");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.common");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.jsfx.dynaFaces");
webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.textField");

/**
 * @class This class contains functions to obtain data asynchronously using JSF
 * Extensions as the underlying transfer protocol.
 * @static
 */
webui.suntheme4_2.widget.jsfx.textField = {
    /**
     * This function is used to process validation events with Object literals.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @return {boolean} true if successful; otherwise, false.
     */
    processValidationEvent: function(props) {
        if (props == null) {
            return false;
        }
        
        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);
        
        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: props.id,
            render: props.id,
            replaceElement: webui.suntheme4_2.widget.jsfx.textField.validationCallback,
            xjson: {
                id : props.id,
                event: "validate"
            }
        });
        return true;
    },  

    /**
     * This function is used to update widgets.
     *
     * @param {String} elementId The HTML element Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to DynaFaces.fireAjaxTransaction.
     * @param {Object} xjson The xjson argument provided to DynaFaces.fireAjaxTransaction.
     * @return {boolean} true if successful; otherwise, false.
     */
    validationCallback: function(elementId, content, closure, xjson) {
        if (elementId == null || content == null) {
            return false;
        }
        
        // Parse JSON text.
        var props = webui.suntheme4_2.json.parse(content);

        // Update text field.
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        widget.setProps({
            valid: props.valid,
            errorImage: {
                title: props.detail
            }
        });

        // Notify decoupled widgets.
        if (widget.notify) {
            // Update each given client ID.
            for (var i = 0; i < widget.notify.length; i++) {
                // Get widget associated with client ID.
                var curWidget = webui.suntheme4_2.dijit.byId(widget.notify[i]);
                if (curWidget && typeof curWidget.notify == "function") {
                    curWidget.notify(props);
                }
            }
        }

        // Publish an event for custom AJAX implementations to listen for.
        webui.suntheme4_2.dojo.publish(
            webui.suntheme4_2.widget.textField.event.validation.endTopic, [props]);
        return true;
    },

    /**
     * This function is used to process autoComplete events.
     *
     * @param props Key-Value pairs of properties.
     * @config {String} id The HTML element Id.
     * @return {boolean} true if successful; otherwise, false.
     */
    processAutoCompleteEvent: function(props) {
        if (props == null) {
            return false;
        }
        
        // Dynamic Faces requires a DOM node as the source property.
        var domNode = document.getElementById(props.id);
        
        // Generate AJAX request using the JSF Extensions library.
        DynaFaces.fireAjaxTransaction(
            (domNode) ? domNode : document.forms[0], {
            execute: props.id,
            render: props.id,
            replaceElement: webui.suntheme4_2.widget.jsfx.textField.autoCompleteCallback,
            xjson: {
                id : props.id,
                event: "autocomplete"
            }
        });
        return true;
    },  

    /**
     * This function is used to update widgets with new autoComplete options list.
     *
     * @param {String} elementId The HTML element Id.
     * @param {String} content The content returned by the AJAX response.
     * @param {Object} closure The closure argument provided to DynaFaces.fireAjaxTransaction.
     * @param {Object} xjson The xjson argument provided to DynaFaces.fireAjaxTransaction.
     * @return {boolean} true if successful; otherwise, false.
     */
    autoCompleteCallback: function(elementId, content, closure, xjson) {
        if (elementId == null || content == null) {
            return false;
        }
        
        // Parse JSON text.
        var props = webui.suntheme4_2.json.parse(content);

        // Update text field.
        var widget = webui.suntheme4_2.dijit.byId(elementId);
        widget.setProps(props);       


        // Publish an event for custom AJAX implementations to listen for.
        webui.suntheme4_2.dojo.publish(
            webui.suntheme4_2.widget.textField.event.autoComplete.endTopic, [props]);
        return true;
    }    
};

// Listen for Dojo Widget events.
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textField.event.refresh.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processRefreshEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textField.event.state.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processStateEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textField.event.submit.beginTopic,
    webui.suntheme4_2.widget.jsfx.common, "processSubmitEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textField.event.validation.beginTopic,
    webui.suntheme4_2.widget.jsfx.textField, "processValidationEvent");
webui.suntheme4_2.dojo.subscribe(webui.suntheme4_2.widget.textField.event.autoComplete.beginTopic,
    webui.suntheme4_2.widget.jsfx.textField, "processAutoCompleteEvent");


webui.suntheme4_2.dojo.provide("webui.suntheme4_2.widget.passwordField");

webui.suntheme4_2.dojo.require("webui.suntheme4_2.widget.fieldBase");

/**
 * @name webui.suntheme4_2.widget.passwordField
 * @extends webui.suntheme4_2.widget.fieldBase
 * @class This class contains functions for the passwordField widget.
 * @constructor This function is used to construct a passwordField widget.
 */
webui.suntheme4_2.dojo.declare("webui.suntheme4_2.widget.passwordField", webui.suntheme4_2.widget.fieldBase, {
    // Set defaults.
    widgetName: "passwordField" // Required for theme properties.
});

/**
 * Helper function to obtain HTML input element class names.
 *
 * @return {String} The HTML input element class name.
 */
webui.suntheme4_2.widget.passwordField.prototype.getInputClassName = function() {
    if (this.fieldNode.readOnly) {
        return this.widget.getClassName("PASSWORD_FIELD_READONLY", "");
    }

    //invalid style
    var validStyle =  (this.valid == false) 
        ? " " + this.widget.getClassName("PASSWORD_FIELD_INVALID", "")
        : " " + this.widget.getClassName("PASSWORD_FIELD_VALID", "");
    
    // Set default style.    
    return (this.disabled == true)
        ? this.widget.getClassName("PASSWORD_FIELD_DISABLED", "") 
        : this.widget.getClassName("PASSWORD_FIELD", "") + validStyle;
};

/**
 * This function is used to set widget properties using Object literals.
 * <p>
 * Note: This function extends the widget object for later updates. Further, the
 * widget shall be updated only for the given key-value pairs.
 * </p><p>
 * If the notify param is true, the widget's state change event shall be
 * published. This is typically used to keep client-side state in sync with the
 * server.
 * </p>
 *
 * @param {Object} props Key-Value pairs of properties.
 * @config {String} accesskey 
 * @config {String} className CSS selector.
 * @config {String} dir Specifies the directionality of text.
 * @config {boolean} disabled Disable element.
 * @config {String} id Uniquely identifies an element within a document.
 * @config {String} label
 * @config {String} lang Specifies the language of attribute values and content.
 * @config {int} maxLength 
 * @config {Array} notify 
 * @config {String} onBlur Element lost focus.
 * @config {String} onClick Mouse button is clicked on element.
 * @config {String} onDblClick Mouse button is double-clicked on element.
 * @config {String} onFocus Element received focus.
 * @config {String} onKeyDown Key is pressed down over element.
 * @config {String} onKeyPress Key is pressed and released over element.
 * @config {String} onKeyUp Key is released over element.
 * @config {String} onMouseDown Mouse button is pressed over element.
 * @config {String} onMouseOut Mouse is moved away from element.
 * @config {String} onMouseOver Mouse is moved onto element.
 * @config {String} onMouseUp Mouse button is released over element.
 * @config {String} onMouseMove Mouse is moved while over element.
 * @config {boolean} readOnly 
 * @config {boolean} required 
 * @config {int} size 
 * @config {String} style Specify style rules inline.
 * @config {boolean} submitForm
 * @config {int} tabIndex Position in tabbing order.
 * @config {String} title Provides a title for element.
 * @config {boolean} valid
 * @config {String} value Value of input.
 * @config {boolean} visible Hide or show element.
 * @return {boolean} true if successful; otherwise, false.
 */
webui.suntheme4_2.widget.passwordField.prototype.setProps = function(props) {
    // Note: This function is overridden for JsDoc.
    return this.inherited("setProps", arguments);
};
