/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.netbeans.microedition.databinding.el;

//import java.math.BigInteger;
//import java.math.BigDecimal;

/**
 * <p>This class contains the logic for coercing data types before
 * operators are applied to them.
 * <p/>
 * <p>The following is the list of rules applied for various type
 * conversions.
 * <p/>
 * <ul><pre>
 * Applying arithmetic operator
 *   Binary operator - A {+,-,*} B
 *     if A and B are null
 *       return 0
 *     if A or B is BigDecimal, coerce both to BigDecimal and then:
 *       if operator is +, return <code>A.add(B)</code>
 *       if operator is -, return <code>A.subtract(B)</code>
 *       if operator is *, return <code>A.multiply(B)</code>
 *     if A or B is Float, Double, or String containing ".", "e", or "E"
 *       if A or B is BigInteger, coerce both A and B to BigDecimal and apply operator
 *       coerce both A and B to Double and apply operator
 *     if A or B is BigInteger, coerce both to BigInteger and then:
 *       if operator is +, return <code>A.add(B)</code>
 *       if operator is -, return <code>A.subtract(B)</code>
 *       if operator is *, return <code>A.multiply(B)</code>
 *     otherwise
 *       coerce both A and B to Long
 *       apply operator
 *     if operator results in exception (such as divide by 0), error
 * <p/>
 *   Binary operator - A {/,div} B
 *     if A and B are null
 *       return 0
 *     if A or B is a BigDecimal or BigInteger, coerce both to BigDecimal and
 *      return <code>A.divide(B, BigDecimal.ROUND_HALF_UP)</code>
 *     otherwise
 *       coerce both A and B to Double
 *       apply operator
 *     if operator results in exception (such as divide by 0), error
 * <p/>
 *   Binary operator - A {%,mod} B
 *     if A and B are null
 *       return 0
 *     if A or B is BigDecimal, Float, Double, or String containing ".", "e" or "E"
 *       coerce both to Double
 *       apply operator
 *     if A or B is BigInteger, coerce both to BigInteger and return
 *      <code>A.remainder(B)</code>
 *     otherwise
 *       coerce both A and B to Long
 *       apply operator
 *     if operator results in exception (such as divide by 0), error
 * <p/>
 *   Unary minus operator - -A
 *     if A is null
 *       return 0
 *     if A is BigInteger or BigDecimal, return <code>A.negate()</code>
 *     if A is String
 *       if A contains ".", "e", or "E"
 *         coerce to Double, apply operator
 *       otherwise
 *         coerce to a Long and apply operator
 *     if A is Byte,Short,Integer,Long,Float,Double
 *       retain type, apply operator
 *     if operator results in exception, error
 *     otherwise
 *       error
 * <p/>
 * Applying "empty" operator - empty A
 *   if A is null
 *     return true
 *   if A is zero-length String
 *     return true
 *   if A is zero-length array
 *     return true
 *   if A is List and ((List) A).isEmpty()
 *     return true
 *   if A is Map and ((Map) A).isEmpty()
 *     return true
 *   if A is Collection an ((Collection) A).isEmpty()
 *     return true
 *   otherwise
 *     return false
 * <p/>
 * Applying logical operators
 *   Binary operator - A {and,or} B
 *     coerce both A and B to Boolean, apply operator
 *   NOTE - operator stops as soon as expression can be determined, i.e.,
 *     A and B and C and D - if B is false, then only A and B is evaluated
 *   Unary not operator - not A
 *     coerce A to Boolean, apply operator
 * <p/>
 * Applying relational operator
 *   A {<,>,<=,>=,lt,gt,lte,gte} B
 *     if A==B
 *       if operator is >= or <=
 *         return true
 *       otherwise
 *         return false
 *     if A or B is null
 *       return false
 *     if A or B is BigDecimal, coerce both A and B to BigDecimal and use the
 *      return value of <code>A.compareTo(B)</code>
 *     if A or B is Float or Double
 *       coerce both A and B to Double
 *       apply operator
 *     if A or B is BigInteger, coerce both A and B to BigInteger and use the
 *      return value of <code>A.compareTo(B)</code>
 *     if A or B is Byte,Short,Character,Integer,Long
 *       coerce both A and B to Long
 *       apply operator
 *     if A or B is String
 *       coerce both A and B to String, compare lexically
 *     if A is Comparable
 *       if A.compareTo (B) throws exception
 *         error
 *       otherwise
 *         use result of A.compareTo(B)
 *     if B is Comparable
 *       if B.compareTo (A) throws exception
 *         error
 *       otherwise
 *         use result of B.compareTo(A)
 *     otherwise
 *       error
 * <p/>
 * Applying equality operator
 *   A {==,!=} B
 *     if A==B
 *       apply operator
 *     if A or B is null
 *       return false for ==, true for !=
 *     if A or B is BigDecimal, coerce both A and B to BigDecimal and then:
 *       if operator is == or eq, return <code>A.equals(B)</code>
 *       if operator is != or ne, return <code>!A.equals(B)</code>
 *     if A or B is Float or Double
 *       coerce both A and B to Double
 *       apply operator
 *     if A or B is BigInteger, coerce both A and B to BigInteger and then:
 *       if operator is == or eq, return <code>A.equals(B)</code>
 *       if operator is != or ne, return <code>!A.equals(B)</code>
 *     if A or B is Byte,Short,Character,Integer,Long
 *       coerce both A and B to Long
 *       apply operator
 *     if A or B is Boolean
 *       coerce both A and B to Boolean
 *       apply operator
 *     if A or B is String
 *       coerce both A and B to String, compare lexically
 *     otherwise
 *       if an error occurs while calling A.equals(B)
 *         error
 *       apply operator to result of A.equals(B)
 * <p/>
 * coercions
 * <p/>
 *   coerce A to String
 *     A is String
 *       return A
 *     A is null
 *       return ""
 *     A.toString throws exception
 *       error
 *     otherwise
 *       return A.toString
 * <p/>
 *   coerce A to Number type N
 *     A is null or ""
 *       return 0
 *     A is Character
 *       convert to short, apply following rules
 *     A is Boolean
 *       error
 *     A is Number type N
 *       return A
 *     A is Number, coerce quietly to type N using the following algorithm
 *         If N is BigInteger
 *             If A is BigDecimal, return <code>A.toBigInteger()</code>
 *             Otherwise, return <code>BigInteger.valueOf(A.longValue())</code>
 *        if N is BigDecimal
 *             If A is a BigInteger, return <code>new BigDecimal(A)</code>
 *             Otherwise, return <code>new BigDecimal(A.doubleValue())</code>
 *        If N is Byte, return <code>new Byte(A.byteValue())</code>
 *        If N is Short, return <code>new Short(A.shortValue())</code>
 *        If N is Integer, return <code>new Integer(A.integerValue())</code>
 *        If N is Long, return <code>new Long(A.longValue())</code>
 *        If N is Float, return <code>new Float(A.floatValue())</code>
 *        If N is Double, return <code>new Double(A.doubleValue())</code>
 *        otherwise ERROR
 *     A is String
 *       If N is BigDecimal then:
 *            If <code>new BigDecimal(A)</code> throws an exception then ERROR
 *            Otherwise, return <code>new BigDecimal(A)</code>
 *       If N is BigInteger then:
 *            If <code>new BigInteger(A)</code> throws an exception, then ERROR
 *            Otherwise, return <code>new BigInteger(A)</code>
 *       new <code>N.valueOf(A)</code> throws exception
 *         error
 *       return <code>N.valueOf(A)</code>
 *     otherwise
 *       error
 * <p/>
 *   coerce A to Character should be
 *     A is null or ""
 *       return (char) 0
 *     A is Character
 *       return A
 *     A is Boolean
 *       error
 *     A is Number with less precision than short
 *       coerce quietly - return (char) A
 *     A is Number with greater precision than short
 *       coerce quietly - return (char) A
 *     A is String
 *       return A.charAt (0)
 *     otherwise
 *       error
 * <p/>
 *   coerce A to Boolean
 *     A is null or ""
 *       return false
 *     A is Boolean
 *       return A
 *     A is String
 *       Boolean.valueOf(A) throws exception
 *         error
 *       return Boolean.valueOf(A)
 *     otherwise
 *       error
 * <p/>
 *   coerce A to any other type T
 *     A is null
 *       return null
 *     A is assignable to T
 *       coerce quietly
 *     A is String
 *       T has no PropertyEditor
 *         if A is "", return null
 *         otherwise error
 *       T's PropertyEditor throws exception
 *         if A is "", return null
 *         otherwise error
 *       otherwise
 *         apply T's PropertyEditor
 *     otherwise
 *       error
 * </pre></ul>
 * @author Nathan Abramson - Art Technology Group
 * @version $Change: 181177 $$DateTime: 2001/06/26 08:45:09 $$Author: luehe $
 */

public class Coercions {

    private static final Integer ZERO = new Integer (0);
    //-------------------------------------

//    /**
//     * Coerces the given value to the specified class.
//     */
//    private static Object coerce (Object pValue, Class pClass, Logger pLogger) {
//        if (pClass == String.class) {
//            return coerceObjectToString (pValue, pLogger);
//        } else if (isNumberClass (pClass)) {
//            return coerceNumberToPrimitiveNumber (pValue, pClass, pLogger);
//        } else if (pClass == Character.class ||
//            pClass == Character.TYPE) {
//            return coerceObjectToCharacter (pValue, pLogger);
//        } else if (pClass == Boolean.class ||
//            pClass == Boolean.TYPE) {
//            return coerceObjectToBoolean (pValue, pLogger);
//        } else {
//            return coerceToObject (pValue, pClass, pLogger);
//        }
//    }

    //-------------------------------------

//    /**
//     * Returns true if the given class is Byte, Short, Integer, Long,
//     * Float, Double, BigInteger, or BigDecimal
//     */
//    private static boolean isNumberClass (Class pClass) {
//        return
//            pClass == Byte.class ||
//                pClass == Byte.TYPE ||
//                pClass == Short.class ||
//                pClass == Short.TYPE ||
//                pClass == Integer.class ||
//                pClass == Integer.TYPE ||
//                pClass == Long.class ||
//                pClass == Long.TYPE ||
//                pClass == Float.class ||
//                pClass == Float.TYPE ||
//                pClass == Double.class ||
//                pClass == Double.TYPE ||
//                pClass == BigInteger.class ||
//                pClass == BigDecimal.class;
//    }

    //-------------------------------------

    /**
     * Coerces the specified value to a String
     */
    private static String coerceObjectToString (Object pValue, Logger pLogger) {
        if (pValue == null) {
            return "";
        } else if (pValue instanceof String) {
            return (String) pValue;
        } else {
            try {
                return pValue.toString ();
            } catch (Exception exc) {
                if (pLogger.isLoggingError ()) {
                    pLogger.logError (Constants.TOSTRING_EXCEPTION, exc, pValue.getClass ().getName ());
                }
                return "";
            }
        }
    }

    //-------------------------------------

    /**
     * Coerces a value to the given primitive number class
     */
    public static Object coerceObjectToPrimitiveNumber (Object pValue, Class pClass, Logger pLogger) {
        if (pValue == null || "".equals (pValue)) {
            return coerceNumberToPrimitiveNumber (ZERO, pClass);
        } else if (pValue instanceof Character) {
            char val = ((Character) pValue).charValue ();
            return coerceNumberToPrimitiveNumber (new Short ((short) val), pClass);
        } else if (pValue instanceof Boolean) {
            if (pLogger.isLoggingError ()) {
                pLogger.logError (Constants.BOOLEAN_TO_NUMBER, pValue, pClass.getName ());
            }
            return coerceNumberToPrimitiveNumber (ZERO, pClass);
        } else if (pValue.getClass () == pClass) {
            return pValue;
        } else if (isInstanceOfNumber (pValue)) {// instanceof Number) {
            return coerceNumberToPrimitiveNumber (pValue, pClass);
        } else if (pValue instanceof String) {
            try {
                return coerceStringToPrimitiveNumber ((String) pValue, pClass);
            } catch (Exception exc) {
                if (pLogger.isLoggingError ()) {
                    pLogger.logError (Constants.STRING_TO_NUMBER_EXCEPTION, (String) pValue, pClass.getName ());
                }
                return coerceNumberToPrimitiveNumber (ZERO, pClass);
            }
        } else {
            if (pLogger.isLoggingError ()) {
                pLogger.logError (Constants.COERCE_TO_NUMBER, pValue.getClass ().getName (), pClass.getName ());
            }
            return coerceLongToPrimitiveNumber (0, pClass);
        }
    }

    //-------------------------------------

//    /**
//     * Coerces a value to an Integer, returning null if the coercion
//     * isn't possible.
//     */
//    private static Integer coerceObjectToInteger (Object pValue, Logger pLogger) {
//        if (pValue == null) {
//            return null;
//        } else if (pValue instanceof Character) {
//            return new Integer ((int) (((Character) pValue).charValue ()));
//        } else if (pValue instanceof Boolean) {
//            if (pLogger.isLoggingWarning ()) {
//                pLogger.logWarning (Constants.BOOLEAN_TO_NUMBER, pValue, Integer.class.getName ());
//            }
//            return new Integer (((Boolean) pValue).booleanValue () ? 1 : 0);
//        } else if (pValue instanceof Integer) {
//            return (Integer) pValue;
//        } else if (isInstanceOfNumber (pValue)) {// instanceof Number) {
//            return new Integer (intValue (pValue));
//        } else if (pValue instanceof String) {
//            try {
//                return new Integer (Integer.parseInt ((String) pValue));
//            } catch (Exception exc) {
//                if (pLogger.isLoggingWarning ()) {
//                    pLogger.logWarning (Constants.STRING_TO_NUMBER_EXCEPTION, pValue, Integer.class.getName ());
//                }
//                return null;
//            }
//        } else {
//            if (pLogger.isLoggingWarning ()) {
//                pLogger.logWarning (Constants.COERCE_TO_NUMBER, pValue.getClass ().getName (),
//                        Integer.class.getName ());
//            }
//            return null;
//        }
//    }

    //-------------------------------------

    /**
     * Coerces a long to the given primitive number class
     */
    static Object coerceLongToPrimitiveNumber (long pValue, Class pClass) {
        if (pClass == Byte.class /*|| pClass == Byte.TYPE*/) {
            return new Byte ((byte) pValue);
        } else if (pClass == Short.class /*|| pClass == Short.TYPE*/) {
            return new Short ((short) pValue);
        } else if (pClass == Integer.class /*|| pClass == Integer.TYPE*/) {
            return new Integer ((int) pValue);
        } else if (pClass == Long.class /*|| pClass == Long.TYPE*/) {
            return new Long (pValue);
        } else if (pClass == Float.class /*|| pClass == Float.TYPE*/) {
            return new Float ((float) pValue);
        } else if (pClass == Double.class /*|| pClass == Double.TYPE*/) {
            return new Double ((double) pValue);
        } else {
            return ZERO;
        }
    }

    //-------------------------------------

//    /**
//     * Coerces a double to the given primitive number class
//     */
//    static Object coerceDoubleToPrimitiveNumber (double pValue, Class pClass) {
//        if (pClass == Byte.class || pClass == Byte.TYPE) {
//            return new Byte ((byte) pValue);
//        } else if (pClass == Short.class || pClass == Short.TYPE) {
//            return new Short ((short) pValue);
//        } else if (pClass == Integer.class || pClass == Integer.TYPE) {
//            return new Integer ((int) pValue);
//        } else if (pClass == Long.class || pClass == Long.TYPE) {
//            return new Long ((long) pValue);
//        } else if (pClass == Float.class || pClass == Float.TYPE) {
//            return new Float ((float) pValue);
//        } else if (pClass == Double.class || pClass == Double.TYPE) {
//            return new Double (pValue);
//        } else {
//            return ZERO;
//        }
//    }

    //-------------------------------------

    /**
     * Coerces a Number to the given primitive number class
     */
    private static Object coerceNumberToPrimitiveNumber (Object pValue, Class pClass) {
        if (pClass == Byte.class /*|| pClass == Byte.TYPE*/) {
            return new Byte (byteValue (pValue));
        } else if (pClass == Short.class /*|| pClass == Short.TYPE*/) {
            return new Short (shortValue (pValue));
        } else if (pClass == Integer.class /*|| pClass == Integer.TYPE*/) {
            return new Integer (intValue (pValue));
        } else if (pClass == Long.class /*|| pClass == Long.TYPE*/) {
            return new Long (longValue (pValue));
        } else if (pClass == Float.class /*|| pClass == Float.TYPE*/) {
            return new Float (floatValue (pValue));
        } else if (pClass == Double.class /*|| pClass == Double.TYPE*/) {
            return new Double (doubleValue (pValue));
//        } else if (pClass == BigInteger.class) {
//            if (pValue instanceof BigDecimal)
//                return ((BigDecimal) pValue).toBigInteger ();
//            else
//                return BigInteger.valueOf (pValue.longValue ());
//        } else if (pClass == BigDecimal.class) {
//            if (pValue instanceof BigInteger)
//                return new BigDecimal ((BigInteger) pValue);
//            else
//                return new BigDecimal (pValue.doubleValue ());
        } else {
            return ZERO;
        }
    }

    //-------------------------------------

    /**
     * Coerces a String to the given primitive number class
     */
    private static Object coerceStringToPrimitiveNumber (String pValue, Class pClass) {
        if (pClass == Byte.class /*|| pClass == Byte.TYPE*/) {
            return new Byte (Byte.parseByte (pValue));
        } else if (pClass == Short.class /*|| pClass == Short.TYPE*/) {
            return new Short (Short.parseShort (pValue));
        } else if (pClass == Integer.class /*|| pClass == Integer.TYPE*/) {
            return new Integer (Integer.parseInt (pValue));
        } else if (pClass == Long.class /*|| pClass == Long.TYPE*/) {
            return new Long (Long.parseLong (pValue));
        } else if (pClass == Float.class /*|| pClass == Float.TYPE*/) {
            return new Float (Float.parseFloat (pValue));
        } else if (pClass == Double.class /*|| pClass == Double.TYPE*/) {
            return new Double (Double.parseDouble (pValue));
//        } else if (pClass == BigInteger.class) {
//            return new BigInteger (pValue);
//        } else if (pClass == BigDecimal.class) {
//            return new BigDecimal (pValue);
        } else {
            return new Integer (0);
        }
    }

    //-------------------------------------

//    /**
//     * Coerces a value to a Character
//     */
//    private static Character coerceObjectToCharacter (Object pValue, Logger pLogger) {
//        if (pValue == null || "".equals (pValue)) {
//            return new Character ((char) 0);
//        } else if (pValue instanceof Character) {
//            return (Character) pValue;
//        } else if (pValue instanceof Boolean) {
//            if (pLogger.isLoggingError ()) {
//                pLogger.logError (Constants.BOOLEAN_TO_CHARACTER, pValue);
//            }
//            return new Character ((char) 0);
//        } else if (isInstanceOfNumber (pValue)) { //pValue instanceof Number) {
//            return new Character ((char) shortValue (pValue));
//        } else if (pValue instanceof String) {
//            String str = (String) pValue;
//            return new Character (str.charAt (0));
//        } else {
//            if (pLogger.isLoggingError ()) {
//                pLogger.logError (Constants.COERCE_TO_CHARACTER, pValue.getClass ().getName ());
//            }
//            return new Character ((char) 0);
//        }
//    }

    //-------------------------------------

    /**
     * Coerces a value to a Boolean
     */
    public static Boolean coerceObjectToBoolean (Object pValue, Logger pLogger) {
        if (pValue == null || "".equals (pValue)) {
            return Boolean.FALSE;
        } else if (pValue instanceof Boolean) {
            return (Boolean) pValue;
        } else if (pValue instanceof String) {
            String str = (String) pValue;
            return "true".equalsIgnoreCase (str) ? Boolean.TRUE : Boolean.FALSE;
//            try {
//                return Boolean.valueOf (str) ? Boolean.TRUE : Boolean.FALSE;
//            } catch (Exception exc) {
//                if (pLogger.isLoggingError ()) {
//                    pLogger.logError (Constants.STRING_TO_BOOLEAN, exc, (String) pValue);
//                }
//                return Boolean.FALSE;
//            }
        } else {
            if (pLogger.isLoggingError ()) {
                pLogger.logError (Constants.COERCE_TO_BOOLEAN, pValue.getClass ().getName ());
            }
            return Boolean.TRUE;
        }
    }

    //-------------------------------------

//    /**
//     * Coerces a value to the specified Class that is not covered by any
//     * of the above cases
//     */
//    private static Object coerceObjectToObject (Object pValue, Class pClass, Logger pLogger) {
//        if (pValue == null) {
//            return null;
//        } else if (pClass.isAssignableFrom (pValue.getClass ())) {
//            return pValue;
//        } else if (pValue instanceof String) {
//            String str = (String) pValue;
////            PropertyEditor pe = PropertyEditorManager.findEditor (pClass);
////            if (pe == null) {
//                if ("".equals (str)) {
//                    return null;
//                } else {
//                    if (pLogger.isLoggingError ()) {
//                        pLogger.logError (Constants.NO_PROPERTY_EDITOR, str, pClass.getName ());
//                    }
//                    return null;
//                }
////            }
////            try {
////                pe.setAsText (str);
////                return pe.getValue ();
////            } catch (IllegalArgumentException exc) {
////                if ("".equals (str)) {
////                    return null;
////                } else {
////                    if (pLogger.isLoggingError ()) {
////                        pLogger.logError (Constants.PROPERTY_EDITOR_ERROR, exc, pValue, pClass.getName ());
////                    }
////                    return null;
////                }
////            }
//        } else {
//            if (pLogger.isLoggingError ()) {
//                pLogger.logError (Constants.COERCE_TO_OBJECT, pValue.getClass ().getName (), pClass.getName ());
//            }
//            return null;
//        }
//    }

    //-------------------------------------
    // Applying operators
    //-------------------------------------

//    private static BigDecimal operateArithm (BigDecimal left, BigDecimal right, int operator) {
//        switch (operator) {
//            case ELUtils.OPERATOR_PLUS:
//                return left.add (right);
//            case ELUtils.OPERATOR_MINUS:
//                return left.subtract (right);
//            case ELUtils.OPERATOR_MULTIPLY:
//                return left.multiply (right);
//            default:
//                throw new IllegalStateException ();
//        }
//    }

//    private static BigInteger operateArithm (BigInteger left, BigInteger right, int operator) {
//        switch (operator) {
//            case ELUtils.OPERATOR_PLUS:
//                return left.add (right);
//            case ELUtils.OPERATOR_MINUS:
//                return left.subtract (right);
//            case ELUtils.OPERATOR_MULTIPLY:
//                return left.multiply (right);
//            default:
//                throw new IllegalStateException ();
//        }
//    }

    private static double operateArithm (double left, double right, int operator) {
        switch (operator) {
            case ELUtils.OPERATOR_PLUS:
                return left + right;
            case ELUtils.OPERATOR_MINUS:
                return left - right;
            case ELUtils.OPERATOR_MULTIPLY:
                return left * right;
            default:
                throw new IllegalStateException ();
        }
    }

    private static long operateArithm (long left, long right, int operator) {
        switch (operator) {
            case ELUtils.OPERATOR_PLUS:
                return left + right;
            case ELUtils.OPERATOR_MINUS:
                return left - right;
            case ELUtils.OPERATOR_MULTIPLY:
                return left * right;
            default:
                throw new IllegalStateException ();
        }
    }

    /**
     * Performs all of the necessary type conversions, then calls on the
     * appropriate operator.
     */
    public static Object applyArithmeticOperator (Object pLeft, Object pRight, int pOperator, Logger pLogger) {
        if (pLeft == null && pRight == null) {
            if (pLogger.isLoggingWarning ()) {
                pLogger.logWarning (Constants.ARITH_OP_NULL, String.valueOf(pOperator));
            }
            return ZERO;
//        } else if (isBigDecimal (pLeft) || isBigDecimal (pRight)) {
//            BigDecimal left = (BigDecimal) coerceObjectToPrimitiveNumber (pLeft, BigDecimal.class, pLogger);
//            BigDecimal right = (BigDecimal) coerceObjectToPrimitiveNumber (pRight, BigDecimal.class, pLogger);
//            return operateArithm (left, right, pOperator);
        } else if (isFloatingPointType (pLeft) ||
            isFloatingPointType (pRight) ||
            isFloatingPointString (pLeft) ||
            isFloatingPointString (pRight)) {
//            if (isBigInteger (pLeft) || isBigInteger (pRight)) {
//                BigDecimal left = (BigDecimal)
//                    coerceObjectToPrimitiveNumber (pLeft, BigDecimal.class, pLogger);
//                BigDecimal right = (BigDecimal)
//                    coerceObjectToPrimitiveNumber (pRight, BigDecimal.class, pLogger);
//                return operateArithm (left, right, pOperator);
//            } else {
                double left = doubleValue (coerceObjectToPrimitiveNumber (pLeft, Double.class, pLogger));
                double right = doubleValue (coerceObjectToPrimitiveNumber (pRight, Double.class, pLogger));
                return new Double (operateArithm (left, right, pOperator));
//            }
//        } else if (isBigInteger (pLeft) || isBigInteger (pRight)) {
//            BigInteger left = (BigInteger) coerceObjectToPrimitiveNumber (pLeft, BigInteger.class, pLogger);
//            BigInteger right = (BigInteger) coerceObjectToPrimitiveNumber (pRight, BigInteger.class, pLogger);
//            return operateArithm (left, right, pOperator);
        } else {
            long left = longValue (coerceObjectToPrimitiveNumber (pLeft, Long.class, pLogger));
            long right = longValue (coerceObjectToPrimitiveNumber (pRight, Long.class, pLogger));
            return new Long (operateArithm (left, right, pOperator));
        }
    }

    //-------------------------------------

//    private static boolean operateRelational (Comparable left, Comparable right, int operator) {
//        switch (operator) {
//            case ELUtils.OPERATOR_LT:
//                return left.compareTo (right) < 0;
//            case ELUtils.OPERATOR_GT:
//                return left.compareTo (right) > 0;
//            case ELUtils.OPERATOR_GE:
//                return left.compareTo (right) >= 0;
//            case ELUtils.OPERATOR_LE:
//                return left.compareTo (right) <= 0;
//            default:
//                throw new IllegalStateException ();
//        }
//    }

    private static boolean operateRelational (String left, String right, int operator) {
        switch (operator) {
            case ELUtils.OPERATOR_LT:
                return left.compareTo (right) < 0;
            case ELUtils.OPERATOR_GT:
                return left.compareTo (right) > 0;
            case ELUtils.OPERATOR_GE:
                return left.compareTo (right) >= 0;
            case ELUtils.OPERATOR_LE:
                return left.compareTo (right) <= 0;
            default:
                throw new IllegalStateException ();
        }
    }

    private static boolean operateRelational (long left, long right, int operator) {
        switch (operator) {
            case ELUtils.OPERATOR_LT:
                return left < right;
            case ELUtils.OPERATOR_GT:
                return left > right;
            case ELUtils.OPERATOR_GE:
                return left >= right;
            case ELUtils.OPERATOR_LE:
                return left <= right;
            default:
                throw new IllegalStateException ();
        }
    }

    private static boolean operateRelational (double left, double right, int operator) {
        switch (operator) {
            case ELUtils.OPERATOR_LT:
                return left < right;
            case ELUtils.OPERATOR_GT:
                return left > right;
            case ELUtils.OPERATOR_GE:
                return left >= right;
            case ELUtils.OPERATOR_LE:
                return left <= right;
            default:
                throw new IllegalStateException ();
        }
    }

    /**
     * Performs all of the necessary type conversions, then calls on the
     * appropriate operator.
     */
    public static Boolean applyRelationalOperator (Object pLeft, Object pRight, int pOperator, Logger pLogger) {
//        if (isBigDecimal (pLeft) || isBigDecimal (pRight)) {
//            BigDecimal left = (BigDecimal) coerceObjectToPrimitiveNumber (pLeft, BigDecimal.class, pLogger);
//            BigDecimal right = (BigDecimal) coerceObjectToPrimitiveNumber (pRight, BigDecimal.class, pLogger);
//            return Boolean.valueOf (operateRelational (left, right, pOperator));
//        } else
        if (isFloatingPointType (pLeft) || isFloatingPointType (pRight)) {
            double left = doubleValue (coerceObjectToPrimitiveNumber (pLeft, Double.class, pLogger));
            double right = doubleValue (coerceObjectToPrimitiveNumber (pRight, Double.class, pLogger));
            return operateRelational (left, right, pOperator) ? Boolean.TRUE : Boolean.FALSE;
//        } else if (isBigInteger (pLeft) || isBigInteger (pRight)) {
//            BigInteger left = (BigInteger) coerceObjectToPrimitiveNumber (pLeft, BigInteger.class, pLogger);
//            BigInteger right = (BigInteger) coerceObjectToPrimitiveNumber (pRight, BigInteger.class, pLogger);
//            return Boolean.valueOf (operateRelational (left, right, pOperator));
        } else if (isIntegerType (pLeft) || isIntegerType (pRight)) {
            long left = longValue (coerceObjectToPrimitiveNumber (pLeft, Long.class, pLogger));
            long right = longValue (coerceObjectToPrimitiveNumber (pRight, Long.class, pLogger));
            return operateRelational (left, right, pOperator) ? Boolean.TRUE : Boolean.FALSE;
        } else if (pLeft instanceof String || pRight instanceof String) {
            String left = coerceObjectToString (pLeft, pLogger);
            String right = coerceObjectToString (pRight, pLogger);
            return operateRelational (left, right, pOperator) ? Boolean.TRUE : Boolean.FALSE;
//        } else if (pLeft instanceof Comparable) {
//            try {
//                int result = ((Comparable) pLeft).compareTo (pRight);
//                return Boolean.valueOf (operateRelational (result, -result, pOperator));
//            } catch (Exception exc) {
////                if (pLogger.isLoggingError ()) {
////                    pLogger.logError (Constants.COMPARABLE_ERROR, exc, pLeft.getClass ().getName (),
////                            (pRight == null) ? "null" : pRight.getClass ().getName (),
////                            pOperator.getOperatorSymbol ());
////                }
//                return Boolean.FALSE;
//            }
//        } else if (pRight instanceof Comparable) {
//            try {
//                int result = ((Comparable) pRight).compareTo (pLeft);
//                return Boolean.valueOf (operateRelational (-result, result, pOperator));
//            } catch (Exception exc) {
////                if (pLogger.isLoggingError ()) {
////                    pLogger.logError (Constants.COMPARABLE_ERROR, exc, pRight.getClass ().getName (),
////                            (pLeft == null) ? "null" : pLeft.getClass ().getName (),
////                            pOperator.getOperatorSymbol ());
////                }
//                return Boolean.FALSE;
//            }
        } else {
//            if (pLogger.isLoggingError ()) {
//                pLogger.logError (Constants.ARITH_OP_BAD_TYPE, pOperator.getOperatorSymbol (),
//                        pLeft.getClass ().getName (),
//                        pRight.getClass ().getName ());
//            }
            return Boolean.FALSE;
        }
    }

    //-------------------------------------

    private static Boolean operateEquality (boolean value, int operator) {
        switch (operator) {
            case ELUtils.OPERATOR_EQUAL:
                return value ? Boolean.TRUE : Boolean.FALSE;
            case ELUtils.OPERATOR_NON_EQUAL:
                return ! value ? Boolean.TRUE : Boolean.FALSE;
            default:
                throw new IllegalStateException ();
        }
    }

    /**
     * Performs all of the necessary type conversions, then calls on the
     * appropriate operator.
     */
    public static Boolean applyEqualityOperator (Object pLeft, Object pRight, int pOperator, Logger pLogger) {
        if (pLeft == pRight) {
            return operateEquality (true, pOperator);
        } else if (pLeft == null || pRight == null) {
            return operateEquality (false, pOperator);
//        } else if (isBigDecimal (pLeft) || isBigDecimal (pRight)) {
//            BigDecimal left = (BigDecimal) coerceObjectToPrimitiveNumber (pLeft, BigDecimal.class, pLogger);
//            BigDecimal right = (BigDecimal) coerceObjectToPrimitiveNumber (pRight, BigDecimal.class, pLogger);
//            return operateEquality (left.equals (right), pOperator);
        } else if (isFloatingPointType (pLeft) || isFloatingPointType (pRight)) {
            double left = doubleValue (coerceObjectToPrimitiveNumber (pLeft, Double.class, pLogger));
            double right = doubleValue (coerceObjectToPrimitiveNumber (pRight, Double.class, pLogger));
            return operateEquality (left == right, pOperator);
//        } else if (isBigInteger (pLeft) || isBigInteger (pRight)) {
//            BigInteger left = (BigInteger) coerceObjectToPrimitiveNumber (pLeft, BigInteger.class, pLogger);
//            BigInteger right = (BigInteger) coerceObjectToPrimitiveNumber (pRight, BigInteger.class, pLogger);
//            return operateEquality (left.equals (right), pOperator);
        } else if (isIntegerType (pLeft) || isIntegerType (pRight)) {
            long left = longValue (coerceObjectToPrimitiveNumber (pLeft, Long.class, pLogger));
            long right = longValue (coerceObjectToPrimitiveNumber (pRight, Long.class, pLogger));
            return operateEquality (left == right, pOperator);
        } else if (pLeft instanceof Boolean || pRight instanceof Boolean) {
            boolean left = coerceObjectToBoolean (pLeft, pLogger).booleanValue ();
            boolean right = coerceObjectToBoolean (pRight, pLogger).booleanValue ();
            return operateEquality (left == right, pOperator);
        } else if (pLeft instanceof String || pRight instanceof String) {
            String left = coerceObjectToString (pLeft, pLogger);
            String right = coerceObjectToString (pRight, pLogger);
            return operateEquality (left.equals (right), pOperator);
        } else {
            try {
                return operateEquality (pLeft.equals (pRight), pOperator);
            } catch (Exception exc) {
                if (pLogger.isLoggingError ()) {
//                    pLogger.logError (Constants.ERROR_IN_EQUALS, exc,
//                            pLeft.getClass ().getName (),
//                            pRight.getClass ().getName (),
//                            pOperator.getOperatorSymbol ());
                }
                return Boolean.FALSE;
            }
        }
    }

    //-------------------------------------

    private static boolean isInstanceOfNumber (Object object) {
        return object != null  &&  (isIntegerType (object.getClass ())  ||  isFloatingPointType (object.getClass ()));
    }

    private static byte byteValue (Object object) {
        return (byte) intValue (object);
    }

    private static short shortValue (Object object) {
        return (short) intValue (object);
    }

    private static int intValue (Object object) {
        if (object instanceof Character)
            return ((Character) object).charValue ();
        if (object instanceof Byte)
            return ((Byte) object).byteValue ();
        if (object instanceof Short)
            return ((Short) object).shortValue ();
        if (object instanceof Integer)
            return ((Integer) object).intValue ();
        if (object instanceof Long)
            return (int) ((Long) object).longValue ();
        if (object instanceof Float)
            return (int) ((Float) object).floatValue ();
        if (object instanceof Double)
            return (int) ((Double) object).doubleValue ();
        return 0;
    }

    public static long longValue (Object object) {
        if (object instanceof Character)
            return ((Character) object).charValue ();
        if (object instanceof Byte)
            return ((Byte) object).byteValue ();
        if (object instanceof Short)
            return ((Short) object).shortValue ();
        if (object instanceof Integer)
            return ((Integer) object).intValue ();
        if (object instanceof Long)
            return ((Long) object).longValue ();
        if (object instanceof Float)
            return (long) ((Float) object).floatValue ();
        if (object instanceof Double)
            return (long) ((Double) object).doubleValue ();
        return 0l;
    }

    private static float floatValue (Object object) {
        if (object instanceof Character)
            return ((Character) object).charValue ();
        if (object instanceof Byte)
            return ((Byte) object).byteValue ();
        if (object instanceof Short)
            return ((Short) object).shortValue ();
        if (object instanceof Integer)
            return ((Integer) object).intValue ();
        if (object instanceof Long)
            return ((Long) object).longValue ();
        if (object instanceof Float)
            return ((Float) object).floatValue ();
        if (object instanceof Double)
            return (float) ((Double) object).doubleValue ();
        return 0.0f;
    }

    public static double doubleValue (Object object) {
        if (object instanceof Character)
            return ((Character) object).charValue ();
        if (object instanceof Byte)
            return ((Byte) object).byteValue ();
        if (object instanceof Short)
            return ((Short) object).shortValue ();
        if (object instanceof Integer)
            return ((Integer) object).intValue ();
        if (object instanceof Long)
            return ((Long) object).longValue ();
        if (object instanceof Float)
            return ((Float) object).floatValue ();
        if (object instanceof Double)
            return ((Double) object).doubleValue ();
        return 0.0;
    }

    /**
     * Returns true if the given Object is of a floating point type
     */
    public static boolean isFloatingPointType (Object pObject) {
        return pObject != null && isFloatingPointType (pObject.getClass ());
    }

    //-------------------------------------

    /**
     * Returns true if the given class is of a floating point type
     */
    private static boolean isFloatingPointType (Class pClass) {
        return pClass == Float.class ||
//                pClass == Float.TYPE ||
                pClass == Double.class;// ||
//                pClass == Double.TYPE;
    }

    //-------------------------------------

    /**
     * Returns true if the given string might contain a floating point
     * number - i.e., it contains ".", "e", or "E"
     */
    public static boolean isFloatingPointString (Object pObject) {
        if (pObject instanceof String) {
            String str = (String) pObject;
            int len = str.length ();
            for (int i = 0; i < len; i++) {
                char ch = str.charAt (i);
                if (ch == '.' ||
                    ch == 'e' ||
                    ch == 'E') {
                    return true;
                }
            }
            return false;
        } else {
            return false;
        }
    }

    //-------------------------------------

    /**
     * Returns true if the given Object is of an integer type
     */
    private static boolean isIntegerType (Object pObject) {
        return pObject != null && isIntegerType (pObject.getClass ());
    }

    //-------------------------------------

    /**
     * Returns true if the given class is of an integer type
     */
    private static boolean isIntegerType (Class pClass) {
        return pClass == Byte.class ||
//                pClass == Byte.TYPE ||
                pClass == Short.class ||
//                pClass == Short.TYPE ||
                pClass == Character.class ||
//                pClass == Character.TYPE ||
                pClass == Integer.class ||
//                pClass == Integer.TYPE ||
                pClass == Long.class;// ||
//                pClass == Long.TYPE;
    }

    //-------------------------------------

//    /**
//     * Returns true if the given object is BigInteger.
//     * @param pObject - Object to evaluate
//     * @return - true if the given object is BigInteger
//     */
//    private static boolean isBigInteger (Object pObject) {
//        return pObject != null && pObject instanceof BigInteger;
//    }

//    /**
//     * Returns true if the given object is BigDecimal.
//     * @param pObject - Object to evaluate
//     * @return - true if the given object is BigDecimal
//     */
//    private static boolean isBigDecimal (Object pObject) {
//        return pObject != null && pObject instanceof BigDecimal;
//    }

}
