/***************************************************************************
 *
 * Routine to read a file to trace group for a given time window.
 *
 * Written by Lion Krischer and Robert Barsch
 *
 * Modified from original version by Chad Trabant.
 *
 * modified: 2010.099
 ***************************************************************************/

#include <stdio.h>
#include "libmseed.h"


/*********************************************************************
 * ms_readtraces_window 
 *
 * This routine will open and read all Mini-SEED records in specified
 * file and populate a trace group.  This routine is thread safe.
 *
 * If reclen is 0 the length of the first record is automatically
 * detected, all subsequent records are then expected to have the same
 * length as the first.
 *
 * If reclen is negative the length of every record is automatically
 * detected.
 *
 * Will only read records that have at least one sample in between
 * starttime and endtime.
 *
 * Returns MS_NOERROR and populates an MSTraceGroup struct at *ppmstg
 * on successful read, otherwise returns a libmseed error code (listed
 * in libmseed.h).
 *********************************************************************/
int ms_readtraces_window (MSTraceGroup **ppmstg, char *msfile, int reclen,
	       double timetol, double sampratetol, flag dataquality,
	       flag skipnotdata, flag dataflag, flag verbose,
	       hptime_t starttime, hptime_t endtime)
{
  MSRecord *msr = 0;
  MSFileParam *msfp = 0;
  int retcode;
  /* File pos at start */
  int start_pos = 0;

  if ( ! ppmstg )
    return MS_GENERROR;
  
  /* Initialize MSTraceGroup if needed */
  if ( ! *ppmstg )
    {
      *ppmstg = mst_initgroup (*ppmstg);
      
      if ( ! *ppmstg )
      {
        return MS_GENERROR;
      }
    }

  
  /* Loop over the input file */
  while ( (retcode = ms_readmsr_main (&msfp, &msr, msfile, reclen, NULL, NULL,
           skipnotdata, 0, NULL, verbose)) == MS_NOERROR)
    {
      if (starttime > msr_endtime(msr) || endtime < msr->starttime)
         {
          start_pos = ftell (msfp->fp);
          continue;
         }

      fseek (msfp->fp, start_pos, SEEK_SET);
      ms_readmsr_main (&msfp, &msr, msfile, msr->reclen, NULL, NULL,
                       skipnotdata, dataflag, NULL, verbose);
      mst_addmsrtogroup (*ppmstg, msr, dataquality, timetol, sampratetol);
      start_pos = ftell (msfp->fp);
    }
  
  /* Reset return code to MS_NOERROR on successful read by ms_readmsr() */
  if ( retcode == MS_ENDOFFILE )
    retcode = MS_NOERROR;

  ms_readmsr_main (&msfp, &msr, NULL, 0, NULL, NULL, 0, 0, NULL, 0);

  return retcode;
}
