/*
Copyright (c) 2007, John Hurst
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/*! \file    S429-5-cgi.cpp
    \version $Id: S429-5-cgi.cpp,v 1.7 2007/10/29 21:24:32 msheby Exp $       
    \brief   AS-DCP Timed Text CGI tool -- deliver SMPTE 429-5 Timed Text resources via HTTP
*/

/*
This is a Common Gateway Interface program that allows a web server to deliver
Timed Text resources from a SMPTE 429-5 MXF file.

NOTE: The URL structure and filesystem layout given below are determined by this
program. The underlying AS-DCP library accessors can be used in many other schemes.
If this program does not meet your needs, you can use the AS-DCP API directly to
create your own solution.

To test this module from the command line, execute the program with the appropriate
environment variables:

        GATEWAY_INTERFACE=CGI/1.1 PATH_INFO=<MXF-UUID>,<Asset-UUID> S429-5-cgi

where <MXF-UUID> identifies the PackageUID of an MXF container file in the working
directory and <Asset-UUID> identifies a timed-text resource within the MXF file.
A different working directory can be used by setting the optional MXF_TIMED_TEXT
environment variable with a directory name.

The working directory should contain one or more Timed Text Track Files having
filenames ending in '.mxf'. In addition, the directory must contain an index file
named 'tt-mxf-idx.txt'. The index file provides a mapping between the UUID values
in the MXF files and the filenames of the MXF files. The index file format is very
simple. For each UUID in the set of MXF files, a line in the file will contain the
MXF file's PackageUID, the Asset UUID and the associated filename:

        <MXF-UUID>,<Asset-UUID>/<filename>

Each line must end with a newline (0x0a). The index file can be created by invoking
this program from the command line with a '-i' flag. The index will be automatically
created if it does not already exist.

A handy shell script that retrieves all indexed resources:

export MXF_TIMED_TEXT=my_mxf_root
for id in `cat $MXF_TIMED_TEXT/tt-mxf-idx.txt | awk -F/ '{print $1}'`
do
    GATEWAY_INTERFACE=CGI/1.1 PATH_INFO=$id S429-5-cgi >/dev/null
done

*/

#include <AS_DCP.h>
#include <KM_fileio.h>
#include <vector>

using namespace Kumu;
using namespace ASDCP;
using std::endl;

const std::string IndexName = "tt-mxf-idx.txt";

#ifdef KM_WIN32
#error "This program uses non-Windows programming idioms."
#endif

// print an HTML-formatted error message with HTTP header
void
do_error(const std::string& err_str)
{
  printf("ContentType: text/html\r\n\r\n\
<html>\n\
  <head><title>Timed Text provider error</title>\n\
  <body>\n\
    <p>Timed Text processing error, please check your server configuration.</p>\n\
    <pre>\n\
%s\n\
</pre>\n\
  </body>\n\
</html>\n", err_str.c_str());
}

//
bool
create_index(const std::string& ResourceDir)
{
  std::string IndexData;
  PathList_t MXFFiles;
  PathList_t::iterator pi;
  Result_t result = RESULT_OK;
  char buf[64], puid_buf[64];

  FindInPath(PathMatchGlob("*.mxf"), ResourceDir, MXFFiles);

  for ( pi = MXFFiles.begin(); pi != MXFFiles.end() && KM_SUCCESS(result); pi++ )
    {
      std::string Basename = PathBasename(*pi);
      TimedText::MXFReader  Reader;
      result = Reader.OpenRead((*pi).c_str());

      if ( KM_SUCCESS(result) )
	{
	  UUID PackageUID, TmpID;
	  TimedText::TimedTextDescriptor TDesc;
	  WriterInfo Info;

	  Reader.FillWriterInfo(Info);
	  Reader.FillDescriptor(TDesc);
	  PackageUID.Set(Info.AssetUUID);
	  PackageUID.EncodeHex(puid_buf, 64);
	  strcat(puid_buf, ",");

	  TmpID.Set(TDesc.AssetID);
	  IndexData += puid_buf;
	  IndexData += TmpID.EncodeHex(buf, 64);
	  IndexData += "/" + Basename + "\n";

	  TimedText::ResourceList_t::const_iterator ri;
	  for ( ri = TDesc.ResourceList.begin() ; ri != TDesc.ResourceList.end(); ri++ )
	    {
	      TmpID.Set((*ri).ResourceID);
	      IndexData += puid_buf;
	      IndexData += TmpID.EncodeHex(buf, 64);
	      IndexData += "/" + Basename + "\n";
	    }
	}
    }

  if ( KM_SUCCESS(result) )
    {
      std::string IndexPath = ResourceDir + "/" + IndexName;
      result = WriteStringIntoFile(IndexPath.c_str(), IndexData);

      if ( KM_FAILURE(result) )
	{
	  fprintf(stderr, "%s: %s", IndexPath.c_str(), result.Label());
	  return false;
	}
    }

  return true;
}

//
Result_t
fetch_resource(const std::string ResourceDir, const std::string& ResourcePath,
	       TimedText::FrameBuffer& ResourceBuffer, std::string& error)
{
  if ( ! PathIsDirectory(ResourceDir) )
    {
      error = "Missing MXF source directory: " + ResourceDir;
      return RESULT_FAIL;
    }

  std::string IndexPath = ResourceDir + "/" + IndexName;

  if ( ! PathIsFile(IndexPath) )
    {
      if ( ! create_index(ResourceDir) )
	{
	  error = "Unable to create MXF index in source directory: " + ResourceDir;
	  return RESULT_FAIL;
	}
    }

  std::string IndexData;
  Result_t result = ReadFileIntoString(IndexPath.c_str(), IndexData);
  if ( KM_FAILURE(result) )
    return result;

  const char* p = strstr(IndexData.c_str(), ResourcePath.c_str());

  if ( p == 0 )
    {
      error = "UUID not found in index: ";
      error += ResourcePath.c_str();
      return RESULT_FAIL;
    }

  const char* fn = strchr(p, '/');

  if ( fn == 0 || *(fn+1) == 0 )
    {
      error = "Index file format error 1.";
      return RESULT_FAIL;
    }

  const char* nl = strchr(fn++, '\n');

  if ( nl == 0 )
    {
      error = "Index file format error 2.";
      return RESULT_FAIL;
    }

  const char* cp = strchr(ResourcePath.c_str(), ',');

  if ( cp == 0 )
    {
      error = "Index file format error 3.";
      return RESULT_FAIL;
    }

  UUID RID;
  TimedText::MXFReader  Reader;
  std::string mxf_filename;

  if ( ! RID.DecodeHex(cp+1) )
    {
      error = "Index file format error 4.";
      return RESULT_FAIL;
    }

  mxf_filename.assign(fn, nl - fn);
  mxf_filename = ResourceDir + "/" + mxf_filename;
  result = Reader.OpenRead(mxf_filename.c_str());

  if ( KM_SUCCESS(result) )
    {
      TimedText::TimedTextDescriptor TDesc;
      Reader.FillDescriptor(TDesc);

      if ( RID == TDesc.AssetID )
	result = Reader.ReadTimedTextResource(ResourceBuffer);

      else
	result = Reader.ReadAncillaryResource(RID.Value(), ResourceBuffer);
    }

  return result;
}


//
int 
main(int argc, const char **argv)
{

  const char* mxf_dirname = getenv("MXF_TIMED_TEXT");
  if ( mxf_dirname == 0 )
    mxf_dirname = ".";

  const char* cgi_version = getenv("GATEWAY_INTERFACE");

  if ( cgi_version == 0 )
    {
      if ( argc == 2 && ( argv[1][0] == '-' && argv[1][1] == 'i' && argv[1][2] == '\0' ) )
	{
	  if ( ! create_index(mxf_dirname) )
	    {
	      fprintf(stderr, "Create index failed.\n");
	      return 2;
	    }

	  return 0;
	}
      else
	{
	  fprintf(stderr, "Command line usage: S429-5-cgi -i (or else run as CGI).\n");
	  return 1;
	}
    }

  const char* path_info = getenv("PATH_INFO");
  std::string error;

  if ( path_info == 0 )
    error = "Path Info elements required.";
  else
    {
      TimedText::FrameBuffer ResourceBuffer(2*Kumu::Megabyte);
      std::string MIMEType;

      Result_t result = fetch_resource(mxf_dirname, path_info, ResourceBuffer, error);

      if ( KM_SUCCESS(result) )
	{
	  fprintf(stderr, "%s %s (%u bytes)\n", path_info, ResourceBuffer.MIMEType(), ResourceBuffer.Size());
	  printf("ContentType: %s\r\n\r\n", ResourceBuffer.MIMEType());
	  write(1, ResourceBuffer.Data(), ResourceBuffer.Size());
	}
      else
	{
	  if ( result != RESULT_FAIL )
	    error = result.Label();
	}
    }

  if ( ! error.empty() )
    do_error(error);

  return 0;
}

//
// end S429-cgi.cpp
//
