/*
 * Copyright 2010 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

package sun.java2d.xr;

import java.util.*;

/**
 * Growable int array, designed to allow subclasses to emulate
 * the behaviour of value types.
 *
 * @author Clemens Eisserer
 */

public class GrowableByteArray
{

        byte[] array;
        int size;
        int cellSize;

        public GrowableByteArray(int cellSize, int initialSize)
        {
                array = new byte[initialSize];
                size = 0;
                this.cellSize = cellSize;
        }

        private int getNextCellIndex()
        {
                int oldSize = size;
                size += cellSize;

                if (size >= array.length)
                {
                        growArray();
                }

                return oldSize;
        }

        /**
         * @return a direct reference to the backing array.
         */
        public byte[] getArray()
        {
                return array;
        }

        /**
         * @return a copy of the backing array.
         */
        public byte[] getSizedArray()
        {
                return Arrays.copyOf(array, getSize());
        }

        public final int getByte(int index)
        {
                return array[getCellIndex(index)];
        }

        /**
         * Returns the index of the next free cell,
         * and grows the backing arrays if required.
         */
        public final int getNextIndex()
        {
                return getNextCellIndex() / cellSize;
        }

        protected final int getCellIndex(int cellIndex)
        {
                return cellSize * cellIndex;
        }

        public final void addByte(byte i)
        {
            int nextIndex = getNextIndex();
            array[nextIndex] = i;
        }

        /**
         * @return The number of stored cells.
         */
        public final int getSize()
        {
                return size / cellSize;
        }

        public void clear()
        {
                size = 0;
        }

        protected void growArray()
        {
                int newSize = Math.max(array.length * 2, 10);
                byte[] oldArray = array;
                array = new byte[newSize];

                System.arraycopy(oldArray, 0, array, 0, oldArray.length);
        }

}
