/*
 * Copyright (c) 2008, 2009, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.dyn;

import java.dyn.*;
import static sun.dyn.MemberName.uncaughtException;

/**
 * Parts of CallSite known to the JVM.
 * @author jrose
 */
public class CallSiteImpl {
    // this implements the upcall from the JVM, MethodHandleNatives.makeDynamicCallSite:
    static CallSite makeSite(MethodHandle bootstrapMethod,
                             // Callee information:
                             String name, MethodType type,
                             // Call-site attributes, if any:
                             Object info,
                             // Caller information:
                             MemberName callerMethod, int callerBCI) {
        Class<?> caller = callerMethod.getDeclaringClass();
        if (bootstrapMethod == null) {
            // If there is no bootstrap method, throw IncompatibleClassChangeError.
            // This is a valid generic error type for resolution (JLS 12.3.3).
            throw new IncompatibleClassChangeError
                ("Class "+caller.getName()+" has not declared a bootstrap method for invokedynamic");
        }
        CallSite site;
        try {
            Object binding;
            if (false)  // switch when invokeGeneric works
                binding = bootstrapMethod.invokeGeneric(caller, name, type);
            else
                binding = bootstrapMethod.invokeVarargs(new Object[]{ caller, name, type });
            //System.out.println("BSM for "+name+type+" => "+binding);
            if (binding instanceof CallSite) {
                site = (CallSite) binding;
            } else if (binding instanceof MethodHandleProvider) {
                MethodHandle target = ((MethodHandleProvider) binding).asMethodHandle();
                site = new ConstantCallSite(target);
            } else {
                throw new ClassCastException("bootstrap method failed to produce a MethodHandle or CallSite");
            }
            PRIVATE_INITIALIZE_CALL_SITE.<void>invokeExact(site, name, type,
                                                           callerMethod, callerBCI);
            assert(site.getTarget() != null);
            assert(site.getTarget().type().equals(type));
        } catch (Throwable ex) {
            InvokeDynamicBootstrapError bex;
            if (ex instanceof InvokeDynamicBootstrapError)
                bex = (InvokeDynamicBootstrapError) ex;
            else
                bex = new InvokeDynamicBootstrapError("call site initialization exception", ex);
            throw bex;
        }
        return site;
    }

    // This method is private in CallSite because it touches private fields in CallSite.
    // These private fields (vmmethod, vmindex) are specific to the JVM.
    private static final MethodHandle PRIVATE_INITIALIZE_CALL_SITE;
    static {
        try {
            PRIVATE_INITIALIZE_CALL_SITE =
            MethodHandleImpl.IMPL_LOOKUP.findVirtual(CallSite.class, "initializeFromJVM",
                MethodType.methodType(void.class,
                                      String.class, MethodType.class,
                                      MemberName.class, int.class));
        } catch (NoAccessException ex) {
            throw uncaughtException(ex);
        }
    }

    public static void setCallSiteTarget(Access token, CallSite site, MethodHandle target) {
        Access.check(token);
        MethodHandleNatives.setCallSiteTarget(site, target);
    }
}
