using namespace std;
/*****************************************************************************
*
* File: PrecisionTimer.cpp
* Project: Osmose emulator.
*
* Description: This class handle accurate timer measure, using rdtsc pentium
* instruction. This code compile with gcc, under Linux, and windows. 
* Windows Sleep() methods works with millisecond, while Linux sleep works with
* seconds.
*
* Usage note:
* The Timer MUST be initialised with the method calibrate(int s). The longest is s
* the more accurate is the mesaurment. 
* 
* The timer handle two measure mode: ONE_SHOT or CUMULATIVE.
* -ONE_SHOT (default mode) works like this:
*
*  Call start() to start measuring,
*  Call stop() to stop it. (in this mode, stop() method return always true)
*  Call getDuration() to  get time duration in seconds.
*
* -CUMULATIVE:
*
* Call SetMeasureMode(ONE_SHOT | CUMULATIVE) with CUMULATIVE parameter.
* Call setCumulativeBufferSize(int) with number of measure to do before
* computing average duration.
* Call Start()
* Call Stop();	the delay is stored into private buffer. nb value++.
* When We have enough values (depending on cumulativeBufferSize), stop() returns
* true: There is a usable value available.
* Average duration is computed and displayed on screen, if verbose is true.
* Else get value with getDuration() method.
*
* Author: Vedder Bruno
* Date: 19/04/2004
* URL: http://bcz.emu-france.com/
*****************************************************************************/
#include <iostream>
#ifndef _RDTSC_TIMER_H
#define _RDTSC_TIMER_H
#ifdef linux
    #define PORTABLE_SLEEP_FUNC  sleep( s );
    #include <unistd.h>
#else
    #define PORTABLE_SLEEP_FUNC  Sleep( s * 1000 );
    #include <windows.h>
#endif

// Measurment mode.
#define ONE_SHOT		0
#define CUMULATIVE		1
#define VALUES_BUFFER_SIZE	1024

class PrecisionTimer
{
    public:
    
        PrecisionTimer();		                     // Constructor.
        unsigned long long getT0();	                     // Get tick nbr from boot at T0.
        unsigned long long getT1();                          // Get tick nbr from boot at T1.
	void setVerbose(bool v);	                     // Silent or verbose mode.
        void calibrate(int cal_s);                           // Compute tick per seconds.
        void start();			                     // Start measure.
        bool stop();                                         // Stop measure.
        unsigned long long getDeltaT1_T0();                  // Get T1 - T0 tick nbr.
        unsigned long long getTickPerSecond();               // Get number of tick per seconds.
	void setMeasureMode(int mode);			     // CUMUL/ ONE SHOT measure.
	void setCumulativeBufferSize(unsigned int buf_size); // Resize cumulative buffer size.
        double getDuration();                                // T1- T0 in seconds.
	void reset();					     // Reset Timer.
    
    private:

        int mode;					     // Cumulative or one shot.
	bool               verbose;                          // Quiet or silence mode.
	double	          *cumulative_buffer;                // Delay cumulative buffer.
	double             average_duration;                 // average length.
	unsigned int       values_number; 		     // total number of values in cumul mode.
	unsigned int       cumul_buffer_index;               // index in array of values.
        unsigned long long tick_per_second;                  // Self explanatory.
        unsigned long long T0;                               // Self explanatory.
        unsigned long long T1;                               // Self explanatory.
        double PrecisionTimer::getOneShotDuration();
};
#endif
