/*
    Written on 2004 by Bruno Vedder

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*****************************************************************************
*
* File: main.cpp
*
* Project: Osmose emulator.
*
* Description: This file contains Osmose main loop, handling keyboard, SDL
* event, and hardware emulation.
*
* Author: Vedder Bruno
* Date: 28/10/2004, 13h00
*
* URL: http://bcz.emu-france.com/
*****************************************************************************/

#include <iostream>
#include <iomanip>
#include "OsmoseCore.h"
#include "Options.h"
#include "Version.h"
#include "VideoFilterList.h"

using namespace std;


// Functions prototypes.
void usage();
void parseCommandLine(int a, char *rv[]);
void splashText();

Options              opt;        // SMS Machine related options.
EmulatorOptions      emu_opt;    // Emulator related options.

/*--------------------------------------------------------------------*/
/* This is the main method of osmose. It check args number, correct   */
/* Emulator options, and start emulation is Ok.                       */
/*--------------------------------------------------------------------*/
int main(int argc,char *argv[])
{
    // Reset default options.
    emu_opt.reset();
    opt.reset();
    
    // Display Osmose version, author.
    splashText();

    // Check arguments number. If not enough display usage.
    if (argc < 2)
    {
        usage();
        exit(0);
    }

    // emu_opt will be corrected depending on cmd line.
    parseCommandLine(argc, argv);
    OsmoseCore core(argv[1]);
    core.run();
    return 0;    
}

/*--------------------------------------------------------------------*/
/* This method displays osmose usage.				      */
/*--------------------------------------------------------------------*/
void usage()
{
    cout << "Usage is osmose filename, ex: osmose asterix.zip -fs -tv -nosound" << endl << endl;
    cout << "Options: "<< endl;
    cout << "    -paddle           emulates one axis paddle." << endl;
    cout << "    -acceleration x.x paddle acceleration (0.1 to 5 default "<< DEFAULT_ACCELERATION << ")" << endl;
    cout << "    -fs               run in fullscreen   (default: windowed)." << endl;
    cout << "    -jap              run as japanese sms (default: export)." << endl;
    cout << "    -exp              run as exported sms (default)." << endl;
    cout << "    -nosound          do not play sounds. (default: sound on)."<< endl;
    cout << "    -cm               codemaster games mem. mapper (EXPERIMENTAL default: off). " << endl;
    cout << "    -dp               use dark palette for screen emulation (default: off)." << endl;
    cout << "    -inifile          xxx use xxx as configuration file. " <<endl;
    cout << "    -fps              display fps in title bar." << endl;
    cout << "    -pal              emulates PAL/SECAM video timing (default: NTSC)."<< endl;
    cout << "    -monochrom        emulates B&W tv (default: off)." << endl;
    cout << "    -scale2x          scale2x filter implementation (default: off)." << endl;
    cout << "    -bilinear         bilinear filter (default: off)." << endl;
    cout << "    -tv               emulates TV interleaves (default: off)." << endl;
}

/*--------------------------------------------------------------------*/
/* This method parses command line, and update options structure with */
/* right options.						      */
/*--------------------------------------------------------------------*/
void parseCommandLine(int a, char *rv[])
{
    for (int i=0; i<a; i++)
    {
        if (strcmp(rv[i],"-fs")==0)
        {
            emu_opt.fullscreen_flag = true;
        }

        if (strcmp(rv[i],"-pal")==0)
        {
            opt.ntsc = false;
        }

        if (strcmp(rv[i],"-fps")==0)
        {
            emu_opt.display_fps = true;
        }

        if (strcmp(rv[i],"-jap")==0)
        {
            // Act as Japanese machine.
            opt.smsVersion = false;  
        }

        if (strcmp(rv[i],"-exp")==0)
        {
            // Act as Exported machine.
            opt.smsVersion = true;
        }

        if (strcmp(rv[i],"-tv")==0)
        {
            emu_opt.videoFilterType = TV_FILTER;
        }

        if (strcmp(rv[i],"-scale2x")==0)
        {
            emu_opt.videoFilterType = SCALE2X_FILTER;
        }

        if (strcmp(rv[i],"-monochrom")==0)
        {
            emu_opt.videoFilterType = MONOCHROM_FILTER;
        }

        if (strcmp(rv[i],"-acceleration")==0)
        {
	    // If actual argument nbr < argument number eg is there any args ?
	    if (i < a-1)
	    {
                opt.acceleration = atof( rv[i+1]);
		if (opt.acceleration == 0)
		{
	        cerr << "No valid acceleration parameter was given. Using default " << (float)DEFAULT_ACCELERATION << " value." << endl;
                    opt.acceleration = DEFAULT_ACCELERATION;
		}
		cout << "Paddle acceleration used : " << opt.acceleration << endl;
	    }
	    else
	    {
	        cerr << "No acceleration parameter was given. Using default " << (float)DEFAULT_ACCELERATION << " value." << endl;
                opt.acceleration = DEFAULT_ACCELERATION;
	    }
        }

        if (strcmp(rv[i],"-bilinear")==0)
        {
            emu_opt.videoFilterType = BILINEAR_FILTER;
        }

        if (strcmp(rv[i],"-nosound")==0)
        {
            emu_opt.sound = false;
        }

        if (strcmp(rv[i],"-cm")==0)
        {
            opt.codemaster = true;
        }

        if (strcmp(rv[i],"-paddle")==0)
        {
            opt.inputType = PADDLE;
        }

        if (strcmp(rv[i],"-dp")==0)
        {
            emu_opt.bright_palette = false;
        }

        if (strcmp(rv[i],"-time_limited")==0)
        {
            emu_opt.time_limited = true;
        }

        // The ini file MUST be the next parameter after -inifile option.
        if (strcmp(rv[i],"-inifile")==0)
        {
	    // If actual argument nbr < argument number eg is there any args ?
	    if (i < a-1)
	    {
                emu_opt.default_config = false;
	        strncpy(emu_opt.ini_file, rv[++i],512);
	    }
	    else
	    {
	        cerr << "No ini file was given, using default configuration." << endl;
                emu_opt.default_config = true;
	    }
        }
    }
}

/*--------------------------------------------------------------------*/
/* This method writes Osmose splash screen, version, author and build */
/* informations.						      */
/*--------------------------------------------------------------------*/
void splashText()
{
    cout << endl <<  __OSMOSE_VERSION__ <<" build on " << __DATE__<< " at " << __TIME__<< " Written by Bruno Vedder." << endl;
    cout << "Web site: http://bcz.emu-france.com" << endl << endl;
}

